/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribe.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an asynchronous transcription job that was created with the <code>StartTranscriptionJob</code> operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TranscriptionJob implements SdkPojo, Serializable,
        ToCopyableBuilder<TranscriptionJob.Builder, TranscriptionJob> {
    private static final SdkField<String> TRANSCRIPTION_JOB_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(TranscriptionJob::transcriptionJobName)).setter(setter(Builder::transcriptionJobName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TranscriptionJobName").build())
            .build();

    private static final SdkField<String> TRANSCRIPTION_JOB_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(TranscriptionJob::transcriptionJobStatusAsString)).setter(setter(Builder::transcriptionJobStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TranscriptionJobStatus").build())
            .build();

    private static final SdkField<String> LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(TranscriptionJob::languageCodeAsString)).setter(setter(Builder::languageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageCode").build()).build();

    private static final SdkField<Integer> MEDIA_SAMPLE_RATE_HERTZ_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(TranscriptionJob::mediaSampleRateHertz)).setter(setter(Builder::mediaSampleRateHertz))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MediaSampleRateHertz").build())
            .build();

    private static final SdkField<String> MEDIA_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(TranscriptionJob::mediaFormatAsString)).setter(setter(Builder::mediaFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MediaFormat").build()).build();

    private static final SdkField<Media> MEDIA_FIELD = SdkField.<Media> builder(MarshallingType.SDK_POJO)
            .getter(getter(TranscriptionJob::media)).setter(setter(Builder::media)).constructor(Media::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Media").build()).build();

    private static final SdkField<Transcript> TRANSCRIPT_FIELD = SdkField.<Transcript> builder(MarshallingType.SDK_POJO)
            .getter(getter(TranscriptionJob::transcript)).setter(setter(Builder::transcript)).constructor(Transcript::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Transcript").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(TranscriptionJob::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(TranscriptionJob::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<Instant> COMPLETION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(TranscriptionJob::completionTime)).setter(setter(Builder::completionTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletionTime").build()).build();

    private static final SdkField<String> FAILURE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(TranscriptionJob::failureReason)).setter(setter(Builder::failureReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureReason").build()).build();

    private static final SdkField<Settings> SETTINGS_FIELD = SdkField.<Settings> builder(MarshallingType.SDK_POJO)
            .getter(getter(TranscriptionJob::settings)).setter(setter(Builder::settings)).constructor(Settings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Settings").build()).build();

    private static final SdkField<JobExecutionSettings> JOB_EXECUTION_SETTINGS_FIELD = SdkField
            .<JobExecutionSettings> builder(MarshallingType.SDK_POJO).getter(getter(TranscriptionJob::jobExecutionSettings))
            .setter(setter(Builder::jobExecutionSettings)).constructor(JobExecutionSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobExecutionSettings").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TRANSCRIPTION_JOB_NAME_FIELD,
            TRANSCRIPTION_JOB_STATUS_FIELD, LANGUAGE_CODE_FIELD, MEDIA_SAMPLE_RATE_HERTZ_FIELD, MEDIA_FORMAT_FIELD, MEDIA_FIELD,
            TRANSCRIPT_FIELD, START_TIME_FIELD, CREATION_TIME_FIELD, COMPLETION_TIME_FIELD, FAILURE_REASON_FIELD, SETTINGS_FIELD,
            JOB_EXECUTION_SETTINGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String transcriptionJobName;

    private final String transcriptionJobStatus;

    private final String languageCode;

    private final Integer mediaSampleRateHertz;

    private final String mediaFormat;

    private final Media media;

    private final Transcript transcript;

    private final Instant startTime;

    private final Instant creationTime;

    private final Instant completionTime;

    private final String failureReason;

    private final Settings settings;

    private final JobExecutionSettings jobExecutionSettings;

    private TranscriptionJob(BuilderImpl builder) {
        this.transcriptionJobName = builder.transcriptionJobName;
        this.transcriptionJobStatus = builder.transcriptionJobStatus;
        this.languageCode = builder.languageCode;
        this.mediaSampleRateHertz = builder.mediaSampleRateHertz;
        this.mediaFormat = builder.mediaFormat;
        this.media = builder.media;
        this.transcript = builder.transcript;
        this.startTime = builder.startTime;
        this.creationTime = builder.creationTime;
        this.completionTime = builder.completionTime;
        this.failureReason = builder.failureReason;
        this.settings = builder.settings;
        this.jobExecutionSettings = builder.jobExecutionSettings;
    }

    /**
     * <p>
     * The name of the transcription job.
     * </p>
     * 
     * @return The name of the transcription job.
     */
    public String transcriptionJobName() {
        return transcriptionJobName;
    }

    /**
     * <p>
     * The status of the transcription job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #transcriptionJobStatus} will return {@link TranscriptionJobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #transcriptionJobStatusAsString}.
     * </p>
     * 
     * @return The status of the transcription job.
     * @see TranscriptionJobStatus
     */
    public TranscriptionJobStatus transcriptionJobStatus() {
        return TranscriptionJobStatus.fromValue(transcriptionJobStatus);
    }

    /**
     * <p>
     * The status of the transcription job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #transcriptionJobStatus} will return {@link TranscriptionJobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #transcriptionJobStatusAsString}.
     * </p>
     * 
     * @return The status of the transcription job.
     * @see TranscriptionJobStatus
     */
    public String transcriptionJobStatusAsString() {
        return transcriptionJobStatus;
    }

    /**
     * <p>
     * The language code for the input speech.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return The language code for the input speech.
     * @see LanguageCode
     */
    public LanguageCode languageCode() {
        return LanguageCode.fromValue(languageCode);
    }

    /**
     * <p>
     * The language code for the input speech.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return The language code for the input speech.
     * @see LanguageCode
     */
    public String languageCodeAsString() {
        return languageCode;
    }

    /**
     * <p>
     * The sample rate, in Hertz, of the audio track in the input media file.
     * </p>
     * 
     * @return The sample rate, in Hertz, of the audio track in the input media file.
     */
    public Integer mediaSampleRateHertz() {
        return mediaSampleRateHertz;
    }

    /**
     * <p>
     * The format of the input media file.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mediaFormat} will
     * return {@link MediaFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #mediaFormatAsString}.
     * </p>
     * 
     * @return The format of the input media file.
     * @see MediaFormat
     */
    public MediaFormat mediaFormat() {
        return MediaFormat.fromValue(mediaFormat);
    }

    /**
     * <p>
     * The format of the input media file.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mediaFormat} will
     * return {@link MediaFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #mediaFormatAsString}.
     * </p>
     * 
     * @return The format of the input media file.
     * @see MediaFormat
     */
    public String mediaFormatAsString() {
        return mediaFormat;
    }

    /**
     * <p>
     * An object that describes the input media for the transcription job.
     * </p>
     * 
     * @return An object that describes the input media for the transcription job.
     */
    public Media media() {
        return media;
    }

    /**
     * <p>
     * An object that describes the output of the transcription job.
     * </p>
     * 
     * @return An object that describes the output of the transcription job.
     */
    public Transcript transcript() {
        return transcript;
    }

    /**
     * <p>
     * A timestamp that shows with the job was started processing.
     * </p>
     * 
     * @return A timestamp that shows with the job was started processing.
     */
    public Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * A timestamp that shows when the job was created.
     * </p>
     * 
     * @return A timestamp that shows when the job was created.
     */
    public Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * A timestamp that shows when the job was completed.
     * </p>
     * 
     * @return A timestamp that shows when the job was completed.
     */
    public Instant completionTime() {
        return completionTime;
    }

    /**
     * <p>
     * If the <code>TranscriptionJobStatus</code> field is <code>FAILED</code>, this field contains information about
     * why the job failed.
     * </p>
     * <p>
     * The <code>FailureReason</code> field can contain one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Unsupported media format</code> - The media format specified in the <code>MediaFormat</code> field of the
     * request isn't valid. See the description of the <code>MediaFormat</code> field for a list of valid values.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>The media format provided does not match the detected media format</code> - The media format of the audio
     * file doesn't match the format specified in the <code>MediaFormat</code> field in the request. Check the media
     * format of your media file and make sure that the two values match.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Invalid sample rate for audio file</code> - The sample rate specified in the
     * <code>MediaSampleRateHertz</code> of the request isn't valid. The sample rate must be between 8000 and 48000
     * Hertz.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>The sample rate provided does not match the detected sample rate</code> - The sample rate in the audio file
     * doesn't match the sample rate specified in the <code>MediaSampleRateHertz</code> field in the request. Check the
     * sample rate of your media file and make sure that the two values match.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Invalid file size: file size too large</code> - The size of your audio file is larger than Amazon
     * Transcribe can process. For more information, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Limits</a> in the <i>Amazon
     * Transcribe Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Invalid number of channels: number of channels too large</code> - Your audio contains more channels than
     * Amazon Transcribe is configured to process. To request additional channels, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws_service_limits.html#limits-amazon-transcribe">Amazon
     * Transcribe Limits</a> in the <i>Amazon Web Services General Reference</i>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return If the <code>TranscriptionJobStatus</code> field is <code>FAILED</code>, this field contains information
     *         about why the job failed.</p>
     *         <p>
     *         The <code>FailureReason</code> field can contain one of the following values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Unsupported media format</code> - The media format specified in the <code>MediaFormat</code> field
     *         of the request isn't valid. See the description of the <code>MediaFormat</code> field for a list of valid
     *         values.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>The media format provided does not match the detected media format</code> - The media format of the
     *         audio file doesn't match the format specified in the <code>MediaFormat</code> field in the request. Check
     *         the media format of your media file and make sure that the two values match.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Invalid sample rate for audio file</code> - The sample rate specified in the
     *         <code>MediaSampleRateHertz</code> of the request isn't valid. The sample rate must be between 8000 and
     *         48000 Hertz.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>The sample rate provided does not match the detected sample rate</code> - The sample rate in the
     *         audio file doesn't match the sample rate specified in the <code>MediaSampleRateHertz</code> field in the
     *         request. Check the sample rate of your media file and make sure that the two values match.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Invalid file size: file size too large</code> - The size of your audio file is larger than Amazon
     *         Transcribe can process. For more information, see <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Limits</a> in the
     *         <i>Amazon Transcribe Developer Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Invalid number of channels: number of channels too large</code> - Your audio contains more channels
     *         than Amazon Transcribe is configured to process. To request additional channels, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws_service_limits.html#limits-amazon-transcribe"
     *         >Amazon Transcribe Limits</a> in the <i>Amazon Web Services General Reference</i>.
     *         </p>
     *         </li>
     */
    public String failureReason() {
        return failureReason;
    }

    /**
     * <p>
     * Optional settings for the transcription job. Use these settings to turn on speaker recognition, to set the
     * maximum number of speakers that should be identified and to specify a custom vocabulary to use when processing
     * the transcription job.
     * </p>
     * 
     * @return Optional settings for the transcription job. Use these settings to turn on speaker recognition, to set
     *         the maximum number of speakers that should be identified and to specify a custom vocabulary to use when
     *         processing the transcription job.
     */
    public Settings settings() {
        return settings;
    }

    /**
     * <p>
     * Provides information about how a transcription job is executed.
     * </p>
     * 
     * @return Provides information about how a transcription job is executed.
     */
    public JobExecutionSettings jobExecutionSettings() {
        return jobExecutionSettings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(transcriptionJobName());
        hashCode = 31 * hashCode + Objects.hashCode(transcriptionJobStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(languageCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(mediaSampleRateHertz());
        hashCode = 31 * hashCode + Objects.hashCode(mediaFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(media());
        hashCode = 31 * hashCode + Objects.hashCode(transcript());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(completionTime());
        hashCode = 31 * hashCode + Objects.hashCode(failureReason());
        hashCode = 31 * hashCode + Objects.hashCode(settings());
        hashCode = 31 * hashCode + Objects.hashCode(jobExecutionSettings());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TranscriptionJob)) {
            return false;
        }
        TranscriptionJob other = (TranscriptionJob) obj;
        return Objects.equals(transcriptionJobName(), other.transcriptionJobName())
                && Objects.equals(transcriptionJobStatusAsString(), other.transcriptionJobStatusAsString())
                && Objects.equals(languageCodeAsString(), other.languageCodeAsString())
                && Objects.equals(mediaSampleRateHertz(), other.mediaSampleRateHertz())
                && Objects.equals(mediaFormatAsString(), other.mediaFormatAsString()) && Objects.equals(media(), other.media())
                && Objects.equals(transcript(), other.transcript()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(completionTime(), other.completionTime())
                && Objects.equals(failureReason(), other.failureReason()) && Objects.equals(settings(), other.settings())
                && Objects.equals(jobExecutionSettings(), other.jobExecutionSettings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("TranscriptionJob").add("TranscriptionJobName", transcriptionJobName())
                .add("TranscriptionJobStatus", transcriptionJobStatusAsString()).add("LanguageCode", languageCodeAsString())
                .add("MediaSampleRateHertz", mediaSampleRateHertz()).add("MediaFormat", mediaFormatAsString())
                .add("Media", media()).add("Transcript", transcript()).add("StartTime", startTime())
                .add("CreationTime", creationTime()).add("CompletionTime", completionTime())
                .add("FailureReason", failureReason()).add("Settings", settings())
                .add("JobExecutionSettings", jobExecutionSettings()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TranscriptionJobName":
            return Optional.ofNullable(clazz.cast(transcriptionJobName()));
        case "TranscriptionJobStatus":
            return Optional.ofNullable(clazz.cast(transcriptionJobStatusAsString()));
        case "LanguageCode":
            return Optional.ofNullable(clazz.cast(languageCodeAsString()));
        case "MediaSampleRateHertz":
            return Optional.ofNullable(clazz.cast(mediaSampleRateHertz()));
        case "MediaFormat":
            return Optional.ofNullable(clazz.cast(mediaFormatAsString()));
        case "Media":
            return Optional.ofNullable(clazz.cast(media()));
        case "Transcript":
            return Optional.ofNullable(clazz.cast(transcript()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "CompletionTime":
            return Optional.ofNullable(clazz.cast(completionTime()));
        case "FailureReason":
            return Optional.ofNullable(clazz.cast(failureReason()));
        case "Settings":
            return Optional.ofNullable(clazz.cast(settings()));
        case "JobExecutionSettings":
            return Optional.ofNullable(clazz.cast(jobExecutionSettings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TranscriptionJob, T> g) {
        return obj -> g.apply((TranscriptionJob) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TranscriptionJob> {
        /**
         * <p>
         * The name of the transcription job.
         * </p>
         * 
         * @param transcriptionJobName
         *        The name of the transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transcriptionJobName(String transcriptionJobName);

        /**
         * <p>
         * The status of the transcription job.
         * </p>
         * 
         * @param transcriptionJobStatus
         *        The status of the transcription job.
         * @see TranscriptionJobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscriptionJobStatus
         */
        Builder transcriptionJobStatus(String transcriptionJobStatus);

        /**
         * <p>
         * The status of the transcription job.
         * </p>
         * 
         * @param transcriptionJobStatus
         *        The status of the transcription job.
         * @see TranscriptionJobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscriptionJobStatus
         */
        Builder transcriptionJobStatus(TranscriptionJobStatus transcriptionJobStatus);

        /**
         * <p>
         * The language code for the input speech.
         * </p>
         * 
         * @param languageCode
         *        The language code for the input speech.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(String languageCode);

        /**
         * <p>
         * The language code for the input speech.
         * </p>
         * 
         * @param languageCode
         *        The language code for the input speech.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(LanguageCode languageCode);

        /**
         * <p>
         * The sample rate, in Hertz, of the audio track in the input media file.
         * </p>
         * 
         * @param mediaSampleRateHertz
         *        The sample rate, in Hertz, of the audio track in the input media file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mediaSampleRateHertz(Integer mediaSampleRateHertz);

        /**
         * <p>
         * The format of the input media file.
         * </p>
         * 
         * @param mediaFormat
         *        The format of the input media file.
         * @see MediaFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MediaFormat
         */
        Builder mediaFormat(String mediaFormat);

        /**
         * <p>
         * The format of the input media file.
         * </p>
         * 
         * @param mediaFormat
         *        The format of the input media file.
         * @see MediaFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MediaFormat
         */
        Builder mediaFormat(MediaFormat mediaFormat);

        /**
         * <p>
         * An object that describes the input media for the transcription job.
         * </p>
         * 
         * @param media
         *        An object that describes the input media for the transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder media(Media media);

        /**
         * <p>
         * An object that describes the input media for the transcription job.
         * </p>
         * This is a convenience that creates an instance of the {@link Media.Builder} avoiding the need to create one
         * manually via {@link Media#builder()}.
         *
         * When the {@link Consumer} completes, {@link Media.Builder#build()} is called immediately and its result is
         * passed to {@link #media(Media)}.
         * 
         * @param media
         *        a consumer that will call methods on {@link Media.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #media(Media)
         */
        default Builder media(Consumer<Media.Builder> media) {
            return media(Media.builder().applyMutation(media).build());
        }

        /**
         * <p>
         * An object that describes the output of the transcription job.
         * </p>
         * 
         * @param transcript
         *        An object that describes the output of the transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transcript(Transcript transcript);

        /**
         * <p>
         * An object that describes the output of the transcription job.
         * </p>
         * This is a convenience that creates an instance of the {@link Transcript.Builder} avoiding the need to create
         * one manually via {@link Transcript#builder()}.
         *
         * When the {@link Consumer} completes, {@link Transcript.Builder#build()} is called immediately and its result
         * is passed to {@link #transcript(Transcript)}.
         * 
         * @param transcript
         *        a consumer that will call methods on {@link Transcript.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #transcript(Transcript)
         */
        default Builder transcript(Consumer<Transcript.Builder> transcript) {
            return transcript(Transcript.builder().applyMutation(transcript).build());
        }

        /**
         * <p>
         * A timestamp that shows with the job was started processing.
         * </p>
         * 
         * @param startTime
         *        A timestamp that shows with the job was started processing.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * A timestamp that shows when the job was created.
         * </p>
         * 
         * @param creationTime
         *        A timestamp that shows when the job was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * A timestamp that shows when the job was completed.
         * </p>
         * 
         * @param completionTime
         *        A timestamp that shows when the job was completed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completionTime(Instant completionTime);

        /**
         * <p>
         * If the <code>TranscriptionJobStatus</code> field is <code>FAILED</code>, this field contains information
         * about why the job failed.
         * </p>
         * <p>
         * The <code>FailureReason</code> field can contain one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Unsupported media format</code> - The media format specified in the <code>MediaFormat</code> field of
         * the request isn't valid. See the description of the <code>MediaFormat</code> field for a list of valid
         * values.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>The media format provided does not match the detected media format</code> - The media format of the
         * audio file doesn't match the format specified in the <code>MediaFormat</code> field in the request. Check the
         * media format of your media file and make sure that the two values match.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Invalid sample rate for audio file</code> - The sample rate specified in the
         * <code>MediaSampleRateHertz</code> of the request isn't valid. The sample rate must be between 8000 and 48000
         * Hertz.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>The sample rate provided does not match the detected sample rate</code> - The sample rate in the audio
         * file doesn't match the sample rate specified in the <code>MediaSampleRateHertz</code> field in the request.
         * Check the sample rate of your media file and make sure that the two values match.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Invalid file size: file size too large</code> - The size of your audio file is larger than Amazon
         * Transcribe can process. For more information, see <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Limits</a> in the
         * <i>Amazon Transcribe Developer Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Invalid number of channels: number of channels too large</code> - Your audio contains more channels
         * than Amazon Transcribe is configured to process. To request additional channels, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws_service_limits.html#limits-amazon-transcribe">Amazon
         * Transcribe Limits</a> in the <i>Amazon Web Services General Reference</i>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param failureReason
         *        If the <code>TranscriptionJobStatus</code> field is <code>FAILED</code>, this field contains
         *        information about why the job failed.</p>
         *        <p>
         *        The <code>FailureReason</code> field can contain one of the following values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Unsupported media format</code> - The media format specified in the <code>MediaFormat</code>
         *        field of the request isn't valid. See the description of the <code>MediaFormat</code> field for a list
         *        of valid values.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>The media format provided does not match the detected media format</code> - The media format of
         *        the audio file doesn't match the format specified in the <code>MediaFormat</code> field in the
         *        request. Check the media format of your media file and make sure that the two values match.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Invalid sample rate for audio file</code> - The sample rate specified in the
         *        <code>MediaSampleRateHertz</code> of the request isn't valid. The sample rate must be between 8000 and
         *        48000 Hertz.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>The sample rate provided does not match the detected sample rate</code> - The sample rate in the
         *        audio file doesn't match the sample rate specified in the <code>MediaSampleRateHertz</code> field in
         *        the request. Check the sample rate of your media file and make sure that the two values match.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Invalid file size: file size too large</code> - The size of your audio file is larger than
         *        Amazon Transcribe can process. For more information, see <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Limits</a> in
         *        the <i>Amazon Transcribe Developer Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Invalid number of channels: number of channels too large</code> - Your audio contains more
         *        channels than Amazon Transcribe is configured to process. To request additional channels, see <a href=
         *        "https://docs.aws.amazon.com/general/latest/gr/aws_service_limits.html#limits-amazon-transcribe"
         *        >Amazon Transcribe Limits</a> in the <i>Amazon Web Services General Reference</i>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReason(String failureReason);

        /**
         * <p>
         * Optional settings for the transcription job. Use these settings to turn on speaker recognition, to set the
         * maximum number of speakers that should be identified and to specify a custom vocabulary to use when
         * processing the transcription job.
         * </p>
         * 
         * @param settings
         *        Optional settings for the transcription job. Use these settings to turn on speaker recognition, to set
         *        the maximum number of speakers that should be identified and to specify a custom vocabulary to use
         *        when processing the transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder settings(Settings settings);

        /**
         * <p>
         * Optional settings for the transcription job. Use these settings to turn on speaker recognition, to set the
         * maximum number of speakers that should be identified and to specify a custom vocabulary to use when
         * processing the transcription job.
         * </p>
         * This is a convenience that creates an instance of the {@link Settings.Builder} avoiding the need to create
         * one manually via {@link Settings#builder()}.
         *
         * When the {@link Consumer} completes, {@link Settings.Builder#build()} is called immediately and its result is
         * passed to {@link #settings(Settings)}.
         * 
         * @param settings
         *        a consumer that will call methods on {@link Settings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #settings(Settings)
         */
        default Builder settings(Consumer<Settings.Builder> settings) {
            return settings(Settings.builder().applyMutation(settings).build());
        }

        /**
         * <p>
         * Provides information about how a transcription job is executed.
         * </p>
         * 
         * @param jobExecutionSettings
         *        Provides information about how a transcription job is executed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobExecutionSettings(JobExecutionSettings jobExecutionSettings);

        /**
         * <p>
         * Provides information about how a transcription job is executed.
         * </p>
         * This is a convenience that creates an instance of the {@link JobExecutionSettings.Builder} avoiding the need
         * to create one manually via {@link JobExecutionSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link JobExecutionSettings.Builder#build()} is called immediately and
         * its result is passed to {@link #jobExecutionSettings(JobExecutionSettings)}.
         * 
         * @param jobExecutionSettings
         *        a consumer that will call methods on {@link JobExecutionSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #jobExecutionSettings(JobExecutionSettings)
         */
        default Builder jobExecutionSettings(Consumer<JobExecutionSettings.Builder> jobExecutionSettings) {
            return jobExecutionSettings(JobExecutionSettings.builder().applyMutation(jobExecutionSettings).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String transcriptionJobName;

        private String transcriptionJobStatus;

        private String languageCode;

        private Integer mediaSampleRateHertz;

        private String mediaFormat;

        private Media media;

        private Transcript transcript;

        private Instant startTime;

        private Instant creationTime;

        private Instant completionTime;

        private String failureReason;

        private Settings settings;

        private JobExecutionSettings jobExecutionSettings;

        private BuilderImpl() {
        }

        private BuilderImpl(TranscriptionJob model) {
            transcriptionJobName(model.transcriptionJobName);
            transcriptionJobStatus(model.transcriptionJobStatus);
            languageCode(model.languageCode);
            mediaSampleRateHertz(model.mediaSampleRateHertz);
            mediaFormat(model.mediaFormat);
            media(model.media);
            transcript(model.transcript);
            startTime(model.startTime);
            creationTime(model.creationTime);
            completionTime(model.completionTime);
            failureReason(model.failureReason);
            settings(model.settings);
            jobExecutionSettings(model.jobExecutionSettings);
        }

        public final String getTranscriptionJobName() {
            return transcriptionJobName;
        }

        @Override
        public final Builder transcriptionJobName(String transcriptionJobName) {
            this.transcriptionJobName = transcriptionJobName;
            return this;
        }

        public final void setTranscriptionJobName(String transcriptionJobName) {
            this.transcriptionJobName = transcriptionJobName;
        }

        public final String getTranscriptionJobStatusAsString() {
            return transcriptionJobStatus;
        }

        @Override
        public final Builder transcriptionJobStatus(String transcriptionJobStatus) {
            this.transcriptionJobStatus = transcriptionJobStatus;
            return this;
        }

        @Override
        public final Builder transcriptionJobStatus(TranscriptionJobStatus transcriptionJobStatus) {
            this.transcriptionJobStatus(transcriptionJobStatus == null ? null : transcriptionJobStatus.toString());
            return this;
        }

        public final void setTranscriptionJobStatus(String transcriptionJobStatus) {
            this.transcriptionJobStatus = transcriptionJobStatus;
        }

        public final String getLanguageCodeAsString() {
            return languageCode;
        }

        @Override
        public final Builder languageCode(String languageCode) {
            this.languageCode = languageCode;
            return this;
        }

        @Override
        public final Builder languageCode(LanguageCode languageCode) {
            this.languageCode(languageCode == null ? null : languageCode.toString());
            return this;
        }

        public final void setLanguageCode(String languageCode) {
            this.languageCode = languageCode;
        }

        public final Integer getMediaSampleRateHertz() {
            return mediaSampleRateHertz;
        }

        @Override
        public final Builder mediaSampleRateHertz(Integer mediaSampleRateHertz) {
            this.mediaSampleRateHertz = mediaSampleRateHertz;
            return this;
        }

        public final void setMediaSampleRateHertz(Integer mediaSampleRateHertz) {
            this.mediaSampleRateHertz = mediaSampleRateHertz;
        }

        public final String getMediaFormatAsString() {
            return mediaFormat;
        }

        @Override
        public final Builder mediaFormat(String mediaFormat) {
            this.mediaFormat = mediaFormat;
            return this;
        }

        @Override
        public final Builder mediaFormat(MediaFormat mediaFormat) {
            this.mediaFormat(mediaFormat == null ? null : mediaFormat.toString());
            return this;
        }

        public final void setMediaFormat(String mediaFormat) {
            this.mediaFormat = mediaFormat;
        }

        public final Media.Builder getMedia() {
            return media != null ? media.toBuilder() : null;
        }

        @Override
        public final Builder media(Media media) {
            this.media = media;
            return this;
        }

        public final void setMedia(Media.BuilderImpl media) {
            this.media = media != null ? media.build() : null;
        }

        public final Transcript.Builder getTranscript() {
            return transcript != null ? transcript.toBuilder() : null;
        }

        @Override
        public final Builder transcript(Transcript transcript) {
            this.transcript = transcript;
            return this;
        }

        public final void setTranscript(Transcript.BuilderImpl transcript) {
            this.transcript = transcript != null ? transcript.build() : null;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        public final Instant getCompletionTime() {
            return completionTime;
        }

        @Override
        public final Builder completionTime(Instant completionTime) {
            this.completionTime = completionTime;
            return this;
        }

        public final void setCompletionTime(Instant completionTime) {
            this.completionTime = completionTime;
        }

        public final String getFailureReason() {
            return failureReason;
        }

        @Override
        public final Builder failureReason(String failureReason) {
            this.failureReason = failureReason;
            return this;
        }

        public final void setFailureReason(String failureReason) {
            this.failureReason = failureReason;
        }

        public final Settings.Builder getSettings() {
            return settings != null ? settings.toBuilder() : null;
        }

        @Override
        public final Builder settings(Settings settings) {
            this.settings = settings;
            return this;
        }

        public final void setSettings(Settings.BuilderImpl settings) {
            this.settings = settings != null ? settings.build() : null;
        }

        public final JobExecutionSettings.Builder getJobExecutionSettings() {
            return jobExecutionSettings != null ? jobExecutionSettings.toBuilder() : null;
        }

        @Override
        public final Builder jobExecutionSettings(JobExecutionSettings jobExecutionSettings) {
            this.jobExecutionSettings = jobExecutionSettings;
            return this;
        }

        public final void setJobExecutionSettings(JobExecutionSettings.BuilderImpl jobExecutionSettings) {
            this.jobExecutionSettings = jobExecutionSettings != null ? jobExecutionSettings.build() : null;
        }

        @Override
        public TranscriptionJob build() {
            return new TranscriptionJob(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
