/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribe;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.transcribe.model.BadRequestException;
import software.amazon.awssdk.services.transcribe.model.ConflictException;
import software.amazon.awssdk.services.transcribe.model.CreateLanguageModelRequest;
import software.amazon.awssdk.services.transcribe.model.CreateLanguageModelResponse;
import software.amazon.awssdk.services.transcribe.model.CreateMedicalVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.CreateMedicalVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.CreateVocabularyFilterRequest;
import software.amazon.awssdk.services.transcribe.model.CreateVocabularyFilterResponse;
import software.amazon.awssdk.services.transcribe.model.CreateVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.CreateVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteLanguageModelRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteLanguageModelResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteMedicalTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteMedicalTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteMedicalVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteMedicalVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteVocabularyFilterRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteVocabularyFilterResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.DescribeLanguageModelRequest;
import software.amazon.awssdk.services.transcribe.model.DescribeLanguageModelResponse;
import software.amazon.awssdk.services.transcribe.model.GetMedicalTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.GetMedicalTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.GetMedicalVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.GetMedicalVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.GetTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.GetTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.GetVocabularyFilterRequest;
import software.amazon.awssdk.services.transcribe.model.GetVocabularyFilterResponse;
import software.amazon.awssdk.services.transcribe.model.GetVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.GetVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.InternalFailureException;
import software.amazon.awssdk.services.transcribe.model.LimitExceededException;
import software.amazon.awssdk.services.transcribe.model.ListLanguageModelsRequest;
import software.amazon.awssdk.services.transcribe.model.ListLanguageModelsResponse;
import software.amazon.awssdk.services.transcribe.model.ListMedicalTranscriptionJobsRequest;
import software.amazon.awssdk.services.transcribe.model.ListMedicalTranscriptionJobsResponse;
import software.amazon.awssdk.services.transcribe.model.ListMedicalVocabulariesRequest;
import software.amazon.awssdk.services.transcribe.model.ListMedicalVocabulariesResponse;
import software.amazon.awssdk.services.transcribe.model.ListTranscriptionJobsRequest;
import software.amazon.awssdk.services.transcribe.model.ListTranscriptionJobsResponse;
import software.amazon.awssdk.services.transcribe.model.ListVocabulariesRequest;
import software.amazon.awssdk.services.transcribe.model.ListVocabulariesResponse;
import software.amazon.awssdk.services.transcribe.model.ListVocabularyFiltersRequest;
import software.amazon.awssdk.services.transcribe.model.ListVocabularyFiltersResponse;
import software.amazon.awssdk.services.transcribe.model.NotFoundException;
import software.amazon.awssdk.services.transcribe.model.StartMedicalTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.StartMedicalTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.StartTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.StartTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.TranscribeException;
import software.amazon.awssdk.services.transcribe.model.TranscribeRequest;
import software.amazon.awssdk.services.transcribe.model.UpdateMedicalVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.UpdateMedicalVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.UpdateVocabularyFilterRequest;
import software.amazon.awssdk.services.transcribe.model.UpdateVocabularyFilterResponse;
import software.amazon.awssdk.services.transcribe.model.UpdateVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.UpdateVocabularyResponse;
import software.amazon.awssdk.services.transcribe.paginators.ListLanguageModelsIterable;
import software.amazon.awssdk.services.transcribe.paginators.ListMedicalTranscriptionJobsIterable;
import software.amazon.awssdk.services.transcribe.paginators.ListMedicalVocabulariesIterable;
import software.amazon.awssdk.services.transcribe.paginators.ListTranscriptionJobsIterable;
import software.amazon.awssdk.services.transcribe.paginators.ListVocabulariesIterable;
import software.amazon.awssdk.services.transcribe.paginators.ListVocabularyFiltersIterable;
import software.amazon.awssdk.services.transcribe.transform.CreateLanguageModelRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.CreateMedicalVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.CreateVocabularyFilterRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.CreateVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteLanguageModelRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteMedicalTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteMedicalVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteVocabularyFilterRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DescribeLanguageModelRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetMedicalTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetMedicalVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetVocabularyFilterRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListLanguageModelsRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListMedicalTranscriptionJobsRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListMedicalVocabulariesRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListTranscriptionJobsRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListVocabulariesRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListVocabularyFiltersRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.StartMedicalTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.StartTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.UpdateMedicalVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.UpdateVocabularyFilterRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.UpdateVocabularyRequestMarshaller;

/**
 * Internal implementation of {@link TranscribeClient}.
 *
 * @see TranscribeClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTranscribeClient implements TranscribeClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultTranscribeClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a new custom language model. Use Amazon S3 prefixes to provide the location of your input files. The time
     * it takes to create your model depends on the size of your training data.
     * </p>
     *
     * @param createLanguageModelRequest
     * @return Result of the CreateLanguageModel operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws ConflictException
     *         There is already a resource with that name.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.CreateLanguageModel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/CreateLanguageModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateLanguageModelResponse createLanguageModel(CreateLanguageModelRequest createLanguageModelRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, ConflictException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateLanguageModelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateLanguageModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLanguageModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLanguageModel");

            return clientHandler.execute(new ClientExecutionParams<CreateLanguageModelRequest, CreateLanguageModelResponse>()
                    .withOperationName("CreateLanguageModel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createLanguageModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateLanguageModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new custom vocabulary that you can use to change how Amazon Transcribe Medical transcribes your audio
     * file.
     * </p>
     *
     * @param createMedicalVocabularyRequest
     * @return Result of the CreateMedicalVocabulary operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws ConflictException
     *         There is already a resource with that name.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.CreateMedicalVocabulary
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/CreateMedicalVocabulary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateMedicalVocabularyResponse createMedicalVocabulary(CreateMedicalVocabularyRequest createMedicalVocabularyRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, ConflictException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateMedicalVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateMedicalVocabularyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMedicalVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMedicalVocabulary");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateMedicalVocabularyRequest, CreateMedicalVocabularyResponse>()
                            .withOperationName("CreateMedicalVocabulary").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createMedicalVocabularyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateMedicalVocabularyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new custom vocabulary that you can use to change the way Amazon Transcribe handles transcription of an
     * audio file.
     * </p>
     *
     * @param createVocabularyRequest
     * @return Result of the CreateVocabulary operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws ConflictException
     *         There is already a resource with that name.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.CreateVocabulary
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/CreateVocabulary" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateVocabularyResponse createVocabulary(CreateVocabularyRequest createVocabularyRequest) throws BadRequestException,
            LimitExceededException, InternalFailureException, ConflictException, AwsServiceException, SdkClientException,
            TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateVocabularyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVocabulary");

            return clientHandler.execute(new ClientExecutionParams<CreateVocabularyRequest, CreateVocabularyResponse>()
                    .withOperationName("CreateVocabulary").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createVocabularyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateVocabularyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new vocabulary filter that you can use to filter words, such as profane words, from the output of a
     * transcription job.
     * </p>
     *
     * @param createVocabularyFilterRequest
     * @return Result of the CreateVocabularyFilter operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws ConflictException
     *         There is already a resource with that name.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.CreateVocabularyFilter
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/CreateVocabularyFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateVocabularyFilterResponse createVocabularyFilter(CreateVocabularyFilterRequest createVocabularyFilterRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, ConflictException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateVocabularyFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateVocabularyFilterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVocabularyFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVocabularyFilter");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateVocabularyFilterRequest, CreateVocabularyFilterResponse>()
                            .withOperationName("CreateVocabularyFilter").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createVocabularyFilterRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateVocabularyFilterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a custom language model using its name.
     * </p>
     *
     * @param deleteLanguageModelRequest
     * @return Result of the DeleteLanguageModel operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.DeleteLanguageModel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteLanguageModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteLanguageModelResponse deleteLanguageModel(DeleteLanguageModelRequest deleteLanguageModelRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteLanguageModelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteLanguageModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLanguageModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLanguageModel");

            return clientHandler.execute(new ClientExecutionParams<DeleteLanguageModelRequest, DeleteLanguageModelResponse>()
                    .withOperationName("DeleteLanguageModel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteLanguageModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteLanguageModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a transcription job generated by Amazon Transcribe Medical and any related information.
     * </p>
     *
     * @param deleteMedicalTranscriptionJobRequest
     * @return Result of the DeleteMedicalTranscriptionJob operation returned by the service.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.DeleteMedicalTranscriptionJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteMedicalTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteMedicalTranscriptionJobResponse deleteMedicalTranscriptionJob(
            DeleteMedicalTranscriptionJobRequest deleteMedicalTranscriptionJobRequest) throws LimitExceededException,
            BadRequestException, InternalFailureException, AwsServiceException, SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteMedicalTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteMedicalTranscriptionJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteMedicalTranscriptionJobRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMedicalTranscriptionJob");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteMedicalTranscriptionJobRequest, DeleteMedicalTranscriptionJobResponse>()
                            .withOperationName("DeleteMedicalTranscriptionJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteMedicalTranscriptionJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteMedicalTranscriptionJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a vocabulary from Amazon Transcribe Medical.
     * </p>
     *
     * @param deleteMedicalVocabularyRequest
     * @return Result of the DeleteMedicalVocabulary operation returned by the service.
     * @throws NotFoundException
     *         We can't find the requested resource. Check the name and try your request again.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.DeleteMedicalVocabulary
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteMedicalVocabulary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteMedicalVocabularyResponse deleteMedicalVocabulary(DeleteMedicalVocabularyRequest deleteMedicalVocabularyRequest)
            throws NotFoundException, LimitExceededException, BadRequestException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteMedicalVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteMedicalVocabularyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMedicalVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMedicalVocabulary");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteMedicalVocabularyRequest, DeleteMedicalVocabularyResponse>()
                            .withOperationName("DeleteMedicalVocabulary").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteMedicalVocabularyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteMedicalVocabularyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a previously submitted transcription job along with any other generated results such as the
     * transcription, models, and so on.
     * </p>
     *
     * @param deleteTranscriptionJobRequest
     * @return Result of the DeleteTranscriptionJob operation returned by the service.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.DeleteTranscriptionJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteTranscriptionJobResponse deleteTranscriptionJob(DeleteTranscriptionJobRequest deleteTranscriptionJobRequest)
            throws LimitExceededException, BadRequestException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteTranscriptionJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTranscriptionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTranscriptionJob");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteTranscriptionJobRequest, DeleteTranscriptionJobResponse>()
                            .withOperationName("DeleteTranscriptionJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteTranscriptionJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteTranscriptionJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a vocabulary from Amazon Transcribe.
     * </p>
     *
     * @param deleteVocabularyRequest
     * @return Result of the DeleteVocabulary operation returned by the service.
     * @throws NotFoundException
     *         We can't find the requested resource. Check the name and try your request again.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.DeleteVocabulary
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteVocabulary" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteVocabularyResponse deleteVocabulary(DeleteVocabularyRequest deleteVocabularyRequest) throws NotFoundException,
            LimitExceededException, BadRequestException, InternalFailureException, AwsServiceException, SdkClientException,
            TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteVocabularyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVocabulary");

            return clientHandler.execute(new ClientExecutionParams<DeleteVocabularyRequest, DeleteVocabularyResponse>()
                    .withOperationName("DeleteVocabulary").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteVocabularyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteVocabularyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a vocabulary filter.
     * </p>
     *
     * @param deleteVocabularyFilterRequest
     * @return Result of the DeleteVocabularyFilter operation returned by the service.
     * @throws NotFoundException
     *         We can't find the requested resource. Check the name and try your request again.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.DeleteVocabularyFilter
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteVocabularyFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteVocabularyFilterResponse deleteVocabularyFilter(DeleteVocabularyFilterRequest deleteVocabularyFilterRequest)
            throws NotFoundException, LimitExceededException, BadRequestException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteVocabularyFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteVocabularyFilterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVocabularyFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVocabularyFilter");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteVocabularyFilterRequest, DeleteVocabularyFilterResponse>()
                            .withOperationName("DeleteVocabularyFilter").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteVocabularyFilterRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteVocabularyFilterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a single custom language model. Use this information to see details about the language
     * model in your AWS account. You can also see whether the base language model used to create your custom language
     * model has been updated. If Amazon Transcribe has updated the base model, you can create a new custom language
     * model using the updated base model. If the language model wasn't created, you can use this operation to
     * understand why Amazon Transcribe couldn't create it.
     * </p>
     *
     * @param describeLanguageModelRequest
     * @return Result of the DescribeLanguageModel operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws NotFoundException
     *         We can't find the requested resource. Check the name and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.DescribeLanguageModel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DescribeLanguageModel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeLanguageModelResponse describeLanguageModel(DescribeLanguageModelRequest describeLanguageModelRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, NotFoundException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeLanguageModelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeLanguageModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLanguageModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLanguageModel");

            return clientHandler.execute(new ClientExecutionParams<DescribeLanguageModelRequest, DescribeLanguageModelResponse>()
                    .withOperationName("DescribeLanguageModel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeLanguageModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeLanguageModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a transcription job from Amazon Transcribe Medical. To see the status of the job, check
     * the <code>TranscriptionJobStatus</code> field. If the status is <code>COMPLETED</code>, the job is finished. You
     * find the results of the completed job in the <code>TranscriptFileUri</code> field.
     * </p>
     *
     * @param getMedicalTranscriptionJobRequest
     * @return Result of the GetMedicalTranscriptionJob operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws NotFoundException
     *         We can't find the requested resource. Check the name and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.GetMedicalTranscriptionJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetMedicalTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetMedicalTranscriptionJobResponse getMedicalTranscriptionJob(
            GetMedicalTranscriptionJobRequest getMedicalTranscriptionJobRequest) throws BadRequestException,
            LimitExceededException, InternalFailureException, NotFoundException, AwsServiceException, SdkClientException,
            TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMedicalTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetMedicalTranscriptionJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMedicalTranscriptionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMedicalTranscriptionJob");

            return clientHandler
                    .execute(new ClientExecutionParams<GetMedicalTranscriptionJobRequest, GetMedicalTranscriptionJobResponse>()
                            .withOperationName("GetMedicalTranscriptionJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getMedicalTranscriptionJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetMedicalTranscriptionJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about a medical vocabulary.
     * </p>
     *
     * @param getMedicalVocabularyRequest
     * @return Result of the GetMedicalVocabulary operation returned by the service.
     * @throws NotFoundException
     *         We can't find the requested resource. Check the name and try your request again.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.GetMedicalVocabulary
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetMedicalVocabulary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetMedicalVocabularyResponse getMedicalVocabulary(GetMedicalVocabularyRequest getMedicalVocabularyRequest)
            throws NotFoundException, LimitExceededException, InternalFailureException, BadRequestException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMedicalVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetMedicalVocabularyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMedicalVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMedicalVocabulary");

            return clientHandler.execute(new ClientExecutionParams<GetMedicalVocabularyRequest, GetMedicalVocabularyResponse>()
                    .withOperationName("GetMedicalVocabulary").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getMedicalVocabularyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMedicalVocabularyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a transcription job. To see the status of the job, check the
     * <code>TranscriptionJobStatus</code> field. If the status is <code>COMPLETED</code>, the job is finished and you
     * can find the results at the location specified in the <code>TranscriptFileUri</code> field. If you enable content
     * redaction, the redacted transcript appears in <code>RedactedTranscriptFileUri</code>.
     * </p>
     *
     * @param getTranscriptionJobRequest
     * @return Result of the GetTranscriptionJob operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws NotFoundException
     *         We can't find the requested resource. Check the name and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.GetTranscriptionJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetTranscriptionJob" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetTranscriptionJobResponse getTranscriptionJob(GetTranscriptionJobRequest getTranscriptionJobRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, NotFoundException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetTranscriptionJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTranscriptionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTranscriptionJob");

            return clientHandler.execute(new ClientExecutionParams<GetTranscriptionJobRequest, GetTranscriptionJobResponse>()
                    .withOperationName("GetTranscriptionJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getTranscriptionJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTranscriptionJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a vocabulary.
     * </p>
     *
     * @param getVocabularyRequest
     * @return Result of the GetVocabulary operation returned by the service.
     * @throws NotFoundException
     *         We can't find the requested resource. Check the name and try your request again.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.GetVocabulary
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetVocabulary" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetVocabularyResponse getVocabulary(GetVocabularyRequest getVocabularyRequest) throws NotFoundException,
            LimitExceededException, InternalFailureException, BadRequestException, AwsServiceException, SdkClientException,
            TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetVocabularyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetVocabulary");

            return clientHandler.execute(new ClientExecutionParams<GetVocabularyRequest, GetVocabularyResponse>()
                    .withOperationName("GetVocabulary").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getVocabularyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetVocabularyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a vocabulary filter.
     * </p>
     *
     * @param getVocabularyFilterRequest
     * @return Result of the GetVocabularyFilter operation returned by the service.
     * @throws NotFoundException
     *         We can't find the requested resource. Check the name and try your request again.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.GetVocabularyFilter
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetVocabularyFilter" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetVocabularyFilterResponse getVocabularyFilter(GetVocabularyFilterRequest getVocabularyFilterRequest)
            throws NotFoundException, LimitExceededException, InternalFailureException, BadRequestException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetVocabularyFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetVocabularyFilterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getVocabularyFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetVocabularyFilter");

            return clientHandler.execute(new ClientExecutionParams<GetVocabularyFilterRequest, GetVocabularyFilterResponse>()
                    .withOperationName("GetVocabularyFilter").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getVocabularyFilterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetVocabularyFilterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides more information about the custom language models you've created. You can use the information in this
     * list to find a specific custom language model. You can then use the operation to get more information about it.
     * </p>
     *
     * @param listLanguageModelsRequest
     * @return Result of the ListLanguageModels operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.ListLanguageModels
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListLanguageModels" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListLanguageModelsResponse listLanguageModels(ListLanguageModelsRequest listLanguageModelsRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListLanguageModelsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListLanguageModelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLanguageModelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLanguageModels");

            return clientHandler.execute(new ClientExecutionParams<ListLanguageModelsRequest, ListLanguageModelsResponse>()
                    .withOperationName("ListLanguageModels").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listLanguageModelsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListLanguageModelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides more information about the custom language models you've created. You can use the information in this
     * list to find a specific custom language model. You can then use the operation to get more information about it.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listLanguageModels(software.amazon.awssdk.services.transcribe.model.ListLanguageModelsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListLanguageModelsIterable responses = client.listLanguageModelsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.transcribe.paginators.ListLanguageModelsIterable responses = client
     *             .listLanguageModelsPaginator(request);
     *     for (software.amazon.awssdk.services.transcribe.model.ListLanguageModelsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListLanguageModelsIterable responses = client.listLanguageModelsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLanguageModels(software.amazon.awssdk.services.transcribe.model.ListLanguageModelsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listLanguageModelsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.ListLanguageModels
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListLanguageModels" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListLanguageModelsIterable listLanguageModelsPaginator(ListLanguageModelsRequest listLanguageModelsRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        return new ListLanguageModelsIterable(this, applyPaginatorUserAgent(listLanguageModelsRequest));
    }

    /**
     * <p>
     * Lists medical transcription jobs with a specified status or substring that matches their names.
     * </p>
     *
     * @param listMedicalTranscriptionJobsRequest
     * @return Result of the ListMedicalTranscriptionJobs operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.ListMedicalTranscriptionJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListMedicalTranscriptionJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListMedicalTranscriptionJobsResponse listMedicalTranscriptionJobs(
            ListMedicalTranscriptionJobsRequest listMedicalTranscriptionJobsRequest) throws BadRequestException,
            LimitExceededException, InternalFailureException, AwsServiceException, SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListMedicalTranscriptionJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListMedicalTranscriptionJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMedicalTranscriptionJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMedicalTranscriptionJobs");

            return clientHandler
                    .execute(new ClientExecutionParams<ListMedicalTranscriptionJobsRequest, ListMedicalTranscriptionJobsResponse>()
                            .withOperationName("ListMedicalTranscriptionJobs").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listMedicalTranscriptionJobsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListMedicalTranscriptionJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists medical transcription jobs with a specified status or substring that matches their names.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMedicalTranscriptionJobs(software.amazon.awssdk.services.transcribe.model.ListMedicalTranscriptionJobsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListMedicalTranscriptionJobsIterable responses = client.listMedicalTranscriptionJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.transcribe.paginators.ListMedicalTranscriptionJobsIterable responses = client
     *             .listMedicalTranscriptionJobsPaginator(request);
     *     for (software.amazon.awssdk.services.transcribe.model.ListMedicalTranscriptionJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListMedicalTranscriptionJobsIterable responses = client.listMedicalTranscriptionJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMedicalTranscriptionJobs(software.amazon.awssdk.services.transcribe.model.ListMedicalTranscriptionJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listMedicalTranscriptionJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.ListMedicalTranscriptionJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListMedicalTranscriptionJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListMedicalTranscriptionJobsIterable listMedicalTranscriptionJobsPaginator(
            ListMedicalTranscriptionJobsRequest listMedicalTranscriptionJobsRequest) throws BadRequestException,
            LimitExceededException, InternalFailureException, AwsServiceException, SdkClientException, TranscribeException {
        return new ListMedicalTranscriptionJobsIterable(this, applyPaginatorUserAgent(listMedicalTranscriptionJobsRequest));
    }

    /**
     * <p>
     * Returns a list of vocabularies that match the specified criteria. If you don't enter a value in any of the
     * request parameters, returns the entire list of vocabularies.
     * </p>
     *
     * @param listMedicalVocabulariesRequest
     * @return Result of the ListMedicalVocabularies operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.ListMedicalVocabularies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListMedicalVocabularies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListMedicalVocabulariesResponse listMedicalVocabularies(ListMedicalVocabulariesRequest listMedicalVocabulariesRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListMedicalVocabulariesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListMedicalVocabulariesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMedicalVocabulariesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMedicalVocabularies");

            return clientHandler
                    .execute(new ClientExecutionParams<ListMedicalVocabulariesRequest, ListMedicalVocabulariesResponse>()
                            .withOperationName("ListMedicalVocabularies").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listMedicalVocabulariesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListMedicalVocabulariesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of vocabularies that match the specified criteria. If you don't enter a value in any of the
     * request parameters, returns the entire list of vocabularies.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMedicalVocabularies(software.amazon.awssdk.services.transcribe.model.ListMedicalVocabulariesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListMedicalVocabulariesIterable responses = client.listMedicalVocabulariesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.transcribe.paginators.ListMedicalVocabulariesIterable responses = client
     *             .listMedicalVocabulariesPaginator(request);
     *     for (software.amazon.awssdk.services.transcribe.model.ListMedicalVocabulariesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListMedicalVocabulariesIterable responses = client.listMedicalVocabulariesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMedicalVocabularies(software.amazon.awssdk.services.transcribe.model.ListMedicalVocabulariesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listMedicalVocabulariesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.ListMedicalVocabularies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListMedicalVocabularies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListMedicalVocabulariesIterable listMedicalVocabulariesPaginator(
            ListMedicalVocabulariesRequest listMedicalVocabulariesRequest) throws BadRequestException, LimitExceededException,
            InternalFailureException, AwsServiceException, SdkClientException, TranscribeException {
        return new ListMedicalVocabulariesIterable(this, applyPaginatorUserAgent(listMedicalVocabulariesRequest));
    }

    /**
     * <p>
     * Lists transcription jobs with the specified status.
     * </p>
     *
     * @param listTranscriptionJobsRequest
     * @return Result of the ListTranscriptionJobs operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.ListTranscriptionJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListTranscriptionJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTranscriptionJobsResponse listTranscriptionJobs(ListTranscriptionJobsRequest listTranscriptionJobsRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTranscriptionJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTranscriptionJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTranscriptionJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTranscriptionJobs");

            return clientHandler.execute(new ClientExecutionParams<ListTranscriptionJobsRequest, ListTranscriptionJobsResponse>()
                    .withOperationName("ListTranscriptionJobs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTranscriptionJobsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTranscriptionJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists transcription jobs with the specified status.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTranscriptionJobs(software.amazon.awssdk.services.transcribe.model.ListTranscriptionJobsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListTranscriptionJobsIterable responses = client.listTranscriptionJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.transcribe.paginators.ListTranscriptionJobsIterable responses = client
     *             .listTranscriptionJobsPaginator(request);
     *     for (software.amazon.awssdk.services.transcribe.model.ListTranscriptionJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListTranscriptionJobsIterable responses = client.listTranscriptionJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTranscriptionJobs(software.amazon.awssdk.services.transcribe.model.ListTranscriptionJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTranscriptionJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.ListTranscriptionJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListTranscriptionJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTranscriptionJobsIterable listTranscriptionJobsPaginator(ListTranscriptionJobsRequest listTranscriptionJobsRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        return new ListTranscriptionJobsIterable(this, applyPaginatorUserAgent(listTranscriptionJobsRequest));
    }

    /**
     * <p>
     * Returns a list of vocabularies that match the specified criteria. If no criteria are specified, returns the
     * entire list of vocabularies.
     * </p>
     *
     * @param listVocabulariesRequest
     * @return Result of the ListVocabularies operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.ListVocabularies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListVocabularies" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListVocabulariesResponse listVocabularies(ListVocabulariesRequest listVocabulariesRequest) throws BadRequestException,
            LimitExceededException, InternalFailureException, AwsServiceException, SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListVocabulariesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListVocabulariesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVocabulariesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVocabularies");

            return clientHandler.execute(new ClientExecutionParams<ListVocabulariesRequest, ListVocabulariesResponse>()
                    .withOperationName("ListVocabularies").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listVocabulariesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListVocabulariesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of vocabularies that match the specified criteria. If no criteria are specified, returns the
     * entire list of vocabularies.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listVocabularies(software.amazon.awssdk.services.transcribe.model.ListVocabulariesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListVocabulariesIterable responses = client.listVocabulariesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.transcribe.paginators.ListVocabulariesIterable responses = client
     *             .listVocabulariesPaginator(request);
     *     for (software.amazon.awssdk.services.transcribe.model.ListVocabulariesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListVocabulariesIterable responses = client.listVocabulariesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listVocabularies(software.amazon.awssdk.services.transcribe.model.ListVocabulariesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listVocabulariesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.ListVocabularies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListVocabularies" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListVocabulariesIterable listVocabulariesPaginator(ListVocabulariesRequest listVocabulariesRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        return new ListVocabulariesIterable(this, applyPaginatorUserAgent(listVocabulariesRequest));
    }

    /**
     * <p>
     * Gets information about vocabulary filters.
     * </p>
     *
     * @param listVocabularyFiltersRequest
     * @return Result of the ListVocabularyFilters operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.ListVocabularyFilters
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListVocabularyFilters"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListVocabularyFiltersResponse listVocabularyFilters(ListVocabularyFiltersRequest listVocabularyFiltersRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListVocabularyFiltersResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListVocabularyFiltersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVocabularyFiltersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVocabularyFilters");

            return clientHandler.execute(new ClientExecutionParams<ListVocabularyFiltersRequest, ListVocabularyFiltersResponse>()
                    .withOperationName("ListVocabularyFilters").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listVocabularyFiltersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListVocabularyFiltersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about vocabulary filters.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listVocabularyFilters(software.amazon.awssdk.services.transcribe.model.ListVocabularyFiltersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListVocabularyFiltersIterable responses = client.listVocabularyFiltersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.transcribe.paginators.ListVocabularyFiltersIterable responses = client
     *             .listVocabularyFiltersPaginator(request);
     *     for (software.amazon.awssdk.services.transcribe.model.ListVocabularyFiltersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListVocabularyFiltersIterable responses = client.listVocabularyFiltersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listVocabularyFilters(software.amazon.awssdk.services.transcribe.model.ListVocabularyFiltersRequest)}
     * operation.</b>
     * </p>
     *
     * @param listVocabularyFiltersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.ListVocabularyFilters
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListVocabularyFilters"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListVocabularyFiltersIterable listVocabularyFiltersPaginator(ListVocabularyFiltersRequest listVocabularyFiltersRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, AwsServiceException,
            SdkClientException, TranscribeException {
        return new ListVocabularyFiltersIterable(this, applyPaginatorUserAgent(listVocabularyFiltersRequest));
    }

    /**
     * <p>
     * Starts a batch job to transcribe medical speech to text.
     * </p>
     *
     * @param startMedicalTranscriptionJobRequest
     * @return Result of the StartMedicalTranscriptionJob operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws ConflictException
     *         There is already a resource with that name.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.StartMedicalTranscriptionJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/StartMedicalTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartMedicalTranscriptionJobResponse startMedicalTranscriptionJob(
            StartMedicalTranscriptionJobRequest startMedicalTranscriptionJobRequest) throws BadRequestException,
            LimitExceededException, InternalFailureException, ConflictException, AwsServiceException, SdkClientException,
            TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartMedicalTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartMedicalTranscriptionJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startMedicalTranscriptionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMedicalTranscriptionJob");

            return clientHandler
                    .execute(new ClientExecutionParams<StartMedicalTranscriptionJobRequest, StartMedicalTranscriptionJobResponse>()
                            .withOperationName("StartMedicalTranscriptionJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(startMedicalTranscriptionJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartMedicalTranscriptionJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts an asynchronous job to transcribe speech to text.
     * </p>
     *
     * @param startTranscriptionJobRequest
     * @return Result of the StartTranscriptionJob operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws ConflictException
     *         There is already a resource with that name.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.StartTranscriptionJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/StartTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartTranscriptionJobResponse startTranscriptionJob(StartTranscriptionJobRequest startTranscriptionJobRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, ConflictException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartTranscriptionJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startTranscriptionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartTranscriptionJob");

            return clientHandler.execute(new ClientExecutionParams<StartTranscriptionJobRequest, StartTranscriptionJobResponse>()
                    .withOperationName("StartTranscriptionJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startTranscriptionJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartTranscriptionJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a vocabulary with new values that you provide in a different text file from the one you used to create
     * the vocabulary. The <code>UpdateMedicalVocabulary</code> operation overwrites all of the existing information
     * with the values that you provide in the request.
     * </p>
     *
     * @param updateMedicalVocabularyRequest
     * @return Result of the UpdateMedicalVocabulary operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws NotFoundException
     *         We can't find the requested resource. Check the name and try your request again.
     * @throws ConflictException
     *         There is already a resource with that name.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.UpdateMedicalVocabulary
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/UpdateMedicalVocabulary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateMedicalVocabularyResponse updateMedicalVocabulary(UpdateMedicalVocabularyRequest updateMedicalVocabularyRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, NotFoundException, ConflictException,
            AwsServiceException, SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateMedicalVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateMedicalVocabularyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMedicalVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMedicalVocabulary");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateMedicalVocabularyRequest, UpdateMedicalVocabularyResponse>()
                            .withOperationName("UpdateMedicalVocabulary").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateMedicalVocabularyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateMedicalVocabularyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing vocabulary with new values. The <code>UpdateVocabulary</code> operation overwrites all of the
     * existing information with the values that you provide in the request.
     * </p>
     *
     * @param updateVocabularyRequest
     * @return Result of the UpdateVocabulary operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws NotFoundException
     *         We can't find the requested resource. Check the name and try your request again.
     * @throws ConflictException
     *         There is already a resource with that name.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.UpdateVocabulary
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/UpdateVocabulary" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateVocabularyResponse updateVocabulary(UpdateVocabularyRequest updateVocabularyRequest) throws BadRequestException,
            LimitExceededException, InternalFailureException, NotFoundException, ConflictException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateVocabularyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVocabulary");

            return clientHandler.execute(new ClientExecutionParams<UpdateVocabularyRequest, UpdateVocabularyResponse>()
                    .withOperationName("UpdateVocabulary").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateVocabularyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateVocabularyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a vocabulary filter with a new list of filtered words.
     * </p>
     *
     * @param updateVocabularyFilterRequest
     * @return Result of the UpdateVocabularyFilter operation returned by the service.
     * @throws BadRequestException
     *         Your request didn't pass one or more validation tests. For example, if the entity that you're trying to
     *         delete doesn't exist or if it is in a non-terminal state (for example, it's "in progress"). See the
     *         exception <code>Message</code> field for more information.
     * @throws LimitExceededException
     *         Either you have sent too many requests or your input file is too long. Wait before you resend your
     *         request, or use a smaller file and resend the request.
     * @throws InternalFailureException
     *         There was an internal error. Check the error message and try your request again.
     * @throws NotFoundException
     *         We can't find the requested resource. Check the name and try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranscribeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranscribeClient.UpdateVocabularyFilter
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/UpdateVocabularyFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateVocabularyFilterResponse updateVocabularyFilter(UpdateVocabularyFilterRequest updateVocabularyFilterRequest)
            throws BadRequestException, LimitExceededException, InternalFailureException, NotFoundException, AwsServiceException,
            SdkClientException, TranscribeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateVocabularyFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateVocabularyFilterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVocabularyFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVocabularyFilter");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateVocabularyFilterRequest, UpdateVocabularyFilterResponse>()
                            .withOperationName("UpdateVocabularyFilter").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateVocabularyFilterRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateVocabularyFilterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(TranscribeException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends TranscribeRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
