/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribe.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an asynchronous transcription job that was created with the <code>StartTranscriptionJob</code> operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TranscriptionJob implements SdkPojo, Serializable,
        ToCopyableBuilder<TranscriptionJob.Builder, TranscriptionJob> {
    private static final SdkField<String> TRANSCRIPTION_JOB_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TranscriptionJobName").getter(getter(TranscriptionJob::transcriptionJobName))
            .setter(setter(Builder::transcriptionJobName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TranscriptionJobName").build())
            .build();

    private static final SdkField<String> TRANSCRIPTION_JOB_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TranscriptionJobStatus").getter(getter(TranscriptionJob::transcriptionJobStatusAsString))
            .setter(setter(Builder::transcriptionJobStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TranscriptionJobStatus").build())
            .build();

    private static final SdkField<String> LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LanguageCode").getter(getter(TranscriptionJob::languageCodeAsString))
            .setter(setter(Builder::languageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageCode").build()).build();

    private static final SdkField<Integer> MEDIA_SAMPLE_RATE_HERTZ_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MediaSampleRateHertz").getter(getter(TranscriptionJob::mediaSampleRateHertz))
            .setter(setter(Builder::mediaSampleRateHertz))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MediaSampleRateHertz").build())
            .build();

    private static final SdkField<String> MEDIA_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MediaFormat").getter(getter(TranscriptionJob::mediaFormatAsString)).setter(setter(Builder::mediaFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MediaFormat").build()).build();

    private static final SdkField<Media> MEDIA_FIELD = SdkField.<Media> builder(MarshallingType.SDK_POJO).memberName("Media")
            .getter(getter(TranscriptionJob::media)).setter(setter(Builder::media)).constructor(Media::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Media").build()).build();

    private static final SdkField<Transcript> TRANSCRIPT_FIELD = SdkField.<Transcript> builder(MarshallingType.SDK_POJO)
            .memberName("Transcript").getter(getter(TranscriptionJob::transcript)).setter(setter(Builder::transcript))
            .constructor(Transcript::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Transcript").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime").getter(getter(TranscriptionJob::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(TranscriptionJob::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<Instant> COMPLETION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CompletionTime").getter(getter(TranscriptionJob::completionTime))
            .setter(setter(Builder::completionTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletionTime").build()).build();

    private static final SdkField<String> FAILURE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FailureReason").getter(getter(TranscriptionJob::failureReason)).setter(setter(Builder::failureReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureReason").build()).build();

    private static final SdkField<Settings> SETTINGS_FIELD = SdkField.<Settings> builder(MarshallingType.SDK_POJO)
            .memberName("Settings").getter(getter(TranscriptionJob::settings)).setter(setter(Builder::settings))
            .constructor(Settings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Settings").build()).build();

    private static final SdkField<ModelSettings> MODEL_SETTINGS_FIELD = SdkField
            .<ModelSettings> builder(MarshallingType.SDK_POJO).memberName("ModelSettings")
            .getter(getter(TranscriptionJob::modelSettings)).setter(setter(Builder::modelSettings))
            .constructor(ModelSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelSettings").build()).build();

    private static final SdkField<JobExecutionSettings> JOB_EXECUTION_SETTINGS_FIELD = SdkField
            .<JobExecutionSettings> builder(MarshallingType.SDK_POJO).memberName("JobExecutionSettings")
            .getter(getter(TranscriptionJob::jobExecutionSettings)).setter(setter(Builder::jobExecutionSettings))
            .constructor(JobExecutionSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobExecutionSettings").build())
            .build();

    private static final SdkField<ContentRedaction> CONTENT_REDACTION_FIELD = SdkField
            .<ContentRedaction> builder(MarshallingType.SDK_POJO).memberName("ContentRedaction")
            .getter(getter(TranscriptionJob::contentRedaction)).setter(setter(Builder::contentRedaction))
            .constructor(ContentRedaction::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContentRedaction").build()).build();

    private static final SdkField<Boolean> IDENTIFY_LANGUAGE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IdentifyLanguage").getter(getter(TranscriptionJob::identifyLanguage))
            .setter(setter(Builder::identifyLanguage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdentifyLanguage").build()).build();

    private static final SdkField<List<String>> LANGUAGE_OPTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("LanguageOptions")
            .getter(getter(TranscriptionJob::languageOptionsAsStrings))
            .setter(setter(Builder::languageOptionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageOptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Float> IDENTIFIED_LANGUAGE_SCORE_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("IdentifiedLanguageScore").getter(getter(TranscriptionJob::identifiedLanguageScore))
            .setter(setter(Builder::identifiedLanguageScore))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdentifiedLanguageScore").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TRANSCRIPTION_JOB_NAME_FIELD,
            TRANSCRIPTION_JOB_STATUS_FIELD, LANGUAGE_CODE_FIELD, MEDIA_SAMPLE_RATE_HERTZ_FIELD, MEDIA_FORMAT_FIELD, MEDIA_FIELD,
            TRANSCRIPT_FIELD, START_TIME_FIELD, CREATION_TIME_FIELD, COMPLETION_TIME_FIELD, FAILURE_REASON_FIELD, SETTINGS_FIELD,
            MODEL_SETTINGS_FIELD, JOB_EXECUTION_SETTINGS_FIELD, CONTENT_REDACTION_FIELD, IDENTIFY_LANGUAGE_FIELD,
            LANGUAGE_OPTIONS_FIELD, IDENTIFIED_LANGUAGE_SCORE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String transcriptionJobName;

    private final String transcriptionJobStatus;

    private final String languageCode;

    private final Integer mediaSampleRateHertz;

    private final String mediaFormat;

    private final Media media;

    private final Transcript transcript;

    private final Instant startTime;

    private final Instant creationTime;

    private final Instant completionTime;

    private final String failureReason;

    private final Settings settings;

    private final ModelSettings modelSettings;

    private final JobExecutionSettings jobExecutionSettings;

    private final ContentRedaction contentRedaction;

    private final Boolean identifyLanguage;

    private final List<String> languageOptions;

    private final Float identifiedLanguageScore;

    private TranscriptionJob(BuilderImpl builder) {
        this.transcriptionJobName = builder.transcriptionJobName;
        this.transcriptionJobStatus = builder.transcriptionJobStatus;
        this.languageCode = builder.languageCode;
        this.mediaSampleRateHertz = builder.mediaSampleRateHertz;
        this.mediaFormat = builder.mediaFormat;
        this.media = builder.media;
        this.transcript = builder.transcript;
        this.startTime = builder.startTime;
        this.creationTime = builder.creationTime;
        this.completionTime = builder.completionTime;
        this.failureReason = builder.failureReason;
        this.settings = builder.settings;
        this.modelSettings = builder.modelSettings;
        this.jobExecutionSettings = builder.jobExecutionSettings;
        this.contentRedaction = builder.contentRedaction;
        this.identifyLanguage = builder.identifyLanguage;
        this.languageOptions = builder.languageOptions;
        this.identifiedLanguageScore = builder.identifiedLanguageScore;
    }

    /**
     * <p>
     * The name of the transcription job.
     * </p>
     * 
     * @return The name of the transcription job.
     */
    public String transcriptionJobName() {
        return transcriptionJobName;
    }

    /**
     * <p>
     * The status of the transcription job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #transcriptionJobStatus} will return {@link TranscriptionJobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #transcriptionJobStatusAsString}.
     * </p>
     * 
     * @return The status of the transcription job.
     * @see TranscriptionJobStatus
     */
    public TranscriptionJobStatus transcriptionJobStatus() {
        return TranscriptionJobStatus.fromValue(transcriptionJobStatus);
    }

    /**
     * <p>
     * The status of the transcription job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #transcriptionJobStatus} will return {@link TranscriptionJobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #transcriptionJobStatusAsString}.
     * </p>
     * 
     * @return The status of the transcription job.
     * @see TranscriptionJobStatus
     */
    public String transcriptionJobStatusAsString() {
        return transcriptionJobStatus;
    }

    /**
     * <p>
     * The language code for the input speech.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return The language code for the input speech.
     * @see LanguageCode
     */
    public LanguageCode languageCode() {
        return LanguageCode.fromValue(languageCode);
    }

    /**
     * <p>
     * The language code for the input speech.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return The language code for the input speech.
     * @see LanguageCode
     */
    public String languageCodeAsString() {
        return languageCode;
    }

    /**
     * <p>
     * The sample rate, in Hertz, of the audio track in the input media file.
     * </p>
     * 
     * @return The sample rate, in Hertz, of the audio track in the input media file.
     */
    public Integer mediaSampleRateHertz() {
        return mediaSampleRateHertz;
    }

    /**
     * <p>
     * The format of the input media file.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mediaFormat} will
     * return {@link MediaFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #mediaFormatAsString}.
     * </p>
     * 
     * @return The format of the input media file.
     * @see MediaFormat
     */
    public MediaFormat mediaFormat() {
        return MediaFormat.fromValue(mediaFormat);
    }

    /**
     * <p>
     * The format of the input media file.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mediaFormat} will
     * return {@link MediaFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #mediaFormatAsString}.
     * </p>
     * 
     * @return The format of the input media file.
     * @see MediaFormat
     */
    public String mediaFormatAsString() {
        return mediaFormat;
    }

    /**
     * <p>
     * An object that describes the input media for the transcription job.
     * </p>
     * 
     * @return An object that describes the input media for the transcription job.
     */
    public Media media() {
        return media;
    }

    /**
     * <p>
     * An object that describes the output of the transcription job.
     * </p>
     * 
     * @return An object that describes the output of the transcription job.
     */
    public Transcript transcript() {
        return transcript;
    }

    /**
     * <p>
     * A timestamp that shows with the job was started processing.
     * </p>
     * 
     * @return A timestamp that shows with the job was started processing.
     */
    public Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * A timestamp that shows when the job was created.
     * </p>
     * 
     * @return A timestamp that shows when the job was created.
     */
    public Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * A timestamp that shows when the job was completed.
     * </p>
     * 
     * @return A timestamp that shows when the job was completed.
     */
    public Instant completionTime() {
        return completionTime;
    }

    /**
     * <p>
     * If the <code>TranscriptionJobStatus</code> field is <code>FAILED</code>, this field contains information about
     * why the job failed.
     * </p>
     * <p>
     * The <code>FailureReason</code> field can contain one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Unsupported media format</code> - The media format specified in the <code>MediaFormat</code> field of the
     * request isn't valid. See the description of the <code>MediaFormat</code> field for a list of valid values.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>The media format provided does not match the detected media format</code> - The media format of the audio
     * file doesn't match the format specified in the <code>MediaFormat</code> field in the request. Check the media
     * format of your media file and make sure that the two values match.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Invalid sample rate for audio file</code> - The sample rate specified in the
     * <code>MediaSampleRateHertz</code> of the request isn't valid. The sample rate must be between 8000 and 48000
     * Hertz.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>The sample rate provided does not match the detected sample rate</code> - The sample rate in the audio file
     * doesn't match the sample rate specified in the <code>MediaSampleRateHertz</code> field in the request. Check the
     * sample rate of your media file and make sure that the two values match.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Invalid file size: file size too large</code> - The size of your audio file is larger than Amazon
     * Transcribe can process. For more information, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Limits</a> in the <i>Amazon
     * Transcribe Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Invalid number of channels: number of channels too large</code> - Your audio contains more channels than
     * Amazon Transcribe is configured to process. To request additional channels, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws_service_limits.html#limits-amazon-transcribe">Amazon
     * Transcribe Limits</a> in the <i>Amazon Web Services General Reference</i>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return If the <code>TranscriptionJobStatus</code> field is <code>FAILED</code>, this field contains information
     *         about why the job failed.</p>
     *         <p>
     *         The <code>FailureReason</code> field can contain one of the following values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Unsupported media format</code> - The media format specified in the <code>MediaFormat</code> field
     *         of the request isn't valid. See the description of the <code>MediaFormat</code> field for a list of valid
     *         values.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>The media format provided does not match the detected media format</code> - The media format of the
     *         audio file doesn't match the format specified in the <code>MediaFormat</code> field in the request. Check
     *         the media format of your media file and make sure that the two values match.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Invalid sample rate for audio file</code> - The sample rate specified in the
     *         <code>MediaSampleRateHertz</code> of the request isn't valid. The sample rate must be between 8000 and
     *         48000 Hertz.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>The sample rate provided does not match the detected sample rate</code> - The sample rate in the
     *         audio file doesn't match the sample rate specified in the <code>MediaSampleRateHertz</code> field in the
     *         request. Check the sample rate of your media file and make sure that the two values match.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Invalid file size: file size too large</code> - The size of your audio file is larger than Amazon
     *         Transcribe can process. For more information, see <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Limits</a> in the
     *         <i>Amazon Transcribe Developer Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Invalid number of channels: number of channels too large</code> - Your audio contains more channels
     *         than Amazon Transcribe is configured to process. To request additional channels, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws_service_limits.html#limits-amazon-transcribe"
     *         >Amazon Transcribe Limits</a> in the <i>Amazon Web Services General Reference</i>.
     *         </p>
     *         </li>
     */
    public String failureReason() {
        return failureReason;
    }

    /**
     * <p>
     * Optional settings for the transcription job. Use these settings to turn on speaker recognition, to set the
     * maximum number of speakers that should be identified and to specify a custom vocabulary to use when processing
     * the transcription job.
     * </p>
     * 
     * @return Optional settings for the transcription job. Use these settings to turn on speaker recognition, to set
     *         the maximum number of speakers that should be identified and to specify a custom vocabulary to use when
     *         processing the transcription job.
     */
    public Settings settings() {
        return settings;
    }

    /**
     * <p>
     * An object containing the details of your custom language model.
     * </p>
     * 
     * @return An object containing the details of your custom language model.
     */
    public ModelSettings modelSettings() {
        return modelSettings;
    }

    /**
     * <p>
     * Provides information about how a transcription job is executed.
     * </p>
     * 
     * @return Provides information about how a transcription job is executed.
     */
    public JobExecutionSettings jobExecutionSettings() {
        return jobExecutionSettings;
    }

    /**
     * <p>
     * An object that describes content redaction settings for the transcription job.
     * </p>
     * 
     * @return An object that describes content redaction settings for the transcription job.
     */
    public ContentRedaction contentRedaction() {
        return contentRedaction;
    }

    /**
     * <p>
     * A value that shows if automatic language identification was enabled for a transcription job.
     * </p>
     * 
     * @return A value that shows if automatic language identification was enabled for a transcription job.
     */
    public Boolean identifyLanguage() {
        return identifyLanguage;
    }

    /**
     * <p>
     * An object that shows the optional array of languages inputted for transcription jobs with automatic language
     * identification enabled.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasLanguageOptions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An object that shows the optional array of languages inputted for transcription jobs with automatic
     *         language identification enabled.
     */
    public List<LanguageCode> languageOptions() {
        return LanguageOptionsCopier.copyStringToEnum(languageOptions);
    }

    /**
     * Returns true if the LanguageOptions property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasLanguageOptions() {
        return languageOptions != null && !(languageOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An object that shows the optional array of languages inputted for transcription jobs with automatic language
     * identification enabled.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasLanguageOptions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An object that shows the optional array of languages inputted for transcription jobs with automatic
     *         language identification enabled.
     */
    public List<String> languageOptionsAsStrings() {
        return languageOptions;
    }

    /**
     * <p>
     * A value between zero and one that Amazon Transcribe assigned to the language that it identified in the source
     * audio. Larger values indicate that Amazon Transcribe has higher confidence in the language it identified.
     * </p>
     * 
     * @return A value between zero and one that Amazon Transcribe assigned to the language that it identified in the
     *         source audio. Larger values indicate that Amazon Transcribe has higher confidence in the language it
     *         identified.
     */
    public Float identifiedLanguageScore() {
        return identifiedLanguageScore;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(transcriptionJobName());
        hashCode = 31 * hashCode + Objects.hashCode(transcriptionJobStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(languageCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(mediaSampleRateHertz());
        hashCode = 31 * hashCode + Objects.hashCode(mediaFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(media());
        hashCode = 31 * hashCode + Objects.hashCode(transcript());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(completionTime());
        hashCode = 31 * hashCode + Objects.hashCode(failureReason());
        hashCode = 31 * hashCode + Objects.hashCode(settings());
        hashCode = 31 * hashCode + Objects.hashCode(modelSettings());
        hashCode = 31 * hashCode + Objects.hashCode(jobExecutionSettings());
        hashCode = 31 * hashCode + Objects.hashCode(contentRedaction());
        hashCode = 31 * hashCode + Objects.hashCode(identifyLanguage());
        hashCode = 31 * hashCode + Objects.hashCode(hasLanguageOptions() ? languageOptionsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(identifiedLanguageScore());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TranscriptionJob)) {
            return false;
        }
        TranscriptionJob other = (TranscriptionJob) obj;
        return Objects.equals(transcriptionJobName(), other.transcriptionJobName())
                && Objects.equals(transcriptionJobStatusAsString(), other.transcriptionJobStatusAsString())
                && Objects.equals(languageCodeAsString(), other.languageCodeAsString())
                && Objects.equals(mediaSampleRateHertz(), other.mediaSampleRateHertz())
                && Objects.equals(mediaFormatAsString(), other.mediaFormatAsString()) && Objects.equals(media(), other.media())
                && Objects.equals(transcript(), other.transcript()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(completionTime(), other.completionTime())
                && Objects.equals(failureReason(), other.failureReason()) && Objects.equals(settings(), other.settings())
                && Objects.equals(modelSettings(), other.modelSettings())
                && Objects.equals(jobExecutionSettings(), other.jobExecutionSettings())
                && Objects.equals(contentRedaction(), other.contentRedaction())
                && Objects.equals(identifyLanguage(), other.identifyLanguage())
                && hasLanguageOptions() == other.hasLanguageOptions()
                && Objects.equals(languageOptionsAsStrings(), other.languageOptionsAsStrings())
                && Objects.equals(identifiedLanguageScore(), other.identifiedLanguageScore());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("TranscriptionJob").add("TranscriptionJobName", transcriptionJobName())
                .add("TranscriptionJobStatus", transcriptionJobStatusAsString()).add("LanguageCode", languageCodeAsString())
                .add("MediaSampleRateHertz", mediaSampleRateHertz()).add("MediaFormat", mediaFormatAsString())
                .add("Media", media()).add("Transcript", transcript()).add("StartTime", startTime())
                .add("CreationTime", creationTime()).add("CompletionTime", completionTime())
                .add("FailureReason", failureReason()).add("Settings", settings()).add("ModelSettings", modelSettings())
                .add("JobExecutionSettings", jobExecutionSettings()).add("ContentRedaction", contentRedaction())
                .add("IdentifyLanguage", identifyLanguage())
                .add("LanguageOptions", hasLanguageOptions() ? languageOptionsAsStrings() : null)
                .add("IdentifiedLanguageScore", identifiedLanguageScore()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TranscriptionJobName":
            return Optional.ofNullable(clazz.cast(transcriptionJobName()));
        case "TranscriptionJobStatus":
            return Optional.ofNullable(clazz.cast(transcriptionJobStatusAsString()));
        case "LanguageCode":
            return Optional.ofNullable(clazz.cast(languageCodeAsString()));
        case "MediaSampleRateHertz":
            return Optional.ofNullable(clazz.cast(mediaSampleRateHertz()));
        case "MediaFormat":
            return Optional.ofNullable(clazz.cast(mediaFormatAsString()));
        case "Media":
            return Optional.ofNullable(clazz.cast(media()));
        case "Transcript":
            return Optional.ofNullable(clazz.cast(transcript()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "CompletionTime":
            return Optional.ofNullable(clazz.cast(completionTime()));
        case "FailureReason":
            return Optional.ofNullable(clazz.cast(failureReason()));
        case "Settings":
            return Optional.ofNullable(clazz.cast(settings()));
        case "ModelSettings":
            return Optional.ofNullable(clazz.cast(modelSettings()));
        case "JobExecutionSettings":
            return Optional.ofNullable(clazz.cast(jobExecutionSettings()));
        case "ContentRedaction":
            return Optional.ofNullable(clazz.cast(contentRedaction()));
        case "IdentifyLanguage":
            return Optional.ofNullable(clazz.cast(identifyLanguage()));
        case "LanguageOptions":
            return Optional.ofNullable(clazz.cast(languageOptionsAsStrings()));
        case "IdentifiedLanguageScore":
            return Optional.ofNullable(clazz.cast(identifiedLanguageScore()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TranscriptionJob, T> g) {
        return obj -> g.apply((TranscriptionJob) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TranscriptionJob> {
        /**
         * <p>
         * The name of the transcription job.
         * </p>
         * 
         * @param transcriptionJobName
         *        The name of the transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transcriptionJobName(String transcriptionJobName);

        /**
         * <p>
         * The status of the transcription job.
         * </p>
         * 
         * @param transcriptionJobStatus
         *        The status of the transcription job.
         * @see TranscriptionJobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscriptionJobStatus
         */
        Builder transcriptionJobStatus(String transcriptionJobStatus);

        /**
         * <p>
         * The status of the transcription job.
         * </p>
         * 
         * @param transcriptionJobStatus
         *        The status of the transcription job.
         * @see TranscriptionJobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscriptionJobStatus
         */
        Builder transcriptionJobStatus(TranscriptionJobStatus transcriptionJobStatus);

        /**
         * <p>
         * The language code for the input speech.
         * </p>
         * 
         * @param languageCode
         *        The language code for the input speech.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(String languageCode);

        /**
         * <p>
         * The language code for the input speech.
         * </p>
         * 
         * @param languageCode
         *        The language code for the input speech.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(LanguageCode languageCode);

        /**
         * <p>
         * The sample rate, in Hertz, of the audio track in the input media file.
         * </p>
         * 
         * @param mediaSampleRateHertz
         *        The sample rate, in Hertz, of the audio track in the input media file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mediaSampleRateHertz(Integer mediaSampleRateHertz);

        /**
         * <p>
         * The format of the input media file.
         * </p>
         * 
         * @param mediaFormat
         *        The format of the input media file.
         * @see MediaFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MediaFormat
         */
        Builder mediaFormat(String mediaFormat);

        /**
         * <p>
         * The format of the input media file.
         * </p>
         * 
         * @param mediaFormat
         *        The format of the input media file.
         * @see MediaFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MediaFormat
         */
        Builder mediaFormat(MediaFormat mediaFormat);

        /**
         * <p>
         * An object that describes the input media for the transcription job.
         * </p>
         * 
         * @param media
         *        An object that describes the input media for the transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder media(Media media);

        /**
         * <p>
         * An object that describes the input media for the transcription job.
         * </p>
         * This is a convenience that creates an instance of the {@link Media.Builder} avoiding the need to create one
         * manually via {@link Media#builder()}.
         *
         * When the {@link Consumer} completes, {@link Media.Builder#build()} is called immediately and its result is
         * passed to {@link #media(Media)}.
         * 
         * @param media
         *        a consumer that will call methods on {@link Media.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #media(Media)
         */
        default Builder media(Consumer<Media.Builder> media) {
            return media(Media.builder().applyMutation(media).build());
        }

        /**
         * <p>
         * An object that describes the output of the transcription job.
         * </p>
         * 
         * @param transcript
         *        An object that describes the output of the transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transcript(Transcript transcript);

        /**
         * <p>
         * An object that describes the output of the transcription job.
         * </p>
         * This is a convenience that creates an instance of the {@link Transcript.Builder} avoiding the need to create
         * one manually via {@link Transcript#builder()}.
         *
         * When the {@link Consumer} completes, {@link Transcript.Builder#build()} is called immediately and its result
         * is passed to {@link #transcript(Transcript)}.
         * 
         * @param transcript
         *        a consumer that will call methods on {@link Transcript.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #transcript(Transcript)
         */
        default Builder transcript(Consumer<Transcript.Builder> transcript) {
            return transcript(Transcript.builder().applyMutation(transcript).build());
        }

        /**
         * <p>
         * A timestamp that shows with the job was started processing.
         * </p>
         * 
         * @param startTime
         *        A timestamp that shows with the job was started processing.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * A timestamp that shows when the job was created.
         * </p>
         * 
         * @param creationTime
         *        A timestamp that shows when the job was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * A timestamp that shows when the job was completed.
         * </p>
         * 
         * @param completionTime
         *        A timestamp that shows when the job was completed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completionTime(Instant completionTime);

        /**
         * <p>
         * If the <code>TranscriptionJobStatus</code> field is <code>FAILED</code>, this field contains information
         * about why the job failed.
         * </p>
         * <p>
         * The <code>FailureReason</code> field can contain one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Unsupported media format</code> - The media format specified in the <code>MediaFormat</code> field of
         * the request isn't valid. See the description of the <code>MediaFormat</code> field for a list of valid
         * values.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>The media format provided does not match the detected media format</code> - The media format of the
         * audio file doesn't match the format specified in the <code>MediaFormat</code> field in the request. Check the
         * media format of your media file and make sure that the two values match.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Invalid sample rate for audio file</code> - The sample rate specified in the
         * <code>MediaSampleRateHertz</code> of the request isn't valid. The sample rate must be between 8000 and 48000
         * Hertz.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>The sample rate provided does not match the detected sample rate</code> - The sample rate in the audio
         * file doesn't match the sample rate specified in the <code>MediaSampleRateHertz</code> field in the request.
         * Check the sample rate of your media file and make sure that the two values match.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Invalid file size: file size too large</code> - The size of your audio file is larger than Amazon
         * Transcribe can process. For more information, see <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Limits</a> in the
         * <i>Amazon Transcribe Developer Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Invalid number of channels: number of channels too large</code> - Your audio contains more channels
         * than Amazon Transcribe is configured to process. To request additional channels, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws_service_limits.html#limits-amazon-transcribe">Amazon
         * Transcribe Limits</a> in the <i>Amazon Web Services General Reference</i>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param failureReason
         *        If the <code>TranscriptionJobStatus</code> field is <code>FAILED</code>, this field contains
         *        information about why the job failed.</p>
         *        <p>
         *        The <code>FailureReason</code> field can contain one of the following values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Unsupported media format</code> - The media format specified in the <code>MediaFormat</code>
         *        field of the request isn't valid. See the description of the <code>MediaFormat</code> field for a list
         *        of valid values.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>The media format provided does not match the detected media format</code> - The media format of
         *        the audio file doesn't match the format specified in the <code>MediaFormat</code> field in the
         *        request. Check the media format of your media file and make sure that the two values match.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Invalid sample rate for audio file</code> - The sample rate specified in the
         *        <code>MediaSampleRateHertz</code> of the request isn't valid. The sample rate must be between 8000 and
         *        48000 Hertz.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>The sample rate provided does not match the detected sample rate</code> - The sample rate in the
         *        audio file doesn't match the sample rate specified in the <code>MediaSampleRateHertz</code> field in
         *        the request. Check the sample rate of your media file and make sure that the two values match.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Invalid file size: file size too large</code> - The size of your audio file is larger than
         *        Amazon Transcribe can process. For more information, see <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Limits</a> in
         *        the <i>Amazon Transcribe Developer Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Invalid number of channels: number of channels too large</code> - Your audio contains more
         *        channels than Amazon Transcribe is configured to process. To request additional channels, see <a href=
         *        "https://docs.aws.amazon.com/general/latest/gr/aws_service_limits.html#limits-amazon-transcribe"
         *        >Amazon Transcribe Limits</a> in the <i>Amazon Web Services General Reference</i>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReason(String failureReason);

        /**
         * <p>
         * Optional settings for the transcription job. Use these settings to turn on speaker recognition, to set the
         * maximum number of speakers that should be identified and to specify a custom vocabulary to use when
         * processing the transcription job.
         * </p>
         * 
         * @param settings
         *        Optional settings for the transcription job. Use these settings to turn on speaker recognition, to set
         *        the maximum number of speakers that should be identified and to specify a custom vocabulary to use
         *        when processing the transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder settings(Settings settings);

        /**
         * <p>
         * Optional settings for the transcription job. Use these settings to turn on speaker recognition, to set the
         * maximum number of speakers that should be identified and to specify a custom vocabulary to use when
         * processing the transcription job.
         * </p>
         * This is a convenience that creates an instance of the {@link Settings.Builder} avoiding the need to create
         * one manually via {@link Settings#builder()}.
         *
         * When the {@link Consumer} completes, {@link Settings.Builder#build()} is called immediately and its result is
         * passed to {@link #settings(Settings)}.
         * 
         * @param settings
         *        a consumer that will call methods on {@link Settings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #settings(Settings)
         */
        default Builder settings(Consumer<Settings.Builder> settings) {
            return settings(Settings.builder().applyMutation(settings).build());
        }

        /**
         * <p>
         * An object containing the details of your custom language model.
         * </p>
         * 
         * @param modelSettings
         *        An object containing the details of your custom language model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelSettings(ModelSettings modelSettings);

        /**
         * <p>
         * An object containing the details of your custom language model.
         * </p>
         * This is a convenience that creates an instance of the {@link ModelSettings.Builder} avoiding the need to
         * create one manually via {@link ModelSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link ModelSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #modelSettings(ModelSettings)}.
         * 
         * @param modelSettings
         *        a consumer that will call methods on {@link ModelSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #modelSettings(ModelSettings)
         */
        default Builder modelSettings(Consumer<ModelSettings.Builder> modelSettings) {
            return modelSettings(ModelSettings.builder().applyMutation(modelSettings).build());
        }

        /**
         * <p>
         * Provides information about how a transcription job is executed.
         * </p>
         * 
         * @param jobExecutionSettings
         *        Provides information about how a transcription job is executed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobExecutionSettings(JobExecutionSettings jobExecutionSettings);

        /**
         * <p>
         * Provides information about how a transcription job is executed.
         * </p>
         * This is a convenience that creates an instance of the {@link JobExecutionSettings.Builder} avoiding the need
         * to create one manually via {@link JobExecutionSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link JobExecutionSettings.Builder#build()} is called immediately and
         * its result is passed to {@link #jobExecutionSettings(JobExecutionSettings)}.
         * 
         * @param jobExecutionSettings
         *        a consumer that will call methods on {@link JobExecutionSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #jobExecutionSettings(JobExecutionSettings)
         */
        default Builder jobExecutionSettings(Consumer<JobExecutionSettings.Builder> jobExecutionSettings) {
            return jobExecutionSettings(JobExecutionSettings.builder().applyMutation(jobExecutionSettings).build());
        }

        /**
         * <p>
         * An object that describes content redaction settings for the transcription job.
         * </p>
         * 
         * @param contentRedaction
         *        An object that describes content redaction settings for the transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contentRedaction(ContentRedaction contentRedaction);

        /**
         * <p>
         * An object that describes content redaction settings for the transcription job.
         * </p>
         * This is a convenience that creates an instance of the {@link ContentRedaction.Builder} avoiding the need to
         * create one manually via {@link ContentRedaction#builder()}.
         *
         * When the {@link Consumer} completes, {@link ContentRedaction.Builder#build()} is called immediately and its
         * result is passed to {@link #contentRedaction(ContentRedaction)}.
         * 
         * @param contentRedaction
         *        a consumer that will call methods on {@link ContentRedaction.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #contentRedaction(ContentRedaction)
         */
        default Builder contentRedaction(Consumer<ContentRedaction.Builder> contentRedaction) {
            return contentRedaction(ContentRedaction.builder().applyMutation(contentRedaction).build());
        }

        /**
         * <p>
         * A value that shows if automatic language identification was enabled for a transcription job.
         * </p>
         * 
         * @param identifyLanguage
         *        A value that shows if automatic language identification was enabled for a transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identifyLanguage(Boolean identifyLanguage);

        /**
         * <p>
         * An object that shows the optional array of languages inputted for transcription jobs with automatic language
         * identification enabled.
         * </p>
         * 
         * @param languageOptions
         *        An object that shows the optional array of languages inputted for transcription jobs with automatic
         *        language identification enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageOptionsWithStrings(Collection<String> languageOptions);

        /**
         * <p>
         * An object that shows the optional array of languages inputted for transcription jobs with automatic language
         * identification enabled.
         * </p>
         * 
         * @param languageOptions
         *        An object that shows the optional array of languages inputted for transcription jobs with automatic
         *        language identification enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageOptionsWithStrings(String... languageOptions);

        /**
         * <p>
         * An object that shows the optional array of languages inputted for transcription jobs with automatic language
         * identification enabled.
         * </p>
         * 
         * @param languageOptions
         *        An object that shows the optional array of languages inputted for transcription jobs with automatic
         *        language identification enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageOptions(Collection<LanguageCode> languageOptions);

        /**
         * <p>
         * An object that shows the optional array of languages inputted for transcription jobs with automatic language
         * identification enabled.
         * </p>
         * 
         * @param languageOptions
         *        An object that shows the optional array of languages inputted for transcription jobs with automatic
         *        language identification enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageOptions(LanguageCode... languageOptions);

        /**
         * <p>
         * A value between zero and one that Amazon Transcribe assigned to the language that it identified in the source
         * audio. Larger values indicate that Amazon Transcribe has higher confidence in the language it identified.
         * </p>
         * 
         * @param identifiedLanguageScore
         *        A value between zero and one that Amazon Transcribe assigned to the language that it identified in the
         *        source audio. Larger values indicate that Amazon Transcribe has higher confidence in the language it
         *        identified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identifiedLanguageScore(Float identifiedLanguageScore);
    }

    static final class BuilderImpl implements Builder {
        private String transcriptionJobName;

        private String transcriptionJobStatus;

        private String languageCode;

        private Integer mediaSampleRateHertz;

        private String mediaFormat;

        private Media media;

        private Transcript transcript;

        private Instant startTime;

        private Instant creationTime;

        private Instant completionTime;

        private String failureReason;

        private Settings settings;

        private ModelSettings modelSettings;

        private JobExecutionSettings jobExecutionSettings;

        private ContentRedaction contentRedaction;

        private Boolean identifyLanguage;

        private List<String> languageOptions = DefaultSdkAutoConstructList.getInstance();

        private Float identifiedLanguageScore;

        private BuilderImpl() {
        }

        private BuilderImpl(TranscriptionJob model) {
            transcriptionJobName(model.transcriptionJobName);
            transcriptionJobStatus(model.transcriptionJobStatus);
            languageCode(model.languageCode);
            mediaSampleRateHertz(model.mediaSampleRateHertz);
            mediaFormat(model.mediaFormat);
            media(model.media);
            transcript(model.transcript);
            startTime(model.startTime);
            creationTime(model.creationTime);
            completionTime(model.completionTime);
            failureReason(model.failureReason);
            settings(model.settings);
            modelSettings(model.modelSettings);
            jobExecutionSettings(model.jobExecutionSettings);
            contentRedaction(model.contentRedaction);
            identifyLanguage(model.identifyLanguage);
            languageOptionsWithStrings(model.languageOptions);
            identifiedLanguageScore(model.identifiedLanguageScore);
        }

        public final String getTranscriptionJobName() {
            return transcriptionJobName;
        }

        @Override
        public final Builder transcriptionJobName(String transcriptionJobName) {
            this.transcriptionJobName = transcriptionJobName;
            return this;
        }

        public final void setTranscriptionJobName(String transcriptionJobName) {
            this.transcriptionJobName = transcriptionJobName;
        }

        public final String getTranscriptionJobStatus() {
            return transcriptionJobStatus;
        }

        @Override
        public final Builder transcriptionJobStatus(String transcriptionJobStatus) {
            this.transcriptionJobStatus = transcriptionJobStatus;
            return this;
        }

        @Override
        public final Builder transcriptionJobStatus(TranscriptionJobStatus transcriptionJobStatus) {
            this.transcriptionJobStatus(transcriptionJobStatus == null ? null : transcriptionJobStatus.toString());
            return this;
        }

        public final void setTranscriptionJobStatus(String transcriptionJobStatus) {
            this.transcriptionJobStatus = transcriptionJobStatus;
        }

        public final String getLanguageCode() {
            return languageCode;
        }

        @Override
        public final Builder languageCode(String languageCode) {
            this.languageCode = languageCode;
            return this;
        }

        @Override
        public final Builder languageCode(LanguageCode languageCode) {
            this.languageCode(languageCode == null ? null : languageCode.toString());
            return this;
        }

        public final void setLanguageCode(String languageCode) {
            this.languageCode = languageCode;
        }

        public final Integer getMediaSampleRateHertz() {
            return mediaSampleRateHertz;
        }

        @Override
        public final Builder mediaSampleRateHertz(Integer mediaSampleRateHertz) {
            this.mediaSampleRateHertz = mediaSampleRateHertz;
            return this;
        }

        public final void setMediaSampleRateHertz(Integer mediaSampleRateHertz) {
            this.mediaSampleRateHertz = mediaSampleRateHertz;
        }

        public final String getMediaFormat() {
            return mediaFormat;
        }

        @Override
        public final Builder mediaFormat(String mediaFormat) {
            this.mediaFormat = mediaFormat;
            return this;
        }

        @Override
        public final Builder mediaFormat(MediaFormat mediaFormat) {
            this.mediaFormat(mediaFormat == null ? null : mediaFormat.toString());
            return this;
        }

        public final void setMediaFormat(String mediaFormat) {
            this.mediaFormat = mediaFormat;
        }

        public final Media.Builder getMedia() {
            return media != null ? media.toBuilder() : null;
        }

        @Override
        public final Builder media(Media media) {
            this.media = media;
            return this;
        }

        public final void setMedia(Media.BuilderImpl media) {
            this.media = media != null ? media.build() : null;
        }

        public final Transcript.Builder getTranscript() {
            return transcript != null ? transcript.toBuilder() : null;
        }

        @Override
        public final Builder transcript(Transcript transcript) {
            this.transcript = transcript;
            return this;
        }

        public final void setTranscript(Transcript.BuilderImpl transcript) {
            this.transcript = transcript != null ? transcript.build() : null;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        public final Instant getCompletionTime() {
            return completionTime;
        }

        @Override
        public final Builder completionTime(Instant completionTime) {
            this.completionTime = completionTime;
            return this;
        }

        public final void setCompletionTime(Instant completionTime) {
            this.completionTime = completionTime;
        }

        public final String getFailureReason() {
            return failureReason;
        }

        @Override
        public final Builder failureReason(String failureReason) {
            this.failureReason = failureReason;
            return this;
        }

        public final void setFailureReason(String failureReason) {
            this.failureReason = failureReason;
        }

        public final Settings.Builder getSettings() {
            return settings != null ? settings.toBuilder() : null;
        }

        @Override
        public final Builder settings(Settings settings) {
            this.settings = settings;
            return this;
        }

        public final void setSettings(Settings.BuilderImpl settings) {
            this.settings = settings != null ? settings.build() : null;
        }

        public final ModelSettings.Builder getModelSettings() {
            return modelSettings != null ? modelSettings.toBuilder() : null;
        }

        @Override
        public final Builder modelSettings(ModelSettings modelSettings) {
            this.modelSettings = modelSettings;
            return this;
        }

        public final void setModelSettings(ModelSettings.BuilderImpl modelSettings) {
            this.modelSettings = modelSettings != null ? modelSettings.build() : null;
        }

        public final JobExecutionSettings.Builder getJobExecutionSettings() {
            return jobExecutionSettings != null ? jobExecutionSettings.toBuilder() : null;
        }

        @Override
        public final Builder jobExecutionSettings(JobExecutionSettings jobExecutionSettings) {
            this.jobExecutionSettings = jobExecutionSettings;
            return this;
        }

        public final void setJobExecutionSettings(JobExecutionSettings.BuilderImpl jobExecutionSettings) {
            this.jobExecutionSettings = jobExecutionSettings != null ? jobExecutionSettings.build() : null;
        }

        public final ContentRedaction.Builder getContentRedaction() {
            return contentRedaction != null ? contentRedaction.toBuilder() : null;
        }

        @Override
        public final Builder contentRedaction(ContentRedaction contentRedaction) {
            this.contentRedaction = contentRedaction;
            return this;
        }

        public final void setContentRedaction(ContentRedaction.BuilderImpl contentRedaction) {
            this.contentRedaction = contentRedaction != null ? contentRedaction.build() : null;
        }

        public final Boolean getIdentifyLanguage() {
            return identifyLanguage;
        }

        @Override
        public final Builder identifyLanguage(Boolean identifyLanguage) {
            this.identifyLanguage = identifyLanguage;
            return this;
        }

        public final void setIdentifyLanguage(Boolean identifyLanguage) {
            this.identifyLanguage = identifyLanguage;
        }

        public final Collection<String> getLanguageOptions() {
            if (languageOptions instanceof SdkAutoConstructList) {
                return null;
            }
            return languageOptions;
        }

        @Override
        public final Builder languageOptionsWithStrings(Collection<String> languageOptions) {
            this.languageOptions = LanguageOptionsCopier.copy(languageOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder languageOptionsWithStrings(String... languageOptions) {
            languageOptionsWithStrings(Arrays.asList(languageOptions));
            return this;
        }

        @Override
        public final Builder languageOptions(Collection<LanguageCode> languageOptions) {
            this.languageOptions = LanguageOptionsCopier.copyEnumToString(languageOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder languageOptions(LanguageCode... languageOptions) {
            languageOptions(Arrays.asList(languageOptions));
            return this;
        }

        public final void setLanguageOptions(Collection<String> languageOptions) {
            this.languageOptions = LanguageOptionsCopier.copy(languageOptions);
        }

        public final Float getIdentifiedLanguageScore() {
            return identifiedLanguageScore;
        }

        @Override
        public final Builder identifiedLanguageScore(Float identifiedLanguageScore) {
            this.identifiedLanguageScore = identifiedLanguageScore;
            return this;
        }

        public final void setIdentifiedLanguageScore(Float identifiedLanguageScore) {
            this.identifiedLanguageScore = identifiedLanguageScore;
        }

        @Override
        public TranscriptionJob build() {
            return new TranscriptionJob(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
