/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribe;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.transcribe.model.BadRequestException;
import software.amazon.awssdk.services.transcribe.model.ConflictException;
import software.amazon.awssdk.services.transcribe.model.CreateLanguageModelRequest;
import software.amazon.awssdk.services.transcribe.model.CreateLanguageModelResponse;
import software.amazon.awssdk.services.transcribe.model.CreateMedicalVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.CreateMedicalVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.CreateVocabularyFilterRequest;
import software.amazon.awssdk.services.transcribe.model.CreateVocabularyFilterResponse;
import software.amazon.awssdk.services.transcribe.model.CreateVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.CreateVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteLanguageModelRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteLanguageModelResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteMedicalTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteMedicalTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteMedicalVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteMedicalVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteVocabularyFilterRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteVocabularyFilterResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.DescribeLanguageModelRequest;
import software.amazon.awssdk.services.transcribe.model.DescribeLanguageModelResponse;
import software.amazon.awssdk.services.transcribe.model.GetMedicalTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.GetMedicalTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.GetMedicalVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.GetMedicalVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.GetTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.GetTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.GetVocabularyFilterRequest;
import software.amazon.awssdk.services.transcribe.model.GetVocabularyFilterResponse;
import software.amazon.awssdk.services.transcribe.model.GetVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.GetVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.InternalFailureException;
import software.amazon.awssdk.services.transcribe.model.LimitExceededException;
import software.amazon.awssdk.services.transcribe.model.ListLanguageModelsRequest;
import software.amazon.awssdk.services.transcribe.model.ListLanguageModelsResponse;
import software.amazon.awssdk.services.transcribe.model.ListMedicalTranscriptionJobsRequest;
import software.amazon.awssdk.services.transcribe.model.ListMedicalTranscriptionJobsResponse;
import software.amazon.awssdk.services.transcribe.model.ListMedicalVocabulariesRequest;
import software.amazon.awssdk.services.transcribe.model.ListMedicalVocabulariesResponse;
import software.amazon.awssdk.services.transcribe.model.ListTranscriptionJobsRequest;
import software.amazon.awssdk.services.transcribe.model.ListTranscriptionJobsResponse;
import software.amazon.awssdk.services.transcribe.model.ListVocabulariesRequest;
import software.amazon.awssdk.services.transcribe.model.ListVocabulariesResponse;
import software.amazon.awssdk.services.transcribe.model.ListVocabularyFiltersRequest;
import software.amazon.awssdk.services.transcribe.model.ListVocabularyFiltersResponse;
import software.amazon.awssdk.services.transcribe.model.NotFoundException;
import software.amazon.awssdk.services.transcribe.model.StartMedicalTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.StartMedicalTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.StartTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.StartTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.TranscribeException;
import software.amazon.awssdk.services.transcribe.model.TranscribeRequest;
import software.amazon.awssdk.services.transcribe.model.UpdateMedicalVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.UpdateMedicalVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.UpdateVocabularyFilterRequest;
import software.amazon.awssdk.services.transcribe.model.UpdateVocabularyFilterResponse;
import software.amazon.awssdk.services.transcribe.model.UpdateVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.UpdateVocabularyResponse;
import software.amazon.awssdk.services.transcribe.paginators.ListLanguageModelsPublisher;
import software.amazon.awssdk.services.transcribe.paginators.ListMedicalTranscriptionJobsPublisher;
import software.amazon.awssdk.services.transcribe.paginators.ListMedicalVocabulariesPublisher;
import software.amazon.awssdk.services.transcribe.paginators.ListTranscriptionJobsPublisher;
import software.amazon.awssdk.services.transcribe.paginators.ListVocabulariesPublisher;
import software.amazon.awssdk.services.transcribe.paginators.ListVocabularyFiltersPublisher;
import software.amazon.awssdk.services.transcribe.transform.CreateLanguageModelRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.CreateMedicalVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.CreateVocabularyFilterRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.CreateVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteLanguageModelRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteMedicalTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteMedicalVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteVocabularyFilterRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DescribeLanguageModelRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetMedicalTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetMedicalVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetVocabularyFilterRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListLanguageModelsRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListMedicalTranscriptionJobsRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListMedicalVocabulariesRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListTranscriptionJobsRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListVocabulariesRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListVocabularyFiltersRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.StartMedicalTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.StartTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.UpdateMedicalVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.UpdateVocabularyFilterRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.UpdateVocabularyRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link TranscribeAsyncClient}.
 *
 * @see TranscribeAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTranscribeAsyncClient implements TranscribeAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultTranscribeAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultTranscribeAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a new custom language model. Use Amazon S3 prefixes to provide the location of your input files. The time
     * it takes to create your model depends on the size of your training data.
     * </p>
     *
     * @param createLanguageModelRequest
     * @return A Java Future containing the result of the CreateLanguageModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>ConflictException There is already a resource with that name.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.CreateLanguageModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/CreateLanguageModel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLanguageModelResponse> createLanguageModel(
            CreateLanguageModelRequest createLanguageModelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLanguageModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLanguageModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLanguageModelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLanguageModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLanguageModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLanguageModelRequest, CreateLanguageModelResponse>()
                            .withOperationName("CreateLanguageModel")
                            .withMarshaller(new CreateLanguageModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLanguageModelRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createLanguageModelRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<CreateLanguageModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new custom vocabulary that you can use to change how Amazon Transcribe Medical transcribes your audio
     * file.
     * </p>
     *
     * @param createMedicalVocabularyRequest
     * @return A Java Future containing the result of the CreateMedicalVocabulary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>ConflictException There is already a resource with that name.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.CreateMedicalVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/CreateMedicalVocabulary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMedicalVocabularyResponse> createMedicalVocabulary(
            CreateMedicalVocabularyRequest createMedicalVocabularyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMedicalVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMedicalVocabulary");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMedicalVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateMedicalVocabularyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMedicalVocabularyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMedicalVocabularyRequest, CreateMedicalVocabularyResponse>()
                            .withOperationName("CreateMedicalVocabulary")
                            .withMarshaller(new CreateMedicalVocabularyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createMedicalVocabularyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createMedicalVocabularyRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreateMedicalVocabularyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new custom vocabulary that you can use to change the way Amazon Transcribe handles transcription of an
     * audio file.
     * </p>
     *
     * @param createVocabularyRequest
     * @return A Java Future containing the result of the CreateVocabulary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>ConflictException There is already a resource with that name.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.CreateVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/CreateVocabulary" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateVocabularyResponse> createVocabulary(CreateVocabularyRequest createVocabularyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVocabulary");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateVocabularyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateVocabularyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateVocabularyRequest, CreateVocabularyResponse>()
                            .withOperationName("CreateVocabulary")
                            .withMarshaller(new CreateVocabularyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createVocabularyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createVocabularyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateVocabularyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new vocabulary filter that you can use to filter words, such as profane words, from the output of a
     * transcription job.
     * </p>
     *
     * @param createVocabularyFilterRequest
     * @return A Java Future containing the result of the CreateVocabularyFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>ConflictException There is already a resource with that name.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.CreateVocabularyFilter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/CreateVocabularyFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateVocabularyFilterResponse> createVocabularyFilter(
            CreateVocabularyFilterRequest createVocabularyFilterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVocabularyFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVocabularyFilter");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateVocabularyFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateVocabularyFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateVocabularyFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateVocabularyFilterRequest, CreateVocabularyFilterResponse>()
                            .withOperationName("CreateVocabularyFilter")
                            .withMarshaller(new CreateVocabularyFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createVocabularyFilterRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createVocabularyFilterRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<CreateVocabularyFilterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a custom language model using its name.
     * </p>
     *
     * @param deleteLanguageModelRequest
     * @return A Java Future containing the result of the DeleteLanguageModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.DeleteLanguageModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteLanguageModel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLanguageModelResponse> deleteLanguageModel(
            DeleteLanguageModelRequest deleteLanguageModelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLanguageModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLanguageModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLanguageModelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteLanguageModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLanguageModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLanguageModelRequest, DeleteLanguageModelResponse>()
                            .withOperationName("DeleteLanguageModel")
                            .withMarshaller(new DeleteLanguageModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteLanguageModelRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteLanguageModelRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeleteLanguageModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a transcription job generated by Amazon Transcribe Medical and any related information.
     * </p>
     *
     * @param deleteMedicalTranscriptionJobRequest
     * @return A Java Future containing the result of the DeleteMedicalTranscriptionJob operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.DeleteMedicalTranscriptionJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteMedicalTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMedicalTranscriptionJobResponse> deleteMedicalTranscriptionJob(
            DeleteMedicalTranscriptionJobRequest deleteMedicalTranscriptionJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteMedicalTranscriptionJobRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMedicalTranscriptionJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMedicalTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteMedicalTranscriptionJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMedicalTranscriptionJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMedicalTranscriptionJobRequest, DeleteMedicalTranscriptionJobResponse>()
                            .withOperationName("DeleteMedicalTranscriptionJob")
                            .withMarshaller(new DeleteMedicalTranscriptionJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteMedicalTranscriptionJobRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteMedicalTranscriptionJobRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteMedicalTranscriptionJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a vocabulary from Amazon Transcribe Medical.
     * </p>
     *
     * @param deleteMedicalVocabularyRequest
     * @return A Java Future containing the result of the DeleteMedicalVocabulary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException We can't find the requested resource. Check the name and try your request again.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.DeleteMedicalVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteMedicalVocabulary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMedicalVocabularyResponse> deleteMedicalVocabulary(
            DeleteMedicalVocabularyRequest deleteMedicalVocabularyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMedicalVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMedicalVocabulary");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMedicalVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteMedicalVocabularyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMedicalVocabularyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMedicalVocabularyRequest, DeleteMedicalVocabularyResponse>()
                            .withOperationName("DeleteMedicalVocabulary")
                            .withMarshaller(new DeleteMedicalVocabularyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteMedicalVocabularyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteMedicalVocabularyRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteMedicalVocabularyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a previously submitted transcription job along with any other generated results such as the
     * transcription, models, and so on.
     * </p>
     *
     * @param deleteTranscriptionJobRequest
     * @return A Java Future containing the result of the DeleteTranscriptionJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.DeleteTranscriptionJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTranscriptionJobResponse> deleteTranscriptionJob(
            DeleteTranscriptionJobRequest deleteTranscriptionJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTranscriptionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTranscriptionJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteTranscriptionJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTranscriptionJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTranscriptionJobRequest, DeleteTranscriptionJobResponse>()
                            .withOperationName("DeleteTranscriptionJob")
                            .withMarshaller(new DeleteTranscriptionJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteTranscriptionJobRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteTranscriptionJobRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeleteTranscriptionJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a vocabulary from Amazon Transcribe.
     * </p>
     *
     * @param deleteVocabularyRequest
     * @return A Java Future containing the result of the DeleteVocabulary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException We can't find the requested resource. Check the name and try your request again.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.DeleteVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteVocabulary" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVocabularyResponse> deleteVocabulary(DeleteVocabularyRequest deleteVocabularyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVocabulary");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteVocabularyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteVocabularyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVocabularyRequest, DeleteVocabularyResponse>()
                            .withOperationName("DeleteVocabulary")
                            .withMarshaller(new DeleteVocabularyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteVocabularyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteVocabularyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteVocabularyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a vocabulary filter.
     * </p>
     *
     * @param deleteVocabularyFilterRequest
     * @return A Java Future containing the result of the DeleteVocabularyFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException We can't find the requested resource. Check the name and try your request again.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.DeleteVocabularyFilter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteVocabularyFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVocabularyFilterResponse> deleteVocabularyFilter(
            DeleteVocabularyFilterRequest deleteVocabularyFilterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVocabularyFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVocabularyFilter");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVocabularyFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteVocabularyFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteVocabularyFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVocabularyFilterRequest, DeleteVocabularyFilterResponse>()
                            .withOperationName("DeleteVocabularyFilter")
                            .withMarshaller(new DeleteVocabularyFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteVocabularyFilterRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteVocabularyFilterRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeleteVocabularyFilterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a single custom language model. Use this information to see details about the language
     * model in your AWS account. You can also see whether the base language model used to create your custom language
     * model has been updated. If Amazon Transcribe has updated the base model, you can create a new custom language
     * model using the updated base model. If the language model wasn't created, you can use this operation to
     * understand why Amazon Transcribe couldn't create it.
     * </p>
     *
     * @param describeLanguageModelRequest
     * @return A Java Future containing the result of the DescribeLanguageModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>NotFoundException We can't find the requested resource. Check the name and try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.DescribeLanguageModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DescribeLanguageModel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLanguageModelResponse> describeLanguageModel(
            DescribeLanguageModelRequest describeLanguageModelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLanguageModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLanguageModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLanguageModelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLanguageModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeLanguageModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLanguageModelRequest, DescribeLanguageModelResponse>()
                            .withOperationName("DescribeLanguageModel")
                            .withMarshaller(new DescribeLanguageModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeLanguageModelRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeLanguageModelRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DescribeLanguageModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a transcription job from Amazon Transcribe Medical. To see the status of the job, check
     * the <code>TranscriptionJobStatus</code> field. If the status is <code>COMPLETED</code>, the job is finished. You
     * find the results of the completed job in the <code>TranscriptFileUri</code> field.
     * </p>
     *
     * @param getMedicalTranscriptionJobRequest
     * @return A Java Future containing the result of the GetMedicalTranscriptionJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>NotFoundException We can't find the requested resource. Check the name and try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.GetMedicalTranscriptionJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetMedicalTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMedicalTranscriptionJobResponse> getMedicalTranscriptionJob(
            GetMedicalTranscriptionJobRequest getMedicalTranscriptionJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMedicalTranscriptionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMedicalTranscriptionJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMedicalTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMedicalTranscriptionJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMedicalTranscriptionJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMedicalTranscriptionJobRequest, GetMedicalTranscriptionJobResponse>()
                            .withOperationName("GetMedicalTranscriptionJob")
                            .withMarshaller(new GetMedicalTranscriptionJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getMedicalTranscriptionJobRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getMedicalTranscriptionJobRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetMedicalTranscriptionJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about a medical vocabulary.
     * </p>
     *
     * @param getMedicalVocabularyRequest
     * @return A Java Future containing the result of the GetMedicalVocabulary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException We can't find the requested resource. Check the name and try your request again.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.GetMedicalVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetMedicalVocabulary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMedicalVocabularyResponse> getMedicalVocabulary(
            GetMedicalVocabularyRequest getMedicalVocabularyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMedicalVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMedicalVocabulary");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMedicalVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMedicalVocabularyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMedicalVocabularyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMedicalVocabularyRequest, GetMedicalVocabularyResponse>()
                            .withOperationName("GetMedicalVocabulary")
                            .withMarshaller(new GetMedicalVocabularyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getMedicalVocabularyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getMedicalVocabularyRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetMedicalVocabularyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a transcription job. To see the status of the job, check the
     * <code>TranscriptionJobStatus</code> field. If the status is <code>COMPLETED</code>, the job is finished and you
     * can find the results at the location specified in the <code>TranscriptFileUri</code> field. If you enable content
     * redaction, the redacted transcript appears in <code>RedactedTranscriptFileUri</code>.
     * </p>
     *
     * @param getTranscriptionJobRequest
     * @return A Java Future containing the result of the GetTranscriptionJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>NotFoundException We can't find the requested resource. Check the name and try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.GetTranscriptionJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTranscriptionJobResponse> getTranscriptionJob(
            GetTranscriptionJobRequest getTranscriptionJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTranscriptionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTranscriptionJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTranscriptionJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTranscriptionJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTranscriptionJobRequest, GetTranscriptionJobResponse>()
                            .withOperationName("GetTranscriptionJob")
                            .withMarshaller(new GetTranscriptionJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getTranscriptionJobRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getTranscriptionJobRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetTranscriptionJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a vocabulary.
     * </p>
     *
     * @param getVocabularyRequest
     * @return A Java Future containing the result of the GetVocabulary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException We can't find the requested resource. Check the name and try your request again.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.GetVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetVocabulary" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetVocabularyResponse> getVocabulary(GetVocabularyRequest getVocabularyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetVocabulary");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetVocabularyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetVocabularyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetVocabularyRequest, GetVocabularyResponse>()
                            .withOperationName("GetVocabulary")
                            .withMarshaller(new GetVocabularyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getVocabularyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getVocabularyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetVocabularyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a vocabulary filter.
     * </p>
     *
     * @param getVocabularyFilterRequest
     * @return A Java Future containing the result of the GetVocabularyFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException We can't find the requested resource. Check the name and try your request again.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.GetVocabularyFilter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetVocabularyFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetVocabularyFilterResponse> getVocabularyFilter(
            GetVocabularyFilterRequest getVocabularyFilterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getVocabularyFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetVocabularyFilter");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetVocabularyFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetVocabularyFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetVocabularyFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetVocabularyFilterRequest, GetVocabularyFilterResponse>()
                            .withOperationName("GetVocabularyFilter")
                            .withMarshaller(new GetVocabularyFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getVocabularyFilterRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getVocabularyFilterRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetVocabularyFilterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides more information about the custom language models you've created. You can use the information in this
     * list to find a specific custom language model. You can then use the operation to get more information about it.
     * </p>
     *
     * @param listLanguageModelsRequest
     * @return A Java Future containing the result of the ListLanguageModels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListLanguageModels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListLanguageModels" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLanguageModelsResponse> listLanguageModels(ListLanguageModelsRequest listLanguageModelsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLanguageModelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLanguageModels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLanguageModelsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLanguageModelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLanguageModelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLanguageModelsRequest, ListLanguageModelsResponse>()
                            .withOperationName("ListLanguageModels")
                            .withMarshaller(new ListLanguageModelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLanguageModelsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listLanguageModelsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListLanguageModelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides more information about the custom language models you've created. You can use the information in this
     * list to find a specific custom language model. You can then use the operation to get more information about it.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listLanguageModels(software.amazon.awssdk.services.transcribe.model.ListLanguageModelsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListLanguageModelsPublisher publisher = client.listLanguageModelsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListLanguageModelsPublisher publisher = client.listLanguageModelsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transcribe.model.ListLanguageModelsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transcribe.model.ListLanguageModelsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLanguageModels(software.amazon.awssdk.services.transcribe.model.ListLanguageModelsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listLanguageModelsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListLanguageModels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListLanguageModels" target="_top">AWS
     *      API Documentation</a>
     */
    public ListLanguageModelsPublisher listLanguageModelsPaginator(ListLanguageModelsRequest listLanguageModelsRequest) {
        return new ListLanguageModelsPublisher(this, applyPaginatorUserAgent(listLanguageModelsRequest));
    }

    /**
     * <p>
     * Lists medical transcription jobs with a specified status or substring that matches their names.
     * </p>
     *
     * @param listMedicalTranscriptionJobsRequest
     * @return A Java Future containing the result of the ListMedicalTranscriptionJobs operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListMedicalTranscriptionJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListMedicalTranscriptionJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMedicalTranscriptionJobsResponse> listMedicalTranscriptionJobs(
            ListMedicalTranscriptionJobsRequest listMedicalTranscriptionJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMedicalTranscriptionJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMedicalTranscriptionJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMedicalTranscriptionJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMedicalTranscriptionJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMedicalTranscriptionJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMedicalTranscriptionJobsRequest, ListMedicalTranscriptionJobsResponse>()
                            .withOperationName("ListMedicalTranscriptionJobs")
                            .withMarshaller(new ListMedicalTranscriptionJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listMedicalTranscriptionJobsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listMedicalTranscriptionJobsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListMedicalTranscriptionJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists medical transcription jobs with a specified status or substring that matches their names.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMedicalTranscriptionJobs(software.amazon.awssdk.services.transcribe.model.ListMedicalTranscriptionJobsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListMedicalTranscriptionJobsPublisher publisher = client.listMedicalTranscriptionJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListMedicalTranscriptionJobsPublisher publisher = client.listMedicalTranscriptionJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transcribe.model.ListMedicalTranscriptionJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transcribe.model.ListMedicalTranscriptionJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMedicalTranscriptionJobs(software.amazon.awssdk.services.transcribe.model.ListMedicalTranscriptionJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listMedicalTranscriptionJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListMedicalTranscriptionJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListMedicalTranscriptionJobs"
     *      target="_top">AWS API Documentation</a>
     */
    public ListMedicalTranscriptionJobsPublisher listMedicalTranscriptionJobsPaginator(
            ListMedicalTranscriptionJobsRequest listMedicalTranscriptionJobsRequest) {
        return new ListMedicalTranscriptionJobsPublisher(this, applyPaginatorUserAgent(listMedicalTranscriptionJobsRequest));
    }

    /**
     * <p>
     * Returns a list of vocabularies that match the specified criteria. If you don't enter a value in any of the
     * request parameters, returns the entire list of vocabularies.
     * </p>
     *
     * @param listMedicalVocabulariesRequest
     * @return A Java Future containing the result of the ListMedicalVocabularies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListMedicalVocabularies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListMedicalVocabularies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMedicalVocabulariesResponse> listMedicalVocabularies(
            ListMedicalVocabulariesRequest listMedicalVocabulariesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMedicalVocabulariesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMedicalVocabularies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMedicalVocabulariesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMedicalVocabulariesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMedicalVocabulariesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMedicalVocabulariesRequest, ListMedicalVocabulariesResponse>()
                            .withOperationName("ListMedicalVocabularies")
                            .withMarshaller(new ListMedicalVocabulariesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listMedicalVocabulariesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listMedicalVocabulariesRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListMedicalVocabulariesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of vocabularies that match the specified criteria. If you don't enter a value in any of the
     * request parameters, returns the entire list of vocabularies.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMedicalVocabularies(software.amazon.awssdk.services.transcribe.model.ListMedicalVocabulariesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListMedicalVocabulariesPublisher publisher = client.listMedicalVocabulariesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListMedicalVocabulariesPublisher publisher = client.listMedicalVocabulariesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transcribe.model.ListMedicalVocabulariesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transcribe.model.ListMedicalVocabulariesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMedicalVocabularies(software.amazon.awssdk.services.transcribe.model.ListMedicalVocabulariesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listMedicalVocabulariesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListMedicalVocabularies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListMedicalVocabularies"
     *      target="_top">AWS API Documentation</a>
     */
    public ListMedicalVocabulariesPublisher listMedicalVocabulariesPaginator(
            ListMedicalVocabulariesRequest listMedicalVocabulariesRequest) {
        return new ListMedicalVocabulariesPublisher(this, applyPaginatorUserAgent(listMedicalVocabulariesRequest));
    }

    /**
     * <p>
     * Lists transcription jobs with the specified status.
     * </p>
     *
     * @param listTranscriptionJobsRequest
     * @return A Java Future containing the result of the ListTranscriptionJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListTranscriptionJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListTranscriptionJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTranscriptionJobsResponse> listTranscriptionJobs(
            ListTranscriptionJobsRequest listTranscriptionJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTranscriptionJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTranscriptionJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTranscriptionJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTranscriptionJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTranscriptionJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTranscriptionJobsRequest, ListTranscriptionJobsResponse>()
                            .withOperationName("ListTranscriptionJobs")
                            .withMarshaller(new ListTranscriptionJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTranscriptionJobsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTranscriptionJobsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTranscriptionJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists transcription jobs with the specified status.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTranscriptionJobs(software.amazon.awssdk.services.transcribe.model.ListTranscriptionJobsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListTranscriptionJobsPublisher publisher = client.listTranscriptionJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListTranscriptionJobsPublisher publisher = client.listTranscriptionJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transcribe.model.ListTranscriptionJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transcribe.model.ListTranscriptionJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTranscriptionJobs(software.amazon.awssdk.services.transcribe.model.ListTranscriptionJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTranscriptionJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListTranscriptionJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListTranscriptionJobs"
     *      target="_top">AWS API Documentation</a>
     */
    public ListTranscriptionJobsPublisher listTranscriptionJobsPaginator(ListTranscriptionJobsRequest listTranscriptionJobsRequest) {
        return new ListTranscriptionJobsPublisher(this, applyPaginatorUserAgent(listTranscriptionJobsRequest));
    }

    /**
     * <p>
     * Returns a list of vocabularies that match the specified criteria. If no criteria are specified, returns the
     * entire list of vocabularies.
     * </p>
     *
     * @param listVocabulariesRequest
     * @return A Java Future containing the result of the ListVocabularies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListVocabularies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListVocabularies" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListVocabulariesResponse> listVocabularies(ListVocabulariesRequest listVocabulariesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVocabulariesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVocabularies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVocabulariesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListVocabulariesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListVocabulariesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVocabulariesRequest, ListVocabulariesResponse>()
                            .withOperationName("ListVocabularies")
                            .withMarshaller(new ListVocabulariesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listVocabulariesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listVocabulariesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListVocabulariesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of vocabularies that match the specified criteria. If no criteria are specified, returns the
     * entire list of vocabularies.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listVocabularies(software.amazon.awssdk.services.transcribe.model.ListVocabulariesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListVocabulariesPublisher publisher = client.listVocabulariesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListVocabulariesPublisher publisher = client.listVocabulariesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transcribe.model.ListVocabulariesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transcribe.model.ListVocabulariesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listVocabularies(software.amazon.awssdk.services.transcribe.model.ListVocabulariesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listVocabulariesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListVocabularies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListVocabularies" target="_top">AWS
     *      API Documentation</a>
     */
    public ListVocabulariesPublisher listVocabulariesPaginator(ListVocabulariesRequest listVocabulariesRequest) {
        return new ListVocabulariesPublisher(this, applyPaginatorUserAgent(listVocabulariesRequest));
    }

    /**
     * <p>
     * Gets information about vocabulary filters.
     * </p>
     *
     * @param listVocabularyFiltersRequest
     * @return A Java Future containing the result of the ListVocabularyFilters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListVocabularyFilters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListVocabularyFilters"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListVocabularyFiltersResponse> listVocabularyFilters(
            ListVocabularyFiltersRequest listVocabularyFiltersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVocabularyFiltersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVocabularyFilters");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVocabularyFiltersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListVocabularyFiltersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListVocabularyFiltersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVocabularyFiltersRequest, ListVocabularyFiltersResponse>()
                            .withOperationName("ListVocabularyFilters")
                            .withMarshaller(new ListVocabularyFiltersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listVocabularyFiltersRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listVocabularyFiltersRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListVocabularyFiltersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about vocabulary filters.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listVocabularyFilters(software.amazon.awssdk.services.transcribe.model.ListVocabularyFiltersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListVocabularyFiltersPublisher publisher = client.listVocabularyFiltersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transcribe.paginators.ListVocabularyFiltersPublisher publisher = client.listVocabularyFiltersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transcribe.model.ListVocabularyFiltersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transcribe.model.ListVocabularyFiltersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listVocabularyFilters(software.amazon.awssdk.services.transcribe.model.ListVocabularyFiltersRequest)}
     * operation.</b>
     * </p>
     *
     * @param listVocabularyFiltersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListVocabularyFilters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListVocabularyFilters"
     *      target="_top">AWS API Documentation</a>
     */
    public ListVocabularyFiltersPublisher listVocabularyFiltersPaginator(ListVocabularyFiltersRequest listVocabularyFiltersRequest) {
        return new ListVocabularyFiltersPublisher(this, applyPaginatorUserAgent(listVocabularyFiltersRequest));
    }

    /**
     * <p>
     * Starts a batch job to transcribe medical speech to text.
     * </p>
     *
     * @param startMedicalTranscriptionJobRequest
     * @return A Java Future containing the result of the StartMedicalTranscriptionJob operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>ConflictException There is already a resource with that name.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.StartMedicalTranscriptionJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/StartMedicalTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartMedicalTranscriptionJobResponse> startMedicalTranscriptionJob(
            StartMedicalTranscriptionJobRequest startMedicalTranscriptionJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startMedicalTranscriptionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMedicalTranscriptionJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartMedicalTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartMedicalTranscriptionJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartMedicalTranscriptionJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartMedicalTranscriptionJobRequest, StartMedicalTranscriptionJobResponse>()
                            .withOperationName("StartMedicalTranscriptionJob")
                            .withMarshaller(new StartMedicalTranscriptionJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startMedicalTranscriptionJobRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = startMedicalTranscriptionJobRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<StartMedicalTranscriptionJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts an asynchronous job to transcribe speech to text.
     * </p>
     *
     * @param startTranscriptionJobRequest
     * @return A Java Future containing the result of the StartTranscriptionJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>ConflictException There is already a resource with that name.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.StartTranscriptionJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/StartTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartTranscriptionJobResponse> startTranscriptionJob(
            StartTranscriptionJobRequest startTranscriptionJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startTranscriptionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartTranscriptionJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartTranscriptionJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartTranscriptionJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartTranscriptionJobRequest, StartTranscriptionJobResponse>()
                            .withOperationName("StartTranscriptionJob")
                            .withMarshaller(new StartTranscriptionJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startTranscriptionJobRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = startTranscriptionJobRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<StartTranscriptionJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a vocabulary with new values that you provide in a different text file from the one you used to create
     * the vocabulary. The <code>UpdateMedicalVocabulary</code> operation overwrites all of the existing information
     * with the values that you provide in the request.
     * </p>
     *
     * @param updateMedicalVocabularyRequest
     * @return A Java Future containing the result of the UpdateMedicalVocabulary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>NotFoundException We can't find the requested resource. Check the name and try your request again.</li>
     *         <li>ConflictException There is already a resource with that name.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.UpdateMedicalVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/UpdateMedicalVocabulary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateMedicalVocabularyResponse> updateMedicalVocabulary(
            UpdateMedicalVocabularyRequest updateMedicalVocabularyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMedicalVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMedicalVocabulary");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateMedicalVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateMedicalVocabularyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateMedicalVocabularyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateMedicalVocabularyRequest, UpdateMedicalVocabularyResponse>()
                            .withOperationName("UpdateMedicalVocabulary")
                            .withMarshaller(new UpdateMedicalVocabularyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateMedicalVocabularyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateMedicalVocabularyRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UpdateMedicalVocabularyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing vocabulary with new values. The <code>UpdateVocabulary</code> operation overwrites all of the
     * existing information with the values that you provide in the request.
     * </p>
     *
     * @param updateVocabularyRequest
     * @return A Java Future containing the result of the UpdateVocabulary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>NotFoundException We can't find the requested resource. Check the name and try your request again.</li>
     *         <li>ConflictException There is already a resource with that name.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.UpdateVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/UpdateVocabulary" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateVocabularyResponse> updateVocabulary(UpdateVocabularyRequest updateVocabularyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVocabulary");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateVocabularyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateVocabularyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateVocabularyRequest, UpdateVocabularyResponse>()
                            .withOperationName("UpdateVocabulary")
                            .withMarshaller(new UpdateVocabularyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateVocabularyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateVocabularyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateVocabularyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a vocabulary filter with a new list of filtered words.
     * </p>
     *
     * @param updateVocabularyFilterRequest
     * @return A Java Future containing the result of the UpdateVocabularyFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. For example, if the entity
     *         that you're trying to delete doesn't exist or if it is in a non-terminal state (for example, it's
     *         "in progress"). See the exception <code>Message</code> field for more information.</li>
     *         <li>LimitExceededException Either you have sent too many requests or your input file is too long. Wait
     *         before you resend your request, or use a smaller file and resend the request.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message and try your request
     *         again.</li>
     *         <li>NotFoundException We can't find the requested resource. Check the name and try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.UpdateVocabularyFilter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/UpdateVocabularyFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateVocabularyFilterResponse> updateVocabularyFilter(
            UpdateVocabularyFilterRequest updateVocabularyFilterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVocabularyFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVocabularyFilter");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateVocabularyFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateVocabularyFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateVocabularyFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateVocabularyFilterRequest, UpdateVocabularyFilterResponse>()
                            .withOperationName("UpdateVocabularyFilter")
                            .withMarshaller(new UpdateVocabularyFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateVocabularyFilterRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateVocabularyFilterRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<UpdateVocabularyFilterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(TranscribeException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends TranscribeRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
