/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribe.model;

import java.beans.Transient;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class StartTranscriptionJobRequest extends TranscribeRequest implements
        ToCopyableBuilder<StartTranscriptionJobRequest.Builder, StartTranscriptionJobRequest> {
    private static final SdkField<String> TRANSCRIPTION_JOB_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TranscriptionJobName").getter(getter(StartTranscriptionJobRequest::transcriptionJobName))
            .setter(setter(Builder::transcriptionJobName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TranscriptionJobName").build())
            .build();

    private static final SdkField<String> LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LanguageCode").getter(getter(StartTranscriptionJobRequest::languageCodeAsString))
            .setter(setter(Builder::languageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageCode").build()).build();

    private static final SdkField<Integer> MEDIA_SAMPLE_RATE_HERTZ_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MediaSampleRateHertz").getter(getter(StartTranscriptionJobRequest::mediaSampleRateHertz))
            .setter(setter(Builder::mediaSampleRateHertz))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MediaSampleRateHertz").build())
            .build();

    private static final SdkField<String> MEDIA_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MediaFormat").getter(getter(StartTranscriptionJobRequest::mediaFormatAsString))
            .setter(setter(Builder::mediaFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MediaFormat").build()).build();

    private static final SdkField<Media> MEDIA_FIELD = SdkField.<Media> builder(MarshallingType.SDK_POJO).memberName("Media")
            .getter(getter(StartTranscriptionJobRequest::media)).setter(setter(Builder::media)).constructor(Media::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Media").build()).build();

    private static final SdkField<String> OUTPUT_BUCKET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OutputBucketName").getter(getter(StartTranscriptionJobRequest::outputBucketName))
            .setter(setter(Builder::outputBucketName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputBucketName").build()).build();

    private static final SdkField<String> OUTPUT_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OutputKey").getter(getter(StartTranscriptionJobRequest::outputKey)).setter(setter(Builder::outputKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputKey").build()).build();

    private static final SdkField<String> OUTPUT_ENCRYPTION_KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OutputEncryptionKMSKeyId").getter(getter(StartTranscriptionJobRequest::outputEncryptionKMSKeyId))
            .setter(setter(Builder::outputEncryptionKMSKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputEncryptionKMSKeyId").build())
            .build();

    private static final SdkField<Map<String, String>> KMS_ENCRYPTION_CONTEXT_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("KMSEncryptionContext")
            .getter(getter(StartTranscriptionJobRequest::kmsEncryptionContext))
            .setter(setter(Builder::kmsEncryptionContext))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KMSEncryptionContext").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Settings> SETTINGS_FIELD = SdkField.<Settings> builder(MarshallingType.SDK_POJO)
            .memberName("Settings").getter(getter(StartTranscriptionJobRequest::settings)).setter(setter(Builder::settings))
            .constructor(Settings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Settings").build()).build();

    private static final SdkField<ModelSettings> MODEL_SETTINGS_FIELD = SdkField
            .<ModelSettings> builder(MarshallingType.SDK_POJO).memberName("ModelSettings")
            .getter(getter(StartTranscriptionJobRequest::modelSettings)).setter(setter(Builder::modelSettings))
            .constructor(ModelSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelSettings").build()).build();

    private static final SdkField<JobExecutionSettings> JOB_EXECUTION_SETTINGS_FIELD = SdkField
            .<JobExecutionSettings> builder(MarshallingType.SDK_POJO).memberName("JobExecutionSettings")
            .getter(getter(StartTranscriptionJobRequest::jobExecutionSettings)).setter(setter(Builder::jobExecutionSettings))
            .constructor(JobExecutionSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobExecutionSettings").build())
            .build();

    private static final SdkField<ContentRedaction> CONTENT_REDACTION_FIELD = SdkField
            .<ContentRedaction> builder(MarshallingType.SDK_POJO).memberName("ContentRedaction")
            .getter(getter(StartTranscriptionJobRequest::contentRedaction)).setter(setter(Builder::contentRedaction))
            .constructor(ContentRedaction::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContentRedaction").build()).build();

    private static final SdkField<Boolean> IDENTIFY_LANGUAGE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IdentifyLanguage").getter(getter(StartTranscriptionJobRequest::identifyLanguage))
            .setter(setter(Builder::identifyLanguage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdentifyLanguage").build()).build();

    private static final SdkField<List<String>> LANGUAGE_OPTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("LanguageOptions")
            .getter(getter(StartTranscriptionJobRequest::languageOptionsAsStrings))
            .setter(setter(Builder::languageOptionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageOptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Subtitles> SUBTITLES_FIELD = SdkField.<Subtitles> builder(MarshallingType.SDK_POJO)
            .memberName("Subtitles").getter(getter(StartTranscriptionJobRequest::subtitles)).setter(setter(Builder::subtitles))
            .constructor(Subtitles::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Subtitles").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(StartTranscriptionJobRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, LanguageIdSettings>> LANGUAGE_ID_SETTINGS_FIELD = SdkField
            .<Map<String, LanguageIdSettings>> builder(MarshallingType.MAP)
            .memberName("LanguageIdSettings")
            .getter(getter(StartTranscriptionJobRequest::languageIdSettingsAsStrings))
            .setter(setter(Builder::languageIdSettingsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageIdSettings").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<LanguageIdSettings> builder(MarshallingType.SDK_POJO)
                                            .constructor(LanguageIdSettings::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TRANSCRIPTION_JOB_NAME_FIELD,
            LANGUAGE_CODE_FIELD, MEDIA_SAMPLE_RATE_HERTZ_FIELD, MEDIA_FORMAT_FIELD, MEDIA_FIELD, OUTPUT_BUCKET_NAME_FIELD,
            OUTPUT_KEY_FIELD, OUTPUT_ENCRYPTION_KMS_KEY_ID_FIELD, KMS_ENCRYPTION_CONTEXT_FIELD, SETTINGS_FIELD,
            MODEL_SETTINGS_FIELD, JOB_EXECUTION_SETTINGS_FIELD, CONTENT_REDACTION_FIELD, IDENTIFY_LANGUAGE_FIELD,
            LANGUAGE_OPTIONS_FIELD, SUBTITLES_FIELD, TAGS_FIELD, LANGUAGE_ID_SETTINGS_FIELD));

    private final String transcriptionJobName;

    private final String languageCode;

    private final Integer mediaSampleRateHertz;

    private final String mediaFormat;

    private final Media media;

    private final String outputBucketName;

    private final String outputKey;

    private final String outputEncryptionKMSKeyId;

    private final Map<String, String> kmsEncryptionContext;

    private final Settings settings;

    private final ModelSettings modelSettings;

    private final JobExecutionSettings jobExecutionSettings;

    private final ContentRedaction contentRedaction;

    private final Boolean identifyLanguage;

    private final List<String> languageOptions;

    private final Subtitles subtitles;

    private final List<Tag> tags;

    private final Map<String, LanguageIdSettings> languageIdSettings;

    private StartTranscriptionJobRequest(BuilderImpl builder) {
        super(builder);
        this.transcriptionJobName = builder.transcriptionJobName;
        this.languageCode = builder.languageCode;
        this.mediaSampleRateHertz = builder.mediaSampleRateHertz;
        this.mediaFormat = builder.mediaFormat;
        this.media = builder.media;
        this.outputBucketName = builder.outputBucketName;
        this.outputKey = builder.outputKey;
        this.outputEncryptionKMSKeyId = builder.outputEncryptionKMSKeyId;
        this.kmsEncryptionContext = builder.kmsEncryptionContext;
        this.settings = builder.settings;
        this.modelSettings = builder.modelSettings;
        this.jobExecutionSettings = builder.jobExecutionSettings;
        this.contentRedaction = builder.contentRedaction;
        this.identifyLanguage = builder.identifyLanguage;
        this.languageOptions = builder.languageOptions;
        this.subtitles = builder.subtitles;
        this.tags = builder.tags;
        this.languageIdSettings = builder.languageIdSettings;
    }

    /**
     * <p>
     * The name of the job. You can't use the strings "<code>.</code>" or "<code>..</code>" by themselves as the job
     * name. The name must also be unique within an Amazon Web Services account. If you try to create a transcription
     * job with the same name as a previous transcription job, you get a <code>ConflictException</code> error.
     * </p>
     * 
     * @return The name of the job. You can't use the strings "<code>.</code>" or "<code>..</code>" by themselves as the
     *         job name. The name must also be unique within an Amazon Web Services account. If you try to create a
     *         transcription job with the same name as a previous transcription job, you get a
     *         <code>ConflictException</code> error.
     */
    public final String transcriptionJobName() {
        return transcriptionJobName;
    }

    /**
     * <p>
     * The language code for the language used in the input media file.
     * </p>
     * <p>
     * To transcribe speech in Modern Standard Arabic (ar-SA), your audio or video file must be encoded at a sample rate
     * of 16,000 Hz or higher.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return The language code for the language used in the input media file.</p>
     *         <p>
     *         To transcribe speech in Modern Standard Arabic (ar-SA), your audio or video file must be encoded at a
     *         sample rate of 16,000 Hz or higher.
     * @see LanguageCode
     */
    public final LanguageCode languageCode() {
        return LanguageCode.fromValue(languageCode);
    }

    /**
     * <p>
     * The language code for the language used in the input media file.
     * </p>
     * <p>
     * To transcribe speech in Modern Standard Arabic (ar-SA), your audio or video file must be encoded at a sample rate
     * of 16,000 Hz or higher.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return The language code for the language used in the input media file.</p>
     *         <p>
     *         To transcribe speech in Modern Standard Arabic (ar-SA), your audio or video file must be encoded at a
     *         sample rate of 16,000 Hz or higher.
     * @see LanguageCode
     */
    public final String languageCodeAsString() {
        return languageCode;
    }

    /**
     * <p>
     * The sample rate, in Hertz, of the audio track in the input media file.
     * </p>
     * <p>
     * If you do not specify the media sample rate, Amazon Transcribe determines the sample rate. If you specify the
     * sample rate, it must match the sample rate detected by Amazon Transcribe. In most cases, you should leave the
     * <code>MediaSampleRateHertz</code> field blank and let Amazon Transcribe determine the sample rate.
     * </p>
     * 
     * @return The sample rate, in Hertz, of the audio track in the input media file. </p>
     *         <p>
     *         If you do not specify the media sample rate, Amazon Transcribe determines the sample rate. If you specify
     *         the sample rate, it must match the sample rate detected by Amazon Transcribe. In most cases, you should
     *         leave the <code>MediaSampleRateHertz</code> field blank and let Amazon Transcribe determine the sample
     *         rate.
     */
    public final Integer mediaSampleRateHertz() {
        return mediaSampleRateHertz;
    }

    /**
     * <p>
     * The format of the input media file.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mediaFormat} will
     * return {@link MediaFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #mediaFormatAsString}.
     * </p>
     * 
     * @return The format of the input media file.
     * @see MediaFormat
     */
    public final MediaFormat mediaFormat() {
        return MediaFormat.fromValue(mediaFormat);
    }

    /**
     * <p>
     * The format of the input media file.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mediaFormat} will
     * return {@link MediaFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #mediaFormatAsString}.
     * </p>
     * 
     * @return The format of the input media file.
     * @see MediaFormat
     */
    public final String mediaFormatAsString() {
        return mediaFormat;
    }

    /**
     * <p>
     * An object that describes the input media for a transcription job.
     * </p>
     * 
     * @return An object that describes the input media for a transcription job.
     */
    public final Media media() {
        return media;
    }

    /**
     * <p>
     * The location where the transcription is stored.
     * </p>
     * <p>
     * If you set the <code>OutputBucketName</code>, Amazon Transcribe puts the transcript in the specified S3 bucket.
     * When you call the <a>GetTranscriptionJob</a> operation, the operation returns this location in the
     * <code>TranscriptFileUri</code> field. If you enable content redaction, the redacted transcript appears in
     * <code>RedactedTranscriptFileUri</code>. If you enable content redaction and choose to output an unredacted
     * transcript, that transcript's location still appears in the <code>TranscriptFileUri</code>. The S3 bucket must
     * have permissions that allow Amazon Transcribe to put files in the bucket. For more information, see <a href=
     * "https://docs.aws.amazon.com/transcribe/latest/dg/security_iam_id-based-policy-examples.html#auth-role-iam-user"
     * >Permissions Required for IAM User Roles</a>.
     * </p>
     * <p>
     * You can specify an Amazon Web Services Key Management Service (KMS) key to encrypt the output of your
     * transcription using the <code>OutputEncryptionKMSKeyId</code> parameter. If you don't specify a KMS key, Amazon
     * Transcribe uses the default Amazon S3 key for server-side encryption of transcripts that are placed in your S3
     * bucket.
     * </p>
     * <p>
     * If you don't set the <code>OutputBucketName</code>, Amazon Transcribe generates a pre-signed URL, a shareable URL
     * that provides secure access to your transcription, and returns it in the <code>TranscriptFileUri</code> field.
     * Use this URL to download the transcription.
     * </p>
     * 
     * @return The location where the transcription is stored.</p>
     *         <p>
     *         If you set the <code>OutputBucketName</code>, Amazon Transcribe puts the transcript in the specified S3
     *         bucket. When you call the <a>GetTranscriptionJob</a> operation, the operation returns this location in
     *         the <code>TranscriptFileUri</code> field. If you enable content redaction, the redacted transcript
     *         appears in <code>RedactedTranscriptFileUri</code>. If you enable content redaction and choose to output
     *         an unredacted transcript, that transcript's location still appears in the <code>TranscriptFileUri</code>.
     *         The S3 bucket must have permissions that allow Amazon Transcribe to put files in the bucket. For more
     *         information, see <a href=
     *         "https://docs.aws.amazon.com/transcribe/latest/dg/security_iam_id-based-policy-examples.html#auth-role-iam-user"
     *         >Permissions Required for IAM User Roles</a>.
     *         </p>
     *         <p>
     *         You can specify an Amazon Web Services Key Management Service (KMS) key to encrypt the output of your
     *         transcription using the <code>OutputEncryptionKMSKeyId</code> parameter. If you don't specify a KMS key,
     *         Amazon Transcribe uses the default Amazon S3 key for server-side encryption of transcripts that are
     *         placed in your S3 bucket.
     *         </p>
     *         <p>
     *         If you don't set the <code>OutputBucketName</code>, Amazon Transcribe generates a pre-signed URL, a
     *         shareable URL that provides secure access to your transcription, and returns it in the
     *         <code>TranscriptFileUri</code> field. Use this URL to download the transcription.
     */
    public final String outputBucketName() {
        return outputBucketName;
    }

    /**
     * <p>
     * You can specify a location in an Amazon S3 bucket to store the output of your transcription job.
     * </p>
     * <p>
     * If you don't specify an output key, Amazon Transcribe stores the output of your transcription job in the Amazon
     * S3 bucket you specified. By default, the object key is "your-transcription-job-name.json".
     * </p>
     * <p>
     * You can use output keys to specify the Amazon S3 prefix and file name of the transcription output. For example,
     * specifying the Amazon S3 prefix, "folder1/folder2/", as an output key would lead to the output being stored as
     * "folder1/folder2/your-transcription-job-name.json". If you specify "my-other-job-name.json" as the output key,
     * the object key is changed to "my-other-job-name.json". You can use an output key to change both the prefix and
     * the file name, for example "folder/my-other-job-name.json".
     * </p>
     * <p>
     * If you specify an output key, you must also specify an S3 bucket in the <code>OutputBucketName</code> parameter.
     * </p>
     * 
     * @return You can specify a location in an Amazon S3 bucket to store the output of your transcription job.</p>
     *         <p>
     *         If you don't specify an output key, Amazon Transcribe stores the output of your transcription job in the
     *         Amazon S3 bucket you specified. By default, the object key is "your-transcription-job-name.json".
     *         </p>
     *         <p>
     *         You can use output keys to specify the Amazon S3 prefix and file name of the transcription output. For
     *         example, specifying the Amazon S3 prefix, "folder1/folder2/", as an output key would lead to the output
     *         being stored as "folder1/folder2/your-transcription-job-name.json". If you specify
     *         "my-other-job-name.json" as the output key, the object key is changed to "my-other-job-name.json". You
     *         can use an output key to change both the prefix and the file name, for example
     *         "folder/my-other-job-name.json".
     *         </p>
     *         <p>
     *         If you specify an output key, you must also specify an S3 bucket in the <code>OutputBucketName</code>
     *         parameter.
     */
    public final String outputKey() {
        return outputKey;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Amazon Web Services Key Management Service (KMS) key used to encrypt the
     * output of the transcription job. The user calling the <code>StartTranscriptionJob</code> operation must have
     * permission to use the specified KMS key.
     * </p>
     * <p>
     * You can use either of the following to identify a KMS key in the current account:
     * </p>
     * <ul>
     * <li>
     * <p>
     * KMS Key ID: "1234abcd-12ab-34cd-56ef-1234567890ab"
     * </p>
     * </li>
     * <li>
     * <p>
     * KMS Key Alias: "alias/ExampleAlias"
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can use either of the following to identify a KMS key in the current account or another account:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Amazon Resource Name (ARN) of a KMS Key: "arn:aws:kms:region:account ID:key/1234abcd-12ab-34cd-56ef-1234567890ab"
     * </p>
     * </li>
     * <li>
     * <p>
     * ARN of a KMS Key Alias: "arn:aws:kms:region:account-ID:alias/ExampleAlias"
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you don't specify an encryption key, the output of the transcription job is encrypted with the default Amazon
     * S3 key (SSE-S3).
     * </p>
     * <p>
     * If you specify a KMS key to encrypt your output, you must also specify an output location in the
     * <code>OutputBucketName</code> parameter.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Amazon Web Services Key Management Service (KMS) key used to
     *         encrypt the output of the transcription job. The user calling the <code>StartTranscriptionJob</code>
     *         operation must have permission to use the specified KMS key.</p>
     *         <p>
     *         You can use either of the following to identify a KMS key in the current account:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         KMS Key ID: "1234abcd-12ab-34cd-56ef-1234567890ab"
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         KMS Key Alias: "alias/ExampleAlias"
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         You can use either of the following to identify a KMS key in the current account or another account:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Amazon Resource Name (ARN) of a KMS Key:
     *         "arn:aws:kms:region:account ID:key/1234abcd-12ab-34cd-56ef-1234567890ab"
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ARN of a KMS Key Alias: "arn:aws:kms:region:account-ID:alias/ExampleAlias"
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you don't specify an encryption key, the output of the transcription job is encrypted with the default
     *         Amazon S3 key (SSE-S3).
     *         </p>
     *         <p>
     *         If you specify a KMS key to encrypt your output, you must also specify an output location in the
     *         <code>OutputBucketName</code> parameter.
     */
    public final String outputEncryptionKMSKeyId() {
        return outputEncryptionKMSKeyId;
    }

    /**
     * For responses, this returns true if the service returned a value for the KMSEncryptionContext property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasKmsEncryptionContext() {
        return kmsEncryptionContext != null && !(kmsEncryptionContext instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of plain text, non-secret key:value pairs, known as encryption context pairs, that provide an added layer
     * of security for your data.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasKmsEncryptionContext} method.
     * </p>
     * 
     * @return A map of plain text, non-secret key:value pairs, known as encryption context pairs, that provide an added
     *         layer of security for your data.
     */
    public final Map<String, String> kmsEncryptionContext() {
        return kmsEncryptionContext;
    }

    /**
     * <p>
     * A <code>Settings</code> object that provides optional settings for a transcription job.
     * </p>
     * 
     * @return A <code>Settings</code> object that provides optional settings for a transcription job.
     */
    public final Settings settings() {
        return settings;
    }

    /**
     * <p>
     * Choose the custom language model you use for your transcription job in this parameter.
     * </p>
     * 
     * @return Choose the custom language model you use for your transcription job in this parameter.
     */
    public final ModelSettings modelSettings() {
        return modelSettings;
    }

    /**
     * <p>
     * Provides information about how a transcription job is executed. Use this field to indicate that the job can be
     * queued for deferred execution if the concurrency limit is reached and there are no slots available to immediately
     * run the job.
     * </p>
     * 
     * @return Provides information about how a transcription job is executed. Use this field to indicate that the job
     *         can be queued for deferred execution if the concurrency limit is reached and there are no slots available
     *         to immediately run the job.
     */
    public final JobExecutionSettings jobExecutionSettings() {
        return jobExecutionSettings;
    }

    /**
     * <p>
     * An object that contains the request parameters for content redaction.
     * </p>
     * 
     * @return An object that contains the request parameters for content redaction.
     */
    public final ContentRedaction contentRedaction() {
        return contentRedaction;
    }

    /**
     * <p>
     * Set this field to <code>true</code> to enable automatic language identification. Automatic language
     * identification is disabled by default. You receive a <code>BadRequestException</code> error if you enter a value
     * for a <code>LanguageCode</code>.
     * </p>
     * 
     * @return Set this field to <code>true</code> to enable automatic language identification. Automatic language
     *         identification is disabled by default. You receive a <code>BadRequestException</code> error if you enter
     *         a value for a <code>LanguageCode</code>.
     */
    public final Boolean identifyLanguage() {
        return identifyLanguage;
    }

    /**
     * <p>
     * An object containing a list of languages that might be present in your collection of audio files. Automatic
     * language identification chooses a language that best matches the source audio from that list.
     * </p>
     * <p>
     * To transcribe speech in Modern Standard Arabic (ar-SA), your audio or video file must be encoded at a sample rate
     * of 16,000 Hz or higher.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLanguageOptions} method.
     * </p>
     * 
     * @return An object containing a list of languages that might be present in your collection of audio files.
     *         Automatic language identification chooses a language that best matches the source audio from that
     *         list.</p>
     *         <p>
     *         To transcribe speech in Modern Standard Arabic (ar-SA), your audio or video file must be encoded at a
     *         sample rate of 16,000 Hz or higher.
     */
    public final List<LanguageCode> languageOptions() {
        return LanguageOptionsCopier.copyStringToEnum(languageOptions);
    }

    /**
     * For responses, this returns true if the service returned a value for the LanguageOptions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLanguageOptions() {
        return languageOptions != null && !(languageOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An object containing a list of languages that might be present in your collection of audio files. Automatic
     * language identification chooses a language that best matches the source audio from that list.
     * </p>
     * <p>
     * To transcribe speech in Modern Standard Arabic (ar-SA), your audio or video file must be encoded at a sample rate
     * of 16,000 Hz or higher.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLanguageOptions} method.
     * </p>
     * 
     * @return An object containing a list of languages that might be present in your collection of audio files.
     *         Automatic language identification chooses a language that best matches the source audio from that
     *         list.</p>
     *         <p>
     *         To transcribe speech in Modern Standard Arabic (ar-SA), your audio or video file must be encoded at a
     *         sample rate of 16,000 Hz or higher.
     */
    public final List<String> languageOptionsAsStrings() {
        return languageOptions;
    }

    /**
     * <p>
     * Add subtitles to your batch transcription job.
     * </p>
     * 
     * @return Add subtitles to your batch transcription job.
     */
    public final Subtitles subtitles() {
        return subtitles;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Add tags to an Amazon Transcribe transcription job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Add tags to an Amazon Transcribe transcription job.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The language identification settings associated with your transcription job. These settings include
     * <code>VocabularyName</code>, <code>VocabularyFilterName</code>, and <code>LanguageModelName</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLanguageIdSettings} method.
     * </p>
     * 
     * @return The language identification settings associated with your transcription job. These settings include
     *         <code>VocabularyName</code>, <code>VocabularyFilterName</code>, and <code>LanguageModelName</code>.
     */
    public final Map<LanguageCode, LanguageIdSettings> languageIdSettings() {
        return LanguageIdSettingsMapCopier.copyStringToEnum(languageIdSettings);
    }

    /**
     * For responses, this returns true if the service returned a value for the LanguageIdSettings property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLanguageIdSettings() {
        return languageIdSettings != null && !(languageIdSettings instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The language identification settings associated with your transcription job. These settings include
     * <code>VocabularyName</code>, <code>VocabularyFilterName</code>, and <code>LanguageModelName</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLanguageIdSettings} method.
     * </p>
     * 
     * @return The language identification settings associated with your transcription job. These settings include
     *         <code>VocabularyName</code>, <code>VocabularyFilterName</code>, and <code>LanguageModelName</code>.
     */
    public final Map<String, LanguageIdSettings> languageIdSettingsAsStrings() {
        return languageIdSettings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(transcriptionJobName());
        hashCode = 31 * hashCode + Objects.hashCode(languageCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(mediaSampleRateHertz());
        hashCode = 31 * hashCode + Objects.hashCode(mediaFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(media());
        hashCode = 31 * hashCode + Objects.hashCode(outputBucketName());
        hashCode = 31 * hashCode + Objects.hashCode(outputKey());
        hashCode = 31 * hashCode + Objects.hashCode(outputEncryptionKMSKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(hasKmsEncryptionContext() ? kmsEncryptionContext() : null);
        hashCode = 31 * hashCode + Objects.hashCode(settings());
        hashCode = 31 * hashCode + Objects.hashCode(modelSettings());
        hashCode = 31 * hashCode + Objects.hashCode(jobExecutionSettings());
        hashCode = 31 * hashCode + Objects.hashCode(contentRedaction());
        hashCode = 31 * hashCode + Objects.hashCode(identifyLanguage());
        hashCode = 31 * hashCode + Objects.hashCode(hasLanguageOptions() ? languageOptionsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(subtitles());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasLanguageIdSettings() ? languageIdSettingsAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StartTranscriptionJobRequest)) {
            return false;
        }
        StartTranscriptionJobRequest other = (StartTranscriptionJobRequest) obj;
        return Objects.equals(transcriptionJobName(), other.transcriptionJobName())
                && Objects.equals(languageCodeAsString(), other.languageCodeAsString())
                && Objects.equals(mediaSampleRateHertz(), other.mediaSampleRateHertz())
                && Objects.equals(mediaFormatAsString(), other.mediaFormatAsString()) && Objects.equals(media(), other.media())
                && Objects.equals(outputBucketName(), other.outputBucketName()) && Objects.equals(outputKey(), other.outputKey())
                && Objects.equals(outputEncryptionKMSKeyId(), other.outputEncryptionKMSKeyId())
                && hasKmsEncryptionContext() == other.hasKmsEncryptionContext()
                && Objects.equals(kmsEncryptionContext(), other.kmsEncryptionContext())
                && Objects.equals(settings(), other.settings()) && Objects.equals(modelSettings(), other.modelSettings())
                && Objects.equals(jobExecutionSettings(), other.jobExecutionSettings())
                && Objects.equals(contentRedaction(), other.contentRedaction())
                && Objects.equals(identifyLanguage(), other.identifyLanguage())
                && hasLanguageOptions() == other.hasLanguageOptions()
                && Objects.equals(languageOptionsAsStrings(), other.languageOptionsAsStrings())
                && Objects.equals(subtitles(), other.subtitles()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && hasLanguageIdSettings() == other.hasLanguageIdSettings()
                && Objects.equals(languageIdSettingsAsStrings(), other.languageIdSettingsAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StartTranscriptionJobRequest").add("TranscriptionJobName", transcriptionJobName())
                .add("LanguageCode", languageCodeAsString()).add("MediaSampleRateHertz", mediaSampleRateHertz())
                .add("MediaFormat", mediaFormatAsString()).add("Media", media()).add("OutputBucketName", outputBucketName())
                .add("OutputKey", outputKey()).add("OutputEncryptionKMSKeyId", outputEncryptionKMSKeyId())
                .add("KMSEncryptionContext", hasKmsEncryptionContext() ? kmsEncryptionContext() : null)
                .add("Settings", settings()).add("ModelSettings", modelSettings())
                .add("JobExecutionSettings", jobExecutionSettings()).add("ContentRedaction", contentRedaction())
                .add("IdentifyLanguage", identifyLanguage())
                .add("LanguageOptions", hasLanguageOptions() ? languageOptionsAsStrings() : null).add("Subtitles", subtitles())
                .add("Tags", hasTags() ? tags() : null)
                .add("LanguageIdSettings", hasLanguageIdSettings() ? languageIdSettingsAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TranscriptionJobName":
            return Optional.ofNullable(clazz.cast(transcriptionJobName()));
        case "LanguageCode":
            return Optional.ofNullable(clazz.cast(languageCodeAsString()));
        case "MediaSampleRateHertz":
            return Optional.ofNullable(clazz.cast(mediaSampleRateHertz()));
        case "MediaFormat":
            return Optional.ofNullable(clazz.cast(mediaFormatAsString()));
        case "Media":
            return Optional.ofNullable(clazz.cast(media()));
        case "OutputBucketName":
            return Optional.ofNullable(clazz.cast(outputBucketName()));
        case "OutputKey":
            return Optional.ofNullable(clazz.cast(outputKey()));
        case "OutputEncryptionKMSKeyId":
            return Optional.ofNullable(clazz.cast(outputEncryptionKMSKeyId()));
        case "KMSEncryptionContext":
            return Optional.ofNullable(clazz.cast(kmsEncryptionContext()));
        case "Settings":
            return Optional.ofNullable(clazz.cast(settings()));
        case "ModelSettings":
            return Optional.ofNullable(clazz.cast(modelSettings()));
        case "JobExecutionSettings":
            return Optional.ofNullable(clazz.cast(jobExecutionSettings()));
        case "ContentRedaction":
            return Optional.ofNullable(clazz.cast(contentRedaction()));
        case "IdentifyLanguage":
            return Optional.ofNullable(clazz.cast(identifyLanguage()));
        case "LanguageOptions":
            return Optional.ofNullable(clazz.cast(languageOptionsAsStrings()));
        case "Subtitles":
            return Optional.ofNullable(clazz.cast(subtitles()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "LanguageIdSettings":
            return Optional.ofNullable(clazz.cast(languageIdSettingsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StartTranscriptionJobRequest, T> g) {
        return obj -> g.apply((StartTranscriptionJobRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends TranscribeRequest.Builder, SdkPojo, CopyableBuilder<Builder, StartTranscriptionJobRequest> {
        /**
         * <p>
         * The name of the job. You can't use the strings "<code>.</code>" or "<code>..</code>" by themselves as the job
         * name. The name must also be unique within an Amazon Web Services account. If you try to create a
         * transcription job with the same name as a previous transcription job, you get a
         * <code>ConflictException</code> error.
         * </p>
         * 
         * @param transcriptionJobName
         *        The name of the job. You can't use the strings "<code>.</code>" or "<code>..</code>" by themselves as
         *        the job name. The name must also be unique within an Amazon Web Services account. If you try to create
         *        a transcription job with the same name as a previous transcription job, you get a
         *        <code>ConflictException</code> error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transcriptionJobName(String transcriptionJobName);

        /**
         * <p>
         * The language code for the language used in the input media file.
         * </p>
         * <p>
         * To transcribe speech in Modern Standard Arabic (ar-SA), your audio or video file must be encoded at a sample
         * rate of 16,000 Hz or higher.
         * </p>
         * 
         * @param languageCode
         *        The language code for the language used in the input media file.</p>
         *        <p>
         *        To transcribe speech in Modern Standard Arabic (ar-SA), your audio or video file must be encoded at a
         *        sample rate of 16,000 Hz or higher.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(String languageCode);

        /**
         * <p>
         * The language code for the language used in the input media file.
         * </p>
         * <p>
         * To transcribe speech in Modern Standard Arabic (ar-SA), your audio or video file must be encoded at a sample
         * rate of 16,000 Hz or higher.
         * </p>
         * 
         * @param languageCode
         *        The language code for the language used in the input media file.</p>
         *        <p>
         *        To transcribe speech in Modern Standard Arabic (ar-SA), your audio or video file must be encoded at a
         *        sample rate of 16,000 Hz or higher.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(LanguageCode languageCode);

        /**
         * <p>
         * The sample rate, in Hertz, of the audio track in the input media file.
         * </p>
         * <p>
         * If you do not specify the media sample rate, Amazon Transcribe determines the sample rate. If you specify the
         * sample rate, it must match the sample rate detected by Amazon Transcribe. In most cases, you should leave the
         * <code>MediaSampleRateHertz</code> field blank and let Amazon Transcribe determine the sample rate.
         * </p>
         * 
         * @param mediaSampleRateHertz
         *        The sample rate, in Hertz, of the audio track in the input media file. </p>
         *        <p>
         *        If you do not specify the media sample rate, Amazon Transcribe determines the sample rate. If you
         *        specify the sample rate, it must match the sample rate detected by Amazon Transcribe. In most cases,
         *        you should leave the <code>MediaSampleRateHertz</code> field blank and let Amazon Transcribe determine
         *        the sample rate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mediaSampleRateHertz(Integer mediaSampleRateHertz);

        /**
         * <p>
         * The format of the input media file.
         * </p>
         * 
         * @param mediaFormat
         *        The format of the input media file.
         * @see MediaFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MediaFormat
         */
        Builder mediaFormat(String mediaFormat);

        /**
         * <p>
         * The format of the input media file.
         * </p>
         * 
         * @param mediaFormat
         *        The format of the input media file.
         * @see MediaFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MediaFormat
         */
        Builder mediaFormat(MediaFormat mediaFormat);

        /**
         * <p>
         * An object that describes the input media for a transcription job.
         * </p>
         * 
         * @param media
         *        An object that describes the input media for a transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder media(Media media);

        /**
         * <p>
         * An object that describes the input media for a transcription job.
         * </p>
         * This is a convenience that creates an instance of the {@link Media.Builder} avoiding the need to create one
         * manually via {@link Media#builder()}.
         *
         * When the {@link Consumer} completes, {@link Media.Builder#build()} is called immediately and its result is
         * passed to {@link #media(Media)}.
         * 
         * @param media
         *        a consumer that will call methods on {@link Media.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #media(Media)
         */
        default Builder media(Consumer<Media.Builder> media) {
            return media(Media.builder().applyMutation(media).build());
        }

        /**
         * <p>
         * The location where the transcription is stored.
         * </p>
         * <p>
         * If you set the <code>OutputBucketName</code>, Amazon Transcribe puts the transcript in the specified S3
         * bucket. When you call the <a>GetTranscriptionJob</a> operation, the operation returns this location in the
         * <code>TranscriptFileUri</code> field. If you enable content redaction, the redacted transcript appears in
         * <code>RedactedTranscriptFileUri</code>. If you enable content redaction and choose to output an unredacted
         * transcript, that transcript's location still appears in the <code>TranscriptFileUri</code>. The S3 bucket
         * must have permissions that allow Amazon Transcribe to put files in the bucket. For more information, see <a
         * href=
         * "https://docs.aws.amazon.com/transcribe/latest/dg/security_iam_id-based-policy-examples.html#auth-role-iam-user"
         * >Permissions Required for IAM User Roles</a>.
         * </p>
         * <p>
         * You can specify an Amazon Web Services Key Management Service (KMS) key to encrypt the output of your
         * transcription using the <code>OutputEncryptionKMSKeyId</code> parameter. If you don't specify a KMS key,
         * Amazon Transcribe uses the default Amazon S3 key for server-side encryption of transcripts that are placed in
         * your S3 bucket.
         * </p>
         * <p>
         * If you don't set the <code>OutputBucketName</code>, Amazon Transcribe generates a pre-signed URL, a shareable
         * URL that provides secure access to your transcription, and returns it in the <code>TranscriptFileUri</code>
         * field. Use this URL to download the transcription.
         * </p>
         * 
         * @param outputBucketName
         *        The location where the transcription is stored.</p>
         *        <p>
         *        If you set the <code>OutputBucketName</code>, Amazon Transcribe puts the transcript in the specified
         *        S3 bucket. When you call the <a>GetTranscriptionJob</a> operation, the operation returns this location
         *        in the <code>TranscriptFileUri</code> field. If you enable content redaction, the redacted transcript
         *        appears in <code>RedactedTranscriptFileUri</code>. If you enable content redaction and choose to
         *        output an unredacted transcript, that transcript's location still appears in the
         *        <code>TranscriptFileUri</code>. The S3 bucket must have permissions that allow Amazon Transcribe to
         *        put files in the bucket. For more information, see <a href=
         *        "https://docs.aws.amazon.com/transcribe/latest/dg/security_iam_id-based-policy-examples.html#auth-role-iam-user"
         *        >Permissions Required for IAM User Roles</a>.
         *        </p>
         *        <p>
         *        You can specify an Amazon Web Services Key Management Service (KMS) key to encrypt the output of your
         *        transcription using the <code>OutputEncryptionKMSKeyId</code> parameter. If you don't specify a KMS
         *        key, Amazon Transcribe uses the default Amazon S3 key for server-side encryption of transcripts that
         *        are placed in your S3 bucket.
         *        </p>
         *        <p>
         *        If you don't set the <code>OutputBucketName</code>, Amazon Transcribe generates a pre-signed URL, a
         *        shareable URL that provides secure access to your transcription, and returns it in the
         *        <code>TranscriptFileUri</code> field. Use this URL to download the transcription.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputBucketName(String outputBucketName);

        /**
         * <p>
         * You can specify a location in an Amazon S3 bucket to store the output of your transcription job.
         * </p>
         * <p>
         * If you don't specify an output key, Amazon Transcribe stores the output of your transcription job in the
         * Amazon S3 bucket you specified. By default, the object key is "your-transcription-job-name.json".
         * </p>
         * <p>
         * You can use output keys to specify the Amazon S3 prefix and file name of the transcription output. For
         * example, specifying the Amazon S3 prefix, "folder1/folder2/", as an output key would lead to the output being
         * stored as "folder1/folder2/your-transcription-job-name.json". If you specify "my-other-job-name.json" as the
         * output key, the object key is changed to "my-other-job-name.json". You can use an output key to change both
         * the prefix and the file name, for example "folder/my-other-job-name.json".
         * </p>
         * <p>
         * If you specify an output key, you must also specify an S3 bucket in the <code>OutputBucketName</code>
         * parameter.
         * </p>
         * 
         * @param outputKey
         *        You can specify a location in an Amazon S3 bucket to store the output of your transcription job.</p>
         *        <p>
         *        If you don't specify an output key, Amazon Transcribe stores the output of your transcription job in
         *        the Amazon S3 bucket you specified. By default, the object key is "your-transcription-job-name.json".
         *        </p>
         *        <p>
         *        You can use output keys to specify the Amazon S3 prefix and file name of the transcription output. For
         *        example, specifying the Amazon S3 prefix, "folder1/folder2/", as an output key would lead to the
         *        output being stored as "folder1/folder2/your-transcription-job-name.json". If you specify
         *        "my-other-job-name.json" as the output key, the object key is changed to "my-other-job-name.json". You
         *        can use an output key to change both the prefix and the file name, for example
         *        "folder/my-other-job-name.json".
         *        </p>
         *        <p>
         *        If you specify an output key, you must also specify an S3 bucket in the <code>OutputBucketName</code>
         *        parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputKey(String outputKey);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Amazon Web Services Key Management Service (KMS) key used to encrypt
         * the output of the transcription job. The user calling the <code>StartTranscriptionJob</code> operation must
         * have permission to use the specified KMS key.
         * </p>
         * <p>
         * You can use either of the following to identify a KMS key in the current account:
         * </p>
         * <ul>
         * <li>
         * <p>
         * KMS Key ID: "1234abcd-12ab-34cd-56ef-1234567890ab"
         * </p>
         * </li>
         * <li>
         * <p>
         * KMS Key Alias: "alias/ExampleAlias"
         * </p>
         * </li>
         * </ul>
         * <p>
         * You can use either of the following to identify a KMS key in the current account or another account:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Amazon Resource Name (ARN) of a KMS Key:
         * "arn:aws:kms:region:account ID:key/1234abcd-12ab-34cd-56ef-1234567890ab"
         * </p>
         * </li>
         * <li>
         * <p>
         * ARN of a KMS Key Alias: "arn:aws:kms:region:account-ID:alias/ExampleAlias"
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you don't specify an encryption key, the output of the transcription job is encrypted with the default
         * Amazon S3 key (SSE-S3).
         * </p>
         * <p>
         * If you specify a KMS key to encrypt your output, you must also specify an output location in the
         * <code>OutputBucketName</code> parameter.
         * </p>
         * 
         * @param outputEncryptionKMSKeyId
         *        The Amazon Resource Name (ARN) of the Amazon Web Services Key Management Service (KMS) key used to
         *        encrypt the output of the transcription job. The user calling the <code>StartTranscriptionJob</code>
         *        operation must have permission to use the specified KMS key.</p>
         *        <p>
         *        You can use either of the following to identify a KMS key in the current account:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        KMS Key ID: "1234abcd-12ab-34cd-56ef-1234567890ab"
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        KMS Key Alias: "alias/ExampleAlias"
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        You can use either of the following to identify a KMS key in the current account or another account:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Amazon Resource Name (ARN) of a KMS Key:
         *        "arn:aws:kms:region:account ID:key/1234abcd-12ab-34cd-56ef-1234567890ab"
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ARN of a KMS Key Alias: "arn:aws:kms:region:account-ID:alias/ExampleAlias"
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you don't specify an encryption key, the output of the transcription job is encrypted with the
         *        default Amazon S3 key (SSE-S3).
         *        </p>
         *        <p>
         *        If you specify a KMS key to encrypt your output, you must also specify an output location in the
         *        <code>OutputBucketName</code> parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputEncryptionKMSKeyId(String outputEncryptionKMSKeyId);

        /**
         * <p>
         * A map of plain text, non-secret key:value pairs, known as encryption context pairs, that provide an added
         * layer of security for your data.
         * </p>
         * 
         * @param kmsEncryptionContext
         *        A map of plain text, non-secret key:value pairs, known as encryption context pairs, that provide an
         *        added layer of security for your data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsEncryptionContext(Map<String, String> kmsEncryptionContext);

        /**
         * <p>
         * A <code>Settings</code> object that provides optional settings for a transcription job.
         * </p>
         * 
         * @param settings
         *        A <code>Settings</code> object that provides optional settings for a transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder settings(Settings settings);

        /**
         * <p>
         * A <code>Settings</code> object that provides optional settings for a transcription job.
         * </p>
         * This is a convenience that creates an instance of the {@link Settings.Builder} avoiding the need to create
         * one manually via {@link Settings#builder()}.
         *
         * When the {@link Consumer} completes, {@link Settings.Builder#build()} is called immediately and its result is
         * passed to {@link #settings(Settings)}.
         * 
         * @param settings
         *        a consumer that will call methods on {@link Settings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #settings(Settings)
         */
        default Builder settings(Consumer<Settings.Builder> settings) {
            return settings(Settings.builder().applyMutation(settings).build());
        }

        /**
         * <p>
         * Choose the custom language model you use for your transcription job in this parameter.
         * </p>
         * 
         * @param modelSettings
         *        Choose the custom language model you use for your transcription job in this parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelSettings(ModelSettings modelSettings);

        /**
         * <p>
         * Choose the custom language model you use for your transcription job in this parameter.
         * </p>
         * This is a convenience that creates an instance of the {@link ModelSettings.Builder} avoiding the need to
         * create one manually via {@link ModelSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link ModelSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #modelSettings(ModelSettings)}.
         * 
         * @param modelSettings
         *        a consumer that will call methods on {@link ModelSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #modelSettings(ModelSettings)
         */
        default Builder modelSettings(Consumer<ModelSettings.Builder> modelSettings) {
            return modelSettings(ModelSettings.builder().applyMutation(modelSettings).build());
        }

        /**
         * <p>
         * Provides information about how a transcription job is executed. Use this field to indicate that the job can
         * be queued for deferred execution if the concurrency limit is reached and there are no slots available to
         * immediately run the job.
         * </p>
         * 
         * @param jobExecutionSettings
         *        Provides information about how a transcription job is executed. Use this field to indicate that the
         *        job can be queued for deferred execution if the concurrency limit is reached and there are no slots
         *        available to immediately run the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobExecutionSettings(JobExecutionSettings jobExecutionSettings);

        /**
         * <p>
         * Provides information about how a transcription job is executed. Use this field to indicate that the job can
         * be queued for deferred execution if the concurrency limit is reached and there are no slots available to
         * immediately run the job.
         * </p>
         * This is a convenience that creates an instance of the {@link JobExecutionSettings.Builder} avoiding the need
         * to create one manually via {@link JobExecutionSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link JobExecutionSettings.Builder#build()} is called immediately and
         * its result is passed to {@link #jobExecutionSettings(JobExecutionSettings)}.
         * 
         * @param jobExecutionSettings
         *        a consumer that will call methods on {@link JobExecutionSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #jobExecutionSettings(JobExecutionSettings)
         */
        default Builder jobExecutionSettings(Consumer<JobExecutionSettings.Builder> jobExecutionSettings) {
            return jobExecutionSettings(JobExecutionSettings.builder().applyMutation(jobExecutionSettings).build());
        }

        /**
         * <p>
         * An object that contains the request parameters for content redaction.
         * </p>
         * 
         * @param contentRedaction
         *        An object that contains the request parameters for content redaction.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contentRedaction(ContentRedaction contentRedaction);

        /**
         * <p>
         * An object that contains the request parameters for content redaction.
         * </p>
         * This is a convenience that creates an instance of the {@link ContentRedaction.Builder} avoiding the need to
         * create one manually via {@link ContentRedaction#builder()}.
         *
         * When the {@link Consumer} completes, {@link ContentRedaction.Builder#build()} is called immediately and its
         * result is passed to {@link #contentRedaction(ContentRedaction)}.
         * 
         * @param contentRedaction
         *        a consumer that will call methods on {@link ContentRedaction.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #contentRedaction(ContentRedaction)
         */
        default Builder contentRedaction(Consumer<ContentRedaction.Builder> contentRedaction) {
            return contentRedaction(ContentRedaction.builder().applyMutation(contentRedaction).build());
        }

        /**
         * <p>
         * Set this field to <code>true</code> to enable automatic language identification. Automatic language
         * identification is disabled by default. You receive a <code>BadRequestException</code> error if you enter a
         * value for a <code>LanguageCode</code>.
         * </p>
         * 
         * @param identifyLanguage
         *        Set this field to <code>true</code> to enable automatic language identification. Automatic language
         *        identification is disabled by default. You receive a <code>BadRequestException</code> error if you
         *        enter a value for a <code>LanguageCode</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identifyLanguage(Boolean identifyLanguage);

        /**
         * <p>
         * An object containing a list of languages that might be present in your collection of audio files. Automatic
         * language identification chooses a language that best matches the source audio from that list.
         * </p>
         * <p>
         * To transcribe speech in Modern Standard Arabic (ar-SA), your audio or video file must be encoded at a sample
         * rate of 16,000 Hz or higher.
         * </p>
         * 
         * @param languageOptions
         *        An object containing a list of languages that might be present in your collection of audio files.
         *        Automatic language identification chooses a language that best matches the source audio from that
         *        list.</p>
         *        <p>
         *        To transcribe speech in Modern Standard Arabic (ar-SA), your audio or video file must be encoded at a
         *        sample rate of 16,000 Hz or higher.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageOptionsWithStrings(Collection<String> languageOptions);

        /**
         * <p>
         * An object containing a list of languages that might be present in your collection of audio files. Automatic
         * language identification chooses a language that best matches the source audio from that list.
         * </p>
         * <p>
         * To transcribe speech in Modern Standard Arabic (ar-SA), your audio or video file must be encoded at a sample
         * rate of 16,000 Hz or higher.
         * </p>
         * 
         * @param languageOptions
         *        An object containing a list of languages that might be present in your collection of audio files.
         *        Automatic language identification chooses a language that best matches the source audio from that
         *        list.</p>
         *        <p>
         *        To transcribe speech in Modern Standard Arabic (ar-SA), your audio or video file must be encoded at a
         *        sample rate of 16,000 Hz or higher.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageOptionsWithStrings(String... languageOptions);

        /**
         * <p>
         * An object containing a list of languages that might be present in your collection of audio files. Automatic
         * language identification chooses a language that best matches the source audio from that list.
         * </p>
         * <p>
         * To transcribe speech in Modern Standard Arabic (ar-SA), your audio or video file must be encoded at a sample
         * rate of 16,000 Hz or higher.
         * </p>
         * 
         * @param languageOptions
         *        An object containing a list of languages that might be present in your collection of audio files.
         *        Automatic language identification chooses a language that best matches the source audio from that
         *        list.</p>
         *        <p>
         *        To transcribe speech in Modern Standard Arabic (ar-SA), your audio or video file must be encoded at a
         *        sample rate of 16,000 Hz or higher.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageOptions(Collection<LanguageCode> languageOptions);

        /**
         * <p>
         * An object containing a list of languages that might be present in your collection of audio files. Automatic
         * language identification chooses a language that best matches the source audio from that list.
         * </p>
         * <p>
         * To transcribe speech in Modern Standard Arabic (ar-SA), your audio or video file must be encoded at a sample
         * rate of 16,000 Hz or higher.
         * </p>
         * 
         * @param languageOptions
         *        An object containing a list of languages that might be present in your collection of audio files.
         *        Automatic language identification chooses a language that best matches the source audio from that
         *        list.</p>
         *        <p>
         *        To transcribe speech in Modern Standard Arabic (ar-SA), your audio or video file must be encoded at a
         *        sample rate of 16,000 Hz or higher.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageOptions(LanguageCode... languageOptions);

        /**
         * <p>
         * Add subtitles to your batch transcription job.
         * </p>
         * 
         * @param subtitles
         *        Add subtitles to your batch transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subtitles(Subtitles subtitles);

        /**
         * <p>
         * Add subtitles to your batch transcription job.
         * </p>
         * This is a convenience that creates an instance of the {@link Subtitles.Builder} avoiding the need to create
         * one manually via {@link Subtitles#builder()}.
         *
         * When the {@link Consumer} completes, {@link Subtitles.Builder#build()} is called immediately and its result
         * is passed to {@link #subtitles(Subtitles)}.
         * 
         * @param subtitles
         *        a consumer that will call methods on {@link Subtitles.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #subtitles(Subtitles)
         */
        default Builder subtitles(Consumer<Subtitles.Builder> subtitles) {
            return subtitles(Subtitles.builder().applyMutation(subtitles).build());
        }

        /**
         * <p>
         * Add tags to an Amazon Transcribe transcription job.
         * </p>
         * 
         * @param tags
         *        Add tags to an Amazon Transcribe transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Add tags to an Amazon Transcribe transcription job.
         * </p>
         * 
         * @param tags
         *        Add tags to an Amazon Transcribe transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Add tags to an Amazon Transcribe transcription job.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The language identification settings associated with your transcription job. These settings include
         * <code>VocabularyName</code>, <code>VocabularyFilterName</code>, and <code>LanguageModelName</code>.
         * </p>
         * 
         * @param languageIdSettings
         *        The language identification settings associated with your transcription job. These settings include
         *        <code>VocabularyName</code>, <code>VocabularyFilterName</code>, and <code>LanguageModelName</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageIdSettingsWithStrings(Map<String, LanguageIdSettings> languageIdSettings);

        /**
         * <p>
         * The language identification settings associated with your transcription job. These settings include
         * <code>VocabularyName</code>, <code>VocabularyFilterName</code>, and <code>LanguageModelName</code>.
         * </p>
         * 
         * @param languageIdSettings
         *        The language identification settings associated with your transcription job. These settings include
         *        <code>VocabularyName</code>, <code>VocabularyFilterName</code>, and <code>LanguageModelName</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageIdSettings(Map<LanguageCode, LanguageIdSettings> languageIdSettings);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends TranscribeRequest.BuilderImpl implements Builder {
        private String transcriptionJobName;

        private String languageCode;

        private Integer mediaSampleRateHertz;

        private String mediaFormat;

        private Media media;

        private String outputBucketName;

        private String outputKey;

        private String outputEncryptionKMSKeyId;

        private Map<String, String> kmsEncryptionContext = DefaultSdkAutoConstructMap.getInstance();

        private Settings settings;

        private ModelSettings modelSettings;

        private JobExecutionSettings jobExecutionSettings;

        private ContentRedaction contentRedaction;

        private Boolean identifyLanguage;

        private List<String> languageOptions = DefaultSdkAutoConstructList.getInstance();

        private Subtitles subtitles;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private Map<String, LanguageIdSettings> languageIdSettings = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(StartTranscriptionJobRequest model) {
            super(model);
            transcriptionJobName(model.transcriptionJobName);
            languageCode(model.languageCode);
            mediaSampleRateHertz(model.mediaSampleRateHertz);
            mediaFormat(model.mediaFormat);
            media(model.media);
            outputBucketName(model.outputBucketName);
            outputKey(model.outputKey);
            outputEncryptionKMSKeyId(model.outputEncryptionKMSKeyId);
            kmsEncryptionContext(model.kmsEncryptionContext);
            settings(model.settings);
            modelSettings(model.modelSettings);
            jobExecutionSettings(model.jobExecutionSettings);
            contentRedaction(model.contentRedaction);
            identifyLanguage(model.identifyLanguage);
            languageOptionsWithStrings(model.languageOptions);
            subtitles(model.subtitles);
            tags(model.tags);
            languageIdSettingsWithStrings(model.languageIdSettings);
        }

        public final String getTranscriptionJobName() {
            return transcriptionJobName;
        }

        public final void setTranscriptionJobName(String transcriptionJobName) {
            this.transcriptionJobName = transcriptionJobName;
        }

        @Override
        @Transient
        public final Builder transcriptionJobName(String transcriptionJobName) {
            this.transcriptionJobName = transcriptionJobName;
            return this;
        }

        public final String getLanguageCode() {
            return languageCode;
        }

        public final void setLanguageCode(String languageCode) {
            this.languageCode = languageCode;
        }

        @Override
        @Transient
        public final Builder languageCode(String languageCode) {
            this.languageCode = languageCode;
            return this;
        }

        @Override
        @Transient
        public final Builder languageCode(LanguageCode languageCode) {
            this.languageCode(languageCode == null ? null : languageCode.toString());
            return this;
        }

        public final Integer getMediaSampleRateHertz() {
            return mediaSampleRateHertz;
        }

        public final void setMediaSampleRateHertz(Integer mediaSampleRateHertz) {
            this.mediaSampleRateHertz = mediaSampleRateHertz;
        }

        @Override
        @Transient
        public final Builder mediaSampleRateHertz(Integer mediaSampleRateHertz) {
            this.mediaSampleRateHertz = mediaSampleRateHertz;
            return this;
        }

        public final String getMediaFormat() {
            return mediaFormat;
        }

        public final void setMediaFormat(String mediaFormat) {
            this.mediaFormat = mediaFormat;
        }

        @Override
        @Transient
        public final Builder mediaFormat(String mediaFormat) {
            this.mediaFormat = mediaFormat;
            return this;
        }

        @Override
        @Transient
        public final Builder mediaFormat(MediaFormat mediaFormat) {
            this.mediaFormat(mediaFormat == null ? null : mediaFormat.toString());
            return this;
        }

        public final Media.Builder getMedia() {
            return media != null ? media.toBuilder() : null;
        }

        public final void setMedia(Media.BuilderImpl media) {
            this.media = media != null ? media.build() : null;
        }

        @Override
        @Transient
        public final Builder media(Media media) {
            this.media = media;
            return this;
        }

        public final String getOutputBucketName() {
            return outputBucketName;
        }

        public final void setOutputBucketName(String outputBucketName) {
            this.outputBucketName = outputBucketName;
        }

        @Override
        @Transient
        public final Builder outputBucketName(String outputBucketName) {
            this.outputBucketName = outputBucketName;
            return this;
        }

        public final String getOutputKey() {
            return outputKey;
        }

        public final void setOutputKey(String outputKey) {
            this.outputKey = outputKey;
        }

        @Override
        @Transient
        public final Builder outputKey(String outputKey) {
            this.outputKey = outputKey;
            return this;
        }

        public final String getOutputEncryptionKMSKeyId() {
            return outputEncryptionKMSKeyId;
        }

        public final void setOutputEncryptionKMSKeyId(String outputEncryptionKMSKeyId) {
            this.outputEncryptionKMSKeyId = outputEncryptionKMSKeyId;
        }

        @Override
        @Transient
        public final Builder outputEncryptionKMSKeyId(String outputEncryptionKMSKeyId) {
            this.outputEncryptionKMSKeyId = outputEncryptionKMSKeyId;
            return this;
        }

        public final Map<String, String> getKmsEncryptionContext() {
            if (kmsEncryptionContext instanceof SdkAutoConstructMap) {
                return null;
            }
            return kmsEncryptionContext;
        }

        public final void setKmsEncryptionContext(Map<String, String> kmsEncryptionContext) {
            this.kmsEncryptionContext = KMSEncryptionContextMapCopier.copy(kmsEncryptionContext);
        }

        @Override
        @Transient
        public final Builder kmsEncryptionContext(Map<String, String> kmsEncryptionContext) {
            this.kmsEncryptionContext = KMSEncryptionContextMapCopier.copy(kmsEncryptionContext);
            return this;
        }

        public final Settings.Builder getSettings() {
            return settings != null ? settings.toBuilder() : null;
        }

        public final void setSettings(Settings.BuilderImpl settings) {
            this.settings = settings != null ? settings.build() : null;
        }

        @Override
        @Transient
        public final Builder settings(Settings settings) {
            this.settings = settings;
            return this;
        }

        public final ModelSettings.Builder getModelSettings() {
            return modelSettings != null ? modelSettings.toBuilder() : null;
        }

        public final void setModelSettings(ModelSettings.BuilderImpl modelSettings) {
            this.modelSettings = modelSettings != null ? modelSettings.build() : null;
        }

        @Override
        @Transient
        public final Builder modelSettings(ModelSettings modelSettings) {
            this.modelSettings = modelSettings;
            return this;
        }

        public final JobExecutionSettings.Builder getJobExecutionSettings() {
            return jobExecutionSettings != null ? jobExecutionSettings.toBuilder() : null;
        }

        public final void setJobExecutionSettings(JobExecutionSettings.BuilderImpl jobExecutionSettings) {
            this.jobExecutionSettings = jobExecutionSettings != null ? jobExecutionSettings.build() : null;
        }

        @Override
        @Transient
        public final Builder jobExecutionSettings(JobExecutionSettings jobExecutionSettings) {
            this.jobExecutionSettings = jobExecutionSettings;
            return this;
        }

        public final ContentRedaction.Builder getContentRedaction() {
            return contentRedaction != null ? contentRedaction.toBuilder() : null;
        }

        public final void setContentRedaction(ContentRedaction.BuilderImpl contentRedaction) {
            this.contentRedaction = contentRedaction != null ? contentRedaction.build() : null;
        }

        @Override
        @Transient
        public final Builder contentRedaction(ContentRedaction contentRedaction) {
            this.contentRedaction = contentRedaction;
            return this;
        }

        public final Boolean getIdentifyLanguage() {
            return identifyLanguage;
        }

        public final void setIdentifyLanguage(Boolean identifyLanguage) {
            this.identifyLanguage = identifyLanguage;
        }

        @Override
        @Transient
        public final Builder identifyLanguage(Boolean identifyLanguage) {
            this.identifyLanguage = identifyLanguage;
            return this;
        }

        public final Collection<String> getLanguageOptions() {
            if (languageOptions instanceof SdkAutoConstructList) {
                return null;
            }
            return languageOptions;
        }

        public final void setLanguageOptions(Collection<String> languageOptions) {
            this.languageOptions = LanguageOptionsCopier.copy(languageOptions);
        }

        @Override
        @Transient
        public final Builder languageOptionsWithStrings(Collection<String> languageOptions) {
            this.languageOptions = LanguageOptionsCopier.copy(languageOptions);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder languageOptionsWithStrings(String... languageOptions) {
            languageOptionsWithStrings(Arrays.asList(languageOptions));
            return this;
        }

        @Override
        @Transient
        public final Builder languageOptions(Collection<LanguageCode> languageOptions) {
            this.languageOptions = LanguageOptionsCopier.copyEnumToString(languageOptions);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder languageOptions(LanguageCode... languageOptions) {
            languageOptions(Arrays.asList(languageOptions));
            return this;
        }

        public final Subtitles.Builder getSubtitles() {
            return subtitles != null ? subtitles.toBuilder() : null;
        }

        public final void setSubtitles(Subtitles.BuilderImpl subtitles) {
            this.subtitles = subtitles != null ? subtitles.build() : null;
        }

        @Override
        @Transient
        public final Builder subtitles(Subtitles subtitles) {
            this.subtitles = subtitles;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        @Transient
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Map<String, LanguageIdSettings.Builder> getLanguageIdSettings() {
            Map<String, LanguageIdSettings.Builder> result = LanguageIdSettingsMapCopier.copyToBuilder(this.languageIdSettings);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setLanguageIdSettings(Map<String, LanguageIdSettings.BuilderImpl> languageIdSettings) {
            this.languageIdSettings = LanguageIdSettingsMapCopier.copyFromBuilder(languageIdSettings);
        }

        @Override
        @Transient
        public final Builder languageIdSettingsWithStrings(Map<String, LanguageIdSettings> languageIdSettings) {
            this.languageIdSettings = LanguageIdSettingsMapCopier.copy(languageIdSettings);
            return this;
        }

        @Override
        @Transient
        public final Builder languageIdSettings(Map<LanguageCode, LanguageIdSettings> languageIdSettings) {
            this.languageIdSettings = LanguageIdSettingsMapCopier.copyEnumToString(languageIdSettings);
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public StartTranscriptionJobRequest build() {
            return new StartTranscriptionJobRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
