/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribe.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * If using automatic language identification (<code>IdentifyLanguage</code>) in your request and you want to apply a
 * custom language model, a custom vocabulary, or a custom vocabulary filter, include <code>LanguageIdSettings</code>
 * with the relevant sub-parameters (<code>VocabularyName</code>, <code>LanguageModelName</code>, and
 * <code>VocabularyFilterName</code>).
 * </p>
 * <p>
 * You can specify two or more language codes that represent the languages you think may be present in your media;
 * including more than five is not recommended. Each language code you include can have an associated custom language
 * model, custom vocabulary, and custom vocabulary filter. The languages you specify must match the languages of the
 * specified custom language models, custom vocabularies, and custom vocabulary filters.
 * </p>
 * <p>
 * To include language options using <code>IdentifyLanguage</code> <b>without</b> including a custom language model, a
 * custom vocabulary, or a custom vocabulary filter, use <code>LanguageOptions</code> instead of
 * <code>LanguageIdSettings</code>. Including language options can improve the accuracy of automatic language
 * identification.
 * </p>
 * <p>
 * If you want to include a custom language model with your request but <b>do not</b> want to use automatic language
 * identification, use instead the <code/> parameter with the <code>LanguageModelName</code> sub-parameter.
 * </p>
 * <p>
 * If you want to include a custom vocabulary or a custom vocabulary filter (or both) with your request but <b>do
 * not</b> want to use automatic language identification, use instead the
 * <code/> parameter with the <code>VocabularyName</code> or <code>VocabularyFilterName</code> (or both) sub-parameter.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LanguageIdSettings implements SdkPojo, Serializable,
        ToCopyableBuilder<LanguageIdSettings.Builder, LanguageIdSettings> {
    private static final SdkField<String> VOCABULARY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VocabularyName").getter(getter(LanguageIdSettings::vocabularyName))
            .setter(setter(Builder::vocabularyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VocabularyName").build()).build();

    private static final SdkField<String> VOCABULARY_FILTER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VocabularyFilterName").getter(getter(LanguageIdSettings::vocabularyFilterName))
            .setter(setter(Builder::vocabularyFilterName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VocabularyFilterName").build())
            .build();

    private static final SdkField<String> LANGUAGE_MODEL_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LanguageModelName").getter(getter(LanguageIdSettings::languageModelName))
            .setter(setter(Builder::languageModelName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageModelName").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VOCABULARY_NAME_FIELD,
            VOCABULARY_FILTER_NAME_FIELD, LANGUAGE_MODEL_NAME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String vocabularyName;

    private final String vocabularyFilterName;

    private final String languageModelName;

    private LanguageIdSettings(BuilderImpl builder) {
        this.vocabularyName = builder.vocabularyName;
        this.vocabularyFilterName = builder.vocabularyFilterName;
        this.languageModelName = builder.languageModelName;
    }

    /**
     * <p>
     * The name of the custom vocabulary you want to use when processing your transcription job. Vocabulary names are
     * case sensitive.
     * </p>
     * <p>
     * The language of the specified vocabulary must match the language code you specify in your transcription request.
     * If the languages don't match, the vocabulary isn't applied. There are no errors or warnings associated with a
     * language mismatch.
     * </p>
     * 
     * @return The name of the custom vocabulary you want to use when processing your transcription job. Vocabulary
     *         names are case sensitive.</p>
     *         <p>
     *         The language of the specified vocabulary must match the language code you specify in your transcription
     *         request. If the languages don't match, the vocabulary isn't applied. There are no errors or warnings
     *         associated with a language mismatch.
     */
    public final String vocabularyName() {
        return vocabularyName;
    }

    /**
     * <p>
     * The name of the custom vocabulary filter you want to use when processing your transcription job. Vocabulary
     * filter names are case sensitive.
     * </p>
     * <p>
     * The language of the specified vocabulary filter must match the language code you specify in your transcription
     * request. If the languages don't match, the vocabulary filter isn't applied. There are no errors or warnings
     * associated with a language mismatch.
     * </p>
     * <p>
     * Note that if you include <code>VocabularyFilterName</code> in your request, you must also include
     * <code>VocabularyFilterMethod</code>.
     * </p>
     * 
     * @return The name of the custom vocabulary filter you want to use when processing your transcription job.
     *         Vocabulary filter names are case sensitive.</p>
     *         <p>
     *         The language of the specified vocabulary filter must match the language code you specify in your
     *         transcription request. If the languages don't match, the vocabulary filter isn't applied. There are no
     *         errors or warnings associated with a language mismatch.
     *         </p>
     *         <p>
     *         Note that if you include <code>VocabularyFilterName</code> in your request, you must also include
     *         <code>VocabularyFilterMethod</code>.
     */
    public final String vocabularyFilterName() {
        return vocabularyFilterName;
    }

    /**
     * <p>
     * The name of the custom language model you want to use when processing your transcription job. Note that language
     * model names are case sensitive.
     * </p>
     * <p>
     * The language of the specified language model must match the language code you specify in your transcription
     * request. If the languages don't match, the language model isn't applied. There are no errors or warnings
     * associated with a language mismatch.
     * </p>
     * 
     * @return The name of the custom language model you want to use when processing your transcription job. Note that
     *         language model names are case sensitive.</p>
     *         <p>
     *         The language of the specified language model must match the language code you specify in your
     *         transcription request. If the languages don't match, the language model isn't applied. There are no
     *         errors or warnings associated with a language mismatch.
     */
    public final String languageModelName() {
        return languageModelName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(vocabularyName());
        hashCode = 31 * hashCode + Objects.hashCode(vocabularyFilterName());
        hashCode = 31 * hashCode + Objects.hashCode(languageModelName());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LanguageIdSettings)) {
            return false;
        }
        LanguageIdSettings other = (LanguageIdSettings) obj;
        return Objects.equals(vocabularyName(), other.vocabularyName())
                && Objects.equals(vocabularyFilterName(), other.vocabularyFilterName())
                && Objects.equals(languageModelName(), other.languageModelName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LanguageIdSettings").add("VocabularyName", vocabularyName())
                .add("VocabularyFilterName", vocabularyFilterName()).add("LanguageModelName", languageModelName()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VocabularyName":
            return Optional.ofNullable(clazz.cast(vocabularyName()));
        case "VocabularyFilterName":
            return Optional.ofNullable(clazz.cast(vocabularyFilterName()));
        case "LanguageModelName":
            return Optional.ofNullable(clazz.cast(languageModelName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LanguageIdSettings, T> g) {
        return obj -> g.apply((LanguageIdSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LanguageIdSettings> {
        /**
         * <p>
         * The name of the custom vocabulary you want to use when processing your transcription job. Vocabulary names
         * are case sensitive.
         * </p>
         * <p>
         * The language of the specified vocabulary must match the language code you specify in your transcription
         * request. If the languages don't match, the vocabulary isn't applied. There are no errors or warnings
         * associated with a language mismatch.
         * </p>
         * 
         * @param vocabularyName
         *        The name of the custom vocabulary you want to use when processing your transcription job. Vocabulary
         *        names are case sensitive.</p>
         *        <p>
         *        The language of the specified vocabulary must match the language code you specify in your
         *        transcription request. If the languages don't match, the vocabulary isn't applied. There are no errors
         *        or warnings associated with a language mismatch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vocabularyName(String vocabularyName);

        /**
         * <p>
         * The name of the custom vocabulary filter you want to use when processing your transcription job. Vocabulary
         * filter names are case sensitive.
         * </p>
         * <p>
         * The language of the specified vocabulary filter must match the language code you specify in your
         * transcription request. If the languages don't match, the vocabulary filter isn't applied. There are no errors
         * or warnings associated with a language mismatch.
         * </p>
         * <p>
         * Note that if you include <code>VocabularyFilterName</code> in your request, you must also include
         * <code>VocabularyFilterMethod</code>.
         * </p>
         * 
         * @param vocabularyFilterName
         *        The name of the custom vocabulary filter you want to use when processing your transcription job.
         *        Vocabulary filter names are case sensitive.</p>
         *        <p>
         *        The language of the specified vocabulary filter must match the language code you specify in your
         *        transcription request. If the languages don't match, the vocabulary filter isn't applied. There are no
         *        errors or warnings associated with a language mismatch.
         *        </p>
         *        <p>
         *        Note that if you include <code>VocabularyFilterName</code> in your request, you must also include
         *        <code>VocabularyFilterMethod</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vocabularyFilterName(String vocabularyFilterName);

        /**
         * <p>
         * The name of the custom language model you want to use when processing your transcription job. Note that
         * language model names are case sensitive.
         * </p>
         * <p>
         * The language of the specified language model must match the language code you specify in your transcription
         * request. If the languages don't match, the language model isn't applied. There are no errors or warnings
         * associated with a language mismatch.
         * </p>
         * 
         * @param languageModelName
         *        The name of the custom language model you want to use when processing your transcription job. Note
         *        that language model names are case sensitive.</p>
         *        <p>
         *        The language of the specified language model must match the language code you specify in your
         *        transcription request. If the languages don't match, the language model isn't applied. There are no
         *        errors or warnings associated with a language mismatch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageModelName(String languageModelName);
    }

    static final class BuilderImpl implements Builder {
        private String vocabularyName;

        private String vocabularyFilterName;

        private String languageModelName;

        private BuilderImpl() {
        }

        private BuilderImpl(LanguageIdSettings model) {
            vocabularyName(model.vocabularyName);
            vocabularyFilterName(model.vocabularyFilterName);
            languageModelName(model.languageModelName);
        }

        public final String getVocabularyName() {
            return vocabularyName;
        }

        public final void setVocabularyName(String vocabularyName) {
            this.vocabularyName = vocabularyName;
        }

        @Override
        public final Builder vocabularyName(String vocabularyName) {
            this.vocabularyName = vocabularyName;
            return this;
        }

        public final String getVocabularyFilterName() {
            return vocabularyFilterName;
        }

        public final void setVocabularyFilterName(String vocabularyFilterName) {
            this.vocabularyFilterName = vocabularyFilterName;
        }

        @Override
        public final Builder vocabularyFilterName(String vocabularyFilterName) {
            this.vocabularyFilterName = vocabularyFilterName;
            return this;
        }

        public final String getLanguageModelName() {
            return languageModelName;
        }

        public final void setLanguageModelName(String languageModelName) {
            this.languageModelName = languageModelName;
        }

        @Override
        public final Builder languageModelName(String languageModelName) {
            this.languageModelName = languageModelName;
            return this;
        }

        @Override
        public LanguageIdSettings build() {
            return new LanguageIdSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
