/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribe.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides optional settings for the <code>CallAnalyticsJob</code> operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CallAnalyticsJobSettings implements SdkPojo, Serializable,
        ToCopyableBuilder<CallAnalyticsJobSettings.Builder, CallAnalyticsJobSettings> {
    private static final SdkField<String> VOCABULARY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VocabularyName").getter(getter(CallAnalyticsJobSettings::vocabularyName))
            .setter(setter(Builder::vocabularyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VocabularyName").build()).build();

    private static final SdkField<String> VOCABULARY_FILTER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VocabularyFilterName").getter(getter(CallAnalyticsJobSettings::vocabularyFilterName))
            .setter(setter(Builder::vocabularyFilterName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VocabularyFilterName").build())
            .build();

    private static final SdkField<String> VOCABULARY_FILTER_METHOD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VocabularyFilterMethod").getter(getter(CallAnalyticsJobSettings::vocabularyFilterMethodAsString))
            .setter(setter(Builder::vocabularyFilterMethod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VocabularyFilterMethod").build())
            .build();

    private static final SdkField<String> LANGUAGE_MODEL_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LanguageModelName").getter(getter(CallAnalyticsJobSettings::languageModelName))
            .setter(setter(Builder::languageModelName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageModelName").build()).build();

    private static final SdkField<ContentRedaction> CONTENT_REDACTION_FIELD = SdkField
            .<ContentRedaction> builder(MarshallingType.SDK_POJO).memberName("ContentRedaction")
            .getter(getter(CallAnalyticsJobSettings::contentRedaction)).setter(setter(Builder::contentRedaction))
            .constructor(ContentRedaction::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContentRedaction").build()).build();

    private static final SdkField<List<String>> LANGUAGE_OPTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("LanguageOptions")
            .getter(getter(CallAnalyticsJobSettings::languageOptionsAsStrings))
            .setter(setter(Builder::languageOptionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageOptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VOCABULARY_NAME_FIELD,
            VOCABULARY_FILTER_NAME_FIELD, VOCABULARY_FILTER_METHOD_FIELD, LANGUAGE_MODEL_NAME_FIELD, CONTENT_REDACTION_FIELD,
            LANGUAGE_OPTIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String vocabularyName;

    private final String vocabularyFilterName;

    private final String vocabularyFilterMethod;

    private final String languageModelName;

    private final ContentRedaction contentRedaction;

    private final List<String> languageOptions;

    private CallAnalyticsJobSettings(BuilderImpl builder) {
        this.vocabularyName = builder.vocabularyName;
        this.vocabularyFilterName = builder.vocabularyFilterName;
        this.vocabularyFilterMethod = builder.vocabularyFilterMethod;
        this.languageModelName = builder.languageModelName;
        this.contentRedaction = builder.contentRedaction;
        this.languageOptions = builder.languageOptions;
    }

    /**
     * <p>
     * The name of a vocabulary to use when processing the call analytics job.
     * </p>
     * 
     * @return The name of a vocabulary to use when processing the call analytics job.
     */
    public final String vocabularyName() {
        return vocabularyName;
    }

    /**
     * <p>
     * The name of the vocabulary filter to use when running a call analytics job. The filter that you specify must have
     * the same language code as the analytics job.
     * </p>
     * 
     * @return The name of the vocabulary filter to use when running a call analytics job. The filter that you specify
     *         must have the same language code as the analytics job.
     */
    public final String vocabularyFilterName() {
        return vocabularyFilterName;
    }

    /**
     * <p>
     * Set to mask to remove filtered text from the transcript and replace it with three asterisks ("***") as
     * placeholder text. Set to <code>remove</code> to remove filtered text from the transcript without using
     * placeholder text. Set to <code>tag</code> to mark the word in the transcription output that matches the
     * vocabulary filter. When you set the filter method to <code>tag</code>, the words matching your vocabulary filter
     * are not masked or removed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #vocabularyFilterMethod} will return {@link VocabularyFilterMethod#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #vocabularyFilterMethodAsString}.
     * </p>
     * 
     * @return Set to mask to remove filtered text from the transcript and replace it with three asterisks ("***") as
     *         placeholder text. Set to <code>remove</code> to remove filtered text from the transcript without using
     *         placeholder text. Set to <code>tag</code> to mark the word in the transcription output that matches the
     *         vocabulary filter. When you set the filter method to <code>tag</code>, the words matching your vocabulary
     *         filter are not masked or removed.
     * @see VocabularyFilterMethod
     */
    public final VocabularyFilterMethod vocabularyFilterMethod() {
        return VocabularyFilterMethod.fromValue(vocabularyFilterMethod);
    }

    /**
     * <p>
     * Set to mask to remove filtered text from the transcript and replace it with three asterisks ("***") as
     * placeholder text. Set to <code>remove</code> to remove filtered text from the transcript without using
     * placeholder text. Set to <code>tag</code> to mark the word in the transcription output that matches the
     * vocabulary filter. When you set the filter method to <code>tag</code>, the words matching your vocabulary filter
     * are not masked or removed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #vocabularyFilterMethod} will return {@link VocabularyFilterMethod#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #vocabularyFilterMethodAsString}.
     * </p>
     * 
     * @return Set to mask to remove filtered text from the transcript and replace it with three asterisks ("***") as
     *         placeholder text. Set to <code>remove</code> to remove filtered text from the transcript without using
     *         placeholder text. Set to <code>tag</code> to mark the word in the transcription output that matches the
     *         vocabulary filter. When you set the filter method to <code>tag</code>, the words matching your vocabulary
     *         filter are not masked or removed.
     * @see VocabularyFilterMethod
     */
    public final String vocabularyFilterMethodAsString() {
        return vocabularyFilterMethod;
    }

    /**
     * <p>
     * The structure used to describe a custom language model.
     * </p>
     * 
     * @return The structure used to describe a custom language model.
     */
    public final String languageModelName() {
        return languageModelName;
    }

    /**
     * Returns the value of the ContentRedaction property for this object.
     * 
     * @return The value of the ContentRedaction property for this object.
     */
    public final ContentRedaction contentRedaction() {
        return contentRedaction;
    }

    /**
     * <p>
     * When you run a call analytics job, you can specify the language spoken in the audio, or you can have Amazon
     * Transcribe identify the language for you.
     * </p>
     * <p>
     * To specify a language, specify an array with one language code. If you don't know the language, you can leave
     * this field blank and Amazon Transcribe will use machine learning to identify the language for you. To improve the
     * ability of Amazon Transcribe to correctly identify the language, you can provide an array of the languages that
     * can be present in the audio. Refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/how-it-works.html">Supported languages and
     * language-specific features</a> for additional information.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLanguageOptions} method.
     * </p>
     * 
     * @return When you run a call analytics job, you can specify the language spoken in the audio, or you can have
     *         Amazon Transcribe identify the language for you.</p>
     *         <p>
     *         To specify a language, specify an array with one language code. If you don't know the language, you can
     *         leave this field blank and Amazon Transcribe will use machine learning to identify the language for you.
     *         To improve the ability of Amazon Transcribe to correctly identify the language, you can provide an array
     *         of the languages that can be present in the audio. Refer to <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/dg/how-it-works.html">Supported languages and
     *         language-specific features</a> for additional information.
     */
    public final List<LanguageCode> languageOptions() {
        return LanguageOptionsCopier.copyStringToEnum(languageOptions);
    }

    /**
     * For responses, this returns true if the service returned a value for the LanguageOptions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLanguageOptions() {
        return languageOptions != null && !(languageOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * When you run a call analytics job, you can specify the language spoken in the audio, or you can have Amazon
     * Transcribe identify the language for you.
     * </p>
     * <p>
     * To specify a language, specify an array with one language code. If you don't know the language, you can leave
     * this field blank and Amazon Transcribe will use machine learning to identify the language for you. To improve the
     * ability of Amazon Transcribe to correctly identify the language, you can provide an array of the languages that
     * can be present in the audio. Refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/how-it-works.html">Supported languages and
     * language-specific features</a> for additional information.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLanguageOptions} method.
     * </p>
     * 
     * @return When you run a call analytics job, you can specify the language spoken in the audio, or you can have
     *         Amazon Transcribe identify the language for you.</p>
     *         <p>
     *         To specify a language, specify an array with one language code. If you don't know the language, you can
     *         leave this field blank and Amazon Transcribe will use machine learning to identify the language for you.
     *         To improve the ability of Amazon Transcribe to correctly identify the language, you can provide an array
     *         of the languages that can be present in the audio. Refer to <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/dg/how-it-works.html">Supported languages and
     *         language-specific features</a> for additional information.
     */
    public final List<String> languageOptionsAsStrings() {
        return languageOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(vocabularyName());
        hashCode = 31 * hashCode + Objects.hashCode(vocabularyFilterName());
        hashCode = 31 * hashCode + Objects.hashCode(vocabularyFilterMethodAsString());
        hashCode = 31 * hashCode + Objects.hashCode(languageModelName());
        hashCode = 31 * hashCode + Objects.hashCode(contentRedaction());
        hashCode = 31 * hashCode + Objects.hashCode(hasLanguageOptions() ? languageOptionsAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CallAnalyticsJobSettings)) {
            return false;
        }
        CallAnalyticsJobSettings other = (CallAnalyticsJobSettings) obj;
        return Objects.equals(vocabularyName(), other.vocabularyName())
                && Objects.equals(vocabularyFilterName(), other.vocabularyFilterName())
                && Objects.equals(vocabularyFilterMethodAsString(), other.vocabularyFilterMethodAsString())
                && Objects.equals(languageModelName(), other.languageModelName())
                && Objects.equals(contentRedaction(), other.contentRedaction())
                && hasLanguageOptions() == other.hasLanguageOptions()
                && Objects.equals(languageOptionsAsStrings(), other.languageOptionsAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CallAnalyticsJobSettings").add("VocabularyName", vocabularyName())
                .add("VocabularyFilterName", vocabularyFilterName())
                .add("VocabularyFilterMethod", vocabularyFilterMethodAsString()).add("LanguageModelName", languageModelName())
                .add("ContentRedaction", contentRedaction())
                .add("LanguageOptions", hasLanguageOptions() ? languageOptionsAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VocabularyName":
            return Optional.ofNullable(clazz.cast(vocabularyName()));
        case "VocabularyFilterName":
            return Optional.ofNullable(clazz.cast(vocabularyFilterName()));
        case "VocabularyFilterMethod":
            return Optional.ofNullable(clazz.cast(vocabularyFilterMethodAsString()));
        case "LanguageModelName":
            return Optional.ofNullable(clazz.cast(languageModelName()));
        case "ContentRedaction":
            return Optional.ofNullable(clazz.cast(contentRedaction()));
        case "LanguageOptions":
            return Optional.ofNullable(clazz.cast(languageOptionsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CallAnalyticsJobSettings, T> g) {
        return obj -> g.apply((CallAnalyticsJobSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CallAnalyticsJobSettings> {
        /**
         * <p>
         * The name of a vocabulary to use when processing the call analytics job.
         * </p>
         * 
         * @param vocabularyName
         *        The name of a vocabulary to use when processing the call analytics job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vocabularyName(String vocabularyName);

        /**
         * <p>
         * The name of the vocabulary filter to use when running a call analytics job. The filter that you specify must
         * have the same language code as the analytics job.
         * </p>
         * 
         * @param vocabularyFilterName
         *        The name of the vocabulary filter to use when running a call analytics job. The filter that you
         *        specify must have the same language code as the analytics job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vocabularyFilterName(String vocabularyFilterName);

        /**
         * <p>
         * Set to mask to remove filtered text from the transcript and replace it with three asterisks ("***") as
         * placeholder text. Set to <code>remove</code> to remove filtered text from the transcript without using
         * placeholder text. Set to <code>tag</code> to mark the word in the transcription output that matches the
         * vocabulary filter. When you set the filter method to <code>tag</code>, the words matching your vocabulary
         * filter are not masked or removed.
         * </p>
         * 
         * @param vocabularyFilterMethod
         *        Set to mask to remove filtered text from the transcript and replace it with three asterisks ("***") as
         *        placeholder text. Set to <code>remove</code> to remove filtered text from the transcript without using
         *        placeholder text. Set to <code>tag</code> to mark the word in the transcription output that matches
         *        the vocabulary filter. When you set the filter method to <code>tag</code>, the words matching your
         *        vocabulary filter are not masked or removed.
         * @see VocabularyFilterMethod
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VocabularyFilterMethod
         */
        Builder vocabularyFilterMethod(String vocabularyFilterMethod);

        /**
         * <p>
         * Set to mask to remove filtered text from the transcript and replace it with three asterisks ("***") as
         * placeholder text. Set to <code>remove</code> to remove filtered text from the transcript without using
         * placeholder text. Set to <code>tag</code> to mark the word in the transcription output that matches the
         * vocabulary filter. When you set the filter method to <code>tag</code>, the words matching your vocabulary
         * filter are not masked or removed.
         * </p>
         * 
         * @param vocabularyFilterMethod
         *        Set to mask to remove filtered text from the transcript and replace it with three asterisks ("***") as
         *        placeholder text. Set to <code>remove</code> to remove filtered text from the transcript without using
         *        placeholder text. Set to <code>tag</code> to mark the word in the transcription output that matches
         *        the vocabulary filter. When you set the filter method to <code>tag</code>, the words matching your
         *        vocabulary filter are not masked or removed.
         * @see VocabularyFilterMethod
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VocabularyFilterMethod
         */
        Builder vocabularyFilterMethod(VocabularyFilterMethod vocabularyFilterMethod);

        /**
         * <p>
         * The structure used to describe a custom language model.
         * </p>
         * 
         * @param languageModelName
         *        The structure used to describe a custom language model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageModelName(String languageModelName);

        /**
         * Sets the value of the ContentRedaction property for this object.
         *
         * @param contentRedaction
         *        The new value for the ContentRedaction property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contentRedaction(ContentRedaction contentRedaction);

        /**
         * Sets the value of the ContentRedaction property for this object.
         *
         * This is a convenience that creates an instance of the {@link ContentRedaction.Builder} avoiding the need to
         * create one manually via {@link ContentRedaction#builder()}.
         *
         * When the {@link Consumer} completes, {@link ContentRedaction.Builder#build()} is called immediately and its
         * result is passed to {@link #contentRedaction(ContentRedaction)}.
         * 
         * @param contentRedaction
         *        a consumer that will call methods on {@link ContentRedaction.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #contentRedaction(ContentRedaction)
         */
        default Builder contentRedaction(Consumer<ContentRedaction.Builder> contentRedaction) {
            return contentRedaction(ContentRedaction.builder().applyMutation(contentRedaction).build());
        }

        /**
         * <p>
         * When you run a call analytics job, you can specify the language spoken in the audio, or you can have Amazon
         * Transcribe identify the language for you.
         * </p>
         * <p>
         * To specify a language, specify an array with one language code. If you don't know the language, you can leave
         * this field blank and Amazon Transcribe will use machine learning to identify the language for you. To improve
         * the ability of Amazon Transcribe to correctly identify the language, you can provide an array of the
         * languages that can be present in the audio. Refer to <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/how-it-works.html">Supported languages and
         * language-specific features</a> for additional information.
         * </p>
         * 
         * @param languageOptions
         *        When you run a call analytics job, you can specify the language spoken in the audio, or you can have
         *        Amazon Transcribe identify the language for you.</p>
         *        <p>
         *        To specify a language, specify an array with one language code. If you don't know the language, you
         *        can leave this field blank and Amazon Transcribe will use machine learning to identify the language
         *        for you. To improve the ability of Amazon Transcribe to correctly identify the language, you can
         *        provide an array of the languages that can be present in the audio. Refer to <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/how-it-works.html">Supported languages and
         *        language-specific features</a> for additional information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageOptionsWithStrings(Collection<String> languageOptions);

        /**
         * <p>
         * When you run a call analytics job, you can specify the language spoken in the audio, or you can have Amazon
         * Transcribe identify the language for you.
         * </p>
         * <p>
         * To specify a language, specify an array with one language code. If you don't know the language, you can leave
         * this field blank and Amazon Transcribe will use machine learning to identify the language for you. To improve
         * the ability of Amazon Transcribe to correctly identify the language, you can provide an array of the
         * languages that can be present in the audio. Refer to <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/how-it-works.html">Supported languages and
         * language-specific features</a> for additional information.
         * </p>
         * 
         * @param languageOptions
         *        When you run a call analytics job, you can specify the language spoken in the audio, or you can have
         *        Amazon Transcribe identify the language for you.</p>
         *        <p>
         *        To specify a language, specify an array with one language code. If you don't know the language, you
         *        can leave this field blank and Amazon Transcribe will use machine learning to identify the language
         *        for you. To improve the ability of Amazon Transcribe to correctly identify the language, you can
         *        provide an array of the languages that can be present in the audio. Refer to <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/how-it-works.html">Supported languages and
         *        language-specific features</a> for additional information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageOptionsWithStrings(String... languageOptions);

        /**
         * <p>
         * When you run a call analytics job, you can specify the language spoken in the audio, or you can have Amazon
         * Transcribe identify the language for you.
         * </p>
         * <p>
         * To specify a language, specify an array with one language code. If you don't know the language, you can leave
         * this field blank and Amazon Transcribe will use machine learning to identify the language for you. To improve
         * the ability of Amazon Transcribe to correctly identify the language, you can provide an array of the
         * languages that can be present in the audio. Refer to <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/how-it-works.html">Supported languages and
         * language-specific features</a> for additional information.
         * </p>
         * 
         * @param languageOptions
         *        When you run a call analytics job, you can specify the language spoken in the audio, or you can have
         *        Amazon Transcribe identify the language for you.</p>
         *        <p>
         *        To specify a language, specify an array with one language code. If you don't know the language, you
         *        can leave this field blank and Amazon Transcribe will use machine learning to identify the language
         *        for you. To improve the ability of Amazon Transcribe to correctly identify the language, you can
         *        provide an array of the languages that can be present in the audio. Refer to <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/how-it-works.html">Supported languages and
         *        language-specific features</a> for additional information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageOptions(Collection<LanguageCode> languageOptions);

        /**
         * <p>
         * When you run a call analytics job, you can specify the language spoken in the audio, or you can have Amazon
         * Transcribe identify the language for you.
         * </p>
         * <p>
         * To specify a language, specify an array with one language code. If you don't know the language, you can leave
         * this field blank and Amazon Transcribe will use machine learning to identify the language for you. To improve
         * the ability of Amazon Transcribe to correctly identify the language, you can provide an array of the
         * languages that can be present in the audio. Refer to <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/how-it-works.html">Supported languages and
         * language-specific features</a> for additional information.
         * </p>
         * 
         * @param languageOptions
         *        When you run a call analytics job, you can specify the language spoken in the audio, or you can have
         *        Amazon Transcribe identify the language for you.</p>
         *        <p>
         *        To specify a language, specify an array with one language code. If you don't know the language, you
         *        can leave this field blank and Amazon Transcribe will use machine learning to identify the language
         *        for you. To improve the ability of Amazon Transcribe to correctly identify the language, you can
         *        provide an array of the languages that can be present in the audio. Refer to <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/how-it-works.html">Supported languages and
         *        language-specific features</a> for additional information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageOptions(LanguageCode... languageOptions);
    }

    static final class BuilderImpl implements Builder {
        private String vocabularyName;

        private String vocabularyFilterName;

        private String vocabularyFilterMethod;

        private String languageModelName;

        private ContentRedaction contentRedaction;

        private List<String> languageOptions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CallAnalyticsJobSettings model) {
            vocabularyName(model.vocabularyName);
            vocabularyFilterName(model.vocabularyFilterName);
            vocabularyFilterMethod(model.vocabularyFilterMethod);
            languageModelName(model.languageModelName);
            contentRedaction(model.contentRedaction);
            languageOptionsWithStrings(model.languageOptions);
        }

        public final String getVocabularyName() {
            return vocabularyName;
        }

        public final void setVocabularyName(String vocabularyName) {
            this.vocabularyName = vocabularyName;
        }

        @Override
        @Transient
        public final Builder vocabularyName(String vocabularyName) {
            this.vocabularyName = vocabularyName;
            return this;
        }

        public final String getVocabularyFilterName() {
            return vocabularyFilterName;
        }

        public final void setVocabularyFilterName(String vocabularyFilterName) {
            this.vocabularyFilterName = vocabularyFilterName;
        }

        @Override
        @Transient
        public final Builder vocabularyFilterName(String vocabularyFilterName) {
            this.vocabularyFilterName = vocabularyFilterName;
            return this;
        }

        public final String getVocabularyFilterMethod() {
            return vocabularyFilterMethod;
        }

        public final void setVocabularyFilterMethod(String vocabularyFilterMethod) {
            this.vocabularyFilterMethod = vocabularyFilterMethod;
        }

        @Override
        @Transient
        public final Builder vocabularyFilterMethod(String vocabularyFilterMethod) {
            this.vocabularyFilterMethod = vocabularyFilterMethod;
            return this;
        }

        @Override
        @Transient
        public final Builder vocabularyFilterMethod(VocabularyFilterMethod vocabularyFilterMethod) {
            this.vocabularyFilterMethod(vocabularyFilterMethod == null ? null : vocabularyFilterMethod.toString());
            return this;
        }

        public final String getLanguageModelName() {
            return languageModelName;
        }

        public final void setLanguageModelName(String languageModelName) {
            this.languageModelName = languageModelName;
        }

        @Override
        @Transient
        public final Builder languageModelName(String languageModelName) {
            this.languageModelName = languageModelName;
            return this;
        }

        public final ContentRedaction.Builder getContentRedaction() {
            return contentRedaction != null ? contentRedaction.toBuilder() : null;
        }

        public final void setContentRedaction(ContentRedaction.BuilderImpl contentRedaction) {
            this.contentRedaction = contentRedaction != null ? contentRedaction.build() : null;
        }

        @Override
        @Transient
        public final Builder contentRedaction(ContentRedaction contentRedaction) {
            this.contentRedaction = contentRedaction;
            return this;
        }

        public final Collection<String> getLanguageOptions() {
            if (languageOptions instanceof SdkAutoConstructList) {
                return null;
            }
            return languageOptions;
        }

        public final void setLanguageOptions(Collection<String> languageOptions) {
            this.languageOptions = LanguageOptionsCopier.copy(languageOptions);
        }

        @Override
        @Transient
        public final Builder languageOptionsWithStrings(Collection<String> languageOptions) {
            this.languageOptions = LanguageOptionsCopier.copy(languageOptions);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder languageOptionsWithStrings(String... languageOptions) {
            languageOptionsWithStrings(Arrays.asList(languageOptions));
            return this;
        }

        @Override
        @Transient
        public final Builder languageOptions(Collection<LanguageCode> languageOptions) {
            this.languageOptions = LanguageOptionsCopier.copyEnumToString(languageOptions);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder languageOptions(LanguageCode... languageOptions) {
            languageOptions(Arrays.asList(languageOptions));
            return this;
        }

        @Override
        public CallAnalyticsJobSettings build() {
            return new CallAnalyticsJobSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
