/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribe.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an asynchronous analytics job that was created with the <code>StartAnalyticsJob</code> operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CallAnalyticsJob implements SdkPojo, Serializable,
        ToCopyableBuilder<CallAnalyticsJob.Builder, CallAnalyticsJob> {
    private static final SdkField<String> CALL_ANALYTICS_JOB_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CallAnalyticsJobName").getter(getter(CallAnalyticsJob::callAnalyticsJobName))
            .setter(setter(Builder::callAnalyticsJobName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CallAnalyticsJobName").build())
            .build();

    private static final SdkField<String> CALL_ANALYTICS_JOB_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CallAnalyticsJobStatus").getter(getter(CallAnalyticsJob::callAnalyticsJobStatusAsString))
            .setter(setter(Builder::callAnalyticsJobStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CallAnalyticsJobStatus").build())
            .build();

    private static final SdkField<String> LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LanguageCode").getter(getter(CallAnalyticsJob::languageCodeAsString))
            .setter(setter(Builder::languageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageCode").build()).build();

    private static final SdkField<Integer> MEDIA_SAMPLE_RATE_HERTZ_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MediaSampleRateHertz").getter(getter(CallAnalyticsJob::mediaSampleRateHertz))
            .setter(setter(Builder::mediaSampleRateHertz))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MediaSampleRateHertz").build())
            .build();

    private static final SdkField<String> MEDIA_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MediaFormat").getter(getter(CallAnalyticsJob::mediaFormatAsString)).setter(setter(Builder::mediaFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MediaFormat").build()).build();

    private static final SdkField<Media> MEDIA_FIELD = SdkField.<Media> builder(MarshallingType.SDK_POJO).memberName("Media")
            .getter(getter(CallAnalyticsJob::media)).setter(setter(Builder::media)).constructor(Media::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Media").build()).build();

    private static final SdkField<Transcript> TRANSCRIPT_FIELD = SdkField.<Transcript> builder(MarshallingType.SDK_POJO)
            .memberName("Transcript").getter(getter(CallAnalyticsJob::transcript)).setter(setter(Builder::transcript))
            .constructor(Transcript::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Transcript").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime").getter(getter(CallAnalyticsJob::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(CallAnalyticsJob::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<Instant> COMPLETION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CompletionTime").getter(getter(CallAnalyticsJob::completionTime))
            .setter(setter(Builder::completionTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletionTime").build()).build();

    private static final SdkField<String> FAILURE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FailureReason").getter(getter(CallAnalyticsJob::failureReason)).setter(setter(Builder::failureReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureReason").build()).build();

    private static final SdkField<String> DATA_ACCESS_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataAccessRoleArn").getter(getter(CallAnalyticsJob::dataAccessRoleArn))
            .setter(setter(Builder::dataAccessRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataAccessRoleArn").build()).build();

    private static final SdkField<Float> IDENTIFIED_LANGUAGE_SCORE_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("IdentifiedLanguageScore").getter(getter(CallAnalyticsJob::identifiedLanguageScore))
            .setter(setter(Builder::identifiedLanguageScore))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdentifiedLanguageScore").build())
            .build();

    private static final SdkField<CallAnalyticsJobSettings> SETTINGS_FIELD = SdkField
            .<CallAnalyticsJobSettings> builder(MarshallingType.SDK_POJO).memberName("Settings")
            .getter(getter(CallAnalyticsJob::settings)).setter(setter(Builder::settings))
            .constructor(CallAnalyticsJobSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Settings").build()).build();

    private static final SdkField<List<ChannelDefinition>> CHANNEL_DEFINITIONS_FIELD = SdkField
            .<List<ChannelDefinition>> builder(MarshallingType.LIST)
            .memberName("ChannelDefinitions")
            .getter(getter(CallAnalyticsJob::channelDefinitions))
            .setter(setter(Builder::channelDefinitions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChannelDefinitions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ChannelDefinition> builder(MarshallingType.SDK_POJO)
                                            .constructor(ChannelDefinition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CALL_ANALYTICS_JOB_NAME_FIELD,
            CALL_ANALYTICS_JOB_STATUS_FIELD, LANGUAGE_CODE_FIELD, MEDIA_SAMPLE_RATE_HERTZ_FIELD, MEDIA_FORMAT_FIELD, MEDIA_FIELD,
            TRANSCRIPT_FIELD, START_TIME_FIELD, CREATION_TIME_FIELD, COMPLETION_TIME_FIELD, FAILURE_REASON_FIELD,
            DATA_ACCESS_ROLE_ARN_FIELD, IDENTIFIED_LANGUAGE_SCORE_FIELD, SETTINGS_FIELD, CHANNEL_DEFINITIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String callAnalyticsJobName;

    private final String callAnalyticsJobStatus;

    private final String languageCode;

    private final Integer mediaSampleRateHertz;

    private final String mediaFormat;

    private final Media media;

    private final Transcript transcript;

    private final Instant startTime;

    private final Instant creationTime;

    private final Instant completionTime;

    private final String failureReason;

    private final String dataAccessRoleArn;

    private final Float identifiedLanguageScore;

    private final CallAnalyticsJobSettings settings;

    private final List<ChannelDefinition> channelDefinitions;

    private CallAnalyticsJob(BuilderImpl builder) {
        this.callAnalyticsJobName = builder.callAnalyticsJobName;
        this.callAnalyticsJobStatus = builder.callAnalyticsJobStatus;
        this.languageCode = builder.languageCode;
        this.mediaSampleRateHertz = builder.mediaSampleRateHertz;
        this.mediaFormat = builder.mediaFormat;
        this.media = builder.media;
        this.transcript = builder.transcript;
        this.startTime = builder.startTime;
        this.creationTime = builder.creationTime;
        this.completionTime = builder.completionTime;
        this.failureReason = builder.failureReason;
        this.dataAccessRoleArn = builder.dataAccessRoleArn;
        this.identifiedLanguageScore = builder.identifiedLanguageScore;
        this.settings = builder.settings;
        this.channelDefinitions = builder.channelDefinitions;
    }

    /**
     * <p>
     * The name of the call analytics job.
     * </p>
     * 
     * @return The name of the call analytics job.
     */
    public final String callAnalyticsJobName() {
        return callAnalyticsJobName;
    }

    /**
     * <p>
     * The status of the analytics job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #callAnalyticsJobStatus} will return {@link CallAnalyticsJobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #callAnalyticsJobStatusAsString}.
     * </p>
     * 
     * @return The status of the analytics job.
     * @see CallAnalyticsJobStatus
     */
    public final CallAnalyticsJobStatus callAnalyticsJobStatus() {
        return CallAnalyticsJobStatus.fromValue(callAnalyticsJobStatus);
    }

    /**
     * <p>
     * The status of the analytics job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #callAnalyticsJobStatus} will return {@link CallAnalyticsJobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #callAnalyticsJobStatusAsString}.
     * </p>
     * 
     * @return The status of the analytics job.
     * @see CallAnalyticsJobStatus
     */
    public final String callAnalyticsJobStatusAsString() {
        return callAnalyticsJobStatus;
    }

    /**
     * <p>
     * If you know the language spoken between the customer and the agent, specify a language code for this field.
     * </p>
     * <p>
     * If you don't know the language, you can leave this field blank, and Amazon Transcribe will use machine learning
     * to automatically identify the language. To improve the accuracy of language identification, you can provide an
     * array containing the possible language codes for the language spoken in your audio. Refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/how-it-works.html">Supported languages and
     * language-specific features</a> for additional information.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return If you know the language spoken between the customer and the agent, specify a language code for this
     *         field.</p>
     *         <p>
     *         If you don't know the language, you can leave this field blank, and Amazon Transcribe will use machine
     *         learning to automatically identify the language. To improve the accuracy of language identification, you
     *         can provide an array containing the possible language codes for the language spoken in your audio. Refer
     *         to <a href="https://docs.aws.amazon.com/transcribe/latest/dg/how-it-works.html">Supported languages and
     *         language-specific features</a> for additional information.
     * @see LanguageCode
     */
    public final LanguageCode languageCode() {
        return LanguageCode.fromValue(languageCode);
    }

    /**
     * <p>
     * If you know the language spoken between the customer and the agent, specify a language code for this field.
     * </p>
     * <p>
     * If you don't know the language, you can leave this field blank, and Amazon Transcribe will use machine learning
     * to automatically identify the language. To improve the accuracy of language identification, you can provide an
     * array containing the possible language codes for the language spoken in your audio. Refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/how-it-works.html">Supported languages and
     * language-specific features</a> for additional information.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return If you know the language spoken between the customer and the agent, specify a language code for this
     *         field.</p>
     *         <p>
     *         If you don't know the language, you can leave this field blank, and Amazon Transcribe will use machine
     *         learning to automatically identify the language. To improve the accuracy of language identification, you
     *         can provide an array containing the possible language codes for the language spoken in your audio. Refer
     *         to <a href="https://docs.aws.amazon.com/transcribe/latest/dg/how-it-works.html">Supported languages and
     *         language-specific features</a> for additional information.
     * @see LanguageCode
     */
    public final String languageCodeAsString() {
        return languageCode;
    }

    /**
     * <p>
     * The sample rate, in Hertz, of the audio.
     * </p>
     * 
     * @return The sample rate, in Hertz, of the audio.
     */
    public final Integer mediaSampleRateHertz() {
        return mediaSampleRateHertz;
    }

    /**
     * <p>
     * The format of the input audio file. Note: for call analytics jobs, only the following media formats are
     * supported: MP3, MP4, WAV, FLAC, OGG, and WebM.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mediaFormat} will
     * return {@link MediaFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #mediaFormatAsString}.
     * </p>
     * 
     * @return The format of the input audio file. Note: for call analytics jobs, only the following media formats are
     *         supported: MP3, MP4, WAV, FLAC, OGG, and WebM.
     * @see MediaFormat
     */
    public final MediaFormat mediaFormat() {
        return MediaFormat.fromValue(mediaFormat);
    }

    /**
     * <p>
     * The format of the input audio file. Note: for call analytics jobs, only the following media formats are
     * supported: MP3, MP4, WAV, FLAC, OGG, and WebM.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mediaFormat} will
     * return {@link MediaFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #mediaFormatAsString}.
     * </p>
     * 
     * @return The format of the input audio file. Note: for call analytics jobs, only the following media formats are
     *         supported: MP3, MP4, WAV, FLAC, OGG, and WebM.
     * @see MediaFormat
     */
    public final String mediaFormatAsString() {
        return mediaFormat;
    }

    /**
     * Returns the value of the Media property for this object.
     * 
     * @return The value of the Media property for this object.
     */
    public final Media media() {
        return media;
    }

    /**
     * Returns the value of the Transcript property for this object.
     * 
     * @return The value of the Transcript property for this object.
     */
    public final Transcript transcript() {
        return transcript;
    }

    /**
     * <p>
     * A timestamp that shows when the analytics job started processing.
     * </p>
     * 
     * @return A timestamp that shows when the analytics job started processing.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * A timestamp that shows when the analytics job was created.
     * </p>
     * 
     * @return A timestamp that shows when the analytics job was created.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * A timestamp that shows when the analytics job was completed.
     * </p>
     * 
     * @return A timestamp that shows when the analytics job was completed.
     */
    public final Instant completionTime() {
        return completionTime;
    }

    /**
     * <p>
     * If the <code>AnalyticsJobStatus</code> is <code>FAILED</code>, this field contains information about why the job
     * failed.
     * </p>
     * <p>
     * The <code>FailureReason</code> field can contain one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Unsupported media format</code>: The media format specified in the <code>MediaFormat</code> field of the
     * request isn't valid. See the description of the <code>MediaFormat</code> field for a list of valid values.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>The media format provided does not match the detected media format</code>: The media format of the audio
     * file doesn't match the format specified in the <code>MediaFormat</code> field in the request. Check the media
     * format of your media file and make sure the two values match.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Invalid sample rate for audio file</code>: The sample rate specified in the
     * <code>MediaSampleRateHertz</code> of the request isn't valid. The sample rate must be between 8,000 and 48,000
     * Hertz.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>The sample rate provided does not match the detected sample rate</code>: The sample rate in the audio file
     * doesn't match the sample rate specified in the <code>MediaSampleRateHertz</code> field in the request. Check the
     * sample rate of your media file and make sure that the two values match.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Invalid file size: file size too large</code>: The size of your audio file is larger than what Amazon
     * Transcribe Medical can process. For more information, see <i>Guidelines and Quotas</i> in the Amazon Transcribe
     * Medical Guide.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Invalid number of channels: number of channels too large</code>: Your audio contains more channels than
     * Amazon Transcribe Medical is configured to process. To request additional channels, see Amazon Transcribe Medical
     * Endpoints and Quotas in the <a href="https://docs.aws.amazon.com/general/latest/gr/Welcome.html">Amazon Web
     * Services General Reference</a>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return If the <code>AnalyticsJobStatus</code> is <code>FAILED</code>, this field contains information about why
     *         the job failed.</p>
     *         <p>
     *         The <code>FailureReason</code> field can contain one of the following values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Unsupported media format</code>: The media format specified in the <code>MediaFormat</code> field
     *         of the request isn't valid. See the description of the <code>MediaFormat</code> field for a list of valid
     *         values.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>The media format provided does not match the detected media format</code>: The media format of the
     *         audio file doesn't match the format specified in the <code>MediaFormat</code> field in the request. Check
     *         the media format of your media file and make sure the two values match.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Invalid sample rate for audio file</code>: The sample rate specified in the
     *         <code>MediaSampleRateHertz</code> of the request isn't valid. The sample rate must be between 8,000 and
     *         48,000 Hertz.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>The sample rate provided does not match the detected sample rate</code>: The sample rate in the
     *         audio file doesn't match the sample rate specified in the <code>MediaSampleRateHertz</code> field in the
     *         request. Check the sample rate of your media file and make sure that the two values match.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Invalid file size: file size too large</code>: The size of your audio file is larger than what
     *         Amazon Transcribe Medical can process. For more information, see <i>Guidelines and Quotas</i> in the
     *         Amazon Transcribe Medical Guide.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Invalid number of channels: number of channels too large</code>: Your audio contains more channels
     *         than Amazon Transcribe Medical is configured to process. To request additional channels, see Amazon
     *         Transcribe Medical Endpoints and Quotas in the <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/Welcome.html">Amazon Web Services General
     *         Reference</a>.
     *         </p>
     *         </li>
     */
    public final String failureReason() {
        return failureReason;
    }

    /**
     * <p>
     * The Amazon Resource Number (ARN) that you use to get access to the analytics job.
     * </p>
     * 
     * @return The Amazon Resource Number (ARN) that you use to get access to the analytics job.
     */
    public final String dataAccessRoleArn() {
        return dataAccessRoleArn;
    }

    /**
     * <p>
     * A value between zero and one that Amazon Transcribe assigned to the language that it identified in the source
     * audio. This value appears only when you don't provide a single language code. Larger values indicate that Amazon
     * Transcribe has higher confidence in the language that it identified
     * </p>
     * 
     * @return A value between zero and one that Amazon Transcribe assigned to the language that it identified in the
     *         source audio. This value appears only when you don't provide a single language code. Larger values
     *         indicate that Amazon Transcribe has higher confidence in the language that it identified
     */
    public final Float identifiedLanguageScore() {
        return identifiedLanguageScore;
    }

    /**
     * <p>
     * Provides information about the settings used to run a transcription job.
     * </p>
     * 
     * @return Provides information about the settings used to run a transcription job.
     */
    public final CallAnalyticsJobSettings settings() {
        return settings;
    }

    /**
     * For responses, this returns true if the service returned a value for the ChannelDefinitions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasChannelDefinitions() {
        return channelDefinitions != null && !(channelDefinitions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Shows numeric values to indicate the channel assigned to the agent's audio and the channel assigned to the
     * customer's audio.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasChannelDefinitions} method.
     * </p>
     * 
     * @return Shows numeric values to indicate the channel assigned to the agent's audio and the channel assigned to
     *         the customer's audio.
     */
    public final List<ChannelDefinition> channelDefinitions() {
        return channelDefinitions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(callAnalyticsJobName());
        hashCode = 31 * hashCode + Objects.hashCode(callAnalyticsJobStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(languageCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(mediaSampleRateHertz());
        hashCode = 31 * hashCode + Objects.hashCode(mediaFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(media());
        hashCode = 31 * hashCode + Objects.hashCode(transcript());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(completionTime());
        hashCode = 31 * hashCode + Objects.hashCode(failureReason());
        hashCode = 31 * hashCode + Objects.hashCode(dataAccessRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(identifiedLanguageScore());
        hashCode = 31 * hashCode + Objects.hashCode(settings());
        hashCode = 31 * hashCode + Objects.hashCode(hasChannelDefinitions() ? channelDefinitions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CallAnalyticsJob)) {
            return false;
        }
        CallAnalyticsJob other = (CallAnalyticsJob) obj;
        return Objects.equals(callAnalyticsJobName(), other.callAnalyticsJobName())
                && Objects.equals(callAnalyticsJobStatusAsString(), other.callAnalyticsJobStatusAsString())
                && Objects.equals(languageCodeAsString(), other.languageCodeAsString())
                && Objects.equals(mediaSampleRateHertz(), other.mediaSampleRateHertz())
                && Objects.equals(mediaFormatAsString(), other.mediaFormatAsString()) && Objects.equals(media(), other.media())
                && Objects.equals(transcript(), other.transcript()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(completionTime(), other.completionTime())
                && Objects.equals(failureReason(), other.failureReason())
                && Objects.equals(dataAccessRoleArn(), other.dataAccessRoleArn())
                && Objects.equals(identifiedLanguageScore(), other.identifiedLanguageScore())
                && Objects.equals(settings(), other.settings()) && hasChannelDefinitions() == other.hasChannelDefinitions()
                && Objects.equals(channelDefinitions(), other.channelDefinitions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CallAnalyticsJob").add("CallAnalyticsJobName", callAnalyticsJobName())
                .add("CallAnalyticsJobStatus", callAnalyticsJobStatusAsString()).add("LanguageCode", languageCodeAsString())
                .add("MediaSampleRateHertz", mediaSampleRateHertz()).add("MediaFormat", mediaFormatAsString())
                .add("Media", media()).add("Transcript", transcript()).add("StartTime", startTime())
                .add("CreationTime", creationTime()).add("CompletionTime", completionTime())
                .add("FailureReason", failureReason()).add("DataAccessRoleArn", dataAccessRoleArn())
                .add("IdentifiedLanguageScore", identifiedLanguageScore()).add("Settings", settings())
                .add("ChannelDefinitions", hasChannelDefinitions() ? channelDefinitions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CallAnalyticsJobName":
            return Optional.ofNullable(clazz.cast(callAnalyticsJobName()));
        case "CallAnalyticsJobStatus":
            return Optional.ofNullable(clazz.cast(callAnalyticsJobStatusAsString()));
        case "LanguageCode":
            return Optional.ofNullable(clazz.cast(languageCodeAsString()));
        case "MediaSampleRateHertz":
            return Optional.ofNullable(clazz.cast(mediaSampleRateHertz()));
        case "MediaFormat":
            return Optional.ofNullable(clazz.cast(mediaFormatAsString()));
        case "Media":
            return Optional.ofNullable(clazz.cast(media()));
        case "Transcript":
            return Optional.ofNullable(clazz.cast(transcript()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "CompletionTime":
            return Optional.ofNullable(clazz.cast(completionTime()));
        case "FailureReason":
            return Optional.ofNullable(clazz.cast(failureReason()));
        case "DataAccessRoleArn":
            return Optional.ofNullable(clazz.cast(dataAccessRoleArn()));
        case "IdentifiedLanguageScore":
            return Optional.ofNullable(clazz.cast(identifiedLanguageScore()));
        case "Settings":
            return Optional.ofNullable(clazz.cast(settings()));
        case "ChannelDefinitions":
            return Optional.ofNullable(clazz.cast(channelDefinitions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CallAnalyticsJob, T> g) {
        return obj -> g.apply((CallAnalyticsJob) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CallAnalyticsJob> {
        /**
         * <p>
         * The name of the call analytics job.
         * </p>
         * 
         * @param callAnalyticsJobName
         *        The name of the call analytics job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder callAnalyticsJobName(String callAnalyticsJobName);

        /**
         * <p>
         * The status of the analytics job.
         * </p>
         * 
         * @param callAnalyticsJobStatus
         *        The status of the analytics job.
         * @see CallAnalyticsJobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CallAnalyticsJobStatus
         */
        Builder callAnalyticsJobStatus(String callAnalyticsJobStatus);

        /**
         * <p>
         * The status of the analytics job.
         * </p>
         * 
         * @param callAnalyticsJobStatus
         *        The status of the analytics job.
         * @see CallAnalyticsJobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CallAnalyticsJobStatus
         */
        Builder callAnalyticsJobStatus(CallAnalyticsJobStatus callAnalyticsJobStatus);

        /**
         * <p>
         * If you know the language spoken between the customer and the agent, specify a language code for this field.
         * </p>
         * <p>
         * If you don't know the language, you can leave this field blank, and Amazon Transcribe will use machine
         * learning to automatically identify the language. To improve the accuracy of language identification, you can
         * provide an array containing the possible language codes for the language spoken in your audio. Refer to <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/how-it-works.html">Supported languages and
         * language-specific features</a> for additional information.
         * </p>
         * 
         * @param languageCode
         *        If you know the language spoken between the customer and the agent, specify a language code for this
         *        field.</p>
         *        <p>
         *        If you don't know the language, you can leave this field blank, and Amazon Transcribe will use machine
         *        learning to automatically identify the language. To improve the accuracy of language identification,
         *        you can provide an array containing the possible language codes for the language spoken in your audio.
         *        Refer to <a href="https://docs.aws.amazon.com/transcribe/latest/dg/how-it-works.html">Supported
         *        languages and language-specific features</a> for additional information.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(String languageCode);

        /**
         * <p>
         * If you know the language spoken between the customer and the agent, specify a language code for this field.
         * </p>
         * <p>
         * If you don't know the language, you can leave this field blank, and Amazon Transcribe will use machine
         * learning to automatically identify the language. To improve the accuracy of language identification, you can
         * provide an array containing the possible language codes for the language spoken in your audio. Refer to <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/how-it-works.html">Supported languages and
         * language-specific features</a> for additional information.
         * </p>
         * 
         * @param languageCode
         *        If you know the language spoken between the customer and the agent, specify a language code for this
         *        field.</p>
         *        <p>
         *        If you don't know the language, you can leave this field blank, and Amazon Transcribe will use machine
         *        learning to automatically identify the language. To improve the accuracy of language identification,
         *        you can provide an array containing the possible language codes for the language spoken in your audio.
         *        Refer to <a href="https://docs.aws.amazon.com/transcribe/latest/dg/how-it-works.html">Supported
         *        languages and language-specific features</a> for additional information.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(LanguageCode languageCode);

        /**
         * <p>
         * The sample rate, in Hertz, of the audio.
         * </p>
         * 
         * @param mediaSampleRateHertz
         *        The sample rate, in Hertz, of the audio.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mediaSampleRateHertz(Integer mediaSampleRateHertz);

        /**
         * <p>
         * The format of the input audio file. Note: for call analytics jobs, only the following media formats are
         * supported: MP3, MP4, WAV, FLAC, OGG, and WebM.
         * </p>
         * 
         * @param mediaFormat
         *        The format of the input audio file. Note: for call analytics jobs, only the following media formats
         *        are supported: MP3, MP4, WAV, FLAC, OGG, and WebM.
         * @see MediaFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MediaFormat
         */
        Builder mediaFormat(String mediaFormat);

        /**
         * <p>
         * The format of the input audio file. Note: for call analytics jobs, only the following media formats are
         * supported: MP3, MP4, WAV, FLAC, OGG, and WebM.
         * </p>
         * 
         * @param mediaFormat
         *        The format of the input audio file. Note: for call analytics jobs, only the following media formats
         *        are supported: MP3, MP4, WAV, FLAC, OGG, and WebM.
         * @see MediaFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MediaFormat
         */
        Builder mediaFormat(MediaFormat mediaFormat);

        /**
         * Sets the value of the Media property for this object.
         *
         * @param media
         *        The new value for the Media property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder media(Media media);

        /**
         * Sets the value of the Media property for this object.
         *
         * This is a convenience that creates an instance of the {@link Media.Builder} avoiding the need to create one
         * manually via {@link Media#builder()}.
         *
         * When the {@link Consumer} completes, {@link Media.Builder#build()} is called immediately and its result is
         * passed to {@link #media(Media)}.
         * 
         * @param media
         *        a consumer that will call methods on {@link Media.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #media(Media)
         */
        default Builder media(Consumer<Media.Builder> media) {
            return media(Media.builder().applyMutation(media).build());
        }

        /**
         * Sets the value of the Transcript property for this object.
         *
         * @param transcript
         *        The new value for the Transcript property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transcript(Transcript transcript);

        /**
         * Sets the value of the Transcript property for this object.
         *
         * This is a convenience that creates an instance of the {@link Transcript.Builder} avoiding the need to create
         * one manually via {@link Transcript#builder()}.
         *
         * When the {@link Consumer} completes, {@link Transcript.Builder#build()} is called immediately and its result
         * is passed to {@link #transcript(Transcript)}.
         * 
         * @param transcript
         *        a consumer that will call methods on {@link Transcript.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #transcript(Transcript)
         */
        default Builder transcript(Consumer<Transcript.Builder> transcript) {
            return transcript(Transcript.builder().applyMutation(transcript).build());
        }

        /**
         * <p>
         * A timestamp that shows when the analytics job started processing.
         * </p>
         * 
         * @param startTime
         *        A timestamp that shows when the analytics job started processing.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * A timestamp that shows when the analytics job was created.
         * </p>
         * 
         * @param creationTime
         *        A timestamp that shows when the analytics job was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * A timestamp that shows when the analytics job was completed.
         * </p>
         * 
         * @param completionTime
         *        A timestamp that shows when the analytics job was completed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completionTime(Instant completionTime);

        /**
         * <p>
         * If the <code>AnalyticsJobStatus</code> is <code>FAILED</code>, this field contains information about why the
         * job failed.
         * </p>
         * <p>
         * The <code>FailureReason</code> field can contain one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Unsupported media format</code>: The media format specified in the <code>MediaFormat</code> field of
         * the request isn't valid. See the description of the <code>MediaFormat</code> field for a list of valid
         * values.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>The media format provided does not match the detected media format</code>: The media format of the
         * audio file doesn't match the format specified in the <code>MediaFormat</code> field in the request. Check the
         * media format of your media file and make sure the two values match.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Invalid sample rate for audio file</code>: The sample rate specified in the
         * <code>MediaSampleRateHertz</code> of the request isn't valid. The sample rate must be between 8,000 and
         * 48,000 Hertz.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>The sample rate provided does not match the detected sample rate</code>: The sample rate in the audio
         * file doesn't match the sample rate specified in the <code>MediaSampleRateHertz</code> field in the request.
         * Check the sample rate of your media file and make sure that the two values match.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Invalid file size: file size too large</code>: The size of your audio file is larger than what Amazon
         * Transcribe Medical can process. For more information, see <i>Guidelines and Quotas</i> in the Amazon
         * Transcribe Medical Guide.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Invalid number of channels: number of channels too large</code>: Your audio contains more channels than
         * Amazon Transcribe Medical is configured to process. To request additional channels, see Amazon Transcribe
         * Medical Endpoints and Quotas in the <a
         * href="https://docs.aws.amazon.com/general/latest/gr/Welcome.html">Amazon Web Services General Reference</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param failureReason
         *        If the <code>AnalyticsJobStatus</code> is <code>FAILED</code>, this field contains information about
         *        why the job failed.</p>
         *        <p>
         *        The <code>FailureReason</code> field can contain one of the following values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Unsupported media format</code>: The media format specified in the <code>MediaFormat</code>
         *        field of the request isn't valid. See the description of the <code>MediaFormat</code> field for a list
         *        of valid values.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>The media format provided does not match the detected media format</code>: The media format of
         *        the audio file doesn't match the format specified in the <code>MediaFormat</code> field in the
         *        request. Check the media format of your media file and make sure the two values match.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Invalid sample rate for audio file</code>: The sample rate specified in the
         *        <code>MediaSampleRateHertz</code> of the request isn't valid. The sample rate must be between 8,000
         *        and 48,000 Hertz.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>The sample rate provided does not match the detected sample rate</code>: The sample rate in the
         *        audio file doesn't match the sample rate specified in the <code>MediaSampleRateHertz</code> field in
         *        the request. Check the sample rate of your media file and make sure that the two values match.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Invalid file size: file size too large</code>: The size of your audio file is larger than what
         *        Amazon Transcribe Medical can process. For more information, see <i>Guidelines and Quotas</i> in the
         *        Amazon Transcribe Medical Guide.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Invalid number of channels: number of channels too large</code>: Your audio contains more
         *        channels than Amazon Transcribe Medical is configured to process. To request additional channels, see
         *        Amazon Transcribe Medical Endpoints and Quotas in the <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/Welcome.html">Amazon Web Services General
         *        Reference</a>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReason(String failureReason);

        /**
         * <p>
         * The Amazon Resource Number (ARN) that you use to get access to the analytics job.
         * </p>
         * 
         * @param dataAccessRoleArn
         *        The Amazon Resource Number (ARN) that you use to get access to the analytics job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataAccessRoleArn(String dataAccessRoleArn);

        /**
         * <p>
         * A value between zero and one that Amazon Transcribe assigned to the language that it identified in the source
         * audio. This value appears only when you don't provide a single language code. Larger values indicate that
         * Amazon Transcribe has higher confidence in the language that it identified
         * </p>
         * 
         * @param identifiedLanguageScore
         *        A value between zero and one that Amazon Transcribe assigned to the language that it identified in the
         *        source audio. This value appears only when you don't provide a single language code. Larger values
         *        indicate that Amazon Transcribe has higher confidence in the language that it identified
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identifiedLanguageScore(Float identifiedLanguageScore);

        /**
         * <p>
         * Provides information about the settings used to run a transcription job.
         * </p>
         * 
         * @param settings
         *        Provides information about the settings used to run a transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder settings(CallAnalyticsJobSettings settings);

        /**
         * <p>
         * Provides information about the settings used to run a transcription job.
         * </p>
         * This is a convenience that creates an instance of the {@link CallAnalyticsJobSettings.Builder} avoiding the
         * need to create one manually via {@link CallAnalyticsJobSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link CallAnalyticsJobSettings.Builder#build()} is called immediately
         * and its result is passed to {@link #settings(CallAnalyticsJobSettings)}.
         * 
         * @param settings
         *        a consumer that will call methods on {@link CallAnalyticsJobSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #settings(CallAnalyticsJobSettings)
         */
        default Builder settings(Consumer<CallAnalyticsJobSettings.Builder> settings) {
            return settings(CallAnalyticsJobSettings.builder().applyMutation(settings).build());
        }

        /**
         * <p>
         * Shows numeric values to indicate the channel assigned to the agent's audio and the channel assigned to the
         * customer's audio.
         * </p>
         * 
         * @param channelDefinitions
         *        Shows numeric values to indicate the channel assigned to the agent's audio and the channel assigned to
         *        the customer's audio.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder channelDefinitions(Collection<ChannelDefinition> channelDefinitions);

        /**
         * <p>
         * Shows numeric values to indicate the channel assigned to the agent's audio and the channel assigned to the
         * customer's audio.
         * </p>
         * 
         * @param channelDefinitions
         *        Shows numeric values to indicate the channel assigned to the agent's audio and the channel assigned to
         *        the customer's audio.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder channelDefinitions(ChannelDefinition... channelDefinitions);

        /**
         * <p>
         * Shows numeric values to indicate the channel assigned to the agent's audio and the channel assigned to the
         * customer's audio.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ChannelDefinition>.Builder} avoiding the
         * need to create one manually via {@link List<ChannelDefinition>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ChannelDefinition>.Builder#build()} is called immediately
         * and its result is passed to {@link #channelDefinitions(List<ChannelDefinition>)}.
         * 
         * @param channelDefinitions
         *        a consumer that will call methods on {@link List<ChannelDefinition>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #channelDefinitions(List<ChannelDefinition>)
         */
        Builder channelDefinitions(Consumer<ChannelDefinition.Builder>... channelDefinitions);
    }

    static final class BuilderImpl implements Builder {
        private String callAnalyticsJobName;

        private String callAnalyticsJobStatus;

        private String languageCode;

        private Integer mediaSampleRateHertz;

        private String mediaFormat;

        private Media media;

        private Transcript transcript;

        private Instant startTime;

        private Instant creationTime;

        private Instant completionTime;

        private String failureReason;

        private String dataAccessRoleArn;

        private Float identifiedLanguageScore;

        private CallAnalyticsJobSettings settings;

        private List<ChannelDefinition> channelDefinitions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CallAnalyticsJob model) {
            callAnalyticsJobName(model.callAnalyticsJobName);
            callAnalyticsJobStatus(model.callAnalyticsJobStatus);
            languageCode(model.languageCode);
            mediaSampleRateHertz(model.mediaSampleRateHertz);
            mediaFormat(model.mediaFormat);
            media(model.media);
            transcript(model.transcript);
            startTime(model.startTime);
            creationTime(model.creationTime);
            completionTime(model.completionTime);
            failureReason(model.failureReason);
            dataAccessRoleArn(model.dataAccessRoleArn);
            identifiedLanguageScore(model.identifiedLanguageScore);
            settings(model.settings);
            channelDefinitions(model.channelDefinitions);
        }

        public final String getCallAnalyticsJobName() {
            return callAnalyticsJobName;
        }

        public final void setCallAnalyticsJobName(String callAnalyticsJobName) {
            this.callAnalyticsJobName = callAnalyticsJobName;
        }

        @Override
        @Transient
        public final Builder callAnalyticsJobName(String callAnalyticsJobName) {
            this.callAnalyticsJobName = callAnalyticsJobName;
            return this;
        }

        public final String getCallAnalyticsJobStatus() {
            return callAnalyticsJobStatus;
        }

        public final void setCallAnalyticsJobStatus(String callAnalyticsJobStatus) {
            this.callAnalyticsJobStatus = callAnalyticsJobStatus;
        }

        @Override
        @Transient
        public final Builder callAnalyticsJobStatus(String callAnalyticsJobStatus) {
            this.callAnalyticsJobStatus = callAnalyticsJobStatus;
            return this;
        }

        @Override
        @Transient
        public final Builder callAnalyticsJobStatus(CallAnalyticsJobStatus callAnalyticsJobStatus) {
            this.callAnalyticsJobStatus(callAnalyticsJobStatus == null ? null : callAnalyticsJobStatus.toString());
            return this;
        }

        public final String getLanguageCode() {
            return languageCode;
        }

        public final void setLanguageCode(String languageCode) {
            this.languageCode = languageCode;
        }

        @Override
        @Transient
        public final Builder languageCode(String languageCode) {
            this.languageCode = languageCode;
            return this;
        }

        @Override
        @Transient
        public final Builder languageCode(LanguageCode languageCode) {
            this.languageCode(languageCode == null ? null : languageCode.toString());
            return this;
        }

        public final Integer getMediaSampleRateHertz() {
            return mediaSampleRateHertz;
        }

        public final void setMediaSampleRateHertz(Integer mediaSampleRateHertz) {
            this.mediaSampleRateHertz = mediaSampleRateHertz;
        }

        @Override
        @Transient
        public final Builder mediaSampleRateHertz(Integer mediaSampleRateHertz) {
            this.mediaSampleRateHertz = mediaSampleRateHertz;
            return this;
        }

        public final String getMediaFormat() {
            return mediaFormat;
        }

        public final void setMediaFormat(String mediaFormat) {
            this.mediaFormat = mediaFormat;
        }

        @Override
        @Transient
        public final Builder mediaFormat(String mediaFormat) {
            this.mediaFormat = mediaFormat;
            return this;
        }

        @Override
        @Transient
        public final Builder mediaFormat(MediaFormat mediaFormat) {
            this.mediaFormat(mediaFormat == null ? null : mediaFormat.toString());
            return this;
        }

        public final Media.Builder getMedia() {
            return media != null ? media.toBuilder() : null;
        }

        public final void setMedia(Media.BuilderImpl media) {
            this.media = media != null ? media.build() : null;
        }

        @Override
        @Transient
        public final Builder media(Media media) {
            this.media = media;
            return this;
        }

        public final Transcript.Builder getTranscript() {
            return transcript != null ? transcript.toBuilder() : null;
        }

        public final void setTranscript(Transcript.BuilderImpl transcript) {
            this.transcript = transcript != null ? transcript.build() : null;
        }

        @Override
        @Transient
        public final Builder transcript(Transcript transcript) {
            this.transcript = transcript;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        @Transient
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        @Transient
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final Instant getCompletionTime() {
            return completionTime;
        }

        public final void setCompletionTime(Instant completionTime) {
            this.completionTime = completionTime;
        }

        @Override
        @Transient
        public final Builder completionTime(Instant completionTime) {
            this.completionTime = completionTime;
            return this;
        }

        public final String getFailureReason() {
            return failureReason;
        }

        public final void setFailureReason(String failureReason) {
            this.failureReason = failureReason;
        }

        @Override
        @Transient
        public final Builder failureReason(String failureReason) {
            this.failureReason = failureReason;
            return this;
        }

        public final String getDataAccessRoleArn() {
            return dataAccessRoleArn;
        }

        public final void setDataAccessRoleArn(String dataAccessRoleArn) {
            this.dataAccessRoleArn = dataAccessRoleArn;
        }

        @Override
        @Transient
        public final Builder dataAccessRoleArn(String dataAccessRoleArn) {
            this.dataAccessRoleArn = dataAccessRoleArn;
            return this;
        }

        public final Float getIdentifiedLanguageScore() {
            return identifiedLanguageScore;
        }

        public final void setIdentifiedLanguageScore(Float identifiedLanguageScore) {
            this.identifiedLanguageScore = identifiedLanguageScore;
        }

        @Override
        @Transient
        public final Builder identifiedLanguageScore(Float identifiedLanguageScore) {
            this.identifiedLanguageScore = identifiedLanguageScore;
            return this;
        }

        public final CallAnalyticsJobSettings.Builder getSettings() {
            return settings != null ? settings.toBuilder() : null;
        }

        public final void setSettings(CallAnalyticsJobSettings.BuilderImpl settings) {
            this.settings = settings != null ? settings.build() : null;
        }

        @Override
        @Transient
        public final Builder settings(CallAnalyticsJobSettings settings) {
            this.settings = settings;
            return this;
        }

        public final List<ChannelDefinition.Builder> getChannelDefinitions() {
            List<ChannelDefinition.Builder> result = ChannelDefinitionsCopier.copyToBuilder(this.channelDefinitions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setChannelDefinitions(Collection<ChannelDefinition.BuilderImpl> channelDefinitions) {
            this.channelDefinitions = ChannelDefinitionsCopier.copyFromBuilder(channelDefinitions);
        }

        @Override
        @Transient
        public final Builder channelDefinitions(Collection<ChannelDefinition> channelDefinitions) {
            this.channelDefinitions = ChannelDefinitionsCopier.copy(channelDefinitions);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder channelDefinitions(ChannelDefinition... channelDefinitions) {
            channelDefinitions(Arrays.asList(channelDefinitions));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder channelDefinitions(Consumer<ChannelDefinition.Builder>... channelDefinitions) {
            channelDefinitions(Stream.of(channelDefinitions).map(c -> ChannelDefinition.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public CallAnalyticsJob build() {
            return new CallAnalyticsJob(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
