/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribe.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides detailed information about a transcription job.
 * </p>
 * <p>
 * To view the status of the specified transcription job, check the <code>TranscriptionJobStatus</code> field. If the
 * status is <code>COMPLETED</code>, the job is finished and you can find the results at the location specified in
 * <code>TranscriptFileUri</code>. If the status is <code>FAILED</code>, <code>FailureReason</code> provides details on
 * why your transcription job failed.
 * </p>
 * <p>
 * If you enabled content redaction, the redacted transcript can be found at the location specified in
 * <code>RedactedTranscriptFileUri</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TranscriptionJob implements SdkPojo, Serializable,
        ToCopyableBuilder<TranscriptionJob.Builder, TranscriptionJob> {
    private static final SdkField<String> TRANSCRIPTION_JOB_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TranscriptionJobName").getter(getter(TranscriptionJob::transcriptionJobName))
            .setter(setter(Builder::transcriptionJobName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TranscriptionJobName").build())
            .build();

    private static final SdkField<String> TRANSCRIPTION_JOB_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TranscriptionJobStatus").getter(getter(TranscriptionJob::transcriptionJobStatusAsString))
            .setter(setter(Builder::transcriptionJobStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TranscriptionJobStatus").build())
            .build();

    private static final SdkField<String> LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LanguageCode").getter(getter(TranscriptionJob::languageCodeAsString))
            .setter(setter(Builder::languageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageCode").build()).build();

    private static final SdkField<Integer> MEDIA_SAMPLE_RATE_HERTZ_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MediaSampleRateHertz").getter(getter(TranscriptionJob::mediaSampleRateHertz))
            .setter(setter(Builder::mediaSampleRateHertz))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MediaSampleRateHertz").build())
            .build();

    private static final SdkField<String> MEDIA_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MediaFormat").getter(getter(TranscriptionJob::mediaFormatAsString)).setter(setter(Builder::mediaFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MediaFormat").build()).build();

    private static final SdkField<Media> MEDIA_FIELD = SdkField.<Media> builder(MarshallingType.SDK_POJO).memberName("Media")
            .getter(getter(TranscriptionJob::media)).setter(setter(Builder::media)).constructor(Media::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Media").build()).build();

    private static final SdkField<Transcript> TRANSCRIPT_FIELD = SdkField.<Transcript> builder(MarshallingType.SDK_POJO)
            .memberName("Transcript").getter(getter(TranscriptionJob::transcript)).setter(setter(Builder::transcript))
            .constructor(Transcript::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Transcript").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime").getter(getter(TranscriptionJob::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(TranscriptionJob::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<Instant> COMPLETION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CompletionTime").getter(getter(TranscriptionJob::completionTime))
            .setter(setter(Builder::completionTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletionTime").build()).build();

    private static final SdkField<String> FAILURE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FailureReason").getter(getter(TranscriptionJob::failureReason)).setter(setter(Builder::failureReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureReason").build()).build();

    private static final SdkField<Settings> SETTINGS_FIELD = SdkField.<Settings> builder(MarshallingType.SDK_POJO)
            .memberName("Settings").getter(getter(TranscriptionJob::settings)).setter(setter(Builder::settings))
            .constructor(Settings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Settings").build()).build();

    private static final SdkField<ModelSettings> MODEL_SETTINGS_FIELD = SdkField
            .<ModelSettings> builder(MarshallingType.SDK_POJO).memberName("ModelSettings")
            .getter(getter(TranscriptionJob::modelSettings)).setter(setter(Builder::modelSettings))
            .constructor(ModelSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelSettings").build()).build();

    private static final SdkField<JobExecutionSettings> JOB_EXECUTION_SETTINGS_FIELD = SdkField
            .<JobExecutionSettings> builder(MarshallingType.SDK_POJO).memberName("JobExecutionSettings")
            .getter(getter(TranscriptionJob::jobExecutionSettings)).setter(setter(Builder::jobExecutionSettings))
            .constructor(JobExecutionSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobExecutionSettings").build())
            .build();

    private static final SdkField<ContentRedaction> CONTENT_REDACTION_FIELD = SdkField
            .<ContentRedaction> builder(MarshallingType.SDK_POJO).memberName("ContentRedaction")
            .getter(getter(TranscriptionJob::contentRedaction)).setter(setter(Builder::contentRedaction))
            .constructor(ContentRedaction::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContentRedaction").build()).build();

    private static final SdkField<Boolean> IDENTIFY_LANGUAGE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IdentifyLanguage").getter(getter(TranscriptionJob::identifyLanguage))
            .setter(setter(Builder::identifyLanguage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdentifyLanguage").build()).build();

    private static final SdkField<Boolean> IDENTIFY_MULTIPLE_LANGUAGES_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN).memberName("IdentifyMultipleLanguages")
            .getter(getter(TranscriptionJob::identifyMultipleLanguages)).setter(setter(Builder::identifyMultipleLanguages))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdentifyMultipleLanguages").build())
            .build();

    private static final SdkField<List<String>> LANGUAGE_OPTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("LanguageOptions")
            .getter(getter(TranscriptionJob::languageOptionsAsStrings))
            .setter(setter(Builder::languageOptionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageOptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Float> IDENTIFIED_LANGUAGE_SCORE_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("IdentifiedLanguageScore").getter(getter(TranscriptionJob::identifiedLanguageScore))
            .setter(setter(Builder::identifiedLanguageScore))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdentifiedLanguageScore").build())
            .build();

    private static final SdkField<List<LanguageCodeItem>> LANGUAGE_CODES_FIELD = SdkField
            .<List<LanguageCodeItem>> builder(MarshallingType.LIST)
            .memberName("LanguageCodes")
            .getter(getter(TranscriptionJob::languageCodes))
            .setter(setter(Builder::languageCodes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageCodes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LanguageCodeItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(LanguageCodeItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(TranscriptionJob::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<SubtitlesOutput> SUBTITLES_FIELD = SdkField.<SubtitlesOutput> builder(MarshallingType.SDK_POJO)
            .memberName("Subtitles").getter(getter(TranscriptionJob::subtitles)).setter(setter(Builder::subtitles))
            .constructor(SubtitlesOutput::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Subtitles").build()).build();

    private static final SdkField<Map<String, LanguageIdSettings>> LANGUAGE_ID_SETTINGS_FIELD = SdkField
            .<Map<String, LanguageIdSettings>> builder(MarshallingType.MAP)
            .memberName("LanguageIdSettings")
            .getter(getter(TranscriptionJob::languageIdSettingsAsStrings))
            .setter(setter(Builder::languageIdSettingsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageIdSettings").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<LanguageIdSettings> builder(MarshallingType.SDK_POJO)
                                            .constructor(LanguageIdSettings::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TRANSCRIPTION_JOB_NAME_FIELD,
            TRANSCRIPTION_JOB_STATUS_FIELD, LANGUAGE_CODE_FIELD, MEDIA_SAMPLE_RATE_HERTZ_FIELD, MEDIA_FORMAT_FIELD, MEDIA_FIELD,
            TRANSCRIPT_FIELD, START_TIME_FIELD, CREATION_TIME_FIELD, COMPLETION_TIME_FIELD, FAILURE_REASON_FIELD, SETTINGS_FIELD,
            MODEL_SETTINGS_FIELD, JOB_EXECUTION_SETTINGS_FIELD, CONTENT_REDACTION_FIELD, IDENTIFY_LANGUAGE_FIELD,
            IDENTIFY_MULTIPLE_LANGUAGES_FIELD, LANGUAGE_OPTIONS_FIELD, IDENTIFIED_LANGUAGE_SCORE_FIELD, LANGUAGE_CODES_FIELD,
            TAGS_FIELD, SUBTITLES_FIELD, LANGUAGE_ID_SETTINGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String transcriptionJobName;

    private final String transcriptionJobStatus;

    private final String languageCode;

    private final Integer mediaSampleRateHertz;

    private final String mediaFormat;

    private final Media media;

    private final Transcript transcript;

    private final Instant startTime;

    private final Instant creationTime;

    private final Instant completionTime;

    private final String failureReason;

    private final Settings settings;

    private final ModelSettings modelSettings;

    private final JobExecutionSettings jobExecutionSettings;

    private final ContentRedaction contentRedaction;

    private final Boolean identifyLanguage;

    private final Boolean identifyMultipleLanguages;

    private final List<String> languageOptions;

    private final Float identifiedLanguageScore;

    private final List<LanguageCodeItem> languageCodes;

    private final List<Tag> tags;

    private final SubtitlesOutput subtitles;

    private final Map<String, LanguageIdSettings> languageIdSettings;

    private TranscriptionJob(BuilderImpl builder) {
        this.transcriptionJobName = builder.transcriptionJobName;
        this.transcriptionJobStatus = builder.transcriptionJobStatus;
        this.languageCode = builder.languageCode;
        this.mediaSampleRateHertz = builder.mediaSampleRateHertz;
        this.mediaFormat = builder.mediaFormat;
        this.media = builder.media;
        this.transcript = builder.transcript;
        this.startTime = builder.startTime;
        this.creationTime = builder.creationTime;
        this.completionTime = builder.completionTime;
        this.failureReason = builder.failureReason;
        this.settings = builder.settings;
        this.modelSettings = builder.modelSettings;
        this.jobExecutionSettings = builder.jobExecutionSettings;
        this.contentRedaction = builder.contentRedaction;
        this.identifyLanguage = builder.identifyLanguage;
        this.identifyMultipleLanguages = builder.identifyMultipleLanguages;
        this.languageOptions = builder.languageOptions;
        this.identifiedLanguageScore = builder.identifiedLanguageScore;
        this.languageCodes = builder.languageCodes;
        this.tags = builder.tags;
        this.subtitles = builder.subtitles;
        this.languageIdSettings = builder.languageIdSettings;
    }

    /**
     * <p>
     * The name of the transcription job. Job names are case sensitive and must be unique within an Amazon Web Services
     * account.
     * </p>
     * 
     * @return The name of the transcription job. Job names are case sensitive and must be unique within an Amazon Web
     *         Services account.
     */
    public final String transcriptionJobName() {
        return transcriptionJobName;
    }

    /**
     * <p>
     * Provides the status of the specified transcription job.
     * </p>
     * <p>
     * If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
     * specified in <code>TranscriptFileUri</code> (or <code>RedactedTranscriptFileUri</code>, if you requested
     * transcript redaction). If the status is <code>FAILED</code>, <code>FailureReason</code> provides details on why
     * your transcription job failed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #transcriptionJobStatus} will return {@link TranscriptionJobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #transcriptionJobStatusAsString}.
     * </p>
     * 
     * @return Provides the status of the specified transcription job.</p>
     *         <p>
     *         If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
     *         specified in <code>TranscriptFileUri</code> (or <code>RedactedTranscriptFileUri</code>, if you requested
     *         transcript redaction). If the status is <code>FAILED</code>, <code>FailureReason</code> provides details
     *         on why your transcription job failed.
     * @see TranscriptionJobStatus
     */
    public final TranscriptionJobStatus transcriptionJobStatus() {
        return TranscriptionJobStatus.fromValue(transcriptionJobStatus);
    }

    /**
     * <p>
     * Provides the status of the specified transcription job.
     * </p>
     * <p>
     * If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
     * specified in <code>TranscriptFileUri</code> (or <code>RedactedTranscriptFileUri</code>, if you requested
     * transcript redaction). If the status is <code>FAILED</code>, <code>FailureReason</code> provides details on why
     * your transcription job failed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #transcriptionJobStatus} will return {@link TranscriptionJobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #transcriptionJobStatusAsString}.
     * </p>
     * 
     * @return Provides the status of the specified transcription job.</p>
     *         <p>
     *         If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
     *         specified in <code>TranscriptFileUri</code> (or <code>RedactedTranscriptFileUri</code>, if you requested
     *         transcript redaction). If the status is <code>FAILED</code>, <code>FailureReason</code> provides details
     *         on why your transcription job failed.
     * @see TranscriptionJobStatus
     */
    public final String transcriptionJobStatusAsString() {
        return transcriptionJobStatus;
    }

    /**
     * <p>
     * The language code used to create your transcription job. This parameter is used with single-language
     * identification. For multi-language identification requests, refer to the plural version of this parameter,
     * <code>LanguageCodes</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return The language code used to create your transcription job. This parameter is used with single-language
     *         identification. For multi-language identification requests, refer to the plural version of this
     *         parameter, <code>LanguageCodes</code>.
     * @see LanguageCode
     */
    public final LanguageCode languageCode() {
        return LanguageCode.fromValue(languageCode);
    }

    /**
     * <p>
     * The language code used to create your transcription job. This parameter is used with single-language
     * identification. For multi-language identification requests, refer to the plural version of this parameter,
     * <code>LanguageCodes</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return The language code used to create your transcription job. This parameter is used with single-language
     *         identification. For multi-language identification requests, refer to the plural version of this
     *         parameter, <code>LanguageCodes</code>.
     * @see LanguageCode
     */
    public final String languageCodeAsString() {
        return languageCode;
    }

    /**
     * <p>
     * The sample rate, in hertz, of the audio track in your input media file.
     * </p>
     * 
     * @return The sample rate, in hertz, of the audio track in your input media file.
     */
    public final Integer mediaSampleRateHertz() {
        return mediaSampleRateHertz;
    }

    /**
     * <p>
     * The format of the input media file.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mediaFormat} will
     * return {@link MediaFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #mediaFormatAsString}.
     * </p>
     * 
     * @return The format of the input media file.
     * @see MediaFormat
     */
    public final MediaFormat mediaFormat() {
        return MediaFormat.fromValue(mediaFormat);
    }

    /**
     * <p>
     * The format of the input media file.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mediaFormat} will
     * return {@link MediaFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #mediaFormatAsString}.
     * </p>
     * 
     * @return The format of the input media file.
     * @see MediaFormat
     */
    public final String mediaFormatAsString() {
        return mediaFormat;
    }

    /**
     * <p>
     * Provides the Amazon S3 location of the media file you used in your request.
     * </p>
     * 
     * @return Provides the Amazon S3 location of the media file you used in your request.
     */
    public final Media media() {
        return media;
    }

    /**
     * <p>
     * Provides you with the Amazon S3 URI you can use to access your transcript.
     * </p>
     * 
     * @return Provides you with the Amazon S3 URI you can use to access your transcript.
     */
    public final Transcript transcript() {
        return transcript;
    }

    /**
     * <p>
     * The date and time the specified transcription job began processing.
     * </p>
     * <p>
     * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     * <code>2022-05-04T12:32:58.789000-07:00</code> represents a transcription job that started processing at 12:32 PM
     * UTC-7 on May 4, 2022.
     * </p>
     * 
     * @return The date and time the specified transcription job began processing.</p>
     *         <p>
     *         Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     *         <code>2022-05-04T12:32:58.789000-07:00</code> represents a transcription job that started processing at
     *         12:32 PM UTC-7 on May 4, 2022.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The date and time the specified transcription job request was made.
     * </p>
     * <p>
     * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     * <code>2022-05-04T12:32:58.761000-07:00</code> represents a transcription job that started processing at 12:32 PM
     * UTC-7 on May 4, 2022.
     * </p>
     * 
     * @return The date and time the specified transcription job request was made.</p>
     *         <p>
     *         Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     *         <code>2022-05-04T12:32:58.761000-07:00</code> represents a transcription job that started processing at
     *         12:32 PM UTC-7 on May 4, 2022.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The date and time the specified transcription job finished processing.
     * </p>
     * <p>
     * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     * <code>2022-05-04T12:33:13.922000-07:00</code> represents a transcription job that started processing at 12:33 PM
     * UTC-7 on May 4, 2022.
     * </p>
     * 
     * @return The date and time the specified transcription job finished processing.</p>
     *         <p>
     *         Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     *         <code>2022-05-04T12:33:13.922000-07:00</code> represents a transcription job that started processing at
     *         12:33 PM UTC-7 on May 4, 2022.
     */
    public final Instant completionTime() {
        return completionTime;
    }

    /**
     * <p>
     * If <code>TranscriptionJobStatus</code> is <code>FAILED</code>, <code>FailureReason</code> contains information
     * about why the transcription job request failed.
     * </p>
     * <p>
     * The <code>FailureReason</code> field contains one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Unsupported media format</code>.
     * </p>
     * <p>
     * The media format specified in <code>MediaFormat</code> isn't valid. Refer to <b>MediaFormat</b> for a list of
     * supported formats.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>The media format provided does not match the detected media format</code>.
     * </p>
     * <p>
     * The media format specified in <code>MediaFormat</code> doesn't match the format of the input file. Check the
     * media format of your media file and correct the specified value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Invalid sample rate for audio file</code>.
     * </p>
     * <p>
     * The sample rate specified in <code>MediaSampleRateHertz</code> isn't valid. The sample rate must be between 8,000
     * and 48,000 hertz.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>The sample rate provided does not match the detected sample rate</code>.
     * </p>
     * <p>
     * The sample rate specified in <code>MediaSampleRateHertz</code> doesn't match the sample rate detected in your
     * input media file. Check the sample rate of your media file and correct the specified value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Invalid file size: file size too large</code>.
     * </p>
     * <p>
     * The size of your media file is larger than what Amazon Transcribe can process. For more information, refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Guidelines and quotas</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Invalid number of channels: number of channels too large</code>.
     * </p>
     * <p>
     * Your audio contains more channels than Amazon Transcribe is able to process. For more information, refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Guidelines and quotas</a>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return If <code>TranscriptionJobStatus</code> is <code>FAILED</code>, <code>FailureReason</code> contains
     *         information about why the transcription job request failed.</p>
     *         <p>
     *         The <code>FailureReason</code> field contains one of the following values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Unsupported media format</code>.
     *         </p>
     *         <p>
     *         The media format specified in <code>MediaFormat</code> isn't valid. Refer to <b>MediaFormat</b> for a
     *         list of supported formats.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>The media format provided does not match the detected media format</code>.
     *         </p>
     *         <p>
     *         The media format specified in <code>MediaFormat</code> doesn't match the format of the input file. Check
     *         the media format of your media file and correct the specified value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Invalid sample rate for audio file</code>.
     *         </p>
     *         <p>
     *         The sample rate specified in <code>MediaSampleRateHertz</code> isn't valid. The sample rate must be
     *         between 8,000 and 48,000 hertz.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>The sample rate provided does not match the detected sample rate</code>.
     *         </p>
     *         <p>
     *         The sample rate specified in <code>MediaSampleRateHertz</code> doesn't match the sample rate detected in
     *         your input media file. Check the sample rate of your media file and correct the specified value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Invalid file size: file size too large</code>.
     *         </p>
     *         <p>
     *         The size of your media file is larger than what Amazon Transcribe can process. For more information,
     *         refer to <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Guidelines and
     *         quotas</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Invalid number of channels: number of channels too large</code>.
     *         </p>
     *         <p>
     *         Your audio contains more channels than Amazon Transcribe is able to process. For more information, refer
     *         to <a href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Guidelines
     *         and quotas</a>.
     *         </p>
     *         </li>
     */
    public final String failureReason() {
        return failureReason;
    }

    /**
     * <p>
     * Provides information on any additional settings that were included in your request. Additional settings include
     * channel identification, alternative transcriptions, speaker partitioning, custom vocabularies, and custom
     * vocabulary filters.
     * </p>
     * 
     * @return Provides information on any additional settings that were included in your request. Additional settings
     *         include channel identification, alternative transcriptions, speaker partitioning, custom vocabularies,
     *         and custom vocabulary filters.
     */
    public final Settings settings() {
        return settings;
    }

    /**
     * <p>
     * Provides information on the custom language model you included in your request.
     * </p>
     * 
     * @return Provides information on the custom language model you included in your request.
     */
    public final ModelSettings modelSettings() {
        return modelSettings;
    }

    /**
     * <p>
     * Provides information about how your transcription job was processed. This parameter shows if your request was
     * queued and what data access role was used.
     * </p>
     * 
     * @return Provides information about how your transcription job was processed. This parameter shows if your request
     *         was queued and what data access role was used.
     */
    public final JobExecutionSettings jobExecutionSettings() {
        return jobExecutionSettings;
    }

    /**
     * <p>
     * Indicates whether redaction was enabled in your transcript.
     * </p>
     * 
     * @return Indicates whether redaction was enabled in your transcript.
     */
    public final ContentRedaction contentRedaction() {
        return contentRedaction;
    }

    /**
     * <p>
     * Indicates whether automatic language identification was enabled (<code>TRUE</code>) for the specified
     * transcription job.
     * </p>
     * 
     * @return Indicates whether automatic language identification was enabled (<code>TRUE</code>) for the specified
     *         transcription job.
     */
    public final Boolean identifyLanguage() {
        return identifyLanguage;
    }

    /**
     * <p>
     * Indicates whether automatic multi-language identification was enabled (<code>TRUE</code>) for the specified
     * transcription job.
     * </p>
     * 
     * @return Indicates whether automatic multi-language identification was enabled (<code>TRUE</code>) for the
     *         specified transcription job.
     */
    public final Boolean identifyMultipleLanguages() {
        return identifyMultipleLanguages;
    }

    /**
     * <p>
     * Provides the language codes you specified in your request.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLanguageOptions} method.
     * </p>
     * 
     * @return Provides the language codes you specified in your request.
     */
    public final List<LanguageCode> languageOptions() {
        return LanguageOptionsCopier.copyStringToEnum(languageOptions);
    }

    /**
     * For responses, this returns true if the service returned a value for the LanguageOptions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLanguageOptions() {
        return languageOptions != null && !(languageOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Provides the language codes you specified in your request.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLanguageOptions} method.
     * </p>
     * 
     * @return Provides the language codes you specified in your request.
     */
    public final List<String> languageOptionsAsStrings() {
        return languageOptions;
    }

    /**
     * <p>
     * The confidence score associated with the language identified in your media file.
     * </p>
     * <p>
     * Confidence scores are values between 0 and 1; a larger value indicates a higher probability that the identified
     * language correctly matches the language spoken in your media.
     * </p>
     * 
     * @return The confidence score associated with the language identified in your media file.</p>
     *         <p>
     *         Confidence scores are values between 0 and 1; a larger value indicates a higher probability that the
     *         identified language correctly matches the language spoken in your media.
     */
    public final Float identifiedLanguageScore() {
        return identifiedLanguageScore;
    }

    /**
     * For responses, this returns true if the service returned a value for the LanguageCodes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLanguageCodes() {
        return languageCodes != null && !(languageCodes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The language codes used to create your transcription job. This parameter is used with multi-language
     * identification. For single-language identification requests, refer to the singular version of this parameter,
     * <code>LanguageCode</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLanguageCodes} method.
     * </p>
     * 
     * @return The language codes used to create your transcription job. This parameter is used with multi-language
     *         identification. For single-language identification requests, refer to the singular version of this
     *         parameter, <code>LanguageCode</code>.
     */
    public final List<LanguageCodeItem> languageCodes() {
        return languageCodes;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags, each in the form of a key:value pair, assigned to the specified transcription job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags, each in the form of a key:value pair, assigned to the specified transcription job.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Indicates whether subtitles were generated with your transcription.
     * </p>
     * 
     * @return Indicates whether subtitles were generated with your transcription.
     */
    public final SubtitlesOutput subtitles() {
        return subtitles;
    }

    /**
     * <p>
     * Provides the name and language of all custom language models, custom vocabularies, and custom vocabulary filters
     * that you included in your request.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLanguageIdSettings} method.
     * </p>
     * 
     * @return Provides the name and language of all custom language models, custom vocabularies, and custom vocabulary
     *         filters that you included in your request.
     */
    public final Map<LanguageCode, LanguageIdSettings> languageIdSettings() {
        return LanguageIdSettingsMapCopier.copyStringToEnum(languageIdSettings);
    }

    /**
     * For responses, this returns true if the service returned a value for the LanguageIdSettings property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLanguageIdSettings() {
        return languageIdSettings != null && !(languageIdSettings instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Provides the name and language of all custom language models, custom vocabularies, and custom vocabulary filters
     * that you included in your request.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLanguageIdSettings} method.
     * </p>
     * 
     * @return Provides the name and language of all custom language models, custom vocabularies, and custom vocabulary
     *         filters that you included in your request.
     */
    public final Map<String, LanguageIdSettings> languageIdSettingsAsStrings() {
        return languageIdSettings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(transcriptionJobName());
        hashCode = 31 * hashCode + Objects.hashCode(transcriptionJobStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(languageCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(mediaSampleRateHertz());
        hashCode = 31 * hashCode + Objects.hashCode(mediaFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(media());
        hashCode = 31 * hashCode + Objects.hashCode(transcript());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(completionTime());
        hashCode = 31 * hashCode + Objects.hashCode(failureReason());
        hashCode = 31 * hashCode + Objects.hashCode(settings());
        hashCode = 31 * hashCode + Objects.hashCode(modelSettings());
        hashCode = 31 * hashCode + Objects.hashCode(jobExecutionSettings());
        hashCode = 31 * hashCode + Objects.hashCode(contentRedaction());
        hashCode = 31 * hashCode + Objects.hashCode(identifyLanguage());
        hashCode = 31 * hashCode + Objects.hashCode(identifyMultipleLanguages());
        hashCode = 31 * hashCode + Objects.hashCode(hasLanguageOptions() ? languageOptionsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(identifiedLanguageScore());
        hashCode = 31 * hashCode + Objects.hashCode(hasLanguageCodes() ? languageCodes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(subtitles());
        hashCode = 31 * hashCode + Objects.hashCode(hasLanguageIdSettings() ? languageIdSettingsAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TranscriptionJob)) {
            return false;
        }
        TranscriptionJob other = (TranscriptionJob) obj;
        return Objects.equals(transcriptionJobName(), other.transcriptionJobName())
                && Objects.equals(transcriptionJobStatusAsString(), other.transcriptionJobStatusAsString())
                && Objects.equals(languageCodeAsString(), other.languageCodeAsString())
                && Objects.equals(mediaSampleRateHertz(), other.mediaSampleRateHertz())
                && Objects.equals(mediaFormatAsString(), other.mediaFormatAsString()) && Objects.equals(media(), other.media())
                && Objects.equals(transcript(), other.transcript()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(completionTime(), other.completionTime())
                && Objects.equals(failureReason(), other.failureReason()) && Objects.equals(settings(), other.settings())
                && Objects.equals(modelSettings(), other.modelSettings())
                && Objects.equals(jobExecutionSettings(), other.jobExecutionSettings())
                && Objects.equals(contentRedaction(), other.contentRedaction())
                && Objects.equals(identifyLanguage(), other.identifyLanguage())
                && Objects.equals(identifyMultipleLanguages(), other.identifyMultipleLanguages())
                && hasLanguageOptions() == other.hasLanguageOptions()
                && Objects.equals(languageOptionsAsStrings(), other.languageOptionsAsStrings())
                && Objects.equals(identifiedLanguageScore(), other.identifiedLanguageScore())
                && hasLanguageCodes() == other.hasLanguageCodes() && Objects.equals(languageCodes(), other.languageCodes())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(subtitles(), other.subtitles()) && hasLanguageIdSettings() == other.hasLanguageIdSettings()
                && Objects.equals(languageIdSettingsAsStrings(), other.languageIdSettingsAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TranscriptionJob").add("TranscriptionJobName", transcriptionJobName())
                .add("TranscriptionJobStatus", transcriptionJobStatusAsString()).add("LanguageCode", languageCodeAsString())
                .add("MediaSampleRateHertz", mediaSampleRateHertz()).add("MediaFormat", mediaFormatAsString())
                .add("Media", media()).add("Transcript", transcript()).add("StartTime", startTime())
                .add("CreationTime", creationTime()).add("CompletionTime", completionTime())
                .add("FailureReason", failureReason()).add("Settings", settings()).add("ModelSettings", modelSettings())
                .add("JobExecutionSettings", jobExecutionSettings()).add("ContentRedaction", contentRedaction())
                .add("IdentifyLanguage", identifyLanguage()).add("IdentifyMultipleLanguages", identifyMultipleLanguages())
                .add("LanguageOptions", hasLanguageOptions() ? languageOptionsAsStrings() : null)
                .add("IdentifiedLanguageScore", identifiedLanguageScore())
                .add("LanguageCodes", hasLanguageCodes() ? languageCodes() : null).add("Tags", hasTags() ? tags() : null)
                .add("Subtitles", subtitles())
                .add("LanguageIdSettings", hasLanguageIdSettings() ? languageIdSettingsAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TranscriptionJobName":
            return Optional.ofNullable(clazz.cast(transcriptionJobName()));
        case "TranscriptionJobStatus":
            return Optional.ofNullable(clazz.cast(transcriptionJobStatusAsString()));
        case "LanguageCode":
            return Optional.ofNullable(clazz.cast(languageCodeAsString()));
        case "MediaSampleRateHertz":
            return Optional.ofNullable(clazz.cast(mediaSampleRateHertz()));
        case "MediaFormat":
            return Optional.ofNullable(clazz.cast(mediaFormatAsString()));
        case "Media":
            return Optional.ofNullable(clazz.cast(media()));
        case "Transcript":
            return Optional.ofNullable(clazz.cast(transcript()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "CompletionTime":
            return Optional.ofNullable(clazz.cast(completionTime()));
        case "FailureReason":
            return Optional.ofNullable(clazz.cast(failureReason()));
        case "Settings":
            return Optional.ofNullable(clazz.cast(settings()));
        case "ModelSettings":
            return Optional.ofNullable(clazz.cast(modelSettings()));
        case "JobExecutionSettings":
            return Optional.ofNullable(clazz.cast(jobExecutionSettings()));
        case "ContentRedaction":
            return Optional.ofNullable(clazz.cast(contentRedaction()));
        case "IdentifyLanguage":
            return Optional.ofNullable(clazz.cast(identifyLanguage()));
        case "IdentifyMultipleLanguages":
            return Optional.ofNullable(clazz.cast(identifyMultipleLanguages()));
        case "LanguageOptions":
            return Optional.ofNullable(clazz.cast(languageOptionsAsStrings()));
        case "IdentifiedLanguageScore":
            return Optional.ofNullable(clazz.cast(identifiedLanguageScore()));
        case "LanguageCodes":
            return Optional.ofNullable(clazz.cast(languageCodes()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "Subtitles":
            return Optional.ofNullable(clazz.cast(subtitles()));
        case "LanguageIdSettings":
            return Optional.ofNullable(clazz.cast(languageIdSettingsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TranscriptionJob, T> g) {
        return obj -> g.apply((TranscriptionJob) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TranscriptionJob> {
        /**
         * <p>
         * The name of the transcription job. Job names are case sensitive and must be unique within an Amazon Web
         * Services account.
         * </p>
         * 
         * @param transcriptionJobName
         *        The name of the transcription job. Job names are case sensitive and must be unique within an Amazon
         *        Web Services account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transcriptionJobName(String transcriptionJobName);

        /**
         * <p>
         * Provides the status of the specified transcription job.
         * </p>
         * <p>
         * If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
         * specified in <code>TranscriptFileUri</code> (or <code>RedactedTranscriptFileUri</code>, if you requested
         * transcript redaction). If the status is <code>FAILED</code>, <code>FailureReason</code> provides details on
         * why your transcription job failed.
         * </p>
         * 
         * @param transcriptionJobStatus
         *        Provides the status of the specified transcription job.</p>
         *        <p>
         *        If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the
         *        location specified in <code>TranscriptFileUri</code> (or <code>RedactedTranscriptFileUri</code>, if
         *        you requested transcript redaction). If the status is <code>FAILED</code>, <code>FailureReason</code>
         *        provides details on why your transcription job failed.
         * @see TranscriptionJobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscriptionJobStatus
         */
        Builder transcriptionJobStatus(String transcriptionJobStatus);

        /**
         * <p>
         * Provides the status of the specified transcription job.
         * </p>
         * <p>
         * If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
         * specified in <code>TranscriptFileUri</code> (or <code>RedactedTranscriptFileUri</code>, if you requested
         * transcript redaction). If the status is <code>FAILED</code>, <code>FailureReason</code> provides details on
         * why your transcription job failed.
         * </p>
         * 
         * @param transcriptionJobStatus
         *        Provides the status of the specified transcription job.</p>
         *        <p>
         *        If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the
         *        location specified in <code>TranscriptFileUri</code> (or <code>RedactedTranscriptFileUri</code>, if
         *        you requested transcript redaction). If the status is <code>FAILED</code>, <code>FailureReason</code>
         *        provides details on why your transcription job failed.
         * @see TranscriptionJobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscriptionJobStatus
         */
        Builder transcriptionJobStatus(TranscriptionJobStatus transcriptionJobStatus);

        /**
         * <p>
         * The language code used to create your transcription job. This parameter is used with single-language
         * identification. For multi-language identification requests, refer to the plural version of this parameter,
         * <code>LanguageCodes</code>.
         * </p>
         * 
         * @param languageCode
         *        The language code used to create your transcription job. This parameter is used with single-language
         *        identification. For multi-language identification requests, refer to the plural version of this
         *        parameter, <code>LanguageCodes</code>.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(String languageCode);

        /**
         * <p>
         * The language code used to create your transcription job. This parameter is used with single-language
         * identification. For multi-language identification requests, refer to the plural version of this parameter,
         * <code>LanguageCodes</code>.
         * </p>
         * 
         * @param languageCode
         *        The language code used to create your transcription job. This parameter is used with single-language
         *        identification. For multi-language identification requests, refer to the plural version of this
         *        parameter, <code>LanguageCodes</code>.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(LanguageCode languageCode);

        /**
         * <p>
         * The sample rate, in hertz, of the audio track in your input media file.
         * </p>
         * 
         * @param mediaSampleRateHertz
         *        The sample rate, in hertz, of the audio track in your input media file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mediaSampleRateHertz(Integer mediaSampleRateHertz);

        /**
         * <p>
         * The format of the input media file.
         * </p>
         * 
         * @param mediaFormat
         *        The format of the input media file.
         * @see MediaFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MediaFormat
         */
        Builder mediaFormat(String mediaFormat);

        /**
         * <p>
         * The format of the input media file.
         * </p>
         * 
         * @param mediaFormat
         *        The format of the input media file.
         * @see MediaFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MediaFormat
         */
        Builder mediaFormat(MediaFormat mediaFormat);

        /**
         * <p>
         * Provides the Amazon S3 location of the media file you used in your request.
         * </p>
         * 
         * @param media
         *        Provides the Amazon S3 location of the media file you used in your request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder media(Media media);

        /**
         * <p>
         * Provides the Amazon S3 location of the media file you used in your request.
         * </p>
         * This is a convenience method that creates an instance of the {@link Media.Builder} avoiding the need to
         * create one manually via {@link Media#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Media.Builder#build()} is called immediately and its result is
         * passed to {@link #media(Media)}.
         * 
         * @param media
         *        a consumer that will call methods on {@link Media.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #media(Media)
         */
        default Builder media(Consumer<Media.Builder> media) {
            return media(Media.builder().applyMutation(media).build());
        }

        /**
         * <p>
         * Provides you with the Amazon S3 URI you can use to access your transcript.
         * </p>
         * 
         * @param transcript
         *        Provides you with the Amazon S3 URI you can use to access your transcript.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transcript(Transcript transcript);

        /**
         * <p>
         * Provides you with the Amazon S3 URI you can use to access your transcript.
         * </p>
         * This is a convenience method that creates an instance of the {@link Transcript.Builder} avoiding the need to
         * create one manually via {@link Transcript#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Transcript.Builder#build()} is called immediately and its result
         * is passed to {@link #transcript(Transcript)}.
         * 
         * @param transcript
         *        a consumer that will call methods on {@link Transcript.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #transcript(Transcript)
         */
        default Builder transcript(Consumer<Transcript.Builder> transcript) {
            return transcript(Transcript.builder().applyMutation(transcript).build());
        }

        /**
         * <p>
         * The date and time the specified transcription job began processing.
         * </p>
         * <p>
         * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         * <code>2022-05-04T12:32:58.789000-07:00</code> represents a transcription job that started processing at 12:32
         * PM UTC-7 on May 4, 2022.
         * </p>
         * 
         * @param startTime
         *        The date and time the specified transcription job began processing.</p>
         *        <p>
         *        Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         *        <code>2022-05-04T12:32:58.789000-07:00</code> represents a transcription job that started processing
         *        at 12:32 PM UTC-7 on May 4, 2022.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The date and time the specified transcription job request was made.
         * </p>
         * <p>
         * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         * <code>2022-05-04T12:32:58.761000-07:00</code> represents a transcription job that started processing at 12:32
         * PM UTC-7 on May 4, 2022.
         * </p>
         * 
         * @param creationTime
         *        The date and time the specified transcription job request was made.</p>
         *        <p>
         *        Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         *        <code>2022-05-04T12:32:58.761000-07:00</code> represents a transcription job that started processing
         *        at 12:32 PM UTC-7 on May 4, 2022.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The date and time the specified transcription job finished processing.
         * </p>
         * <p>
         * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         * <code>2022-05-04T12:33:13.922000-07:00</code> represents a transcription job that started processing at 12:33
         * PM UTC-7 on May 4, 2022.
         * </p>
         * 
         * @param completionTime
         *        The date and time the specified transcription job finished processing.</p>
         *        <p>
         *        Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         *        <code>2022-05-04T12:33:13.922000-07:00</code> represents a transcription job that started processing
         *        at 12:33 PM UTC-7 on May 4, 2022.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completionTime(Instant completionTime);

        /**
         * <p>
         * If <code>TranscriptionJobStatus</code> is <code>FAILED</code>, <code>FailureReason</code> contains
         * information about why the transcription job request failed.
         * </p>
         * <p>
         * The <code>FailureReason</code> field contains one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Unsupported media format</code>.
         * </p>
         * <p>
         * The media format specified in <code>MediaFormat</code> isn't valid. Refer to <b>MediaFormat</b> for a list of
         * supported formats.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>The media format provided does not match the detected media format</code>.
         * </p>
         * <p>
         * The media format specified in <code>MediaFormat</code> doesn't match the format of the input file. Check the
         * media format of your media file and correct the specified value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Invalid sample rate for audio file</code>.
         * </p>
         * <p>
         * The sample rate specified in <code>MediaSampleRateHertz</code> isn't valid. The sample rate must be between
         * 8,000 and 48,000 hertz.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>The sample rate provided does not match the detected sample rate</code>.
         * </p>
         * <p>
         * The sample rate specified in <code>MediaSampleRateHertz</code> doesn't match the sample rate detected in your
         * input media file. Check the sample rate of your media file and correct the specified value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Invalid file size: file size too large</code>.
         * </p>
         * <p>
         * The size of your media file is larger than what Amazon Transcribe can process. For more information, refer to
         * <a href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Guidelines and
         * quotas</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Invalid number of channels: number of channels too large</code>.
         * </p>
         * <p>
         * Your audio contains more channels than Amazon Transcribe is able to process. For more information, refer to
         * <a href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Guidelines and
         * quotas</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param failureReason
         *        If <code>TranscriptionJobStatus</code> is <code>FAILED</code>, <code>FailureReason</code> contains
         *        information about why the transcription job request failed.</p>
         *        <p>
         *        The <code>FailureReason</code> field contains one of the following values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Unsupported media format</code>.
         *        </p>
         *        <p>
         *        The media format specified in <code>MediaFormat</code> isn't valid. Refer to <b>MediaFormat</b> for a
         *        list of supported formats.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>The media format provided does not match the detected media format</code>.
         *        </p>
         *        <p>
         *        The media format specified in <code>MediaFormat</code> doesn't match the format of the input file.
         *        Check the media format of your media file and correct the specified value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Invalid sample rate for audio file</code>.
         *        </p>
         *        <p>
         *        The sample rate specified in <code>MediaSampleRateHertz</code> isn't valid. The sample rate must be
         *        between 8,000 and 48,000 hertz.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>The sample rate provided does not match the detected sample rate</code>.
         *        </p>
         *        <p>
         *        The sample rate specified in <code>MediaSampleRateHertz</code> doesn't match the sample rate detected
         *        in your input media file. Check the sample rate of your media file and correct the specified value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Invalid file size: file size too large</code>.
         *        </p>
         *        <p>
         *        The size of your media file is larger than what Amazon Transcribe can process. For more information,
         *        refer to <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Guidelines and
         *        quotas</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Invalid number of channels: number of channels too large</code>.
         *        </p>
         *        <p>
         *        Your audio contains more channels than Amazon Transcribe is able to process. For more information,
         *        refer to <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/limits-guidelines.html#limits">Guidelines and
         *        quotas</a>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReason(String failureReason);

        /**
         * <p>
         * Provides information on any additional settings that were included in your request. Additional settings
         * include channel identification, alternative transcriptions, speaker partitioning, custom vocabularies, and
         * custom vocabulary filters.
         * </p>
         * 
         * @param settings
         *        Provides information on any additional settings that were included in your request. Additional
         *        settings include channel identification, alternative transcriptions, speaker partitioning, custom
         *        vocabularies, and custom vocabulary filters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder settings(Settings settings);

        /**
         * <p>
         * Provides information on any additional settings that were included in your request. Additional settings
         * include channel identification, alternative transcriptions, speaker partitioning, custom vocabularies, and
         * custom vocabulary filters.
         * </p>
         * This is a convenience method that creates an instance of the {@link Settings.Builder} avoiding the need to
         * create one manually via {@link Settings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Settings.Builder#build()} is called immediately and its result is
         * passed to {@link #settings(Settings)}.
         * 
         * @param settings
         *        a consumer that will call methods on {@link Settings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #settings(Settings)
         */
        default Builder settings(Consumer<Settings.Builder> settings) {
            return settings(Settings.builder().applyMutation(settings).build());
        }

        /**
         * <p>
         * Provides information on the custom language model you included in your request.
         * </p>
         * 
         * @param modelSettings
         *        Provides information on the custom language model you included in your request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelSettings(ModelSettings modelSettings);

        /**
         * <p>
         * Provides information on the custom language model you included in your request.
         * </p>
         * This is a convenience method that creates an instance of the {@link ModelSettings.Builder} avoiding the need
         * to create one manually via {@link ModelSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ModelSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #modelSettings(ModelSettings)}.
         * 
         * @param modelSettings
         *        a consumer that will call methods on {@link ModelSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #modelSettings(ModelSettings)
         */
        default Builder modelSettings(Consumer<ModelSettings.Builder> modelSettings) {
            return modelSettings(ModelSettings.builder().applyMutation(modelSettings).build());
        }

        /**
         * <p>
         * Provides information about how your transcription job was processed. This parameter shows if your request was
         * queued and what data access role was used.
         * </p>
         * 
         * @param jobExecutionSettings
         *        Provides information about how your transcription job was processed. This parameter shows if your
         *        request was queued and what data access role was used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobExecutionSettings(JobExecutionSettings jobExecutionSettings);

        /**
         * <p>
         * Provides information about how your transcription job was processed. This parameter shows if your request was
         * queued and what data access role was used.
         * </p>
         * This is a convenience method that creates an instance of the {@link JobExecutionSettings.Builder} avoiding
         * the need to create one manually via {@link JobExecutionSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link JobExecutionSettings.Builder#build()} is called immediately and
         * its result is passed to {@link #jobExecutionSettings(JobExecutionSettings)}.
         * 
         * @param jobExecutionSettings
         *        a consumer that will call methods on {@link JobExecutionSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #jobExecutionSettings(JobExecutionSettings)
         */
        default Builder jobExecutionSettings(Consumer<JobExecutionSettings.Builder> jobExecutionSettings) {
            return jobExecutionSettings(JobExecutionSettings.builder().applyMutation(jobExecutionSettings).build());
        }

        /**
         * <p>
         * Indicates whether redaction was enabled in your transcript.
         * </p>
         * 
         * @param contentRedaction
         *        Indicates whether redaction was enabled in your transcript.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contentRedaction(ContentRedaction contentRedaction);

        /**
         * <p>
         * Indicates whether redaction was enabled in your transcript.
         * </p>
         * This is a convenience method that creates an instance of the {@link ContentRedaction.Builder} avoiding the
         * need to create one manually via {@link ContentRedaction#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ContentRedaction.Builder#build()} is called immediately and its
         * result is passed to {@link #contentRedaction(ContentRedaction)}.
         * 
         * @param contentRedaction
         *        a consumer that will call methods on {@link ContentRedaction.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #contentRedaction(ContentRedaction)
         */
        default Builder contentRedaction(Consumer<ContentRedaction.Builder> contentRedaction) {
            return contentRedaction(ContentRedaction.builder().applyMutation(contentRedaction).build());
        }

        /**
         * <p>
         * Indicates whether automatic language identification was enabled (<code>TRUE</code>) for the specified
         * transcription job.
         * </p>
         * 
         * @param identifyLanguage
         *        Indicates whether automatic language identification was enabled (<code>TRUE</code>) for the specified
         *        transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identifyLanguage(Boolean identifyLanguage);

        /**
         * <p>
         * Indicates whether automatic multi-language identification was enabled (<code>TRUE</code>) for the specified
         * transcription job.
         * </p>
         * 
         * @param identifyMultipleLanguages
         *        Indicates whether automatic multi-language identification was enabled (<code>TRUE</code>) for the
         *        specified transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identifyMultipleLanguages(Boolean identifyMultipleLanguages);

        /**
         * <p>
         * Provides the language codes you specified in your request.
         * </p>
         * 
         * @param languageOptions
         *        Provides the language codes you specified in your request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageOptionsWithStrings(Collection<String> languageOptions);

        /**
         * <p>
         * Provides the language codes you specified in your request.
         * </p>
         * 
         * @param languageOptions
         *        Provides the language codes you specified in your request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageOptionsWithStrings(String... languageOptions);

        /**
         * <p>
         * Provides the language codes you specified in your request.
         * </p>
         * 
         * @param languageOptions
         *        Provides the language codes you specified in your request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageOptions(Collection<LanguageCode> languageOptions);

        /**
         * <p>
         * Provides the language codes you specified in your request.
         * </p>
         * 
         * @param languageOptions
         *        Provides the language codes you specified in your request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageOptions(LanguageCode... languageOptions);

        /**
         * <p>
         * The confidence score associated with the language identified in your media file.
         * </p>
         * <p>
         * Confidence scores are values between 0 and 1; a larger value indicates a higher probability that the
         * identified language correctly matches the language spoken in your media.
         * </p>
         * 
         * @param identifiedLanguageScore
         *        The confidence score associated with the language identified in your media file.</p>
         *        <p>
         *        Confidence scores are values between 0 and 1; a larger value indicates a higher probability that the
         *        identified language correctly matches the language spoken in your media.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identifiedLanguageScore(Float identifiedLanguageScore);

        /**
         * <p>
         * The language codes used to create your transcription job. This parameter is used with multi-language
         * identification. For single-language identification requests, refer to the singular version of this parameter,
         * <code>LanguageCode</code>.
         * </p>
         * 
         * @param languageCodes
         *        The language codes used to create your transcription job. This parameter is used with multi-language
         *        identification. For single-language identification requests, refer to the singular version of this
         *        parameter, <code>LanguageCode</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageCodes(Collection<LanguageCodeItem> languageCodes);

        /**
         * <p>
         * The language codes used to create your transcription job. This parameter is used with multi-language
         * identification. For single-language identification requests, refer to the singular version of this parameter,
         * <code>LanguageCode</code>.
         * </p>
         * 
         * @param languageCodes
         *        The language codes used to create your transcription job. This parameter is used with multi-language
         *        identification. For single-language identification requests, refer to the singular version of this
         *        parameter, <code>LanguageCode</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageCodes(LanguageCodeItem... languageCodes);

        /**
         * <p>
         * The language codes used to create your transcription job. This parameter is used with multi-language
         * identification. For single-language identification requests, refer to the singular version of this parameter,
         * <code>LanguageCode</code>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.transcribe.model.LanguageCodeItem.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.transcribe.model.LanguageCodeItem#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.transcribe.model.LanguageCodeItem.Builder#build()} is called
         * immediately and its result is passed to {@link #languageCodes(List<LanguageCodeItem>)}.
         * 
         * @param languageCodes
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.transcribe.model.LanguageCodeItem.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #languageCodes(java.util.Collection<LanguageCodeItem>)
         */
        Builder languageCodes(Consumer<LanguageCodeItem.Builder>... languageCodes);

        /**
         * <p>
         * The tags, each in the form of a key:value pair, assigned to the specified transcription job.
         * </p>
         * 
         * @param tags
         *        The tags, each in the form of a key:value pair, assigned to the specified transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags, each in the form of a key:value pair, assigned to the specified transcription job.
         * </p>
         * 
         * @param tags
         *        The tags, each in the form of a key:value pair, assigned to the specified transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags, each in the form of a key:value pair, assigned to the specified transcription job.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.transcribe.model.Tag.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.transcribe.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.transcribe.model.Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.transcribe.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Indicates whether subtitles were generated with your transcription.
         * </p>
         * 
         * @param subtitles
         *        Indicates whether subtitles were generated with your transcription.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subtitles(SubtitlesOutput subtitles);

        /**
         * <p>
         * Indicates whether subtitles were generated with your transcription.
         * </p>
         * This is a convenience method that creates an instance of the {@link SubtitlesOutput.Builder} avoiding the
         * need to create one manually via {@link SubtitlesOutput#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SubtitlesOutput.Builder#build()} is called immediately and its
         * result is passed to {@link #subtitles(SubtitlesOutput)}.
         * 
         * @param subtitles
         *        a consumer that will call methods on {@link SubtitlesOutput.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #subtitles(SubtitlesOutput)
         */
        default Builder subtitles(Consumer<SubtitlesOutput.Builder> subtitles) {
            return subtitles(SubtitlesOutput.builder().applyMutation(subtitles).build());
        }

        /**
         * <p>
         * Provides the name and language of all custom language models, custom vocabularies, and custom vocabulary
         * filters that you included in your request.
         * </p>
         * 
         * @param languageIdSettings
         *        Provides the name and language of all custom language models, custom vocabularies, and custom
         *        vocabulary filters that you included in your request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageIdSettingsWithStrings(Map<String, LanguageIdSettings> languageIdSettings);

        /**
         * <p>
         * Provides the name and language of all custom language models, custom vocabularies, and custom vocabulary
         * filters that you included in your request.
         * </p>
         * 
         * @param languageIdSettings
         *        Provides the name and language of all custom language models, custom vocabularies, and custom
         *        vocabulary filters that you included in your request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageIdSettings(Map<LanguageCode, LanguageIdSettings> languageIdSettings);
    }

    static final class BuilderImpl implements Builder {
        private String transcriptionJobName;

        private String transcriptionJobStatus;

        private String languageCode;

        private Integer mediaSampleRateHertz;

        private String mediaFormat;

        private Media media;

        private Transcript transcript;

        private Instant startTime;

        private Instant creationTime;

        private Instant completionTime;

        private String failureReason;

        private Settings settings;

        private ModelSettings modelSettings;

        private JobExecutionSettings jobExecutionSettings;

        private ContentRedaction contentRedaction;

        private Boolean identifyLanguage;

        private Boolean identifyMultipleLanguages;

        private List<String> languageOptions = DefaultSdkAutoConstructList.getInstance();

        private Float identifiedLanguageScore;

        private List<LanguageCodeItem> languageCodes = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private SubtitlesOutput subtitles;

        private Map<String, LanguageIdSettings> languageIdSettings = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(TranscriptionJob model) {
            transcriptionJobName(model.transcriptionJobName);
            transcriptionJobStatus(model.transcriptionJobStatus);
            languageCode(model.languageCode);
            mediaSampleRateHertz(model.mediaSampleRateHertz);
            mediaFormat(model.mediaFormat);
            media(model.media);
            transcript(model.transcript);
            startTime(model.startTime);
            creationTime(model.creationTime);
            completionTime(model.completionTime);
            failureReason(model.failureReason);
            settings(model.settings);
            modelSettings(model.modelSettings);
            jobExecutionSettings(model.jobExecutionSettings);
            contentRedaction(model.contentRedaction);
            identifyLanguage(model.identifyLanguage);
            identifyMultipleLanguages(model.identifyMultipleLanguages);
            languageOptionsWithStrings(model.languageOptions);
            identifiedLanguageScore(model.identifiedLanguageScore);
            languageCodes(model.languageCodes);
            tags(model.tags);
            subtitles(model.subtitles);
            languageIdSettingsWithStrings(model.languageIdSettings);
        }

        public final String getTranscriptionJobName() {
            return transcriptionJobName;
        }

        public final void setTranscriptionJobName(String transcriptionJobName) {
            this.transcriptionJobName = transcriptionJobName;
        }

        @Override
        public final Builder transcriptionJobName(String transcriptionJobName) {
            this.transcriptionJobName = transcriptionJobName;
            return this;
        }

        public final String getTranscriptionJobStatus() {
            return transcriptionJobStatus;
        }

        public final void setTranscriptionJobStatus(String transcriptionJobStatus) {
            this.transcriptionJobStatus = transcriptionJobStatus;
        }

        @Override
        public final Builder transcriptionJobStatus(String transcriptionJobStatus) {
            this.transcriptionJobStatus = transcriptionJobStatus;
            return this;
        }

        @Override
        public final Builder transcriptionJobStatus(TranscriptionJobStatus transcriptionJobStatus) {
            this.transcriptionJobStatus(transcriptionJobStatus == null ? null : transcriptionJobStatus.toString());
            return this;
        }

        public final String getLanguageCode() {
            return languageCode;
        }

        public final void setLanguageCode(String languageCode) {
            this.languageCode = languageCode;
        }

        @Override
        public final Builder languageCode(String languageCode) {
            this.languageCode = languageCode;
            return this;
        }

        @Override
        public final Builder languageCode(LanguageCode languageCode) {
            this.languageCode(languageCode == null ? null : languageCode.toString());
            return this;
        }

        public final Integer getMediaSampleRateHertz() {
            return mediaSampleRateHertz;
        }

        public final void setMediaSampleRateHertz(Integer mediaSampleRateHertz) {
            this.mediaSampleRateHertz = mediaSampleRateHertz;
        }

        @Override
        public final Builder mediaSampleRateHertz(Integer mediaSampleRateHertz) {
            this.mediaSampleRateHertz = mediaSampleRateHertz;
            return this;
        }

        public final String getMediaFormat() {
            return mediaFormat;
        }

        public final void setMediaFormat(String mediaFormat) {
            this.mediaFormat = mediaFormat;
        }

        @Override
        public final Builder mediaFormat(String mediaFormat) {
            this.mediaFormat = mediaFormat;
            return this;
        }

        @Override
        public final Builder mediaFormat(MediaFormat mediaFormat) {
            this.mediaFormat(mediaFormat == null ? null : mediaFormat.toString());
            return this;
        }

        public final Media.Builder getMedia() {
            return media != null ? media.toBuilder() : null;
        }

        public final void setMedia(Media.BuilderImpl media) {
            this.media = media != null ? media.build() : null;
        }

        @Override
        public final Builder media(Media media) {
            this.media = media;
            return this;
        }

        public final Transcript.Builder getTranscript() {
            return transcript != null ? transcript.toBuilder() : null;
        }

        public final void setTranscript(Transcript.BuilderImpl transcript) {
            this.transcript = transcript != null ? transcript.build() : null;
        }

        @Override
        public final Builder transcript(Transcript transcript) {
            this.transcript = transcript;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final Instant getCompletionTime() {
            return completionTime;
        }

        public final void setCompletionTime(Instant completionTime) {
            this.completionTime = completionTime;
        }

        @Override
        public final Builder completionTime(Instant completionTime) {
            this.completionTime = completionTime;
            return this;
        }

        public final String getFailureReason() {
            return failureReason;
        }

        public final void setFailureReason(String failureReason) {
            this.failureReason = failureReason;
        }

        @Override
        public final Builder failureReason(String failureReason) {
            this.failureReason = failureReason;
            return this;
        }

        public final Settings.Builder getSettings() {
            return settings != null ? settings.toBuilder() : null;
        }

        public final void setSettings(Settings.BuilderImpl settings) {
            this.settings = settings != null ? settings.build() : null;
        }

        @Override
        @Transient
        public final Builder settings(Settings settings) {
            this.settings = settings;
            return this;
        }

        public final ModelSettings.Builder getModelSettings() {
            return modelSettings != null ? modelSettings.toBuilder() : null;
        }

        public final void setModelSettings(ModelSettings.BuilderImpl modelSettings) {
            this.modelSettings = modelSettings != null ? modelSettings.build() : null;
        }

        @Override
        public final Builder modelSettings(ModelSettings modelSettings) {
            this.modelSettings = modelSettings;
            return this;
        }

        public final JobExecutionSettings.Builder getJobExecutionSettings() {
            return jobExecutionSettings != null ? jobExecutionSettings.toBuilder() : null;
        }

        public final void setJobExecutionSettings(JobExecutionSettings.BuilderImpl jobExecutionSettings) {
            this.jobExecutionSettings = jobExecutionSettings != null ? jobExecutionSettings.build() : null;
        }

        @Override
        public final Builder jobExecutionSettings(JobExecutionSettings jobExecutionSettings) {
            this.jobExecutionSettings = jobExecutionSettings;
            return this;
        }

        public final ContentRedaction.Builder getContentRedaction() {
            return contentRedaction != null ? contentRedaction.toBuilder() : null;
        }

        public final void setContentRedaction(ContentRedaction.BuilderImpl contentRedaction) {
            this.contentRedaction = contentRedaction != null ? contentRedaction.build() : null;
        }

        @Override
        public final Builder contentRedaction(ContentRedaction contentRedaction) {
            this.contentRedaction = contentRedaction;
            return this;
        }

        public final Boolean getIdentifyLanguage() {
            return identifyLanguage;
        }

        public final void setIdentifyLanguage(Boolean identifyLanguage) {
            this.identifyLanguage = identifyLanguage;
        }

        @Override
        public final Builder identifyLanguage(Boolean identifyLanguage) {
            this.identifyLanguage = identifyLanguage;
            return this;
        }

        public final Boolean getIdentifyMultipleLanguages() {
            return identifyMultipleLanguages;
        }

        public final void setIdentifyMultipleLanguages(Boolean identifyMultipleLanguages) {
            this.identifyMultipleLanguages = identifyMultipleLanguages;
        }

        @Override
        public final Builder identifyMultipleLanguages(Boolean identifyMultipleLanguages) {
            this.identifyMultipleLanguages = identifyMultipleLanguages;
            return this;
        }

        public final Collection<String> getLanguageOptions() {
            if (languageOptions instanceof SdkAutoConstructList) {
                return null;
            }
            return languageOptions;
        }

        public final void setLanguageOptions(Collection<String> languageOptions) {
            this.languageOptions = LanguageOptionsCopier.copy(languageOptions);
        }

        @Override
        public final Builder languageOptionsWithStrings(Collection<String> languageOptions) {
            this.languageOptions = LanguageOptionsCopier.copy(languageOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder languageOptionsWithStrings(String... languageOptions) {
            languageOptionsWithStrings(Arrays.asList(languageOptions));
            return this;
        }

        @Override
        public final Builder languageOptions(Collection<LanguageCode> languageOptions) {
            this.languageOptions = LanguageOptionsCopier.copyEnumToString(languageOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder languageOptions(LanguageCode... languageOptions) {
            languageOptions(Arrays.asList(languageOptions));
            return this;
        }

        public final Float getIdentifiedLanguageScore() {
            return identifiedLanguageScore;
        }

        public final void setIdentifiedLanguageScore(Float identifiedLanguageScore) {
            this.identifiedLanguageScore = identifiedLanguageScore;
        }

        @Override
        public final Builder identifiedLanguageScore(Float identifiedLanguageScore) {
            this.identifiedLanguageScore = identifiedLanguageScore;
            return this;
        }

        public final List<LanguageCodeItem.Builder> getLanguageCodes() {
            List<LanguageCodeItem.Builder> result = LanguageCodeListCopier.copyToBuilder(this.languageCodes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setLanguageCodes(Collection<LanguageCodeItem.BuilderImpl> languageCodes) {
            this.languageCodes = LanguageCodeListCopier.copyFromBuilder(languageCodes);
        }

        @Override
        public final Builder languageCodes(Collection<LanguageCodeItem> languageCodes) {
            this.languageCodes = LanguageCodeListCopier.copy(languageCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder languageCodes(LanguageCodeItem... languageCodes) {
            languageCodes(Arrays.asList(languageCodes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder languageCodes(Consumer<LanguageCodeItem.Builder>... languageCodes) {
            languageCodes(Stream.of(languageCodes).map(c -> LanguageCodeItem.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final SubtitlesOutput.Builder getSubtitles() {
            return subtitles != null ? subtitles.toBuilder() : null;
        }

        public final void setSubtitles(SubtitlesOutput.BuilderImpl subtitles) {
            this.subtitles = subtitles != null ? subtitles.build() : null;
        }

        @Override
        public final Builder subtitles(SubtitlesOutput subtitles) {
            this.subtitles = subtitles;
            return this;
        }

        public final Map<String, LanguageIdSettings.Builder> getLanguageIdSettings() {
            Map<String, LanguageIdSettings.Builder> result = LanguageIdSettingsMapCopier.copyToBuilder(this.languageIdSettings);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setLanguageIdSettings(Map<String, LanguageIdSettings.BuilderImpl> languageIdSettings) {
            this.languageIdSettings = LanguageIdSettingsMapCopier.copyFromBuilder(languageIdSettings);
        }

        @Override
        public final Builder languageIdSettingsWithStrings(Map<String, LanguageIdSettings> languageIdSettings) {
            this.languageIdSettings = LanguageIdSettingsMapCopier.copy(languageIdSettings);
            return this;
        }

        @Override
        public final Builder languageIdSettings(Map<LanguageCode, LanguageIdSettings> languageIdSettings) {
            this.languageIdSettings = LanguageIdSettingsMapCopier.copyEnumToString(languageIdSettings);
            return this;
        }

        @Override
        public TranscriptionJob build() {
            return new TranscriptionJob(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
