/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribe;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.transcribe.internal.TranscribeServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.transcribe.model.BadRequestException;
import software.amazon.awssdk.services.transcribe.model.ConflictException;
import software.amazon.awssdk.services.transcribe.model.CreateCallAnalyticsCategoryRequest;
import software.amazon.awssdk.services.transcribe.model.CreateCallAnalyticsCategoryResponse;
import software.amazon.awssdk.services.transcribe.model.CreateLanguageModelRequest;
import software.amazon.awssdk.services.transcribe.model.CreateLanguageModelResponse;
import software.amazon.awssdk.services.transcribe.model.CreateMedicalVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.CreateMedicalVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.CreateVocabularyFilterRequest;
import software.amazon.awssdk.services.transcribe.model.CreateVocabularyFilterResponse;
import software.amazon.awssdk.services.transcribe.model.CreateVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.CreateVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteCallAnalyticsCategoryRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteCallAnalyticsCategoryResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteCallAnalyticsJobRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteCallAnalyticsJobResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteLanguageModelRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteLanguageModelResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteMedicalScribeJobRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteMedicalScribeJobResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteMedicalTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteMedicalTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteMedicalVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteMedicalVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteVocabularyFilterRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteVocabularyFilterResponse;
import software.amazon.awssdk.services.transcribe.model.DeleteVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.DeleteVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.DescribeLanguageModelRequest;
import software.amazon.awssdk.services.transcribe.model.DescribeLanguageModelResponse;
import software.amazon.awssdk.services.transcribe.model.GetCallAnalyticsCategoryRequest;
import software.amazon.awssdk.services.transcribe.model.GetCallAnalyticsCategoryResponse;
import software.amazon.awssdk.services.transcribe.model.GetCallAnalyticsJobRequest;
import software.amazon.awssdk.services.transcribe.model.GetCallAnalyticsJobResponse;
import software.amazon.awssdk.services.transcribe.model.GetMedicalScribeJobRequest;
import software.amazon.awssdk.services.transcribe.model.GetMedicalScribeJobResponse;
import software.amazon.awssdk.services.transcribe.model.GetMedicalTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.GetMedicalTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.GetMedicalVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.GetMedicalVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.GetTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.GetTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.GetVocabularyFilterRequest;
import software.amazon.awssdk.services.transcribe.model.GetVocabularyFilterResponse;
import software.amazon.awssdk.services.transcribe.model.GetVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.GetVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.InternalFailureException;
import software.amazon.awssdk.services.transcribe.model.LimitExceededException;
import software.amazon.awssdk.services.transcribe.model.ListCallAnalyticsCategoriesRequest;
import software.amazon.awssdk.services.transcribe.model.ListCallAnalyticsCategoriesResponse;
import software.amazon.awssdk.services.transcribe.model.ListCallAnalyticsJobsRequest;
import software.amazon.awssdk.services.transcribe.model.ListCallAnalyticsJobsResponse;
import software.amazon.awssdk.services.transcribe.model.ListLanguageModelsRequest;
import software.amazon.awssdk.services.transcribe.model.ListLanguageModelsResponse;
import software.amazon.awssdk.services.transcribe.model.ListMedicalScribeJobsRequest;
import software.amazon.awssdk.services.transcribe.model.ListMedicalScribeJobsResponse;
import software.amazon.awssdk.services.transcribe.model.ListMedicalTranscriptionJobsRequest;
import software.amazon.awssdk.services.transcribe.model.ListMedicalTranscriptionJobsResponse;
import software.amazon.awssdk.services.transcribe.model.ListMedicalVocabulariesRequest;
import software.amazon.awssdk.services.transcribe.model.ListMedicalVocabulariesResponse;
import software.amazon.awssdk.services.transcribe.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.transcribe.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.transcribe.model.ListTranscriptionJobsRequest;
import software.amazon.awssdk.services.transcribe.model.ListTranscriptionJobsResponse;
import software.amazon.awssdk.services.transcribe.model.ListVocabulariesRequest;
import software.amazon.awssdk.services.transcribe.model.ListVocabulariesResponse;
import software.amazon.awssdk.services.transcribe.model.ListVocabularyFiltersRequest;
import software.amazon.awssdk.services.transcribe.model.ListVocabularyFiltersResponse;
import software.amazon.awssdk.services.transcribe.model.NotFoundException;
import software.amazon.awssdk.services.transcribe.model.StartCallAnalyticsJobRequest;
import software.amazon.awssdk.services.transcribe.model.StartCallAnalyticsJobResponse;
import software.amazon.awssdk.services.transcribe.model.StartMedicalScribeJobRequest;
import software.amazon.awssdk.services.transcribe.model.StartMedicalScribeJobResponse;
import software.amazon.awssdk.services.transcribe.model.StartMedicalTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.StartMedicalTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.StartTranscriptionJobRequest;
import software.amazon.awssdk.services.transcribe.model.StartTranscriptionJobResponse;
import software.amazon.awssdk.services.transcribe.model.TagResourceRequest;
import software.amazon.awssdk.services.transcribe.model.TagResourceResponse;
import software.amazon.awssdk.services.transcribe.model.TranscribeException;
import software.amazon.awssdk.services.transcribe.model.UntagResourceRequest;
import software.amazon.awssdk.services.transcribe.model.UntagResourceResponse;
import software.amazon.awssdk.services.transcribe.model.UpdateCallAnalyticsCategoryRequest;
import software.amazon.awssdk.services.transcribe.model.UpdateCallAnalyticsCategoryResponse;
import software.amazon.awssdk.services.transcribe.model.UpdateMedicalVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.UpdateMedicalVocabularyResponse;
import software.amazon.awssdk.services.transcribe.model.UpdateVocabularyFilterRequest;
import software.amazon.awssdk.services.transcribe.model.UpdateVocabularyFilterResponse;
import software.amazon.awssdk.services.transcribe.model.UpdateVocabularyRequest;
import software.amazon.awssdk.services.transcribe.model.UpdateVocabularyResponse;
import software.amazon.awssdk.services.transcribe.transform.CreateCallAnalyticsCategoryRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.CreateLanguageModelRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.CreateMedicalVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.CreateVocabularyFilterRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.CreateVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteCallAnalyticsCategoryRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteCallAnalyticsJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteLanguageModelRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteMedicalScribeJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteMedicalTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteMedicalVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteVocabularyFilterRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DeleteVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.DescribeLanguageModelRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetCallAnalyticsCategoryRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetCallAnalyticsJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetMedicalScribeJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetMedicalTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetMedicalVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetVocabularyFilterRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.GetVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListCallAnalyticsCategoriesRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListCallAnalyticsJobsRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListLanguageModelsRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListMedicalScribeJobsRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListMedicalTranscriptionJobsRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListMedicalVocabulariesRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListTranscriptionJobsRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListVocabulariesRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.ListVocabularyFiltersRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.StartCallAnalyticsJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.StartMedicalScribeJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.StartMedicalTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.StartTranscriptionJobRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.UpdateCallAnalyticsCategoryRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.UpdateMedicalVocabularyRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.UpdateVocabularyFilterRequestMarshaller;
import software.amazon.awssdk.services.transcribe.transform.UpdateVocabularyRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link TranscribeAsyncClient}.
 *
 * @see TranscribeAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTranscribeAsyncClient implements TranscribeAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultTranscribeAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultTranscribeAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a new Call Analytics category.
     * </p>
     * <p>
     * All categories are automatically applied to your Call Analytics transcriptions. Note that in order to apply
     * categories to your transcriptions, you must create them before submitting your transcription request, as
     * categories cannot be applied retroactively.
     * </p>
     * <p>
     * When creating a new category, you can use the <code>InputType</code> parameter to label the category as a
     * <code>POST_CALL</code> or a <code>REAL_TIME</code> category. <code>POST_CALL</code> categories can only be
     * applied to post-call transcriptions and <code>REAL_TIME</code> categories can only be applied to real-time
     * transcriptions. If you do not include <code>InputType</code>, your category is created as a
     * <code>POST_CALL</code> category by default.
     * </p>
     * <p>
     * Call Analytics categories are composed of rules. For each category, you must create between 1 and 20 rules. Rules
     * can include these parameters: , , , and .
     * </p>
     * <p>
     * To update an existing category, see .
     * </p>
     * <p>
     * To learn more about Call Analytics categories, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/tca-categories-batch.html">Creating categories for
     * post-call transcriptions</a> and <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/tca-categories-stream.html">Creating categories for
     * real-time transcriptions</a>.
     * </p>
     *
     * @param createCallAnalyticsCategoryRequest
     * @return A Java Future containing the result of the CreateCallAnalyticsCategory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>ConflictException A resource already exists with this name. Resource names must be unique within an
     *         Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.CreateCallAnalyticsCategory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/CreateCallAnalyticsCategory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateCallAnalyticsCategoryResponse> createCallAnalyticsCategory(
            CreateCallAnalyticsCategoryRequest createCallAnalyticsCategoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createCallAnalyticsCategoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCallAnalyticsCategoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCallAnalyticsCategory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateCallAnalyticsCategoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateCallAnalyticsCategoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateCallAnalyticsCategoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateCallAnalyticsCategoryRequest, CreateCallAnalyticsCategoryResponse>()
                            .withOperationName("CreateCallAnalyticsCategory").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateCallAnalyticsCategoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createCallAnalyticsCategoryRequest));
            CompletableFuture<CreateCallAnalyticsCategoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new custom language model.
     * </p>
     * <p>
     * When creating a new custom language model, you must specify:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you want a Wideband (audio sample rates over 16,000 Hz) or Narrowband (audio sample rates under 16,000 Hz)
     * base model
     * </p>
     * </li>
     * <li>
     * <p>
     * The location of your training and tuning files (this must be an Amazon S3 URI)
     * </p>
     * </li>
     * <li>
     * <p>
     * The language of your model
     * </p>
     * </li>
     * <li>
     * <p>
     * A unique name for your model
     * </p>
     * </li>
     * </ul>
     *
     * @param createLanguageModelRequest
     * @return A Java Future containing the result of the CreateLanguageModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>ConflictException A resource already exists with this name. Resource names must be unique within an
     *         Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.CreateLanguageModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/CreateLanguageModel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLanguageModelResponse> createLanguageModel(
            CreateLanguageModelRequest createLanguageModelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createLanguageModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLanguageModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLanguageModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLanguageModelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLanguageModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLanguageModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLanguageModelRequest, CreateLanguageModelResponse>()
                            .withOperationName("CreateLanguageModel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateLanguageModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createLanguageModelRequest));
            CompletableFuture<CreateLanguageModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new custom medical vocabulary.
     * </p>
     * <p>
     * Before creating a new custom medical vocabulary, you must first upload a text file that contains your vocabulary
     * table into an Amazon S3 bucket. Note that this differs from , where you can include a list of terms within your
     * request using the <code>Phrases</code> flag; <code>CreateMedicalVocabulary</code> does not support the
     * <code>Phrases</code> flag and only accepts vocabularies in table format.
     * </p>
     * <p>
     * Each language has a character set that contains all allowed characters for that specific language. If you use
     * unsupported characters, your custom vocabulary request fails. Refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/charsets.html">Character Sets for Custom Vocabularies</a>
     * to get the character set for your language.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/custom-vocabulary.html">Custom vocabularies</a>.
     * </p>
     *
     * @param createMedicalVocabularyRequest
     * @return A Java Future containing the result of the CreateMedicalVocabulary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>ConflictException A resource already exists with this name. Resource names must be unique within an
     *         Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.CreateMedicalVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/CreateMedicalVocabulary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMedicalVocabularyResponse> createMedicalVocabulary(
            CreateMedicalVocabularyRequest createMedicalVocabularyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createMedicalVocabularyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMedicalVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMedicalVocabulary");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMedicalVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateMedicalVocabularyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMedicalVocabularyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMedicalVocabularyRequest, CreateMedicalVocabularyResponse>()
                            .withOperationName("CreateMedicalVocabulary").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateMedicalVocabularyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createMedicalVocabularyRequest));
            CompletableFuture<CreateMedicalVocabularyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new custom vocabulary.
     * </p>
     * <p>
     * When creating a new custom vocabulary, you can either upload a text file that contains your new entries, phrases,
     * and terms into an Amazon S3 bucket and include the URI in your request. Or you can include a list of terms
     * directly in your request using the <code>Phrases</code> flag.
     * </p>
     * <p>
     * Each language has a character set that contains all allowed characters for that specific language. If you use
     * unsupported characters, your custom vocabulary request fails. Refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/charsets.html">Character Sets for Custom Vocabularies</a>
     * to get the character set for your language.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/custom-vocabulary.html">Custom vocabularies</a>.
     * </p>
     *
     * @param createVocabularyRequest
     * @return A Java Future containing the result of the CreateVocabulary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>ConflictException A resource already exists with this name. Resource names must be unique within an
     *         Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.CreateVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/CreateVocabulary" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateVocabularyResponse> createVocabulary(CreateVocabularyRequest createVocabularyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createVocabularyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVocabulary");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateVocabularyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateVocabularyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateVocabularyRequest, CreateVocabularyResponse>()
                            .withOperationName("CreateVocabulary").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateVocabularyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createVocabularyRequest));
            CompletableFuture<CreateVocabularyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new custom vocabulary filter.
     * </p>
     * <p>
     * You can use custom vocabulary filters to mask, delete, or flag specific words from your transcript. Custom
     * vocabulary filters are commonly used to mask profanity in transcripts.
     * </p>
     * <p>
     * Each language has a character set that contains all allowed characters for that specific language. If you use
     * unsupported characters, your custom vocabulary filter request fails. Refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/charsets.html">Character Sets for Custom Vocabularies</a>
     * to get the character set for your language.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/vocabulary-filtering.html">Vocabulary filtering</a>.
     * </p>
     *
     * @param createVocabularyFilterRequest
     * @return A Java Future containing the result of the CreateVocabularyFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>ConflictException A resource already exists with this name. Resource names must be unique within an
     *         Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.CreateVocabularyFilter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/CreateVocabularyFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateVocabularyFilterResponse> createVocabularyFilter(
            CreateVocabularyFilterRequest createVocabularyFilterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createVocabularyFilterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVocabularyFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVocabularyFilter");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateVocabularyFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateVocabularyFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateVocabularyFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateVocabularyFilterRequest, CreateVocabularyFilterResponse>()
                            .withOperationName("CreateVocabularyFilter").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateVocabularyFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createVocabularyFilterRequest));
            CompletableFuture<CreateVocabularyFilterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a Call Analytics category. To use this operation, specify the name of the category you want to delete
     * using <code>CategoryName</code>. Category names are case sensitive.
     * </p>
     *
     * @param deleteCallAnalyticsCategoryRequest
     * @return A Java Future containing the result of the DeleteCallAnalyticsCategory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException We can't find the requested resource. Check that the specified name is correct and
     *         try your request again.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.DeleteCallAnalyticsCategory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteCallAnalyticsCategory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCallAnalyticsCategoryResponse> deleteCallAnalyticsCategory(
            DeleteCallAnalyticsCategoryRequest deleteCallAnalyticsCategoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCallAnalyticsCategoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCallAnalyticsCategoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCallAnalyticsCategory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteCallAnalyticsCategoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteCallAnalyticsCategoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteCallAnalyticsCategoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCallAnalyticsCategoryRequest, DeleteCallAnalyticsCategoryResponse>()
                            .withOperationName("DeleteCallAnalyticsCategory").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteCallAnalyticsCategoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteCallAnalyticsCategoryRequest));
            CompletableFuture<DeleteCallAnalyticsCategoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a Call Analytics job. To use this operation, specify the name of the job you want to delete using
     * <code>CallAnalyticsJobName</code>. Job names are case sensitive.
     * </p>
     *
     * @param deleteCallAnalyticsJobRequest
     * @return A Java Future containing the result of the DeleteCallAnalyticsJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.DeleteCallAnalyticsJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteCallAnalyticsJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCallAnalyticsJobResponse> deleteCallAnalyticsJob(
            DeleteCallAnalyticsJobRequest deleteCallAnalyticsJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCallAnalyticsJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCallAnalyticsJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCallAnalyticsJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteCallAnalyticsJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteCallAnalyticsJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteCallAnalyticsJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCallAnalyticsJobRequest, DeleteCallAnalyticsJobResponse>()
                            .withOperationName("DeleteCallAnalyticsJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteCallAnalyticsJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteCallAnalyticsJobRequest));
            CompletableFuture<DeleteCallAnalyticsJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a custom language model. To use this operation, specify the name of the language model you want to delete
     * using <code>ModelName</code>. custom language model names are case sensitive.
     * </p>
     *
     * @param deleteLanguageModelRequest
     * @return A Java Future containing the result of the DeleteLanguageModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.DeleteLanguageModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteLanguageModel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLanguageModelResponse> deleteLanguageModel(
            DeleteLanguageModelRequest deleteLanguageModelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteLanguageModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLanguageModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLanguageModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLanguageModelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteLanguageModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLanguageModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLanguageModelRequest, DeleteLanguageModelResponse>()
                            .withOperationName("DeleteLanguageModel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteLanguageModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteLanguageModelRequest));
            CompletableFuture<DeleteLanguageModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a Medical Scribe job. To use this operation, specify the name of the job you want to delete using
     * <code>MedicalScribeJobName</code>. Job names are case sensitive.
     * </p>
     *
     * @param deleteMedicalScribeJobRequest
     * @return A Java Future containing the result of the DeleteMedicalScribeJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.DeleteMedicalScribeJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteMedicalScribeJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMedicalScribeJobResponse> deleteMedicalScribeJob(
            DeleteMedicalScribeJobRequest deleteMedicalScribeJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMedicalScribeJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMedicalScribeJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMedicalScribeJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMedicalScribeJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteMedicalScribeJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMedicalScribeJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMedicalScribeJobRequest, DeleteMedicalScribeJobResponse>()
                            .withOperationName("DeleteMedicalScribeJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteMedicalScribeJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteMedicalScribeJobRequest));
            CompletableFuture<DeleteMedicalScribeJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a medical transcription job. To use this operation, specify the name of the job you want to delete using
     * <code>MedicalTranscriptionJobName</code>. Job names are case sensitive.
     * </p>
     *
     * @param deleteMedicalTranscriptionJobRequest
     * @return A Java Future containing the result of the DeleteMedicalTranscriptionJob operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.DeleteMedicalTranscriptionJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteMedicalTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMedicalTranscriptionJobResponse> deleteMedicalTranscriptionJob(
            DeleteMedicalTranscriptionJobRequest deleteMedicalTranscriptionJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMedicalTranscriptionJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteMedicalTranscriptionJobRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMedicalTranscriptionJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMedicalTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteMedicalTranscriptionJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMedicalTranscriptionJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMedicalTranscriptionJobRequest, DeleteMedicalTranscriptionJobResponse>()
                            .withOperationName("DeleteMedicalTranscriptionJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteMedicalTranscriptionJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteMedicalTranscriptionJobRequest));
            CompletableFuture<DeleteMedicalTranscriptionJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a custom medical vocabulary. To use this operation, specify the name of the custom vocabulary you want to
     * delete using <code>VocabularyName</code>. Custom vocabulary names are case sensitive.
     * </p>
     *
     * @param deleteMedicalVocabularyRequest
     * @return A Java Future containing the result of the DeleteMedicalVocabulary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException We can't find the requested resource. Check that the specified name is correct and
     *         try your request again.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.DeleteMedicalVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteMedicalVocabulary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMedicalVocabularyResponse> deleteMedicalVocabulary(
            DeleteMedicalVocabularyRequest deleteMedicalVocabularyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMedicalVocabularyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMedicalVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMedicalVocabulary");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMedicalVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteMedicalVocabularyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMedicalVocabularyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMedicalVocabularyRequest, DeleteMedicalVocabularyResponse>()
                            .withOperationName("DeleteMedicalVocabulary").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteMedicalVocabularyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteMedicalVocabularyRequest));
            CompletableFuture<DeleteMedicalVocabularyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a transcription job. To use this operation, specify the name of the job you want to delete using
     * <code>TranscriptionJobName</code>. Job names are case sensitive.
     * </p>
     *
     * @param deleteTranscriptionJobRequest
     * @return A Java Future containing the result of the DeleteTranscriptionJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.DeleteTranscriptionJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTranscriptionJobResponse> deleteTranscriptionJob(
            DeleteTranscriptionJobRequest deleteTranscriptionJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTranscriptionJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTranscriptionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTranscriptionJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteTranscriptionJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTranscriptionJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTranscriptionJobRequest, DeleteTranscriptionJobResponse>()
                            .withOperationName("DeleteTranscriptionJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteTranscriptionJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteTranscriptionJobRequest));
            CompletableFuture<DeleteTranscriptionJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a custom vocabulary. To use this operation, specify the name of the custom vocabulary you want to delete
     * using <code>VocabularyName</code>. Custom vocabulary names are case sensitive.
     * </p>
     *
     * @param deleteVocabularyRequest
     * @return A Java Future containing the result of the DeleteVocabulary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException We can't find the requested resource. Check that the specified name is correct and
     *         try your request again.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.DeleteVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteVocabulary" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVocabularyResponse> deleteVocabulary(DeleteVocabularyRequest deleteVocabularyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVocabularyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVocabulary");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteVocabularyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteVocabularyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVocabularyRequest, DeleteVocabularyResponse>()
                            .withOperationName("DeleteVocabulary").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteVocabularyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteVocabularyRequest));
            CompletableFuture<DeleteVocabularyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a custom vocabulary filter. To use this operation, specify the name of the custom vocabulary filter you
     * want to delete using <code>VocabularyFilterName</code>. Custom vocabulary filter names are case sensitive.
     * </p>
     *
     * @param deleteVocabularyFilterRequest
     * @return A Java Future containing the result of the DeleteVocabularyFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException We can't find the requested resource. Check that the specified name is correct and
     *         try your request again.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.DeleteVocabularyFilter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DeleteVocabularyFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVocabularyFilterResponse> deleteVocabularyFilter(
            DeleteVocabularyFilterRequest deleteVocabularyFilterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVocabularyFilterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVocabularyFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVocabularyFilter");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVocabularyFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteVocabularyFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteVocabularyFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVocabularyFilterRequest, DeleteVocabularyFilterResponse>()
                            .withOperationName("DeleteVocabularyFilter").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteVocabularyFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteVocabularyFilterRequest));
            CompletableFuture<DeleteVocabularyFilterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about the specified custom language model.
     * </p>
     * <p>
     * This operation also shows if the base language model that you used to create your custom language model has been
     * updated. If Amazon Transcribe has updated the base model, you can create a new custom language model using the
     * updated base model.
     * </p>
     * <p>
     * If you tried to create a new custom language model and the request wasn't successful, you can use
     * <code>DescribeLanguageModel</code> to help identify the reason for this failure.
     * </p>
     *
     * @param describeLanguageModelRequest
     * @return A Java Future containing the result of the DescribeLanguageModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>NotFoundException We can't find the requested resource. Check that the specified name is correct and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.DescribeLanguageModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/DescribeLanguageModel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLanguageModelResponse> describeLanguageModel(
            DescribeLanguageModelRequest describeLanguageModelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeLanguageModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLanguageModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLanguageModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLanguageModelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLanguageModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeLanguageModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLanguageModelRequest, DescribeLanguageModelResponse>()
                            .withOperationName("DescribeLanguageModel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeLanguageModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeLanguageModelRequest));
            CompletableFuture<DescribeLanguageModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about the specified Call Analytics category.
     * </p>
     * <p>
     * To get a list of your Call Analytics categories, use the operation.
     * </p>
     *
     * @param getCallAnalyticsCategoryRequest
     * @return A Java Future containing the result of the GetCallAnalyticsCategory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException We can't find the requested resource. Check that the specified name is correct and
     *         try your request again.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.GetCallAnalyticsCategory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetCallAnalyticsCategory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCallAnalyticsCategoryResponse> getCallAnalyticsCategory(
            GetCallAnalyticsCategoryRequest getCallAnalyticsCategoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCallAnalyticsCategoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCallAnalyticsCategoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCallAnalyticsCategory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCallAnalyticsCategoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCallAnalyticsCategoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCallAnalyticsCategoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCallAnalyticsCategoryRequest, GetCallAnalyticsCategoryResponse>()
                            .withOperationName("GetCallAnalyticsCategory").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetCallAnalyticsCategoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getCallAnalyticsCategoryRequest));
            CompletableFuture<GetCallAnalyticsCategoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about the specified Call Analytics job.
     * </p>
     * <p>
     * To view the job's status, refer to <code>CallAnalyticsJobStatus</code>. If the status is <code>COMPLETED</code>,
     * the job is finished. You can find your completed transcript at the URI specified in
     * <code>TranscriptFileUri</code>. If the status is <code>FAILED</code>, <code>FailureReason</code> provides details
     * on why your transcription job failed.
     * </p>
     * <p>
     * If you enabled personally identifiable information (PII) redaction, the redacted transcript appears at the
     * location specified in <code>RedactedTranscriptFileUri</code>.
     * </p>
     * <p>
     * If you chose to redact the audio in your media file, you can find your redacted media file at the location
     * specified in <code>RedactedMediaFileUri</code>.
     * </p>
     * <p>
     * To get a list of your Call Analytics jobs, use the operation.
     * </p>
     *
     * @param getCallAnalyticsJobRequest
     * @return A Java Future containing the result of the GetCallAnalyticsJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>NotFoundException We can't find the requested resource. Check that the specified name is correct and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.GetCallAnalyticsJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetCallAnalyticsJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCallAnalyticsJobResponse> getCallAnalyticsJob(
            GetCallAnalyticsJobRequest getCallAnalyticsJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCallAnalyticsJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCallAnalyticsJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCallAnalyticsJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCallAnalyticsJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCallAnalyticsJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCallAnalyticsJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCallAnalyticsJobRequest, GetCallAnalyticsJobResponse>()
                            .withOperationName("GetCallAnalyticsJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetCallAnalyticsJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getCallAnalyticsJobRequest));
            CompletableFuture<GetCallAnalyticsJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about the specified Medical Scribe job.
     * </p>
     * <p>
     * To view the status of the specified medical transcription job, check the <code>MedicalScribeJobStatus</code>
     * field. If the status is <code>COMPLETED</code>, the job is finished. You can find the results at the location
     * specified in <code>MedicalScribeOutput</code>. If the status is <code>FAILED</code>, <code>FailureReason</code>
     * provides details on why your Medical Scribe job failed.
     * </p>
     * <p>
     * To get a list of your Medical Scribe jobs, use the operation.
     * </p>
     *
     * @param getMedicalScribeJobRequest
     * @return A Java Future containing the result of the GetMedicalScribeJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>NotFoundException We can't find the requested resource. Check that the specified name is correct and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.GetMedicalScribeJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetMedicalScribeJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMedicalScribeJobResponse> getMedicalScribeJob(
            GetMedicalScribeJobRequest getMedicalScribeJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMedicalScribeJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMedicalScribeJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMedicalScribeJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMedicalScribeJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMedicalScribeJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMedicalScribeJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMedicalScribeJobRequest, GetMedicalScribeJobResponse>()
                            .withOperationName("GetMedicalScribeJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetMedicalScribeJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getMedicalScribeJobRequest));
            CompletableFuture<GetMedicalScribeJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about the specified medical transcription job.
     * </p>
     * <p>
     * To view the status of the specified medical transcription job, check the <code>TranscriptionJobStatus</code>
     * field. If the status is <code>COMPLETED</code>, the job is finished. You can find the results at the location
     * specified in <code>TranscriptFileUri</code>. If the status is <code>FAILED</code>, <code>FailureReason</code>
     * provides details on why your transcription job failed.
     * </p>
     * <p>
     * To get a list of your medical transcription jobs, use the operation.
     * </p>
     *
     * @param getMedicalTranscriptionJobRequest
     * @return A Java Future containing the result of the GetMedicalTranscriptionJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>NotFoundException We can't find the requested resource. Check that the specified name is correct and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.GetMedicalTranscriptionJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetMedicalTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMedicalTranscriptionJobResponse> getMedicalTranscriptionJob(
            GetMedicalTranscriptionJobRequest getMedicalTranscriptionJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMedicalTranscriptionJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMedicalTranscriptionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMedicalTranscriptionJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMedicalTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMedicalTranscriptionJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMedicalTranscriptionJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMedicalTranscriptionJobRequest, GetMedicalTranscriptionJobResponse>()
                            .withOperationName("GetMedicalTranscriptionJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetMedicalTranscriptionJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getMedicalTranscriptionJobRequest));
            CompletableFuture<GetMedicalTranscriptionJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about the specified custom medical vocabulary.
     * </p>
     * <p>
     * To view the status of the specified custom medical vocabulary, check the <code>VocabularyState</code> field. If
     * the status is <code>READY</code>, your custom vocabulary is available to use. If the status is
     * <code>FAILED</code>, <code>FailureReason</code> provides details on why your vocabulary failed.
     * </p>
     * <p>
     * To get a list of your custom medical vocabularies, use the operation.
     * </p>
     *
     * @param getMedicalVocabularyRequest
     * @return A Java Future containing the result of the GetMedicalVocabulary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException We can't find the requested resource. Check that the specified name is correct and
     *         try your request again.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.GetMedicalVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetMedicalVocabulary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMedicalVocabularyResponse> getMedicalVocabulary(
            GetMedicalVocabularyRequest getMedicalVocabularyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMedicalVocabularyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMedicalVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMedicalVocabulary");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMedicalVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMedicalVocabularyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMedicalVocabularyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMedicalVocabularyRequest, GetMedicalVocabularyResponse>()
                            .withOperationName("GetMedicalVocabulary").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetMedicalVocabularyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getMedicalVocabularyRequest));
            CompletableFuture<GetMedicalVocabularyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about the specified transcription job.
     * </p>
     * <p>
     * To view the status of the specified transcription job, check the <code>TranscriptionJobStatus</code> field. If
     * the status is <code>COMPLETED</code>, the job is finished. You can find the results at the location specified in
     * <code>TranscriptFileUri</code>. If the status is <code>FAILED</code>, <code>FailureReason</code> provides details
     * on why your transcription job failed.
     * </p>
     * <p>
     * If you enabled content redaction, the redacted transcript can be found at the location specified in
     * <code>RedactedTranscriptFileUri</code>.
     * </p>
     * <p>
     * To get a list of your transcription jobs, use the operation.
     * </p>
     *
     * @param getTranscriptionJobRequest
     * @return A Java Future containing the result of the GetTranscriptionJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>NotFoundException We can't find the requested resource. Check that the specified name is correct and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.GetTranscriptionJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTranscriptionJobResponse> getTranscriptionJob(
            GetTranscriptionJobRequest getTranscriptionJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTranscriptionJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTranscriptionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTranscriptionJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTranscriptionJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTranscriptionJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTranscriptionJobRequest, GetTranscriptionJobResponse>()
                            .withOperationName("GetTranscriptionJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTranscriptionJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTranscriptionJobRequest));
            CompletableFuture<GetTranscriptionJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about the specified custom vocabulary.
     * </p>
     * <p>
     * To view the status of the specified custom vocabulary, check the <code>VocabularyState</code> field. If the
     * status is <code>READY</code>, your custom vocabulary is available to use. If the status is <code>FAILED</code>,
     * <code>FailureReason</code> provides details on why your custom vocabulary failed.
     * </p>
     * <p>
     * To get a list of your custom vocabularies, use the operation.
     * </p>
     *
     * @param getVocabularyRequest
     * @return A Java Future containing the result of the GetVocabulary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException We can't find the requested resource. Check that the specified name is correct and
     *         try your request again.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.GetVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetVocabulary" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetVocabularyResponse> getVocabulary(GetVocabularyRequest getVocabularyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getVocabularyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetVocabulary");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetVocabularyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetVocabularyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetVocabularyRequest, GetVocabularyResponse>()
                            .withOperationName("GetVocabulary").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetVocabularyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getVocabularyRequest));
            CompletableFuture<GetVocabularyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about the specified custom vocabulary filter.
     * </p>
     * <p>
     * To get a list of your custom vocabulary filters, use the operation.
     * </p>
     *
     * @param getVocabularyFilterRequest
     * @return A Java Future containing the result of the GetVocabularyFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException We can't find the requested resource. Check that the specified name is correct and
     *         try your request again.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.GetVocabularyFilter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/GetVocabularyFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetVocabularyFilterResponse> getVocabularyFilter(
            GetVocabularyFilterRequest getVocabularyFilterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getVocabularyFilterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getVocabularyFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetVocabularyFilter");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetVocabularyFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetVocabularyFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetVocabularyFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetVocabularyFilterRequest, GetVocabularyFilterResponse>()
                            .withOperationName("GetVocabularyFilter").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetVocabularyFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getVocabularyFilterRequest));
            CompletableFuture<GetVocabularyFilterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a list of Call Analytics categories, including all rules that make up each category.
     * </p>
     * <p>
     * To get detailed information about a specific Call Analytics category, use the operation.
     * </p>
     *
     * @param listCallAnalyticsCategoriesRequest
     * @return A Java Future containing the result of the ListCallAnalyticsCategories operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListCallAnalyticsCategories
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListCallAnalyticsCategories"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListCallAnalyticsCategoriesResponse> listCallAnalyticsCategories(
            ListCallAnalyticsCategoriesRequest listCallAnalyticsCategoriesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCallAnalyticsCategoriesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCallAnalyticsCategoriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCallAnalyticsCategories");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCallAnalyticsCategoriesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCallAnalyticsCategoriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListCallAnalyticsCategoriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCallAnalyticsCategoriesRequest, ListCallAnalyticsCategoriesResponse>()
                            .withOperationName("ListCallAnalyticsCategories").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListCallAnalyticsCategoriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listCallAnalyticsCategoriesRequest));
            CompletableFuture<ListCallAnalyticsCategoriesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a list of Call Analytics jobs that match the specified criteria. If no criteria are specified, all Call
     * Analytics jobs are returned.
     * </p>
     * <p>
     * To get detailed information about a specific Call Analytics job, use the operation.
     * </p>
     *
     * @param listCallAnalyticsJobsRequest
     * @return A Java Future containing the result of the ListCallAnalyticsJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListCallAnalyticsJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListCallAnalyticsJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListCallAnalyticsJobsResponse> listCallAnalyticsJobs(
            ListCallAnalyticsJobsRequest listCallAnalyticsJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCallAnalyticsJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCallAnalyticsJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCallAnalyticsJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCallAnalyticsJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCallAnalyticsJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListCallAnalyticsJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCallAnalyticsJobsRequest, ListCallAnalyticsJobsResponse>()
                            .withOperationName("ListCallAnalyticsJobs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListCallAnalyticsJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listCallAnalyticsJobsRequest));
            CompletableFuture<ListCallAnalyticsJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a list of custom language models that match the specified criteria. If no criteria are specified, all
     * custom language models are returned.
     * </p>
     * <p>
     * To get detailed information about a specific custom language model, use the operation.
     * </p>
     *
     * @param listLanguageModelsRequest
     * @return A Java Future containing the result of the ListLanguageModels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListLanguageModels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListLanguageModels" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLanguageModelsResponse> listLanguageModels(ListLanguageModelsRequest listLanguageModelsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listLanguageModelsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLanguageModelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLanguageModels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLanguageModelsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLanguageModelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLanguageModelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLanguageModelsRequest, ListLanguageModelsResponse>()
                            .withOperationName("ListLanguageModels").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListLanguageModelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listLanguageModelsRequest));
            CompletableFuture<ListLanguageModelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a list of Medical Scribe jobs that match the specified criteria. If no criteria are specified, all
     * Medical Scribe jobs are returned.
     * </p>
     * <p>
     * To get detailed information about a specific Medical Scribe job, use the operation.
     * </p>
     *
     * @param listMedicalScribeJobsRequest
     * @return A Java Future containing the result of the ListMedicalScribeJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListMedicalScribeJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListMedicalScribeJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMedicalScribeJobsResponse> listMedicalScribeJobs(
            ListMedicalScribeJobsRequest listMedicalScribeJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMedicalScribeJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMedicalScribeJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMedicalScribeJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMedicalScribeJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMedicalScribeJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMedicalScribeJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMedicalScribeJobsRequest, ListMedicalScribeJobsResponse>()
                            .withOperationName("ListMedicalScribeJobs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListMedicalScribeJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listMedicalScribeJobsRequest));
            CompletableFuture<ListMedicalScribeJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a list of medical transcription jobs that match the specified criteria. If no criteria are specified,
     * all medical transcription jobs are returned.
     * </p>
     * <p>
     * To get detailed information about a specific medical transcription job, use the operation.
     * </p>
     *
     * @param listMedicalTranscriptionJobsRequest
     * @return A Java Future containing the result of the ListMedicalTranscriptionJobs operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListMedicalTranscriptionJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListMedicalTranscriptionJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMedicalTranscriptionJobsResponse> listMedicalTranscriptionJobs(
            ListMedicalTranscriptionJobsRequest listMedicalTranscriptionJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMedicalTranscriptionJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMedicalTranscriptionJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMedicalTranscriptionJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMedicalTranscriptionJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMedicalTranscriptionJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMedicalTranscriptionJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMedicalTranscriptionJobsRequest, ListMedicalTranscriptionJobsResponse>()
                            .withOperationName("ListMedicalTranscriptionJobs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListMedicalTranscriptionJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listMedicalTranscriptionJobsRequest));
            CompletableFuture<ListMedicalTranscriptionJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a list of custom medical vocabularies that match the specified criteria. If no criteria are specified,
     * all custom medical vocabularies are returned.
     * </p>
     * <p>
     * To get detailed information about a specific custom medical vocabulary, use the operation.
     * </p>
     *
     * @param listMedicalVocabulariesRequest
     * @return A Java Future containing the result of the ListMedicalVocabularies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListMedicalVocabularies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListMedicalVocabularies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMedicalVocabulariesResponse> listMedicalVocabularies(
            ListMedicalVocabulariesRequest listMedicalVocabulariesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMedicalVocabulariesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMedicalVocabulariesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMedicalVocabularies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMedicalVocabulariesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMedicalVocabulariesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMedicalVocabulariesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMedicalVocabulariesRequest, ListMedicalVocabulariesResponse>()
                            .withOperationName("ListMedicalVocabularies").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListMedicalVocabulariesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listMedicalVocabulariesRequest));
            CompletableFuture<ListMedicalVocabulariesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all tags associated with the specified transcription job, vocabulary, model, or resource.
     * </p>
     * <p>
     * To learn more about using tags with Amazon Transcribe, refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/tagging.html">Tagging resources</a>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>NotFoundException We can't find the requested resource. Check that the specified name is correct and
     *         try your request again.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a list of transcription jobs that match the specified criteria. If no criteria are specified, all
     * transcription jobs are returned.
     * </p>
     * <p>
     * To get detailed information about a specific transcription job, use the operation.
     * </p>
     *
     * @param listTranscriptionJobsRequest
     * @return A Java Future containing the result of the ListTranscriptionJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListTranscriptionJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListTranscriptionJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTranscriptionJobsResponse> listTranscriptionJobs(
            ListTranscriptionJobsRequest listTranscriptionJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTranscriptionJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTranscriptionJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTranscriptionJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTranscriptionJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTranscriptionJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTranscriptionJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTranscriptionJobsRequest, ListTranscriptionJobsResponse>()
                            .withOperationName("ListTranscriptionJobs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTranscriptionJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTranscriptionJobsRequest));
            CompletableFuture<ListTranscriptionJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a list of custom vocabularies that match the specified criteria. If no criteria are specified, all
     * custom vocabularies are returned.
     * </p>
     * <p>
     * To get detailed information about a specific custom vocabulary, use the operation.
     * </p>
     *
     * @param listVocabulariesRequest
     * @return A Java Future containing the result of the ListVocabularies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListVocabularies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListVocabularies" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListVocabulariesResponse> listVocabularies(ListVocabulariesRequest listVocabulariesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listVocabulariesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVocabulariesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVocabularies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVocabulariesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListVocabulariesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListVocabulariesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVocabulariesRequest, ListVocabulariesResponse>()
                            .withOperationName("ListVocabularies").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListVocabulariesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listVocabulariesRequest));
            CompletableFuture<ListVocabulariesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a list of custom vocabulary filters that match the specified criteria. If no criteria are specified, all
     * custom vocabularies are returned.
     * </p>
     * <p>
     * To get detailed information about a specific custom vocabulary filter, use the operation.
     * </p>
     *
     * @param listVocabularyFiltersRequest
     * @return A Java Future containing the result of the ListVocabularyFilters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.ListVocabularyFilters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/ListVocabularyFilters"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListVocabularyFiltersResponse> listVocabularyFilters(
            ListVocabularyFiltersRequest listVocabularyFiltersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listVocabularyFiltersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVocabularyFiltersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVocabularyFilters");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVocabularyFiltersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListVocabularyFiltersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListVocabularyFiltersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVocabularyFiltersRequest, ListVocabularyFiltersResponse>()
                            .withOperationName("ListVocabularyFilters").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListVocabularyFiltersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listVocabularyFiltersRequest));
            CompletableFuture<ListVocabularyFiltersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Transcribes the audio from a customer service call and applies any additional Request Parameters you choose to
     * include in your request.
     * </p>
     * <p>
     * In addition to many standard transcription features, Call Analytics provides you with call characteristics, call
     * summarization, speaker sentiment, and optional redaction of your text transcript and your audio file. You can
     * also apply custom categories to flag specified conditions. To learn more about these features and insights, refer
     * to <a href="https://docs.aws.amazon.com/transcribe/latest/dg/call-analytics.html">Analyzing call center audio
     * with Call Analytics</a>.
     * </p>
     * <p>
     * If you want to apply categories to your Call Analytics job, you must create them before submitting your job
     * request. Categories cannot be retroactively applied to a job. To create a new category, use the operation. To
     * learn more about Call Analytics categories, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/tca-categories-batch.html">Creating categories for
     * post-call transcriptions</a> and <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/tca-categories-stream.html">Creating categories for
     * real-time transcriptions</a>.
     * </p>
     * <p>
     * To make a <code>StartCallAnalyticsJob</code> request, you must first upload your media file into an Amazon S3
     * bucket; you can then specify the Amazon S3 location of the file using the <code>Media</code> parameter.
     * </p>
     * <p>
     * Note that job queuing is enabled by default for Call Analytics jobs.
     * </p>
     * <p>
     * You must include the following parameters in your <code>StartCallAnalyticsJob</code> request:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>region</code>: The Amazon Web Services Region where you are making your request. For a list of Amazon Web
     * Services Regions supported with Amazon Transcribe, refer to <a
     * href="https://docs.aws.amazon.com/general/latest/gr/transcribe.html">Amazon Transcribe endpoints and quotas</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CallAnalyticsJobName</code>: A custom name that you create for your transcription job that's unique within
     * your Amazon Web Services account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DataAccessRoleArn</code>: The Amazon Resource Name (ARN) of an IAM role that has permissions to access the
     * Amazon S3 bucket that contains your input files.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Media</code> (<code>MediaFileUri</code> or <code>RedactedMediaFileUri</code>): The Amazon S3 location of
     * your media file.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * With Call Analytics, you can redact the audio contained in your media file by including
     * <code>RedactedMediaFileUri</code>, instead of <code>MediaFileUri</code>, to specify the location of your input
     * audio. If you choose to redact your audio, you can find your redacted media at the location specified in the
     * <code>RedactedMediaFileUri</code> field of your response.
     * </p>
     * </note>
     *
     * @param startCallAnalyticsJobRequest
     * @return A Java Future containing the result of the StartCallAnalyticsJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>ConflictException A resource already exists with this name. Resource names must be unique within an
     *         Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.StartCallAnalyticsJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/StartCallAnalyticsJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartCallAnalyticsJobResponse> startCallAnalyticsJob(
            StartCallAnalyticsJobRequest startCallAnalyticsJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startCallAnalyticsJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startCallAnalyticsJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartCallAnalyticsJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartCallAnalyticsJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartCallAnalyticsJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartCallAnalyticsJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartCallAnalyticsJobRequest, StartCallAnalyticsJobResponse>()
                            .withOperationName("StartCallAnalyticsJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartCallAnalyticsJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startCallAnalyticsJobRequest));
            CompletableFuture<StartCallAnalyticsJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Transcribes patient-clinician conversations and generates clinical notes.
     * </p>
     * <p>
     * Amazon Web Services HealthScribe automatically provides rich conversation transcripts, identifies speaker roles,
     * classifies dialogues, extracts medical terms, and generates preliminary clinical notes. To learn more about these
     * features, refer to <a href="https://docs.aws.amazon.com/transcribe/latest/dg/health-scribe.html">Amazon Web
     * Services HealthScribe</a>.
     * </p>
     * <p>
     * To make a <code>StartMedicalScribeJob</code> request, you must first upload your media file into an Amazon S3
     * bucket; you can then specify the Amazon S3 location of the file using the <code>Media</code> parameter.
     * </p>
     * <p>
     * You must include the following parameters in your <code>StartMedicalTranscriptionJob</code> request:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DataAccessRoleArn</code>: The ARN of an IAM role with the these minimum permissions: read permission on
     * input file Amazon S3 bucket specified in <code>Media</code>, write permission on the Amazon S3 bucket specified
     * in <code>OutputBucketName</code>, and full permissions on the KMS key specified in
     * <code>OutputEncryptionKMSKeyId</code> (if set). The role should also allow <code>transcribe.amazonaws.com</code>
     * to assume it.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Media</code> (<code>MediaFileUri</code>): The Amazon S3 location of your media file.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MedicalScribeJobName</code>: A custom name you create for your MedicalScribe job that is unique within your
     * Amazon Web Services account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OutputBucketName</code>: The Amazon S3 bucket where you want your output files stored.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Settings</code>: A <code>MedicalScribeSettings</code> obect that must set exactly one of
     * <code>ShowSpeakerLabels</code> or <code>ChannelIdentification</code> to true. If <code>ShowSpeakerLabels</code>
     * is true, <code>MaxSpeakerLabels</code> must also be set.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ChannelDefinitions</code>: A <code>MedicalScribeChannelDefinitions</code> array should be set if and only
     * if the <code>ChannelIdentification</code> value of <code>Settings</code> is set to true.
     * </p>
     * </li>
     * </ul>
     *
     * @param startMedicalScribeJobRequest
     * @return A Java Future containing the result of the StartMedicalScribeJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>ConflictException A resource already exists with this name. Resource names must be unique within an
     *         Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.StartMedicalScribeJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/StartMedicalScribeJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartMedicalScribeJobResponse> startMedicalScribeJob(
            StartMedicalScribeJobRequest startMedicalScribeJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startMedicalScribeJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startMedicalScribeJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMedicalScribeJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartMedicalScribeJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartMedicalScribeJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartMedicalScribeJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartMedicalScribeJobRequest, StartMedicalScribeJobResponse>()
                            .withOperationName("StartMedicalScribeJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartMedicalScribeJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startMedicalScribeJobRequest));
            CompletableFuture<StartMedicalScribeJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Transcribes the audio from a medical dictation or conversation and applies any additional Request Parameters you
     * choose to include in your request.
     * </p>
     * <p>
     * In addition to many standard transcription features, Amazon Transcribe Medical provides you with a robust medical
     * vocabulary and, optionally, content identification, which adds flags to personal health information (PHI). To
     * learn more about these features, refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/how-it-works-med.html">How Amazon Transcribe Medical
     * works</a>.
     * </p>
     * <p>
     * To make a <code>StartMedicalTranscriptionJob</code> request, you must first upload your media file into an Amazon
     * S3 bucket; you can then specify the Amazon S3 location of the file using the <code>Media</code> parameter.
     * </p>
     * <p>
     * You must include the following parameters in your <code>StartMedicalTranscriptionJob</code> request:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>region</code>: The Amazon Web Services Region where you are making your request. For a list of Amazon Web
     * Services Regions supported with Amazon Transcribe, refer to <a
     * href="https://docs.aws.amazon.com/general/latest/gr/transcribe.html">Amazon Transcribe endpoints and quotas</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MedicalTranscriptionJobName</code>: A custom name you create for your transcription job that is unique
     * within your Amazon Web Services account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Media</code> (<code>MediaFileUri</code>): The Amazon S3 location of your media file.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LanguageCode</code>: This must be <code>en-US</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OutputBucketName</code>: The Amazon S3 bucket where you want your transcript stored. If you want your
     * output stored in a sub-folder of this bucket, you must also include <code>OutputKey</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Specialty</code>: This must be <code>PRIMARYCARE</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Type</code>: Choose whether your audio is a conversation or a dictation.
     * </p>
     * </li>
     * </ul>
     *
     * @param startMedicalTranscriptionJobRequest
     * @return A Java Future containing the result of the StartMedicalTranscriptionJob operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>ConflictException A resource already exists with this name. Resource names must be unique within an
     *         Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.StartMedicalTranscriptionJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/StartMedicalTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartMedicalTranscriptionJobResponse> startMedicalTranscriptionJob(
            StartMedicalTranscriptionJobRequest startMedicalTranscriptionJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startMedicalTranscriptionJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startMedicalTranscriptionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMedicalTranscriptionJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartMedicalTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartMedicalTranscriptionJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartMedicalTranscriptionJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartMedicalTranscriptionJobRequest, StartMedicalTranscriptionJobResponse>()
                            .withOperationName("StartMedicalTranscriptionJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartMedicalTranscriptionJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startMedicalTranscriptionJobRequest));
            CompletableFuture<StartMedicalTranscriptionJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Transcribes the audio from a media file and applies any additional Request Parameters you choose to include in
     * your request.
     * </p>
     * <p>
     * To make a <code>StartTranscriptionJob</code> request, you must first upload your media file into an Amazon S3
     * bucket; you can then specify the Amazon S3 location of the file using the <code>Media</code> parameter.
     * </p>
     * <p>
     * You must include the following parameters in your <code>StartTranscriptionJob</code> request:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>region</code>: The Amazon Web Services Region where you are making your request. For a list of Amazon Web
     * Services Regions supported with Amazon Transcribe, refer to <a
     * href="https://docs.aws.amazon.com/general/latest/gr/transcribe.html">Amazon Transcribe endpoints and quotas</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TranscriptionJobName</code>: A custom name you create for your transcription job that is unique within your
     * Amazon Web Services account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Media</code> (<code>MediaFileUri</code>): The Amazon S3 location of your media file.
     * </p>
     * </li>
     * <li>
     * <p>
     * One of <code>LanguageCode</code>, <code>IdentifyLanguage</code>, or <code>IdentifyMultipleLanguages</code>: If
     * you know the language of your media file, specify it using the <code>LanguageCode</code> parameter; you can find
     * all valid language codes in the <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages</a> table.
     * If you do not know the languages spoken in your media, use either <code>IdentifyLanguage</code> or
     * <code>IdentifyMultipleLanguages</code> and let Amazon Transcribe identify the languages for you.
     * </p>
     * </li>
     * </ul>
     *
     * @param startTranscriptionJobRequest
     * @return A Java Future containing the result of the StartTranscriptionJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>ConflictException A resource already exists with this name. Resource names must be unique within an
     *         Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.StartTranscriptionJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/StartTranscriptionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartTranscriptionJobResponse> startTranscriptionJob(
            StartTranscriptionJobRequest startTranscriptionJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startTranscriptionJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startTranscriptionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartTranscriptionJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartTranscriptionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartTranscriptionJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartTranscriptionJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartTranscriptionJobRequest, StartTranscriptionJobResponse>()
                            .withOperationName("StartTranscriptionJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartTranscriptionJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startTranscriptionJobRequest));
            CompletableFuture<StartTranscriptionJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more custom tags, each in the form of a key:value pair, to the specified resource.
     * </p>
     * <p>
     * To learn more about using tags with Amazon Transcribe, refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/tagging.html">Tagging resources</a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>ConflictException A resource already exists with this name. Resource names must be unique within an
     *         Amazon Web Services account.</li>
     *         <li>NotFoundException We can't find the requested resource. Check that the specified name is correct and
     *         try your request again.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified tags from the specified Amazon Transcribe resource.
     * </p>
     * <p>
     * If you include <code>UntagResource</code> in your request, you must also include <code>ResourceArn</code> and
     * <code>TagKeys</code>.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>ConflictException A resource already exists with this name. Resource names must be unique within an
     *         Amazon Web Services account.</li>
     *         <li>NotFoundException We can't find the requested resource. Check that the specified name is correct and
     *         try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified Call Analytics category with new rules. Note that the
     * <code>UpdateCallAnalyticsCategory</code> operation overwrites all existing rules contained in the specified
     * category. You cannot append additional rules onto an existing category.
     * </p>
     * <p>
     * To create a new category, see .
     * </p>
     *
     * @param updateCallAnalyticsCategoryRequest
     * @return A Java Future containing the result of the UpdateCallAnalyticsCategory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>NotFoundException We can't find the requested resource. Check that the specified name is correct and
     *         try your request again.</li>
     *         <li>ConflictException A resource already exists with this name. Resource names must be unique within an
     *         Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.UpdateCallAnalyticsCategory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/UpdateCallAnalyticsCategory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateCallAnalyticsCategoryResponse> updateCallAnalyticsCategory(
            UpdateCallAnalyticsCategoryRequest updateCallAnalyticsCategoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateCallAnalyticsCategoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateCallAnalyticsCategoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCallAnalyticsCategory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateCallAnalyticsCategoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateCallAnalyticsCategoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateCallAnalyticsCategoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateCallAnalyticsCategoryRequest, UpdateCallAnalyticsCategoryResponse>()
                            .withOperationName("UpdateCallAnalyticsCategory").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateCallAnalyticsCategoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateCallAnalyticsCategoryRequest));
            CompletableFuture<UpdateCallAnalyticsCategoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing custom medical vocabulary with new values. This operation overwrites all existing information
     * with your new values; you cannot append new terms onto an existing custom vocabulary.
     * </p>
     *
     * @param updateMedicalVocabularyRequest
     * @return A Java Future containing the result of the UpdateMedicalVocabulary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>NotFoundException We can't find the requested resource. Check that the specified name is correct and
     *         try your request again.</li>
     *         <li>ConflictException A resource already exists with this name. Resource names must be unique within an
     *         Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.UpdateMedicalVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/UpdateMedicalVocabulary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateMedicalVocabularyResponse> updateMedicalVocabulary(
            UpdateMedicalVocabularyRequest updateMedicalVocabularyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateMedicalVocabularyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMedicalVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMedicalVocabulary");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateMedicalVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateMedicalVocabularyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateMedicalVocabularyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateMedicalVocabularyRequest, UpdateMedicalVocabularyResponse>()
                            .withOperationName("UpdateMedicalVocabulary").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateMedicalVocabularyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateMedicalVocabularyRequest));
            CompletableFuture<UpdateMedicalVocabularyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing custom vocabulary with new values. This operation overwrites all existing information with
     * your new values; you cannot append new terms onto an existing custom vocabulary.
     * </p>
     *
     * @param updateVocabularyRequest
     * @return A Java Future containing the result of the UpdateVocabulary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>NotFoundException We can't find the requested resource. Check that the specified name is correct and
     *         try your request again.</li>
     *         <li>ConflictException A resource already exists with this name. Resource names must be unique within an
     *         Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.UpdateVocabulary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/UpdateVocabulary" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateVocabularyResponse> updateVocabulary(UpdateVocabularyRequest updateVocabularyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateVocabularyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVocabularyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVocabulary");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateVocabularyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateVocabularyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateVocabularyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateVocabularyRequest, UpdateVocabularyResponse>()
                            .withOperationName("UpdateVocabulary").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateVocabularyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateVocabularyRequest));
            CompletableFuture<UpdateVocabularyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing custom vocabulary filter with a new list of words. The new list you provide overwrites all
     * previous entries; you cannot append new terms onto an existing custom vocabulary filter.
     * </p>
     *
     * @param updateVocabularyFilterRequest
     * @return A Java Future containing the result of the UpdateVocabularyFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Your request didn't pass one or more validation tests. This can occur when the
     *         entity you're trying to delete doesn't exist or if it's in a non-terminal state (such as
     *         <code>IN PROGRESS</code>). See the exception message field for more information.</li>
     *         <li>LimitExceededException You've either sent too many requests or your input file is too long. Wait
     *         before retrying your request, or use a smaller file and try your request again.</li>
     *         <li>InternalFailureException There was an internal error. Check the error message, correct the issue, and
     *         try your request again.</li>
     *         <li>NotFoundException We can't find the requested resource. Check that the specified name is correct and
     *         try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranscribeAsyncClient.UpdateVocabularyFilter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-2017-10-26/UpdateVocabularyFilter"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateVocabularyFilterResponse> updateVocabularyFilter(
            UpdateVocabularyFilterRequest updateVocabularyFilterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateVocabularyFilterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVocabularyFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVocabularyFilter");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateVocabularyFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateVocabularyFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateVocabularyFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateVocabularyFilterRequest, UpdateVocabularyFilterResponse>()
                            .withOperationName("UpdateVocabularyFilter").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateVocabularyFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateVocabularyFilterRequest));
            CompletableFuture<UpdateVocabularyFilterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final TranscribeServiceClientConfiguration serviceClientConfiguration() {
        return new TranscribeServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(TranscribeException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        TranscribeServiceClientConfigurationBuilder serviceConfigBuilder = new TranscribeServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
