/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribe.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides detailed information about a Medical Scribe job.
 * </p>
 * <p>
 * To view the status of the specified Medical Scribe job, check the <code>MedicalScribeJobStatus</code> field. If the
 * status is <code>COMPLETED</code>, the job is finished and you can find the results at the locations specified in
 * <code>MedicalScribeOutput</code>. If the status is <code>FAILED</code>, <code>FailureReason</code> provides details
 * on why your Medical Scribe job failed.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MedicalScribeJob implements SdkPojo, Serializable,
        ToCopyableBuilder<MedicalScribeJob.Builder, MedicalScribeJob> {
    private static final SdkField<String> MEDICAL_SCRIBE_JOB_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MedicalScribeJobName").getter(getter(MedicalScribeJob::medicalScribeJobName))
            .setter(setter(Builder::medicalScribeJobName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MedicalScribeJobName").build())
            .build();

    private static final SdkField<String> MEDICAL_SCRIBE_JOB_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MedicalScribeJobStatus").getter(getter(MedicalScribeJob::medicalScribeJobStatusAsString))
            .setter(setter(Builder::medicalScribeJobStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MedicalScribeJobStatus").build())
            .build();

    private static final SdkField<String> LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LanguageCode").getter(getter(MedicalScribeJob::languageCodeAsString))
            .setter(setter(Builder::languageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageCode").build()).build();

    private static final SdkField<Media> MEDIA_FIELD = SdkField.<Media> builder(MarshallingType.SDK_POJO).memberName("Media")
            .getter(getter(MedicalScribeJob::media)).setter(setter(Builder::media)).constructor(Media::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Media").build()).build();

    private static final SdkField<MedicalScribeOutput> MEDICAL_SCRIBE_OUTPUT_FIELD = SdkField
            .<MedicalScribeOutput> builder(MarshallingType.SDK_POJO).memberName("MedicalScribeOutput")
            .getter(getter(MedicalScribeJob::medicalScribeOutput)).setter(setter(Builder::medicalScribeOutput))
            .constructor(MedicalScribeOutput::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MedicalScribeOutput").build())
            .build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime").getter(getter(MedicalScribeJob::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(MedicalScribeJob::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<Instant> COMPLETION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CompletionTime").getter(getter(MedicalScribeJob::completionTime))
            .setter(setter(Builder::completionTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletionTime").build()).build();

    private static final SdkField<String> FAILURE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FailureReason").getter(getter(MedicalScribeJob::failureReason)).setter(setter(Builder::failureReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureReason").build()).build();

    private static final SdkField<MedicalScribeSettings> SETTINGS_FIELD = SdkField
            .<MedicalScribeSettings> builder(MarshallingType.SDK_POJO).memberName("Settings")
            .getter(getter(MedicalScribeJob::settings)).setter(setter(Builder::settings))
            .constructor(MedicalScribeSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Settings").build()).build();

    private static final SdkField<String> DATA_ACCESS_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataAccessRoleArn").getter(getter(MedicalScribeJob::dataAccessRoleArn))
            .setter(setter(Builder::dataAccessRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataAccessRoleArn").build()).build();

    private static final SdkField<List<MedicalScribeChannelDefinition>> CHANNEL_DEFINITIONS_FIELD = SdkField
            .<List<MedicalScribeChannelDefinition>> builder(MarshallingType.LIST)
            .memberName("ChannelDefinitions")
            .getter(getter(MedicalScribeJob::channelDefinitions))
            .setter(setter(Builder::channelDefinitions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChannelDefinitions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MedicalScribeChannelDefinition> builder(MarshallingType.SDK_POJO)
                                            .constructor(MedicalScribeChannelDefinition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(MedicalScribeJob::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MEDICAL_SCRIBE_JOB_NAME_FIELD,
            MEDICAL_SCRIBE_JOB_STATUS_FIELD, LANGUAGE_CODE_FIELD, MEDIA_FIELD, MEDICAL_SCRIBE_OUTPUT_FIELD, START_TIME_FIELD,
            CREATION_TIME_FIELD, COMPLETION_TIME_FIELD, FAILURE_REASON_FIELD, SETTINGS_FIELD, DATA_ACCESS_ROLE_ARN_FIELD,
            CHANNEL_DEFINITIONS_FIELD, TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("MedicalScribeJobName", MEDICAL_SCRIBE_JOB_NAME_FIELD);
                    put("MedicalScribeJobStatus", MEDICAL_SCRIBE_JOB_STATUS_FIELD);
                    put("LanguageCode", LANGUAGE_CODE_FIELD);
                    put("Media", MEDIA_FIELD);
                    put("MedicalScribeOutput", MEDICAL_SCRIBE_OUTPUT_FIELD);
                    put("StartTime", START_TIME_FIELD);
                    put("CreationTime", CREATION_TIME_FIELD);
                    put("CompletionTime", COMPLETION_TIME_FIELD);
                    put("FailureReason", FAILURE_REASON_FIELD);
                    put("Settings", SETTINGS_FIELD);
                    put("DataAccessRoleArn", DATA_ACCESS_ROLE_ARN_FIELD);
                    put("ChannelDefinitions", CHANNEL_DEFINITIONS_FIELD);
                    put("Tags", TAGS_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String medicalScribeJobName;

    private final String medicalScribeJobStatus;

    private final String languageCode;

    private final Media media;

    private final MedicalScribeOutput medicalScribeOutput;

    private final Instant startTime;

    private final Instant creationTime;

    private final Instant completionTime;

    private final String failureReason;

    private final MedicalScribeSettings settings;

    private final String dataAccessRoleArn;

    private final List<MedicalScribeChannelDefinition> channelDefinitions;

    private final List<Tag> tags;

    private MedicalScribeJob(BuilderImpl builder) {
        this.medicalScribeJobName = builder.medicalScribeJobName;
        this.medicalScribeJobStatus = builder.medicalScribeJobStatus;
        this.languageCode = builder.languageCode;
        this.media = builder.media;
        this.medicalScribeOutput = builder.medicalScribeOutput;
        this.startTime = builder.startTime;
        this.creationTime = builder.creationTime;
        this.completionTime = builder.completionTime;
        this.failureReason = builder.failureReason;
        this.settings = builder.settings;
        this.dataAccessRoleArn = builder.dataAccessRoleArn;
        this.channelDefinitions = builder.channelDefinitions;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name of the Medical Scribe job. Job names are case sensitive and must be unique within an Amazon Web Services
     * account.
     * </p>
     * 
     * @return The name of the Medical Scribe job. Job names are case sensitive and must be unique within an Amazon Web
     *         Services account.
     */
    public final String medicalScribeJobName() {
        return medicalScribeJobName;
    }

    /**
     * <p>
     * Provides the status of the specified Medical Scribe job.
     * </p>
     * <p>
     * If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
     * specified in <code>MedicalScribeOutput</code> If the status is <code>FAILED</code>, <code>FailureReason</code>
     * provides details on why your Medical Scribe job failed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #medicalScribeJobStatus} will return {@link MedicalScribeJobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #medicalScribeJobStatusAsString}.
     * </p>
     * 
     * @return Provides the status of the specified Medical Scribe job.</p>
     *         <p>
     *         If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
     *         specified in <code>MedicalScribeOutput</code> If the status is <code>FAILED</code>,
     *         <code>FailureReason</code> provides details on why your Medical Scribe job failed.
     * @see MedicalScribeJobStatus
     */
    public final MedicalScribeJobStatus medicalScribeJobStatus() {
        return MedicalScribeJobStatus.fromValue(medicalScribeJobStatus);
    }

    /**
     * <p>
     * Provides the status of the specified Medical Scribe job.
     * </p>
     * <p>
     * If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
     * specified in <code>MedicalScribeOutput</code> If the status is <code>FAILED</code>, <code>FailureReason</code>
     * provides details on why your Medical Scribe job failed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #medicalScribeJobStatus} will return {@link MedicalScribeJobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #medicalScribeJobStatusAsString}.
     * </p>
     * 
     * @return Provides the status of the specified Medical Scribe job.</p>
     *         <p>
     *         If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
     *         specified in <code>MedicalScribeOutput</code> If the status is <code>FAILED</code>,
     *         <code>FailureReason</code> provides details on why your Medical Scribe job failed.
     * @see MedicalScribeJobStatus
     */
    public final String medicalScribeJobStatusAsString() {
        return medicalScribeJobStatus;
    }

    /**
     * <p>
     * The language code used to create your Medical Scribe job. US English (<code>en-US</code>) is the only supported
     * language for Medical Scribe jobs.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link MedicalScribeLanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #languageCodeAsString}.
     * </p>
     * 
     * @return The language code used to create your Medical Scribe job. US English (<code>en-US</code>) is the only
     *         supported language for Medical Scribe jobs.
     * @see MedicalScribeLanguageCode
     */
    public final MedicalScribeLanguageCode languageCode() {
        return MedicalScribeLanguageCode.fromValue(languageCode);
    }

    /**
     * <p>
     * The language code used to create your Medical Scribe job. US English (<code>en-US</code>) is the only supported
     * language for Medical Scribe jobs.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link MedicalScribeLanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #languageCodeAsString}.
     * </p>
     * 
     * @return The language code used to create your Medical Scribe job. US English (<code>en-US</code>) is the only
     *         supported language for Medical Scribe jobs.
     * @see MedicalScribeLanguageCode
     */
    public final String languageCodeAsString() {
        return languageCode;
    }

    /**
     * Returns the value of the Media property for this object.
     * 
     * @return The value of the Media property for this object.
     */
    public final Media media() {
        return media;
    }

    /**
     * <p>
     * The location of the output of your Medical Scribe job. <code>ClinicalDocumentUri</code> holds the Amazon S3 URI
     * for the Clinical Document and <code>TranscriptFileUri</code> holds the Amazon S3 URI for the Transcript.
     * </p>
     * 
     * @return The location of the output of your Medical Scribe job. <code>ClinicalDocumentUri</code> holds the Amazon
     *         S3 URI for the Clinical Document and <code>TranscriptFileUri</code> holds the Amazon S3 URI for the
     *         Transcript.
     */
    public final MedicalScribeOutput medicalScribeOutput() {
        return medicalScribeOutput;
    }

    /**
     * <p>
     * The date and time your Medical Scribe job began processing.
     * </p>
     * <p>
     * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     * <code>2022-05-04T12:32:58.789000-07:00</code> represents a Medical Scribe job that started processing at 12:32 PM
     * UTC-7 on May 4, 2022.
     * </p>
     * 
     * @return The date and time your Medical Scribe job began processing.</p>
     *         <p>
     *         Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     *         <code>2022-05-04T12:32:58.789000-07:00</code> represents a Medical Scribe job that started processing at
     *         12:32 PM UTC-7 on May 4, 2022.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The date and time the specified Medical Scribe job request was made.
     * </p>
     * <p>
     * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     * <code>2022-05-04T12:32:58.761000-07:00</code> represents a Medical Scribe job that started processing at 12:32 PM
     * UTC-7 on May 4, 2022.
     * </p>
     * 
     * @return The date and time the specified Medical Scribe job request was made.</p>
     *         <p>
     *         Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     *         <code>2022-05-04T12:32:58.761000-07:00</code> represents a Medical Scribe job that started processing at
     *         12:32 PM UTC-7 on May 4, 2022.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The date and time the specified Medical Scribe job finished processing.
     * </p>
     * <p>
     * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     * <code>2022-05-04T12:32:58.761000-07:00</code> represents a Medical Scribe job that finished processing at 12:32
     * PM UTC-7 on May 4, 2022.
     * </p>
     * 
     * @return The date and time the specified Medical Scribe job finished processing.</p>
     *         <p>
     *         Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     *         <code>2022-05-04T12:32:58.761000-07:00</code> represents a Medical Scribe job that finished processing at
     *         12:32 PM UTC-7 on May 4, 2022.
     */
    public final Instant completionTime() {
        return completionTime;
    }

    /**
     * <p>
     * If <code>MedicalScribeJobStatus</code> is <code>FAILED</code>, <code>FailureReason</code> contains information
     * about why the transcription job failed. See also: <a
     * href="https://docs.aws.amazon.com/transcribe/latest/APIReference/CommonErrors.html">Common Errors</a>.
     * </p>
     * 
     * @return If <code>MedicalScribeJobStatus</code> is <code>FAILED</code>, <code>FailureReason</code> contains
     *         information about why the transcription job failed. See also: <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/APIReference/CommonErrors.html">Common Errors</a>.
     */
    public final String failureReason() {
        return failureReason;
    }

    /**
     * <p>
     * Makes it possible to control how your Medical Scribe job is processed using a <code>MedicalScribeSettings</code>
     * object. Specify <code>ChannelIdentification</code> if <code>ChannelDefinitions</code> are set. Enabled
     * <code>ShowSpeakerLabels</code> if <code>ChannelIdentification</code> and <code>ChannelDefinitions</code> are not
     * set. One and only one of <code>ChannelIdentification</code> and <code>ShowSpeakerLabels</code> must be set. If
     * <code>ShowSpeakerLabels</code> is set, <code>MaxSpeakerLabels</code> must also be set. Use <code>Settings</code>
     * to specify a vocabulary or vocabulary filter or both using <code>VocabularyName</code>,
     * <code>VocabularyFilterName</code>. <code>VocabularyFilterMethod</code> must be specified if
     * <code>VocabularyFilterName</code> is set.
     * </p>
     * 
     * @return Makes it possible to control how your Medical Scribe job is processed using a
     *         <code>MedicalScribeSettings</code> object. Specify <code>ChannelIdentification</code> if
     *         <code>ChannelDefinitions</code> are set. Enabled <code>ShowSpeakerLabels</code> if
     *         <code>ChannelIdentification</code> and <code>ChannelDefinitions</code> are not set. One and only one of
     *         <code>ChannelIdentification</code> and <code>ShowSpeakerLabels</code> must be set. If
     *         <code>ShowSpeakerLabels</code> is set, <code>MaxSpeakerLabels</code> must also be set. Use
     *         <code>Settings</code> to specify a vocabulary or vocabulary filter or both using
     *         <code>VocabularyName</code>, <code>VocabularyFilterName</code>. <code>VocabularyFilterMethod</code> must
     *         be specified if <code>VocabularyFilterName</code> is set.
     */
    public final MedicalScribeSettings settings() {
        return settings;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an IAM role that has permissions to access the Amazon S3 bucket that contains
     * your input files, write to the output bucket, and use your KMS key if supplied. If the role that you specify
     * doesn’t have the appropriate permissions your request fails.
     * </p>
     * <p>
     * IAM role ARNs have the format <code>arn:partition:iam::account:role/role-name-with-path</code>. For example:
     * <code>arn:aws:iam::111122223333:role/Admin</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-arns">IAM ARNs</a>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of an IAM role that has permissions to access the Amazon S3 bucket that
     *         contains your input files, write to the output bucket, and use your KMS key if supplied. If the role that
     *         you specify doesn’t have the appropriate permissions your request fails.</p>
     *         <p>
     *         IAM role ARNs have the format <code>arn:partition:iam::account:role/role-name-with-path</code>. For
     *         example: <code>arn:aws:iam::111122223333:role/Admin</code>.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-arns">IAM
     *         ARNs</a>.
     */
    public final String dataAccessRoleArn() {
        return dataAccessRoleArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the ChannelDefinitions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasChannelDefinitions() {
        return channelDefinitions != null && !(channelDefinitions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Makes it possible to specify which speaker is on which channel. For example, if the clinician is the first
     * participant to speak, you would set <code>ChannelId</code> of the first <code>ChannelDefinition</code> in the
     * list to <code>0</code> (to indicate the first channel) and <code>ParticipantRole</code> to <code>CLINICIAN</code>
     * (to indicate that it's the clinician speaking). Then you would set the <code>ChannelId</code> of the second
     * <code>ChannelDefinition</code> in the list to <code>1</code> (to indicate the second channel) and
     * <code>ParticipantRole</code> to <code>PATIENT</code> (to indicate that it's the patient speaking).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasChannelDefinitions} method.
     * </p>
     * 
     * @return Makes it possible to specify which speaker is on which channel. For example, if the clinician is the
     *         first participant to speak, you would set <code>ChannelId</code> of the first
     *         <code>ChannelDefinition</code> in the list to <code>0</code> (to indicate the first channel) and
     *         <code>ParticipantRole</code> to <code>CLINICIAN</code> (to indicate that it's the clinician speaking).
     *         Then you would set the <code>ChannelId</code> of the second <code>ChannelDefinition</code> in the list to
     *         <code>1</code> (to indicate the second channel) and <code>ParticipantRole</code> to <code>PATIENT</code>
     *         (to indicate that it's the patient speaking).
     */
    public final List<MedicalScribeChannelDefinition> channelDefinitions() {
        return channelDefinitions;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Adds one or more custom tags, each in the form of a key:value pair, to the Medica Scribe job.
     * </p>
     * <p>
     * To learn more about using tags with Amazon Transcribe, refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/tagging.html">Tagging resources</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Adds one or more custom tags, each in the form of a key:value pair, to the Medica Scribe job.</p>
     *         <p>
     *         To learn more about using tags with Amazon Transcribe, refer to <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/dg/tagging.html">Tagging resources</a>.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(medicalScribeJobName());
        hashCode = 31 * hashCode + Objects.hashCode(medicalScribeJobStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(languageCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(media());
        hashCode = 31 * hashCode + Objects.hashCode(medicalScribeOutput());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(completionTime());
        hashCode = 31 * hashCode + Objects.hashCode(failureReason());
        hashCode = 31 * hashCode + Objects.hashCode(settings());
        hashCode = 31 * hashCode + Objects.hashCode(dataAccessRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasChannelDefinitions() ? channelDefinitions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MedicalScribeJob)) {
            return false;
        }
        MedicalScribeJob other = (MedicalScribeJob) obj;
        return Objects.equals(medicalScribeJobName(), other.medicalScribeJobName())
                && Objects.equals(medicalScribeJobStatusAsString(), other.medicalScribeJobStatusAsString())
                && Objects.equals(languageCodeAsString(), other.languageCodeAsString()) && Objects.equals(media(), other.media())
                && Objects.equals(medicalScribeOutput(), other.medicalScribeOutput())
                && Objects.equals(startTime(), other.startTime()) && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(completionTime(), other.completionTime())
                && Objects.equals(failureReason(), other.failureReason()) && Objects.equals(settings(), other.settings())
                && Objects.equals(dataAccessRoleArn(), other.dataAccessRoleArn())
                && hasChannelDefinitions() == other.hasChannelDefinitions()
                && Objects.equals(channelDefinitions(), other.channelDefinitions()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MedicalScribeJob").add("MedicalScribeJobName", medicalScribeJobName())
                .add("MedicalScribeJobStatus", medicalScribeJobStatusAsString()).add("LanguageCode", languageCodeAsString())
                .add("Media", media()).add("MedicalScribeOutput", medicalScribeOutput()).add("StartTime", startTime())
                .add("CreationTime", creationTime()).add("CompletionTime", completionTime())
                .add("FailureReason", failureReason()).add("Settings", settings()).add("DataAccessRoleArn", dataAccessRoleArn())
                .add("ChannelDefinitions", hasChannelDefinitions() ? channelDefinitions() : null)
                .add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MedicalScribeJobName":
            return Optional.ofNullable(clazz.cast(medicalScribeJobName()));
        case "MedicalScribeJobStatus":
            return Optional.ofNullable(clazz.cast(medicalScribeJobStatusAsString()));
        case "LanguageCode":
            return Optional.ofNullable(clazz.cast(languageCodeAsString()));
        case "Media":
            return Optional.ofNullable(clazz.cast(media()));
        case "MedicalScribeOutput":
            return Optional.ofNullable(clazz.cast(medicalScribeOutput()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "CompletionTime":
            return Optional.ofNullable(clazz.cast(completionTime()));
        case "FailureReason":
            return Optional.ofNullable(clazz.cast(failureReason()));
        case "Settings":
            return Optional.ofNullable(clazz.cast(settings()));
        case "DataAccessRoleArn":
            return Optional.ofNullable(clazz.cast(dataAccessRoleArn()));
        case "ChannelDefinitions":
            return Optional.ofNullable(clazz.cast(channelDefinitions()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<MedicalScribeJob, T> g) {
        return obj -> g.apply((MedicalScribeJob) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MedicalScribeJob> {
        /**
         * <p>
         * The name of the Medical Scribe job. Job names are case sensitive and must be unique within an Amazon Web
         * Services account.
         * </p>
         * 
         * @param medicalScribeJobName
         *        The name of the Medical Scribe job. Job names are case sensitive and must be unique within an Amazon
         *        Web Services account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder medicalScribeJobName(String medicalScribeJobName);

        /**
         * <p>
         * Provides the status of the specified Medical Scribe job.
         * </p>
         * <p>
         * If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
         * specified in <code>MedicalScribeOutput</code> If the status is <code>FAILED</code>,
         * <code>FailureReason</code> provides details on why your Medical Scribe job failed.
         * </p>
         * 
         * @param medicalScribeJobStatus
         *        Provides the status of the specified Medical Scribe job.</p>
         *        <p>
         *        If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the
         *        location specified in <code>MedicalScribeOutput</code> If the status is <code>FAILED</code>,
         *        <code>FailureReason</code> provides details on why your Medical Scribe job failed.
         * @see MedicalScribeJobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MedicalScribeJobStatus
         */
        Builder medicalScribeJobStatus(String medicalScribeJobStatus);

        /**
         * <p>
         * Provides the status of the specified Medical Scribe job.
         * </p>
         * <p>
         * If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
         * specified in <code>MedicalScribeOutput</code> If the status is <code>FAILED</code>,
         * <code>FailureReason</code> provides details on why your Medical Scribe job failed.
         * </p>
         * 
         * @param medicalScribeJobStatus
         *        Provides the status of the specified Medical Scribe job.</p>
         *        <p>
         *        If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the
         *        location specified in <code>MedicalScribeOutput</code> If the status is <code>FAILED</code>,
         *        <code>FailureReason</code> provides details on why your Medical Scribe job failed.
         * @see MedicalScribeJobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MedicalScribeJobStatus
         */
        Builder medicalScribeJobStatus(MedicalScribeJobStatus medicalScribeJobStatus);

        /**
         * <p>
         * The language code used to create your Medical Scribe job. US English (<code>en-US</code>) is the only
         * supported language for Medical Scribe jobs.
         * </p>
         * 
         * @param languageCode
         *        The language code used to create your Medical Scribe job. US English (<code>en-US</code>) is the only
         *        supported language for Medical Scribe jobs.
         * @see MedicalScribeLanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MedicalScribeLanguageCode
         */
        Builder languageCode(String languageCode);

        /**
         * <p>
         * The language code used to create your Medical Scribe job. US English (<code>en-US</code>) is the only
         * supported language for Medical Scribe jobs.
         * </p>
         * 
         * @param languageCode
         *        The language code used to create your Medical Scribe job. US English (<code>en-US</code>) is the only
         *        supported language for Medical Scribe jobs.
         * @see MedicalScribeLanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MedicalScribeLanguageCode
         */
        Builder languageCode(MedicalScribeLanguageCode languageCode);

        /**
         * Sets the value of the Media property for this object.
         *
         * @param media
         *        The new value for the Media property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder media(Media media);

        /**
         * Sets the value of the Media property for this object.
         *
         * This is a convenience method that creates an instance of the {@link Media.Builder} avoiding the need to
         * create one manually via {@link Media#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Media.Builder#build()} is called immediately and its result is
         * passed to {@link #media(Media)}.
         * 
         * @param media
         *        a consumer that will call methods on {@link Media.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #media(Media)
         */
        default Builder media(Consumer<Media.Builder> media) {
            return media(Media.builder().applyMutation(media).build());
        }

        /**
         * <p>
         * The location of the output of your Medical Scribe job. <code>ClinicalDocumentUri</code> holds the Amazon S3
         * URI for the Clinical Document and <code>TranscriptFileUri</code> holds the Amazon S3 URI for the Transcript.
         * </p>
         * 
         * @param medicalScribeOutput
         *        The location of the output of your Medical Scribe job. <code>ClinicalDocumentUri</code> holds the
         *        Amazon S3 URI for the Clinical Document and <code>TranscriptFileUri</code> holds the Amazon S3 URI for
         *        the Transcript.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder medicalScribeOutput(MedicalScribeOutput medicalScribeOutput);

        /**
         * <p>
         * The location of the output of your Medical Scribe job. <code>ClinicalDocumentUri</code> holds the Amazon S3
         * URI for the Clinical Document and <code>TranscriptFileUri</code> holds the Amazon S3 URI for the Transcript.
         * </p>
         * This is a convenience method that creates an instance of the {@link MedicalScribeOutput.Builder} avoiding the
         * need to create one manually via {@link MedicalScribeOutput#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MedicalScribeOutput.Builder#build()} is called immediately and
         * its result is passed to {@link #medicalScribeOutput(MedicalScribeOutput)}.
         * 
         * @param medicalScribeOutput
         *        a consumer that will call methods on {@link MedicalScribeOutput.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #medicalScribeOutput(MedicalScribeOutput)
         */
        default Builder medicalScribeOutput(Consumer<MedicalScribeOutput.Builder> medicalScribeOutput) {
            return medicalScribeOutput(MedicalScribeOutput.builder().applyMutation(medicalScribeOutput).build());
        }

        /**
         * <p>
         * The date and time your Medical Scribe job began processing.
         * </p>
         * <p>
         * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         * <code>2022-05-04T12:32:58.789000-07:00</code> represents a Medical Scribe job that started processing at
         * 12:32 PM UTC-7 on May 4, 2022.
         * </p>
         * 
         * @param startTime
         *        The date and time your Medical Scribe job began processing.</p>
         *        <p>
         *        Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         *        <code>2022-05-04T12:32:58.789000-07:00</code> represents a Medical Scribe job that started processing
         *        at 12:32 PM UTC-7 on May 4, 2022.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The date and time the specified Medical Scribe job request was made.
         * </p>
         * <p>
         * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         * <code>2022-05-04T12:32:58.761000-07:00</code> represents a Medical Scribe job that started processing at
         * 12:32 PM UTC-7 on May 4, 2022.
         * </p>
         * 
         * @param creationTime
         *        The date and time the specified Medical Scribe job request was made.</p>
         *        <p>
         *        Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         *        <code>2022-05-04T12:32:58.761000-07:00</code> represents a Medical Scribe job that started processing
         *        at 12:32 PM UTC-7 on May 4, 2022.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The date and time the specified Medical Scribe job finished processing.
         * </p>
         * <p>
         * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         * <code>2022-05-04T12:32:58.761000-07:00</code> represents a Medical Scribe job that finished processing at
         * 12:32 PM UTC-7 on May 4, 2022.
         * </p>
         * 
         * @param completionTime
         *        The date and time the specified Medical Scribe job finished processing.</p>
         *        <p>
         *        Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         *        <code>2022-05-04T12:32:58.761000-07:00</code> represents a Medical Scribe job that finished processing
         *        at 12:32 PM UTC-7 on May 4, 2022.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completionTime(Instant completionTime);

        /**
         * <p>
         * If <code>MedicalScribeJobStatus</code> is <code>FAILED</code>, <code>FailureReason</code> contains
         * information about why the transcription job failed. See also: <a
         * href="https://docs.aws.amazon.com/transcribe/latest/APIReference/CommonErrors.html">Common Errors</a>.
         * </p>
         * 
         * @param failureReason
         *        If <code>MedicalScribeJobStatus</code> is <code>FAILED</code>, <code>FailureReason</code> contains
         *        information about why the transcription job failed. See also: <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/APIReference/CommonErrors.html">Common Errors</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReason(String failureReason);

        /**
         * <p>
         * Makes it possible to control how your Medical Scribe job is processed using a
         * <code>MedicalScribeSettings</code> object. Specify <code>ChannelIdentification</code> if
         * <code>ChannelDefinitions</code> are set. Enabled <code>ShowSpeakerLabels</code> if
         * <code>ChannelIdentification</code> and <code>ChannelDefinitions</code> are not set. One and only one of
         * <code>ChannelIdentification</code> and <code>ShowSpeakerLabels</code> must be set. If
         * <code>ShowSpeakerLabels</code> is set, <code>MaxSpeakerLabels</code> must also be set. Use
         * <code>Settings</code> to specify a vocabulary or vocabulary filter or both using <code>VocabularyName</code>,
         * <code>VocabularyFilterName</code>. <code>VocabularyFilterMethod</code> must be specified if
         * <code>VocabularyFilterName</code> is set.
         * </p>
         * 
         * @param settings
         *        Makes it possible to control how your Medical Scribe job is processed using a
         *        <code>MedicalScribeSettings</code> object. Specify <code>ChannelIdentification</code> if
         *        <code>ChannelDefinitions</code> are set. Enabled <code>ShowSpeakerLabels</code> if
         *        <code>ChannelIdentification</code> and <code>ChannelDefinitions</code> are not set. One and only one
         *        of <code>ChannelIdentification</code> and <code>ShowSpeakerLabels</code> must be set. If
         *        <code>ShowSpeakerLabels</code> is set, <code>MaxSpeakerLabels</code> must also be set. Use
         *        <code>Settings</code> to specify a vocabulary or vocabulary filter or both using
         *        <code>VocabularyName</code>, <code>VocabularyFilterName</code>. <code>VocabularyFilterMethod</code>
         *        must be specified if <code>VocabularyFilterName</code> is set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder settings(MedicalScribeSettings settings);

        /**
         * <p>
         * Makes it possible to control how your Medical Scribe job is processed using a
         * <code>MedicalScribeSettings</code> object. Specify <code>ChannelIdentification</code> if
         * <code>ChannelDefinitions</code> are set. Enabled <code>ShowSpeakerLabels</code> if
         * <code>ChannelIdentification</code> and <code>ChannelDefinitions</code> are not set. One and only one of
         * <code>ChannelIdentification</code> and <code>ShowSpeakerLabels</code> must be set. If
         * <code>ShowSpeakerLabels</code> is set, <code>MaxSpeakerLabels</code> must also be set. Use
         * <code>Settings</code> to specify a vocabulary or vocabulary filter or both using <code>VocabularyName</code>,
         * <code>VocabularyFilterName</code>. <code>VocabularyFilterMethod</code> must be specified if
         * <code>VocabularyFilterName</code> is set.
         * </p>
         * This is a convenience method that creates an instance of the {@link MedicalScribeSettings.Builder} avoiding
         * the need to create one manually via {@link MedicalScribeSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MedicalScribeSettings.Builder#build()} is called immediately and
         * its result is passed to {@link #settings(MedicalScribeSettings)}.
         * 
         * @param settings
         *        a consumer that will call methods on {@link MedicalScribeSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #settings(MedicalScribeSettings)
         */
        default Builder settings(Consumer<MedicalScribeSettings.Builder> settings) {
            return settings(MedicalScribeSettings.builder().applyMutation(settings).build());
        }

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an IAM role that has permissions to access the Amazon S3 bucket that
         * contains your input files, write to the output bucket, and use your KMS key if supplied. If the role that you
         * specify doesn’t have the appropriate permissions your request fails.
         * </p>
         * <p>
         * IAM role ARNs have the format <code>arn:partition:iam::account:role/role-name-with-path</code>. For example:
         * <code>arn:aws:iam::111122223333:role/Admin</code>.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-arns">IAM
         * ARNs</a>.
         * </p>
         * 
         * @param dataAccessRoleArn
         *        The Amazon Resource Name (ARN) of an IAM role that has permissions to access the Amazon S3 bucket that
         *        contains your input files, write to the output bucket, and use your KMS key if supplied. If the role
         *        that you specify doesn’t have the appropriate permissions your request fails.</p>
         *        <p>
         *        IAM role ARNs have the format <code>arn:partition:iam::account:role/role-name-with-path</code>. For
         *        example: <code>arn:aws:iam::111122223333:role/Admin</code>.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-arns"
         *        >IAM ARNs</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataAccessRoleArn(String dataAccessRoleArn);

        /**
         * <p>
         * Makes it possible to specify which speaker is on which channel. For example, if the clinician is the first
         * participant to speak, you would set <code>ChannelId</code> of the first <code>ChannelDefinition</code> in the
         * list to <code>0</code> (to indicate the first channel) and <code>ParticipantRole</code> to
         * <code>CLINICIAN</code> (to indicate that it's the clinician speaking). Then you would set the
         * <code>ChannelId</code> of the second <code>ChannelDefinition</code> in the list to <code>1</code> (to
         * indicate the second channel) and <code>ParticipantRole</code> to <code>PATIENT</code> (to indicate that it's
         * the patient speaking).
         * </p>
         * 
         * @param channelDefinitions
         *        Makes it possible to specify which speaker is on which channel. For example, if the clinician is the
         *        first participant to speak, you would set <code>ChannelId</code> of the first
         *        <code>ChannelDefinition</code> in the list to <code>0</code> (to indicate the first channel) and
         *        <code>ParticipantRole</code> to <code>CLINICIAN</code> (to indicate that it's the clinician speaking).
         *        Then you would set the <code>ChannelId</code> of the second <code>ChannelDefinition</code> in the list
         *        to <code>1</code> (to indicate the second channel) and <code>ParticipantRole</code> to
         *        <code>PATIENT</code> (to indicate that it's the patient speaking).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder channelDefinitions(Collection<MedicalScribeChannelDefinition> channelDefinitions);

        /**
         * <p>
         * Makes it possible to specify which speaker is on which channel. For example, if the clinician is the first
         * participant to speak, you would set <code>ChannelId</code> of the first <code>ChannelDefinition</code> in the
         * list to <code>0</code> (to indicate the first channel) and <code>ParticipantRole</code> to
         * <code>CLINICIAN</code> (to indicate that it's the clinician speaking). Then you would set the
         * <code>ChannelId</code> of the second <code>ChannelDefinition</code> in the list to <code>1</code> (to
         * indicate the second channel) and <code>ParticipantRole</code> to <code>PATIENT</code> (to indicate that it's
         * the patient speaking).
         * </p>
         * 
         * @param channelDefinitions
         *        Makes it possible to specify which speaker is on which channel. For example, if the clinician is the
         *        first participant to speak, you would set <code>ChannelId</code> of the first
         *        <code>ChannelDefinition</code> in the list to <code>0</code> (to indicate the first channel) and
         *        <code>ParticipantRole</code> to <code>CLINICIAN</code> (to indicate that it's the clinician speaking).
         *        Then you would set the <code>ChannelId</code> of the second <code>ChannelDefinition</code> in the list
         *        to <code>1</code> (to indicate the second channel) and <code>ParticipantRole</code> to
         *        <code>PATIENT</code> (to indicate that it's the patient speaking).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder channelDefinitions(MedicalScribeChannelDefinition... channelDefinitions);

        /**
         * <p>
         * Makes it possible to specify which speaker is on which channel. For example, if the clinician is the first
         * participant to speak, you would set <code>ChannelId</code> of the first <code>ChannelDefinition</code> in the
         * list to <code>0</code> (to indicate the first channel) and <code>ParticipantRole</code> to
         * <code>CLINICIAN</code> (to indicate that it's the clinician speaking). Then you would set the
         * <code>ChannelId</code> of the second <code>ChannelDefinition</code> in the list to <code>1</code> (to
         * indicate the second channel) and <code>ParticipantRole</code> to <code>PATIENT</code> (to indicate that it's
         * the patient speaking).
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.transcribe.model.MedicalScribeChannelDefinition.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.transcribe.model.MedicalScribeChannelDefinition#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.transcribe.model.MedicalScribeChannelDefinition.Builder#build()} is
         * called immediately and its result is passed to {@link
         * #channelDefinitions(List<MedicalScribeChannelDefinition>)}.
         * 
         * @param channelDefinitions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.transcribe.model.MedicalScribeChannelDefinition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #channelDefinitions(java.util.Collection<MedicalScribeChannelDefinition>)
         */
        Builder channelDefinitions(Consumer<MedicalScribeChannelDefinition.Builder>... channelDefinitions);

        /**
         * <p>
         * Adds one or more custom tags, each in the form of a key:value pair, to the Medica Scribe job.
         * </p>
         * <p>
         * To learn more about using tags with Amazon Transcribe, refer to <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/tagging.html">Tagging resources</a>.
         * </p>
         * 
         * @param tags
         *        Adds one or more custom tags, each in the form of a key:value pair, to the Medica Scribe job.</p>
         *        <p>
         *        To learn more about using tags with Amazon Transcribe, refer to <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/tagging.html">Tagging resources</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Adds one or more custom tags, each in the form of a key:value pair, to the Medica Scribe job.
         * </p>
         * <p>
         * To learn more about using tags with Amazon Transcribe, refer to <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/tagging.html">Tagging resources</a>.
         * </p>
         * 
         * @param tags
         *        Adds one or more custom tags, each in the form of a key:value pair, to the Medica Scribe job.</p>
         *        <p>
         *        To learn more about using tags with Amazon Transcribe, refer to <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/tagging.html">Tagging resources</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Adds one or more custom tags, each in the form of a key:value pair, to the Medica Scribe job.
         * </p>
         * <p>
         * To learn more about using tags with Amazon Transcribe, refer to <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/tagging.html">Tagging resources</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.transcribe.model.Tag.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.transcribe.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.transcribe.model.Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.transcribe.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String medicalScribeJobName;

        private String medicalScribeJobStatus;

        private String languageCode;

        private Media media;

        private MedicalScribeOutput medicalScribeOutput;

        private Instant startTime;

        private Instant creationTime;

        private Instant completionTime;

        private String failureReason;

        private MedicalScribeSettings settings;

        private String dataAccessRoleArn;

        private List<MedicalScribeChannelDefinition> channelDefinitions = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MedicalScribeJob model) {
            medicalScribeJobName(model.medicalScribeJobName);
            medicalScribeJobStatus(model.medicalScribeJobStatus);
            languageCode(model.languageCode);
            media(model.media);
            medicalScribeOutput(model.medicalScribeOutput);
            startTime(model.startTime);
            creationTime(model.creationTime);
            completionTime(model.completionTime);
            failureReason(model.failureReason);
            settings(model.settings);
            dataAccessRoleArn(model.dataAccessRoleArn);
            channelDefinitions(model.channelDefinitions);
            tags(model.tags);
        }

        public final String getMedicalScribeJobName() {
            return medicalScribeJobName;
        }

        public final void setMedicalScribeJobName(String medicalScribeJobName) {
            this.medicalScribeJobName = medicalScribeJobName;
        }

        @Override
        public final Builder medicalScribeJobName(String medicalScribeJobName) {
            this.medicalScribeJobName = medicalScribeJobName;
            return this;
        }

        public final String getMedicalScribeJobStatus() {
            return medicalScribeJobStatus;
        }

        public final void setMedicalScribeJobStatus(String medicalScribeJobStatus) {
            this.medicalScribeJobStatus = medicalScribeJobStatus;
        }

        @Override
        public final Builder medicalScribeJobStatus(String medicalScribeJobStatus) {
            this.medicalScribeJobStatus = medicalScribeJobStatus;
            return this;
        }

        @Override
        public final Builder medicalScribeJobStatus(MedicalScribeJobStatus medicalScribeJobStatus) {
            this.medicalScribeJobStatus(medicalScribeJobStatus == null ? null : medicalScribeJobStatus.toString());
            return this;
        }

        public final String getLanguageCode() {
            return languageCode;
        }

        public final void setLanguageCode(String languageCode) {
            this.languageCode = languageCode;
        }

        @Override
        public final Builder languageCode(String languageCode) {
            this.languageCode = languageCode;
            return this;
        }

        @Override
        public final Builder languageCode(MedicalScribeLanguageCode languageCode) {
            this.languageCode(languageCode == null ? null : languageCode.toString());
            return this;
        }

        public final Media.Builder getMedia() {
            return media != null ? media.toBuilder() : null;
        }

        public final void setMedia(Media.BuilderImpl media) {
            this.media = media != null ? media.build() : null;
        }

        @Override
        public final Builder media(Media media) {
            this.media = media;
            return this;
        }

        public final MedicalScribeOutput.Builder getMedicalScribeOutput() {
            return medicalScribeOutput != null ? medicalScribeOutput.toBuilder() : null;
        }

        public final void setMedicalScribeOutput(MedicalScribeOutput.BuilderImpl medicalScribeOutput) {
            this.medicalScribeOutput = medicalScribeOutput != null ? medicalScribeOutput.build() : null;
        }

        @Override
        public final Builder medicalScribeOutput(MedicalScribeOutput medicalScribeOutput) {
            this.medicalScribeOutput = medicalScribeOutput;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final Instant getCompletionTime() {
            return completionTime;
        }

        public final void setCompletionTime(Instant completionTime) {
            this.completionTime = completionTime;
        }

        @Override
        public final Builder completionTime(Instant completionTime) {
            this.completionTime = completionTime;
            return this;
        }

        public final String getFailureReason() {
            return failureReason;
        }

        public final void setFailureReason(String failureReason) {
            this.failureReason = failureReason;
        }

        @Override
        public final Builder failureReason(String failureReason) {
            this.failureReason = failureReason;
            return this;
        }

        public final MedicalScribeSettings.Builder getSettings() {
            return settings != null ? settings.toBuilder() : null;
        }

        public final void setSettings(MedicalScribeSettings.BuilderImpl settings) {
            this.settings = settings != null ? settings.build() : null;
        }

        @Override
        @Transient
        public final Builder settings(MedicalScribeSettings settings) {
            this.settings = settings;
            return this;
        }

        public final String getDataAccessRoleArn() {
            return dataAccessRoleArn;
        }

        public final void setDataAccessRoleArn(String dataAccessRoleArn) {
            this.dataAccessRoleArn = dataAccessRoleArn;
        }

        @Override
        public final Builder dataAccessRoleArn(String dataAccessRoleArn) {
            this.dataAccessRoleArn = dataAccessRoleArn;
            return this;
        }

        public final List<MedicalScribeChannelDefinition.Builder> getChannelDefinitions() {
            List<MedicalScribeChannelDefinition.Builder> result = MedicalScribeChannelDefinitionsCopier
                    .copyToBuilder(this.channelDefinitions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setChannelDefinitions(Collection<MedicalScribeChannelDefinition.BuilderImpl> channelDefinitions) {
            this.channelDefinitions = MedicalScribeChannelDefinitionsCopier.copyFromBuilder(channelDefinitions);
        }

        @Override
        public final Builder channelDefinitions(Collection<MedicalScribeChannelDefinition> channelDefinitions) {
            this.channelDefinitions = MedicalScribeChannelDefinitionsCopier.copy(channelDefinitions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder channelDefinitions(MedicalScribeChannelDefinition... channelDefinitions) {
            channelDefinitions(Arrays.asList(channelDefinitions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder channelDefinitions(Consumer<MedicalScribeChannelDefinition.Builder>... channelDefinitions) {
            channelDefinitions(Stream.of(channelDefinitions)
                    .map(c -> MedicalScribeChannelDefinition.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public MedicalScribeJob build() {
            return new MedicalScribeJob(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
