/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribe.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Flag the presence or absence of specific words or phrases detected in your Call Analytics transcription output.
 * </p>
 * <p>
 * Rules using <code>TranscriptFilter</code> are designed to match:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Custom words or phrases spoken by the agent, the customer, or both
 * </p>
 * </li>
 * <li>
 * <p>
 * Custom words or phrases <b>not</b> spoken by the agent, the customer, or either
 * </p>
 * </li>
 * <li>
 * <p>
 * Custom words or phrases that occur at a specific time frame
 * </p>
 * </li>
 * </ul>
 * <p>
 * See <a href="https://docs.aws.amazon.com/transcribe/latest/dg/tca-categories-batch.html#tca-rules-batch">Rule
 * criteria for post-call categories</a> and <a
 * href="https://docs.aws.amazon.com/transcribe/latest/dg/tca-categories-stream.html#tca-rules-stream">Rule criteria for
 * streaming categories</a> for usage examples.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TranscriptFilter implements SdkPojo, Serializable,
        ToCopyableBuilder<TranscriptFilter.Builder, TranscriptFilter> {
    private static final SdkField<String> TRANSCRIPT_FILTER_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TranscriptFilterType").getter(getter(TranscriptFilter::transcriptFilterTypeAsString))
            .setter(setter(Builder::transcriptFilterType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TranscriptFilterType").build())
            .build();

    private static final SdkField<AbsoluteTimeRange> ABSOLUTE_TIME_RANGE_FIELD = SdkField
            .<AbsoluteTimeRange> builder(MarshallingType.SDK_POJO).memberName("AbsoluteTimeRange")
            .getter(getter(TranscriptFilter::absoluteTimeRange)).setter(setter(Builder::absoluteTimeRange))
            .constructor(AbsoluteTimeRange::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AbsoluteTimeRange").build()).build();

    private static final SdkField<RelativeTimeRange> RELATIVE_TIME_RANGE_FIELD = SdkField
            .<RelativeTimeRange> builder(MarshallingType.SDK_POJO).memberName("RelativeTimeRange")
            .getter(getter(TranscriptFilter::relativeTimeRange)).setter(setter(Builder::relativeTimeRange))
            .constructor(RelativeTimeRange::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RelativeTimeRange").build()).build();

    private static final SdkField<String> PARTICIPANT_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ParticipantRole").getter(getter(TranscriptFilter::participantRoleAsString))
            .setter(setter(Builder::participantRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParticipantRole").build()).build();

    private static final SdkField<Boolean> NEGATE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Negate").getter(getter(TranscriptFilter::negate)).setter(setter(Builder::negate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Negate").build()).build();

    private static final SdkField<List<String>> TARGETS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Targets")
            .getter(getter(TranscriptFilter::targets))
            .setter(setter(Builder::targets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Targets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TRANSCRIPT_FILTER_TYPE_FIELD,
            ABSOLUTE_TIME_RANGE_FIELD, RELATIVE_TIME_RANGE_FIELD, PARTICIPANT_ROLE_FIELD, NEGATE_FIELD, TARGETS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("TranscriptFilterType", TRANSCRIPT_FILTER_TYPE_FIELD);
                    put("AbsoluteTimeRange", ABSOLUTE_TIME_RANGE_FIELD);
                    put("RelativeTimeRange", RELATIVE_TIME_RANGE_FIELD);
                    put("ParticipantRole", PARTICIPANT_ROLE_FIELD);
                    put("Negate", NEGATE_FIELD);
                    put("Targets", TARGETS_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String transcriptFilterType;

    private final AbsoluteTimeRange absoluteTimeRange;

    private final RelativeTimeRange relativeTimeRange;

    private final String participantRole;

    private final Boolean negate;

    private final List<String> targets;

    private TranscriptFilter(BuilderImpl builder) {
        this.transcriptFilterType = builder.transcriptFilterType;
        this.absoluteTimeRange = builder.absoluteTimeRange;
        this.relativeTimeRange = builder.relativeTimeRange;
        this.participantRole = builder.participantRole;
        this.negate = builder.negate;
        this.targets = builder.targets;
    }

    /**
     * <p>
     * Flag the presence or absence of an exact match to the phrases that you specify. For example, if you specify the
     * phrase "speak to a manager" as your <code>Targets</code> value, only that exact phrase is flagged.
     * </p>
     * <p>
     * Note that semantic matching is not supported. For example, if your customer says "speak to <i>the</i> manager",
     * instead of "speak to <i>a</i> manager", your content is not flagged.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #transcriptFilterType} will return {@link TranscriptFilterType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #transcriptFilterTypeAsString}.
     * </p>
     * 
     * @return Flag the presence or absence of an exact match to the phrases that you specify. For example, if you
     *         specify the phrase "speak to a manager" as your <code>Targets</code> value, only that exact phrase is
     *         flagged.</p>
     *         <p>
     *         Note that semantic matching is not supported. For example, if your customer says
     *         "speak to <i>the</i> manager", instead of "speak to <i>a</i> manager", your content is not flagged.
     * @see TranscriptFilterType
     */
    public final TranscriptFilterType transcriptFilterType() {
        return TranscriptFilterType.fromValue(transcriptFilterType);
    }

    /**
     * <p>
     * Flag the presence or absence of an exact match to the phrases that you specify. For example, if you specify the
     * phrase "speak to a manager" as your <code>Targets</code> value, only that exact phrase is flagged.
     * </p>
     * <p>
     * Note that semantic matching is not supported. For example, if your customer says "speak to <i>the</i> manager",
     * instead of "speak to <i>a</i> manager", your content is not flagged.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #transcriptFilterType} will return {@link TranscriptFilterType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #transcriptFilterTypeAsString}.
     * </p>
     * 
     * @return Flag the presence or absence of an exact match to the phrases that you specify. For example, if you
     *         specify the phrase "speak to a manager" as your <code>Targets</code> value, only that exact phrase is
     *         flagged.</p>
     *         <p>
     *         Note that semantic matching is not supported. For example, if your customer says
     *         "speak to <i>the</i> manager", instead of "speak to <i>a</i> manager", your content is not flagged.
     * @see TranscriptFilterType
     */
    public final String transcriptFilterTypeAsString() {
        return transcriptFilterType;
    }

    /**
     * <p>
     * Makes it possible to specify a time range (in milliseconds) in your audio, during which you want to search for
     * the specified key words or phrases. See for more detail.
     * </p>
     * 
     * @return Makes it possible to specify a time range (in milliseconds) in your audio, during which you want to
     *         search for the specified key words or phrases. See for more detail.
     */
    public final AbsoluteTimeRange absoluteTimeRange() {
        return absoluteTimeRange;
    }

    /**
     * <p>
     * Makes it possible to specify a time range (in percentage) in your media file, during which you want to search for
     * the specified key words or phrases. See for more detail.
     * </p>
     * 
     * @return Makes it possible to specify a time range (in percentage) in your media file, during which you want to
     *         search for the specified key words or phrases. See for more detail.
     */
    public final RelativeTimeRange relativeTimeRange() {
        return relativeTimeRange;
    }

    /**
     * <p>
     * Specify the participant that you want to flag. Omitting this parameter is equivalent to specifying both
     * participants.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #participantRole}
     * will return {@link ParticipantRole#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #participantRoleAsString}.
     * </p>
     * 
     * @return Specify the participant that you want to flag. Omitting this parameter is equivalent to specifying both
     *         participants.
     * @see ParticipantRole
     */
    public final ParticipantRole participantRole() {
        return ParticipantRole.fromValue(participantRole);
    }

    /**
     * <p>
     * Specify the participant that you want to flag. Omitting this parameter is equivalent to specifying both
     * participants.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #participantRole}
     * will return {@link ParticipantRole#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #participantRoleAsString}.
     * </p>
     * 
     * @return Specify the participant that you want to flag. Omitting this parameter is equivalent to specifying both
     *         participants.
     * @see ParticipantRole
     */
    public final String participantRoleAsString() {
        return participantRole;
    }

    /**
     * <p>
     * Set to <code>TRUE</code> to flag the absence of the phrase that you specified in your request. Set to
     * <code>FALSE</code> to flag the presence of the phrase that you specified in your request.
     * </p>
     * 
     * @return Set to <code>TRUE</code> to flag the absence of the phrase that you specified in your request. Set to
     *         <code>FALSE</code> to flag the presence of the phrase that you specified in your request.
     */
    public final Boolean negate() {
        return negate;
    }

    /**
     * For responses, this returns true if the service returned a value for the Targets property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTargets() {
        return targets != null && !(targets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specify the phrases that you want to flag.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTargets} method.
     * </p>
     * 
     * @return Specify the phrases that you want to flag.
     */
    public final List<String> targets() {
        return targets;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(transcriptFilterTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(absoluteTimeRange());
        hashCode = 31 * hashCode + Objects.hashCode(relativeTimeRange());
        hashCode = 31 * hashCode + Objects.hashCode(participantRoleAsString());
        hashCode = 31 * hashCode + Objects.hashCode(negate());
        hashCode = 31 * hashCode + Objects.hashCode(hasTargets() ? targets() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TranscriptFilter)) {
            return false;
        }
        TranscriptFilter other = (TranscriptFilter) obj;
        return Objects.equals(transcriptFilterTypeAsString(), other.transcriptFilterTypeAsString())
                && Objects.equals(absoluteTimeRange(), other.absoluteTimeRange())
                && Objects.equals(relativeTimeRange(), other.relativeTimeRange())
                && Objects.equals(participantRoleAsString(), other.participantRoleAsString())
                && Objects.equals(negate(), other.negate()) && hasTargets() == other.hasTargets()
                && Objects.equals(targets(), other.targets());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TranscriptFilter").add("TranscriptFilterType", transcriptFilterTypeAsString())
                .add("AbsoluteTimeRange", absoluteTimeRange()).add("RelativeTimeRange", relativeTimeRange())
                .add("ParticipantRole", participantRoleAsString()).add("Negate", negate())
                .add("Targets", hasTargets() ? targets() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TranscriptFilterType":
            return Optional.ofNullable(clazz.cast(transcriptFilterTypeAsString()));
        case "AbsoluteTimeRange":
            return Optional.ofNullable(clazz.cast(absoluteTimeRange()));
        case "RelativeTimeRange":
            return Optional.ofNullable(clazz.cast(relativeTimeRange()));
        case "ParticipantRole":
            return Optional.ofNullable(clazz.cast(participantRoleAsString()));
        case "Negate":
            return Optional.ofNullable(clazz.cast(negate()));
        case "Targets":
            return Optional.ofNullable(clazz.cast(targets()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<TranscriptFilter, T> g) {
        return obj -> g.apply((TranscriptFilter) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TranscriptFilter> {
        /**
         * <p>
         * Flag the presence or absence of an exact match to the phrases that you specify. For example, if you specify
         * the phrase "speak to a manager" as your <code>Targets</code> value, only that exact phrase is flagged.
         * </p>
         * <p>
         * Note that semantic matching is not supported. For example, if your customer says
         * "speak to <i>the</i> manager", instead of "speak to <i>a</i> manager", your content is not flagged.
         * </p>
         * 
         * @param transcriptFilterType
         *        Flag the presence or absence of an exact match to the phrases that you specify. For example, if you
         *        specify the phrase "speak to a manager" as your <code>Targets</code> value, only that exact phrase is
         *        flagged.</p>
         *        <p>
         *        Note that semantic matching is not supported. For example, if your customer says
         *        "speak to <i>the</i> manager", instead of "speak to <i>a</i> manager", your content is not flagged.
         * @see TranscriptFilterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscriptFilterType
         */
        Builder transcriptFilterType(String transcriptFilterType);

        /**
         * <p>
         * Flag the presence or absence of an exact match to the phrases that you specify. For example, if you specify
         * the phrase "speak to a manager" as your <code>Targets</code> value, only that exact phrase is flagged.
         * </p>
         * <p>
         * Note that semantic matching is not supported. For example, if your customer says
         * "speak to <i>the</i> manager", instead of "speak to <i>a</i> manager", your content is not flagged.
         * </p>
         * 
         * @param transcriptFilterType
         *        Flag the presence or absence of an exact match to the phrases that you specify. For example, if you
         *        specify the phrase "speak to a manager" as your <code>Targets</code> value, only that exact phrase is
         *        flagged.</p>
         *        <p>
         *        Note that semantic matching is not supported. For example, if your customer says
         *        "speak to <i>the</i> manager", instead of "speak to <i>a</i> manager", your content is not flagged.
         * @see TranscriptFilterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscriptFilterType
         */
        Builder transcriptFilterType(TranscriptFilterType transcriptFilterType);

        /**
         * <p>
         * Makes it possible to specify a time range (in milliseconds) in your audio, during which you want to search
         * for the specified key words or phrases. See for more detail.
         * </p>
         * 
         * @param absoluteTimeRange
         *        Makes it possible to specify a time range (in milliseconds) in your audio, during which you want to
         *        search for the specified key words or phrases. See for more detail.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder absoluteTimeRange(AbsoluteTimeRange absoluteTimeRange);

        /**
         * <p>
         * Makes it possible to specify a time range (in milliseconds) in your audio, during which you want to search
         * for the specified key words or phrases. See for more detail.
         * </p>
         * This is a convenience method that creates an instance of the {@link AbsoluteTimeRange.Builder} avoiding the
         * need to create one manually via {@link AbsoluteTimeRange#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AbsoluteTimeRange.Builder#build()} is called immediately and its
         * result is passed to {@link #absoluteTimeRange(AbsoluteTimeRange)}.
         * 
         * @param absoluteTimeRange
         *        a consumer that will call methods on {@link AbsoluteTimeRange.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #absoluteTimeRange(AbsoluteTimeRange)
         */
        default Builder absoluteTimeRange(Consumer<AbsoluteTimeRange.Builder> absoluteTimeRange) {
            return absoluteTimeRange(AbsoluteTimeRange.builder().applyMutation(absoluteTimeRange).build());
        }

        /**
         * <p>
         * Makes it possible to specify a time range (in percentage) in your media file, during which you want to search
         * for the specified key words or phrases. See for more detail.
         * </p>
         * 
         * @param relativeTimeRange
         *        Makes it possible to specify a time range (in percentage) in your media file, during which you want to
         *        search for the specified key words or phrases. See for more detail.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relativeTimeRange(RelativeTimeRange relativeTimeRange);

        /**
         * <p>
         * Makes it possible to specify a time range (in percentage) in your media file, during which you want to search
         * for the specified key words or phrases. See for more detail.
         * </p>
         * This is a convenience method that creates an instance of the {@link RelativeTimeRange.Builder} avoiding the
         * need to create one manually via {@link RelativeTimeRange#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RelativeTimeRange.Builder#build()} is called immediately and its
         * result is passed to {@link #relativeTimeRange(RelativeTimeRange)}.
         * 
         * @param relativeTimeRange
         *        a consumer that will call methods on {@link RelativeTimeRange.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #relativeTimeRange(RelativeTimeRange)
         */
        default Builder relativeTimeRange(Consumer<RelativeTimeRange.Builder> relativeTimeRange) {
            return relativeTimeRange(RelativeTimeRange.builder().applyMutation(relativeTimeRange).build());
        }

        /**
         * <p>
         * Specify the participant that you want to flag. Omitting this parameter is equivalent to specifying both
         * participants.
         * </p>
         * 
         * @param participantRole
         *        Specify the participant that you want to flag. Omitting this parameter is equivalent to specifying
         *        both participants.
         * @see ParticipantRole
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParticipantRole
         */
        Builder participantRole(String participantRole);

        /**
         * <p>
         * Specify the participant that you want to flag. Omitting this parameter is equivalent to specifying both
         * participants.
         * </p>
         * 
         * @param participantRole
         *        Specify the participant that you want to flag. Omitting this parameter is equivalent to specifying
         *        both participants.
         * @see ParticipantRole
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParticipantRole
         */
        Builder participantRole(ParticipantRole participantRole);

        /**
         * <p>
         * Set to <code>TRUE</code> to flag the absence of the phrase that you specified in your request. Set to
         * <code>FALSE</code> to flag the presence of the phrase that you specified in your request.
         * </p>
         * 
         * @param negate
         *        Set to <code>TRUE</code> to flag the absence of the phrase that you specified in your request. Set to
         *        <code>FALSE</code> to flag the presence of the phrase that you specified in your request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder negate(Boolean negate);

        /**
         * <p>
         * Specify the phrases that you want to flag.
         * </p>
         * 
         * @param targets
         *        Specify the phrases that you want to flag.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<String> targets);

        /**
         * <p>
         * Specify the phrases that you want to flag.
         * </p>
         * 
         * @param targets
         *        Specify the phrases that you want to flag.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(String... targets);
    }

    static final class BuilderImpl implements Builder {
        private String transcriptFilterType;

        private AbsoluteTimeRange absoluteTimeRange;

        private RelativeTimeRange relativeTimeRange;

        private String participantRole;

        private Boolean negate;

        private List<String> targets = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(TranscriptFilter model) {
            transcriptFilterType(model.transcriptFilterType);
            absoluteTimeRange(model.absoluteTimeRange);
            relativeTimeRange(model.relativeTimeRange);
            participantRole(model.participantRole);
            negate(model.negate);
            targets(model.targets);
        }

        public final String getTranscriptFilterType() {
            return transcriptFilterType;
        }

        public final void setTranscriptFilterType(String transcriptFilterType) {
            this.transcriptFilterType = transcriptFilterType;
        }

        @Override
        public final Builder transcriptFilterType(String transcriptFilterType) {
            this.transcriptFilterType = transcriptFilterType;
            return this;
        }

        @Override
        public final Builder transcriptFilterType(TranscriptFilterType transcriptFilterType) {
            this.transcriptFilterType(transcriptFilterType == null ? null : transcriptFilterType.toString());
            return this;
        }

        public final AbsoluteTimeRange.Builder getAbsoluteTimeRange() {
            return absoluteTimeRange != null ? absoluteTimeRange.toBuilder() : null;
        }

        public final void setAbsoluteTimeRange(AbsoluteTimeRange.BuilderImpl absoluteTimeRange) {
            this.absoluteTimeRange = absoluteTimeRange != null ? absoluteTimeRange.build() : null;
        }

        @Override
        public final Builder absoluteTimeRange(AbsoluteTimeRange absoluteTimeRange) {
            this.absoluteTimeRange = absoluteTimeRange;
            return this;
        }

        public final RelativeTimeRange.Builder getRelativeTimeRange() {
            return relativeTimeRange != null ? relativeTimeRange.toBuilder() : null;
        }

        public final void setRelativeTimeRange(RelativeTimeRange.BuilderImpl relativeTimeRange) {
            this.relativeTimeRange = relativeTimeRange != null ? relativeTimeRange.build() : null;
        }

        @Override
        public final Builder relativeTimeRange(RelativeTimeRange relativeTimeRange) {
            this.relativeTimeRange = relativeTimeRange;
            return this;
        }

        public final String getParticipantRole() {
            return participantRole;
        }

        public final void setParticipantRole(String participantRole) {
            this.participantRole = participantRole;
        }

        @Override
        public final Builder participantRole(String participantRole) {
            this.participantRole = participantRole;
            return this;
        }

        @Override
        public final Builder participantRole(ParticipantRole participantRole) {
            this.participantRole(participantRole == null ? null : participantRole.toString());
            return this;
        }

        public final Boolean getNegate() {
            return negate;
        }

        public final void setNegate(Boolean negate) {
            this.negate = negate;
        }

        @Override
        public final Builder negate(Boolean negate) {
            this.negate = negate;
            return this;
        }

        public final Collection<String> getTargets() {
            if (targets instanceof SdkAutoConstructList) {
                return null;
            }
            return targets;
        }

        public final void setTargets(Collection<String> targets) {
            this.targets = StringTargetListCopier.copy(targets);
        }

        @Override
        public final Builder targets(Collection<String> targets) {
            this.targets = StringTargetListCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(String... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        @Override
        public TranscriptFilter build() {
            return new TranscriptFilter(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
