/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribe.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Flag the presence or absence of specific sentiments detected in your Call Analytics transcription output.
 * </p>
 * <p>
 * Rules using <code>SentimentFilter</code> are designed to match:
 * </p>
 * <ul>
 * <li>
 * <p>
 * The presence or absence of a positive sentiment felt by the customer, agent, or both at specified points in the call
 * </p>
 * </li>
 * <li>
 * <p>
 * The presence or absence of a negative sentiment felt by the customer, agent, or both at specified points in the call
 * </p>
 * </li>
 * <li>
 * <p>
 * The presence or absence of a neutral sentiment felt by the customer, agent, or both at specified points in the call
 * </p>
 * </li>
 * <li>
 * <p>
 * The presence or absence of a mixed sentiment felt by the customer, the agent, or both at specified points in the call
 * </p>
 * </li>
 * </ul>
 * <p>
 * See <a href="https://docs.aws.amazon.com/transcribe/latest/dg/tca-categories-batch.html#tca-rules-batch">Rule
 * criteria for post-call categories</a> for usage examples.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SentimentFilter implements SdkPojo, Serializable, ToCopyableBuilder<SentimentFilter.Builder, SentimentFilter> {
    private static final SdkField<List<String>> SENTIMENTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Sentiments")
            .getter(getter(SentimentFilter::sentimentsAsStrings))
            .setter(setter(Builder::sentimentsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Sentiments").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<AbsoluteTimeRange> ABSOLUTE_TIME_RANGE_FIELD = SdkField
            .<AbsoluteTimeRange> builder(MarshallingType.SDK_POJO).memberName("AbsoluteTimeRange")
            .getter(getter(SentimentFilter::absoluteTimeRange)).setter(setter(Builder::absoluteTimeRange))
            .constructor(AbsoluteTimeRange::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AbsoluteTimeRange").build()).build();

    private static final SdkField<RelativeTimeRange> RELATIVE_TIME_RANGE_FIELD = SdkField
            .<RelativeTimeRange> builder(MarshallingType.SDK_POJO).memberName("RelativeTimeRange")
            .getter(getter(SentimentFilter::relativeTimeRange)).setter(setter(Builder::relativeTimeRange))
            .constructor(RelativeTimeRange::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RelativeTimeRange").build()).build();

    private static final SdkField<String> PARTICIPANT_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ParticipantRole").getter(getter(SentimentFilter::participantRoleAsString))
            .setter(setter(Builder::participantRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParticipantRole").build()).build();

    private static final SdkField<Boolean> NEGATE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Negate").getter(getter(SentimentFilter::negate)).setter(setter(Builder::negate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Negate").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SENTIMENTS_FIELD,
            ABSOLUTE_TIME_RANGE_FIELD, RELATIVE_TIME_RANGE_FIELD, PARTICIPANT_ROLE_FIELD, NEGATE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<String> sentiments;

    private final AbsoluteTimeRange absoluteTimeRange;

    private final RelativeTimeRange relativeTimeRange;

    private final String participantRole;

    private final Boolean negate;

    private SentimentFilter(BuilderImpl builder) {
        this.sentiments = builder.sentiments;
        this.absoluteTimeRange = builder.absoluteTimeRange;
        this.relativeTimeRange = builder.relativeTimeRange;
        this.participantRole = builder.participantRole;
        this.negate = builder.negate;
    }

    /**
     * <p>
     * Specify the sentiments that you want to flag.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSentiments} method.
     * </p>
     * 
     * @return Specify the sentiments that you want to flag.
     */
    public final List<SentimentValue> sentiments() {
        return SentimentValueListCopier.copyStringToEnum(sentiments);
    }

    /**
     * For responses, this returns true if the service returned a value for the Sentiments property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSentiments() {
        return sentiments != null && !(sentiments instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specify the sentiments that you want to flag.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSentiments} method.
     * </p>
     * 
     * @return Specify the sentiments that you want to flag.
     */
    public final List<String> sentimentsAsStrings() {
        return sentiments;
    }

    /**
     * <p>
     * Makes it possible to specify a time range (in milliseconds) in your audio, during which you want to search for
     * the specified sentiments. See for more detail.
     * </p>
     * 
     * @return Makes it possible to specify a time range (in milliseconds) in your audio, during which you want to
     *         search for the specified sentiments. See for more detail.
     */
    public final AbsoluteTimeRange absoluteTimeRange() {
        return absoluteTimeRange;
    }

    /**
     * <p>
     * Makes it possible to specify a time range (in percentage) in your media file, during which you want to search for
     * the specified sentiments. See for more detail.
     * </p>
     * 
     * @return Makes it possible to specify a time range (in percentage) in your media file, during which you want to
     *         search for the specified sentiments. See for more detail.
     */
    public final RelativeTimeRange relativeTimeRange() {
        return relativeTimeRange;
    }

    /**
     * <p>
     * Specify the participant that you want to flag. Omitting this parameter is equivalent to specifying both
     * participants.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #participantRole}
     * will return {@link ParticipantRole#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #participantRoleAsString}.
     * </p>
     * 
     * @return Specify the participant that you want to flag. Omitting this parameter is equivalent to specifying both
     *         participants.
     * @see ParticipantRole
     */
    public final ParticipantRole participantRole() {
        return ParticipantRole.fromValue(participantRole);
    }

    /**
     * <p>
     * Specify the participant that you want to flag. Omitting this parameter is equivalent to specifying both
     * participants.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #participantRole}
     * will return {@link ParticipantRole#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #participantRoleAsString}.
     * </p>
     * 
     * @return Specify the participant that you want to flag. Omitting this parameter is equivalent to specifying both
     *         participants.
     * @see ParticipantRole
     */
    public final String participantRoleAsString() {
        return participantRole;
    }

    /**
     * <p>
     * Set to <code>TRUE</code> to flag the sentiments that you didn't include in your request. Set to
     * <code>FALSE</code> to flag the sentiments that you specified in your request.
     * </p>
     * 
     * @return Set to <code>TRUE</code> to flag the sentiments that you didn't include in your request. Set to
     *         <code>FALSE</code> to flag the sentiments that you specified in your request.
     */
    public final Boolean negate() {
        return negate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasSentiments() ? sentimentsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(absoluteTimeRange());
        hashCode = 31 * hashCode + Objects.hashCode(relativeTimeRange());
        hashCode = 31 * hashCode + Objects.hashCode(participantRoleAsString());
        hashCode = 31 * hashCode + Objects.hashCode(negate());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SentimentFilter)) {
            return false;
        }
        SentimentFilter other = (SentimentFilter) obj;
        return hasSentiments() == other.hasSentiments() && Objects.equals(sentimentsAsStrings(), other.sentimentsAsStrings())
                && Objects.equals(absoluteTimeRange(), other.absoluteTimeRange())
                && Objects.equals(relativeTimeRange(), other.relativeTimeRange())
                && Objects.equals(participantRoleAsString(), other.participantRoleAsString())
                && Objects.equals(negate(), other.negate());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SentimentFilter").add("Sentiments", hasSentiments() ? sentimentsAsStrings() : null)
                .add("AbsoluteTimeRange", absoluteTimeRange()).add("RelativeTimeRange", relativeTimeRange())
                .add("ParticipantRole", participantRoleAsString()).add("Negate", negate()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Sentiments":
            return Optional.ofNullable(clazz.cast(sentimentsAsStrings()));
        case "AbsoluteTimeRange":
            return Optional.ofNullable(clazz.cast(absoluteTimeRange()));
        case "RelativeTimeRange":
            return Optional.ofNullable(clazz.cast(relativeTimeRange()));
        case "ParticipantRole":
            return Optional.ofNullable(clazz.cast(participantRoleAsString()));
        case "Negate":
            return Optional.ofNullable(clazz.cast(negate()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Sentiments", SENTIMENTS_FIELD);
        map.put("AbsoluteTimeRange", ABSOLUTE_TIME_RANGE_FIELD);
        map.put("RelativeTimeRange", RELATIVE_TIME_RANGE_FIELD);
        map.put("ParticipantRole", PARTICIPANT_ROLE_FIELD);
        map.put("Negate", NEGATE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SentimentFilter, T> g) {
        return obj -> g.apply((SentimentFilter) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SentimentFilter> {
        /**
         * <p>
         * Specify the sentiments that you want to flag.
         * </p>
         * 
         * @param sentiments
         *        Specify the sentiments that you want to flag.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sentimentsWithStrings(Collection<String> sentiments);

        /**
         * <p>
         * Specify the sentiments that you want to flag.
         * </p>
         * 
         * @param sentiments
         *        Specify the sentiments that you want to flag.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sentimentsWithStrings(String... sentiments);

        /**
         * <p>
         * Specify the sentiments that you want to flag.
         * </p>
         * 
         * @param sentiments
         *        Specify the sentiments that you want to flag.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sentiments(Collection<SentimentValue> sentiments);

        /**
         * <p>
         * Specify the sentiments that you want to flag.
         * </p>
         * 
         * @param sentiments
         *        Specify the sentiments that you want to flag.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sentiments(SentimentValue... sentiments);

        /**
         * <p>
         * Makes it possible to specify a time range (in milliseconds) in your audio, during which you want to search
         * for the specified sentiments. See for more detail.
         * </p>
         * 
         * @param absoluteTimeRange
         *        Makes it possible to specify a time range (in milliseconds) in your audio, during which you want to
         *        search for the specified sentiments. See for more detail.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder absoluteTimeRange(AbsoluteTimeRange absoluteTimeRange);

        /**
         * <p>
         * Makes it possible to specify a time range (in milliseconds) in your audio, during which you want to search
         * for the specified sentiments. See for more detail.
         * </p>
         * This is a convenience method that creates an instance of the {@link AbsoluteTimeRange.Builder} avoiding the
         * need to create one manually via {@link AbsoluteTimeRange#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AbsoluteTimeRange.Builder#build()} is called immediately and its
         * result is passed to {@link #absoluteTimeRange(AbsoluteTimeRange)}.
         * 
         * @param absoluteTimeRange
         *        a consumer that will call methods on {@link AbsoluteTimeRange.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #absoluteTimeRange(AbsoluteTimeRange)
         */
        default Builder absoluteTimeRange(Consumer<AbsoluteTimeRange.Builder> absoluteTimeRange) {
            return absoluteTimeRange(AbsoluteTimeRange.builder().applyMutation(absoluteTimeRange).build());
        }

        /**
         * <p>
         * Makes it possible to specify a time range (in percentage) in your media file, during which you want to search
         * for the specified sentiments. See for more detail.
         * </p>
         * 
         * @param relativeTimeRange
         *        Makes it possible to specify a time range (in percentage) in your media file, during which you want to
         *        search for the specified sentiments. See for more detail.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relativeTimeRange(RelativeTimeRange relativeTimeRange);

        /**
         * <p>
         * Makes it possible to specify a time range (in percentage) in your media file, during which you want to search
         * for the specified sentiments. See for more detail.
         * </p>
         * This is a convenience method that creates an instance of the {@link RelativeTimeRange.Builder} avoiding the
         * need to create one manually via {@link RelativeTimeRange#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RelativeTimeRange.Builder#build()} is called immediately and its
         * result is passed to {@link #relativeTimeRange(RelativeTimeRange)}.
         * 
         * @param relativeTimeRange
         *        a consumer that will call methods on {@link RelativeTimeRange.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #relativeTimeRange(RelativeTimeRange)
         */
        default Builder relativeTimeRange(Consumer<RelativeTimeRange.Builder> relativeTimeRange) {
            return relativeTimeRange(RelativeTimeRange.builder().applyMutation(relativeTimeRange).build());
        }

        /**
         * <p>
         * Specify the participant that you want to flag. Omitting this parameter is equivalent to specifying both
         * participants.
         * </p>
         * 
         * @param participantRole
         *        Specify the participant that you want to flag. Omitting this parameter is equivalent to specifying
         *        both participants.
         * @see ParticipantRole
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParticipantRole
         */
        Builder participantRole(String participantRole);

        /**
         * <p>
         * Specify the participant that you want to flag. Omitting this parameter is equivalent to specifying both
         * participants.
         * </p>
         * 
         * @param participantRole
         *        Specify the participant that you want to flag. Omitting this parameter is equivalent to specifying
         *        both participants.
         * @see ParticipantRole
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParticipantRole
         */
        Builder participantRole(ParticipantRole participantRole);

        /**
         * <p>
         * Set to <code>TRUE</code> to flag the sentiments that you didn't include in your request. Set to
         * <code>FALSE</code> to flag the sentiments that you specified in your request.
         * </p>
         * 
         * @param negate
         *        Set to <code>TRUE</code> to flag the sentiments that you didn't include in your request. Set to
         *        <code>FALSE</code> to flag the sentiments that you specified in your request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder negate(Boolean negate);
    }

    static final class BuilderImpl implements Builder {
        private List<String> sentiments = DefaultSdkAutoConstructList.getInstance();

        private AbsoluteTimeRange absoluteTimeRange;

        private RelativeTimeRange relativeTimeRange;

        private String participantRole;

        private Boolean negate;

        private BuilderImpl() {
        }

        private BuilderImpl(SentimentFilter model) {
            sentimentsWithStrings(model.sentiments);
            absoluteTimeRange(model.absoluteTimeRange);
            relativeTimeRange(model.relativeTimeRange);
            participantRole(model.participantRole);
            negate(model.negate);
        }

        public final Collection<String> getSentiments() {
            if (sentiments instanceof SdkAutoConstructList) {
                return null;
            }
            return sentiments;
        }

        public final void setSentiments(Collection<String> sentiments) {
            this.sentiments = SentimentValueListCopier.copy(sentiments);
        }

        @Override
        public final Builder sentimentsWithStrings(Collection<String> sentiments) {
            this.sentiments = SentimentValueListCopier.copy(sentiments);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sentimentsWithStrings(String... sentiments) {
            sentimentsWithStrings(Arrays.asList(sentiments));
            return this;
        }

        @Override
        public final Builder sentiments(Collection<SentimentValue> sentiments) {
            this.sentiments = SentimentValueListCopier.copyEnumToString(sentiments);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sentiments(SentimentValue... sentiments) {
            sentiments(Arrays.asList(sentiments));
            return this;
        }

        public final AbsoluteTimeRange.Builder getAbsoluteTimeRange() {
            return absoluteTimeRange != null ? absoluteTimeRange.toBuilder() : null;
        }

        public final void setAbsoluteTimeRange(AbsoluteTimeRange.BuilderImpl absoluteTimeRange) {
            this.absoluteTimeRange = absoluteTimeRange != null ? absoluteTimeRange.build() : null;
        }

        @Override
        public final Builder absoluteTimeRange(AbsoluteTimeRange absoluteTimeRange) {
            this.absoluteTimeRange = absoluteTimeRange;
            return this;
        }

        public final RelativeTimeRange.Builder getRelativeTimeRange() {
            return relativeTimeRange != null ? relativeTimeRange.toBuilder() : null;
        }

        public final void setRelativeTimeRange(RelativeTimeRange.BuilderImpl relativeTimeRange) {
            this.relativeTimeRange = relativeTimeRange != null ? relativeTimeRange.build() : null;
        }

        @Override
        public final Builder relativeTimeRange(RelativeTimeRange relativeTimeRange) {
            this.relativeTimeRange = relativeTimeRange;
            return this;
        }

        public final String getParticipantRole() {
            return participantRole;
        }

        public final void setParticipantRole(String participantRole) {
            this.participantRole = participantRole;
        }

        @Override
        public final Builder participantRole(String participantRole) {
            this.participantRole = participantRole;
            return this;
        }

        @Override
        public final Builder participantRole(ParticipantRole participantRole) {
            this.participantRole(participantRole == null ? null : participantRole.toString());
            return this;
        }

        public final Boolean getNegate() {
            return negate;
        }

        public final void setNegate(Boolean negate) {
            this.negate = negate;
        }

        @Override
        public final Builder negate(Boolean negate) {
            this.negate = negate;
            return this;
        }

        @Override
        public SentimentFilter build() {
            return new SentimentFilter(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
