/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribe.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides detailed information about a specific transcription job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TranscriptionJobSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<TranscriptionJobSummary.Builder, TranscriptionJobSummary> {
    private static final SdkField<String> TRANSCRIPTION_JOB_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TranscriptionJobName").getter(getter(TranscriptionJobSummary::transcriptionJobName))
            .setter(setter(Builder::transcriptionJobName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TranscriptionJobName").build())
            .build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(TranscriptionJobSummary::creationTime))
            .setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime").getter(getter(TranscriptionJobSummary::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> COMPLETION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CompletionTime").getter(getter(TranscriptionJobSummary::completionTime))
            .setter(setter(Builder::completionTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletionTime").build()).build();

    private static final SdkField<String> LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LanguageCode").getter(getter(TranscriptionJobSummary::languageCodeAsString))
            .setter(setter(Builder::languageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageCode").build()).build();

    private static final SdkField<String> TRANSCRIPTION_JOB_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TranscriptionJobStatus").getter(getter(TranscriptionJobSummary::transcriptionJobStatusAsString))
            .setter(setter(Builder::transcriptionJobStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TranscriptionJobStatus").build())
            .build();

    private static final SdkField<String> FAILURE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FailureReason").getter(getter(TranscriptionJobSummary::failureReason))
            .setter(setter(Builder::failureReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureReason").build()).build();

    private static final SdkField<String> OUTPUT_LOCATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OutputLocationType").getter(getter(TranscriptionJobSummary::outputLocationTypeAsString))
            .setter(setter(Builder::outputLocationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputLocationType").build())
            .build();

    private static final SdkField<ContentRedaction> CONTENT_REDACTION_FIELD = SdkField
            .<ContentRedaction> builder(MarshallingType.SDK_POJO).memberName("ContentRedaction")
            .getter(getter(TranscriptionJobSummary::contentRedaction)).setter(setter(Builder::contentRedaction))
            .constructor(ContentRedaction::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContentRedaction").build()).build();

    private static final SdkField<ModelSettings> MODEL_SETTINGS_FIELD = SdkField
            .<ModelSettings> builder(MarshallingType.SDK_POJO).memberName("ModelSettings")
            .getter(getter(TranscriptionJobSummary::modelSettings)).setter(setter(Builder::modelSettings))
            .constructor(ModelSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelSettings").build()).build();

    private static final SdkField<Boolean> IDENTIFY_LANGUAGE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IdentifyLanguage").getter(getter(TranscriptionJobSummary::identifyLanguage))
            .setter(setter(Builder::identifyLanguage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdentifyLanguage").build()).build();

    private static final SdkField<Boolean> IDENTIFY_MULTIPLE_LANGUAGES_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN).memberName("IdentifyMultipleLanguages")
            .getter(getter(TranscriptionJobSummary::identifyMultipleLanguages))
            .setter(setter(Builder::identifyMultipleLanguages))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdentifyMultipleLanguages").build())
            .build();

    private static final SdkField<Float> IDENTIFIED_LANGUAGE_SCORE_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("IdentifiedLanguageScore").getter(getter(TranscriptionJobSummary::identifiedLanguageScore))
            .setter(setter(Builder::identifiedLanguageScore))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdentifiedLanguageScore").build())
            .build();

    private static final SdkField<List<LanguageCodeItem>> LANGUAGE_CODES_FIELD = SdkField
            .<List<LanguageCodeItem>> builder(MarshallingType.LIST)
            .memberName("LanguageCodes")
            .getter(getter(TranscriptionJobSummary::languageCodes))
            .setter(setter(Builder::languageCodes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageCodes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LanguageCodeItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(LanguageCodeItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ToxicityDetectionSettings>> TOXICITY_DETECTION_FIELD = SdkField
            .<List<ToxicityDetectionSettings>> builder(MarshallingType.LIST)
            .memberName("ToxicityDetection")
            .getter(getter(TranscriptionJobSummary::toxicityDetection))
            .setter(setter(Builder::toxicityDetection))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ToxicityDetection").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ToxicityDetectionSettings> builder(MarshallingType.SDK_POJO)
                                            .constructor(ToxicityDetectionSettings::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TRANSCRIPTION_JOB_NAME_FIELD,
            CREATION_TIME_FIELD, START_TIME_FIELD, COMPLETION_TIME_FIELD, LANGUAGE_CODE_FIELD, TRANSCRIPTION_JOB_STATUS_FIELD,
            FAILURE_REASON_FIELD, OUTPUT_LOCATION_TYPE_FIELD, CONTENT_REDACTION_FIELD, MODEL_SETTINGS_FIELD,
            IDENTIFY_LANGUAGE_FIELD, IDENTIFY_MULTIPLE_LANGUAGES_FIELD, IDENTIFIED_LANGUAGE_SCORE_FIELD, LANGUAGE_CODES_FIELD,
            TOXICITY_DETECTION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String transcriptionJobName;

    private final Instant creationTime;

    private final Instant startTime;

    private final Instant completionTime;

    private final String languageCode;

    private final String transcriptionJobStatus;

    private final String failureReason;

    private final String outputLocationType;

    private final ContentRedaction contentRedaction;

    private final ModelSettings modelSettings;

    private final Boolean identifyLanguage;

    private final Boolean identifyMultipleLanguages;

    private final Float identifiedLanguageScore;

    private final List<LanguageCodeItem> languageCodes;

    private final List<ToxicityDetectionSettings> toxicityDetection;

    private TranscriptionJobSummary(BuilderImpl builder) {
        this.transcriptionJobName = builder.transcriptionJobName;
        this.creationTime = builder.creationTime;
        this.startTime = builder.startTime;
        this.completionTime = builder.completionTime;
        this.languageCode = builder.languageCode;
        this.transcriptionJobStatus = builder.transcriptionJobStatus;
        this.failureReason = builder.failureReason;
        this.outputLocationType = builder.outputLocationType;
        this.contentRedaction = builder.contentRedaction;
        this.modelSettings = builder.modelSettings;
        this.identifyLanguage = builder.identifyLanguage;
        this.identifyMultipleLanguages = builder.identifyMultipleLanguages;
        this.identifiedLanguageScore = builder.identifiedLanguageScore;
        this.languageCodes = builder.languageCodes;
        this.toxicityDetection = builder.toxicityDetection;
    }

    /**
     * <p>
     * The name of the transcription job. Job names are case sensitive and must be unique within an Amazon Web Services
     * account.
     * </p>
     * 
     * @return The name of the transcription job. Job names are case sensitive and must be unique within an Amazon Web
     *         Services account.
     */
    public final String transcriptionJobName() {
        return transcriptionJobName;
    }

    /**
     * <p>
     * The date and time the specified transcription job request was made.
     * </p>
     * <p>
     * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     * <code>2022-05-04T12:32:58.761000-07:00</code> represents a transcription job that started processing at 12:32 PM
     * UTC-7 on May 4, 2022.
     * </p>
     * 
     * @return The date and time the specified transcription job request was made.</p>
     *         <p>
     *         Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     *         <code>2022-05-04T12:32:58.761000-07:00</code> represents a transcription job that started processing at
     *         12:32 PM UTC-7 on May 4, 2022.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The date and time your transcription job began processing.
     * </p>
     * <p>
     * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     * <code>2022-05-04T12:32:58.789000-07:00</code> represents a transcription job that started processing at 12:32 PM
     * UTC-7 on May 4, 2022.
     * </p>
     * 
     * @return The date and time your transcription job began processing.</p>
     *         <p>
     *         Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     *         <code>2022-05-04T12:32:58.789000-07:00</code> represents a transcription job that started processing at
     *         12:32 PM UTC-7 on May 4, 2022.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The date and time the specified transcription job finished processing.
     * </p>
     * <p>
     * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     * <code>2022-05-04T12:33:13.922000-07:00</code> represents a transcription job that started processing at 12:33 PM
     * UTC-7 on May 4, 2022.
     * </p>
     * 
     * @return The date and time the specified transcription job finished processing.</p>
     *         <p>
     *         Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     *         <code>2022-05-04T12:33:13.922000-07:00</code> represents a transcription job that started processing at
     *         12:33 PM UTC-7 on May 4, 2022.
     */
    public final Instant completionTime() {
        return completionTime;
    }

    /**
     * <p>
     * The language code used to create your transcription.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return The language code used to create your transcription.
     * @see LanguageCode
     */
    public final LanguageCode languageCode() {
        return LanguageCode.fromValue(languageCode);
    }

    /**
     * <p>
     * The language code used to create your transcription.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return The language code used to create your transcription.
     * @see LanguageCode
     */
    public final String languageCodeAsString() {
        return languageCode;
    }

    /**
     * <p>
     * Provides the status of your transcription job.
     * </p>
     * <p>
     * If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
     * specified in <code>TranscriptFileUri</code> (or <code>RedactedTranscriptFileUri</code>, if you requested
     * transcript redaction). If the status is <code>FAILED</code>, <code>FailureReason</code> provides details on why
     * your transcription job failed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #transcriptionJobStatus} will return {@link TranscriptionJobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #transcriptionJobStatusAsString}.
     * </p>
     * 
     * @return Provides the status of your transcription job.</p>
     *         <p>
     *         If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
     *         specified in <code>TranscriptFileUri</code> (or <code>RedactedTranscriptFileUri</code>, if you requested
     *         transcript redaction). If the status is <code>FAILED</code>, <code>FailureReason</code> provides details
     *         on why your transcription job failed.
     * @see TranscriptionJobStatus
     */
    public final TranscriptionJobStatus transcriptionJobStatus() {
        return TranscriptionJobStatus.fromValue(transcriptionJobStatus);
    }

    /**
     * <p>
     * Provides the status of your transcription job.
     * </p>
     * <p>
     * If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
     * specified in <code>TranscriptFileUri</code> (or <code>RedactedTranscriptFileUri</code>, if you requested
     * transcript redaction). If the status is <code>FAILED</code>, <code>FailureReason</code> provides details on why
     * your transcription job failed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #transcriptionJobStatus} will return {@link TranscriptionJobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #transcriptionJobStatusAsString}.
     * </p>
     * 
     * @return Provides the status of your transcription job.</p>
     *         <p>
     *         If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
     *         specified in <code>TranscriptFileUri</code> (or <code>RedactedTranscriptFileUri</code>, if you requested
     *         transcript redaction). If the status is <code>FAILED</code>, <code>FailureReason</code> provides details
     *         on why your transcription job failed.
     * @see TranscriptionJobStatus
     */
    public final String transcriptionJobStatusAsString() {
        return transcriptionJobStatus;
    }

    /**
     * <p>
     * If <code>TranscriptionJobStatus</code> is <code>FAILED</code>, <code>FailureReason</code> contains information
     * about why the transcription job failed. See also: <a
     * href="https://docs.aws.amazon.com/transcribe/latest/APIReference/CommonErrors.html">Common Errors</a>.
     * </p>
     * 
     * @return If <code>TranscriptionJobStatus</code> is <code>FAILED</code>, <code>FailureReason</code> contains
     *         information about why the transcription job failed. See also: <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/APIReference/CommonErrors.html">Common Errors</a>.
     */
    public final String failureReason() {
        return failureReason;
    }

    /**
     * <p>
     * Indicates where the specified transcription output is stored.
     * </p>
     * <p>
     * If the value is <code>CUSTOMER_BUCKET</code>, the location is the Amazon S3 bucket you specified using the
     * <code>OutputBucketName</code> parameter in your request. If you also included <code>OutputKey</code> in your
     * request, your output is located in the path you specified in your request.
     * </p>
     * <p>
     * If the value is <code>SERVICE_BUCKET</code>, the location is a service-managed Amazon S3 bucket. To access a
     * transcript stored in a service-managed bucket, use the URI shown in the <code>TranscriptFileUri</code> or
     * <code>RedactedTranscriptFileUri</code> field.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #outputLocationType} will return {@link OutputLocationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #outputLocationTypeAsString}.
     * </p>
     * 
     * @return Indicates where the specified transcription output is stored.</p>
     *         <p>
     *         If the value is <code>CUSTOMER_BUCKET</code>, the location is the Amazon S3 bucket you specified using
     *         the <code>OutputBucketName</code> parameter in your request. If you also included <code>OutputKey</code>
     *         in your request, your output is located in the path you specified in your request.
     *         </p>
     *         <p>
     *         If the value is <code>SERVICE_BUCKET</code>, the location is a service-managed Amazon S3 bucket. To
     *         access a transcript stored in a service-managed bucket, use the URI shown in the
     *         <code>TranscriptFileUri</code> or <code>RedactedTranscriptFileUri</code> field.
     * @see OutputLocationType
     */
    public final OutputLocationType outputLocationType() {
        return OutputLocationType.fromValue(outputLocationType);
    }

    /**
     * <p>
     * Indicates where the specified transcription output is stored.
     * </p>
     * <p>
     * If the value is <code>CUSTOMER_BUCKET</code>, the location is the Amazon S3 bucket you specified using the
     * <code>OutputBucketName</code> parameter in your request. If you also included <code>OutputKey</code> in your
     * request, your output is located in the path you specified in your request.
     * </p>
     * <p>
     * If the value is <code>SERVICE_BUCKET</code>, the location is a service-managed Amazon S3 bucket. To access a
     * transcript stored in a service-managed bucket, use the URI shown in the <code>TranscriptFileUri</code> or
     * <code>RedactedTranscriptFileUri</code> field.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #outputLocationType} will return {@link OutputLocationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #outputLocationTypeAsString}.
     * </p>
     * 
     * @return Indicates where the specified transcription output is stored.</p>
     *         <p>
     *         If the value is <code>CUSTOMER_BUCKET</code>, the location is the Amazon S3 bucket you specified using
     *         the <code>OutputBucketName</code> parameter in your request. If you also included <code>OutputKey</code>
     *         in your request, your output is located in the path you specified in your request.
     *         </p>
     *         <p>
     *         If the value is <code>SERVICE_BUCKET</code>, the location is a service-managed Amazon S3 bucket. To
     *         access a transcript stored in a service-managed bucket, use the URI shown in the
     *         <code>TranscriptFileUri</code> or <code>RedactedTranscriptFileUri</code> field.
     * @see OutputLocationType
     */
    public final String outputLocationTypeAsString() {
        return outputLocationType;
    }

    /**
     * <p>
     * The content redaction settings of the transcription job.
     * </p>
     * 
     * @return The content redaction settings of the transcription job.
     */
    public final ContentRedaction contentRedaction() {
        return contentRedaction;
    }

    /**
     * Returns the value of the ModelSettings property for this object.
     * 
     * @return The value of the ModelSettings property for this object.
     */
    public final ModelSettings modelSettings() {
        return modelSettings;
    }

    /**
     * <p>
     * Indicates whether automatic language identification was enabled (<code>TRUE</code>) for the specified
     * transcription job.
     * </p>
     * 
     * @return Indicates whether automatic language identification was enabled (<code>TRUE</code>) for the specified
     *         transcription job.
     */
    public final Boolean identifyLanguage() {
        return identifyLanguage;
    }

    /**
     * <p>
     * Indicates whether automatic multi-language identification was enabled (<code>TRUE</code>) for the specified
     * transcription job.
     * </p>
     * 
     * @return Indicates whether automatic multi-language identification was enabled (<code>TRUE</code>) for the
     *         specified transcription job.
     */
    public final Boolean identifyMultipleLanguages() {
        return identifyMultipleLanguages;
    }

    /**
     * <p>
     * The confidence score associated with the language identified in your media file.
     * </p>
     * <p>
     * Confidence scores are values between 0 and 1; a larger value indicates a higher probability that the identified
     * language correctly matches the language spoken in your media.
     * </p>
     * 
     * @return The confidence score associated with the language identified in your media file.</p>
     *         <p>
     *         Confidence scores are values between 0 and 1; a larger value indicates a higher probability that the
     *         identified language correctly matches the language spoken in your media.
     */
    public final Float identifiedLanguageScore() {
        return identifiedLanguageScore;
    }

    /**
     * For responses, this returns true if the service returned a value for the LanguageCodes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLanguageCodes() {
        return languageCodes != null && !(languageCodes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The language codes used to create your transcription job. This parameter is used with multi-language
     * identification. For single-language identification, the singular version of this parameter,
     * <code>LanguageCode</code>, is present.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLanguageCodes} method.
     * </p>
     * 
     * @return The language codes used to create your transcription job. This parameter is used with multi-language
     *         identification. For single-language identification, the singular version of this parameter,
     *         <code>LanguageCode</code>, is present.
     */
    public final List<LanguageCodeItem> languageCodes() {
        return languageCodes;
    }

    /**
     * For responses, this returns true if the service returned a value for the ToxicityDetection property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasToxicityDetection() {
        return toxicityDetection != null && !(toxicityDetection instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Indicates whether toxicity detection was enabled for the specified transcription job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasToxicityDetection} method.
     * </p>
     * 
     * @return Indicates whether toxicity detection was enabled for the specified transcription job.
     */
    public final List<ToxicityDetectionSettings> toxicityDetection() {
        return toxicityDetection;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(transcriptionJobName());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(completionTime());
        hashCode = 31 * hashCode + Objects.hashCode(languageCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(transcriptionJobStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(failureReason());
        hashCode = 31 * hashCode + Objects.hashCode(outputLocationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(contentRedaction());
        hashCode = 31 * hashCode + Objects.hashCode(modelSettings());
        hashCode = 31 * hashCode + Objects.hashCode(identifyLanguage());
        hashCode = 31 * hashCode + Objects.hashCode(identifyMultipleLanguages());
        hashCode = 31 * hashCode + Objects.hashCode(identifiedLanguageScore());
        hashCode = 31 * hashCode + Objects.hashCode(hasLanguageCodes() ? languageCodes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasToxicityDetection() ? toxicityDetection() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TranscriptionJobSummary)) {
            return false;
        }
        TranscriptionJobSummary other = (TranscriptionJobSummary) obj;
        return Objects.equals(transcriptionJobName(), other.transcriptionJobName())
                && Objects.equals(creationTime(), other.creationTime()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(completionTime(), other.completionTime())
                && Objects.equals(languageCodeAsString(), other.languageCodeAsString())
                && Objects.equals(transcriptionJobStatusAsString(), other.transcriptionJobStatusAsString())
                && Objects.equals(failureReason(), other.failureReason())
                && Objects.equals(outputLocationTypeAsString(), other.outputLocationTypeAsString())
                && Objects.equals(contentRedaction(), other.contentRedaction())
                && Objects.equals(modelSettings(), other.modelSettings())
                && Objects.equals(identifyLanguage(), other.identifyLanguage())
                && Objects.equals(identifyMultipleLanguages(), other.identifyMultipleLanguages())
                && Objects.equals(identifiedLanguageScore(), other.identifiedLanguageScore())
                && hasLanguageCodes() == other.hasLanguageCodes() && Objects.equals(languageCodes(), other.languageCodes())
                && hasToxicityDetection() == other.hasToxicityDetection()
                && Objects.equals(toxicityDetection(), other.toxicityDetection());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TranscriptionJobSummary").add("TranscriptionJobName", transcriptionJobName())
                .add("CreationTime", creationTime()).add("StartTime", startTime()).add("CompletionTime", completionTime())
                .add("LanguageCode", languageCodeAsString()).add("TranscriptionJobStatus", transcriptionJobStatusAsString())
                .add("FailureReason", failureReason()).add("OutputLocationType", outputLocationTypeAsString())
                .add("ContentRedaction", contentRedaction()).add("ModelSettings", modelSettings())
                .add("IdentifyLanguage", identifyLanguage()).add("IdentifyMultipleLanguages", identifyMultipleLanguages())
                .add("IdentifiedLanguageScore", identifiedLanguageScore())
                .add("LanguageCodes", hasLanguageCodes() ? languageCodes() : null)
                .add("ToxicityDetection", hasToxicityDetection() ? toxicityDetection() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TranscriptionJobName":
            return Optional.ofNullable(clazz.cast(transcriptionJobName()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "CompletionTime":
            return Optional.ofNullable(clazz.cast(completionTime()));
        case "LanguageCode":
            return Optional.ofNullable(clazz.cast(languageCodeAsString()));
        case "TranscriptionJobStatus":
            return Optional.ofNullable(clazz.cast(transcriptionJobStatusAsString()));
        case "FailureReason":
            return Optional.ofNullable(clazz.cast(failureReason()));
        case "OutputLocationType":
            return Optional.ofNullable(clazz.cast(outputLocationTypeAsString()));
        case "ContentRedaction":
            return Optional.ofNullable(clazz.cast(contentRedaction()));
        case "ModelSettings":
            return Optional.ofNullable(clazz.cast(modelSettings()));
        case "IdentifyLanguage":
            return Optional.ofNullable(clazz.cast(identifyLanguage()));
        case "IdentifyMultipleLanguages":
            return Optional.ofNullable(clazz.cast(identifyMultipleLanguages()));
        case "IdentifiedLanguageScore":
            return Optional.ofNullable(clazz.cast(identifiedLanguageScore()));
        case "LanguageCodes":
            return Optional.ofNullable(clazz.cast(languageCodes()));
        case "ToxicityDetection":
            return Optional.ofNullable(clazz.cast(toxicityDetection()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("TranscriptionJobName", TRANSCRIPTION_JOB_NAME_FIELD);
        map.put("CreationTime", CREATION_TIME_FIELD);
        map.put("StartTime", START_TIME_FIELD);
        map.put("CompletionTime", COMPLETION_TIME_FIELD);
        map.put("LanguageCode", LANGUAGE_CODE_FIELD);
        map.put("TranscriptionJobStatus", TRANSCRIPTION_JOB_STATUS_FIELD);
        map.put("FailureReason", FAILURE_REASON_FIELD);
        map.put("OutputLocationType", OUTPUT_LOCATION_TYPE_FIELD);
        map.put("ContentRedaction", CONTENT_REDACTION_FIELD);
        map.put("ModelSettings", MODEL_SETTINGS_FIELD);
        map.put("IdentifyLanguage", IDENTIFY_LANGUAGE_FIELD);
        map.put("IdentifyMultipleLanguages", IDENTIFY_MULTIPLE_LANGUAGES_FIELD);
        map.put("IdentifiedLanguageScore", IDENTIFIED_LANGUAGE_SCORE_FIELD);
        map.put("LanguageCodes", LANGUAGE_CODES_FIELD);
        map.put("ToxicityDetection", TOXICITY_DETECTION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<TranscriptionJobSummary, T> g) {
        return obj -> g.apply((TranscriptionJobSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TranscriptionJobSummary> {
        /**
         * <p>
         * The name of the transcription job. Job names are case sensitive and must be unique within an Amazon Web
         * Services account.
         * </p>
         * 
         * @param transcriptionJobName
         *        The name of the transcription job. Job names are case sensitive and must be unique within an Amazon
         *        Web Services account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transcriptionJobName(String transcriptionJobName);

        /**
         * <p>
         * The date and time the specified transcription job request was made.
         * </p>
         * <p>
         * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         * <code>2022-05-04T12:32:58.761000-07:00</code> represents a transcription job that started processing at 12:32
         * PM UTC-7 on May 4, 2022.
         * </p>
         * 
         * @param creationTime
         *        The date and time the specified transcription job request was made.</p>
         *        <p>
         *        Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         *        <code>2022-05-04T12:32:58.761000-07:00</code> represents a transcription job that started processing
         *        at 12:32 PM UTC-7 on May 4, 2022.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The date and time your transcription job began processing.
         * </p>
         * <p>
         * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         * <code>2022-05-04T12:32:58.789000-07:00</code> represents a transcription job that started processing at 12:32
         * PM UTC-7 on May 4, 2022.
         * </p>
         * 
         * @param startTime
         *        The date and time your transcription job began processing.</p>
         *        <p>
         *        Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         *        <code>2022-05-04T12:32:58.789000-07:00</code> represents a transcription job that started processing
         *        at 12:32 PM UTC-7 on May 4, 2022.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The date and time the specified transcription job finished processing.
         * </p>
         * <p>
         * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         * <code>2022-05-04T12:33:13.922000-07:00</code> represents a transcription job that started processing at 12:33
         * PM UTC-7 on May 4, 2022.
         * </p>
         * 
         * @param completionTime
         *        The date and time the specified transcription job finished processing.</p>
         *        <p>
         *        Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         *        <code>2022-05-04T12:33:13.922000-07:00</code> represents a transcription job that started processing
         *        at 12:33 PM UTC-7 on May 4, 2022.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completionTime(Instant completionTime);

        /**
         * <p>
         * The language code used to create your transcription.
         * </p>
         * 
         * @param languageCode
         *        The language code used to create your transcription.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(String languageCode);

        /**
         * <p>
         * The language code used to create your transcription.
         * </p>
         * 
         * @param languageCode
         *        The language code used to create your transcription.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(LanguageCode languageCode);

        /**
         * <p>
         * Provides the status of your transcription job.
         * </p>
         * <p>
         * If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
         * specified in <code>TranscriptFileUri</code> (or <code>RedactedTranscriptFileUri</code>, if you requested
         * transcript redaction). If the status is <code>FAILED</code>, <code>FailureReason</code> provides details on
         * why your transcription job failed.
         * </p>
         * 
         * @param transcriptionJobStatus
         *        Provides the status of your transcription job.</p>
         *        <p>
         *        If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the
         *        location specified in <code>TranscriptFileUri</code> (or <code>RedactedTranscriptFileUri</code>, if
         *        you requested transcript redaction). If the status is <code>FAILED</code>, <code>FailureReason</code>
         *        provides details on why your transcription job failed.
         * @see TranscriptionJobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscriptionJobStatus
         */
        Builder transcriptionJobStatus(String transcriptionJobStatus);

        /**
         * <p>
         * Provides the status of your transcription job.
         * </p>
         * <p>
         * If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the location
         * specified in <code>TranscriptFileUri</code> (or <code>RedactedTranscriptFileUri</code>, if you requested
         * transcript redaction). If the status is <code>FAILED</code>, <code>FailureReason</code> provides details on
         * why your transcription job failed.
         * </p>
         * 
         * @param transcriptionJobStatus
         *        Provides the status of your transcription job.</p>
         *        <p>
         *        If the status is <code>COMPLETED</code>, the job is finished and you can find the results at the
         *        location specified in <code>TranscriptFileUri</code> (or <code>RedactedTranscriptFileUri</code>, if
         *        you requested transcript redaction). If the status is <code>FAILED</code>, <code>FailureReason</code>
         *        provides details on why your transcription job failed.
         * @see TranscriptionJobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscriptionJobStatus
         */
        Builder transcriptionJobStatus(TranscriptionJobStatus transcriptionJobStatus);

        /**
         * <p>
         * If <code>TranscriptionJobStatus</code> is <code>FAILED</code>, <code>FailureReason</code> contains
         * information about why the transcription job failed. See also: <a
         * href="https://docs.aws.amazon.com/transcribe/latest/APIReference/CommonErrors.html">Common Errors</a>.
         * </p>
         * 
         * @param failureReason
         *        If <code>TranscriptionJobStatus</code> is <code>FAILED</code>, <code>FailureReason</code> contains
         *        information about why the transcription job failed. See also: <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/APIReference/CommonErrors.html">Common Errors</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReason(String failureReason);

        /**
         * <p>
         * Indicates where the specified transcription output is stored.
         * </p>
         * <p>
         * If the value is <code>CUSTOMER_BUCKET</code>, the location is the Amazon S3 bucket you specified using the
         * <code>OutputBucketName</code> parameter in your request. If you also included <code>OutputKey</code> in your
         * request, your output is located in the path you specified in your request.
         * </p>
         * <p>
         * If the value is <code>SERVICE_BUCKET</code>, the location is a service-managed Amazon S3 bucket. To access a
         * transcript stored in a service-managed bucket, use the URI shown in the <code>TranscriptFileUri</code> or
         * <code>RedactedTranscriptFileUri</code> field.
         * </p>
         * 
         * @param outputLocationType
         *        Indicates where the specified transcription output is stored.</p>
         *        <p>
         *        If the value is <code>CUSTOMER_BUCKET</code>, the location is the Amazon S3 bucket you specified using
         *        the <code>OutputBucketName</code> parameter in your request. If you also included
         *        <code>OutputKey</code> in your request, your output is located in the path you specified in your
         *        request.
         *        </p>
         *        <p>
         *        If the value is <code>SERVICE_BUCKET</code>, the location is a service-managed Amazon S3 bucket. To
         *        access a transcript stored in a service-managed bucket, use the URI shown in the
         *        <code>TranscriptFileUri</code> or <code>RedactedTranscriptFileUri</code> field.
         * @see OutputLocationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OutputLocationType
         */
        Builder outputLocationType(String outputLocationType);

        /**
         * <p>
         * Indicates where the specified transcription output is stored.
         * </p>
         * <p>
         * If the value is <code>CUSTOMER_BUCKET</code>, the location is the Amazon S3 bucket you specified using the
         * <code>OutputBucketName</code> parameter in your request. If you also included <code>OutputKey</code> in your
         * request, your output is located in the path you specified in your request.
         * </p>
         * <p>
         * If the value is <code>SERVICE_BUCKET</code>, the location is a service-managed Amazon S3 bucket. To access a
         * transcript stored in a service-managed bucket, use the URI shown in the <code>TranscriptFileUri</code> or
         * <code>RedactedTranscriptFileUri</code> field.
         * </p>
         * 
         * @param outputLocationType
         *        Indicates where the specified transcription output is stored.</p>
         *        <p>
         *        If the value is <code>CUSTOMER_BUCKET</code>, the location is the Amazon S3 bucket you specified using
         *        the <code>OutputBucketName</code> parameter in your request. If you also included
         *        <code>OutputKey</code> in your request, your output is located in the path you specified in your
         *        request.
         *        </p>
         *        <p>
         *        If the value is <code>SERVICE_BUCKET</code>, the location is a service-managed Amazon S3 bucket. To
         *        access a transcript stored in a service-managed bucket, use the URI shown in the
         *        <code>TranscriptFileUri</code> or <code>RedactedTranscriptFileUri</code> field.
         * @see OutputLocationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OutputLocationType
         */
        Builder outputLocationType(OutputLocationType outputLocationType);

        /**
         * <p>
         * The content redaction settings of the transcription job.
         * </p>
         * 
         * @param contentRedaction
         *        The content redaction settings of the transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contentRedaction(ContentRedaction contentRedaction);

        /**
         * <p>
         * The content redaction settings of the transcription job.
         * </p>
         * This is a convenience method that creates an instance of the {@link ContentRedaction.Builder} avoiding the
         * need to create one manually via {@link ContentRedaction#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ContentRedaction.Builder#build()} is called immediately and its
         * result is passed to {@link #contentRedaction(ContentRedaction)}.
         * 
         * @param contentRedaction
         *        a consumer that will call methods on {@link ContentRedaction.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #contentRedaction(ContentRedaction)
         */
        default Builder contentRedaction(Consumer<ContentRedaction.Builder> contentRedaction) {
            return contentRedaction(ContentRedaction.builder().applyMutation(contentRedaction).build());
        }

        /**
         * Sets the value of the ModelSettings property for this object.
         *
         * @param modelSettings
         *        The new value for the ModelSettings property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelSettings(ModelSettings modelSettings);

        /**
         * Sets the value of the ModelSettings property for this object.
         *
         * This is a convenience method that creates an instance of the {@link ModelSettings.Builder} avoiding the need
         * to create one manually via {@link ModelSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ModelSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #modelSettings(ModelSettings)}.
         * 
         * @param modelSettings
         *        a consumer that will call methods on {@link ModelSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #modelSettings(ModelSettings)
         */
        default Builder modelSettings(Consumer<ModelSettings.Builder> modelSettings) {
            return modelSettings(ModelSettings.builder().applyMutation(modelSettings).build());
        }

        /**
         * <p>
         * Indicates whether automatic language identification was enabled (<code>TRUE</code>) for the specified
         * transcription job.
         * </p>
         * 
         * @param identifyLanguage
         *        Indicates whether automatic language identification was enabled (<code>TRUE</code>) for the specified
         *        transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identifyLanguage(Boolean identifyLanguage);

        /**
         * <p>
         * Indicates whether automatic multi-language identification was enabled (<code>TRUE</code>) for the specified
         * transcription job.
         * </p>
         * 
         * @param identifyMultipleLanguages
         *        Indicates whether automatic multi-language identification was enabled (<code>TRUE</code>) for the
         *        specified transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identifyMultipleLanguages(Boolean identifyMultipleLanguages);

        /**
         * <p>
         * The confidence score associated with the language identified in your media file.
         * </p>
         * <p>
         * Confidence scores are values between 0 and 1; a larger value indicates a higher probability that the
         * identified language correctly matches the language spoken in your media.
         * </p>
         * 
         * @param identifiedLanguageScore
         *        The confidence score associated with the language identified in your media file.</p>
         *        <p>
         *        Confidence scores are values between 0 and 1; a larger value indicates a higher probability that the
         *        identified language correctly matches the language spoken in your media.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identifiedLanguageScore(Float identifiedLanguageScore);

        /**
         * <p>
         * The language codes used to create your transcription job. This parameter is used with multi-language
         * identification. For single-language identification, the singular version of this parameter,
         * <code>LanguageCode</code>, is present.
         * </p>
         * 
         * @param languageCodes
         *        The language codes used to create your transcription job. This parameter is used with multi-language
         *        identification. For single-language identification, the singular version of this parameter,
         *        <code>LanguageCode</code>, is present.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageCodes(Collection<LanguageCodeItem> languageCodes);

        /**
         * <p>
         * The language codes used to create your transcription job. This parameter is used with multi-language
         * identification. For single-language identification, the singular version of this parameter,
         * <code>LanguageCode</code>, is present.
         * </p>
         * 
         * @param languageCodes
         *        The language codes used to create your transcription job. This parameter is used with multi-language
         *        identification. For single-language identification, the singular version of this parameter,
         *        <code>LanguageCode</code>, is present.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageCodes(LanguageCodeItem... languageCodes);

        /**
         * <p>
         * The language codes used to create your transcription job. This parameter is used with multi-language
         * identification. For single-language identification, the singular version of this parameter,
         * <code>LanguageCode</code>, is present.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.transcribe.model.LanguageCodeItem.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.transcribe.model.LanguageCodeItem#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.transcribe.model.LanguageCodeItem.Builder#build()} is called
         * immediately and its result is passed to {@link #languageCodes(List<LanguageCodeItem>)}.
         * 
         * @param languageCodes
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.transcribe.model.LanguageCodeItem.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #languageCodes(java.util.Collection<LanguageCodeItem>)
         */
        Builder languageCodes(Consumer<LanguageCodeItem.Builder>... languageCodes);

        /**
         * <p>
         * Indicates whether toxicity detection was enabled for the specified transcription job.
         * </p>
         * 
         * @param toxicityDetection
         *        Indicates whether toxicity detection was enabled for the specified transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder toxicityDetection(Collection<ToxicityDetectionSettings> toxicityDetection);

        /**
         * <p>
         * Indicates whether toxicity detection was enabled for the specified transcription job.
         * </p>
         * 
         * @param toxicityDetection
         *        Indicates whether toxicity detection was enabled for the specified transcription job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder toxicityDetection(ToxicityDetectionSettings... toxicityDetection);

        /**
         * <p>
         * Indicates whether toxicity detection was enabled for the specified transcription job.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.transcribe.model.ToxicityDetectionSettings.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.transcribe.model.ToxicityDetectionSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.transcribe.model.ToxicityDetectionSettings.Builder#build()} is called
         * immediately and its result is passed to {@link #toxicityDetection(List<ToxicityDetectionSettings>)}.
         * 
         * @param toxicityDetection
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.transcribe.model.ToxicityDetectionSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #toxicityDetection(java.util.Collection<ToxicityDetectionSettings>)
         */
        Builder toxicityDetection(Consumer<ToxicityDetectionSettings.Builder>... toxicityDetection);
    }

    static final class BuilderImpl implements Builder {
        private String transcriptionJobName;

        private Instant creationTime;

        private Instant startTime;

        private Instant completionTime;

        private String languageCode;

        private String transcriptionJobStatus;

        private String failureReason;

        private String outputLocationType;

        private ContentRedaction contentRedaction;

        private ModelSettings modelSettings;

        private Boolean identifyLanguage;

        private Boolean identifyMultipleLanguages;

        private Float identifiedLanguageScore;

        private List<LanguageCodeItem> languageCodes = DefaultSdkAutoConstructList.getInstance();

        private List<ToxicityDetectionSettings> toxicityDetection = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(TranscriptionJobSummary model) {
            transcriptionJobName(model.transcriptionJobName);
            creationTime(model.creationTime);
            startTime(model.startTime);
            completionTime(model.completionTime);
            languageCode(model.languageCode);
            transcriptionJobStatus(model.transcriptionJobStatus);
            failureReason(model.failureReason);
            outputLocationType(model.outputLocationType);
            contentRedaction(model.contentRedaction);
            modelSettings(model.modelSettings);
            identifyLanguage(model.identifyLanguage);
            identifyMultipleLanguages(model.identifyMultipleLanguages);
            identifiedLanguageScore(model.identifiedLanguageScore);
            languageCodes(model.languageCodes);
            toxicityDetection(model.toxicityDetection);
        }

        public final String getTranscriptionJobName() {
            return transcriptionJobName;
        }

        public final void setTranscriptionJobName(String transcriptionJobName) {
            this.transcriptionJobName = transcriptionJobName;
        }

        @Override
        public final Builder transcriptionJobName(String transcriptionJobName) {
            this.transcriptionJobName = transcriptionJobName;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getCompletionTime() {
            return completionTime;
        }

        public final void setCompletionTime(Instant completionTime) {
            this.completionTime = completionTime;
        }

        @Override
        public final Builder completionTime(Instant completionTime) {
            this.completionTime = completionTime;
            return this;
        }

        public final String getLanguageCode() {
            return languageCode;
        }

        public final void setLanguageCode(String languageCode) {
            this.languageCode = languageCode;
        }

        @Override
        public final Builder languageCode(String languageCode) {
            this.languageCode = languageCode;
            return this;
        }

        @Override
        public final Builder languageCode(LanguageCode languageCode) {
            this.languageCode(languageCode == null ? null : languageCode.toString());
            return this;
        }

        public final String getTranscriptionJobStatus() {
            return transcriptionJobStatus;
        }

        public final void setTranscriptionJobStatus(String transcriptionJobStatus) {
            this.transcriptionJobStatus = transcriptionJobStatus;
        }

        @Override
        public final Builder transcriptionJobStatus(String transcriptionJobStatus) {
            this.transcriptionJobStatus = transcriptionJobStatus;
            return this;
        }

        @Override
        public final Builder transcriptionJobStatus(TranscriptionJobStatus transcriptionJobStatus) {
            this.transcriptionJobStatus(transcriptionJobStatus == null ? null : transcriptionJobStatus.toString());
            return this;
        }

        public final String getFailureReason() {
            return failureReason;
        }

        public final void setFailureReason(String failureReason) {
            this.failureReason = failureReason;
        }

        @Override
        public final Builder failureReason(String failureReason) {
            this.failureReason = failureReason;
            return this;
        }

        public final String getOutputLocationType() {
            return outputLocationType;
        }

        public final void setOutputLocationType(String outputLocationType) {
            this.outputLocationType = outputLocationType;
        }

        @Override
        public final Builder outputLocationType(String outputLocationType) {
            this.outputLocationType = outputLocationType;
            return this;
        }

        @Override
        public final Builder outputLocationType(OutputLocationType outputLocationType) {
            this.outputLocationType(outputLocationType == null ? null : outputLocationType.toString());
            return this;
        }

        public final ContentRedaction.Builder getContentRedaction() {
            return contentRedaction != null ? contentRedaction.toBuilder() : null;
        }

        public final void setContentRedaction(ContentRedaction.BuilderImpl contentRedaction) {
            this.contentRedaction = contentRedaction != null ? contentRedaction.build() : null;
        }

        @Override
        public final Builder contentRedaction(ContentRedaction contentRedaction) {
            this.contentRedaction = contentRedaction;
            return this;
        }

        public final ModelSettings.Builder getModelSettings() {
            return modelSettings != null ? modelSettings.toBuilder() : null;
        }

        public final void setModelSettings(ModelSettings.BuilderImpl modelSettings) {
            this.modelSettings = modelSettings != null ? modelSettings.build() : null;
        }

        @Override
        public final Builder modelSettings(ModelSettings modelSettings) {
            this.modelSettings = modelSettings;
            return this;
        }

        public final Boolean getIdentifyLanguage() {
            return identifyLanguage;
        }

        public final void setIdentifyLanguage(Boolean identifyLanguage) {
            this.identifyLanguage = identifyLanguage;
        }

        @Override
        public final Builder identifyLanguage(Boolean identifyLanguage) {
            this.identifyLanguage = identifyLanguage;
            return this;
        }

        public final Boolean getIdentifyMultipleLanguages() {
            return identifyMultipleLanguages;
        }

        public final void setIdentifyMultipleLanguages(Boolean identifyMultipleLanguages) {
            this.identifyMultipleLanguages = identifyMultipleLanguages;
        }

        @Override
        public final Builder identifyMultipleLanguages(Boolean identifyMultipleLanguages) {
            this.identifyMultipleLanguages = identifyMultipleLanguages;
            return this;
        }

        public final Float getIdentifiedLanguageScore() {
            return identifiedLanguageScore;
        }

        public final void setIdentifiedLanguageScore(Float identifiedLanguageScore) {
            this.identifiedLanguageScore = identifiedLanguageScore;
        }

        @Override
        public final Builder identifiedLanguageScore(Float identifiedLanguageScore) {
            this.identifiedLanguageScore = identifiedLanguageScore;
            return this;
        }

        public final List<LanguageCodeItem.Builder> getLanguageCodes() {
            List<LanguageCodeItem.Builder> result = LanguageCodeListCopier.copyToBuilder(this.languageCodes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setLanguageCodes(Collection<LanguageCodeItem.BuilderImpl> languageCodes) {
            this.languageCodes = LanguageCodeListCopier.copyFromBuilder(languageCodes);
        }

        @Override
        public final Builder languageCodes(Collection<LanguageCodeItem> languageCodes) {
            this.languageCodes = LanguageCodeListCopier.copy(languageCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder languageCodes(LanguageCodeItem... languageCodes) {
            languageCodes(Arrays.asList(languageCodes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder languageCodes(Consumer<LanguageCodeItem.Builder>... languageCodes) {
            languageCodes(Stream.of(languageCodes).map(c -> LanguageCodeItem.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<ToxicityDetectionSettings.Builder> getToxicityDetection() {
            List<ToxicityDetectionSettings.Builder> result = ToxicityDetectionCopier.copyToBuilder(this.toxicityDetection);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setToxicityDetection(Collection<ToxicityDetectionSettings.BuilderImpl> toxicityDetection) {
            this.toxicityDetection = ToxicityDetectionCopier.copyFromBuilder(toxicityDetection);
        }

        @Override
        public final Builder toxicityDetection(Collection<ToxicityDetectionSettings> toxicityDetection) {
            this.toxicityDetection = ToxicityDetectionCopier.copy(toxicityDetection);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder toxicityDetection(ToxicityDetectionSettings... toxicityDetection) {
            toxicityDetection(Arrays.asList(toxicityDetection));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder toxicityDetection(Consumer<ToxicityDetectionSettings.Builder>... toxicityDetection) {
            toxicityDetection(Stream.of(toxicityDetection).map(c -> ToxicityDetectionSettings.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public TranscriptionJobSummary build() {
            return new TranscriptionJobSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
