/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribe.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides additional optional settings for your request, including content redaction, automatic language
 * identification; allows you to apply custom language models, custom vocabulary filters, and custom vocabularies.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CallAnalyticsJobSettings implements SdkPojo, Serializable,
        ToCopyableBuilder<CallAnalyticsJobSettings.Builder, CallAnalyticsJobSettings> {
    private static final SdkField<String> VOCABULARY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VocabularyName").getter(getter(CallAnalyticsJobSettings::vocabularyName))
            .setter(setter(Builder::vocabularyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VocabularyName").build()).build();

    private static final SdkField<String> VOCABULARY_FILTER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VocabularyFilterName").getter(getter(CallAnalyticsJobSettings::vocabularyFilterName))
            .setter(setter(Builder::vocabularyFilterName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VocabularyFilterName").build())
            .build();

    private static final SdkField<String> VOCABULARY_FILTER_METHOD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VocabularyFilterMethod").getter(getter(CallAnalyticsJobSettings::vocabularyFilterMethodAsString))
            .setter(setter(Builder::vocabularyFilterMethod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VocabularyFilterMethod").build())
            .build();

    private static final SdkField<String> LANGUAGE_MODEL_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LanguageModelName").getter(getter(CallAnalyticsJobSettings::languageModelName))
            .setter(setter(Builder::languageModelName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageModelName").build()).build();

    private static final SdkField<ContentRedaction> CONTENT_REDACTION_FIELD = SdkField
            .<ContentRedaction> builder(MarshallingType.SDK_POJO).memberName("ContentRedaction")
            .getter(getter(CallAnalyticsJobSettings::contentRedaction)).setter(setter(Builder::contentRedaction))
            .constructor(ContentRedaction::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContentRedaction").build()).build();

    private static final SdkField<List<String>> LANGUAGE_OPTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("LanguageOptions")
            .getter(getter(CallAnalyticsJobSettings::languageOptionsAsStrings))
            .setter(setter(Builder::languageOptionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageOptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, LanguageIdSettings>> LANGUAGE_ID_SETTINGS_FIELD = SdkField
            .<Map<String, LanguageIdSettings>> builder(MarshallingType.MAP)
            .memberName("LanguageIdSettings")
            .getter(getter(CallAnalyticsJobSettings::languageIdSettingsAsStrings))
            .setter(setter(Builder::languageIdSettingsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageIdSettings").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<LanguageIdSettings> builder(MarshallingType.SDK_POJO)
                                            .constructor(LanguageIdSettings::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Summarization> SUMMARIZATION_FIELD = SdkField.<Summarization> builder(MarshallingType.SDK_POJO)
            .memberName("Summarization").getter(getter(CallAnalyticsJobSettings::summarization))
            .setter(setter(Builder::summarization)).constructor(Summarization::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Summarization").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VOCABULARY_NAME_FIELD,
            VOCABULARY_FILTER_NAME_FIELD, VOCABULARY_FILTER_METHOD_FIELD, LANGUAGE_MODEL_NAME_FIELD, CONTENT_REDACTION_FIELD,
            LANGUAGE_OPTIONS_FIELD, LANGUAGE_ID_SETTINGS_FIELD, SUMMARIZATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String vocabularyName;

    private final String vocabularyFilterName;

    private final String vocabularyFilterMethod;

    private final String languageModelName;

    private final ContentRedaction contentRedaction;

    private final List<String> languageOptions;

    private final Map<String, LanguageIdSettings> languageIdSettings;

    private final Summarization summarization;

    private CallAnalyticsJobSettings(BuilderImpl builder) {
        this.vocabularyName = builder.vocabularyName;
        this.vocabularyFilterName = builder.vocabularyFilterName;
        this.vocabularyFilterMethod = builder.vocabularyFilterMethod;
        this.languageModelName = builder.languageModelName;
        this.contentRedaction = builder.contentRedaction;
        this.languageOptions = builder.languageOptions;
        this.languageIdSettings = builder.languageIdSettings;
        this.summarization = builder.summarization;
    }

    /**
     * <p>
     * The name of the custom vocabulary you want to include in your Call Analytics transcription request. Custom
     * vocabulary names are case sensitive.
     * </p>
     * 
     * @return The name of the custom vocabulary you want to include in your Call Analytics transcription request.
     *         Custom vocabulary names are case sensitive.
     */
    public final String vocabularyName() {
        return vocabularyName;
    }

    /**
     * <p>
     * The name of the custom vocabulary filter you want to include in your Call Analytics transcription request. Custom
     * vocabulary filter names are case sensitive.
     * </p>
     * <p>
     * Note that if you include <code>VocabularyFilterName</code> in your request, you must also include
     * <code>VocabularyFilterMethod</code>.
     * </p>
     * 
     * @return The name of the custom vocabulary filter you want to include in your Call Analytics transcription
     *         request. Custom vocabulary filter names are case sensitive.</p>
     *         <p>
     *         Note that if you include <code>VocabularyFilterName</code> in your request, you must also include
     *         <code>VocabularyFilterMethod</code>.
     */
    public final String vocabularyFilterName() {
        return vocabularyFilterName;
    }

    /**
     * <p>
     * Specify how you want your custom vocabulary filter applied to your transcript.
     * </p>
     * <p>
     * To replace words with <code>***</code>, choose <code>mask</code>.
     * </p>
     * <p>
     * To delete words, choose <code>remove</code>.
     * </p>
     * <p>
     * To flag words without changing them, choose <code>tag</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #vocabularyFilterMethod} will return {@link VocabularyFilterMethod#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #vocabularyFilterMethodAsString}.
     * </p>
     * 
     * @return Specify how you want your custom vocabulary filter applied to your transcript.</p>
     *         <p>
     *         To replace words with <code>***</code>, choose <code>mask</code>.
     *         </p>
     *         <p>
     *         To delete words, choose <code>remove</code>.
     *         </p>
     *         <p>
     *         To flag words without changing them, choose <code>tag</code>.
     * @see VocabularyFilterMethod
     */
    public final VocabularyFilterMethod vocabularyFilterMethod() {
        return VocabularyFilterMethod.fromValue(vocabularyFilterMethod);
    }

    /**
     * <p>
     * Specify how you want your custom vocabulary filter applied to your transcript.
     * </p>
     * <p>
     * To replace words with <code>***</code>, choose <code>mask</code>.
     * </p>
     * <p>
     * To delete words, choose <code>remove</code>.
     * </p>
     * <p>
     * To flag words without changing them, choose <code>tag</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #vocabularyFilterMethod} will return {@link VocabularyFilterMethod#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #vocabularyFilterMethodAsString}.
     * </p>
     * 
     * @return Specify how you want your custom vocabulary filter applied to your transcript.</p>
     *         <p>
     *         To replace words with <code>***</code>, choose <code>mask</code>.
     *         </p>
     *         <p>
     *         To delete words, choose <code>remove</code>.
     *         </p>
     *         <p>
     *         To flag words without changing them, choose <code>tag</code>.
     * @see VocabularyFilterMethod
     */
    public final String vocabularyFilterMethodAsString() {
        return vocabularyFilterMethod;
    }

    /**
     * <p>
     * The name of the custom language model you want to use when processing your Call Analytics job. Note that custom
     * language model names are case sensitive.
     * </p>
     * <p>
     * The language of the specified custom language model must match the language code that you specify in your
     * transcription request. If the languages do not match, the custom language model isn't applied. There are no
     * errors or warnings associated with a language mismatch.
     * </p>
     * 
     * @return The name of the custom language model you want to use when processing your Call Analytics job. Note that
     *         custom language model names are case sensitive.</p>
     *         <p>
     *         The language of the specified custom language model must match the language code that you specify in your
     *         transcription request. If the languages do not match, the custom language model isn't applied. There are
     *         no errors or warnings associated with a language mismatch.
     */
    public final String languageModelName() {
        return languageModelName;
    }

    /**
     * Returns the value of the ContentRedaction property for this object.
     * 
     * @return The value of the ContentRedaction property for this object.
     */
    public final ContentRedaction contentRedaction() {
        return contentRedaction;
    }

    /**
     * <p>
     * You can specify two or more language codes that represent the languages you think may be present in your media.
     * Including more than five is not recommended. If you're unsure what languages are present, do not include this
     * parameter.
     * </p>
     * <p>
     * Including language options can improve the accuracy of language identification.
     * </p>
     * <p>
     * For a list of languages supported with Call Analytics, refer to the <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages</a> table.
     * </p>
     * <p>
     * To transcribe speech in Modern Standard Arabic (<code>ar-SA</code>) in Amazon Web Services GovCloud (US)
     * (US-West, us-gov-west-1), Amazon Web Services GovCloud (US) (US-East, us-gov-east-1), Canada (Calgary) ca-west-1
     * and Africa (Cape Town) af-south-1, your media file must be encoded at a sample rate of 16,000 Hz or higher.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLanguageOptions} method.
     * </p>
     * 
     * @return You can specify two or more language codes that represent the languages you think may be present in your
     *         media. Including more than five is not recommended. If you're unsure what languages are present, do not
     *         include this parameter.</p>
     *         <p>
     *         Including language options can improve the accuracy of language identification.
     *         </p>
     *         <p>
     *         For a list of languages supported with Call Analytics, refer to the <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages</a>
     *         table.
     *         </p>
     *         <p>
     *         To transcribe speech in Modern Standard Arabic (<code>ar-SA</code>) in Amazon Web Services GovCloud (US)
     *         (US-West, us-gov-west-1), Amazon Web Services GovCloud (US) (US-East, us-gov-east-1), Canada (Calgary)
     *         ca-west-1 and Africa (Cape Town) af-south-1, your media file must be encoded at a sample rate of 16,000
     *         Hz or higher.
     */
    public final List<LanguageCode> languageOptions() {
        return LanguageOptionsCopier.copyStringToEnum(languageOptions);
    }

    /**
     * For responses, this returns true if the service returned a value for the LanguageOptions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLanguageOptions() {
        return languageOptions != null && !(languageOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * You can specify two or more language codes that represent the languages you think may be present in your media.
     * Including more than five is not recommended. If you're unsure what languages are present, do not include this
     * parameter.
     * </p>
     * <p>
     * Including language options can improve the accuracy of language identification.
     * </p>
     * <p>
     * For a list of languages supported with Call Analytics, refer to the <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages</a> table.
     * </p>
     * <p>
     * To transcribe speech in Modern Standard Arabic (<code>ar-SA</code>) in Amazon Web Services GovCloud (US)
     * (US-West, us-gov-west-1), Amazon Web Services GovCloud (US) (US-East, us-gov-east-1), Canada (Calgary) ca-west-1
     * and Africa (Cape Town) af-south-1, your media file must be encoded at a sample rate of 16,000 Hz or higher.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLanguageOptions} method.
     * </p>
     * 
     * @return You can specify two or more language codes that represent the languages you think may be present in your
     *         media. Including more than five is not recommended. If you're unsure what languages are present, do not
     *         include this parameter.</p>
     *         <p>
     *         Including language options can improve the accuracy of language identification.
     *         </p>
     *         <p>
     *         For a list of languages supported with Call Analytics, refer to the <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages</a>
     *         table.
     *         </p>
     *         <p>
     *         To transcribe speech in Modern Standard Arabic (<code>ar-SA</code>) in Amazon Web Services GovCloud (US)
     *         (US-West, us-gov-west-1), Amazon Web Services GovCloud (US) (US-East, us-gov-east-1), Canada (Calgary)
     *         ca-west-1 and Africa (Cape Town) af-south-1, your media file must be encoded at a sample rate of 16,000
     *         Hz or higher.
     */
    public final List<String> languageOptionsAsStrings() {
        return languageOptions;
    }

    /**
     * <p>
     * If using automatic language identification in your request and you want to apply a custom language model, a
     * custom vocabulary, or a custom vocabulary filter, include <code>LanguageIdSettings</code> with the relevant
     * sub-parameters (<code>VocabularyName</code>, <code>LanguageModelName</code>, and
     * <code>VocabularyFilterName</code>).
     * </p>
     * <p>
     * <code>LanguageIdSettings</code> supports two to five language codes. Each language code you include can have an
     * associated custom language model, custom vocabulary, and custom vocabulary filter. The language codes that you
     * specify must match the languages of the associated custom language models, custom vocabularies, and custom
     * vocabulary filters.
     * </p>
     * <p>
     * It's recommended that you include <code>LanguageOptions</code> when using <code>LanguageIdSettings</code> to
     * ensure that the correct language dialect is identified. For example, if you specify a custom vocabulary that is
     * in <code>en-US</code> but Amazon Transcribe determines that the language spoken in your media is
     * <code>en-AU</code>, your custom vocabulary <i>is not</i> applied to your transcription. If you include
     * <code>LanguageOptions</code> and include <code>en-US</code> as the only English language dialect, your custom
     * vocabulary <i>is</i> applied to your transcription.
     * </p>
     * <p>
     * If you want to include a custom language model, custom vocabulary, or custom vocabulary filter with your request
     * but <b>do not</b> want to use automatic language identification, use instead the
     * <code/> parameter with the <code>LanguageModelName</code>, <code>VocabularyName</code>, or
     * <code>VocabularyFilterName</code> sub-parameters.
     * </p>
     * <p>
     * For a list of languages supported with Call Analytics, refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages and
     * language-specific features</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLanguageIdSettings} method.
     * </p>
     * 
     * @return If using automatic language identification in your request and you want to apply a custom language model,
     *         a custom vocabulary, or a custom vocabulary filter, include <code>LanguageIdSettings</code> with the
     *         relevant sub-parameters (<code>VocabularyName</code>, <code>LanguageModelName</code>, and
     *         <code>VocabularyFilterName</code>).</p>
     *         <p>
     *         <code>LanguageIdSettings</code> supports two to five language codes. Each language code you include can
     *         have an associated custom language model, custom vocabulary, and custom vocabulary filter. The language
     *         codes that you specify must match the languages of the associated custom language models, custom
     *         vocabularies, and custom vocabulary filters.
     *         </p>
     *         <p>
     *         It's recommended that you include <code>LanguageOptions</code> when using <code>LanguageIdSettings</code>
     *         to ensure that the correct language dialect is identified. For example, if you specify a custom
     *         vocabulary that is in <code>en-US</code> but Amazon Transcribe determines that the language spoken in
     *         your media is <code>en-AU</code>, your custom vocabulary <i>is not</i> applied to your transcription. If
     *         you include <code>LanguageOptions</code> and include <code>en-US</code> as the only English language
     *         dialect, your custom vocabulary <i>is</i> applied to your transcription.
     *         </p>
     *         <p>
     *         If you want to include a custom language model, custom vocabulary, or custom vocabulary filter with your
     *         request but <b>do not</b> want to use automatic language identification, use instead the
     *         <code/> parameter with the <code>LanguageModelName</code>, <code>VocabularyName</code>, or
     *         <code>VocabularyFilterName</code> sub-parameters.
     *         </p>
     *         <p>
     *         For a list of languages supported with Call Analytics, refer to <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages and
     *         language-specific features</a>.
     */
    public final Map<LanguageCode, LanguageIdSettings> languageIdSettings() {
        return LanguageIdSettingsMapCopier.copyStringToEnum(languageIdSettings);
    }

    /**
     * For responses, this returns true if the service returned a value for the LanguageIdSettings property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLanguageIdSettings() {
        return languageIdSettings != null && !(languageIdSettings instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * If using automatic language identification in your request and you want to apply a custom language model, a
     * custom vocabulary, or a custom vocabulary filter, include <code>LanguageIdSettings</code> with the relevant
     * sub-parameters (<code>VocabularyName</code>, <code>LanguageModelName</code>, and
     * <code>VocabularyFilterName</code>).
     * </p>
     * <p>
     * <code>LanguageIdSettings</code> supports two to five language codes. Each language code you include can have an
     * associated custom language model, custom vocabulary, and custom vocabulary filter. The language codes that you
     * specify must match the languages of the associated custom language models, custom vocabularies, and custom
     * vocabulary filters.
     * </p>
     * <p>
     * It's recommended that you include <code>LanguageOptions</code> when using <code>LanguageIdSettings</code> to
     * ensure that the correct language dialect is identified. For example, if you specify a custom vocabulary that is
     * in <code>en-US</code> but Amazon Transcribe determines that the language spoken in your media is
     * <code>en-AU</code>, your custom vocabulary <i>is not</i> applied to your transcription. If you include
     * <code>LanguageOptions</code> and include <code>en-US</code> as the only English language dialect, your custom
     * vocabulary <i>is</i> applied to your transcription.
     * </p>
     * <p>
     * If you want to include a custom language model, custom vocabulary, or custom vocabulary filter with your request
     * but <b>do not</b> want to use automatic language identification, use instead the
     * <code/> parameter with the <code>LanguageModelName</code>, <code>VocabularyName</code>, or
     * <code>VocabularyFilterName</code> sub-parameters.
     * </p>
     * <p>
     * For a list of languages supported with Call Analytics, refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages and
     * language-specific features</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLanguageIdSettings} method.
     * </p>
     * 
     * @return If using automatic language identification in your request and you want to apply a custom language model,
     *         a custom vocabulary, or a custom vocabulary filter, include <code>LanguageIdSettings</code> with the
     *         relevant sub-parameters (<code>VocabularyName</code>, <code>LanguageModelName</code>, and
     *         <code>VocabularyFilterName</code>).</p>
     *         <p>
     *         <code>LanguageIdSettings</code> supports two to five language codes. Each language code you include can
     *         have an associated custom language model, custom vocabulary, and custom vocabulary filter. The language
     *         codes that you specify must match the languages of the associated custom language models, custom
     *         vocabularies, and custom vocabulary filters.
     *         </p>
     *         <p>
     *         It's recommended that you include <code>LanguageOptions</code> when using <code>LanguageIdSettings</code>
     *         to ensure that the correct language dialect is identified. For example, if you specify a custom
     *         vocabulary that is in <code>en-US</code> but Amazon Transcribe determines that the language spoken in
     *         your media is <code>en-AU</code>, your custom vocabulary <i>is not</i> applied to your transcription. If
     *         you include <code>LanguageOptions</code> and include <code>en-US</code> as the only English language
     *         dialect, your custom vocabulary <i>is</i> applied to your transcription.
     *         </p>
     *         <p>
     *         If you want to include a custom language model, custom vocabulary, or custom vocabulary filter with your
     *         request but <b>do not</b> want to use automatic language identification, use instead the
     *         <code/> parameter with the <code>LanguageModelName</code>, <code>VocabularyName</code>, or
     *         <code>VocabularyFilterName</code> sub-parameters.
     *         </p>
     *         <p>
     *         For a list of languages supported with Call Analytics, refer to <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages and
     *         language-specific features</a>.
     */
    public final Map<String, LanguageIdSettings> languageIdSettingsAsStrings() {
        return languageIdSettings;
    }

    /**
     * <p>
     * Contains <code>GenerateAbstractiveSummary</code>, which is a required parameter if you want to enable Generative
     * call summarization in your Call Analytics request.
     * </p>
     * 
     * @return Contains <code>GenerateAbstractiveSummary</code>, which is a required parameter if you want to enable
     *         Generative call summarization in your Call Analytics request.
     */
    public final Summarization summarization() {
        return summarization;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(vocabularyName());
        hashCode = 31 * hashCode + Objects.hashCode(vocabularyFilterName());
        hashCode = 31 * hashCode + Objects.hashCode(vocabularyFilterMethodAsString());
        hashCode = 31 * hashCode + Objects.hashCode(languageModelName());
        hashCode = 31 * hashCode + Objects.hashCode(contentRedaction());
        hashCode = 31 * hashCode + Objects.hashCode(hasLanguageOptions() ? languageOptionsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasLanguageIdSettings() ? languageIdSettingsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(summarization());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CallAnalyticsJobSettings)) {
            return false;
        }
        CallAnalyticsJobSettings other = (CallAnalyticsJobSettings) obj;
        return Objects.equals(vocabularyName(), other.vocabularyName())
                && Objects.equals(vocabularyFilterName(), other.vocabularyFilterName())
                && Objects.equals(vocabularyFilterMethodAsString(), other.vocabularyFilterMethodAsString())
                && Objects.equals(languageModelName(), other.languageModelName())
                && Objects.equals(contentRedaction(), other.contentRedaction())
                && hasLanguageOptions() == other.hasLanguageOptions()
                && Objects.equals(languageOptionsAsStrings(), other.languageOptionsAsStrings())
                && hasLanguageIdSettings() == other.hasLanguageIdSettings()
                && Objects.equals(languageIdSettingsAsStrings(), other.languageIdSettingsAsStrings())
                && Objects.equals(summarization(), other.summarization());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CallAnalyticsJobSettings").add("VocabularyName", vocabularyName())
                .add("VocabularyFilterName", vocabularyFilterName())
                .add("VocabularyFilterMethod", vocabularyFilterMethodAsString()).add("LanguageModelName", languageModelName())
                .add("ContentRedaction", contentRedaction())
                .add("LanguageOptions", hasLanguageOptions() ? languageOptionsAsStrings() : null)
                .add("LanguageIdSettings", hasLanguageIdSettings() ? languageIdSettingsAsStrings() : null)
                .add("Summarization", summarization()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VocabularyName":
            return Optional.ofNullable(clazz.cast(vocabularyName()));
        case "VocabularyFilterName":
            return Optional.ofNullable(clazz.cast(vocabularyFilterName()));
        case "VocabularyFilterMethod":
            return Optional.ofNullable(clazz.cast(vocabularyFilterMethodAsString()));
        case "LanguageModelName":
            return Optional.ofNullable(clazz.cast(languageModelName()));
        case "ContentRedaction":
            return Optional.ofNullable(clazz.cast(contentRedaction()));
        case "LanguageOptions":
            return Optional.ofNullable(clazz.cast(languageOptionsAsStrings()));
        case "LanguageIdSettings":
            return Optional.ofNullable(clazz.cast(languageIdSettingsAsStrings()));
        case "Summarization":
            return Optional.ofNullable(clazz.cast(summarization()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("VocabularyName", VOCABULARY_NAME_FIELD);
        map.put("VocabularyFilterName", VOCABULARY_FILTER_NAME_FIELD);
        map.put("VocabularyFilterMethod", VOCABULARY_FILTER_METHOD_FIELD);
        map.put("LanguageModelName", LANGUAGE_MODEL_NAME_FIELD);
        map.put("ContentRedaction", CONTENT_REDACTION_FIELD);
        map.put("LanguageOptions", LANGUAGE_OPTIONS_FIELD);
        map.put("LanguageIdSettings", LANGUAGE_ID_SETTINGS_FIELD);
        map.put("Summarization", SUMMARIZATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CallAnalyticsJobSettings, T> g) {
        return obj -> g.apply((CallAnalyticsJobSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CallAnalyticsJobSettings> {
        /**
         * <p>
         * The name of the custom vocabulary you want to include in your Call Analytics transcription request. Custom
         * vocabulary names are case sensitive.
         * </p>
         * 
         * @param vocabularyName
         *        The name of the custom vocabulary you want to include in your Call Analytics transcription request.
         *        Custom vocabulary names are case sensitive.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vocabularyName(String vocabularyName);

        /**
         * <p>
         * The name of the custom vocabulary filter you want to include in your Call Analytics transcription request.
         * Custom vocabulary filter names are case sensitive.
         * </p>
         * <p>
         * Note that if you include <code>VocabularyFilterName</code> in your request, you must also include
         * <code>VocabularyFilterMethod</code>.
         * </p>
         * 
         * @param vocabularyFilterName
         *        The name of the custom vocabulary filter you want to include in your Call Analytics transcription
         *        request. Custom vocabulary filter names are case sensitive.</p>
         *        <p>
         *        Note that if you include <code>VocabularyFilterName</code> in your request, you must also include
         *        <code>VocabularyFilterMethod</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vocabularyFilterName(String vocabularyFilterName);

        /**
         * <p>
         * Specify how you want your custom vocabulary filter applied to your transcript.
         * </p>
         * <p>
         * To replace words with <code>***</code>, choose <code>mask</code>.
         * </p>
         * <p>
         * To delete words, choose <code>remove</code>.
         * </p>
         * <p>
         * To flag words without changing them, choose <code>tag</code>.
         * </p>
         * 
         * @param vocabularyFilterMethod
         *        Specify how you want your custom vocabulary filter applied to your transcript.</p>
         *        <p>
         *        To replace words with <code>***</code>, choose <code>mask</code>.
         *        </p>
         *        <p>
         *        To delete words, choose <code>remove</code>.
         *        </p>
         *        <p>
         *        To flag words without changing them, choose <code>tag</code>.
         * @see VocabularyFilterMethod
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VocabularyFilterMethod
         */
        Builder vocabularyFilterMethod(String vocabularyFilterMethod);

        /**
         * <p>
         * Specify how you want your custom vocabulary filter applied to your transcript.
         * </p>
         * <p>
         * To replace words with <code>***</code>, choose <code>mask</code>.
         * </p>
         * <p>
         * To delete words, choose <code>remove</code>.
         * </p>
         * <p>
         * To flag words without changing them, choose <code>tag</code>.
         * </p>
         * 
         * @param vocabularyFilterMethod
         *        Specify how you want your custom vocabulary filter applied to your transcript.</p>
         *        <p>
         *        To replace words with <code>***</code>, choose <code>mask</code>.
         *        </p>
         *        <p>
         *        To delete words, choose <code>remove</code>.
         *        </p>
         *        <p>
         *        To flag words without changing them, choose <code>tag</code>.
         * @see VocabularyFilterMethod
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VocabularyFilterMethod
         */
        Builder vocabularyFilterMethod(VocabularyFilterMethod vocabularyFilterMethod);

        /**
         * <p>
         * The name of the custom language model you want to use when processing your Call Analytics job. Note that
         * custom language model names are case sensitive.
         * </p>
         * <p>
         * The language of the specified custom language model must match the language code that you specify in your
         * transcription request. If the languages do not match, the custom language model isn't applied. There are no
         * errors or warnings associated with a language mismatch.
         * </p>
         * 
         * @param languageModelName
         *        The name of the custom language model you want to use when processing your Call Analytics job. Note
         *        that custom language model names are case sensitive.</p>
         *        <p>
         *        The language of the specified custom language model must match the language code that you specify in
         *        your transcription request. If the languages do not match, the custom language model isn't applied.
         *        There are no errors or warnings associated with a language mismatch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageModelName(String languageModelName);

        /**
         * Sets the value of the ContentRedaction property for this object.
         *
         * @param contentRedaction
         *        The new value for the ContentRedaction property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contentRedaction(ContentRedaction contentRedaction);

        /**
         * Sets the value of the ContentRedaction property for this object.
         *
         * This is a convenience method that creates an instance of the {@link ContentRedaction.Builder} avoiding the
         * need to create one manually via {@link ContentRedaction#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ContentRedaction.Builder#build()} is called immediately and its
         * result is passed to {@link #contentRedaction(ContentRedaction)}.
         * 
         * @param contentRedaction
         *        a consumer that will call methods on {@link ContentRedaction.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #contentRedaction(ContentRedaction)
         */
        default Builder contentRedaction(Consumer<ContentRedaction.Builder> contentRedaction) {
            return contentRedaction(ContentRedaction.builder().applyMutation(contentRedaction).build());
        }

        /**
         * <p>
         * You can specify two or more language codes that represent the languages you think may be present in your
         * media. Including more than five is not recommended. If you're unsure what languages are present, do not
         * include this parameter.
         * </p>
         * <p>
         * Including language options can improve the accuracy of language identification.
         * </p>
         * <p>
         * For a list of languages supported with Call Analytics, refer to the <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages</a>
         * table.
         * </p>
         * <p>
         * To transcribe speech in Modern Standard Arabic (<code>ar-SA</code>) in Amazon Web Services GovCloud (US)
         * (US-West, us-gov-west-1), Amazon Web Services GovCloud (US) (US-East, us-gov-east-1), Canada (Calgary)
         * ca-west-1 and Africa (Cape Town) af-south-1, your media file must be encoded at a sample rate of 16,000 Hz or
         * higher.
         * </p>
         * 
         * @param languageOptions
         *        You can specify two or more language codes that represent the languages you think may be present in
         *        your media. Including more than five is not recommended. If you're unsure what languages are present,
         *        do not include this parameter.</p>
         *        <p>
         *        Including language options can improve the accuracy of language identification.
         *        </p>
         *        <p>
         *        For a list of languages supported with Call Analytics, refer to the <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported
         *        languages</a> table.
         *        </p>
         *        <p>
         *        To transcribe speech in Modern Standard Arabic (<code>ar-SA</code>) in Amazon Web Services GovCloud
         *        (US) (US-West, us-gov-west-1), Amazon Web Services GovCloud (US) (US-East, us-gov-east-1), Canada
         *        (Calgary) ca-west-1 and Africa (Cape Town) af-south-1, your media file must be encoded at a sample
         *        rate of 16,000 Hz or higher.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageOptionsWithStrings(Collection<String> languageOptions);

        /**
         * <p>
         * You can specify two or more language codes that represent the languages you think may be present in your
         * media. Including more than five is not recommended. If you're unsure what languages are present, do not
         * include this parameter.
         * </p>
         * <p>
         * Including language options can improve the accuracy of language identification.
         * </p>
         * <p>
         * For a list of languages supported with Call Analytics, refer to the <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages</a>
         * table.
         * </p>
         * <p>
         * To transcribe speech in Modern Standard Arabic (<code>ar-SA</code>) in Amazon Web Services GovCloud (US)
         * (US-West, us-gov-west-1), Amazon Web Services GovCloud (US) (US-East, us-gov-east-1), Canada (Calgary)
         * ca-west-1 and Africa (Cape Town) af-south-1, your media file must be encoded at a sample rate of 16,000 Hz or
         * higher.
         * </p>
         * 
         * @param languageOptions
         *        You can specify two or more language codes that represent the languages you think may be present in
         *        your media. Including more than five is not recommended. If you're unsure what languages are present,
         *        do not include this parameter.</p>
         *        <p>
         *        Including language options can improve the accuracy of language identification.
         *        </p>
         *        <p>
         *        For a list of languages supported with Call Analytics, refer to the <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported
         *        languages</a> table.
         *        </p>
         *        <p>
         *        To transcribe speech in Modern Standard Arabic (<code>ar-SA</code>) in Amazon Web Services GovCloud
         *        (US) (US-West, us-gov-west-1), Amazon Web Services GovCloud (US) (US-East, us-gov-east-1), Canada
         *        (Calgary) ca-west-1 and Africa (Cape Town) af-south-1, your media file must be encoded at a sample
         *        rate of 16,000 Hz or higher.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageOptionsWithStrings(String... languageOptions);

        /**
         * <p>
         * You can specify two or more language codes that represent the languages you think may be present in your
         * media. Including more than five is not recommended. If you're unsure what languages are present, do not
         * include this parameter.
         * </p>
         * <p>
         * Including language options can improve the accuracy of language identification.
         * </p>
         * <p>
         * For a list of languages supported with Call Analytics, refer to the <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages</a>
         * table.
         * </p>
         * <p>
         * To transcribe speech in Modern Standard Arabic (<code>ar-SA</code>) in Amazon Web Services GovCloud (US)
         * (US-West, us-gov-west-1), Amazon Web Services GovCloud (US) (US-East, us-gov-east-1), Canada (Calgary)
         * ca-west-1 and Africa (Cape Town) af-south-1, your media file must be encoded at a sample rate of 16,000 Hz or
         * higher.
         * </p>
         * 
         * @param languageOptions
         *        You can specify two or more language codes that represent the languages you think may be present in
         *        your media. Including more than five is not recommended. If you're unsure what languages are present,
         *        do not include this parameter.</p>
         *        <p>
         *        Including language options can improve the accuracy of language identification.
         *        </p>
         *        <p>
         *        For a list of languages supported with Call Analytics, refer to the <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported
         *        languages</a> table.
         *        </p>
         *        <p>
         *        To transcribe speech in Modern Standard Arabic (<code>ar-SA</code>) in Amazon Web Services GovCloud
         *        (US) (US-West, us-gov-west-1), Amazon Web Services GovCloud (US) (US-East, us-gov-east-1), Canada
         *        (Calgary) ca-west-1 and Africa (Cape Town) af-south-1, your media file must be encoded at a sample
         *        rate of 16,000 Hz or higher.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageOptions(Collection<LanguageCode> languageOptions);

        /**
         * <p>
         * You can specify two or more language codes that represent the languages you think may be present in your
         * media. Including more than five is not recommended. If you're unsure what languages are present, do not
         * include this parameter.
         * </p>
         * <p>
         * Including language options can improve the accuracy of language identification.
         * </p>
         * <p>
         * For a list of languages supported with Call Analytics, refer to the <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages</a>
         * table.
         * </p>
         * <p>
         * To transcribe speech in Modern Standard Arabic (<code>ar-SA</code>) in Amazon Web Services GovCloud (US)
         * (US-West, us-gov-west-1), Amazon Web Services GovCloud (US) (US-East, us-gov-east-1), Canada (Calgary)
         * ca-west-1 and Africa (Cape Town) af-south-1, your media file must be encoded at a sample rate of 16,000 Hz or
         * higher.
         * </p>
         * 
         * @param languageOptions
         *        You can specify two or more language codes that represent the languages you think may be present in
         *        your media. Including more than five is not recommended. If you're unsure what languages are present,
         *        do not include this parameter.</p>
         *        <p>
         *        Including language options can improve the accuracy of language identification.
         *        </p>
         *        <p>
         *        For a list of languages supported with Call Analytics, refer to the <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported
         *        languages</a> table.
         *        </p>
         *        <p>
         *        To transcribe speech in Modern Standard Arabic (<code>ar-SA</code>) in Amazon Web Services GovCloud
         *        (US) (US-West, us-gov-west-1), Amazon Web Services GovCloud (US) (US-East, us-gov-east-1), Canada
         *        (Calgary) ca-west-1 and Africa (Cape Town) af-south-1, your media file must be encoded at a sample
         *        rate of 16,000 Hz or higher.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageOptions(LanguageCode... languageOptions);

        /**
         * <p>
         * If using automatic language identification in your request and you want to apply a custom language model, a
         * custom vocabulary, or a custom vocabulary filter, include <code>LanguageIdSettings</code> with the relevant
         * sub-parameters (<code>VocabularyName</code>, <code>LanguageModelName</code>, and
         * <code>VocabularyFilterName</code>).
         * </p>
         * <p>
         * <code>LanguageIdSettings</code> supports two to five language codes. Each language code you include can have
         * an associated custom language model, custom vocabulary, and custom vocabulary filter. The language codes that
         * you specify must match the languages of the associated custom language models, custom vocabularies, and
         * custom vocabulary filters.
         * </p>
         * <p>
         * It's recommended that you include <code>LanguageOptions</code> when using <code>LanguageIdSettings</code> to
         * ensure that the correct language dialect is identified. For example, if you specify a custom vocabulary that
         * is in <code>en-US</code> but Amazon Transcribe determines that the language spoken in your media is
         * <code>en-AU</code>, your custom vocabulary <i>is not</i> applied to your transcription. If you include
         * <code>LanguageOptions</code> and include <code>en-US</code> as the only English language dialect, your custom
         * vocabulary <i>is</i> applied to your transcription.
         * </p>
         * <p>
         * If you want to include a custom language model, custom vocabulary, or custom vocabulary filter with your
         * request but <b>do not</b> want to use automatic language identification, use instead the
         * <code/> parameter with the <code>LanguageModelName</code>, <code>VocabularyName</code>, or
         * <code>VocabularyFilterName</code> sub-parameters.
         * </p>
         * <p>
         * For a list of languages supported with Call Analytics, refer to <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages and
         * language-specific features</a>.
         * </p>
         * 
         * @param languageIdSettings
         *        If using automatic language identification in your request and you want to apply a custom language
         *        model, a custom vocabulary, or a custom vocabulary filter, include <code>LanguageIdSettings</code>
         *        with the relevant sub-parameters (<code>VocabularyName</code>, <code>LanguageModelName</code>, and
         *        <code>VocabularyFilterName</code>).</p>
         *        <p>
         *        <code>LanguageIdSettings</code> supports two to five language codes. Each language code you include
         *        can have an associated custom language model, custom vocabulary, and custom vocabulary filter. The
         *        language codes that you specify must match the languages of the associated custom language models,
         *        custom vocabularies, and custom vocabulary filters.
         *        </p>
         *        <p>
         *        It's recommended that you include <code>LanguageOptions</code> when using
         *        <code>LanguageIdSettings</code> to ensure that the correct language dialect is identified. For
         *        example, if you specify a custom vocabulary that is in <code>en-US</code> but Amazon Transcribe
         *        determines that the language spoken in your media is <code>en-AU</code>, your custom vocabulary <i>is
         *        not</i> applied to your transcription. If you include <code>LanguageOptions</code> and include
         *        <code>en-US</code> as the only English language dialect, your custom vocabulary <i>is</i> applied to
         *        your transcription.
         *        </p>
         *        <p>
         *        If you want to include a custom language model, custom vocabulary, or custom vocabulary filter with
         *        your request but <b>do not</b> want to use automatic language identification, use instead the
         *        <code/> parameter with the <code>LanguageModelName</code>, <code>VocabularyName</code>, or
         *        <code>VocabularyFilterName</code> sub-parameters.
         *        </p>
         *        <p>
         *        For a list of languages supported with Call Analytics, refer to <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages
         *        and language-specific features</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageIdSettingsWithStrings(Map<String, LanguageIdSettings> languageIdSettings);

        /**
         * <p>
         * If using automatic language identification in your request and you want to apply a custom language model, a
         * custom vocabulary, or a custom vocabulary filter, include <code>LanguageIdSettings</code> with the relevant
         * sub-parameters (<code>VocabularyName</code>, <code>LanguageModelName</code>, and
         * <code>VocabularyFilterName</code>).
         * </p>
         * <p>
         * <code>LanguageIdSettings</code> supports two to five language codes. Each language code you include can have
         * an associated custom language model, custom vocabulary, and custom vocabulary filter. The language codes that
         * you specify must match the languages of the associated custom language models, custom vocabularies, and
         * custom vocabulary filters.
         * </p>
         * <p>
         * It's recommended that you include <code>LanguageOptions</code> when using <code>LanguageIdSettings</code> to
         * ensure that the correct language dialect is identified. For example, if you specify a custom vocabulary that
         * is in <code>en-US</code> but Amazon Transcribe determines that the language spoken in your media is
         * <code>en-AU</code>, your custom vocabulary <i>is not</i> applied to your transcription. If you include
         * <code>LanguageOptions</code> and include <code>en-US</code> as the only English language dialect, your custom
         * vocabulary <i>is</i> applied to your transcription.
         * </p>
         * <p>
         * If you want to include a custom language model, custom vocabulary, or custom vocabulary filter with your
         * request but <b>do not</b> want to use automatic language identification, use instead the
         * <code/> parameter with the <code>LanguageModelName</code>, <code>VocabularyName</code>, or
         * <code>VocabularyFilterName</code> sub-parameters.
         * </p>
         * <p>
         * For a list of languages supported with Call Analytics, refer to <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages and
         * language-specific features</a>.
         * </p>
         * 
         * @param languageIdSettings
         *        If using automatic language identification in your request and you want to apply a custom language
         *        model, a custom vocabulary, or a custom vocabulary filter, include <code>LanguageIdSettings</code>
         *        with the relevant sub-parameters (<code>VocabularyName</code>, <code>LanguageModelName</code>, and
         *        <code>VocabularyFilterName</code>).</p>
         *        <p>
         *        <code>LanguageIdSettings</code> supports two to five language codes. Each language code you include
         *        can have an associated custom language model, custom vocabulary, and custom vocabulary filter. The
         *        language codes that you specify must match the languages of the associated custom language models,
         *        custom vocabularies, and custom vocabulary filters.
         *        </p>
         *        <p>
         *        It's recommended that you include <code>LanguageOptions</code> when using
         *        <code>LanguageIdSettings</code> to ensure that the correct language dialect is identified. For
         *        example, if you specify a custom vocabulary that is in <code>en-US</code> but Amazon Transcribe
         *        determines that the language spoken in your media is <code>en-AU</code>, your custom vocabulary <i>is
         *        not</i> applied to your transcription. If you include <code>LanguageOptions</code> and include
         *        <code>en-US</code> as the only English language dialect, your custom vocabulary <i>is</i> applied to
         *        your transcription.
         *        </p>
         *        <p>
         *        If you want to include a custom language model, custom vocabulary, or custom vocabulary filter with
         *        your request but <b>do not</b> want to use automatic language identification, use instead the
         *        <code/> parameter with the <code>LanguageModelName</code>, <code>VocabularyName</code>, or
         *        <code>VocabularyFilterName</code> sub-parameters.
         *        </p>
         *        <p>
         *        For a list of languages supported with Call Analytics, refer to <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages
         *        and language-specific features</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageIdSettings(Map<LanguageCode, LanguageIdSettings> languageIdSettings);

        /**
         * <p>
         * Contains <code>GenerateAbstractiveSummary</code>, which is a required parameter if you want to enable
         * Generative call summarization in your Call Analytics request.
         * </p>
         * 
         * @param summarization
         *        Contains <code>GenerateAbstractiveSummary</code>, which is a required parameter if you want to enable
         *        Generative call summarization in your Call Analytics request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder summarization(Summarization summarization);

        /**
         * <p>
         * Contains <code>GenerateAbstractiveSummary</code>, which is a required parameter if you want to enable
         * Generative call summarization in your Call Analytics request.
         * </p>
         * This is a convenience method that creates an instance of the {@link Summarization.Builder} avoiding the need
         * to create one manually via {@link Summarization#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Summarization.Builder#build()} is called immediately and its
         * result is passed to {@link #summarization(Summarization)}.
         * 
         * @param summarization
         *        a consumer that will call methods on {@link Summarization.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #summarization(Summarization)
         */
        default Builder summarization(Consumer<Summarization.Builder> summarization) {
            return summarization(Summarization.builder().applyMutation(summarization).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String vocabularyName;

        private String vocabularyFilterName;

        private String vocabularyFilterMethod;

        private String languageModelName;

        private ContentRedaction contentRedaction;

        private List<String> languageOptions = DefaultSdkAutoConstructList.getInstance();

        private Map<String, LanguageIdSettings> languageIdSettings = DefaultSdkAutoConstructMap.getInstance();

        private Summarization summarization;

        private BuilderImpl() {
        }

        private BuilderImpl(CallAnalyticsJobSettings model) {
            vocabularyName(model.vocabularyName);
            vocabularyFilterName(model.vocabularyFilterName);
            vocabularyFilterMethod(model.vocabularyFilterMethod);
            languageModelName(model.languageModelName);
            contentRedaction(model.contentRedaction);
            languageOptionsWithStrings(model.languageOptions);
            languageIdSettingsWithStrings(model.languageIdSettings);
            summarization(model.summarization);
        }

        public final String getVocabularyName() {
            return vocabularyName;
        }

        public final void setVocabularyName(String vocabularyName) {
            this.vocabularyName = vocabularyName;
        }

        @Override
        public final Builder vocabularyName(String vocabularyName) {
            this.vocabularyName = vocabularyName;
            return this;
        }

        public final String getVocabularyFilterName() {
            return vocabularyFilterName;
        }

        public final void setVocabularyFilterName(String vocabularyFilterName) {
            this.vocabularyFilterName = vocabularyFilterName;
        }

        @Override
        public final Builder vocabularyFilterName(String vocabularyFilterName) {
            this.vocabularyFilterName = vocabularyFilterName;
            return this;
        }

        public final String getVocabularyFilterMethod() {
            return vocabularyFilterMethod;
        }

        public final void setVocabularyFilterMethod(String vocabularyFilterMethod) {
            this.vocabularyFilterMethod = vocabularyFilterMethod;
        }

        @Override
        public final Builder vocabularyFilterMethod(String vocabularyFilterMethod) {
            this.vocabularyFilterMethod = vocabularyFilterMethod;
            return this;
        }

        @Override
        public final Builder vocabularyFilterMethod(VocabularyFilterMethod vocabularyFilterMethod) {
            this.vocabularyFilterMethod(vocabularyFilterMethod == null ? null : vocabularyFilterMethod.toString());
            return this;
        }

        public final String getLanguageModelName() {
            return languageModelName;
        }

        public final void setLanguageModelName(String languageModelName) {
            this.languageModelName = languageModelName;
        }

        @Override
        public final Builder languageModelName(String languageModelName) {
            this.languageModelName = languageModelName;
            return this;
        }

        public final ContentRedaction.Builder getContentRedaction() {
            return contentRedaction != null ? contentRedaction.toBuilder() : null;
        }

        public final void setContentRedaction(ContentRedaction.BuilderImpl contentRedaction) {
            this.contentRedaction = contentRedaction != null ? contentRedaction.build() : null;
        }

        @Override
        public final Builder contentRedaction(ContentRedaction contentRedaction) {
            this.contentRedaction = contentRedaction;
            return this;
        }

        public final Collection<String> getLanguageOptions() {
            if (languageOptions instanceof SdkAutoConstructList) {
                return null;
            }
            return languageOptions;
        }

        public final void setLanguageOptions(Collection<String> languageOptions) {
            this.languageOptions = LanguageOptionsCopier.copy(languageOptions);
        }

        @Override
        public final Builder languageOptionsWithStrings(Collection<String> languageOptions) {
            this.languageOptions = LanguageOptionsCopier.copy(languageOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder languageOptionsWithStrings(String... languageOptions) {
            languageOptionsWithStrings(Arrays.asList(languageOptions));
            return this;
        }

        @Override
        public final Builder languageOptions(Collection<LanguageCode> languageOptions) {
            this.languageOptions = LanguageOptionsCopier.copyEnumToString(languageOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder languageOptions(LanguageCode... languageOptions) {
            languageOptions(Arrays.asList(languageOptions));
            return this;
        }

        public final Map<String, LanguageIdSettings.Builder> getLanguageIdSettings() {
            Map<String, LanguageIdSettings.Builder> result = LanguageIdSettingsMapCopier.copyToBuilder(this.languageIdSettings);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setLanguageIdSettings(Map<String, LanguageIdSettings.BuilderImpl> languageIdSettings) {
            this.languageIdSettings = LanguageIdSettingsMapCopier.copyFromBuilder(languageIdSettings);
        }

        @Override
        public final Builder languageIdSettingsWithStrings(Map<String, LanguageIdSettings> languageIdSettings) {
            this.languageIdSettings = LanguageIdSettingsMapCopier.copy(languageIdSettings);
            return this;
        }

        @Override
        public final Builder languageIdSettings(Map<LanguageCode, LanguageIdSettings> languageIdSettings) {
            this.languageIdSettings = LanguageIdSettingsMapCopier.copyEnumToString(languageIdSettings);
            return this;
        }

        public final Summarization.Builder getSummarization() {
            return summarization != null ? summarization.toBuilder() : null;
        }

        public final void setSummarization(Summarization.BuilderImpl summarization) {
            this.summarization = summarization != null ? summarization.build() : null;
        }

        @Override
        public final Builder summarization(Summarization summarization) {
            this.summarization = summarization;
            return this;
        }

        @Override
        public CallAnalyticsJobSettings build() {
            return new CallAnalyticsJobSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
