/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribe.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateVocabularyRequest extends TranscribeRequest implements
        ToCopyableBuilder<UpdateVocabularyRequest.Builder, UpdateVocabularyRequest> {
    private static final SdkField<String> VOCABULARY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VocabularyName").getter(getter(UpdateVocabularyRequest::vocabularyName))
            .setter(setter(Builder::vocabularyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VocabularyName").build()).build();

    private static final SdkField<String> LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LanguageCode").getter(getter(UpdateVocabularyRequest::languageCodeAsString))
            .setter(setter(Builder::languageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageCode").build()).build();

    private static final SdkField<List<String>> PHRASES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Phrases")
            .getter(getter(UpdateVocabularyRequest::phrases))
            .setter(setter(Builder::phrases))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Phrases").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> VOCABULARY_FILE_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VocabularyFileUri").getter(getter(UpdateVocabularyRequest::vocabularyFileUri))
            .setter(setter(Builder::vocabularyFileUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VocabularyFileUri").build()).build();

    private static final SdkField<String> DATA_ACCESS_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataAccessRoleArn").getter(getter(UpdateVocabularyRequest::dataAccessRoleArn))
            .setter(setter(Builder::dataAccessRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataAccessRoleArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VOCABULARY_NAME_FIELD,
            LANGUAGE_CODE_FIELD, PHRASES_FIELD, VOCABULARY_FILE_URI_FIELD, DATA_ACCESS_ROLE_ARN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String vocabularyName;

    private final String languageCode;

    private final List<String> phrases;

    private final String vocabularyFileUri;

    private final String dataAccessRoleArn;

    private UpdateVocabularyRequest(BuilderImpl builder) {
        super(builder);
        this.vocabularyName = builder.vocabularyName;
        this.languageCode = builder.languageCode;
        this.phrases = builder.phrases;
        this.vocabularyFileUri = builder.vocabularyFileUri;
        this.dataAccessRoleArn = builder.dataAccessRoleArn;
    }

    /**
     * <p>
     * The name of the custom vocabulary you want to update. Custom vocabulary names are case sensitive.
     * </p>
     * 
     * @return The name of the custom vocabulary you want to update. Custom vocabulary names are case sensitive.
     */
    public final String vocabularyName() {
        return vocabularyName;
    }

    /**
     * <p>
     * The language code that represents the language of the entries in the custom vocabulary you want to update. Each
     * custom vocabulary must contain terms in only one language.
     * </p>
     * <p>
     * A custom vocabulary can only be used to transcribe files in the same language as the custom vocabulary. For
     * example, if you create a custom vocabulary using US English (<code>en-US</code>), you can only apply this custom
     * vocabulary to files that contain English audio.
     * </p>
     * <p>
     * For a list of supported languages and their associated language codes, refer to the <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages</a> table.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return The language code that represents the language of the entries in the custom vocabulary you want to
     *         update. Each custom vocabulary must contain terms in only one language.</p>
     *         <p>
     *         A custom vocabulary can only be used to transcribe files in the same language as the custom vocabulary.
     *         For example, if you create a custom vocabulary using US English (<code>en-US</code>), you can only apply
     *         this custom vocabulary to files that contain English audio.
     *         </p>
     *         <p>
     *         For a list of supported languages and their associated language codes, refer to the <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages</a>
     *         table.
     * @see LanguageCode
     */
    public final LanguageCode languageCode() {
        return LanguageCode.fromValue(languageCode);
    }

    /**
     * <p>
     * The language code that represents the language of the entries in the custom vocabulary you want to update. Each
     * custom vocabulary must contain terms in only one language.
     * </p>
     * <p>
     * A custom vocabulary can only be used to transcribe files in the same language as the custom vocabulary. For
     * example, if you create a custom vocabulary using US English (<code>en-US</code>), you can only apply this custom
     * vocabulary to files that contain English audio.
     * </p>
     * <p>
     * For a list of supported languages and their associated language codes, refer to the <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages</a> table.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return The language code that represents the language of the entries in the custom vocabulary you want to
     *         update. Each custom vocabulary must contain terms in only one language.</p>
     *         <p>
     *         A custom vocabulary can only be used to transcribe files in the same language as the custom vocabulary.
     *         For example, if you create a custom vocabulary using US English (<code>en-US</code>), you can only apply
     *         this custom vocabulary to files that contain English audio.
     *         </p>
     *         <p>
     *         For a list of supported languages and their associated language codes, refer to the <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages</a>
     *         table.
     * @see LanguageCode
     */
    public final String languageCodeAsString() {
        return languageCode;
    }

    /**
     * For responses, this returns true if the service returned a value for the Phrases property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPhrases() {
        return phrases != null && !(phrases instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Use this parameter if you want to update your custom vocabulary by including all desired terms, as
     * comma-separated values, within your request. The other option for updating your custom vocabulary is to save your
     * entries in a text file and upload them to an Amazon S3 bucket, then specify the location of your file using the
     * <code>VocabularyFileUri</code> parameter.
     * </p>
     * <p>
     * Note that if you include <code>Phrases</code> in your request, you cannot use <code>VocabularyFileUri</code>; you
     * must choose one or the other.
     * </p>
     * <p>
     * Each language has a character set that contains all allowed characters for that specific language. If you use
     * unsupported characters, your custom vocabulary filter request fails. Refer to <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/charsets.html">Character Sets for Custom Vocabularies</a>
     * to get the character set for your language.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPhrases} method.
     * </p>
     * 
     * @return Use this parameter if you want to update your custom vocabulary by including all desired terms, as
     *         comma-separated values, within your request. The other option for updating your custom vocabulary is to
     *         save your entries in a text file and upload them to an Amazon S3 bucket, then specify the location of
     *         your file using the <code>VocabularyFileUri</code> parameter.</p>
     *         <p>
     *         Note that if you include <code>Phrases</code> in your request, you cannot use
     *         <code>VocabularyFileUri</code>; you must choose one or the other.
     *         </p>
     *         <p>
     *         Each language has a character set that contains all allowed characters for that specific language. If you
     *         use unsupported characters, your custom vocabulary filter request fails. Refer to <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/dg/charsets.html">Character Sets for Custom
     *         Vocabularies</a> to get the character set for your language.
     */
    public final List<String> phrases() {
        return phrases;
    }

    /**
     * <p>
     * The Amazon S3 location of the text file that contains your custom vocabulary. The URI must be located in the same
     * Amazon Web Services Region as the resource you're calling.
     * </p>
     * <p>
     * Here's an example URI path: <code>s3://DOC-EXAMPLE-BUCKET/my-vocab-file.txt</code>
     * </p>
     * <p>
     * Note that if you include <code>VocabularyFileUri</code> in your request, you cannot use the <code>Phrases</code>
     * flag; you must choose one or the other.
     * </p>
     * 
     * @return The Amazon S3 location of the text file that contains your custom vocabulary. The URI must be located in
     *         the same Amazon Web Services Region as the resource you're calling.</p>
     *         <p>
     *         Here's an example URI path: <code>s3://DOC-EXAMPLE-BUCKET/my-vocab-file.txt</code>
     *         </p>
     *         <p>
     *         Note that if you include <code>VocabularyFileUri</code> in your request, you cannot use the
     *         <code>Phrases</code> flag; you must choose one or the other.
     */
    public final String vocabularyFileUri() {
        return vocabularyFileUri;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an IAM role that has permissions to access the Amazon S3 bucket that contains
     * your input files (in this case, your custom vocabulary). If the role that you specify doesn’t have the
     * appropriate permissions to access the specified Amazon S3 location, your request fails.
     * </p>
     * <p>
     * IAM role ARNs have the format <code>arn:partition:iam::account:role/role-name-with-path</code>. For example:
     * <code>arn:aws:iam::111122223333:role/Admin</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-arns">IAM ARNs</a>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of an IAM role that has permissions to access the Amazon S3 bucket that
     *         contains your input files (in this case, your custom vocabulary). If the role that you specify doesn’t
     *         have the appropriate permissions to access the specified Amazon S3 location, your request fails.</p>
     *         <p>
     *         IAM role ARNs have the format <code>arn:partition:iam::account:role/role-name-with-path</code>. For
     *         example: <code>arn:aws:iam::111122223333:role/Admin</code>.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-arns">IAM
     *         ARNs</a>.
     */
    public final String dataAccessRoleArn() {
        return dataAccessRoleArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(vocabularyName());
        hashCode = 31 * hashCode + Objects.hashCode(languageCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasPhrases() ? phrases() : null);
        hashCode = 31 * hashCode + Objects.hashCode(vocabularyFileUri());
        hashCode = 31 * hashCode + Objects.hashCode(dataAccessRoleArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateVocabularyRequest)) {
            return false;
        }
        UpdateVocabularyRequest other = (UpdateVocabularyRequest) obj;
        return Objects.equals(vocabularyName(), other.vocabularyName())
                && Objects.equals(languageCodeAsString(), other.languageCodeAsString()) && hasPhrases() == other.hasPhrases()
                && Objects.equals(phrases(), other.phrases()) && Objects.equals(vocabularyFileUri(), other.vocabularyFileUri())
                && Objects.equals(dataAccessRoleArn(), other.dataAccessRoleArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UpdateVocabularyRequest").add("VocabularyName", vocabularyName())
                .add("LanguageCode", languageCodeAsString()).add("Phrases", hasPhrases() ? phrases() : null)
                .add("VocabularyFileUri", vocabularyFileUri()).add("DataAccessRoleArn", dataAccessRoleArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VocabularyName":
            return Optional.ofNullable(clazz.cast(vocabularyName()));
        case "LanguageCode":
            return Optional.ofNullable(clazz.cast(languageCodeAsString()));
        case "Phrases":
            return Optional.ofNullable(clazz.cast(phrases()));
        case "VocabularyFileUri":
            return Optional.ofNullable(clazz.cast(vocabularyFileUri()));
        case "DataAccessRoleArn":
            return Optional.ofNullable(clazz.cast(dataAccessRoleArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("VocabularyName", VOCABULARY_NAME_FIELD);
        map.put("LanguageCode", LANGUAGE_CODE_FIELD);
        map.put("Phrases", PHRASES_FIELD);
        map.put("VocabularyFileUri", VOCABULARY_FILE_URI_FIELD);
        map.put("DataAccessRoleArn", DATA_ACCESS_ROLE_ARN_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<UpdateVocabularyRequest, T> g) {
        return obj -> g.apply((UpdateVocabularyRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends TranscribeRequest.Builder, SdkPojo, CopyableBuilder<Builder, UpdateVocabularyRequest> {
        /**
         * <p>
         * The name of the custom vocabulary you want to update. Custom vocabulary names are case sensitive.
         * </p>
         * 
         * @param vocabularyName
         *        The name of the custom vocabulary you want to update. Custom vocabulary names are case sensitive.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vocabularyName(String vocabularyName);

        /**
         * <p>
         * The language code that represents the language of the entries in the custom vocabulary you want to update.
         * Each custom vocabulary must contain terms in only one language.
         * </p>
         * <p>
         * A custom vocabulary can only be used to transcribe files in the same language as the custom vocabulary. For
         * example, if you create a custom vocabulary using US English (<code>en-US</code>), you can only apply this
         * custom vocabulary to files that contain English audio.
         * </p>
         * <p>
         * For a list of supported languages and their associated language codes, refer to the <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages</a>
         * table.
         * </p>
         * 
         * @param languageCode
         *        The language code that represents the language of the entries in the custom vocabulary you want to
         *        update. Each custom vocabulary must contain terms in only one language.</p>
         *        <p>
         *        A custom vocabulary can only be used to transcribe files in the same language as the custom
         *        vocabulary. For example, if you create a custom vocabulary using US English (<code>en-US</code>), you
         *        can only apply this custom vocabulary to files that contain English audio.
         *        </p>
         *        <p>
         *        For a list of supported languages and their associated language codes, refer to the <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported
         *        languages</a> table.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(String languageCode);

        /**
         * <p>
         * The language code that represents the language of the entries in the custom vocabulary you want to update.
         * Each custom vocabulary must contain terms in only one language.
         * </p>
         * <p>
         * A custom vocabulary can only be used to transcribe files in the same language as the custom vocabulary. For
         * example, if you create a custom vocabulary using US English (<code>en-US</code>), you can only apply this
         * custom vocabulary to files that contain English audio.
         * </p>
         * <p>
         * For a list of supported languages and their associated language codes, refer to the <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages</a>
         * table.
         * </p>
         * 
         * @param languageCode
         *        The language code that represents the language of the entries in the custom vocabulary you want to
         *        update. Each custom vocabulary must contain terms in only one language.</p>
         *        <p>
         *        A custom vocabulary can only be used to transcribe files in the same language as the custom
         *        vocabulary. For example, if you create a custom vocabulary using US English (<code>en-US</code>), you
         *        can only apply this custom vocabulary to files that contain English audio.
         *        </p>
         *        <p>
         *        For a list of supported languages and their associated language codes, refer to the <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported
         *        languages</a> table.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(LanguageCode languageCode);

        /**
         * <p>
         * Use this parameter if you want to update your custom vocabulary by including all desired terms, as
         * comma-separated values, within your request. The other option for updating your custom vocabulary is to save
         * your entries in a text file and upload them to an Amazon S3 bucket, then specify the location of your file
         * using the <code>VocabularyFileUri</code> parameter.
         * </p>
         * <p>
         * Note that if you include <code>Phrases</code> in your request, you cannot use <code>VocabularyFileUri</code>;
         * you must choose one or the other.
         * </p>
         * <p>
         * Each language has a character set that contains all allowed characters for that specific language. If you use
         * unsupported characters, your custom vocabulary filter request fails. Refer to <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/charsets.html">Character Sets for Custom
         * Vocabularies</a> to get the character set for your language.
         * </p>
         * 
         * @param phrases
         *        Use this parameter if you want to update your custom vocabulary by including all desired terms, as
         *        comma-separated values, within your request. The other option for updating your custom vocabulary is
         *        to save your entries in a text file and upload them to an Amazon S3 bucket, then specify the location
         *        of your file using the <code>VocabularyFileUri</code> parameter.</p>
         *        <p>
         *        Note that if you include <code>Phrases</code> in your request, you cannot use
         *        <code>VocabularyFileUri</code>; you must choose one or the other.
         *        </p>
         *        <p>
         *        Each language has a character set that contains all allowed characters for that specific language. If
         *        you use unsupported characters, your custom vocabulary filter request fails. Refer to <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/charsets.html">Character Sets for Custom
         *        Vocabularies</a> to get the character set for your language.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder phrases(Collection<String> phrases);

        /**
         * <p>
         * Use this parameter if you want to update your custom vocabulary by including all desired terms, as
         * comma-separated values, within your request. The other option for updating your custom vocabulary is to save
         * your entries in a text file and upload them to an Amazon S3 bucket, then specify the location of your file
         * using the <code>VocabularyFileUri</code> parameter.
         * </p>
         * <p>
         * Note that if you include <code>Phrases</code> in your request, you cannot use <code>VocabularyFileUri</code>;
         * you must choose one or the other.
         * </p>
         * <p>
         * Each language has a character set that contains all allowed characters for that specific language. If you use
         * unsupported characters, your custom vocabulary filter request fails. Refer to <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/charsets.html">Character Sets for Custom
         * Vocabularies</a> to get the character set for your language.
         * </p>
         * 
         * @param phrases
         *        Use this parameter if you want to update your custom vocabulary by including all desired terms, as
         *        comma-separated values, within your request. The other option for updating your custom vocabulary is
         *        to save your entries in a text file and upload them to an Amazon S3 bucket, then specify the location
         *        of your file using the <code>VocabularyFileUri</code> parameter.</p>
         *        <p>
         *        Note that if you include <code>Phrases</code> in your request, you cannot use
         *        <code>VocabularyFileUri</code>; you must choose one or the other.
         *        </p>
         *        <p>
         *        Each language has a character set that contains all allowed characters for that specific language. If
         *        you use unsupported characters, your custom vocabulary filter request fails. Refer to <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/charsets.html">Character Sets for Custom
         *        Vocabularies</a> to get the character set for your language.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder phrases(String... phrases);

        /**
         * <p>
         * The Amazon S3 location of the text file that contains your custom vocabulary. The URI must be located in the
         * same Amazon Web Services Region as the resource you're calling.
         * </p>
         * <p>
         * Here's an example URI path: <code>s3://DOC-EXAMPLE-BUCKET/my-vocab-file.txt</code>
         * </p>
         * <p>
         * Note that if you include <code>VocabularyFileUri</code> in your request, you cannot use the
         * <code>Phrases</code> flag; you must choose one or the other.
         * </p>
         * 
         * @param vocabularyFileUri
         *        The Amazon S3 location of the text file that contains your custom vocabulary. The URI must be located
         *        in the same Amazon Web Services Region as the resource you're calling.</p>
         *        <p>
         *        Here's an example URI path: <code>s3://DOC-EXAMPLE-BUCKET/my-vocab-file.txt</code>
         *        </p>
         *        <p>
         *        Note that if you include <code>VocabularyFileUri</code> in your request, you cannot use the
         *        <code>Phrases</code> flag; you must choose one or the other.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vocabularyFileUri(String vocabularyFileUri);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an IAM role that has permissions to access the Amazon S3 bucket that
         * contains your input files (in this case, your custom vocabulary). If the role that you specify doesn’t have
         * the appropriate permissions to access the specified Amazon S3 location, your request fails.
         * </p>
         * <p>
         * IAM role ARNs have the format <code>arn:partition:iam::account:role/role-name-with-path</code>. For example:
         * <code>arn:aws:iam::111122223333:role/Admin</code>.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-arns">IAM
         * ARNs</a>.
         * </p>
         * 
         * @param dataAccessRoleArn
         *        The Amazon Resource Name (ARN) of an IAM role that has permissions to access the Amazon S3 bucket that
         *        contains your input files (in this case, your custom vocabulary). If the role that you specify doesn’t
         *        have the appropriate permissions to access the specified Amazon S3 location, your request fails.</p>
         *        <p>
         *        IAM role ARNs have the format <code>arn:partition:iam::account:role/role-name-with-path</code>. For
         *        example: <code>arn:aws:iam::111122223333:role/Admin</code>.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-arns"
         *        >IAM ARNs</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataAccessRoleArn(String dataAccessRoleArn);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends TranscribeRequest.BuilderImpl implements Builder {
        private String vocabularyName;

        private String languageCode;

        private List<String> phrases = DefaultSdkAutoConstructList.getInstance();

        private String vocabularyFileUri;

        private String dataAccessRoleArn;

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateVocabularyRequest model) {
            super(model);
            vocabularyName(model.vocabularyName);
            languageCode(model.languageCode);
            phrases(model.phrases);
            vocabularyFileUri(model.vocabularyFileUri);
            dataAccessRoleArn(model.dataAccessRoleArn);
        }

        public final String getVocabularyName() {
            return vocabularyName;
        }

        public final void setVocabularyName(String vocabularyName) {
            this.vocabularyName = vocabularyName;
        }

        @Override
        public final Builder vocabularyName(String vocabularyName) {
            this.vocabularyName = vocabularyName;
            return this;
        }

        public final String getLanguageCode() {
            return languageCode;
        }

        public final void setLanguageCode(String languageCode) {
            this.languageCode = languageCode;
        }

        @Override
        public final Builder languageCode(String languageCode) {
            this.languageCode = languageCode;
            return this;
        }

        @Override
        public final Builder languageCode(LanguageCode languageCode) {
            this.languageCode(languageCode == null ? null : languageCode.toString());
            return this;
        }

        public final Collection<String> getPhrases() {
            if (phrases instanceof SdkAutoConstructList) {
                return null;
            }
            return phrases;
        }

        public final void setPhrases(Collection<String> phrases) {
            this.phrases = PhrasesCopier.copy(phrases);
        }

        @Override
        public final Builder phrases(Collection<String> phrases) {
            this.phrases = PhrasesCopier.copy(phrases);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder phrases(String... phrases) {
            phrases(Arrays.asList(phrases));
            return this;
        }

        public final String getVocabularyFileUri() {
            return vocabularyFileUri;
        }

        public final void setVocabularyFileUri(String vocabularyFileUri) {
            this.vocabularyFileUri = vocabularyFileUri;
        }

        @Override
        public final Builder vocabularyFileUri(String vocabularyFileUri) {
            this.vocabularyFileUri = vocabularyFileUri;
            return this;
        }

        public final String getDataAccessRoleArn() {
            return dataAccessRoleArn;
        }

        public final void setDataAccessRoleArn(String dataAccessRoleArn) {
            this.dataAccessRoleArn = dataAccessRoleArn;
        }

        @Override
        public final Builder dataAccessRoleArn(String dataAccessRoleArn) {
            this.dataAccessRoleArn = dataAccessRoleArn;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public UpdateVocabularyRequest build() {
            return new UpdateVocabularyRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
