/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribestreaming;

import static software.amazon.awssdk.utils.FunctionalUtils.runAndLogError;

import java.nio.ByteBuffer;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.Consumer;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.auth.signer.EventStreamAws4Signer;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.client.handler.AwsClientHandlerUtils;
import software.amazon.awssdk.awscore.eventstream.EventStreamAsyncResponseTransformer;
import software.amazon.awssdk.awscore.eventstream.EventStreamTaggedUnionJsonMarshaller;
import software.amazon.awssdk.awscore.eventstream.EventStreamTaggedUnionPojoSupplier;
import software.amazon.awssdk.awscore.eventstream.RestEventStreamAsyncResponseTransformer;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.SdkResponse;
import software.amazon.awssdk.core.async.SdkPublisher;
import software.amazon.awssdk.core.client.config.SdkAdvancedAsyncClientOption;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.AttachHttpMetadataResponseHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.protocol.VoidSdkResponse;
import software.amazon.awssdk.core.signer.Signer;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.transcribestreaming.model.AudioEvent;
import software.amazon.awssdk.services.transcribestreaming.model.AudioStream;
import software.amazon.awssdk.services.transcribestreaming.model.BadRequestException;
import software.amazon.awssdk.services.transcribestreaming.model.ConflictException;
import software.amazon.awssdk.services.transcribestreaming.model.InternalFailureException;
import software.amazon.awssdk.services.transcribestreaming.model.LimitExceededException;
import software.amazon.awssdk.services.transcribestreaming.model.StartStreamTranscriptionRequest;
import software.amazon.awssdk.services.transcribestreaming.model.StartStreamTranscriptionResponse;
import software.amazon.awssdk.services.transcribestreaming.model.StartStreamTranscriptionResponseHandler;
import software.amazon.awssdk.services.transcribestreaming.model.TranscribeStreamingException;
import software.amazon.awssdk.services.transcribestreaming.model.TranscribeStreamingRequest;
import software.amazon.awssdk.services.transcribestreaming.model.TranscriptEvent;
import software.amazon.awssdk.services.transcribestreaming.model.TranscriptResultStream;
import software.amazon.awssdk.services.transcribestreaming.transform.AudioEventMarshaller;
import software.amazon.awssdk.services.transcribestreaming.transform.StartStreamTranscriptionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link TranscribeStreamingAsyncClient}.
 *
 * @see TranscribeStreamingAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTranscribeStreamingAsyncClient implements TranscribeStreamingAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultTranscribeStreamingAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final Executor executor;

    protected DefaultTranscribeStreamingAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        this.executor = clientConfiguration.option(SdkAdvancedAsyncClientOption.FUTURE_COMPLETION_EXECUTOR);
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Starts a bidirectional HTTP2 stream where audio is streamed to Amazon Transcribe and the transcription results
     * are streamed to your application.
     * </p>
     * <p>
     * The following are encoded as HTTP2 headers:
     * </p>
     * <ul>
     * <li>
     * <p>
     * x-amzn-transcribe-language-code
     * </p>
     * </li>
     * <li>
     * <p>
     * x-amzn-transcribe-media-encoding
     * </p>
     * </li>
     * <li>
     * <p>
     * x-amzn-transcribe-sample-rate
     * </p>
     * </li>
     * <li>
     * <p>
     * x-amzn-transcribe-session-id
     * </p>
     * </li>
     * </ul>
     *
     * @param startStreamTranscriptionRequest
     * @return A Java Future containing the result of the StartStreamTranscription operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code> operation was
     *         invalid. For example, <code>MediaEncoding</code> was not set to <code>pcm</code> or
     *         <code>LanguageCode</code> was not set to a valid code. Check the parameters and try your request again.</li>
     *         <li>LimitExceededException You have exceeded the maximum number of concurrent transcription streams, are
     *         starting transcription streams too quickly, or the maximum audio length of 4 hours. Wait until a stream
     *         has finished processing, or break your audio stream into smaller chunks and try your request again.</li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing. Try your request again.</li>
     *         <li>ConflictException A new stream started with the same session ID. The current stream has been
     *         terminated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.StartStreamTranscription
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/StartStreamTranscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<Void> startStreamTranscription(StartStreamTranscriptionRequest startStreamTranscriptionRequest,
            Publisher<AudioStream> requestStream, StartStreamTranscriptionResponseHandler asyncResponseHandler) {
        try {
            startStreamTranscriptionRequest = applySignerOverride(startStreamTranscriptionRequest, EventStreamAws4Signer.create());
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartStreamTranscriptionResponse> responseHandler = new AttachHttpMetadataResponseHandler(
                    protocolFactory.createResponseHandler(operationMetadata, StartStreamTranscriptionResponse::builder));

            HttpResponseHandler<SdkResponse> voidResponseHandler = protocolFactory.createResponseHandler(JsonOperationMetadata
                    .builder().isPayloadJson(false).hasStreamingSuccessResponse(true).build(), VoidSdkResponse::builder);

            HttpResponseHandler<? extends TranscriptResultStream> eventResponseHandler = protocolFactory.createResponseHandler(
                    JsonOperationMetadata.builder().isPayloadJson(true).hasStreamingSuccessResponse(false).build(),
                    EventStreamTaggedUnionPojoSupplier.builder().putSdkPojoSupplier("TranscriptEvent", TranscriptEvent::builder)
                            .defaultSdkPojoSupplier(() -> TranscriptResultStream.UNKNOWN).build());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            EventStreamTaggedUnionJsonMarshaller eventMarshaller = EventStreamTaggedUnionJsonMarshaller.builder()
                    .putMarshaller(AudioEvent.class, new AudioEventMarshaller(protocolFactory)).build();
            SdkPublisher<AudioStream> eventPublisher = SdkPublisher.adapt(requestStream);
            Publisher<ByteBuffer> adapted = eventPublisher.map(event -> eventMarshaller.marshall(event)).map(
                    AwsClientHandlerUtils::encodeEventStreamRequestToByteBuffer);
            CompletableFuture<Void> future = new CompletableFuture<>();
            EventStreamAsyncResponseTransformer<StartStreamTranscriptionResponse, TranscriptResultStream> asyncResponseTransformer = EventStreamAsyncResponseTransformer
                    .<StartStreamTranscriptionResponse, TranscriptResultStream> builder()
                    .eventStreamResponseHandler(asyncResponseHandler).eventResponseHandler(eventResponseHandler)
                    .initialResponseHandler(responseHandler).exceptionResponseHandler(errorResponseHandler).future(future)
                    .executor(executor).serviceName(serviceName()).build();
            RestEventStreamAsyncResponseTransformer<StartStreamTranscriptionResponse, TranscriptResultStream> restAsyncResponseTransformer = RestEventStreamAsyncResponseTransformer
                    .<StartStreamTranscriptionResponse, TranscriptResultStream> builder()
                    .eventStreamAsyncResponseTransformer(asyncResponseTransformer)
                    .eventStreamResponseHandler(asyncResponseHandler).build();

            clientHandler.execute(
                    new ClientExecutionParams<StartStreamTranscriptionRequest, StartStreamTranscriptionResponse>()
                            .withMarshaller(new StartStreamTranscriptionRequestMarshaller(protocolFactory))
                            .withAsyncRequestBody(software.amazon.awssdk.core.async.AsyncRequestBody.fromPublisher(adapted))
                            .withFullDuplex(true).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(startStreamTranscriptionRequest),
                    restAsyncResponseTransformer).whenComplete((r, e) -> {
                if (e != null) {
                    try {
                        asyncResponseHandler.exceptionOccurred(e);
                    } finally {
                        future.completeExceptionally(e);
                    }
                }
            });
            return future;
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseHandler.exceptionOccurred(t));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(TranscribeStreamingException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(429).build());
    }

    private <T extends TranscribeStreamingRequest> T applySignerOverride(T request, Signer signer) {
        if (request.overrideConfiguration().flatMap(c -> c.signer()).isPresent()) {
            return request;
        }
        Consumer<AwsRequestOverrideConfiguration.Builder> signerOverride = b -> b.signer(signer).build();
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(signerOverride).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(signerOverride).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
