/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribestreaming.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The result of transcribing a portion of the input audio stream.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Result implements SdkPojo, Serializable, ToCopyableBuilder<Result.Builder, Result> {
    private static final SdkField<String> RESULT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResultId").getter(getter(Result::resultId)).setter(setter(Builder::resultId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResultId").build()).build();

    private static final SdkField<Double> START_TIME_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("StartTime").getter(getter(Result::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Double> END_TIME_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("EndTime").getter(getter(Result::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTime").build()).build();

    private static final SdkField<Boolean> IS_PARTIAL_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IsPartial").getter(getter(Result::isPartial)).setter(setter(Builder::isPartial))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsPartial").build()).build();

    private static final SdkField<List<Alternative>> ALTERNATIVES_FIELD = SdkField
            .<List<Alternative>> builder(MarshallingType.LIST)
            .memberName("Alternatives")
            .getter(getter(Result::alternatives))
            .setter(setter(Builder::alternatives))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Alternatives").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Alternative> builder(MarshallingType.SDK_POJO)
                                            .constructor(Alternative::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CHANNEL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ChannelId").getter(getter(Result::channelId)).setter(setter(Builder::channelId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChannelId").build()).build();

    private static final SdkField<String> LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LanguageCode").getter(getter(Result::languageCodeAsString)).setter(setter(Builder::languageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageCode").build()).build();

    private static final SdkField<List<LanguageWithScore>> LANGUAGE_IDENTIFICATION_FIELD = SdkField
            .<List<LanguageWithScore>> builder(MarshallingType.LIST)
            .memberName("LanguageIdentification")
            .getter(getter(Result::languageIdentification))
            .setter(setter(Builder::languageIdentification))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageIdentification").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LanguageWithScore> builder(MarshallingType.SDK_POJO)
                                            .constructor(LanguageWithScore::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESULT_ID_FIELD,
            START_TIME_FIELD, END_TIME_FIELD, IS_PARTIAL_FIELD, ALTERNATIVES_FIELD, CHANNEL_ID_FIELD, LANGUAGE_CODE_FIELD,
            LANGUAGE_IDENTIFICATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String resultId;

    private final Double startTime;

    private final Double endTime;

    private final Boolean isPartial;

    private final List<Alternative> alternatives;

    private final String channelId;

    private final String languageCode;

    private final List<LanguageWithScore> languageIdentification;

    private Result(BuilderImpl builder) {
        this.resultId = builder.resultId;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.isPartial = builder.isPartial;
        this.alternatives = builder.alternatives;
        this.channelId = builder.channelId;
        this.languageCode = builder.languageCode;
        this.languageIdentification = builder.languageIdentification;
    }

    /**
     * <p>
     * A unique identifier for the result.
     * </p>
     * 
     * @return A unique identifier for the result.
     */
    public final String resultId() {
        return resultId;
    }

    /**
     * <p>
     * The offset in seconds from the beginning of the audio stream to the beginning of the result.
     * </p>
     * 
     * @return The offset in seconds from the beginning of the audio stream to the beginning of the result.
     */
    public final Double startTime() {
        return startTime;
    }

    /**
     * <p>
     * The offset in seconds from the beginning of the audio stream to the end of the result.
     * </p>
     * 
     * @return The offset in seconds from the beginning of the audio stream to the end of the result.
     */
    public final Double endTime() {
        return endTime;
    }

    /**
     * <p>
     * Amazon Transcribe divides the incoming audio stream into segments at natural points in the audio. Transcription
     * results are returned based on these segments.
     * </p>
     * <p>
     * The <code>IsPartial</code> field is <code>true</code> to indicate that Amazon Transcribe has additional
     * transcription data to send, <code>false</code> to indicate that this is the last transcription result for the
     * segment.
     * </p>
     * 
     * @return Amazon Transcribe divides the incoming audio stream into segments at natural points in the audio.
     *         Transcription results are returned based on these segments. </p>
     *         <p>
     *         The <code>IsPartial</code> field is <code>true</code> to indicate that Amazon Transcribe has additional
     *         transcription data to send, <code>false</code> to indicate that this is the last transcription result for
     *         the segment.
     */
    public final Boolean isPartial() {
        return isPartial;
    }

    /**
     * For responses, this returns true if the service returned a value for the Alternatives property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAlternatives() {
        return alternatives != null && !(alternatives instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of possible transcriptions for the audio. Each alternative typically contains one <code>item</code> that
     * contains the result of the transcription.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAlternatives} method.
     * </p>
     * 
     * @return A list of possible transcriptions for the audio. Each alternative typically contains one
     *         <code>item</code> that contains the result of the transcription.
     */
    public final List<Alternative> alternatives() {
        return alternatives;
    }

    /**
     * <p>
     * When channel identification is enabled, Amazon Transcribe transcribes the speech from each audio channel
     * separately.
     * </p>
     * <p>
     * You can use <code>ChannelId</code> to retrieve the transcription results for a single channel in your audio
     * stream.
     * </p>
     * 
     * @return When channel identification is enabled, Amazon Transcribe transcribes the speech from each audio channel
     *         separately.</p>
     *         <p>
     *         You can use <code>ChannelId</code> to retrieve the transcription results for a single channel in your
     *         audio stream.
     */
    public final String channelId() {
        return channelId;
    }

    /**
     * <p>
     * The language code of the identified language in your media stream.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return The language code of the identified language in your media stream.
     * @see LanguageCode
     */
    public final LanguageCode languageCode() {
        return LanguageCode.fromValue(languageCode);
    }

    /**
     * <p>
     * The language code of the identified language in your media stream.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return The language code of the identified language in your media stream.
     * @see LanguageCode
     */
    public final String languageCodeAsString() {
        return languageCode;
    }

    /**
     * For responses, this returns true if the service returned a value for the LanguageIdentification property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasLanguageIdentification() {
        return languageIdentification != null && !(languageIdentification instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The language code of the dominant language identified in your media.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLanguageIdentification} method.
     * </p>
     * 
     * @return The language code of the dominant language identified in your media.
     */
    public final List<LanguageWithScore> languageIdentification() {
        return languageIdentification;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resultId());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(isPartial());
        hashCode = 31 * hashCode + Objects.hashCode(hasAlternatives() ? alternatives() : null);
        hashCode = 31 * hashCode + Objects.hashCode(channelId());
        hashCode = 31 * hashCode + Objects.hashCode(languageCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasLanguageIdentification() ? languageIdentification() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Result)) {
            return false;
        }
        Result other = (Result) obj;
        return Objects.equals(resultId(), other.resultId()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(endTime(), other.endTime()) && Objects.equals(isPartial(), other.isPartial())
                && hasAlternatives() == other.hasAlternatives() && Objects.equals(alternatives(), other.alternatives())
                && Objects.equals(channelId(), other.channelId())
                && Objects.equals(languageCodeAsString(), other.languageCodeAsString())
                && hasLanguageIdentification() == other.hasLanguageIdentification()
                && Objects.equals(languageIdentification(), other.languageIdentification());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Result").add("ResultId", resultId()).add("StartTime", startTime()).add("EndTime", endTime())
                .add("IsPartial", isPartial()).add("Alternatives", hasAlternatives() ? alternatives() : null)
                .add("ChannelId", channelId()).add("LanguageCode", languageCodeAsString())
                .add("LanguageIdentification", hasLanguageIdentification() ? languageIdentification() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ResultId":
            return Optional.ofNullable(clazz.cast(resultId()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "EndTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "IsPartial":
            return Optional.ofNullable(clazz.cast(isPartial()));
        case "Alternatives":
            return Optional.ofNullable(clazz.cast(alternatives()));
        case "ChannelId":
            return Optional.ofNullable(clazz.cast(channelId()));
        case "LanguageCode":
            return Optional.ofNullable(clazz.cast(languageCodeAsString()));
        case "LanguageIdentification":
            return Optional.ofNullable(clazz.cast(languageIdentification()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Result, T> g) {
        return obj -> g.apply((Result) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Result> {
        /**
         * <p>
         * A unique identifier for the result.
         * </p>
         * 
         * @param resultId
         *        A unique identifier for the result.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resultId(String resultId);

        /**
         * <p>
         * The offset in seconds from the beginning of the audio stream to the beginning of the result.
         * </p>
         * 
         * @param startTime
         *        The offset in seconds from the beginning of the audio stream to the beginning of the result.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Double startTime);

        /**
         * <p>
         * The offset in seconds from the beginning of the audio stream to the end of the result.
         * </p>
         * 
         * @param endTime
         *        The offset in seconds from the beginning of the audio stream to the end of the result.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Double endTime);

        /**
         * <p>
         * Amazon Transcribe divides the incoming audio stream into segments at natural points in the audio.
         * Transcription results are returned based on these segments.
         * </p>
         * <p>
         * The <code>IsPartial</code> field is <code>true</code> to indicate that Amazon Transcribe has additional
         * transcription data to send, <code>false</code> to indicate that this is the last transcription result for the
         * segment.
         * </p>
         * 
         * @param isPartial
         *        Amazon Transcribe divides the incoming audio stream into segments at natural points in the audio.
         *        Transcription results are returned based on these segments. </p>
         *        <p>
         *        The <code>IsPartial</code> field is <code>true</code> to indicate that Amazon Transcribe has
         *        additional transcription data to send, <code>false</code> to indicate that this is the last
         *        transcription result for the segment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isPartial(Boolean isPartial);

        /**
         * <p>
         * A list of possible transcriptions for the audio. Each alternative typically contains one <code>item</code>
         * that contains the result of the transcription.
         * </p>
         * 
         * @param alternatives
         *        A list of possible transcriptions for the audio. Each alternative typically contains one
         *        <code>item</code> that contains the result of the transcription.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alternatives(Collection<Alternative> alternatives);

        /**
         * <p>
         * A list of possible transcriptions for the audio. Each alternative typically contains one <code>item</code>
         * that contains the result of the transcription.
         * </p>
         * 
         * @param alternatives
         *        A list of possible transcriptions for the audio. Each alternative typically contains one
         *        <code>item</code> that contains the result of the transcription.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alternatives(Alternative... alternatives);

        /**
         * <p>
         * A list of possible transcriptions for the audio. Each alternative typically contains one <code>item</code>
         * that contains the result of the transcription.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Alternative>.Builder} avoiding the need to
         * create one manually via {@link List<Alternative>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Alternative>.Builder#build()} is called immediately and its
         * result is passed to {@link #alternatives(List<Alternative>)}.
         * 
         * @param alternatives
         *        a consumer that will call methods on {@link List<Alternative>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #alternatives(List<Alternative>)
         */
        Builder alternatives(Consumer<Alternative.Builder>... alternatives);

        /**
         * <p>
         * When channel identification is enabled, Amazon Transcribe transcribes the speech from each audio channel
         * separately.
         * </p>
         * <p>
         * You can use <code>ChannelId</code> to retrieve the transcription results for a single channel in your audio
         * stream.
         * </p>
         * 
         * @param channelId
         *        When channel identification is enabled, Amazon Transcribe transcribes the speech from each audio
         *        channel separately.</p>
         *        <p>
         *        You can use <code>ChannelId</code> to retrieve the transcription results for a single channel in your
         *        audio stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder channelId(String channelId);

        /**
         * <p>
         * The language code of the identified language in your media stream.
         * </p>
         * 
         * @param languageCode
         *        The language code of the identified language in your media stream.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(String languageCode);

        /**
         * <p>
         * The language code of the identified language in your media stream.
         * </p>
         * 
         * @param languageCode
         *        The language code of the identified language in your media stream.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(LanguageCode languageCode);

        /**
         * <p>
         * The language code of the dominant language identified in your media.
         * </p>
         * 
         * @param languageIdentification
         *        The language code of the dominant language identified in your media.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageIdentification(Collection<LanguageWithScore> languageIdentification);

        /**
         * <p>
         * The language code of the dominant language identified in your media.
         * </p>
         * 
         * @param languageIdentification
         *        The language code of the dominant language identified in your media.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageIdentification(LanguageWithScore... languageIdentification);

        /**
         * <p>
         * The language code of the dominant language identified in your media.
         * </p>
         * This is a convenience that creates an instance of the {@link List<LanguageWithScore>.Builder} avoiding the
         * need to create one manually via {@link List<LanguageWithScore>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<LanguageWithScore>.Builder#build()} is called immediately
         * and its result is passed to {@link #languageIdentification(List<LanguageWithScore>)}.
         * 
         * @param languageIdentification
         *        a consumer that will call methods on {@link List<LanguageWithScore>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #languageIdentification(List<LanguageWithScore>)
         */
        Builder languageIdentification(Consumer<LanguageWithScore.Builder>... languageIdentification);
    }

    static final class BuilderImpl implements Builder {
        private String resultId;

        private Double startTime;

        private Double endTime;

        private Boolean isPartial;

        private List<Alternative> alternatives = DefaultSdkAutoConstructList.getInstance();

        private String channelId;

        private String languageCode;

        private List<LanguageWithScore> languageIdentification = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Result model) {
            resultId(model.resultId);
            startTime(model.startTime);
            endTime(model.endTime);
            isPartial(model.isPartial);
            alternatives(model.alternatives);
            channelId(model.channelId);
            languageCode(model.languageCode);
            languageIdentification(model.languageIdentification);
        }

        public final String getResultId() {
            return resultId;
        }

        public final void setResultId(String resultId) {
            this.resultId = resultId;
        }

        @Override
        @Transient
        public final Builder resultId(String resultId) {
            this.resultId = resultId;
            return this;
        }

        public final Double getStartTime() {
            return startTime;
        }

        public final void setStartTime(Double startTime) {
            this.startTime = startTime;
        }

        @Override
        @Transient
        public final Builder startTime(Double startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Double getEndTime() {
            return endTime;
        }

        public final void setEndTime(Double endTime) {
            this.endTime = endTime;
        }

        @Override
        @Transient
        public final Builder endTime(Double endTime) {
            this.endTime = endTime;
            return this;
        }

        public final Boolean getIsPartial() {
            return isPartial;
        }

        public final void setIsPartial(Boolean isPartial) {
            this.isPartial = isPartial;
        }

        @Override
        @Transient
        public final Builder isPartial(Boolean isPartial) {
            this.isPartial = isPartial;
            return this;
        }

        public final List<Alternative.Builder> getAlternatives() {
            List<Alternative.Builder> result = AlternativeListCopier.copyToBuilder(this.alternatives);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAlternatives(Collection<Alternative.BuilderImpl> alternatives) {
            this.alternatives = AlternativeListCopier.copyFromBuilder(alternatives);
        }

        @Override
        @Transient
        public final Builder alternatives(Collection<Alternative> alternatives) {
            this.alternatives = AlternativeListCopier.copy(alternatives);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder alternatives(Alternative... alternatives) {
            alternatives(Arrays.asList(alternatives));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder alternatives(Consumer<Alternative.Builder>... alternatives) {
            alternatives(Stream.of(alternatives).map(c -> Alternative.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getChannelId() {
            return channelId;
        }

        public final void setChannelId(String channelId) {
            this.channelId = channelId;
        }

        @Override
        @Transient
        public final Builder channelId(String channelId) {
            this.channelId = channelId;
            return this;
        }

        public final String getLanguageCode() {
            return languageCode;
        }

        public final void setLanguageCode(String languageCode) {
            this.languageCode = languageCode;
        }

        @Override
        @Transient
        public final Builder languageCode(String languageCode) {
            this.languageCode = languageCode;
            return this;
        }

        @Override
        @Transient
        public final Builder languageCode(LanguageCode languageCode) {
            this.languageCode(languageCode == null ? null : languageCode.toString());
            return this;
        }

        public final List<LanguageWithScore.Builder> getLanguageIdentification() {
            List<LanguageWithScore.Builder> result = LanguageIdentificationCopier.copyToBuilder(this.languageIdentification);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setLanguageIdentification(Collection<LanguageWithScore.BuilderImpl> languageIdentification) {
            this.languageIdentification = LanguageIdentificationCopier.copyFromBuilder(languageIdentification);
        }

        @Override
        @Transient
        public final Builder languageIdentification(Collection<LanguageWithScore> languageIdentification) {
            this.languageIdentification = LanguageIdentificationCopier.copy(languageIdentification);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder languageIdentification(LanguageWithScore... languageIdentification) {
            languageIdentification(Arrays.asList(languageIdentification));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder languageIdentification(Consumer<LanguageWithScore.Builder>... languageIdentification) {
            languageIdentification(Stream.of(languageIdentification)
                    .map(c -> LanguageWithScore.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Result build() {
            return new Result(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
