/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribestreaming;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.reactivestreams.Publisher;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.services.transcribestreaming.model.AudioStream;
import software.amazon.awssdk.services.transcribestreaming.model.StartMedicalStreamTranscriptionRequest;
import software.amazon.awssdk.services.transcribestreaming.model.StartMedicalStreamTranscriptionResponseHandler;
import software.amazon.awssdk.services.transcribestreaming.model.StartStreamTranscriptionRequest;
import software.amazon.awssdk.services.transcribestreaming.model.StartStreamTranscriptionResponseHandler;

/**
 * Service client for accessing Amazon Transcribe Streaming Service asynchronously. This can be created using the static
 * {@link #builder()} method.
 *
 * <p>
 * Amazon Transcribe streaming offers two types of real-time transcription: <b>Standard</b> and <b>Medical</b>.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>Standard transcriptions</b> are the most common option. Refer to for details.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Medical transcriptions</b> are tailored to medical professionals and incorporate medical terms. A common use case
 * for this service is transcribing doctor-patient dialogue in real time, so doctors can focus on their patient instead
 * of taking notes. Refer to for details.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface TranscribeStreamingAsyncClient extends SdkClient {
    String SERVICE_NAME = "transcribe";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "transcribestreaming";

    /**
     * Create a {@link TranscribeStreamingAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static TranscribeStreamingAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link TranscribeStreamingAsyncClient}.
     */
    static TranscribeStreamingAsyncClientBuilder builder() {
        return new DefaultTranscribeStreamingAsyncClientBuilder();
    }

    /**
     * <p>
     * Starts a bidirectional HTTP/2 or WebSocket stream where audio is streamed to Amazon Transcribe Medical and the
     * transcription results are streamed to your application.
     * </p>
     * <p>
     * For more information on streaming with Amazon Transcribe Medical, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/streaming.html">Transcribing streaming audio</a>.
     * </p>
     *
     * @param startMedicalStreamTranscriptionRequest
     * @return A Java Future containing the result of the StartMedicalStreamTranscription operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code> or
     *         <code>StartMedicalStreamTranscription</code> operation was not valid. For example,
     *         <code>MediaEncoding</code> or <code>LanguageCode</code> used not valid values. Check the specified
     *         parameters and try your request again.</li>
     *         <li>LimitExceededException Your client has exceeded one of the Amazon Transcribe limits. This is
     *         typically the audio length limit. Break your audio stream into smaller chunks and try your request again.
     *         </li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing.</li>
     *         <li>ConflictException A new stream started with the same session ID. The current stream has been
     *         terminated.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable. Try your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.StartMedicalStreamTranscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/StartMedicalStreamTranscription"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<Void> startMedicalStreamTranscription(
            StartMedicalStreamTranscriptionRequest startMedicalStreamTranscriptionRequest, Publisher<AudioStream> requestStream,
            StartMedicalStreamTranscriptionResponseHandler asyncResponseHandler) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Starts a bidirectional HTTP/2 or WebSocket stream where audio is streamed to Amazon Transcribe Medical and the
     * transcription results are streamed to your application.
     * </p>
     * <p>
     * For more information on streaming with Amazon Transcribe Medical, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/streaming.html">Transcribing streaming audio</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartMedicalStreamTranscriptionRequest.Builder}
     * avoiding the need to create one manually via {@link StartMedicalStreamTranscriptionRequest#builder()}
     * </p>
     *
     * @param startMedicalStreamTranscriptionRequest
     *        A {@link Consumer} that will call methods on {@link StartMedicalStreamTranscriptionRequest.Builder} to
     *        create a request.
     * @return A Java Future containing the result of the StartMedicalStreamTranscription operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code> or
     *         <code>StartMedicalStreamTranscription</code> operation was not valid. For example,
     *         <code>MediaEncoding</code> or <code>LanguageCode</code> used not valid values. Check the specified
     *         parameters and try your request again.</li>
     *         <li>LimitExceededException Your client has exceeded one of the Amazon Transcribe limits. This is
     *         typically the audio length limit. Break your audio stream into smaller chunks and try your request again.
     *         </li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing.</li>
     *         <li>ConflictException A new stream started with the same session ID. The current stream has been
     *         terminated.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable. Try your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.StartMedicalStreamTranscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/StartMedicalStreamTranscription"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<Void> startMedicalStreamTranscription(
            Consumer<StartMedicalStreamTranscriptionRequest.Builder> startMedicalStreamTranscriptionRequest,
            Publisher<AudioStream> requestStream, StartMedicalStreamTranscriptionResponseHandler asyncResponseHandler) {
        return startMedicalStreamTranscription(
                StartMedicalStreamTranscriptionRequest.builder().applyMutation(startMedicalStreamTranscriptionRequest).build(),
                requestStream, asyncResponseHandler);
    }

    /**
     * <p>
     * Starts a bidirectional HTTP/2 or WebSocket stream where audio is streamed to Amazon Transcribe and the
     * transcription results are streamed to your application.
     * </p>
     * <p>
     * The following are encoded as headers:
     * </p>
     * <ul>
     * <li>
     * <p>
     * language-code
     * </p>
     * </li>
     * <li>
     * <p>
     * media-encoding
     * </p>
     * </li>
     * <li>
     * <p>
     * sample-rate
     * </p>
     * </li>
     * <li>
     * <p>
     * session-id
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information on streaming with Amazon Transcribe, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/streaming.html">Transcribing streaming audio</a>.
     * </p>
     *
     * @param startStreamTranscriptionRequest
     * @return A Java Future containing the result of the StartStreamTranscription operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code> or
     *         <code>StartMedicalStreamTranscription</code> operation was not valid. For example,
     *         <code>MediaEncoding</code> or <code>LanguageCode</code> used not valid values. Check the specified
     *         parameters and try your request again.</li>
     *         <li>LimitExceededException Your client has exceeded one of the Amazon Transcribe limits. This is
     *         typically the audio length limit. Break your audio stream into smaller chunks and try your request again.
     *         </li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing.</li>
     *         <li>ConflictException A new stream started with the same session ID. The current stream has been
     *         terminated.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable. Try your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.StartStreamTranscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/StartStreamTranscription"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<Void> startStreamTranscription(StartStreamTranscriptionRequest startStreamTranscriptionRequest,
            Publisher<AudioStream> requestStream, StartStreamTranscriptionResponseHandler asyncResponseHandler) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Starts a bidirectional HTTP/2 or WebSocket stream where audio is streamed to Amazon Transcribe and the
     * transcription results are streamed to your application.
     * </p>
     * <p>
     * The following are encoded as headers:
     * </p>
     * <ul>
     * <li>
     * <p>
     * language-code
     * </p>
     * </li>
     * <li>
     * <p>
     * media-encoding
     * </p>
     * </li>
     * <li>
     * <p>
     * sample-rate
     * </p>
     * </li>
     * <li>
     * <p>
     * session-id
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information on streaming with Amazon Transcribe, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/streaming.html">Transcribing streaming audio</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartStreamTranscriptionRequest.Builder} avoiding
     * the need to create one manually via {@link StartStreamTranscriptionRequest#builder()}
     * </p>
     *
     * @param startStreamTranscriptionRequest
     *        A {@link Consumer} that will call methods on {@link StartStreamTranscriptionRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the StartStreamTranscription operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code> or
     *         <code>StartMedicalStreamTranscription</code> operation was not valid. For example,
     *         <code>MediaEncoding</code> or <code>LanguageCode</code> used not valid values. Check the specified
     *         parameters and try your request again.</li>
     *         <li>LimitExceededException Your client has exceeded one of the Amazon Transcribe limits. This is
     *         typically the audio length limit. Break your audio stream into smaller chunks and try your request again.
     *         </li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing.</li>
     *         <li>ConflictException A new stream started with the same session ID. The current stream has been
     *         terminated.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable. Try your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.StartStreamTranscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/StartStreamTranscription"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<Void> startStreamTranscription(
            Consumer<StartStreamTranscriptionRequest.Builder> startStreamTranscriptionRequest,
            Publisher<AudioStream> requestStream, StartStreamTranscriptionResponseHandler asyncResponseHandler) {
        return startStreamTranscription(StartStreamTranscriptionRequest.builder().applyMutation(startStreamTranscriptionRequest)
                .build(), requestStream, asyncResponseHandler);
    }
}
