/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribestreaming;

import static software.amazon.awssdk.utils.FunctionalUtils.runAndLogError;

import java.nio.ByteBuffer;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.Consumer;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.auth.signer.EventStreamAws4Signer;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.client.handler.AwsClientHandlerUtils;
import software.amazon.awssdk.awscore.eventstream.EventStreamAsyncResponseTransformer;
import software.amazon.awssdk.awscore.eventstream.EventStreamTaggedUnionJsonMarshaller;
import software.amazon.awssdk.awscore.eventstream.EventStreamTaggedUnionPojoSupplier;
import software.amazon.awssdk.awscore.eventstream.RestEventStreamAsyncResponseTransformer;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPojoBuilder;
import software.amazon.awssdk.core.SdkResponse;
import software.amazon.awssdk.core.async.AsyncRequestBody;
import software.amazon.awssdk.core.async.SdkPublisher;
import software.amazon.awssdk.core.client.config.SdkAdvancedAsyncClientOption;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.AttachHttpMetadataResponseHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.protocol.VoidSdkResponse;
import software.amazon.awssdk.core.signer.Signer;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.transcribestreaming.model.AudioEvent;
import software.amazon.awssdk.services.transcribestreaming.model.AudioStream;
import software.amazon.awssdk.services.transcribestreaming.model.BadRequestException;
import software.amazon.awssdk.services.transcribestreaming.model.CallAnalyticsTranscriptResultStream;
import software.amazon.awssdk.services.transcribestreaming.model.ConflictException;
import software.amazon.awssdk.services.transcribestreaming.model.InternalFailureException;
import software.amazon.awssdk.services.transcribestreaming.model.LimitExceededException;
import software.amazon.awssdk.services.transcribestreaming.model.MedicalTranscriptResultStream;
import software.amazon.awssdk.services.transcribestreaming.model.ServiceUnavailableException;
import software.amazon.awssdk.services.transcribestreaming.model.StartCallAnalyticsStreamTranscriptionRequest;
import software.amazon.awssdk.services.transcribestreaming.model.StartCallAnalyticsStreamTranscriptionResponse;
import software.amazon.awssdk.services.transcribestreaming.model.StartCallAnalyticsStreamTranscriptionResponseHandler;
import software.amazon.awssdk.services.transcribestreaming.model.StartMedicalStreamTranscriptionRequest;
import software.amazon.awssdk.services.transcribestreaming.model.StartMedicalStreamTranscriptionResponse;
import software.amazon.awssdk.services.transcribestreaming.model.StartMedicalStreamTranscriptionResponseHandler;
import software.amazon.awssdk.services.transcribestreaming.model.StartStreamTranscriptionRequest;
import software.amazon.awssdk.services.transcribestreaming.model.StartStreamTranscriptionResponse;
import software.amazon.awssdk.services.transcribestreaming.model.StartStreamTranscriptionResponseHandler;
import software.amazon.awssdk.services.transcribestreaming.model.TranscribeStreamingException;
import software.amazon.awssdk.services.transcribestreaming.model.TranscribeStreamingRequest;
import software.amazon.awssdk.services.transcribestreaming.model.TranscriptResultStream;
import software.amazon.awssdk.services.transcribestreaming.model.audiostream.DefaultConfigurationEvent;
import software.amazon.awssdk.services.transcribestreaming.transform.AudioEventMarshaller;
import software.amazon.awssdk.services.transcribestreaming.transform.ConfigurationEventMarshaller;
import software.amazon.awssdk.services.transcribestreaming.transform.StartCallAnalyticsStreamTranscriptionRequestMarshaller;
import software.amazon.awssdk.services.transcribestreaming.transform.StartMedicalStreamTranscriptionRequestMarshaller;
import software.amazon.awssdk.services.transcribestreaming.transform.StartStreamTranscriptionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link TranscribeStreamingAsyncClient}.
 *
 * @see TranscribeStreamingAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTranscribeStreamingAsyncClient implements TranscribeStreamingAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultTranscribeStreamingAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final TranscribeStreamingServiceClientConfiguration serviceClientConfiguration;

    private final Executor executor;

    protected DefaultTranscribeStreamingAsyncClient(TranscribeStreamingServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        this.executor = clientConfiguration.option(SdkAdvancedAsyncClientOption.FUTURE_COMPLETION_EXECUTOR);
    }

    /**
     * <p>
     * Starts a bidirectional HTTP/2 or WebSocket stream where audio is streamed to Amazon Transcribe and the
     * transcription results are streamed to your application. Use this operation for <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/call-analytics.html">Call Analytics</a> transcriptions.
     * </p>
     * <p>
     * The following parameters are required:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>language-code</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>media-encoding</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sample-rate</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information on streaming with Amazon Transcribe, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/streaming.html">Transcribing streaming audio</a>.
     * </p>
     *
     * @param startCallAnalyticsStreamTranscriptionRequest
     * @return A Java Future containing the result of the StartCallAnalyticsStreamTranscription operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code>,
     *         <code>StartMedicalStreamTranscription</code>, or <code>StartCallAnalyticsStreamTranscription</code>
     *         operation was not valid. For example, <code>MediaEncoding</code> or <code>LanguageCode</code> used not
     *         valid values. Check the specified parameters and try your request again.</li>
     *         <li>LimitExceededException Your client has exceeded one of the Amazon Transcribe limits. This is
     *         typically the audio length limit. Break your audio stream into smaller chunks and try your request again.
     *         </li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing.</li>
     *         <li>ConflictException A new stream started with the same session ID. The current stream has been
     *         terminated.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable. Try your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.StartCallAnalyticsStreamTranscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/StartCallAnalyticsStreamTranscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<Void> startCallAnalyticsStreamTranscription(
            StartCallAnalyticsStreamTranscriptionRequest startCallAnalyticsStreamTranscriptionRequest,
            Publisher<AudioStream> requestStream, StartCallAnalyticsStreamTranscriptionResponseHandler asyncResponseHandler) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startCallAnalyticsStreamTranscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe Streaming");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartCallAnalyticsStreamTranscription");
            startCallAnalyticsStreamTranscriptionRequest = applySignerOverride(startCallAnalyticsStreamTranscriptionRequest,
                    EventStreamAws4Signer.create());
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartCallAnalyticsStreamTranscriptionResponse> responseHandler = new AttachHttpMetadataResponseHandler(
                    protocolFactory.createResponseHandler(operationMetadata,
                            StartCallAnalyticsStreamTranscriptionResponse::builder));

            HttpResponseHandler<SdkResponse> voidResponseHandler = protocolFactory.createResponseHandler(JsonOperationMetadata
                    .builder().isPayloadJson(false).hasStreamingSuccessResponse(true).build(), VoidSdkResponse::builder);

            HttpResponseHandler<? extends CallAnalyticsTranscriptResultStream> eventResponseHandler = protocolFactory
                    .createResponseHandler(
                            JsonOperationMetadata.builder().isPayloadJson(true).hasStreamingSuccessResponse(false).build(),
                            EventStreamTaggedUnionPojoSupplier
                                    .builder()
                                    .putSdkPojoSupplier("UtteranceEvent",
                                            CallAnalyticsTranscriptResultStream::utteranceEventBuilder)
                                    .putSdkPojoSupplier("CategoryEvent",
                                            CallAnalyticsTranscriptResultStream::categoryEventBuilder)
                                    .defaultSdkPojoSupplier(() -> new SdkPojoBuilder(CallAnalyticsTranscriptResultStream.UNKNOWN))
                                    .build());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            EventStreamTaggedUnionJsonMarshaller eventMarshaller = EventStreamTaggedUnionJsonMarshaller.builder()
                    .putMarshaller(AudioEvent.class, new AudioEventMarshaller(protocolFactory))
                    .putMarshaller(DefaultConfigurationEvent.class, new ConfigurationEventMarshaller(protocolFactory)).build();
            SdkPublisher<AudioStream> eventPublisher = SdkPublisher.adapt(requestStream);
            Publisher<ByteBuffer> adapted = eventPublisher.map(event -> eventMarshaller.marshall(event)).map(
                    AwsClientHandlerUtils::encodeEventStreamRequestToByteBuffer);
            CompletableFuture<Void> future = new CompletableFuture<>();
            EventStreamAsyncResponseTransformer<StartCallAnalyticsStreamTranscriptionResponse, CallAnalyticsTranscriptResultStream> asyncResponseTransformer = EventStreamAsyncResponseTransformer
                    .<StartCallAnalyticsStreamTranscriptionResponse, CallAnalyticsTranscriptResultStream> builder()
                    .eventStreamResponseHandler(asyncResponseHandler).eventResponseHandler(eventResponseHandler)
                    .initialResponseHandler(responseHandler).exceptionResponseHandler(errorResponseHandler).future(future)
                    .executor(executor).serviceName(serviceName()).build();
            RestEventStreamAsyncResponseTransformer<StartCallAnalyticsStreamTranscriptionResponse, CallAnalyticsTranscriptResultStream> restAsyncResponseTransformer = RestEventStreamAsyncResponseTransformer
                    .<StartCallAnalyticsStreamTranscriptionResponse, CallAnalyticsTranscriptResultStream> builder()
                    .eventStreamAsyncResponseTransformer(asyncResponseTransformer)
                    .eventStreamResponseHandler(asyncResponseHandler).build();

            CompletableFuture<Void> executeFuture = clientHandler
                    .execute(
                            new ClientExecutionParams<StartCallAnalyticsStreamTranscriptionRequest, StartCallAnalyticsStreamTranscriptionResponse>()
                                    .withOperationName("StartCallAnalyticsStreamTranscription")
                                    .withMarshaller(new StartCallAnalyticsStreamTranscriptionRequestMarshaller(protocolFactory))
                                    .withAsyncRequestBody(AsyncRequestBody.fromPublisher(adapted)).withFullDuplex(true)
                                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                                    .withMetricCollector(apiCallMetricCollector)
                                    .withInput(startCallAnalyticsStreamTranscriptionRequest), restAsyncResponseTransformer);
            CompletableFuture<Void> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    try {
                        asyncResponseHandler.exceptionOccurred(e);
                    } finally {
                        future.completeExceptionally(e);
                    }
                }
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return CompletableFutureUtils.forwardExceptionTo(future, executeFuture);
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseHandler.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a bidirectional HTTP/2 or WebSocket stream where audio is streamed to Amazon Transcribe Medical and the
     * transcription results are streamed to your application.
     * </p>
     * <p>
     * The following parameters are required:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>language-code</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>media-encoding</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sample-rate</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information on streaming with Amazon Transcribe Medical, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/streaming.html">Transcribing streaming audio</a>.
     * </p>
     *
     * @param startMedicalStreamTranscriptionRequest
     * @return A Java Future containing the result of the StartMedicalStreamTranscription operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code>,
     *         <code>StartMedicalStreamTranscription</code>, or <code>StartCallAnalyticsStreamTranscription</code>
     *         operation was not valid. For example, <code>MediaEncoding</code> or <code>LanguageCode</code> used not
     *         valid values. Check the specified parameters and try your request again.</li>
     *         <li>LimitExceededException Your client has exceeded one of the Amazon Transcribe limits. This is
     *         typically the audio length limit. Break your audio stream into smaller chunks and try your request again.
     *         </li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing.</li>
     *         <li>ConflictException A new stream started with the same session ID. The current stream has been
     *         terminated.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable. Try your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.StartMedicalStreamTranscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/StartMedicalStreamTranscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<Void> startMedicalStreamTranscription(
            StartMedicalStreamTranscriptionRequest startMedicalStreamTranscriptionRequest, Publisher<AudioStream> requestStream,
            StartMedicalStreamTranscriptionResponseHandler asyncResponseHandler) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startMedicalStreamTranscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe Streaming");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMedicalStreamTranscription");
            startMedicalStreamTranscriptionRequest = applySignerOverride(startMedicalStreamTranscriptionRequest,
                    EventStreamAws4Signer.create());
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartMedicalStreamTranscriptionResponse> responseHandler = new AttachHttpMetadataResponseHandler(
                    protocolFactory.createResponseHandler(operationMetadata, StartMedicalStreamTranscriptionResponse::builder));

            HttpResponseHandler<SdkResponse> voidResponseHandler = protocolFactory.createResponseHandler(JsonOperationMetadata
                    .builder().isPayloadJson(false).hasStreamingSuccessResponse(true).build(), VoidSdkResponse::builder);

            HttpResponseHandler<? extends MedicalTranscriptResultStream> eventResponseHandler = protocolFactory
                    .createResponseHandler(
                            JsonOperationMetadata.builder().isPayloadJson(true).hasStreamingSuccessResponse(false).build(),
                            EventStreamTaggedUnionPojoSupplier.builder()
                                    .putSdkPojoSupplier("TranscriptEvent", MedicalTranscriptResultStream::transcriptEventBuilder)
                                    .defaultSdkPojoSupplier(() -> new SdkPojoBuilder(MedicalTranscriptResultStream.UNKNOWN))
                                    .build());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            EventStreamTaggedUnionJsonMarshaller eventMarshaller = EventStreamTaggedUnionJsonMarshaller.builder()
                    .putMarshaller(AudioEvent.class, new AudioEventMarshaller(protocolFactory))
                    .putMarshaller(DefaultConfigurationEvent.class, new ConfigurationEventMarshaller(protocolFactory)).build();
            SdkPublisher<AudioStream> eventPublisher = SdkPublisher.adapt(requestStream);
            Publisher<ByteBuffer> adapted = eventPublisher.map(event -> eventMarshaller.marshall(event)).map(
                    AwsClientHandlerUtils::encodeEventStreamRequestToByteBuffer);
            CompletableFuture<Void> future = new CompletableFuture<>();
            EventStreamAsyncResponseTransformer<StartMedicalStreamTranscriptionResponse, MedicalTranscriptResultStream> asyncResponseTransformer = EventStreamAsyncResponseTransformer
                    .<StartMedicalStreamTranscriptionResponse, MedicalTranscriptResultStream> builder()
                    .eventStreamResponseHandler(asyncResponseHandler).eventResponseHandler(eventResponseHandler)
                    .initialResponseHandler(responseHandler).exceptionResponseHandler(errorResponseHandler).future(future)
                    .executor(executor).serviceName(serviceName()).build();
            RestEventStreamAsyncResponseTransformer<StartMedicalStreamTranscriptionResponse, MedicalTranscriptResultStream> restAsyncResponseTransformer = RestEventStreamAsyncResponseTransformer
                    .<StartMedicalStreamTranscriptionResponse, MedicalTranscriptResultStream> builder()
                    .eventStreamAsyncResponseTransformer(asyncResponseTransformer)
                    .eventStreamResponseHandler(asyncResponseHandler).build();

            CompletableFuture<Void> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<StartMedicalStreamTranscriptionRequest, StartMedicalStreamTranscriptionResponse>()
                            .withOperationName("StartMedicalStreamTranscription")
                            .withMarshaller(new StartMedicalStreamTranscriptionRequestMarshaller(protocolFactory))
                            .withAsyncRequestBody(AsyncRequestBody.fromPublisher(adapted)).withFullDuplex(true)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startMedicalStreamTranscriptionRequest),
                    restAsyncResponseTransformer);
            CompletableFuture<Void> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    try {
                        asyncResponseHandler.exceptionOccurred(e);
                    } finally {
                        future.completeExceptionally(e);
                    }
                }
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return CompletableFutureUtils.forwardExceptionTo(future, executeFuture);
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseHandler.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a bidirectional HTTP/2 or WebSocket stream where audio is streamed to Amazon Transcribe and the
     * transcription results are streamed to your application.
     * </p>
     * <p>
     * The following parameters are required:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>language-code</code> or <code>identify-language</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>media-encoding</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sample-rate</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information on streaming with Amazon Transcribe, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/streaming.html">Transcribing streaming audio</a>.
     * </p>
     *
     * @param startStreamTranscriptionRequest
     * @return A Java Future containing the result of the StartStreamTranscription operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code>,
     *         <code>StartMedicalStreamTranscription</code>, or <code>StartCallAnalyticsStreamTranscription</code>
     *         operation was not valid. For example, <code>MediaEncoding</code> or <code>LanguageCode</code> used not
     *         valid values. Check the specified parameters and try your request again.</li>
     *         <li>LimitExceededException Your client has exceeded one of the Amazon Transcribe limits. This is
     *         typically the audio length limit. Break your audio stream into smaller chunks and try your request again.
     *         </li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing.</li>
     *         <li>ConflictException A new stream started with the same session ID. The current stream has been
     *         terminated.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable. Try your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.StartStreamTranscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/StartStreamTranscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<Void> startStreamTranscription(StartStreamTranscriptionRequest startStreamTranscriptionRequest,
            Publisher<AudioStream> requestStream, StartStreamTranscriptionResponseHandler asyncResponseHandler) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startStreamTranscriptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe Streaming");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartStreamTranscription");
            startStreamTranscriptionRequest = applySignerOverride(startStreamTranscriptionRequest, EventStreamAws4Signer.create());
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartStreamTranscriptionResponse> responseHandler = new AttachHttpMetadataResponseHandler(
                    protocolFactory.createResponseHandler(operationMetadata, StartStreamTranscriptionResponse::builder));

            HttpResponseHandler<SdkResponse> voidResponseHandler = protocolFactory.createResponseHandler(JsonOperationMetadata
                    .builder().isPayloadJson(false).hasStreamingSuccessResponse(true).build(), VoidSdkResponse::builder);

            HttpResponseHandler<? extends TranscriptResultStream> eventResponseHandler = protocolFactory.createResponseHandler(
                    JsonOperationMetadata.builder().isPayloadJson(true).hasStreamingSuccessResponse(false).build(),
                    EventStreamTaggedUnionPojoSupplier.builder()
                            .putSdkPojoSupplier("TranscriptEvent", TranscriptResultStream::transcriptEventBuilder)
                            .defaultSdkPojoSupplier(() -> new SdkPojoBuilder(TranscriptResultStream.UNKNOWN)).build());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            EventStreamTaggedUnionJsonMarshaller eventMarshaller = EventStreamTaggedUnionJsonMarshaller.builder()
                    .putMarshaller(AudioEvent.class, new AudioEventMarshaller(protocolFactory))
                    .putMarshaller(DefaultConfigurationEvent.class, new ConfigurationEventMarshaller(protocolFactory)).build();
            SdkPublisher<AudioStream> eventPublisher = SdkPublisher.adapt(requestStream);
            Publisher<ByteBuffer> adapted = eventPublisher.map(event -> eventMarshaller.marshall(event)).map(
                    AwsClientHandlerUtils::encodeEventStreamRequestToByteBuffer);
            CompletableFuture<Void> future = new CompletableFuture<>();
            EventStreamAsyncResponseTransformer<StartStreamTranscriptionResponse, TranscriptResultStream> asyncResponseTransformer = EventStreamAsyncResponseTransformer
                    .<StartStreamTranscriptionResponse, TranscriptResultStream> builder()
                    .eventStreamResponseHandler(asyncResponseHandler).eventResponseHandler(eventResponseHandler)
                    .initialResponseHandler(responseHandler).exceptionResponseHandler(errorResponseHandler).future(future)
                    .executor(executor).serviceName(serviceName()).build();
            RestEventStreamAsyncResponseTransformer<StartStreamTranscriptionResponse, TranscriptResultStream> restAsyncResponseTransformer = RestEventStreamAsyncResponseTransformer
                    .<StartStreamTranscriptionResponse, TranscriptResultStream> builder()
                    .eventStreamAsyncResponseTransformer(asyncResponseTransformer)
                    .eventStreamResponseHandler(asyncResponseHandler).build();

            CompletableFuture<Void> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<StartStreamTranscriptionRequest, StartStreamTranscriptionResponse>()
                            .withOperationName("StartStreamTranscription")
                            .withMarshaller(new StartStreamTranscriptionRequestMarshaller(protocolFactory))
                            .withAsyncRequestBody(AsyncRequestBody.fromPublisher(adapted)).withFullDuplex(true)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startStreamTranscriptionRequest),
                    restAsyncResponseTransformer);
            CompletableFuture<Void> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    try {
                        asyncResponseHandler.exceptionOccurred(e);
                    } finally {
                        future.completeExceptionally(e);
                    }
                }
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return CompletableFutureUtils.forwardExceptionTo(future, executeFuture);
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseHandler.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final TranscribeStreamingServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(TranscribeStreamingException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(429).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends TranscribeStreamingRequest> T applySignerOverride(T request, Signer signer) {
        if (request.overrideConfiguration().flatMap(c -> c.signer()).isPresent()) {
            return request;
        }
        Consumer<AwsRequestOverrideConfiguration.Builder> signerOverride = b -> b.signer(signer).build();
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(signerOverride).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(signerOverride).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private static boolean isSignerOverridden(SdkClientConfiguration clientConfiguration) {
        return Boolean.TRUE.equals(clientConfiguration.option(SdkClientOption.SIGNER_OVERRIDDEN));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
