/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribestreaming;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.reactivestreams.Publisher;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.services.transcribestreaming.model.AudioStream;
import software.amazon.awssdk.services.transcribestreaming.model.GetMedicalScribeStreamRequest;
import software.amazon.awssdk.services.transcribestreaming.model.GetMedicalScribeStreamResponse;
import software.amazon.awssdk.services.transcribestreaming.model.MedicalScribeInputStream;
import software.amazon.awssdk.services.transcribestreaming.model.StartCallAnalyticsStreamTranscriptionRequest;
import software.amazon.awssdk.services.transcribestreaming.model.StartCallAnalyticsStreamTranscriptionResponseHandler;
import software.amazon.awssdk.services.transcribestreaming.model.StartMedicalScribeStreamRequest;
import software.amazon.awssdk.services.transcribestreaming.model.StartMedicalScribeStreamResponseHandler;
import software.amazon.awssdk.services.transcribestreaming.model.StartMedicalStreamTranscriptionRequest;
import software.amazon.awssdk.services.transcribestreaming.model.StartMedicalStreamTranscriptionResponseHandler;
import software.amazon.awssdk.services.transcribestreaming.model.StartStreamTranscriptionRequest;
import software.amazon.awssdk.services.transcribestreaming.model.StartStreamTranscriptionResponseHandler;

/**
 * Service client for accessing Amazon Transcribe Streaming Service asynchronously. This can be created using the static
 * {@link #builder()} method.The asynchronous client performs non-blocking I/O when configured with any
 * {@code SdkAsyncHttpClient} supported in the SDK. However, full non-blocking is not guaranteed as the async client may
 * perform blocking calls in some cases such as credentials retrieval and endpoint discovery as part of the async API
 * call.
 *
 * <p>
 * Amazon Transcribe streaming offers four main types of real-time transcription: <b>Standard</b>, <b>Medical</b>,
 * <b>Call Analytics</b>, and <b>Health Scribe</b>.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>Standard transcriptions</b> are the most common option. Refer to for details.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Medical transcriptions</b> are tailored to medical professionals and incorporate medical terms. A common use case
 * for this service is transcribing doctor-patient dialogue in real time, so doctors can focus on their patient instead
 * of taking notes. Refer to for details.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Call Analytics transcriptions</b> are designed for use with call center audio on two different channels; if you're
 * looking for insight into customer service calls, use this option. Refer to for details.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>HealthScribe transcriptions</b> are designed to automatically create clinical notes from patient-clinician
 * conversations using generative AI. Refer to [here] for details.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface TranscribeStreamingAsyncClient extends AwsClient {
    String SERVICE_NAME = "transcribe";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "transcribestreaming";

    /**
     * <p>
     * Provides details about the specified Amazon Web Services HealthScribe streaming session. To view the status of
     * the streaming session, check the <code>StreamStatus</code> field in the response. To get the details of
     * post-stream analytics, including its status, check the <code>PostStreamAnalyticsResult</code> field in the
     * response.
     * </p>
     *
     * @param getMedicalScribeStreamRequest
     * @return A Java Future containing the result of the GetMedicalScribeStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code>,
     *         <code>StartMedicalStreamTranscription</code>, or <code>StartCallAnalyticsStreamTranscription</code>
     *         operation was not valid. For example, <code>MediaEncoding</code> or <code>LanguageCode</code> used
     *         unsupported values. Check the specified parameters and try your request again.</li>
     *         <li>LimitExceededException Your client has exceeded one of the Amazon Transcribe limits. This is
     *         typically the audio length limit. Break your audio stream into smaller chunks and try your request again.
     *         </li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing.</li>
     *         <li>ResourceNotFoundException The request references a resource which doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.GetMedicalScribeStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/GetMedicalScribeStream"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetMedicalScribeStreamResponse> getMedicalScribeStream(
            GetMedicalScribeStreamRequest getMedicalScribeStreamRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Provides details about the specified Amazon Web Services HealthScribe streaming session. To view the status of
     * the streaming session, check the <code>StreamStatus</code> field in the response. To get the details of
     * post-stream analytics, including its status, check the <code>PostStreamAnalyticsResult</code> field in the
     * response.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetMedicalScribeStreamRequest.Builder} avoiding the
     * need to create one manually via {@link GetMedicalScribeStreamRequest#builder()}
     * </p>
     *
     * @param getMedicalScribeStreamRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.transcribestreaming.model.GetMedicalScribeStreamRequest.Builder} to
     *        create a request.
     * @return A Java Future containing the result of the GetMedicalScribeStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code>,
     *         <code>StartMedicalStreamTranscription</code>, or <code>StartCallAnalyticsStreamTranscription</code>
     *         operation was not valid. For example, <code>MediaEncoding</code> or <code>LanguageCode</code> used
     *         unsupported values. Check the specified parameters and try your request again.</li>
     *         <li>LimitExceededException Your client has exceeded one of the Amazon Transcribe limits. This is
     *         typically the audio length limit. Break your audio stream into smaller chunks and try your request again.
     *         </li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing.</li>
     *         <li>ResourceNotFoundException The request references a resource which doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.GetMedicalScribeStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/GetMedicalScribeStream"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetMedicalScribeStreamResponse> getMedicalScribeStream(
            Consumer<GetMedicalScribeStreamRequest.Builder> getMedicalScribeStreamRequest) {
        return getMedicalScribeStream(GetMedicalScribeStreamRequest.builder().applyMutation(getMedicalScribeStreamRequest)
                .build());
    }

    /**
     * <p>
     * Starts a bidirectional HTTP/2 or WebSocket stream where audio is streamed to Amazon Transcribe and the
     * transcription results are streamed to your application. Use this operation for <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/call-analytics.html">Call Analytics</a> transcriptions.
     * </p>
     * <p>
     * The following parameters are required:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>language-code</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>media-encoding</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sample-rate</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information on streaming with Amazon Transcribe, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/streaming.html">Transcribing streaming audio</a>.
     * </p>
     *
     * @param startCallAnalyticsStreamTranscriptionRequest
     * @return A Java Future containing the result of the StartCallAnalyticsStreamTranscription operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code>,
     *         <code>StartMedicalStreamTranscription</code>, or <code>StartCallAnalyticsStreamTranscription</code>
     *         operation was not valid. For example, <code>MediaEncoding</code> or <code>LanguageCode</code> used
     *         unsupported values. Check the specified parameters and try your request again.</li>
     *         <li>LimitExceededException Your client has exceeded one of the Amazon Transcribe limits. This is
     *         typically the audio length limit. Break your audio stream into smaller chunks and try your request again.
     *         </li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing.</li>
     *         <li>ConflictException A new stream started with the same session ID. The current stream has been
     *         terminated.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable. Try your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.StartCallAnalyticsStreamTranscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/StartCallAnalyticsStreamTranscription"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<Void> startCallAnalyticsStreamTranscription(
            StartCallAnalyticsStreamTranscriptionRequest startCallAnalyticsStreamTranscriptionRequest,
            Publisher<AudioStream> requestStream, StartCallAnalyticsStreamTranscriptionResponseHandler asyncResponseHandler) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Starts a bidirectional HTTP/2 or WebSocket stream where audio is streamed to Amazon Transcribe and the
     * transcription results are streamed to your application. Use this operation for <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/call-analytics.html">Call Analytics</a> transcriptions.
     * </p>
     * <p>
     * The following parameters are required:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>language-code</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>media-encoding</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sample-rate</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information on streaming with Amazon Transcribe, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/streaming.html">Transcribing streaming audio</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the
     * {@link StartCallAnalyticsStreamTranscriptionRequest.Builder} avoiding the need to create one manually via
     * {@link StartCallAnalyticsStreamTranscriptionRequest#builder()}
     * </p>
     *
     * @param startCallAnalyticsStreamTranscriptionRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.transcribestreaming.model.StartCallAnalyticsStreamTranscriptionRequest.Builder}
     *        to create a request.
     * @return A Java Future containing the result of the StartCallAnalyticsStreamTranscription operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code>,
     *         <code>StartMedicalStreamTranscription</code>, or <code>StartCallAnalyticsStreamTranscription</code>
     *         operation was not valid. For example, <code>MediaEncoding</code> or <code>LanguageCode</code> used
     *         unsupported values. Check the specified parameters and try your request again.</li>
     *         <li>LimitExceededException Your client has exceeded one of the Amazon Transcribe limits. This is
     *         typically the audio length limit. Break your audio stream into smaller chunks and try your request again.
     *         </li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing.</li>
     *         <li>ConflictException A new stream started with the same session ID. The current stream has been
     *         terminated.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable. Try your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.StartCallAnalyticsStreamTranscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/StartCallAnalyticsStreamTranscription"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<Void> startCallAnalyticsStreamTranscription(
            Consumer<StartCallAnalyticsStreamTranscriptionRequest.Builder> startCallAnalyticsStreamTranscriptionRequest,
            Publisher<AudioStream> requestStream, StartCallAnalyticsStreamTranscriptionResponseHandler asyncResponseHandler) {
        return startCallAnalyticsStreamTranscription(
                StartCallAnalyticsStreamTranscriptionRequest.builder()
                        .applyMutation(startCallAnalyticsStreamTranscriptionRequest).build(), requestStream, asyncResponseHandler);
    }

    /**
     * <p>
     * Starts a bidirectional HTTP/2 stream, where audio is streamed to Amazon Web Services HealthScribe and the
     * transcription results are streamed to your application.
     * </p>
     * <p>
     * When you start a stream, you first specify the stream configuration in a
     * <code>MedicalScribeConfigurationEvent</code>. This event includes channel definitions, encryption settings, and
     * post-stream analytics settings, such as the output configuration for aggregated transcript and clinical note
     * generation. These are additional streaming session configurations beyond those provided in your initial start
     * request headers. Whether you are starting a new session or resuming an existing session, your first event must be
     * a <code>MedicalScribeConfigurationEvent</code>.
     * </p>
     * <p>
     * After you send a <code>MedicalScribeConfigurationEvent</code>, you start <code>AudioEvents</code> and Amazon Web
     * Services HealthScribe responds with real-time transcription results. When you are finished, to start processing
     * the results with the post-stream analytics, send a <code>MedicalScribeSessionControlEvent</code> with a
     * <code>Type</code> of <code>END_OF_SESSION</code> and Amazon Web Services HealthScribe starts the analytics.
     * </p>
     * <p>
     * You can pause or resume streaming. To pause streaming, complete the input stream without sending the
     * <code>MedicalScribeSessionControlEvent</code>. To resume streaming, call the
     * <code>StartMedicalScribeStream</code> and specify the same SessionId you used to start the stream.
     * </p>
     * <p>
     * The following parameters are required:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>language-code</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>media-encoding</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>media-sample-rate-hertz</code>
     * </p>
     * </li>
     * </ul>
     * <p/>
     * <p>
     * For more information on streaming with Amazon Web Services HealthScribe, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/health-scribe-streaming.html">Amazon Web Services
     * HealthScribe</a>.
     * </p>
     *
     * @param startMedicalScribeStreamRequest
     * @return A Java Future containing the result of the StartMedicalScribeStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code>,
     *         <code>StartMedicalStreamTranscription</code>, or <code>StartCallAnalyticsStreamTranscription</code>
     *         operation was not valid. For example, <code>MediaEncoding</code> or <code>LanguageCode</code> used
     *         unsupported values. Check the specified parameters and try your request again.</li>
     *         <li>LimitExceededException Your client has exceeded one of the Amazon Transcribe limits. This is
     *         typically the audio length limit. Break your audio stream into smaller chunks and try your request again.
     *         </li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing.</li>
     *         <li>ConflictException A new stream started with the same session ID. The current stream has been
     *         terminated.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable. Try your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.StartMedicalScribeStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/StartMedicalScribeStream"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<Void> startMedicalScribeStream(StartMedicalScribeStreamRequest startMedicalScribeStreamRequest,
            Publisher<MedicalScribeInputStream> requestStream, StartMedicalScribeStreamResponseHandler asyncResponseHandler) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Starts a bidirectional HTTP/2 stream, where audio is streamed to Amazon Web Services HealthScribe and the
     * transcription results are streamed to your application.
     * </p>
     * <p>
     * When you start a stream, you first specify the stream configuration in a
     * <code>MedicalScribeConfigurationEvent</code>. This event includes channel definitions, encryption settings, and
     * post-stream analytics settings, such as the output configuration for aggregated transcript and clinical note
     * generation. These are additional streaming session configurations beyond those provided in your initial start
     * request headers. Whether you are starting a new session or resuming an existing session, your first event must be
     * a <code>MedicalScribeConfigurationEvent</code>.
     * </p>
     * <p>
     * After you send a <code>MedicalScribeConfigurationEvent</code>, you start <code>AudioEvents</code> and Amazon Web
     * Services HealthScribe responds with real-time transcription results. When you are finished, to start processing
     * the results with the post-stream analytics, send a <code>MedicalScribeSessionControlEvent</code> with a
     * <code>Type</code> of <code>END_OF_SESSION</code> and Amazon Web Services HealthScribe starts the analytics.
     * </p>
     * <p>
     * You can pause or resume streaming. To pause streaming, complete the input stream without sending the
     * <code>MedicalScribeSessionControlEvent</code>. To resume streaming, call the
     * <code>StartMedicalScribeStream</code> and specify the same SessionId you used to start the stream.
     * </p>
     * <p>
     * The following parameters are required:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>language-code</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>media-encoding</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>media-sample-rate-hertz</code>
     * </p>
     * </li>
     * </ul>
     * <p/>
     * <p>
     * For more information on streaming with Amazon Web Services HealthScribe, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/health-scribe-streaming.html">Amazon Web Services
     * HealthScribe</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartMedicalScribeStreamRequest.Builder} avoiding
     * the need to create one manually via {@link StartMedicalScribeStreamRequest#builder()}
     * </p>
     *
     * @param startMedicalScribeStreamRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.transcribestreaming.model.StartMedicalScribeStreamRequest.Builder}
     *        to create a request.
     * @return A Java Future containing the result of the StartMedicalScribeStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code>,
     *         <code>StartMedicalStreamTranscription</code>, or <code>StartCallAnalyticsStreamTranscription</code>
     *         operation was not valid. For example, <code>MediaEncoding</code> or <code>LanguageCode</code> used
     *         unsupported values. Check the specified parameters and try your request again.</li>
     *         <li>LimitExceededException Your client has exceeded one of the Amazon Transcribe limits. This is
     *         typically the audio length limit. Break your audio stream into smaller chunks and try your request again.
     *         </li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing.</li>
     *         <li>ConflictException A new stream started with the same session ID. The current stream has been
     *         terminated.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable. Try your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.StartMedicalScribeStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/StartMedicalScribeStream"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<Void> startMedicalScribeStream(
            Consumer<StartMedicalScribeStreamRequest.Builder> startMedicalScribeStreamRequest,
            Publisher<MedicalScribeInputStream> requestStream, StartMedicalScribeStreamResponseHandler asyncResponseHandler) {
        return startMedicalScribeStream(StartMedicalScribeStreamRequest.builder().applyMutation(startMedicalScribeStreamRequest)
                .build(), requestStream, asyncResponseHandler);
    }

    /**
     * <p>
     * Starts a bidirectional HTTP/2 or WebSocket stream where audio is streamed to Amazon Transcribe Medical and the
     * transcription results are streamed to your application.
     * </p>
     * <p>
     * The following parameters are required:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>language-code</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>media-encoding</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sample-rate</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information on streaming with Amazon Transcribe Medical, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/streaming.html">Transcribing streaming audio</a>.
     * </p>
     *
     * @param startMedicalStreamTranscriptionRequest
     * @return A Java Future containing the result of the StartMedicalStreamTranscription operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code>,
     *         <code>StartMedicalStreamTranscription</code>, or <code>StartCallAnalyticsStreamTranscription</code>
     *         operation was not valid. For example, <code>MediaEncoding</code> or <code>LanguageCode</code> used
     *         unsupported values. Check the specified parameters and try your request again.</li>
     *         <li>LimitExceededException Your client has exceeded one of the Amazon Transcribe limits. This is
     *         typically the audio length limit. Break your audio stream into smaller chunks and try your request again.
     *         </li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing.</li>
     *         <li>ConflictException A new stream started with the same session ID. The current stream has been
     *         terminated.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable. Try your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.StartMedicalStreamTranscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/StartMedicalStreamTranscription"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<Void> startMedicalStreamTranscription(
            StartMedicalStreamTranscriptionRequest startMedicalStreamTranscriptionRequest, Publisher<AudioStream> requestStream,
            StartMedicalStreamTranscriptionResponseHandler asyncResponseHandler) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Starts a bidirectional HTTP/2 or WebSocket stream where audio is streamed to Amazon Transcribe Medical and the
     * transcription results are streamed to your application.
     * </p>
     * <p>
     * The following parameters are required:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>language-code</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>media-encoding</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sample-rate</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information on streaming with Amazon Transcribe Medical, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/streaming.html">Transcribing streaming audio</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartMedicalStreamTranscriptionRequest.Builder}
     * avoiding the need to create one manually via {@link StartMedicalStreamTranscriptionRequest#builder()}
     * </p>
     *
     * @param startMedicalStreamTranscriptionRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.transcribestreaming.model.StartMedicalStreamTranscriptionRequest.Builder}
     *        to create a request.
     * @return A Java Future containing the result of the StartMedicalStreamTranscription operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code>,
     *         <code>StartMedicalStreamTranscription</code>, or <code>StartCallAnalyticsStreamTranscription</code>
     *         operation was not valid. For example, <code>MediaEncoding</code> or <code>LanguageCode</code> used
     *         unsupported values. Check the specified parameters and try your request again.</li>
     *         <li>LimitExceededException Your client has exceeded one of the Amazon Transcribe limits. This is
     *         typically the audio length limit. Break your audio stream into smaller chunks and try your request again.
     *         </li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing.</li>
     *         <li>ConflictException A new stream started with the same session ID. The current stream has been
     *         terminated.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable. Try your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.StartMedicalStreamTranscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/StartMedicalStreamTranscription"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<Void> startMedicalStreamTranscription(
            Consumer<StartMedicalStreamTranscriptionRequest.Builder> startMedicalStreamTranscriptionRequest,
            Publisher<AudioStream> requestStream, StartMedicalStreamTranscriptionResponseHandler asyncResponseHandler) {
        return startMedicalStreamTranscription(
                StartMedicalStreamTranscriptionRequest.builder().applyMutation(startMedicalStreamTranscriptionRequest).build(),
                requestStream, asyncResponseHandler);
    }

    /**
     * <p>
     * Starts a bidirectional HTTP/2 or WebSocket stream where audio is streamed to Amazon Transcribe and the
     * transcription results are streamed to your application.
     * </p>
     * <p>
     * The following parameters are required:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>language-code</code> or <code>identify-language</code> or <code>identify-multiple-language</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>media-encoding</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sample-rate</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information on streaming with Amazon Transcribe, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/streaming.html">Transcribing streaming audio</a>.
     * </p>
     *
     * @param startStreamTranscriptionRequest
     * @return A Java Future containing the result of the StartStreamTranscription operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code>,
     *         <code>StartMedicalStreamTranscription</code>, or <code>StartCallAnalyticsStreamTranscription</code>
     *         operation was not valid. For example, <code>MediaEncoding</code> or <code>LanguageCode</code> used
     *         unsupported values. Check the specified parameters and try your request again.</li>
     *         <li>LimitExceededException Your client has exceeded one of the Amazon Transcribe limits. This is
     *         typically the audio length limit. Break your audio stream into smaller chunks and try your request again.
     *         </li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing.</li>
     *         <li>ConflictException A new stream started with the same session ID. The current stream has been
     *         terminated.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable. Try your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.StartStreamTranscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/StartStreamTranscription"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<Void> startStreamTranscription(StartStreamTranscriptionRequest startStreamTranscriptionRequest,
            Publisher<AudioStream> requestStream, StartStreamTranscriptionResponseHandler asyncResponseHandler) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Starts a bidirectional HTTP/2 or WebSocket stream where audio is streamed to Amazon Transcribe and the
     * transcription results are streamed to your application.
     * </p>
     * <p>
     * The following parameters are required:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>language-code</code> or <code>identify-language</code> or <code>identify-multiple-language</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>media-encoding</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sample-rate</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information on streaming with Amazon Transcribe, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/streaming.html">Transcribing streaming audio</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartStreamTranscriptionRequest.Builder} avoiding
     * the need to create one manually via {@link StartStreamTranscriptionRequest#builder()}
     * </p>
     *
     * @param startStreamTranscriptionRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.transcribestreaming.model.StartStreamTranscriptionRequest.Builder}
     *        to create a request.
     * @return A Java Future containing the result of the StartStreamTranscription operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code>,
     *         <code>StartMedicalStreamTranscription</code>, or <code>StartCallAnalyticsStreamTranscription</code>
     *         operation was not valid. For example, <code>MediaEncoding</code> or <code>LanguageCode</code> used
     *         unsupported values. Check the specified parameters and try your request again.</li>
     *         <li>LimitExceededException Your client has exceeded one of the Amazon Transcribe limits. This is
     *         typically the audio length limit. Break your audio stream into smaller chunks and try your request again.
     *         </li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing.</li>
     *         <li>ConflictException A new stream started with the same session ID. The current stream has been
     *         terminated.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable. Try your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.StartStreamTranscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/StartStreamTranscription"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<Void> startStreamTranscription(
            Consumer<StartStreamTranscriptionRequest.Builder> startStreamTranscriptionRequest,
            Publisher<AudioStream> requestStream, StartStreamTranscriptionResponseHandler asyncResponseHandler) {
        return startStreamTranscription(StartStreamTranscriptionRequest.builder().applyMutation(startStreamTranscriptionRequest)
                .build(), requestStream, asyncResponseHandler);
    }

    @Override
    default TranscribeStreamingServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link TranscribeStreamingAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static TranscribeStreamingAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link TranscribeStreamingAsyncClient}.
     */
    static TranscribeStreamingAsyncClientBuilder builder() {
        return new DefaultTranscribeStreamingAsyncClientBuilder();
    }
}
