/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribestreaming;

import static software.amazon.awssdk.utils.FunctionalUtils.runAndLogError;

import java.nio.ByteBuffer;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.Consumer;
import java.util.function.Function;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.client.handler.AwsClientHandlerUtils;
import software.amazon.awssdk.awscore.eventstream.EventStreamAsyncResponseTransformer;
import software.amazon.awssdk.awscore.eventstream.EventStreamTaggedUnionJsonMarshaller;
import software.amazon.awssdk.awscore.eventstream.EventStreamTaggedUnionPojoSupplier;
import software.amazon.awssdk.awscore.eventstream.RestEventStreamAsyncResponseTransformer;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkPojoBuilder;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.SdkResponse;
import software.amazon.awssdk.core.async.AsyncRequestBody;
import software.amazon.awssdk.core.async.SdkPublisher;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkAdvancedAsyncClientOption;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.AttachHttpMetadataResponseHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.protocol.VoidSdkResponse;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.transcribestreaming.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.transcribestreaming.internal.TranscribeStreamingServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.transcribestreaming.model.AudioEvent;
import software.amazon.awssdk.services.transcribestreaming.model.AudioStream;
import software.amazon.awssdk.services.transcribestreaming.model.BadRequestException;
import software.amazon.awssdk.services.transcribestreaming.model.CallAnalyticsTranscriptResultStream;
import software.amazon.awssdk.services.transcribestreaming.model.ConflictException;
import software.amazon.awssdk.services.transcribestreaming.model.GetMedicalScribeStreamRequest;
import software.amazon.awssdk.services.transcribestreaming.model.GetMedicalScribeStreamResponse;
import software.amazon.awssdk.services.transcribestreaming.model.InternalFailureException;
import software.amazon.awssdk.services.transcribestreaming.model.LimitExceededException;
import software.amazon.awssdk.services.transcribestreaming.model.MedicalScribeInputStream;
import software.amazon.awssdk.services.transcribestreaming.model.MedicalScribeResultStream;
import software.amazon.awssdk.services.transcribestreaming.model.MedicalTranscriptResultStream;
import software.amazon.awssdk.services.transcribestreaming.model.ResourceNotFoundException;
import software.amazon.awssdk.services.transcribestreaming.model.ServiceUnavailableException;
import software.amazon.awssdk.services.transcribestreaming.model.StartCallAnalyticsStreamTranscriptionRequest;
import software.amazon.awssdk.services.transcribestreaming.model.StartCallAnalyticsStreamTranscriptionResponse;
import software.amazon.awssdk.services.transcribestreaming.model.StartCallAnalyticsStreamTranscriptionResponseHandler;
import software.amazon.awssdk.services.transcribestreaming.model.StartMedicalScribeStreamRequest;
import software.amazon.awssdk.services.transcribestreaming.model.StartMedicalScribeStreamResponse;
import software.amazon.awssdk.services.transcribestreaming.model.StartMedicalScribeStreamResponseHandler;
import software.amazon.awssdk.services.transcribestreaming.model.StartMedicalStreamTranscriptionRequest;
import software.amazon.awssdk.services.transcribestreaming.model.StartMedicalStreamTranscriptionResponse;
import software.amazon.awssdk.services.transcribestreaming.model.StartMedicalStreamTranscriptionResponseHandler;
import software.amazon.awssdk.services.transcribestreaming.model.StartStreamTranscriptionRequest;
import software.amazon.awssdk.services.transcribestreaming.model.StartStreamTranscriptionResponse;
import software.amazon.awssdk.services.transcribestreaming.model.StartStreamTranscriptionResponseHandler;
import software.amazon.awssdk.services.transcribestreaming.model.TranscribeStreamingException;
import software.amazon.awssdk.services.transcribestreaming.model.TranscriptResultStream;
import software.amazon.awssdk.services.transcribestreaming.model.audiostream.DefaultConfigurationEvent;
import software.amazon.awssdk.services.transcribestreaming.model.medicalscribeinputstream.DefaultAudioEvent;
import software.amazon.awssdk.services.transcribestreaming.model.medicalscribeinputstream.DefaultSessionControlEvent;
import software.amazon.awssdk.services.transcribestreaming.transform.AudioEventMarshaller;
import software.amazon.awssdk.services.transcribestreaming.transform.ConfigurationEventMarshaller;
import software.amazon.awssdk.services.transcribestreaming.transform.GetMedicalScribeStreamRequestMarshaller;
import software.amazon.awssdk.services.transcribestreaming.transform.MedicalScribeAudioEventMarshaller;
import software.amazon.awssdk.services.transcribestreaming.transform.MedicalScribeConfigurationEventMarshaller;
import software.amazon.awssdk.services.transcribestreaming.transform.MedicalScribeSessionControlEventMarshaller;
import software.amazon.awssdk.services.transcribestreaming.transform.StartCallAnalyticsStreamTranscriptionRequestMarshaller;
import software.amazon.awssdk.services.transcribestreaming.transform.StartMedicalScribeStreamRequestMarshaller;
import software.amazon.awssdk.services.transcribestreaming.transform.StartMedicalStreamTranscriptionRequestMarshaller;
import software.amazon.awssdk.services.transcribestreaming.transform.StartStreamTranscriptionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link TranscribeStreamingAsyncClient}.
 *
 * @see TranscribeStreamingAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTranscribeStreamingAsyncClient implements TranscribeStreamingAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultTranscribeStreamingAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final Executor executor;

    protected DefaultTranscribeStreamingAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "Transcribe_Streaming" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        this.executor = clientConfiguration.option(SdkAdvancedAsyncClientOption.FUTURE_COMPLETION_EXECUTOR);
    }

    /**
     * <p>
     * Provides details about the specified Amazon Web Services HealthScribe streaming session. To view the status of
     * the streaming session, check the <code>StreamStatus</code> field in the response. To get the details of
     * post-stream analytics, including its status, check the <code>PostStreamAnalyticsResult</code> field in the
     * response.
     * </p>
     *
     * @param getMedicalScribeStreamRequest
     * @return A Java Future containing the result of the GetMedicalScribeStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code>,
     *         <code>StartMedicalStreamTranscription</code>, or <code>StartCallAnalyticsStreamTranscription</code>
     *         operation was not valid. For example, <code>MediaEncoding</code> or <code>LanguageCode</code> used
     *         unsupported values. Check the specified parameters and try your request again.</li>
     *         <li>LimitExceededException Your client has exceeded one of the Amazon Transcribe limits. This is
     *         typically the audio length limit. Break your audio stream into smaller chunks and try your request again.
     *         </li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing.</li>
     *         <li>ResourceNotFoundException The request references a resource which doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.GetMedicalScribeStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/GetMedicalScribeStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMedicalScribeStreamResponse> getMedicalScribeStream(
            GetMedicalScribeStreamRequest getMedicalScribeStreamRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMedicalScribeStreamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMedicalScribeStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe Streaming");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMedicalScribeStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMedicalScribeStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMedicalScribeStreamResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetMedicalScribeStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMedicalScribeStreamRequest, GetMedicalScribeStreamResponse>()
                            .withOperationName("GetMedicalScribeStream").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetMedicalScribeStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getMedicalScribeStreamRequest));
            CompletableFuture<GetMedicalScribeStreamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a bidirectional HTTP/2 or WebSocket stream where audio is streamed to Amazon Transcribe and the
     * transcription results are streamed to your application. Use this operation for <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/call-analytics.html">Call Analytics</a> transcriptions.
     * </p>
     * <p>
     * The following parameters are required:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>language-code</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>media-encoding</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sample-rate</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information on streaming with Amazon Transcribe, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/streaming.html">Transcribing streaming audio</a>.
     * </p>
     *
     * @param startCallAnalyticsStreamTranscriptionRequest
     * @return A Java Future containing the result of the StartCallAnalyticsStreamTranscription operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code>,
     *         <code>StartMedicalStreamTranscription</code>, or <code>StartCallAnalyticsStreamTranscription</code>
     *         operation was not valid. For example, <code>MediaEncoding</code> or <code>LanguageCode</code> used
     *         unsupported values. Check the specified parameters and try your request again.</li>
     *         <li>LimitExceededException Your client has exceeded one of the Amazon Transcribe limits. This is
     *         typically the audio length limit. Break your audio stream into smaller chunks and try your request again.
     *         </li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing.</li>
     *         <li>ConflictException A new stream started with the same session ID. The current stream has been
     *         terminated.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable. Try your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.StartCallAnalyticsStreamTranscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/StartCallAnalyticsStreamTranscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<Void> startCallAnalyticsStreamTranscription(
            StartCallAnalyticsStreamTranscriptionRequest startCallAnalyticsStreamTranscriptionRequest,
            Publisher<AudioStream> requestStream, StartCallAnalyticsStreamTranscriptionResponseHandler asyncResponseHandler) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startCallAnalyticsStreamTranscriptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startCallAnalyticsStreamTranscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe Streaming");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartCallAnalyticsStreamTranscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartCallAnalyticsStreamTranscriptionResponse> responseHandler = new AttachHttpMetadataResponseHandler(
                    protocolFactory.createResponseHandler(operationMetadata,
                            StartCallAnalyticsStreamTranscriptionResponse::builder));

            HttpResponseHandler<SdkResponse> voidResponseHandler = protocolFactory.createResponseHandler(JsonOperationMetadata
                    .builder().isPayloadJson(false).hasStreamingSuccessResponse(true).build(), VoidSdkResponse::builder);

            HttpResponseHandler<? extends CallAnalyticsTranscriptResultStream> eventResponseHandler = protocolFactory
                    .createResponseHandler(
                            JsonOperationMetadata.builder().isPayloadJson(true).hasStreamingSuccessResponse(false).build(),
                            EventStreamTaggedUnionPojoSupplier
                                    .builder()
                                    .putSdkPojoSupplier("UtteranceEvent",
                                            CallAnalyticsTranscriptResultStream::utteranceEventBuilder)
                                    .putSdkPojoSupplier("CategoryEvent",
                                            CallAnalyticsTranscriptResultStream::categoryEventBuilder)
                                    .defaultSdkPojoSupplier(() -> new SdkPojoBuilder(CallAnalyticsTranscriptResultStream.UNKNOWN))
                                    .build());

            Function<String, Optional<ExceptionMetadata>> eventstreamExceptionMetadataMapper = errorCode -> {
                switch (errorCode) {
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };

            HttpResponseHandler<AwsServiceException> errorEventResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, eventstreamExceptionMetadataMapper);

            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);
            EventStreamTaggedUnionJsonMarshaller eventMarshaller = EventStreamTaggedUnionJsonMarshaller.builder()
                    .putMarshaller(AudioEvent.class, new AudioEventMarshaller(protocolFactory))
                    .putMarshaller(DefaultConfigurationEvent.class, new ConfigurationEventMarshaller(protocolFactory)).build();
            SdkPublisher<AudioStream> eventPublisher = SdkPublisher.adapt(requestStream);
            Publisher<ByteBuffer> adapted = eventPublisher.map(event -> eventMarshaller.marshall(event)).map(
                    AwsClientHandlerUtils::encodeEventStreamRequestToByteBuffer);
            CompletableFuture<Void> future = new CompletableFuture<>();
            EventStreamAsyncResponseTransformer<StartCallAnalyticsStreamTranscriptionResponse, CallAnalyticsTranscriptResultStream> asyncResponseTransformer = EventStreamAsyncResponseTransformer
                    .<StartCallAnalyticsStreamTranscriptionResponse, CallAnalyticsTranscriptResultStream> builder()
                    .eventStreamResponseHandler(asyncResponseHandler).eventResponseHandler(eventResponseHandler)
                    .initialResponseHandler(responseHandler).exceptionResponseHandler(errorEventResponseHandler).future(future)
                    .executor(executor).serviceName(serviceName()).build();
            RestEventStreamAsyncResponseTransformer<StartCallAnalyticsStreamTranscriptionResponse, CallAnalyticsTranscriptResultStream> restAsyncResponseTransformer = RestEventStreamAsyncResponseTransformer
                    .<StartCallAnalyticsStreamTranscriptionResponse, CallAnalyticsTranscriptResultStream> builder()
                    .eventStreamAsyncResponseTransformer(asyncResponseTransformer)
                    .eventStreamResponseHandler(asyncResponseHandler).build();

            CompletableFuture<Void> executeFuture = clientHandler
                    .execute(
                            new ClientExecutionParams<StartCallAnalyticsStreamTranscriptionRequest, StartCallAnalyticsStreamTranscriptionResponse>()
                                    .withOperationName("StartCallAnalyticsStreamTranscription")
                                    .withProtocolMetadata(protocolMetadata)
                                    .withMarshaller(new StartCallAnalyticsStreamTranscriptionRequestMarshaller(protocolFactory))
                                    .withAsyncRequestBody(AsyncRequestBody.fromPublisher(adapted)).withFullDuplex(true)
                                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                                    .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                                    .withInput(startCallAnalyticsStreamTranscriptionRequest), restAsyncResponseTransformer);
            CompletableFuture<Void> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    try {
                        asyncResponseHandler.exceptionOccurred(e);
                    } finally {
                        future.completeExceptionally(e);
                    }
                }
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return CompletableFutureUtils.forwardExceptionTo(future, executeFuture);
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseHandler.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a bidirectional HTTP/2 stream, where audio is streamed to Amazon Web Services HealthScribe and the
     * transcription results are streamed to your application.
     * </p>
     * <p>
     * When you start a stream, you first specify the stream configuration in a
     * <code>MedicalScribeConfigurationEvent</code>. This event includes channel definitions, encryption settings, and
     * post-stream analytics settings, such as the output configuration for aggregated transcript and clinical note
     * generation. These are additional streaming session configurations beyond those provided in your initial start
     * request headers. Whether you are starting a new session or resuming an existing session, your first event must be
     * a <code>MedicalScribeConfigurationEvent</code>.
     * </p>
     * <p>
     * After you send a <code>MedicalScribeConfigurationEvent</code>, you start <code>AudioEvents</code> and Amazon Web
     * Services HealthScribe responds with real-time transcription results. When you are finished, to start processing
     * the results with the post-stream analytics, send a <code>MedicalScribeSessionControlEvent</code> with a
     * <code>Type</code> of <code>END_OF_SESSION</code> and Amazon Web Services HealthScribe starts the analytics.
     * </p>
     * <p>
     * You can pause or resume streaming. To pause streaming, complete the input stream without sending the
     * <code>MedicalScribeSessionControlEvent</code>. To resume streaming, call the
     * <code>StartMedicalScribeStream</code> and specify the same SessionId you used to start the stream.
     * </p>
     * <p>
     * The following parameters are required:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>language-code</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>media-encoding</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>media-sample-rate-hertz</code>
     * </p>
     * </li>
     * </ul>
     * <p/>
     * <p>
     * For more information on streaming with Amazon Web Services HealthScribe, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/health-scribe-streaming.html">Amazon Web Services
     * HealthScribe</a>.
     * </p>
     *
     * @param startMedicalScribeStreamRequest
     * @return A Java Future containing the result of the StartMedicalScribeStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code>,
     *         <code>StartMedicalStreamTranscription</code>, or <code>StartCallAnalyticsStreamTranscription</code>
     *         operation was not valid. For example, <code>MediaEncoding</code> or <code>LanguageCode</code> used
     *         unsupported values. Check the specified parameters and try your request again.</li>
     *         <li>LimitExceededException Your client has exceeded one of the Amazon Transcribe limits. This is
     *         typically the audio length limit. Break your audio stream into smaller chunks and try your request again.
     *         </li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing.</li>
     *         <li>ConflictException A new stream started with the same session ID. The current stream has been
     *         terminated.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable. Try your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.StartMedicalScribeStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/StartMedicalScribeStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<Void> startMedicalScribeStream(StartMedicalScribeStreamRequest startMedicalScribeStreamRequest,
            Publisher<MedicalScribeInputStream> requestStream, StartMedicalScribeStreamResponseHandler asyncResponseHandler) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startMedicalScribeStreamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startMedicalScribeStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe Streaming");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMedicalScribeStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartMedicalScribeStreamResponse> responseHandler = new AttachHttpMetadataResponseHandler(
                    protocolFactory.createResponseHandler(operationMetadata, StartMedicalScribeStreamResponse::builder));

            HttpResponseHandler<SdkResponse> voidResponseHandler = protocolFactory.createResponseHandler(JsonOperationMetadata
                    .builder().isPayloadJson(false).hasStreamingSuccessResponse(true).build(), VoidSdkResponse::builder);

            HttpResponseHandler<? extends MedicalScribeResultStream> eventResponseHandler = protocolFactory
                    .createResponseHandler(
                            JsonOperationMetadata.builder().isPayloadJson(true).hasStreamingSuccessResponse(false).build(),
                            EventStreamTaggedUnionPojoSupplier.builder()
                                    .putSdkPojoSupplier("TranscriptEvent", MedicalScribeResultStream::transcriptEventBuilder)
                                    .defaultSdkPojoSupplier(() -> new SdkPojoBuilder(MedicalScribeResultStream.UNKNOWN)).build());

            Function<String, Optional<ExceptionMetadata>> eventstreamExceptionMetadataMapper = errorCode -> {
                switch (errorCode) {
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };

            HttpResponseHandler<AwsServiceException> errorEventResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, eventstreamExceptionMetadataMapper);

            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);
            EventStreamTaggedUnionJsonMarshaller eventMarshaller = EventStreamTaggedUnionJsonMarshaller
                    .builder()
                    .putMarshaller(DefaultAudioEvent.class, new MedicalScribeAudioEventMarshaller(protocolFactory))
                    .putMarshaller(DefaultSessionControlEvent.class,
                            new MedicalScribeSessionControlEventMarshaller(protocolFactory))
                    .putMarshaller(
                            software.amazon.awssdk.services.transcribestreaming.model.medicalscribeinputstream.DefaultConfigurationEvent.class,
                            new MedicalScribeConfigurationEventMarshaller(protocolFactory)).build();
            SdkPublisher<MedicalScribeInputStream> eventPublisher = SdkPublisher.adapt(requestStream);
            Publisher<ByteBuffer> adapted = eventPublisher.map(event -> eventMarshaller.marshall(event)).map(
                    AwsClientHandlerUtils::encodeEventStreamRequestToByteBuffer);
            CompletableFuture<Void> future = new CompletableFuture<>();
            EventStreamAsyncResponseTransformer<StartMedicalScribeStreamResponse, MedicalScribeResultStream> asyncResponseTransformer = EventStreamAsyncResponseTransformer
                    .<StartMedicalScribeStreamResponse, MedicalScribeResultStream> builder()
                    .eventStreamResponseHandler(asyncResponseHandler).eventResponseHandler(eventResponseHandler)
                    .initialResponseHandler(responseHandler).exceptionResponseHandler(errorEventResponseHandler).future(future)
                    .executor(executor).serviceName(serviceName()).build();
            RestEventStreamAsyncResponseTransformer<StartMedicalScribeStreamResponse, MedicalScribeResultStream> restAsyncResponseTransformer = RestEventStreamAsyncResponseTransformer
                    .<StartMedicalScribeStreamResponse, MedicalScribeResultStream> builder()
                    .eventStreamAsyncResponseTransformer(asyncResponseTransformer)
                    .eventStreamResponseHandler(asyncResponseHandler).build();

            CompletableFuture<Void> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<StartMedicalScribeStreamRequest, StartMedicalScribeStreamResponse>()
                            .withOperationName("StartMedicalScribeStream").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartMedicalScribeStreamRequestMarshaller(protocolFactory))
                            .withAsyncRequestBody(AsyncRequestBody.fromPublisher(adapted)).withFullDuplex(true)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startMedicalScribeStreamRequest), restAsyncResponseTransformer);
            CompletableFuture<Void> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    try {
                        asyncResponseHandler.exceptionOccurred(e);
                    } finally {
                        future.completeExceptionally(e);
                    }
                }
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return CompletableFutureUtils.forwardExceptionTo(future, executeFuture);
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseHandler.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a bidirectional HTTP/2 or WebSocket stream where audio is streamed to Amazon Transcribe Medical and the
     * transcription results are streamed to your application.
     * </p>
     * <p>
     * The following parameters are required:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>language-code</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>media-encoding</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sample-rate</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information on streaming with Amazon Transcribe Medical, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/streaming.html">Transcribing streaming audio</a>.
     * </p>
     *
     * @param startMedicalStreamTranscriptionRequest
     * @return A Java Future containing the result of the StartMedicalStreamTranscription operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code>,
     *         <code>StartMedicalStreamTranscription</code>, or <code>StartCallAnalyticsStreamTranscription</code>
     *         operation was not valid. For example, <code>MediaEncoding</code> or <code>LanguageCode</code> used
     *         unsupported values. Check the specified parameters and try your request again.</li>
     *         <li>LimitExceededException Your client has exceeded one of the Amazon Transcribe limits. This is
     *         typically the audio length limit. Break your audio stream into smaller chunks and try your request again.
     *         </li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing.</li>
     *         <li>ConflictException A new stream started with the same session ID. The current stream has been
     *         terminated.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable. Try your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.StartMedicalStreamTranscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/StartMedicalStreamTranscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<Void> startMedicalStreamTranscription(
            StartMedicalStreamTranscriptionRequest startMedicalStreamTranscriptionRequest, Publisher<AudioStream> requestStream,
            StartMedicalStreamTranscriptionResponseHandler asyncResponseHandler) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startMedicalStreamTranscriptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startMedicalStreamTranscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe Streaming");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMedicalStreamTranscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartMedicalStreamTranscriptionResponse> responseHandler = new AttachHttpMetadataResponseHandler(
                    protocolFactory.createResponseHandler(operationMetadata, StartMedicalStreamTranscriptionResponse::builder));

            HttpResponseHandler<SdkResponse> voidResponseHandler = protocolFactory.createResponseHandler(JsonOperationMetadata
                    .builder().isPayloadJson(false).hasStreamingSuccessResponse(true).build(), VoidSdkResponse::builder);

            HttpResponseHandler<? extends MedicalTranscriptResultStream> eventResponseHandler = protocolFactory
                    .createResponseHandler(
                            JsonOperationMetadata.builder().isPayloadJson(true).hasStreamingSuccessResponse(false).build(),
                            EventStreamTaggedUnionPojoSupplier.builder()
                                    .putSdkPojoSupplier("TranscriptEvent", MedicalTranscriptResultStream::transcriptEventBuilder)
                                    .defaultSdkPojoSupplier(() -> new SdkPojoBuilder(MedicalTranscriptResultStream.UNKNOWN))
                                    .build());

            Function<String, Optional<ExceptionMetadata>> eventstreamExceptionMetadataMapper = errorCode -> {
                switch (errorCode) {
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };

            HttpResponseHandler<AwsServiceException> errorEventResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, eventstreamExceptionMetadataMapper);

            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);
            EventStreamTaggedUnionJsonMarshaller eventMarshaller = EventStreamTaggedUnionJsonMarshaller.builder()
                    .putMarshaller(AudioEvent.class, new AudioEventMarshaller(protocolFactory))
                    .putMarshaller(DefaultConfigurationEvent.class, new ConfigurationEventMarshaller(protocolFactory)).build();
            SdkPublisher<AudioStream> eventPublisher = SdkPublisher.adapt(requestStream);
            Publisher<ByteBuffer> adapted = eventPublisher.map(event -> eventMarshaller.marshall(event)).map(
                    AwsClientHandlerUtils::encodeEventStreamRequestToByteBuffer);
            CompletableFuture<Void> future = new CompletableFuture<>();
            EventStreamAsyncResponseTransformer<StartMedicalStreamTranscriptionResponse, MedicalTranscriptResultStream> asyncResponseTransformer = EventStreamAsyncResponseTransformer
                    .<StartMedicalStreamTranscriptionResponse, MedicalTranscriptResultStream> builder()
                    .eventStreamResponseHandler(asyncResponseHandler).eventResponseHandler(eventResponseHandler)
                    .initialResponseHandler(responseHandler).exceptionResponseHandler(errorEventResponseHandler).future(future)
                    .executor(executor).serviceName(serviceName()).build();
            RestEventStreamAsyncResponseTransformer<StartMedicalStreamTranscriptionResponse, MedicalTranscriptResultStream> restAsyncResponseTransformer = RestEventStreamAsyncResponseTransformer
                    .<StartMedicalStreamTranscriptionResponse, MedicalTranscriptResultStream> builder()
                    .eventStreamAsyncResponseTransformer(asyncResponseTransformer)
                    .eventStreamResponseHandler(asyncResponseHandler).build();

            CompletableFuture<Void> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<StartMedicalStreamTranscriptionRequest, StartMedicalStreamTranscriptionResponse>()
                            .withOperationName("StartMedicalStreamTranscription").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartMedicalStreamTranscriptionRequestMarshaller(protocolFactory))
                            .withAsyncRequestBody(AsyncRequestBody.fromPublisher(adapted)).withFullDuplex(true)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startMedicalStreamTranscriptionRequest), restAsyncResponseTransformer);
            CompletableFuture<Void> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    try {
                        asyncResponseHandler.exceptionOccurred(e);
                    } finally {
                        future.completeExceptionally(e);
                    }
                }
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return CompletableFutureUtils.forwardExceptionTo(future, executeFuture);
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseHandler.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a bidirectional HTTP/2 or WebSocket stream where audio is streamed to Amazon Transcribe and the
     * transcription results are streamed to your application.
     * </p>
     * <p>
     * The following parameters are required:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>language-code</code> or <code>identify-language</code> or <code>identify-multiple-language</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>media-encoding</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sample-rate</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information on streaming with Amazon Transcribe, see <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/streaming.html">Transcribing streaming audio</a>.
     * </p>
     *
     * @param startStreamTranscriptionRequest
     * @return A Java Future containing the result of the StartStreamTranscription operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException One or more arguments to the <code>StartStreamTranscription</code>,
     *         <code>StartMedicalStreamTranscription</code>, or <code>StartCallAnalyticsStreamTranscription</code>
     *         operation was not valid. For example, <code>MediaEncoding</code> or <code>LanguageCode</code> used
     *         unsupported values. Check the specified parameters and try your request again.</li>
     *         <li>LimitExceededException Your client has exceeded one of the Amazon Transcribe limits. This is
     *         typically the audio length limit. Break your audio stream into smaller chunks and try your request again.
     *         </li>
     *         <li>InternalFailureException A problem occurred while processing the audio. Amazon Transcribe terminated
     *         processing.</li>
     *         <li>ConflictException A new stream started with the same session ID. The current stream has been
     *         terminated.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable. Try your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranscribeStreamingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample TranscribeStreamingAsyncClient.StartStreamTranscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transcribe-streaming-2017-10-26/StartStreamTranscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<Void> startStreamTranscription(StartStreamTranscriptionRequest startStreamTranscriptionRequest,
            Publisher<AudioStream> requestStream, StartStreamTranscriptionResponseHandler asyncResponseHandler) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startStreamTranscriptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startStreamTranscriptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transcribe Streaming");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartStreamTranscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartStreamTranscriptionResponse> responseHandler = new AttachHttpMetadataResponseHandler(
                    protocolFactory.createResponseHandler(operationMetadata, StartStreamTranscriptionResponse::builder));

            HttpResponseHandler<SdkResponse> voidResponseHandler = protocolFactory.createResponseHandler(JsonOperationMetadata
                    .builder().isPayloadJson(false).hasStreamingSuccessResponse(true).build(), VoidSdkResponse::builder);

            HttpResponseHandler<? extends TranscriptResultStream> eventResponseHandler = protocolFactory.createResponseHandler(
                    JsonOperationMetadata.builder().isPayloadJson(true).hasStreamingSuccessResponse(false).build(),
                    EventStreamTaggedUnionPojoSupplier.builder()
                            .putSdkPojoSupplier("TranscriptEvent", TranscriptResultStream::transcriptEventBuilder)
                            .defaultSdkPojoSupplier(() -> new SdkPojoBuilder(TranscriptResultStream.UNKNOWN)).build());

            Function<String, Optional<ExceptionMetadata>> eventstreamExceptionMetadataMapper = errorCode -> {
                switch (errorCode) {
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                default:
                    return Optional.empty();
                }
            };

            HttpResponseHandler<AwsServiceException> errorEventResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, eventstreamExceptionMetadataMapper);

            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);
            EventStreamTaggedUnionJsonMarshaller eventMarshaller = EventStreamTaggedUnionJsonMarshaller.builder()
                    .putMarshaller(AudioEvent.class, new AudioEventMarshaller(protocolFactory))
                    .putMarshaller(DefaultConfigurationEvent.class, new ConfigurationEventMarshaller(protocolFactory)).build();
            SdkPublisher<AudioStream> eventPublisher = SdkPublisher.adapt(requestStream);
            Publisher<ByteBuffer> adapted = eventPublisher.map(event -> eventMarshaller.marshall(event)).map(
                    AwsClientHandlerUtils::encodeEventStreamRequestToByteBuffer);
            CompletableFuture<Void> future = new CompletableFuture<>();
            EventStreamAsyncResponseTransformer<StartStreamTranscriptionResponse, TranscriptResultStream> asyncResponseTransformer = EventStreamAsyncResponseTransformer
                    .<StartStreamTranscriptionResponse, TranscriptResultStream> builder()
                    .eventStreamResponseHandler(asyncResponseHandler).eventResponseHandler(eventResponseHandler)
                    .initialResponseHandler(responseHandler).exceptionResponseHandler(errorEventResponseHandler).future(future)
                    .executor(executor).serviceName(serviceName()).build();
            RestEventStreamAsyncResponseTransformer<StartStreamTranscriptionResponse, TranscriptResultStream> restAsyncResponseTransformer = RestEventStreamAsyncResponseTransformer
                    .<StartStreamTranscriptionResponse, TranscriptResultStream> builder()
                    .eventStreamAsyncResponseTransformer(asyncResponseTransformer)
                    .eventStreamResponseHandler(asyncResponseHandler).build();

            CompletableFuture<Void> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<StartStreamTranscriptionRequest, StartStreamTranscriptionResponse>()
                            .withOperationName("StartStreamTranscription").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartStreamTranscriptionRequestMarshaller(protocolFactory))
                            .withAsyncRequestBody(AsyncRequestBody.fromPublisher(adapted)).withFullDuplex(true)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startStreamTranscriptionRequest), restAsyncResponseTransformer);
            CompletableFuture<Void> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    try {
                        asyncResponseHandler.exceptionOccurred(e);
                    } finally {
                        future.completeExceptionally(e);
                    }
                }
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return CompletableFutureUtils.forwardExceptionTo(future, executeFuture);
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseHandler.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final TranscribeStreamingServiceClientConfiguration serviceClientConfiguration() {
        return new TranscribeStreamingServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(TranscribeStreamingException::builder).protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        TranscribeStreamingServiceClientConfigurationBuilder serviceConfigBuilder = new TranscribeStreamingServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
