/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transfer.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateUserRequest extends TransferRequest implements
        ToCopyableBuilder<CreateUserRequest.Builder, CreateUserRequest> {
    private static final SdkField<String> HOME_DIRECTORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateUserRequest::homeDirectory)).setter(setter(Builder::homeDirectory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HomeDirectory").build()).build();

    private static final SdkField<String> HOME_DIRECTORY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateUserRequest::homeDirectoryTypeAsString)).setter(setter(Builder::homeDirectoryType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HomeDirectoryType").build()).build();

    private static final SdkField<List<HomeDirectoryMapEntry>> HOME_DIRECTORY_MAPPINGS_FIELD = SdkField
            .<List<HomeDirectoryMapEntry>> builder(MarshallingType.LIST)
            .getter(getter(CreateUserRequest::homeDirectoryMappings))
            .setter(setter(Builder::homeDirectoryMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HomeDirectoryMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<HomeDirectoryMapEntry> builder(MarshallingType.SDK_POJO)
                                            .constructor(HomeDirectoryMapEntry::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateUserRequest::policy)).setter(setter(Builder::policy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Policy").build()).build();

    private static final SdkField<String> ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateUserRequest::role)).setter(setter(Builder::role))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Role").build()).build();

    private static final SdkField<String> SERVER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateUserRequest::serverId)).setter(setter(Builder::serverId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerId").build()).build();

    private static final SdkField<String> SSH_PUBLIC_KEY_BODY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateUserRequest::sshPublicKeyBody)).setter(setter(Builder::sshPublicKeyBody))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SshPublicKeyBody").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(CreateUserRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> USER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateUserRequest::userName)).setter(setter(Builder::userName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserName").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(HOME_DIRECTORY_FIELD,
            HOME_DIRECTORY_TYPE_FIELD, HOME_DIRECTORY_MAPPINGS_FIELD, POLICY_FIELD, ROLE_FIELD, SERVER_ID_FIELD,
            SSH_PUBLIC_KEY_BODY_FIELD, TAGS_FIELD, USER_NAME_FIELD));

    private final String homeDirectory;

    private final String homeDirectoryType;

    private final List<HomeDirectoryMapEntry> homeDirectoryMappings;

    private final String policy;

    private final String role;

    private final String serverId;

    private final String sshPublicKeyBody;

    private final List<Tag> tags;

    private final String userName;

    private CreateUserRequest(BuilderImpl builder) {
        super(builder);
        this.homeDirectory = builder.homeDirectory;
        this.homeDirectoryType = builder.homeDirectoryType;
        this.homeDirectoryMappings = builder.homeDirectoryMappings;
        this.policy = builder.policy;
        this.role = builder.role;
        this.serverId = builder.serverId;
        this.sshPublicKeyBody = builder.sshPublicKeyBody;
        this.tags = builder.tags;
        this.userName = builder.userName;
    }

    /**
     * <p>
     * The landing directory (folder) for a user when they log in to the server using their SFTP client.
     * </p>
     * <p>
     * An example is &lt;<code>your-Amazon-S3-bucket-name&gt;/home/username</code>.
     * </p>
     * 
     * @return The landing directory (folder) for a user when they log in to the server using their SFTP client. </p>
     *         <p>
     *         An example is &lt;<code>your-Amazon-S3-bucket-name&gt;/home/username</code>.
     */
    public String homeDirectory() {
        return homeDirectory;
    }

    /**
     * <p>
     * The type of landing directory (folder) you want your users' home directory to be when they log into the SFTP
     * server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket paths as is in their
     * SFTP clients. If you set it <code>LOGICAL</code>, you will need to provide mappings in the
     * <code>HomeDirectoryMappings</code> for how you want to make S3 paths visible to your user.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #homeDirectoryType}
     * will return {@link HomeDirectoryType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #homeDirectoryTypeAsString}.
     * </p>
     * 
     * @return The type of landing directory (folder) you want your users' home directory to be when they log into the
     *         SFTP server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket paths as
     *         is in their SFTP clients. If you set it <code>LOGICAL</code>, you will need to provide mappings in the
     *         <code>HomeDirectoryMappings</code> for how you want to make S3 paths visible to your user.
     * @see HomeDirectoryType
     */
    public HomeDirectoryType homeDirectoryType() {
        return HomeDirectoryType.fromValue(homeDirectoryType);
    }

    /**
     * <p>
     * The type of landing directory (folder) you want your users' home directory to be when they log into the SFTP
     * server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket paths as is in their
     * SFTP clients. If you set it <code>LOGICAL</code>, you will need to provide mappings in the
     * <code>HomeDirectoryMappings</code> for how you want to make S3 paths visible to your user.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #homeDirectoryType}
     * will return {@link HomeDirectoryType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #homeDirectoryTypeAsString}.
     * </p>
     * 
     * @return The type of landing directory (folder) you want your users' home directory to be when they log into the
     *         SFTP server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket paths as
     *         is in their SFTP clients. If you set it <code>LOGICAL</code>, you will need to provide mappings in the
     *         <code>HomeDirectoryMappings</code> for how you want to make S3 paths visible to your user.
     * @see HomeDirectoryType
     */
    public String homeDirectoryTypeAsString() {
        return homeDirectoryType;
    }

    /**
     * <p>
     * Logical directory mappings that specify what S3 paths and keys should be visible to your user and how you want to
     * make them visible. You will need to specify the "<code>Entry</code>" and "<code>Target</code>" pair, where
     * <code>Entry</code> shows how the path is made visible and <code>Target</code> is the actual S3 path. If you only
     * specify a target, it will be displayed as is. You will need to also make sure that your AWS IAM Role provides
     * access to paths in <code>Target</code>. The following is an example.
     * </p>
     * <p>
     * <code>'[ "/bucket2/documentation", { "Entry": "your-personal-report.pdf", "Target": "/bucket3/customized-reports/${transfer:UserName}.pdf" } ]'</code>
     * </p>
     * <p>
     * In most cases, you can use this value instead of the scope down policy to lock your user down to the designated
     * home directory ("chroot"). To do this, you can set <code>Entry</code> to '/' and set <code>Target</code> to the
     * HomeDirectory parameter value.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Logical directory mappings that specify what S3 paths and keys should be visible to your user and how you
     *         want to make them visible. You will need to specify the "<code>Entry</code>" and "<code>Target</code>"
     *         pair, where <code>Entry</code> shows how the path is made visible and <code>Target</code> is the actual
     *         S3 path. If you only specify a target, it will be displayed as is. You will need to also make sure that
     *         your AWS IAM Role provides access to paths in <code>Target</code>. The following is an example.</p>
     *         <p>
     *         <code>'[ "/bucket2/documentation", { "Entry": "your-personal-report.pdf", "Target": "/bucket3/customized-reports/${transfer:UserName}.pdf" } ]'</code>
     *         </p>
     *         <p>
     *         In most cases, you can use this value instead of the scope down policy to lock your user down to the
     *         designated home directory ("chroot"). To do this, you can set <code>Entry</code> to '/' and set
     *         <code>Target</code> to the HomeDirectory parameter value.
     */
    public List<HomeDirectoryMapEntry> homeDirectoryMappings() {
        return homeDirectoryMappings;
    }

    /**
     * <p>
     * A scope-down policy for your user so you can use the same IAM role across multiple users. This policy scopes down
     * user access to portions of their Amazon S3 bucket. Variables that you can use inside this policy include
     * <code>${Transfer:UserName}</code>, <code>${Transfer:HomeDirectory}</code>, and
     * <code>${Transfer:HomeBucket}</code>.
     * </p>
     * <note>
     * <p>
     * For scope-down policies, AWS Transfer for SFTP stores the policy as a JSON blob, instead of the Amazon Resource
     * Name (ARN) of the policy. You save the policy as a JSON blob and pass it in the <code>Policy</code> argument.
     * </p>
     * <p>
     * For an example of a scope-down policy, see
     * "https://docs.aws.amazon.com/transfer/latest/userguide/users.html#users-policies-scope-down"&gt;Creating a
     * Scope-Down Policy.
     * </p>
     * <p>
     * For more information, see "https://docs.aws.amazon.com/STS/latest/APIReference/API_AssumeRole.html" in the <i>AWS
     * Security Token Service API Reference</i>.
     * </p>
     * </note>
     * 
     * @return A scope-down policy for your user so you can use the same IAM role across multiple users. This policy
     *         scopes down user access to portions of their Amazon S3 bucket. Variables that you can use inside this
     *         policy include <code>${Transfer:UserName}</code>, <code>${Transfer:HomeDirectory}</code>, and
     *         <code>${Transfer:HomeBucket}</code>.</p> <note>
     *         <p>
     *         For scope-down policies, AWS Transfer for SFTP stores the policy as a JSON blob, instead of the Amazon
     *         Resource Name (ARN) of the policy. You save the policy as a JSON blob and pass it in the
     *         <code>Policy</code> argument.
     *         </p>
     *         <p>
     *         For an example of a scope-down policy, see
     *         "https://docs.aws.amazon.com/transfer/latest/userguide/users.html#users-policies-scope-down"&gt;Creating
     *         a Scope-Down Policy.
     *         </p>
     *         <p>
     *         For more information, see "https://docs.aws.amazon.com/STS/latest/APIReference/API_AssumeRole.html" in
     *         the <i>AWS Security Token Service API Reference</i>.
     *         </p>
     */
    public String policy() {
        return policy;
    }

    /**
     * <p>
     * The IAM role that controls your user's access to your Amazon S3 bucket. The policies attached to this role will
     * determine the level of access you want to provide your users when transferring files into and out of your Amazon
     * S3 bucket or buckets. The IAM role should also contain a trust relationship that allows the SFTP server to access
     * your resources when servicing your SFTP user's transfer requests.
     * </p>
     * 
     * @return The IAM role that controls your user's access to your Amazon S3 bucket. The policies attached to this
     *         role will determine the level of access you want to provide your users when transferring files into and
     *         out of your Amazon S3 bucket or buckets. The IAM role should also contain a trust relationship that
     *         allows the SFTP server to access your resources when servicing your SFTP user's transfer requests.
     */
    public String role() {
        return role;
    }

    /**
     * <p>
     * A system-assigned unique identifier for an SFTP server instance. This is the specific SFTP server that you added
     * your user to.
     * </p>
     * 
     * @return A system-assigned unique identifier for an SFTP server instance. This is the specific SFTP server that
     *         you added your user to.
     */
    public String serverId() {
        return serverId;
    }

    /**
     * <p>
     * The public portion of the Secure Shell (SSH) key used to authenticate the user to the SFTP server.
     * </p>
     * 
     * @return The public portion of the Secure Shell (SSH) key used to authenticate the user to the SFTP server.
     */
    public String sshPublicKeyBody() {
        return sshPublicKeyBody;
    }

    /**
     * <p>
     * Key-value pairs that can be used to group and search for users. Tags are metadata attached to users for any
     * purpose.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Key-value pairs that can be used to group and search for users. Tags are metadata attached to users for
     *         any purpose.
     */
    public List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * A unique string that identifies a user and is associated with a server as specified by the <code>ServerId</code>.
     * This user name must be a minimum of 3 and a maximum of 32 characters long. The following are valid characters:
     * a-z, A-Z, 0-9, underscore, and hyphen. The user name can't start with a hyphen.
     * </p>
     * 
     * @return A unique string that identifies a user and is associated with a server as specified by the
     *         <code>ServerId</code>. This user name must be a minimum of 3 and a maximum of 32 characters long. The
     *         following are valid characters: a-z, A-Z, 0-9, underscore, and hyphen. The user name can't start with a
     *         hyphen.
     */
    public String userName() {
        return userName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(homeDirectory());
        hashCode = 31 * hashCode + Objects.hashCode(homeDirectoryTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(homeDirectoryMappings());
        hashCode = 31 * hashCode + Objects.hashCode(policy());
        hashCode = 31 * hashCode + Objects.hashCode(role());
        hashCode = 31 * hashCode + Objects.hashCode(serverId());
        hashCode = 31 * hashCode + Objects.hashCode(sshPublicKeyBody());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(userName());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateUserRequest)) {
            return false;
        }
        CreateUserRequest other = (CreateUserRequest) obj;
        return Objects.equals(homeDirectory(), other.homeDirectory())
                && Objects.equals(homeDirectoryTypeAsString(), other.homeDirectoryTypeAsString())
                && Objects.equals(homeDirectoryMappings(), other.homeDirectoryMappings())
                && Objects.equals(policy(), other.policy()) && Objects.equals(role(), other.role())
                && Objects.equals(serverId(), other.serverId()) && Objects.equals(sshPublicKeyBody(), other.sshPublicKeyBody())
                && Objects.equals(tags(), other.tags()) && Objects.equals(userName(), other.userName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateUserRequest").add("HomeDirectory", homeDirectory())
                .add("HomeDirectoryType", homeDirectoryTypeAsString()).add("HomeDirectoryMappings", homeDirectoryMappings())
                .add("Policy", policy()).add("Role", role()).add("ServerId", serverId())
                .add("SshPublicKeyBody", sshPublicKeyBody()).add("Tags", tags()).add("UserName", userName()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "HomeDirectory":
            return Optional.ofNullable(clazz.cast(homeDirectory()));
        case "HomeDirectoryType":
            return Optional.ofNullable(clazz.cast(homeDirectoryTypeAsString()));
        case "HomeDirectoryMappings":
            return Optional.ofNullable(clazz.cast(homeDirectoryMappings()));
        case "Policy":
            return Optional.ofNullable(clazz.cast(policy()));
        case "Role":
            return Optional.ofNullable(clazz.cast(role()));
        case "ServerId":
            return Optional.ofNullable(clazz.cast(serverId()));
        case "SshPublicKeyBody":
            return Optional.ofNullable(clazz.cast(sshPublicKeyBody()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "UserName":
            return Optional.ofNullable(clazz.cast(userName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateUserRequest, T> g) {
        return obj -> g.apply((CreateUserRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends TransferRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateUserRequest> {
        /**
         * <p>
         * The landing directory (folder) for a user when they log in to the server using their SFTP client.
         * </p>
         * <p>
         * An example is &lt;<code>your-Amazon-S3-bucket-name&gt;/home/username</code>.
         * </p>
         * 
         * @param homeDirectory
         *        The landing directory (folder) for a user when they log in to the server using their SFTP client. </p>
         *        <p>
         *        An example is &lt;<code>your-Amazon-S3-bucket-name&gt;/home/username</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder homeDirectory(String homeDirectory);

        /**
         * <p>
         * The type of landing directory (folder) you want your users' home directory to be when they log into the SFTP
         * server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket paths as is in
         * their SFTP clients. If you set it <code>LOGICAL</code>, you will need to provide mappings in the
         * <code>HomeDirectoryMappings</code> for how you want to make S3 paths visible to your user.
         * </p>
         * 
         * @param homeDirectoryType
         *        The type of landing directory (folder) you want your users' home directory to be when they log into
         *        the SFTP server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket
         *        paths as is in their SFTP clients. If you set it <code>LOGICAL</code>, you will need to provide
         *        mappings in the <code>HomeDirectoryMappings</code> for how you want to make S3 paths visible to your
         *        user.
         * @see HomeDirectoryType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HomeDirectoryType
         */
        Builder homeDirectoryType(String homeDirectoryType);

        /**
         * <p>
         * The type of landing directory (folder) you want your users' home directory to be when they log into the SFTP
         * server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket paths as is in
         * their SFTP clients. If you set it <code>LOGICAL</code>, you will need to provide mappings in the
         * <code>HomeDirectoryMappings</code> for how you want to make S3 paths visible to your user.
         * </p>
         * 
         * @param homeDirectoryType
         *        The type of landing directory (folder) you want your users' home directory to be when they log into
         *        the SFTP server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket
         *        paths as is in their SFTP clients. If you set it <code>LOGICAL</code>, you will need to provide
         *        mappings in the <code>HomeDirectoryMappings</code> for how you want to make S3 paths visible to your
         *        user.
         * @see HomeDirectoryType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HomeDirectoryType
         */
        Builder homeDirectoryType(HomeDirectoryType homeDirectoryType);

        /**
         * <p>
         * Logical directory mappings that specify what S3 paths and keys should be visible to your user and how you
         * want to make them visible. You will need to specify the "<code>Entry</code>" and "<code>Target</code>" pair,
         * where <code>Entry</code> shows how the path is made visible and <code>Target</code> is the actual S3 path. If
         * you only specify a target, it will be displayed as is. You will need to also make sure that your AWS IAM Role
         * provides access to paths in <code>Target</code>. The following is an example.
         * </p>
         * <p>
         * <code>'[ "/bucket2/documentation", { "Entry": "your-personal-report.pdf", "Target": "/bucket3/customized-reports/${transfer:UserName}.pdf" } ]'</code>
         * </p>
         * <p>
         * In most cases, you can use this value instead of the scope down policy to lock your user down to the
         * designated home directory ("chroot"). To do this, you can set <code>Entry</code> to '/' and set
         * <code>Target</code> to the HomeDirectory parameter value.
         * </p>
         * 
         * @param homeDirectoryMappings
         *        Logical directory mappings that specify what S3 paths and keys should be visible to your user and how
         *        you want to make them visible. You will need to specify the "<code>Entry</code>" and "
         *        <code>Target</code>" pair, where <code>Entry</code> shows how the path is made visible and
         *        <code>Target</code> is the actual S3 path. If you only specify a target, it will be displayed as is.
         *        You will need to also make sure that your AWS IAM Role provides access to paths in <code>Target</code>
         *        . The following is an example.</p>
         *        <p>
         *        <code>'[ "/bucket2/documentation", { "Entry": "your-personal-report.pdf", "Target": "/bucket3/customized-reports/${transfer:UserName}.pdf" } ]'</code>
         *        </p>
         *        <p>
         *        In most cases, you can use this value instead of the scope down policy to lock your user down to the
         *        designated home directory ("chroot"). To do this, you can set <code>Entry</code> to '/' and set
         *        <code>Target</code> to the HomeDirectory parameter value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder homeDirectoryMappings(Collection<HomeDirectoryMapEntry> homeDirectoryMappings);

        /**
         * <p>
         * Logical directory mappings that specify what S3 paths and keys should be visible to your user and how you
         * want to make them visible. You will need to specify the "<code>Entry</code>" and "<code>Target</code>" pair,
         * where <code>Entry</code> shows how the path is made visible and <code>Target</code> is the actual S3 path. If
         * you only specify a target, it will be displayed as is. You will need to also make sure that your AWS IAM Role
         * provides access to paths in <code>Target</code>. The following is an example.
         * </p>
         * <p>
         * <code>'[ "/bucket2/documentation", { "Entry": "your-personal-report.pdf", "Target": "/bucket3/customized-reports/${transfer:UserName}.pdf" } ]'</code>
         * </p>
         * <p>
         * In most cases, you can use this value instead of the scope down policy to lock your user down to the
         * designated home directory ("chroot"). To do this, you can set <code>Entry</code> to '/' and set
         * <code>Target</code> to the HomeDirectory parameter value.
         * </p>
         * 
         * @param homeDirectoryMappings
         *        Logical directory mappings that specify what S3 paths and keys should be visible to your user and how
         *        you want to make them visible. You will need to specify the "<code>Entry</code>" and "
         *        <code>Target</code>" pair, where <code>Entry</code> shows how the path is made visible and
         *        <code>Target</code> is the actual S3 path. If you only specify a target, it will be displayed as is.
         *        You will need to also make sure that your AWS IAM Role provides access to paths in <code>Target</code>
         *        . The following is an example.</p>
         *        <p>
         *        <code>'[ "/bucket2/documentation", { "Entry": "your-personal-report.pdf", "Target": "/bucket3/customized-reports/${transfer:UserName}.pdf" } ]'</code>
         *        </p>
         *        <p>
         *        In most cases, you can use this value instead of the scope down policy to lock your user down to the
         *        designated home directory ("chroot"). To do this, you can set <code>Entry</code> to '/' and set
         *        <code>Target</code> to the HomeDirectory parameter value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder homeDirectoryMappings(HomeDirectoryMapEntry... homeDirectoryMappings);

        /**
         * <p>
         * Logical directory mappings that specify what S3 paths and keys should be visible to your user and how you
         * want to make them visible. You will need to specify the "<code>Entry</code>" and "<code>Target</code>" pair,
         * where <code>Entry</code> shows how the path is made visible and <code>Target</code> is the actual S3 path. If
         * you only specify a target, it will be displayed as is. You will need to also make sure that your AWS IAM Role
         * provides access to paths in <code>Target</code>. The following is an example.
         * </p>
         * <p>
         * <code>'[ "/bucket2/documentation", { "Entry": "your-personal-report.pdf", "Target": "/bucket3/customized-reports/${transfer:UserName}.pdf" } ]'</code>
         * </p>
         * <p>
         * In most cases, you can use this value instead of the scope down policy to lock your user down to the
         * designated home directory ("chroot"). To do this, you can set <code>Entry</code> to '/' and set
         * <code>Target</code> to the HomeDirectory parameter value.
         * </p>
         * This is a convenience that creates an instance of the {@link List<HomeDirectoryMapEntry>.Builder} avoiding
         * the need to create one manually via {@link List<HomeDirectoryMapEntry>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<HomeDirectoryMapEntry>.Builder#build()} is called
         * immediately and its result is passed to {@link #homeDirectoryMappings(List<HomeDirectoryMapEntry>)}.
         * 
         * @param homeDirectoryMappings
         *        a consumer that will call methods on {@link List<HomeDirectoryMapEntry>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #homeDirectoryMappings(List<HomeDirectoryMapEntry>)
         */
        Builder homeDirectoryMappings(Consumer<HomeDirectoryMapEntry.Builder>... homeDirectoryMappings);

        /**
         * <p>
         * A scope-down policy for your user so you can use the same IAM role across multiple users. This policy scopes
         * down user access to portions of their Amazon S3 bucket. Variables that you can use inside this policy include
         * <code>${Transfer:UserName}</code>, <code>${Transfer:HomeDirectory}</code>, and
         * <code>${Transfer:HomeBucket}</code>.
         * </p>
         * <note>
         * <p>
         * For scope-down policies, AWS Transfer for SFTP stores the policy as a JSON blob, instead of the Amazon
         * Resource Name (ARN) of the policy. You save the policy as a JSON blob and pass it in the <code>Policy</code>
         * argument.
         * </p>
         * <p>
         * For an example of a scope-down policy, see
         * "https://docs.aws.amazon.com/transfer/latest/userguide/users.html#users-policies-scope-down"&gt;Creating a
         * Scope-Down Policy.
         * </p>
         * <p>
         * For more information, see "https://docs.aws.amazon.com/STS/latest/APIReference/API_AssumeRole.html" in the
         * <i>AWS Security Token Service API Reference</i>.
         * </p>
         * </note>
         * 
         * @param policy
         *        A scope-down policy for your user so you can use the same IAM role across multiple users. This policy
         *        scopes down user access to portions of their Amazon S3 bucket. Variables that you can use inside this
         *        policy include <code>${Transfer:UserName}</code>, <code>${Transfer:HomeDirectory}</code>, and
         *        <code>${Transfer:HomeBucket}</code>.</p> <note>
         *        <p>
         *        For scope-down policies, AWS Transfer for SFTP stores the policy as a JSON blob, instead of the Amazon
         *        Resource Name (ARN) of the policy. You save the policy as a JSON blob and pass it in the
         *        <code>Policy</code> argument.
         *        </p>
         *        <p>
         *        For an example of a scope-down policy, see
         *        "https://docs.aws.amazon.com/transfer/latest/userguide/users.html#users-policies-scope-down"
         *        &gt;Creating a Scope-Down Policy.
         *        </p>
         *        <p>
         *        For more information, see "https://docs.aws.amazon.com/STS/latest/APIReference/API_AssumeRole.html" in
         *        the <i>AWS Security Token Service API Reference</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policy(String policy);

        /**
         * <p>
         * The IAM role that controls your user's access to your Amazon S3 bucket. The policies attached to this role
         * will determine the level of access you want to provide your users when transferring files into and out of
         * your Amazon S3 bucket or buckets. The IAM role should also contain a trust relationship that allows the SFTP
         * server to access your resources when servicing your SFTP user's transfer requests.
         * </p>
         * 
         * @param role
         *        The IAM role that controls your user's access to your Amazon S3 bucket. The policies attached to this
         *        role will determine the level of access you want to provide your users when transferring files into
         *        and out of your Amazon S3 bucket or buckets. The IAM role should also contain a trust relationship
         *        that allows the SFTP server to access your resources when servicing your SFTP user's transfer
         *        requests.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder role(String role);

        /**
         * <p>
         * A system-assigned unique identifier for an SFTP server instance. This is the specific SFTP server that you
         * added your user to.
         * </p>
         * 
         * @param serverId
         *        A system-assigned unique identifier for an SFTP server instance. This is the specific SFTP server that
         *        you added your user to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverId(String serverId);

        /**
         * <p>
         * The public portion of the Secure Shell (SSH) key used to authenticate the user to the SFTP server.
         * </p>
         * 
         * @param sshPublicKeyBody
         *        The public portion of the Secure Shell (SSH) key used to authenticate the user to the SFTP server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sshPublicKeyBody(String sshPublicKeyBody);

        /**
         * <p>
         * Key-value pairs that can be used to group and search for users. Tags are metadata attached to users for any
         * purpose.
         * </p>
         * 
         * @param tags
         *        Key-value pairs that can be used to group and search for users. Tags are metadata attached to users
         *        for any purpose.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Key-value pairs that can be used to group and search for users. Tags are metadata attached to users for any
         * purpose.
         * </p>
         * 
         * @param tags
         *        Key-value pairs that can be used to group and search for users. Tags are metadata attached to users
         *        for any purpose.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Key-value pairs that can be used to group and search for users. Tags are metadata attached to users for any
         * purpose.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * A unique string that identifies a user and is associated with a server as specified by the
         * <code>ServerId</code>. This user name must be a minimum of 3 and a maximum of 32 characters long. The
         * following are valid characters: a-z, A-Z, 0-9, underscore, and hyphen. The user name can't start with a
         * hyphen.
         * </p>
         * 
         * @param userName
         *        A unique string that identifies a user and is associated with a server as specified by the
         *        <code>ServerId</code>. This user name must be a minimum of 3 and a maximum of 32 characters long. The
         *        following are valid characters: a-z, A-Z, 0-9, underscore, and hyphen. The user name can't start with
         *        a hyphen.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userName(String userName);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends TransferRequest.BuilderImpl implements Builder {
        private String homeDirectory;

        private String homeDirectoryType;

        private List<HomeDirectoryMapEntry> homeDirectoryMappings = DefaultSdkAutoConstructList.getInstance();

        private String policy;

        private String role;

        private String serverId;

        private String sshPublicKeyBody;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String userName;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateUserRequest model) {
            super(model);
            homeDirectory(model.homeDirectory);
            homeDirectoryType(model.homeDirectoryType);
            homeDirectoryMappings(model.homeDirectoryMappings);
            policy(model.policy);
            role(model.role);
            serverId(model.serverId);
            sshPublicKeyBody(model.sshPublicKeyBody);
            tags(model.tags);
            userName(model.userName);
        }

        public final String getHomeDirectory() {
            return homeDirectory;
        }

        @Override
        public final Builder homeDirectory(String homeDirectory) {
            this.homeDirectory = homeDirectory;
            return this;
        }

        public final void setHomeDirectory(String homeDirectory) {
            this.homeDirectory = homeDirectory;
        }

        public final String getHomeDirectoryTypeAsString() {
            return homeDirectoryType;
        }

        @Override
        public final Builder homeDirectoryType(String homeDirectoryType) {
            this.homeDirectoryType = homeDirectoryType;
            return this;
        }

        @Override
        public final Builder homeDirectoryType(HomeDirectoryType homeDirectoryType) {
            this.homeDirectoryType(homeDirectoryType == null ? null : homeDirectoryType.toString());
            return this;
        }

        public final void setHomeDirectoryType(String homeDirectoryType) {
            this.homeDirectoryType = homeDirectoryType;
        }

        public final Collection<HomeDirectoryMapEntry.Builder> getHomeDirectoryMappings() {
            return homeDirectoryMappings != null ? homeDirectoryMappings.stream().map(HomeDirectoryMapEntry::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder homeDirectoryMappings(Collection<HomeDirectoryMapEntry> homeDirectoryMappings) {
            this.homeDirectoryMappings = HomeDirectoryMappingsCopier.copy(homeDirectoryMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder homeDirectoryMappings(HomeDirectoryMapEntry... homeDirectoryMappings) {
            homeDirectoryMappings(Arrays.asList(homeDirectoryMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder homeDirectoryMappings(Consumer<HomeDirectoryMapEntry.Builder>... homeDirectoryMappings) {
            homeDirectoryMappings(Stream.of(homeDirectoryMappings)
                    .map(c -> HomeDirectoryMapEntry.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setHomeDirectoryMappings(Collection<HomeDirectoryMapEntry.BuilderImpl> homeDirectoryMappings) {
            this.homeDirectoryMappings = HomeDirectoryMappingsCopier.copyFromBuilder(homeDirectoryMappings);
        }

        public final String getPolicy() {
            return policy;
        }

        @Override
        public final Builder policy(String policy) {
            this.policy = policy;
            return this;
        }

        public final void setPolicy(String policy) {
            this.policy = policy;
        }

        public final String getRole() {
            return role;
        }

        @Override
        public final Builder role(String role) {
            this.role = role;
            return this;
        }

        public final void setRole(String role) {
            this.role = role;
        }

        public final String getServerId() {
            return serverId;
        }

        @Override
        public final Builder serverId(String serverId) {
            this.serverId = serverId;
            return this;
        }

        public final void setServerId(String serverId) {
            this.serverId = serverId;
        }

        public final String getSshPublicKeyBody() {
            return sshPublicKeyBody;
        }

        @Override
        public final Builder sshPublicKeyBody(String sshPublicKeyBody) {
            this.sshPublicKeyBody = sshPublicKeyBody;
            return this;
        }

        public final void setSshPublicKeyBody(String sshPublicKeyBody) {
            this.sshPublicKeyBody = sshPublicKeyBody;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        public final String getUserName() {
            return userName;
        }

        @Override
        public final Builder userName(String userName) {
            this.userName = userName;
            return this;
        }

        public final void setUserName(String userName) {
            this.userName = userName;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateUserRequest build() {
            return new CreateUserRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
