/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transfer.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateServerRequest extends TransferRequest implements
        ToCopyableBuilder<CreateServerRequest.Builder, CreateServerRequest> {
    private static final SdkField<String> CERTIFICATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Certificate").getter(getter(CreateServerRequest::certificate)).setter(setter(Builder::certificate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Certificate").build()).build();

    private static final SdkField<EndpointDetails> ENDPOINT_DETAILS_FIELD = SdkField
            .<EndpointDetails> builder(MarshallingType.SDK_POJO).memberName("EndpointDetails")
            .getter(getter(CreateServerRequest::endpointDetails)).setter(setter(Builder::endpointDetails))
            .constructor(EndpointDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointDetails").build()).build();

    private static final SdkField<String> ENDPOINT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EndpointType").getter(getter(CreateServerRequest::endpointTypeAsString))
            .setter(setter(Builder::endpointType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointType").build()).build();

    private static final SdkField<String> HOST_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HostKey").getter(getter(CreateServerRequest::hostKey)).setter(setter(Builder::hostKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HostKey").build()).build();

    private static final SdkField<IdentityProviderDetails> IDENTITY_PROVIDER_DETAILS_FIELD = SdkField
            .<IdentityProviderDetails> builder(MarshallingType.SDK_POJO).memberName("IdentityProviderDetails")
            .getter(getter(CreateServerRequest::identityProviderDetails)).setter(setter(Builder::identityProviderDetails))
            .constructor(IdentityProviderDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdentityProviderDetails").build())
            .build();

    private static final SdkField<String> IDENTITY_PROVIDER_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IdentityProviderType").getter(getter(CreateServerRequest::identityProviderTypeAsString))
            .setter(setter(Builder::identityProviderType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdentityProviderType").build())
            .build();

    private static final SdkField<String> LOGGING_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LoggingRole").getter(getter(CreateServerRequest::loggingRole)).setter(setter(Builder::loggingRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LoggingRole").build()).build();

    private static final SdkField<List<String>> PROTOCOLS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Protocols")
            .getter(getter(CreateServerRequest::protocolsAsStrings))
            .setter(setter(Builder::protocolsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Protocols").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SECURITY_POLICY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecurityPolicyName").getter(getter(CreateServerRequest::securityPolicyName))
            .setter(setter(Builder::securityPolicyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityPolicyName").build())
            .build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateServerRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CERTIFICATE_FIELD,
            ENDPOINT_DETAILS_FIELD, ENDPOINT_TYPE_FIELD, HOST_KEY_FIELD, IDENTITY_PROVIDER_DETAILS_FIELD,
            IDENTITY_PROVIDER_TYPE_FIELD, LOGGING_ROLE_FIELD, PROTOCOLS_FIELD, SECURITY_POLICY_NAME_FIELD, TAGS_FIELD));

    private final String certificate;

    private final EndpointDetails endpointDetails;

    private final String endpointType;

    private final String hostKey;

    private final IdentityProviderDetails identityProviderDetails;

    private final String identityProviderType;

    private final String loggingRole;

    private final List<String> protocols;

    private final String securityPolicyName;

    private final List<Tag> tags;

    private CreateServerRequest(BuilderImpl builder) {
        super(builder);
        this.certificate = builder.certificate;
        this.endpointDetails = builder.endpointDetails;
        this.endpointType = builder.endpointType;
        this.hostKey = builder.hostKey;
        this.identityProviderDetails = builder.identityProviderDetails;
        this.identityProviderType = builder.identityProviderType;
        this.loggingRole = builder.loggingRole;
        this.protocols = builder.protocols;
        this.securityPolicyName = builder.securityPolicyName;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the AWS Certificate Manager (ACM) certificate. Required when
     * <code>Protocols</code> is set to <code>FTPS</code>.
     * </p>
     * <p>
     * To request a new public certificate, see <a
     * href="https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-request-public.html">Request a public
     * certificate</a> in the <i> AWS Certificate Manager User Guide</i>.
     * </p>
     * <p>
     * To import an existing certificate into ACM, see <a
     * href="https://docs.aws.amazon.com/acm/latest/userguide/import-certificate.html">Importing certificates into
     * ACM</a> in the <i> AWS Certificate Manager User Guide</i>.
     * </p>
     * <p>
     * To request a private certificate to use FTPS through private IP addresses, see <a
     * href="https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-request-private.html">Request a private
     * certificate</a> in the <i> AWS Certificate Manager User Guide</i>.
     * </p>
     * <p>
     * Certificates with the following cryptographic algorithms and key sizes are supported:
     * </p>
     * <ul>
     * <li>
     * <p>
     * 2048-bit RSA (RSA_2048)
     * </p>
     * </li>
     * <li>
     * <p>
     * 4096-bit RSA (RSA_4096)
     * </p>
     * </li>
     * <li>
     * <p>
     * Elliptic Prime Curve 256 bit (EC_prime256v1)
     * </p>
     * </li>
     * <li>
     * <p>
     * Elliptic Prime Curve 384 bit (EC_secp384r1)
     * </p>
     * </li>
     * <li>
     * <p>
     * Elliptic Prime Curve 521 bit (EC_secp521r1)
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The certificate must be a valid SSL/TLS X.509 version 3 certificate with FQDN or IP address specified and
     * information about the issuer.
     * </p>
     * </note>
     * 
     * @return The Amazon Resource Name (ARN) of the AWS Certificate Manager (ACM) certificate. Required when
     *         <code>Protocols</code> is set to <code>FTPS</code>.</p>
     *         <p>
     *         To request a new public certificate, see <a
     *         href="https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-request-public.html">Request a public
     *         certificate</a> in the <i> AWS Certificate Manager User Guide</i>.
     *         </p>
     *         <p>
     *         To import an existing certificate into ACM, see <a
     *         href="https://docs.aws.amazon.com/acm/latest/userguide/import-certificate.html">Importing certificates
     *         into ACM</a> in the <i> AWS Certificate Manager User Guide</i>.
     *         </p>
     *         <p>
     *         To request a private certificate to use FTPS through private IP addresses, see <a
     *         href="https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-request-private.html">Request a private
     *         certificate</a> in the <i> AWS Certificate Manager User Guide</i>.
     *         </p>
     *         <p>
     *         Certificates with the following cryptographic algorithms and key sizes are supported:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         2048-bit RSA (RSA_2048)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4096-bit RSA (RSA_4096)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Elliptic Prime Curve 256 bit (EC_prime256v1)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Elliptic Prime Curve 384 bit (EC_secp384r1)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Elliptic Prime Curve 521 bit (EC_secp521r1)
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The certificate must be a valid SSL/TLS X.509 version 3 certificate with FQDN or IP address specified and
     *         information about the issuer.
     *         </p>
     */
    public String certificate() {
        return certificate;
    }

    /**
     * <p>
     * The virtual private cloud (VPC) endpoint settings that are configured for your server. When you host your
     * endpoint within your VPC, you can make it accessible only to resources within your VPC, or you can attach Elastic
     * IPs and make it accessible to clients over the internet. Your VPC's default security groups are automatically
     * assigned to your endpoint.
     * </p>
     * 
     * @return The virtual private cloud (VPC) endpoint settings that are configured for your server. When you host your
     *         endpoint within your VPC, you can make it accessible only to resources within your VPC, or you can attach
     *         Elastic IPs and make it accessible to clients over the internet. Your VPC's default security groups are
     *         automatically assigned to your endpoint.
     */
    public EndpointDetails endpointDetails() {
        return endpointDetails;
    }

    /**
     * <p>
     * The type of VPC endpoint that you want your server to connect to. You can choose to connect to the public
     * internet or a VPC endpoint. With a VPC endpoint, you can restrict access to your server and resources only within
     * your VPC.
     * </p>
     * <note>
     * <p>
     * It is recommended that you use <code>VPC</code> as the <code>EndpointType</code>. With this endpoint type, you
     * have the option to directly associate up to three Elastic IPv4 addresses (BYO IP included) with your server's
     * endpoint and use VPC security groups to restrict traffic by the client's public IP address. This is not possible
     * with <code>EndpointType</code> set to <code>VPC_ENDPOINT</code>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #endpointType} will
     * return {@link EndpointType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #endpointTypeAsString}.
     * </p>
     * 
     * @return The type of VPC endpoint that you want your server to connect to. You can choose to connect to the public
     *         internet or a VPC endpoint. With a VPC endpoint, you can restrict access to your server and resources
     *         only within your VPC.</p> <note>
     *         <p>
     *         It is recommended that you use <code>VPC</code> as the <code>EndpointType</code>. With this endpoint
     *         type, you have the option to directly associate up to three Elastic IPv4 addresses (BYO IP included) with
     *         your server's endpoint and use VPC security groups to restrict traffic by the client's public IP address.
     *         This is not possible with <code>EndpointType</code> set to <code>VPC_ENDPOINT</code>.
     *         </p>
     * @see EndpointType
     */
    public EndpointType endpointType() {
        return EndpointType.fromValue(endpointType);
    }

    /**
     * <p>
     * The type of VPC endpoint that you want your server to connect to. You can choose to connect to the public
     * internet or a VPC endpoint. With a VPC endpoint, you can restrict access to your server and resources only within
     * your VPC.
     * </p>
     * <note>
     * <p>
     * It is recommended that you use <code>VPC</code> as the <code>EndpointType</code>. With this endpoint type, you
     * have the option to directly associate up to three Elastic IPv4 addresses (BYO IP included) with your server's
     * endpoint and use VPC security groups to restrict traffic by the client's public IP address. This is not possible
     * with <code>EndpointType</code> set to <code>VPC_ENDPOINT</code>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #endpointType} will
     * return {@link EndpointType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #endpointTypeAsString}.
     * </p>
     * 
     * @return The type of VPC endpoint that you want your server to connect to. You can choose to connect to the public
     *         internet or a VPC endpoint. With a VPC endpoint, you can restrict access to your server and resources
     *         only within your VPC.</p> <note>
     *         <p>
     *         It is recommended that you use <code>VPC</code> as the <code>EndpointType</code>. With this endpoint
     *         type, you have the option to directly associate up to three Elastic IPv4 addresses (BYO IP included) with
     *         your server's endpoint and use VPC security groups to restrict traffic by the client's public IP address.
     *         This is not possible with <code>EndpointType</code> set to <code>VPC_ENDPOINT</code>.
     *         </p>
     * @see EndpointType
     */
    public String endpointTypeAsString() {
        return endpointType;
    }

    /**
     * <p>
     * The RSA private key as generated by the <code>ssh-keygen -N "" -m PEM -f my-new-server-key</code> command.
     * </p>
     * <important>
     * <p>
     * If you aren't planning to migrate existing users from an existing SFTP-enabled server to a new server, don't
     * update the host key. Accidentally changing a server's host key can be disruptive.
     * </p>
     * </important>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/transfer/latest/userguide/edit-server-config.html#configuring-servers-change-host-key"
     * >Change the host key for your SFTP-enabled server</a> in the <i>AWS Transfer Family User Guide</i>.
     * </p>
     * 
     * @return The RSA private key as generated by the <code>ssh-keygen -N "" -m PEM -f my-new-server-key</code>
     *         command.</p> <important>
     *         <p>
     *         If you aren't planning to migrate existing users from an existing SFTP-enabled server to a new server,
     *         don't update the host key. Accidentally changing a server's host key can be disruptive.
     *         </p>
     *         </important>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/transfer/latest/userguide/edit-server-config.html#configuring-servers-change-host-key"
     *         >Change the host key for your SFTP-enabled server</a> in the <i>AWS Transfer Family User Guide</i>.
     */
    public String hostKey() {
        return hostKey;
    }

    /**
     * <p>
     * Required when <code>IdentityProviderType</code> is set to <code>API_GATEWAY</code>. Accepts an array containing
     * all of the information required to call a customer-supplied authentication API, including the API Gateway URL.
     * Not required when <code>IdentityProviderType</code> is set to <code>SERVICE_MANAGED</code>.
     * </p>
     * 
     * @return Required when <code>IdentityProviderType</code> is set to <code>API_GATEWAY</code>. Accepts an array
     *         containing all of the information required to call a customer-supplied authentication API, including the
     *         API Gateway URL. Not required when <code>IdentityProviderType</code> is set to
     *         <code>SERVICE_MANAGED</code>.
     */
    public IdentityProviderDetails identityProviderDetails() {
        return identityProviderDetails;
    }

    /**
     * <p>
     * Specifies the mode of authentication for a server. The default value is <code>SERVICE_MANAGED</code>, which
     * allows you to store and access user credentials within the AWS Transfer Family service. Use the
     * <code>API_GATEWAY</code> value to integrate with an identity provider of your choosing. The
     * <code>API_GATEWAY</code> setting requires you to provide an API Gateway endpoint URL to call for authentication
     * using the <code>IdentityProviderDetails</code> parameter.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #identityProviderType} will return {@link IdentityProviderType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #identityProviderTypeAsString}.
     * </p>
     * 
     * @return Specifies the mode of authentication for a server. The default value is <code>SERVICE_MANAGED</code>,
     *         which allows you to store and access user credentials within the AWS Transfer Family service. Use the
     *         <code>API_GATEWAY</code> value to integrate with an identity provider of your choosing. The
     *         <code>API_GATEWAY</code> setting requires you to provide an API Gateway endpoint URL to call for
     *         authentication using the <code>IdentityProviderDetails</code> parameter.
     * @see IdentityProviderType
     */
    public IdentityProviderType identityProviderType() {
        return IdentityProviderType.fromValue(identityProviderType);
    }

    /**
     * <p>
     * Specifies the mode of authentication for a server. The default value is <code>SERVICE_MANAGED</code>, which
     * allows you to store and access user credentials within the AWS Transfer Family service. Use the
     * <code>API_GATEWAY</code> value to integrate with an identity provider of your choosing. The
     * <code>API_GATEWAY</code> setting requires you to provide an API Gateway endpoint URL to call for authentication
     * using the <code>IdentityProviderDetails</code> parameter.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #identityProviderType} will return {@link IdentityProviderType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #identityProviderTypeAsString}.
     * </p>
     * 
     * @return Specifies the mode of authentication for a server. The default value is <code>SERVICE_MANAGED</code>,
     *         which allows you to store and access user credentials within the AWS Transfer Family service. Use the
     *         <code>API_GATEWAY</code> value to integrate with an identity provider of your choosing. The
     *         <code>API_GATEWAY</code> setting requires you to provide an API Gateway endpoint URL to call for
     *         authentication using the <code>IdentityProviderDetails</code> parameter.
     * @see IdentityProviderType
     */
    public String identityProviderTypeAsString() {
        return identityProviderType;
    }

    /**
     * <p>
     * Allows the service to write your users' activity to your Amazon CloudWatch logs for monitoring and auditing
     * purposes.
     * </p>
     * 
     * @return Allows the service to write your users' activity to your Amazon CloudWatch logs for monitoring and
     *         auditing purposes.
     */
    public String loggingRole() {
        return loggingRole;
    }

    /**
     * <p>
     * Specifies the file transfer protocol or protocols over which your file transfer protocol client can connect to
     * your server's endpoint. The available protocols are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * If you select <code>FTPS</code>, you must choose a certificate stored in AWS Certificate Manager (ACM) which will
     * be used to identify your server when clients connect to it over FTPS.
     * </p>
     * <p>
     * If <code>Protocol</code> includes either <code>FTP</code> or <code>FTPS</code>, then the
     * <code>EndpointType</code> must be <code>VPC</code> and the <code>IdentityProviderType</code> must be
     * <code>API_GATEWAY</code>.
     * </p>
     * <p>
     * If <code>Protocol</code> includes <code>FTP</code>, then <code>AddressAllocationIds</code> cannot be associated.
     * </p>
     * <p>
     * If <code>Protocol</code> is set only to <code>SFTP</code>, the <code>EndpointType</code> can be set to
     * <code>PUBLIC</code> and the <code>IdentityProviderType</code> can be set to <code>SERVICE_MANAGED</code>.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasProtocols()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Specifies the file transfer protocol or protocols over which your file transfer protocol client can
     *         connect to your server's endpoint. The available protocols are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         If you select <code>FTPS</code>, you must choose a certificate stored in AWS Certificate Manager (ACM)
     *         which will be used to identify your server when clients connect to it over FTPS.
     *         </p>
     *         <p>
     *         If <code>Protocol</code> includes either <code>FTP</code> or <code>FTPS</code>, then the
     *         <code>EndpointType</code> must be <code>VPC</code> and the <code>IdentityProviderType</code> must be
     *         <code>API_GATEWAY</code>.
     *         </p>
     *         <p>
     *         If <code>Protocol</code> includes <code>FTP</code>, then <code>AddressAllocationIds</code> cannot be
     *         associated.
     *         </p>
     *         <p>
     *         If <code>Protocol</code> is set only to <code>SFTP</code>, the <code>EndpointType</code> can be set to
     *         <code>PUBLIC</code> and the <code>IdentityProviderType</code> can be set to <code>SERVICE_MANAGED</code>.
     *         </p>
     */
    public List<Protocol> protocols() {
        return ProtocolsCopier.copyStringToEnum(protocols);
    }

    /**
     * Returns true if the Protocols property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasProtocols() {
        return protocols != null && !(protocols instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the file transfer protocol or protocols over which your file transfer protocol client can connect to
     * your server's endpoint. The available protocols are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * If you select <code>FTPS</code>, you must choose a certificate stored in AWS Certificate Manager (ACM) which will
     * be used to identify your server when clients connect to it over FTPS.
     * </p>
     * <p>
     * If <code>Protocol</code> includes either <code>FTP</code> or <code>FTPS</code>, then the
     * <code>EndpointType</code> must be <code>VPC</code> and the <code>IdentityProviderType</code> must be
     * <code>API_GATEWAY</code>.
     * </p>
     * <p>
     * If <code>Protocol</code> includes <code>FTP</code>, then <code>AddressAllocationIds</code> cannot be associated.
     * </p>
     * <p>
     * If <code>Protocol</code> is set only to <code>SFTP</code>, the <code>EndpointType</code> can be set to
     * <code>PUBLIC</code> and the <code>IdentityProviderType</code> can be set to <code>SERVICE_MANAGED</code>.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasProtocols()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Specifies the file transfer protocol or protocols over which your file transfer protocol client can
     *         connect to your server's endpoint. The available protocols are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         If you select <code>FTPS</code>, you must choose a certificate stored in AWS Certificate Manager (ACM)
     *         which will be used to identify your server when clients connect to it over FTPS.
     *         </p>
     *         <p>
     *         If <code>Protocol</code> includes either <code>FTP</code> or <code>FTPS</code>, then the
     *         <code>EndpointType</code> must be <code>VPC</code> and the <code>IdentityProviderType</code> must be
     *         <code>API_GATEWAY</code>.
     *         </p>
     *         <p>
     *         If <code>Protocol</code> includes <code>FTP</code>, then <code>AddressAllocationIds</code> cannot be
     *         associated.
     *         </p>
     *         <p>
     *         If <code>Protocol</code> is set only to <code>SFTP</code>, the <code>EndpointType</code> can be set to
     *         <code>PUBLIC</code> and the <code>IdentityProviderType</code> can be set to <code>SERVICE_MANAGED</code>.
     *         </p>
     */
    public List<String> protocolsAsStrings() {
        return protocols;
    }

    /**
     * <p>
     * Specifies the name of the security policy that is attached to the server.
     * </p>
     * 
     * @return Specifies the name of the security policy that is attached to the server.
     */
    public String securityPolicyName() {
        return securityPolicyName;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Key-value pairs that can be used to group and search for servers.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Key-value pairs that can be used to group and search for servers.
     */
    public List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(certificate());
        hashCode = 31 * hashCode + Objects.hashCode(endpointDetails());
        hashCode = 31 * hashCode + Objects.hashCode(endpointTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hostKey());
        hashCode = 31 * hashCode + Objects.hashCode(identityProviderDetails());
        hashCode = 31 * hashCode + Objects.hashCode(identityProviderTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(loggingRole());
        hashCode = 31 * hashCode + Objects.hashCode(protocolsAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(securityPolicyName());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateServerRequest)) {
            return false;
        }
        CreateServerRequest other = (CreateServerRequest) obj;
        return Objects.equals(certificate(), other.certificate()) && Objects.equals(endpointDetails(), other.endpointDetails())
                && Objects.equals(endpointTypeAsString(), other.endpointTypeAsString())
                && Objects.equals(hostKey(), other.hostKey())
                && Objects.equals(identityProviderDetails(), other.identityProviderDetails())
                && Objects.equals(identityProviderTypeAsString(), other.identityProviderTypeAsString())
                && Objects.equals(loggingRole(), other.loggingRole())
                && Objects.equals(protocolsAsStrings(), other.protocolsAsStrings())
                && Objects.equals(securityPolicyName(), other.securityPolicyName()) && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateServerRequest").add("Certificate", certificate())
                .add("EndpointDetails", endpointDetails()).add("EndpointType", endpointTypeAsString())
                .add("HostKey", hostKey() == null ? null : "*** Sensitive Data Redacted ***")
                .add("IdentityProviderDetails", identityProviderDetails())
                .add("IdentityProviderType", identityProviderTypeAsString()).add("LoggingRole", loggingRole())
                .add("Protocols", protocolsAsStrings()).add("SecurityPolicyName", securityPolicyName()).add("Tags", tags())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Certificate":
            return Optional.ofNullable(clazz.cast(certificate()));
        case "EndpointDetails":
            return Optional.ofNullable(clazz.cast(endpointDetails()));
        case "EndpointType":
            return Optional.ofNullable(clazz.cast(endpointTypeAsString()));
        case "HostKey":
            return Optional.ofNullable(clazz.cast(hostKey()));
        case "IdentityProviderDetails":
            return Optional.ofNullable(clazz.cast(identityProviderDetails()));
        case "IdentityProviderType":
            return Optional.ofNullable(clazz.cast(identityProviderTypeAsString()));
        case "LoggingRole":
            return Optional.ofNullable(clazz.cast(loggingRole()));
        case "Protocols":
            return Optional.ofNullable(clazz.cast(protocolsAsStrings()));
        case "SecurityPolicyName":
            return Optional.ofNullable(clazz.cast(securityPolicyName()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateServerRequest, T> g) {
        return obj -> g.apply((CreateServerRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends TransferRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateServerRequest> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the AWS Certificate Manager (ACM) certificate. Required when
         * <code>Protocols</code> is set to <code>FTPS</code>.
         * </p>
         * <p>
         * To request a new public certificate, see <a
         * href="https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-request-public.html">Request a public
         * certificate</a> in the <i> AWS Certificate Manager User Guide</i>.
         * </p>
         * <p>
         * To import an existing certificate into ACM, see <a
         * href="https://docs.aws.amazon.com/acm/latest/userguide/import-certificate.html">Importing certificates into
         * ACM</a> in the <i> AWS Certificate Manager User Guide</i>.
         * </p>
         * <p>
         * To request a private certificate to use FTPS through private IP addresses, see <a
         * href="https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-request-private.html">Request a private
         * certificate</a> in the <i> AWS Certificate Manager User Guide</i>.
         * </p>
         * <p>
         * Certificates with the following cryptographic algorithms and key sizes are supported:
         * </p>
         * <ul>
         * <li>
         * <p>
         * 2048-bit RSA (RSA_2048)
         * </p>
         * </li>
         * <li>
         * <p>
         * 4096-bit RSA (RSA_4096)
         * </p>
         * </li>
         * <li>
         * <p>
         * Elliptic Prime Curve 256 bit (EC_prime256v1)
         * </p>
         * </li>
         * <li>
         * <p>
         * Elliptic Prime Curve 384 bit (EC_secp384r1)
         * </p>
         * </li>
         * <li>
         * <p>
         * Elliptic Prime Curve 521 bit (EC_secp521r1)
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The certificate must be a valid SSL/TLS X.509 version 3 certificate with FQDN or IP address specified and
         * information about the issuer.
         * </p>
         * </note>
         * 
         * @param certificate
         *        The Amazon Resource Name (ARN) of the AWS Certificate Manager (ACM) certificate. Required when
         *        <code>Protocols</code> is set to <code>FTPS</code>.</p>
         *        <p>
         *        To request a new public certificate, see <a
         *        href="https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-request-public.html">Request a public
         *        certificate</a> in the <i> AWS Certificate Manager User Guide</i>.
         *        </p>
         *        <p>
         *        To import an existing certificate into ACM, see <a
         *        href="https://docs.aws.amazon.com/acm/latest/userguide/import-certificate.html">Importing certificates
         *        into ACM</a> in the <i> AWS Certificate Manager User Guide</i>.
         *        </p>
         *        <p>
         *        To request a private certificate to use FTPS through private IP addresses, see <a
         *        href="https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-request-private.html">Request a private
         *        certificate</a> in the <i> AWS Certificate Manager User Guide</i>.
         *        </p>
         *        <p>
         *        Certificates with the following cryptographic algorithms and key sizes are supported:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        2048-bit RSA (RSA_2048)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        4096-bit RSA (RSA_4096)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Elliptic Prime Curve 256 bit (EC_prime256v1)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Elliptic Prime Curve 384 bit (EC_secp384r1)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Elliptic Prime Curve 521 bit (EC_secp521r1)
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The certificate must be a valid SSL/TLS X.509 version 3 certificate with FQDN or IP address specified
         *        and information about the issuer.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificate(String certificate);

        /**
         * <p>
         * The virtual private cloud (VPC) endpoint settings that are configured for your server. When you host your
         * endpoint within your VPC, you can make it accessible only to resources within your VPC, or you can attach
         * Elastic IPs and make it accessible to clients over the internet. Your VPC's default security groups are
         * automatically assigned to your endpoint.
         * </p>
         * 
         * @param endpointDetails
         *        The virtual private cloud (VPC) endpoint settings that are configured for your server. When you host
         *        your endpoint within your VPC, you can make it accessible only to resources within your VPC, or you
         *        can attach Elastic IPs and make it accessible to clients over the internet. Your VPC's default
         *        security groups are automatically assigned to your endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointDetails(EndpointDetails endpointDetails);

        /**
         * <p>
         * The virtual private cloud (VPC) endpoint settings that are configured for your server. When you host your
         * endpoint within your VPC, you can make it accessible only to resources within your VPC, or you can attach
         * Elastic IPs and make it accessible to clients over the internet. Your VPC's default security groups are
         * automatically assigned to your endpoint.
         * </p>
         * This is a convenience that creates an instance of the {@link EndpointDetails.Builder} avoiding the need to
         * create one manually via {@link EndpointDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link EndpointDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #endpointDetails(EndpointDetails)}.
         * 
         * @param endpointDetails
         *        a consumer that will call methods on {@link EndpointDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #endpointDetails(EndpointDetails)
         */
        default Builder endpointDetails(Consumer<EndpointDetails.Builder> endpointDetails) {
            return endpointDetails(EndpointDetails.builder().applyMutation(endpointDetails).build());
        }

        /**
         * <p>
         * The type of VPC endpoint that you want your server to connect to. You can choose to connect to the public
         * internet or a VPC endpoint. With a VPC endpoint, you can restrict access to your server and resources only
         * within your VPC.
         * </p>
         * <note>
         * <p>
         * It is recommended that you use <code>VPC</code> as the <code>EndpointType</code>. With this endpoint type,
         * you have the option to directly associate up to three Elastic IPv4 addresses (BYO IP included) with your
         * server's endpoint and use VPC security groups to restrict traffic by the client's public IP address. This is
         * not possible with <code>EndpointType</code> set to <code>VPC_ENDPOINT</code>.
         * </p>
         * </note>
         * 
         * @param endpointType
         *        The type of VPC endpoint that you want your server to connect to. You can choose to connect to the
         *        public internet or a VPC endpoint. With a VPC endpoint, you can restrict access to your server and
         *        resources only within your VPC.</p> <note>
         *        <p>
         *        It is recommended that you use <code>VPC</code> as the <code>EndpointType</code>. With this endpoint
         *        type, you have the option to directly associate up to three Elastic IPv4 addresses (BYO IP included)
         *        with your server's endpoint and use VPC security groups to restrict traffic by the client's public IP
         *        address. This is not possible with <code>EndpointType</code> set to <code>VPC_ENDPOINT</code>.
         *        </p>
         * @see EndpointType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EndpointType
         */
        Builder endpointType(String endpointType);

        /**
         * <p>
         * The type of VPC endpoint that you want your server to connect to. You can choose to connect to the public
         * internet or a VPC endpoint. With a VPC endpoint, you can restrict access to your server and resources only
         * within your VPC.
         * </p>
         * <note>
         * <p>
         * It is recommended that you use <code>VPC</code> as the <code>EndpointType</code>. With this endpoint type,
         * you have the option to directly associate up to three Elastic IPv4 addresses (BYO IP included) with your
         * server's endpoint and use VPC security groups to restrict traffic by the client's public IP address. This is
         * not possible with <code>EndpointType</code> set to <code>VPC_ENDPOINT</code>.
         * </p>
         * </note>
         * 
         * @param endpointType
         *        The type of VPC endpoint that you want your server to connect to. You can choose to connect to the
         *        public internet or a VPC endpoint. With a VPC endpoint, you can restrict access to your server and
         *        resources only within your VPC.</p> <note>
         *        <p>
         *        It is recommended that you use <code>VPC</code> as the <code>EndpointType</code>. With this endpoint
         *        type, you have the option to directly associate up to three Elastic IPv4 addresses (BYO IP included)
         *        with your server's endpoint and use VPC security groups to restrict traffic by the client's public IP
         *        address. This is not possible with <code>EndpointType</code> set to <code>VPC_ENDPOINT</code>.
         *        </p>
         * @see EndpointType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EndpointType
         */
        Builder endpointType(EndpointType endpointType);

        /**
         * <p>
         * The RSA private key as generated by the <code>ssh-keygen -N "" -m PEM -f my-new-server-key</code> command.
         * </p>
         * <important>
         * <p>
         * If you aren't planning to migrate existing users from an existing SFTP-enabled server to a new server, don't
         * update the host key. Accidentally changing a server's host key can be disruptive.
         * </p>
         * </important>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/transfer/latest/userguide/edit-server-config.html#configuring-servers-change-host-key"
         * >Change the host key for your SFTP-enabled server</a> in the <i>AWS Transfer Family User Guide</i>.
         * </p>
         * 
         * @param hostKey
         *        The RSA private key as generated by the <code>ssh-keygen -N "" -m PEM -f my-new-server-key</code>
         *        command.</p> <important>
         *        <p>
         *        If you aren't planning to migrate existing users from an existing SFTP-enabled server to a new server,
         *        don't update the host key. Accidentally changing a server's host key can be disruptive.
         *        </p>
         *        </important>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/transfer/latest/userguide/edit-server-config.html#configuring-servers-change-host-key"
         *        >Change the host key for your SFTP-enabled server</a> in the <i>AWS Transfer Family User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hostKey(String hostKey);

        /**
         * <p>
         * Required when <code>IdentityProviderType</code> is set to <code>API_GATEWAY</code>. Accepts an array
         * containing all of the information required to call a customer-supplied authentication API, including the API
         * Gateway URL. Not required when <code>IdentityProviderType</code> is set to <code>SERVICE_MANAGED</code>.
         * </p>
         * 
         * @param identityProviderDetails
         *        Required when <code>IdentityProviderType</code> is set to <code>API_GATEWAY</code>. Accepts an array
         *        containing all of the information required to call a customer-supplied authentication API, including
         *        the API Gateway URL. Not required when <code>IdentityProviderType</code> is set to
         *        <code>SERVICE_MANAGED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identityProviderDetails(IdentityProviderDetails identityProviderDetails);

        /**
         * <p>
         * Required when <code>IdentityProviderType</code> is set to <code>API_GATEWAY</code>. Accepts an array
         * containing all of the information required to call a customer-supplied authentication API, including the API
         * Gateway URL. Not required when <code>IdentityProviderType</code> is set to <code>SERVICE_MANAGED</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link IdentityProviderDetails.Builder} avoiding the
         * need to create one manually via {@link IdentityProviderDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link IdentityProviderDetails.Builder#build()} is called immediately
         * and its result is passed to {@link #identityProviderDetails(IdentityProviderDetails)}.
         * 
         * @param identityProviderDetails
         *        a consumer that will call methods on {@link IdentityProviderDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #identityProviderDetails(IdentityProviderDetails)
         */
        default Builder identityProviderDetails(Consumer<IdentityProviderDetails.Builder> identityProviderDetails) {
            return identityProviderDetails(IdentityProviderDetails.builder().applyMutation(identityProviderDetails).build());
        }

        /**
         * <p>
         * Specifies the mode of authentication for a server. The default value is <code>SERVICE_MANAGED</code>, which
         * allows you to store and access user credentials within the AWS Transfer Family service. Use the
         * <code>API_GATEWAY</code> value to integrate with an identity provider of your choosing. The
         * <code>API_GATEWAY</code> setting requires you to provide an API Gateway endpoint URL to call for
         * authentication using the <code>IdentityProviderDetails</code> parameter.
         * </p>
         * 
         * @param identityProviderType
         *        Specifies the mode of authentication for a server. The default value is <code>SERVICE_MANAGED</code>,
         *        which allows you to store and access user credentials within the AWS Transfer Family service. Use the
         *        <code>API_GATEWAY</code> value to integrate with an identity provider of your choosing. The
         *        <code>API_GATEWAY</code> setting requires you to provide an API Gateway endpoint URL to call for
         *        authentication using the <code>IdentityProviderDetails</code> parameter.
         * @see IdentityProviderType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IdentityProviderType
         */
        Builder identityProviderType(String identityProviderType);

        /**
         * <p>
         * Specifies the mode of authentication for a server. The default value is <code>SERVICE_MANAGED</code>, which
         * allows you to store and access user credentials within the AWS Transfer Family service. Use the
         * <code>API_GATEWAY</code> value to integrate with an identity provider of your choosing. The
         * <code>API_GATEWAY</code> setting requires you to provide an API Gateway endpoint URL to call for
         * authentication using the <code>IdentityProviderDetails</code> parameter.
         * </p>
         * 
         * @param identityProviderType
         *        Specifies the mode of authentication for a server. The default value is <code>SERVICE_MANAGED</code>,
         *        which allows you to store and access user credentials within the AWS Transfer Family service. Use the
         *        <code>API_GATEWAY</code> value to integrate with an identity provider of your choosing. The
         *        <code>API_GATEWAY</code> setting requires you to provide an API Gateway endpoint URL to call for
         *        authentication using the <code>IdentityProviderDetails</code> parameter.
         * @see IdentityProviderType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IdentityProviderType
         */
        Builder identityProviderType(IdentityProviderType identityProviderType);

        /**
         * <p>
         * Allows the service to write your users' activity to your Amazon CloudWatch logs for monitoring and auditing
         * purposes.
         * </p>
         * 
         * @param loggingRole
         *        Allows the service to write your users' activity to your Amazon CloudWatch logs for monitoring and
         *        auditing purposes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loggingRole(String loggingRole);

        /**
         * <p>
         * Specifies the file transfer protocol or protocols over which your file transfer protocol client can connect
         * to your server's endpoint. The available protocols are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * If you select <code>FTPS</code>, you must choose a certificate stored in AWS Certificate Manager (ACM) which
         * will be used to identify your server when clients connect to it over FTPS.
         * </p>
         * <p>
         * If <code>Protocol</code> includes either <code>FTP</code> or <code>FTPS</code>, then the
         * <code>EndpointType</code> must be <code>VPC</code> and the <code>IdentityProviderType</code> must be
         * <code>API_GATEWAY</code>.
         * </p>
         * <p>
         * If <code>Protocol</code> includes <code>FTP</code>, then <code>AddressAllocationIds</code> cannot be
         * associated.
         * </p>
         * <p>
         * If <code>Protocol</code> is set only to <code>SFTP</code>, the <code>EndpointType</code> can be set to
         * <code>PUBLIC</code> and the <code>IdentityProviderType</code> can be set to <code>SERVICE_MANAGED</code>.
         * </p>
         * </note>
         * 
         * @param protocols
         *        Specifies the file transfer protocol or protocols over which your file transfer protocol client can
         *        connect to your server's endpoint. The available protocols are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        If you select <code>FTPS</code>, you must choose a certificate stored in AWS Certificate Manager (ACM)
         *        which will be used to identify your server when clients connect to it over FTPS.
         *        </p>
         *        <p>
         *        If <code>Protocol</code> includes either <code>FTP</code> or <code>FTPS</code>, then the
         *        <code>EndpointType</code> must be <code>VPC</code> and the <code>IdentityProviderType</code> must be
         *        <code>API_GATEWAY</code>.
         *        </p>
         *        <p>
         *        If <code>Protocol</code> includes <code>FTP</code>, then <code>AddressAllocationIds</code> cannot be
         *        associated.
         *        </p>
         *        <p>
         *        If <code>Protocol</code> is set only to <code>SFTP</code>, the <code>EndpointType</code> can be set to
         *        <code>PUBLIC</code> and the <code>IdentityProviderType</code> can be set to
         *        <code>SERVICE_MANAGED</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocolsWithStrings(Collection<String> protocols);

        /**
         * <p>
         * Specifies the file transfer protocol or protocols over which your file transfer protocol client can connect
         * to your server's endpoint. The available protocols are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * If you select <code>FTPS</code>, you must choose a certificate stored in AWS Certificate Manager (ACM) which
         * will be used to identify your server when clients connect to it over FTPS.
         * </p>
         * <p>
         * If <code>Protocol</code> includes either <code>FTP</code> or <code>FTPS</code>, then the
         * <code>EndpointType</code> must be <code>VPC</code> and the <code>IdentityProviderType</code> must be
         * <code>API_GATEWAY</code>.
         * </p>
         * <p>
         * If <code>Protocol</code> includes <code>FTP</code>, then <code>AddressAllocationIds</code> cannot be
         * associated.
         * </p>
         * <p>
         * If <code>Protocol</code> is set only to <code>SFTP</code>, the <code>EndpointType</code> can be set to
         * <code>PUBLIC</code> and the <code>IdentityProviderType</code> can be set to <code>SERVICE_MANAGED</code>.
         * </p>
         * </note>
         * 
         * @param protocols
         *        Specifies the file transfer protocol or protocols over which your file transfer protocol client can
         *        connect to your server's endpoint. The available protocols are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        If you select <code>FTPS</code>, you must choose a certificate stored in AWS Certificate Manager (ACM)
         *        which will be used to identify your server when clients connect to it over FTPS.
         *        </p>
         *        <p>
         *        If <code>Protocol</code> includes either <code>FTP</code> or <code>FTPS</code>, then the
         *        <code>EndpointType</code> must be <code>VPC</code> and the <code>IdentityProviderType</code> must be
         *        <code>API_GATEWAY</code>.
         *        </p>
         *        <p>
         *        If <code>Protocol</code> includes <code>FTP</code>, then <code>AddressAllocationIds</code> cannot be
         *        associated.
         *        </p>
         *        <p>
         *        If <code>Protocol</code> is set only to <code>SFTP</code>, the <code>EndpointType</code> can be set to
         *        <code>PUBLIC</code> and the <code>IdentityProviderType</code> can be set to
         *        <code>SERVICE_MANAGED</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocolsWithStrings(String... protocols);

        /**
         * <p>
         * Specifies the file transfer protocol or protocols over which your file transfer protocol client can connect
         * to your server's endpoint. The available protocols are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * If you select <code>FTPS</code>, you must choose a certificate stored in AWS Certificate Manager (ACM) which
         * will be used to identify your server when clients connect to it over FTPS.
         * </p>
         * <p>
         * If <code>Protocol</code> includes either <code>FTP</code> or <code>FTPS</code>, then the
         * <code>EndpointType</code> must be <code>VPC</code> and the <code>IdentityProviderType</code> must be
         * <code>API_GATEWAY</code>.
         * </p>
         * <p>
         * If <code>Protocol</code> includes <code>FTP</code>, then <code>AddressAllocationIds</code> cannot be
         * associated.
         * </p>
         * <p>
         * If <code>Protocol</code> is set only to <code>SFTP</code>, the <code>EndpointType</code> can be set to
         * <code>PUBLIC</code> and the <code>IdentityProviderType</code> can be set to <code>SERVICE_MANAGED</code>.
         * </p>
         * </note>
         * 
         * @param protocols
         *        Specifies the file transfer protocol or protocols over which your file transfer protocol client can
         *        connect to your server's endpoint. The available protocols are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        If you select <code>FTPS</code>, you must choose a certificate stored in AWS Certificate Manager (ACM)
         *        which will be used to identify your server when clients connect to it over FTPS.
         *        </p>
         *        <p>
         *        If <code>Protocol</code> includes either <code>FTP</code> or <code>FTPS</code>, then the
         *        <code>EndpointType</code> must be <code>VPC</code> and the <code>IdentityProviderType</code> must be
         *        <code>API_GATEWAY</code>.
         *        </p>
         *        <p>
         *        If <code>Protocol</code> includes <code>FTP</code>, then <code>AddressAllocationIds</code> cannot be
         *        associated.
         *        </p>
         *        <p>
         *        If <code>Protocol</code> is set only to <code>SFTP</code>, the <code>EndpointType</code> can be set to
         *        <code>PUBLIC</code> and the <code>IdentityProviderType</code> can be set to
         *        <code>SERVICE_MANAGED</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocols(Collection<Protocol> protocols);

        /**
         * <p>
         * Specifies the file transfer protocol or protocols over which your file transfer protocol client can connect
         * to your server's endpoint. The available protocols are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * If you select <code>FTPS</code>, you must choose a certificate stored in AWS Certificate Manager (ACM) which
         * will be used to identify your server when clients connect to it over FTPS.
         * </p>
         * <p>
         * If <code>Protocol</code> includes either <code>FTP</code> or <code>FTPS</code>, then the
         * <code>EndpointType</code> must be <code>VPC</code> and the <code>IdentityProviderType</code> must be
         * <code>API_GATEWAY</code>.
         * </p>
         * <p>
         * If <code>Protocol</code> includes <code>FTP</code>, then <code>AddressAllocationIds</code> cannot be
         * associated.
         * </p>
         * <p>
         * If <code>Protocol</code> is set only to <code>SFTP</code>, the <code>EndpointType</code> can be set to
         * <code>PUBLIC</code> and the <code>IdentityProviderType</code> can be set to <code>SERVICE_MANAGED</code>.
         * </p>
         * </note>
         * 
         * @param protocols
         *        Specifies the file transfer protocol or protocols over which your file transfer protocol client can
         *        connect to your server's endpoint. The available protocols are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        If you select <code>FTPS</code>, you must choose a certificate stored in AWS Certificate Manager (ACM)
         *        which will be used to identify your server when clients connect to it over FTPS.
         *        </p>
         *        <p>
         *        If <code>Protocol</code> includes either <code>FTP</code> or <code>FTPS</code>, then the
         *        <code>EndpointType</code> must be <code>VPC</code> and the <code>IdentityProviderType</code> must be
         *        <code>API_GATEWAY</code>.
         *        </p>
         *        <p>
         *        If <code>Protocol</code> includes <code>FTP</code>, then <code>AddressAllocationIds</code> cannot be
         *        associated.
         *        </p>
         *        <p>
         *        If <code>Protocol</code> is set only to <code>SFTP</code>, the <code>EndpointType</code> can be set to
         *        <code>PUBLIC</code> and the <code>IdentityProviderType</code> can be set to
         *        <code>SERVICE_MANAGED</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocols(Protocol... protocols);

        /**
         * <p>
         * Specifies the name of the security policy that is attached to the server.
         * </p>
         * 
         * @param securityPolicyName
         *        Specifies the name of the security policy that is attached to the server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityPolicyName(String securityPolicyName);

        /**
         * <p>
         * Key-value pairs that can be used to group and search for servers.
         * </p>
         * 
         * @param tags
         *        Key-value pairs that can be used to group and search for servers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Key-value pairs that can be used to group and search for servers.
         * </p>
         * 
         * @param tags
         *        Key-value pairs that can be used to group and search for servers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Key-value pairs that can be used to group and search for servers.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends TransferRequest.BuilderImpl implements Builder {
        private String certificate;

        private EndpointDetails endpointDetails;

        private String endpointType;

        private String hostKey;

        private IdentityProviderDetails identityProviderDetails;

        private String identityProviderType;

        private String loggingRole;

        private List<String> protocols = DefaultSdkAutoConstructList.getInstance();

        private String securityPolicyName;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateServerRequest model) {
            super(model);
            certificate(model.certificate);
            endpointDetails(model.endpointDetails);
            endpointType(model.endpointType);
            hostKey(model.hostKey);
            identityProviderDetails(model.identityProviderDetails);
            identityProviderType(model.identityProviderType);
            loggingRole(model.loggingRole);
            protocolsWithStrings(model.protocols);
            securityPolicyName(model.securityPolicyName);
            tags(model.tags);
        }

        public final String getCertificate() {
            return certificate;
        }

        @Override
        public final Builder certificate(String certificate) {
            this.certificate = certificate;
            return this;
        }

        public final void setCertificate(String certificate) {
            this.certificate = certificate;
        }

        public final EndpointDetails.Builder getEndpointDetails() {
            return endpointDetails != null ? endpointDetails.toBuilder() : null;
        }

        @Override
        public final Builder endpointDetails(EndpointDetails endpointDetails) {
            this.endpointDetails = endpointDetails;
            return this;
        }

        public final void setEndpointDetails(EndpointDetails.BuilderImpl endpointDetails) {
            this.endpointDetails = endpointDetails != null ? endpointDetails.build() : null;
        }

        public final String getEndpointType() {
            return endpointType;
        }

        @Override
        public final Builder endpointType(String endpointType) {
            this.endpointType = endpointType;
            return this;
        }

        @Override
        public final Builder endpointType(EndpointType endpointType) {
            this.endpointType(endpointType == null ? null : endpointType.toString());
            return this;
        }

        public final void setEndpointType(String endpointType) {
            this.endpointType = endpointType;
        }

        public final String getHostKey() {
            return hostKey;
        }

        @Override
        public final Builder hostKey(String hostKey) {
            this.hostKey = hostKey;
            return this;
        }

        public final void setHostKey(String hostKey) {
            this.hostKey = hostKey;
        }

        public final IdentityProviderDetails.Builder getIdentityProviderDetails() {
            return identityProviderDetails != null ? identityProviderDetails.toBuilder() : null;
        }

        @Override
        public final Builder identityProviderDetails(IdentityProviderDetails identityProviderDetails) {
            this.identityProviderDetails = identityProviderDetails;
            return this;
        }

        public final void setIdentityProviderDetails(IdentityProviderDetails.BuilderImpl identityProviderDetails) {
            this.identityProviderDetails = identityProviderDetails != null ? identityProviderDetails.build() : null;
        }

        public final String getIdentityProviderType() {
            return identityProviderType;
        }

        @Override
        public final Builder identityProviderType(String identityProviderType) {
            this.identityProviderType = identityProviderType;
            return this;
        }

        @Override
        public final Builder identityProviderType(IdentityProviderType identityProviderType) {
            this.identityProviderType(identityProviderType == null ? null : identityProviderType.toString());
            return this;
        }

        public final void setIdentityProviderType(String identityProviderType) {
            this.identityProviderType = identityProviderType;
        }

        public final String getLoggingRole() {
            return loggingRole;
        }

        @Override
        public final Builder loggingRole(String loggingRole) {
            this.loggingRole = loggingRole;
            return this;
        }

        public final void setLoggingRole(String loggingRole) {
            this.loggingRole = loggingRole;
        }

        public final Collection<String> getProtocols() {
            return protocols;
        }

        @Override
        public final Builder protocolsWithStrings(Collection<String> protocols) {
            this.protocols = ProtocolsCopier.copy(protocols);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder protocolsWithStrings(String... protocols) {
            protocolsWithStrings(Arrays.asList(protocols));
            return this;
        }

        @Override
        public final Builder protocols(Collection<Protocol> protocols) {
            this.protocols = ProtocolsCopier.copyEnumToString(protocols);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder protocols(Protocol... protocols) {
            protocols(Arrays.asList(protocols));
            return this;
        }

        public final void setProtocols(Collection<String> protocols) {
            this.protocols = ProtocolsCopier.copy(protocols);
        }

        public final String getSecurityPolicyName() {
            return securityPolicyName;
        }

        @Override
        public final Builder securityPolicyName(String securityPolicyName) {
            this.securityPolicyName = securityPolicyName;
            return this;
        }

        public final void setSecurityPolicyName(String securityPolicyName) {
            this.securityPolicyName = securityPolicyName;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateServerRequest build() {
            return new CreateServerRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
