/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transfer.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateUserRequest extends TransferRequest implements
        ToCopyableBuilder<CreateUserRequest.Builder, CreateUserRequest> {
    private static final SdkField<String> HOME_DIRECTORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HomeDirectory").getter(getter(CreateUserRequest::homeDirectory)).setter(setter(Builder::homeDirectory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HomeDirectory").build()).build();

    private static final SdkField<String> HOME_DIRECTORY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HomeDirectoryType").getter(getter(CreateUserRequest::homeDirectoryTypeAsString))
            .setter(setter(Builder::homeDirectoryType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HomeDirectoryType").build()).build();

    private static final SdkField<List<HomeDirectoryMapEntry>> HOME_DIRECTORY_MAPPINGS_FIELD = SdkField
            .<List<HomeDirectoryMapEntry>> builder(MarshallingType.LIST)
            .memberName("HomeDirectoryMappings")
            .getter(getter(CreateUserRequest::homeDirectoryMappings))
            .setter(setter(Builder::homeDirectoryMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HomeDirectoryMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<HomeDirectoryMapEntry> builder(MarshallingType.SDK_POJO)
                                            .constructor(HomeDirectoryMapEntry::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Policy")
            .getter(getter(CreateUserRequest::policy)).setter(setter(Builder::policy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Policy").build()).build();

    private static final SdkField<String> ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Role")
            .getter(getter(CreateUserRequest::role)).setter(setter(Builder::role))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Role").build()).build();

    private static final SdkField<String> SERVER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServerId").getter(getter(CreateUserRequest::serverId)).setter(setter(Builder::serverId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerId").build()).build();

    private static final SdkField<String> SSH_PUBLIC_KEY_BODY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SshPublicKeyBody").getter(getter(CreateUserRequest::sshPublicKeyBody))
            .setter(setter(Builder::sshPublicKeyBody))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SshPublicKeyBody").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateUserRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> USER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UserName").getter(getter(CreateUserRequest::userName)).setter(setter(Builder::userName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserName").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(HOME_DIRECTORY_FIELD,
            HOME_DIRECTORY_TYPE_FIELD, HOME_DIRECTORY_MAPPINGS_FIELD, POLICY_FIELD, ROLE_FIELD, SERVER_ID_FIELD,
            SSH_PUBLIC_KEY_BODY_FIELD, TAGS_FIELD, USER_NAME_FIELD));

    private final String homeDirectory;

    private final String homeDirectoryType;

    private final List<HomeDirectoryMapEntry> homeDirectoryMappings;

    private final String policy;

    private final String role;

    private final String serverId;

    private final String sshPublicKeyBody;

    private final List<Tag> tags;

    private final String userName;

    private CreateUserRequest(BuilderImpl builder) {
        super(builder);
        this.homeDirectory = builder.homeDirectory;
        this.homeDirectoryType = builder.homeDirectoryType;
        this.homeDirectoryMappings = builder.homeDirectoryMappings;
        this.policy = builder.policy;
        this.role = builder.role;
        this.serverId = builder.serverId;
        this.sshPublicKeyBody = builder.sshPublicKeyBody;
        this.tags = builder.tags;
        this.userName = builder.userName;
    }

    /**
     * <p>
     * The landing directory (folder) for a user when they log in to the server using the client.
     * </p>
     * <p>
     * An example is <i> <code>your-Amazon-S3-bucket-name&gt;/home/username</code> </i>.
     * </p>
     * 
     * @return The landing directory (folder) for a user when they log in to the server using the client.</p>
     *         <p>
     *         An example is <i> <code>your-Amazon-S3-bucket-name&gt;/home/username</code> </i>.
     */
    public final String homeDirectory() {
        return homeDirectory;
    }

    /**
     * <p>
     * The type of landing directory (folder) you want your users' home directory to be when they log into the server.
     * If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket paths as is in their file
     * transfer protocol clients. If you set it <code>LOGICAL</code>, you will need to provide mappings in the
     * <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 paths visible to your users.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #homeDirectoryType}
     * will return {@link HomeDirectoryType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #homeDirectoryTypeAsString}.
     * </p>
     * 
     * @return The type of landing directory (folder) you want your users' home directory to be when they log into the
     *         server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket paths as is
     *         in their file transfer protocol clients. If you set it <code>LOGICAL</code>, you will need to provide
     *         mappings in the <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 paths visible to
     *         your users.
     * @see HomeDirectoryType
     */
    public final HomeDirectoryType homeDirectoryType() {
        return HomeDirectoryType.fromValue(homeDirectoryType);
    }

    /**
     * <p>
     * The type of landing directory (folder) you want your users' home directory to be when they log into the server.
     * If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket paths as is in their file
     * transfer protocol clients. If you set it <code>LOGICAL</code>, you will need to provide mappings in the
     * <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 paths visible to your users.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #homeDirectoryType}
     * will return {@link HomeDirectoryType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #homeDirectoryTypeAsString}.
     * </p>
     * 
     * @return The type of landing directory (folder) you want your users' home directory to be when they log into the
     *         server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket paths as is
     *         in their file transfer protocol clients. If you set it <code>LOGICAL</code>, you will need to provide
     *         mappings in the <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 paths visible to
     *         your users.
     * @see HomeDirectoryType
     */
    public final String homeDirectoryTypeAsString() {
        return homeDirectoryType;
    }

    /**
     * Returns true if the HomeDirectoryMappings property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasHomeDirectoryMappings() {
        return homeDirectoryMappings != null && !(homeDirectoryMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Logical directory mappings that specify what Amazon S3 paths and keys should be visible to your user and how you
     * want to make them visible. You will need to specify the "<code>Entry</code>" and "<code>Target</code>" pair,
     * where <code>Entry</code> shows how the path is made visible and <code>Target</code> is the actual Amazon S3 path.
     * If you only specify a target, it will be displayed as is. You will need to also make sure that your IAM role
     * provides access to paths in <code>Target</code>. The following is an example.
     * </p>
     * <p>
     * <code>'[ "/bucket2/documentation", { "Entry": "your-personal-report.pdf", "Target": "/bucket3/customized-reports/${transfer:UserName}.pdf" } ]'</code>
     * </p>
     * <p>
     * In most cases, you can use this value instead of the scope-down policy to lock your user down to the designated
     * home directory ("chroot"). To do this, you can set <code>Entry</code> to '/' and set <code>Target</code> to the
     * HomeDirectory parameter value.
     * </p>
     * <note>
     * <p>
     * If the target of a logical directory entry does not exist in Amazon S3, the entry will be ignored. As a
     * workaround, you can use the Amazon S3 API to create 0 byte objects as place holders for your directory. If using
     * the CLI, use the <code>s3api</code> call instead of <code>s3</code> so you can use the put-object operation. For
     * example, you use the following: <code>aws s3api put-object --bucket bucketname --key path/to/folder/</code>. Make
     * sure that the end of the key name ends in a '/' for it to be considered a folder.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasHomeDirectoryMappings()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Logical directory mappings that specify what Amazon S3 paths and keys should be visible to your user and
     *         how you want to make them visible. You will need to specify the "<code>Entry</code>" and "
     *         <code>Target</code>" pair, where <code>Entry</code> shows how the path is made visible and
     *         <code>Target</code> is the actual Amazon S3 path. If you only specify a target, it will be displayed as
     *         is. You will need to also make sure that your IAM role provides access to paths in <code>Target</code>.
     *         The following is an example.</p>
     *         <p>
     *         <code>'[ "/bucket2/documentation", { "Entry": "your-personal-report.pdf", "Target": "/bucket3/customized-reports/${transfer:UserName}.pdf" } ]'</code>
     *         </p>
     *         <p>
     *         In most cases, you can use this value instead of the scope-down policy to lock your user down to the
     *         designated home directory ("chroot"). To do this, you can set <code>Entry</code> to '/' and set
     *         <code>Target</code> to the HomeDirectory parameter value.
     *         </p>
     *         <note>
     *         <p>
     *         If the target of a logical directory entry does not exist in Amazon S3, the entry will be ignored. As a
     *         workaround, you can use the Amazon S3 API to create 0 byte objects as place holders for your directory.
     *         If using the CLI, use the <code>s3api</code> call instead of <code>s3</code> so you can use the
     *         put-object operation. For example, you use the following:
     *         <code>aws s3api put-object --bucket bucketname --key path/to/folder/</code>. Make sure that the end of
     *         the key name ends in a '/' for it to be considered a folder.
     *         </p>
     */
    public final List<HomeDirectoryMapEntry> homeDirectoryMappings() {
        return homeDirectoryMappings;
    }

    /**
     * <p>
     * A scope-down policy for your user so you can use the same IAM role across multiple users. This policy scopes down
     * user access to portions of their Amazon S3 bucket. Variables that you can use inside this policy include
     * <code>${Transfer:UserName}</code>, <code>${Transfer:HomeDirectory}</code>, and
     * <code>${Transfer:HomeBucket}</code>.
     * </p>
     * <note>
     * <p>
     * For scope-down policies, AWS Transfer Family stores the policy as a JSON blob, instead of the Amazon Resource
     * Name (ARN) of the policy. You save the policy as a JSON blob and pass it in the <code>Policy</code> argument.
     * </p>
     * <p>
     * For an example of a scope-down policy, see <a
     * href="https://docs.aws.amazon.com/transfer/latest/userguide/users.html#users-policies-scope-down">Creating a
     * scope-down policy</a>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/STS/latest/APIReference/API_AssumeRole.html">AssumeRole</a> in the <i>AWS
     * Security Token Service API Reference</i>.
     * </p>
     * </note>
     * 
     * @return A scope-down policy for your user so you can use the same IAM role across multiple users. This policy
     *         scopes down user access to portions of their Amazon S3 bucket. Variables that you can use inside this
     *         policy include <code>${Transfer:UserName}</code>, <code>${Transfer:HomeDirectory}</code>, and
     *         <code>${Transfer:HomeBucket}</code>.</p> <note>
     *         <p>
     *         For scope-down policies, AWS Transfer Family stores the policy as a JSON blob, instead of the Amazon
     *         Resource Name (ARN) of the policy. You save the policy as a JSON blob and pass it in the
     *         <code>Policy</code> argument.
     *         </p>
     *         <p>
     *         For an example of a scope-down policy, see <a
     *         href="https://docs.aws.amazon.com/transfer/latest/userguide/users.html#users-policies-scope-down"
     *         >Creating a scope-down policy</a>.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/STS/latest/APIReference/API_AssumeRole.html">AssumeRole</a> in the
     *         <i>AWS Security Token Service API Reference</i>.
     *         </p>
     */
    public final String policy() {
        return policy;
    }

    /**
     * <p>
     * The IAM role that controls your users' access to your Amazon S3 bucket. The policies attached to this role will
     * determine the level of access you want to provide your users when transferring files into and out of your Amazon
     * S3 bucket or buckets. The IAM role should also contain a trust relationship that allows the server to access your
     * resources when servicing your users' transfer requests.
     * </p>
     * 
     * @return The IAM role that controls your users' access to your Amazon S3 bucket. The policies attached to this
     *         role will determine the level of access you want to provide your users when transferring files into and
     *         out of your Amazon S3 bucket or buckets. The IAM role should also contain a trust relationship that
     *         allows the server to access your resources when servicing your users' transfer requests.
     */
    public final String role() {
        return role;
    }

    /**
     * <p>
     * A system-assigned unique identifier for a server instance. This is the specific server that you added your user
     * to.
     * </p>
     * 
     * @return A system-assigned unique identifier for a server instance. This is the specific server that you added
     *         your user to.
     */
    public final String serverId() {
        return serverId;
    }

    /**
     * <p>
     * The public portion of the Secure Shell (SSH) key used to authenticate the user to the server.
     * </p>
     * 
     * @return The public portion of the Secure Shell (SSH) key used to authenticate the user to the server.
     */
    public final String sshPublicKeyBody() {
        return sshPublicKeyBody;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Key-value pairs that can be used to group and search for users. Tags are metadata attached to users for any
     * purpose.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Key-value pairs that can be used to group and search for users. Tags are metadata attached to users for
     *         any purpose.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * A unique string that identifies a user and is associated with a as specified by the <code>ServerId</code>. This
     * user name must be a minimum of 3 and a maximum of 100 characters long. The following are valid characters: a-z,
     * A-Z, 0-9, underscore '_', hyphen '-', period '.', and at sign '@'. The user name can't start with a hyphen,
     * period, or at sign.
     * </p>
     * 
     * @return A unique string that identifies a user and is associated with a as specified by the <code>ServerId</code>
     *         . This user name must be a minimum of 3 and a maximum of 100 characters long. The following are valid
     *         characters: a-z, A-Z, 0-9, underscore '_', hyphen '-', period '.', and at sign '@'. The user name can't
     *         start with a hyphen, period, or at sign.
     */
    public final String userName() {
        return userName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(homeDirectory());
        hashCode = 31 * hashCode + Objects.hashCode(homeDirectoryTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasHomeDirectoryMappings() ? homeDirectoryMappings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(policy());
        hashCode = 31 * hashCode + Objects.hashCode(role());
        hashCode = 31 * hashCode + Objects.hashCode(serverId());
        hashCode = 31 * hashCode + Objects.hashCode(sshPublicKeyBody());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(userName());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateUserRequest)) {
            return false;
        }
        CreateUserRequest other = (CreateUserRequest) obj;
        return Objects.equals(homeDirectory(), other.homeDirectory())
                && Objects.equals(homeDirectoryTypeAsString(), other.homeDirectoryTypeAsString())
                && hasHomeDirectoryMappings() == other.hasHomeDirectoryMappings()
                && Objects.equals(homeDirectoryMappings(), other.homeDirectoryMappings())
                && Objects.equals(policy(), other.policy()) && Objects.equals(role(), other.role())
                && Objects.equals(serverId(), other.serverId()) && Objects.equals(sshPublicKeyBody(), other.sshPublicKeyBody())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(userName(), other.userName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateUserRequest").add("HomeDirectory", homeDirectory())
                .add("HomeDirectoryType", homeDirectoryTypeAsString())
                .add("HomeDirectoryMappings", hasHomeDirectoryMappings() ? homeDirectoryMappings() : null)
                .add("Policy", policy()).add("Role", role()).add("ServerId", serverId())
                .add("SshPublicKeyBody", sshPublicKeyBody()).add("Tags", hasTags() ? tags() : null).add("UserName", userName())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "HomeDirectory":
            return Optional.ofNullable(clazz.cast(homeDirectory()));
        case "HomeDirectoryType":
            return Optional.ofNullable(clazz.cast(homeDirectoryTypeAsString()));
        case "HomeDirectoryMappings":
            return Optional.ofNullable(clazz.cast(homeDirectoryMappings()));
        case "Policy":
            return Optional.ofNullable(clazz.cast(policy()));
        case "Role":
            return Optional.ofNullable(clazz.cast(role()));
        case "ServerId":
            return Optional.ofNullable(clazz.cast(serverId()));
        case "SshPublicKeyBody":
            return Optional.ofNullable(clazz.cast(sshPublicKeyBody()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "UserName":
            return Optional.ofNullable(clazz.cast(userName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateUserRequest, T> g) {
        return obj -> g.apply((CreateUserRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends TransferRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateUserRequest> {
        /**
         * <p>
         * The landing directory (folder) for a user when they log in to the server using the client.
         * </p>
         * <p>
         * An example is <i> <code>your-Amazon-S3-bucket-name&gt;/home/username</code> </i>.
         * </p>
         * 
         * @param homeDirectory
         *        The landing directory (folder) for a user when they log in to the server using the client.</p>
         *        <p>
         *        An example is <i> <code>your-Amazon-S3-bucket-name&gt;/home/username</code> </i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder homeDirectory(String homeDirectory);

        /**
         * <p>
         * The type of landing directory (folder) you want your users' home directory to be when they log into the
         * server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket paths as is in
         * their file transfer protocol clients. If you set it <code>LOGICAL</code>, you will need to provide mappings
         * in the <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 paths visible to your users.
         * </p>
         * 
         * @param homeDirectoryType
         *        The type of landing directory (folder) you want your users' home directory to be when they log into
         *        the server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket paths
         *        as is in their file transfer protocol clients. If you set it <code>LOGICAL</code>, you will need to
         *        provide mappings in the <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 paths
         *        visible to your users.
         * @see HomeDirectoryType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HomeDirectoryType
         */
        Builder homeDirectoryType(String homeDirectoryType);

        /**
         * <p>
         * The type of landing directory (folder) you want your users' home directory to be when they log into the
         * server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket paths as is in
         * their file transfer protocol clients. If you set it <code>LOGICAL</code>, you will need to provide mappings
         * in the <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 paths visible to your users.
         * </p>
         * 
         * @param homeDirectoryType
         *        The type of landing directory (folder) you want your users' home directory to be when they log into
         *        the server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket paths
         *        as is in their file transfer protocol clients. If you set it <code>LOGICAL</code>, you will need to
         *        provide mappings in the <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 paths
         *        visible to your users.
         * @see HomeDirectoryType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HomeDirectoryType
         */
        Builder homeDirectoryType(HomeDirectoryType homeDirectoryType);

        /**
         * <p>
         * Logical directory mappings that specify what Amazon S3 paths and keys should be visible to your user and how
         * you want to make them visible. You will need to specify the "<code>Entry</code>" and "<code>Target</code>"
         * pair, where <code>Entry</code> shows how the path is made visible and <code>Target</code> is the actual
         * Amazon S3 path. If you only specify a target, it will be displayed as is. You will need to also make sure
         * that your IAM role provides access to paths in <code>Target</code>. The following is an example.
         * </p>
         * <p>
         * <code>'[ "/bucket2/documentation", { "Entry": "your-personal-report.pdf", "Target": "/bucket3/customized-reports/${transfer:UserName}.pdf" } ]'</code>
         * </p>
         * <p>
         * In most cases, you can use this value instead of the scope-down policy to lock your user down to the
         * designated home directory ("chroot"). To do this, you can set <code>Entry</code> to '/' and set
         * <code>Target</code> to the HomeDirectory parameter value.
         * </p>
         * <note>
         * <p>
         * If the target of a logical directory entry does not exist in Amazon S3, the entry will be ignored. As a
         * workaround, you can use the Amazon S3 API to create 0 byte objects as place holders for your directory. If
         * using the CLI, use the <code>s3api</code> call instead of <code>s3</code> so you can use the put-object
         * operation. For example, you use the following:
         * <code>aws s3api put-object --bucket bucketname --key path/to/folder/</code>. Make sure that the end of the
         * key name ends in a '/' for it to be considered a folder.
         * </p>
         * </note>
         * 
         * @param homeDirectoryMappings
         *        Logical directory mappings that specify what Amazon S3 paths and keys should be visible to your user
         *        and how you want to make them visible. You will need to specify the "<code>Entry</code>" and "
         *        <code>Target</code>" pair, where <code>Entry</code> shows how the path is made visible and
         *        <code>Target</code> is the actual Amazon S3 path. If you only specify a target, it will be displayed
         *        as is. You will need to also make sure that your IAM role provides access to paths in
         *        <code>Target</code>. The following is an example.</p>
         *        <p>
         *        <code>'[ "/bucket2/documentation", { "Entry": "your-personal-report.pdf", "Target": "/bucket3/customized-reports/${transfer:UserName}.pdf" } ]'</code>
         *        </p>
         *        <p>
         *        In most cases, you can use this value instead of the scope-down policy to lock your user down to the
         *        designated home directory ("chroot"). To do this, you can set <code>Entry</code> to '/' and set
         *        <code>Target</code> to the HomeDirectory parameter value.
         *        </p>
         *        <note>
         *        <p>
         *        If the target of a logical directory entry does not exist in Amazon S3, the entry will be ignored. As
         *        a workaround, you can use the Amazon S3 API to create 0 byte objects as place holders for your
         *        directory. If using the CLI, use the <code>s3api</code> call instead of <code>s3</code> so you can use
         *        the put-object operation. For example, you use the following:
         *        <code>aws s3api put-object --bucket bucketname --key path/to/folder/</code>. Make sure that the end of
         *        the key name ends in a '/' for it to be considered a folder.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder homeDirectoryMappings(Collection<HomeDirectoryMapEntry> homeDirectoryMappings);

        /**
         * <p>
         * Logical directory mappings that specify what Amazon S3 paths and keys should be visible to your user and how
         * you want to make them visible. You will need to specify the "<code>Entry</code>" and "<code>Target</code>"
         * pair, where <code>Entry</code> shows how the path is made visible and <code>Target</code> is the actual
         * Amazon S3 path. If you only specify a target, it will be displayed as is. You will need to also make sure
         * that your IAM role provides access to paths in <code>Target</code>. The following is an example.
         * </p>
         * <p>
         * <code>'[ "/bucket2/documentation", { "Entry": "your-personal-report.pdf", "Target": "/bucket3/customized-reports/${transfer:UserName}.pdf" } ]'</code>
         * </p>
         * <p>
         * In most cases, you can use this value instead of the scope-down policy to lock your user down to the
         * designated home directory ("chroot"). To do this, you can set <code>Entry</code> to '/' and set
         * <code>Target</code> to the HomeDirectory parameter value.
         * </p>
         * <note>
         * <p>
         * If the target of a logical directory entry does not exist in Amazon S3, the entry will be ignored. As a
         * workaround, you can use the Amazon S3 API to create 0 byte objects as place holders for your directory. If
         * using the CLI, use the <code>s3api</code> call instead of <code>s3</code> so you can use the put-object
         * operation. For example, you use the following:
         * <code>aws s3api put-object --bucket bucketname --key path/to/folder/</code>. Make sure that the end of the
         * key name ends in a '/' for it to be considered a folder.
         * </p>
         * </note>
         * 
         * @param homeDirectoryMappings
         *        Logical directory mappings that specify what Amazon S3 paths and keys should be visible to your user
         *        and how you want to make them visible. You will need to specify the "<code>Entry</code>" and "
         *        <code>Target</code>" pair, where <code>Entry</code> shows how the path is made visible and
         *        <code>Target</code> is the actual Amazon S3 path. If you only specify a target, it will be displayed
         *        as is. You will need to also make sure that your IAM role provides access to paths in
         *        <code>Target</code>. The following is an example.</p>
         *        <p>
         *        <code>'[ "/bucket2/documentation", { "Entry": "your-personal-report.pdf", "Target": "/bucket3/customized-reports/${transfer:UserName}.pdf" } ]'</code>
         *        </p>
         *        <p>
         *        In most cases, you can use this value instead of the scope-down policy to lock your user down to the
         *        designated home directory ("chroot"). To do this, you can set <code>Entry</code> to '/' and set
         *        <code>Target</code> to the HomeDirectory parameter value.
         *        </p>
         *        <note>
         *        <p>
         *        If the target of a logical directory entry does not exist in Amazon S3, the entry will be ignored. As
         *        a workaround, you can use the Amazon S3 API to create 0 byte objects as place holders for your
         *        directory. If using the CLI, use the <code>s3api</code> call instead of <code>s3</code> so you can use
         *        the put-object operation. For example, you use the following:
         *        <code>aws s3api put-object --bucket bucketname --key path/to/folder/</code>. Make sure that the end of
         *        the key name ends in a '/' for it to be considered a folder.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder homeDirectoryMappings(HomeDirectoryMapEntry... homeDirectoryMappings);

        /**
         * <p>
         * Logical directory mappings that specify what Amazon S3 paths and keys should be visible to your user and how
         * you want to make them visible. You will need to specify the "<code>Entry</code>" and "<code>Target</code>"
         * pair, where <code>Entry</code> shows how the path is made visible and <code>Target</code> is the actual
         * Amazon S3 path. If you only specify a target, it will be displayed as is. You will need to also make sure
         * that your IAM role provides access to paths in <code>Target</code>. The following is an example.
         * </p>
         * <p>
         * <code>'[ "/bucket2/documentation", { "Entry": "your-personal-report.pdf", "Target": "/bucket3/customized-reports/${transfer:UserName}.pdf" } ]'</code>
         * </p>
         * <p>
         * In most cases, you can use this value instead of the scope-down policy to lock your user down to the
         * designated home directory ("chroot"). To do this, you can set <code>Entry</code> to '/' and set
         * <code>Target</code> to the HomeDirectory parameter value.
         * </p>
         * <note>
         * <p>
         * If the target of a logical directory entry does not exist in Amazon S3, the entry will be ignored. As a
         * workaround, you can use the Amazon S3 API to create 0 byte objects as place holders for your directory. If
         * using the CLI, use the <code>s3api</code> call instead of <code>s3</code> so you can use the put-object
         * operation. For example, you use the following:
         * <code>aws s3api put-object --bucket bucketname --key path/to/folder/</code>. Make sure that the end of the
         * key name ends in a '/' for it to be considered a folder.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link List<HomeDirectoryMapEntry>.Builder}
         * avoiding the need to create one manually via {@link List<HomeDirectoryMapEntry>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<HomeDirectoryMapEntry>.Builder#build()} is called
         * immediately and its result is passed to {@link #homeDirectoryMappings(List<HomeDirectoryMapEntry>)}.
         * 
         * @param homeDirectoryMappings
         *        a consumer that will call methods on {@link List<HomeDirectoryMapEntry>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #homeDirectoryMappings(List<HomeDirectoryMapEntry>)
         */
        Builder homeDirectoryMappings(Consumer<HomeDirectoryMapEntry.Builder>... homeDirectoryMappings);

        /**
         * <p>
         * A scope-down policy for your user so you can use the same IAM role across multiple users. This policy scopes
         * down user access to portions of their Amazon S3 bucket. Variables that you can use inside this policy include
         * <code>${Transfer:UserName}</code>, <code>${Transfer:HomeDirectory}</code>, and
         * <code>${Transfer:HomeBucket}</code>.
         * </p>
         * <note>
         * <p>
         * For scope-down policies, AWS Transfer Family stores the policy as a JSON blob, instead of the Amazon Resource
         * Name (ARN) of the policy. You save the policy as a JSON blob and pass it in the <code>Policy</code> argument.
         * </p>
         * <p>
         * For an example of a scope-down policy, see <a
         * href="https://docs.aws.amazon.com/transfer/latest/userguide/users.html#users-policies-scope-down">Creating a
         * scope-down policy</a>.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/STS/latest/APIReference/API_AssumeRole.html">AssumeRole</a> in the <i>AWS
         * Security Token Service API Reference</i>.
         * </p>
         * </note>
         * 
         * @param policy
         *        A scope-down policy for your user so you can use the same IAM role across multiple users. This policy
         *        scopes down user access to portions of their Amazon S3 bucket. Variables that you can use inside this
         *        policy include <code>${Transfer:UserName}</code>, <code>${Transfer:HomeDirectory}</code>, and
         *        <code>${Transfer:HomeBucket}</code>.</p> <note>
         *        <p>
         *        For scope-down policies, AWS Transfer Family stores the policy as a JSON blob, instead of the Amazon
         *        Resource Name (ARN) of the policy. You save the policy as a JSON blob and pass it in the
         *        <code>Policy</code> argument.
         *        </p>
         *        <p>
         *        For an example of a scope-down policy, see <a
         *        href="https://docs.aws.amazon.com/transfer/latest/userguide/users.html#users-policies-scope-down"
         *        >Creating a scope-down policy</a>.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/STS/latest/APIReference/API_AssumeRole.html">AssumeRole</a> in the
         *        <i>AWS Security Token Service API Reference</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policy(String policy);

        /**
         * <p>
         * The IAM role that controls your users' access to your Amazon S3 bucket. The policies attached to this role
         * will determine the level of access you want to provide your users when transferring files into and out of
         * your Amazon S3 bucket or buckets. The IAM role should also contain a trust relationship that allows the
         * server to access your resources when servicing your users' transfer requests.
         * </p>
         * 
         * @param role
         *        The IAM role that controls your users' access to your Amazon S3 bucket. The policies attached to this
         *        role will determine the level of access you want to provide your users when transferring files into
         *        and out of your Amazon S3 bucket or buckets. The IAM role should also contain a trust relationship
         *        that allows the server to access your resources when servicing your users' transfer requests.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder role(String role);

        /**
         * <p>
         * A system-assigned unique identifier for a server instance. This is the specific server that you added your
         * user to.
         * </p>
         * 
         * @param serverId
         *        A system-assigned unique identifier for a server instance. This is the specific server that you added
         *        your user to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverId(String serverId);

        /**
         * <p>
         * The public portion of the Secure Shell (SSH) key used to authenticate the user to the server.
         * </p>
         * 
         * @param sshPublicKeyBody
         *        The public portion of the Secure Shell (SSH) key used to authenticate the user to the server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sshPublicKeyBody(String sshPublicKeyBody);

        /**
         * <p>
         * Key-value pairs that can be used to group and search for users. Tags are metadata attached to users for any
         * purpose.
         * </p>
         * 
         * @param tags
         *        Key-value pairs that can be used to group and search for users. Tags are metadata attached to users
         *        for any purpose.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Key-value pairs that can be used to group and search for users. Tags are metadata attached to users for any
         * purpose.
         * </p>
         * 
         * @param tags
         *        Key-value pairs that can be used to group and search for users. Tags are metadata attached to users
         *        for any purpose.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Key-value pairs that can be used to group and search for users. Tags are metadata attached to users for any
         * purpose.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * A unique string that identifies a user and is associated with a as specified by the <code>ServerId</code>.
         * This user name must be a minimum of 3 and a maximum of 100 characters long. The following are valid
         * characters: a-z, A-Z, 0-9, underscore '_', hyphen '-', period '.', and at sign '@'. The user name can't start
         * with a hyphen, period, or at sign.
         * </p>
         * 
         * @param userName
         *        A unique string that identifies a user and is associated with a as specified by the
         *        <code>ServerId</code>. This user name must be a minimum of 3 and a maximum of 100 characters long. The
         *        following are valid characters: a-z, A-Z, 0-9, underscore '_', hyphen '-', period '.', and at sign
         *        '@'. The user name can't start with a hyphen, period, or at sign.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userName(String userName);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends TransferRequest.BuilderImpl implements Builder {
        private String homeDirectory;

        private String homeDirectoryType;

        private List<HomeDirectoryMapEntry> homeDirectoryMappings = DefaultSdkAutoConstructList.getInstance();

        private String policy;

        private String role;

        private String serverId;

        private String sshPublicKeyBody;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String userName;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateUserRequest model) {
            super(model);
            homeDirectory(model.homeDirectory);
            homeDirectoryType(model.homeDirectoryType);
            homeDirectoryMappings(model.homeDirectoryMappings);
            policy(model.policy);
            role(model.role);
            serverId(model.serverId);
            sshPublicKeyBody(model.sshPublicKeyBody);
            tags(model.tags);
            userName(model.userName);
        }

        public final String getHomeDirectory() {
            return homeDirectory;
        }

        @Override
        public final Builder homeDirectory(String homeDirectory) {
            this.homeDirectory = homeDirectory;
            return this;
        }

        public final void setHomeDirectory(String homeDirectory) {
            this.homeDirectory = homeDirectory;
        }

        public final String getHomeDirectoryType() {
            return homeDirectoryType;
        }

        @Override
        public final Builder homeDirectoryType(String homeDirectoryType) {
            this.homeDirectoryType = homeDirectoryType;
            return this;
        }

        @Override
        public final Builder homeDirectoryType(HomeDirectoryType homeDirectoryType) {
            this.homeDirectoryType(homeDirectoryType == null ? null : homeDirectoryType.toString());
            return this;
        }

        public final void setHomeDirectoryType(String homeDirectoryType) {
            this.homeDirectoryType = homeDirectoryType;
        }

        public final Collection<HomeDirectoryMapEntry.Builder> getHomeDirectoryMappings() {
            if (homeDirectoryMappings instanceof SdkAutoConstructList) {
                return null;
            }
            return homeDirectoryMappings != null ? homeDirectoryMappings.stream().map(HomeDirectoryMapEntry::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder homeDirectoryMappings(Collection<HomeDirectoryMapEntry> homeDirectoryMappings) {
            this.homeDirectoryMappings = HomeDirectoryMappingsCopier.copy(homeDirectoryMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder homeDirectoryMappings(HomeDirectoryMapEntry... homeDirectoryMappings) {
            homeDirectoryMappings(Arrays.asList(homeDirectoryMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder homeDirectoryMappings(Consumer<HomeDirectoryMapEntry.Builder>... homeDirectoryMappings) {
            homeDirectoryMappings(Stream.of(homeDirectoryMappings)
                    .map(c -> HomeDirectoryMapEntry.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setHomeDirectoryMappings(Collection<HomeDirectoryMapEntry.BuilderImpl> homeDirectoryMappings) {
            this.homeDirectoryMappings = HomeDirectoryMappingsCopier.copyFromBuilder(homeDirectoryMappings);
        }

        public final String getPolicy() {
            return policy;
        }

        @Override
        public final Builder policy(String policy) {
            this.policy = policy;
            return this;
        }

        public final void setPolicy(String policy) {
            this.policy = policy;
        }

        public final String getRole() {
            return role;
        }

        @Override
        public final Builder role(String role) {
            this.role = role;
            return this;
        }

        public final void setRole(String role) {
            this.role = role;
        }

        public final String getServerId() {
            return serverId;
        }

        @Override
        public final Builder serverId(String serverId) {
            this.serverId = serverId;
            return this;
        }

        public final void setServerId(String serverId) {
            this.serverId = serverId;
        }

        public final String getSshPublicKeyBody() {
            return sshPublicKeyBody;
        }

        @Override
        public final Builder sshPublicKeyBody(String sshPublicKeyBody) {
            this.sshPublicKeyBody = sshPublicKeyBody;
            return this;
        }

        public final void setSshPublicKeyBody(String sshPublicKeyBody) {
            this.sshPublicKeyBody = sshPublicKeyBody;
        }

        public final Collection<Tag.Builder> getTags() {
            if (tags instanceof SdkAutoConstructList) {
                return null;
            }
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        public final String getUserName() {
            return userName;
        }

        @Override
        public final Builder userName(String userName) {
            this.userName = userName;
            return this;
        }

        public final void setUserName(String userName) {
            this.userName = userName;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateUserRequest build() {
            return new CreateUserRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
