/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transfer;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.services.transfer.model.CreateServerRequest;
import software.amazon.awssdk.services.transfer.model.CreateServerResponse;
import software.amazon.awssdk.services.transfer.model.CreateUserRequest;
import software.amazon.awssdk.services.transfer.model.CreateUserResponse;
import software.amazon.awssdk.services.transfer.model.DeleteServerRequest;
import software.amazon.awssdk.services.transfer.model.DeleteServerResponse;
import software.amazon.awssdk.services.transfer.model.DeleteSshPublicKeyRequest;
import software.amazon.awssdk.services.transfer.model.DeleteSshPublicKeyResponse;
import software.amazon.awssdk.services.transfer.model.DeleteUserRequest;
import software.amazon.awssdk.services.transfer.model.DeleteUserResponse;
import software.amazon.awssdk.services.transfer.model.DescribeSecurityPolicyRequest;
import software.amazon.awssdk.services.transfer.model.DescribeSecurityPolicyResponse;
import software.amazon.awssdk.services.transfer.model.DescribeServerRequest;
import software.amazon.awssdk.services.transfer.model.DescribeServerResponse;
import software.amazon.awssdk.services.transfer.model.DescribeUserRequest;
import software.amazon.awssdk.services.transfer.model.DescribeUserResponse;
import software.amazon.awssdk.services.transfer.model.ImportSshPublicKeyRequest;
import software.amazon.awssdk.services.transfer.model.ImportSshPublicKeyResponse;
import software.amazon.awssdk.services.transfer.model.ListSecurityPoliciesRequest;
import software.amazon.awssdk.services.transfer.model.ListSecurityPoliciesResponse;
import software.amazon.awssdk.services.transfer.model.ListServersRequest;
import software.amazon.awssdk.services.transfer.model.ListServersResponse;
import software.amazon.awssdk.services.transfer.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.transfer.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.transfer.model.ListUsersRequest;
import software.amazon.awssdk.services.transfer.model.ListUsersResponse;
import software.amazon.awssdk.services.transfer.model.StartServerRequest;
import software.amazon.awssdk.services.transfer.model.StartServerResponse;
import software.amazon.awssdk.services.transfer.model.StopServerRequest;
import software.amazon.awssdk.services.transfer.model.StopServerResponse;
import software.amazon.awssdk.services.transfer.model.TagResourceRequest;
import software.amazon.awssdk.services.transfer.model.TagResourceResponse;
import software.amazon.awssdk.services.transfer.model.TestIdentityProviderRequest;
import software.amazon.awssdk.services.transfer.model.TestIdentityProviderResponse;
import software.amazon.awssdk.services.transfer.model.UntagResourceRequest;
import software.amazon.awssdk.services.transfer.model.UntagResourceResponse;
import software.amazon.awssdk.services.transfer.model.UpdateServerRequest;
import software.amazon.awssdk.services.transfer.model.UpdateServerResponse;
import software.amazon.awssdk.services.transfer.model.UpdateUserRequest;
import software.amazon.awssdk.services.transfer.model.UpdateUserResponse;
import software.amazon.awssdk.services.transfer.paginators.ListSecurityPoliciesPublisher;
import software.amazon.awssdk.services.transfer.paginators.ListServersPublisher;
import software.amazon.awssdk.services.transfer.paginators.ListTagsForResourcePublisher;
import software.amazon.awssdk.services.transfer.paginators.ListUsersPublisher;

/**
 * Service client for accessing AWS Transfer asynchronously. This can be created using the static {@link #builder()}
 * method.
 *
 * <p>
 * AWS Transfer Family is a fully managed service that enables the transfer of files over the File Transfer Protocol
 * (FTP), File Transfer Protocol over SSL (FTPS), or Secure Shell (SSH) File Transfer Protocol (SFTP) directly into and
 * out of Amazon Simple Storage Service (Amazon S3). AWS helps you seamlessly migrate your file transfer workflows to
 * AWS Transfer Family by integrating with existing authentication systems, and providing DNS routing with Amazon Route
 * 53 so nothing changes for your customers and partners, or their applications. With your data in Amazon S3, you can
 * use it with AWS services for processing, analytics, machine learning, and archiving. Getting started with AWS
 * Transfer Family is easy since there is no infrastructure to buy and set up.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public interface TransferAsyncClient extends SdkClient {
    String SERVICE_NAME = "transfer";

    /**
     * Create a {@link TransferAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static TransferAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link TransferAsyncClient}.
     */
    static TransferAsyncClientBuilder builder() {
        return new DefaultTransferAsyncClientBuilder();
    }

    /**
     * <p>
     * Instantiates an autoscaling virtual server based on the selected file transfer protocol in AWS. When you make
     * updates to your file transfer protocol-enabled server or when you work with users, use the service-generated
     * <code>ServerId</code> property that is assigned to the newly created server.
     * </p>
     *
     * @param createServerRequest
     * @return A Java Future containing the result of the CreateServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceExistsException The requested resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.CreateServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/CreateServer" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateServerResponse> createServer(CreateServerRequest createServerRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Instantiates an autoscaling virtual server based on the selected file transfer protocol in AWS. When you make
     * updates to your file transfer protocol-enabled server or when you work with users, use the service-generated
     * <code>ServerId</code> property that is assigned to the newly created server.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateServerRequest.Builder} avoiding the need to
     * create one manually via {@link CreateServerRequest#builder()}
     * </p>
     *
     * @param createServerRequest
     *        A {@link Consumer} that will call methods on {@link CreateServerRequest.Builder} to create a request.
     * @return A Java Future containing the result of the CreateServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceExistsException The requested resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.CreateServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/CreateServer" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateServerResponse> createServer(Consumer<CreateServerRequest.Builder> createServerRequest) {
        return createServer(CreateServerRequest.builder().applyMutation(createServerRequest).build());
    }

    /**
     * <p>
     * Creates a user and associates them with an existing file transfer protocol-enabled server. You can only create
     * and associate users with servers that have the <code>IdentityProviderType</code> set to
     * <code>SERVICE_MANAGED</code>. Using parameters for <code>CreateUser</code>, you can specify the user name, set
     * the home directory, store the user's public key, and assign the user's AWS Identity and Access Management (IAM)
     * role. You can also optionally add a scope-down policy, and assign metadata with tags that can be used to group
     * and search for users.
     * </p>
     *
     * @param createUserRequest
     * @return A Java Future containing the result of the CreateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceExistsException The requested resource does not exist.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.CreateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/CreateUser" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateUserResponse> createUser(CreateUserRequest createUserRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a user and associates them with an existing file transfer protocol-enabled server. You can only create
     * and associate users with servers that have the <code>IdentityProviderType</code> set to
     * <code>SERVICE_MANAGED</code>. Using parameters for <code>CreateUser</code>, you can specify the user name, set
     * the home directory, store the user's public key, and assign the user's AWS Identity and Access Management (IAM)
     * role. You can also optionally add a scope-down policy, and assign metadata with tags that can be used to group
     * and search for users.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateUserRequest.Builder} avoiding the need to
     * create one manually via {@link CreateUserRequest#builder()}
     * </p>
     *
     * @param createUserRequest
     *        A {@link Consumer} that will call methods on {@link CreateUserRequest.Builder} to create a request.
     * @return A Java Future containing the result of the CreateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceExistsException The requested resource does not exist.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.CreateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/CreateUser" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateUserResponse> createUser(Consumer<CreateUserRequest.Builder> createUserRequest) {
        return createUser(CreateUserRequest.builder().applyMutation(createUserRequest).build());
    }

    /**
     * <p>
     * Deletes the file transfer protocol-enabled server that you specify.
     * </p>
     * <p>
     * No response returns from this operation.
     * </p>
     *
     * @param deleteServerRequest
     * @return A Java Future containing the result of the DeleteServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DeleteServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DeleteServer" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteServerResponse> deleteServer(DeleteServerRequest deleteServerRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the file transfer protocol-enabled server that you specify.
     * </p>
     * <p>
     * No response returns from this operation.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteServerRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteServerRequest#builder()}
     * </p>
     *
     * @param deleteServerRequest
     *        A {@link Consumer} that will call methods on {@link DeleteServerRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DeleteServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DeleteServer" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteServerResponse> deleteServer(Consumer<DeleteServerRequest.Builder> deleteServerRequest) {
        return deleteServer(DeleteServerRequest.builder().applyMutation(deleteServerRequest).build());
    }

    /**
     * <p>
     * Deletes a user's Secure Shell (SSH) public key.
     * </p>
     * <p>
     * No response is returned from this operation.
     * </p>
     *
     * @param deleteSshPublicKeyRequest
     * @return A Java Future containing the result of the DeleteSshPublicKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DeleteSshPublicKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DeleteSshPublicKey" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DeleteSshPublicKeyResponse> deleteSshPublicKey(DeleteSshPublicKeyRequest deleteSshPublicKeyRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a user's Secure Shell (SSH) public key.
     * </p>
     * <p>
     * No response is returned from this operation.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteSshPublicKeyRequest.Builder} avoiding the
     * need to create one manually via {@link DeleteSshPublicKeyRequest#builder()}
     * </p>
     *
     * @param deleteSshPublicKeyRequest
     *        A {@link Consumer} that will call methods on {@link DeleteSshPublicKeyRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DeleteSshPublicKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DeleteSshPublicKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DeleteSshPublicKey" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DeleteSshPublicKeyResponse> deleteSshPublicKey(
            Consumer<DeleteSshPublicKeyRequest.Builder> deleteSshPublicKeyRequest) {
        return deleteSshPublicKey(DeleteSshPublicKeyRequest.builder().applyMutation(deleteSshPublicKeyRequest).build());
    }

    /**
     * <p>
     * Deletes the user belonging to a file transfer protocol-enabled server you specify.
     * </p>
     * <p>
     * No response returns from this operation.
     * </p>
     * <note>
     * <p>
     * When you delete a user from a server, the user's information is lost.
     * </p>
     * </note>
     *
     * @param deleteUserRequest
     * @return A Java Future containing the result of the DeleteUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DeleteUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DeleteUser" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteUserResponse> deleteUser(DeleteUserRequest deleteUserRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the user belonging to a file transfer protocol-enabled server you specify.
     * </p>
     * <p>
     * No response returns from this operation.
     * </p>
     * <note>
     * <p>
     * When you delete a user from a server, the user's information is lost.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteUserRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteUserRequest#builder()}
     * </p>
     *
     * @param deleteUserRequest
     *        A {@link Consumer} that will call methods on {@link DeleteUserRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DeleteUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DeleteUser" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteUserResponse> deleteUser(Consumer<DeleteUserRequest.Builder> deleteUserRequest) {
        return deleteUser(DeleteUserRequest.builder().applyMutation(deleteUserRequest).build());
    }

    /**
     * <p>
     * Describes the security policy that is attached to your file transfer protocol-enabled server. The response
     * contains a description of the security policy's properties. For more information about security policies, see <a
     * href="https://docs.aws.amazon.com/transfer/latest/userguide/security-policies.html">Working with security
     * policies</a>.
     * </p>
     *
     * @param describeSecurityPolicyRequest
     * @return A Java Future containing the result of the DescribeSecurityPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DescribeSecurityPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DescribeSecurityPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeSecurityPolicyResponse> describeSecurityPolicy(
            DescribeSecurityPolicyRequest describeSecurityPolicyRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes the security policy that is attached to your file transfer protocol-enabled server. The response
     * contains a description of the security policy's properties. For more information about security policies, see <a
     * href="https://docs.aws.amazon.com/transfer/latest/userguide/security-policies.html">Working with security
     * policies</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeSecurityPolicyRequest.Builder} avoiding the
     * need to create one manually via {@link DescribeSecurityPolicyRequest#builder()}
     * </p>
     *
     * @param describeSecurityPolicyRequest
     *        A {@link Consumer} that will call methods on {@link DescribeSecurityPolicyRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DescribeSecurityPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DescribeSecurityPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DescribeSecurityPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeSecurityPolicyResponse> describeSecurityPolicy(
            Consumer<DescribeSecurityPolicyRequest.Builder> describeSecurityPolicyRequest) {
        return describeSecurityPolicy(DescribeSecurityPolicyRequest.builder().applyMutation(describeSecurityPolicyRequest)
                .build());
    }

    /**
     * <p>
     * Describes a file transfer protocol-enabled server that you specify by passing the <code>ServerId</code>
     * parameter.
     * </p>
     * <p>
     * The response contains a description of a server's properties. When you set <code>EndpointType</code> to VPC, the
     * response will contain the <code>EndpointDetails</code>.
     * </p>
     *
     * @param describeServerRequest
     * @return A Java Future containing the result of the DescribeServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DescribeServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DescribeServer" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribeServerResponse> describeServer(DescribeServerRequest describeServerRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes a file transfer protocol-enabled server that you specify by passing the <code>ServerId</code>
     * parameter.
     * </p>
     * <p>
     * The response contains a description of a server's properties. When you set <code>EndpointType</code> to VPC, the
     * response will contain the <code>EndpointDetails</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeServerRequest.Builder} avoiding the need to
     * create one manually via {@link DescribeServerRequest#builder()}
     * </p>
     *
     * @param describeServerRequest
     *        A {@link Consumer} that will call methods on {@link DescribeServerRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DescribeServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DescribeServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DescribeServer" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribeServerResponse> describeServer(Consumer<DescribeServerRequest.Builder> describeServerRequest) {
        return describeServer(DescribeServerRequest.builder().applyMutation(describeServerRequest).build());
    }

    /**
     * <p>
     * Describes the user assigned to the specific file transfer protocol-enabled server, as identified by its
     * <code>ServerId</code> property.
     * </p>
     * <p>
     * The response from this call returns the properties of the user associated with the <code>ServerId</code> value
     * that was specified.
     * </p>
     *
     * @param describeUserRequest
     * @return A Java Future containing the result of the DescribeUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DescribeUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DescribeUser" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribeUserResponse> describeUser(DescribeUserRequest describeUserRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes the user assigned to the specific file transfer protocol-enabled server, as identified by its
     * <code>ServerId</code> property.
     * </p>
     * <p>
     * The response from this call returns the properties of the user associated with the <code>ServerId</code> value
     * that was specified.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeUserRequest.Builder} avoiding the need to
     * create one manually via {@link DescribeUserRequest#builder()}
     * </p>
     *
     * @param describeUserRequest
     *        A {@link Consumer} that will call methods on {@link DescribeUserRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DescribeUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DescribeUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DescribeUser" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribeUserResponse> describeUser(Consumer<DescribeUserRequest.Builder> describeUserRequest) {
        return describeUser(DescribeUserRequest.builder().applyMutation(describeUserRequest).build());
    }

    /**
     * <p>
     * Adds a Secure Shell (SSH) public key to a user account identified by a <code>UserName</code> value assigned to
     * the specific file transfer protocol-enabled server, identified by <code>ServerId</code>.
     * </p>
     * <p>
     * The response returns the <code>UserName</code> value, the <code>ServerId</code> value, and the name of the
     * <code>SshPublicKeyId</code>.
     * </p>
     *
     * @param importSshPublicKeyRequest
     * @return A Java Future containing the result of the ImportSshPublicKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceExistsException The requested resource does not exist.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ImportSshPublicKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ImportSshPublicKey" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ImportSshPublicKeyResponse> importSshPublicKey(ImportSshPublicKeyRequest importSshPublicKeyRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Adds a Secure Shell (SSH) public key to a user account identified by a <code>UserName</code> value assigned to
     * the specific file transfer protocol-enabled server, identified by <code>ServerId</code>.
     * </p>
     * <p>
     * The response returns the <code>UserName</code> value, the <code>ServerId</code> value, and the name of the
     * <code>SshPublicKeyId</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ImportSshPublicKeyRequest.Builder} avoiding the
     * need to create one manually via {@link ImportSshPublicKeyRequest#builder()}
     * </p>
     *
     * @param importSshPublicKeyRequest
     *        A {@link Consumer} that will call methods on {@link ImportSshPublicKeyRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ImportSshPublicKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceExistsException The requested resource does not exist.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ImportSshPublicKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ImportSshPublicKey" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ImportSshPublicKeyResponse> importSshPublicKey(
            Consumer<ImportSshPublicKeyRequest.Builder> importSshPublicKeyRequest) {
        return importSshPublicKey(ImportSshPublicKeyRequest.builder().applyMutation(importSshPublicKeyRequest).build());
    }

    /**
     * <p>
     * Lists the security policies that are attached to your file transfer protocol-enabled servers.
     * </p>
     *
     * @param listSecurityPoliciesRequest
     * @return A Java Future containing the result of the ListSecurityPolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListSecurityPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListSecurityPolicies" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListSecurityPoliciesResponse> listSecurityPolicies(
            ListSecurityPoliciesRequest listSecurityPoliciesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the security policies that are attached to your file transfer protocol-enabled servers.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSecurityPoliciesRequest.Builder} avoiding the
     * need to create one manually via {@link ListSecurityPoliciesRequest#builder()}
     * </p>
     *
     * @param listSecurityPoliciesRequest
     *        A {@link Consumer} that will call methods on {@link ListSecurityPoliciesRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ListSecurityPolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListSecurityPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListSecurityPolicies" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListSecurityPoliciesResponse> listSecurityPolicies(
            Consumer<ListSecurityPoliciesRequest.Builder> listSecurityPoliciesRequest) {
        return listSecurityPolicies(ListSecurityPoliciesRequest.builder().applyMutation(listSecurityPoliciesRequest).build());
    }

    /**
     * <p>
     * Lists the security policies that are attached to your file transfer protocol-enabled servers.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSecurityPolicies(software.amazon.awssdk.services.transfer.model.ListSecurityPoliciesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListSecurityPoliciesPublisher publisher = client.listSecurityPoliciesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListSecurityPoliciesPublisher publisher = client.listSecurityPoliciesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transfer.model.ListSecurityPoliciesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transfer.model.ListSecurityPoliciesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSecurityPolicies(software.amazon.awssdk.services.transfer.model.ListSecurityPoliciesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSecurityPoliciesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListSecurityPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListSecurityPolicies" target="_top">AWS
     *      API Documentation</a>
     */
    default ListSecurityPoliciesPublisher listSecurityPoliciesPaginator(ListSecurityPoliciesRequest listSecurityPoliciesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the security policies that are attached to your file transfer protocol-enabled servers.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSecurityPolicies(software.amazon.awssdk.services.transfer.model.ListSecurityPoliciesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListSecurityPoliciesPublisher publisher = client.listSecurityPoliciesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListSecurityPoliciesPublisher publisher = client.listSecurityPoliciesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transfer.model.ListSecurityPoliciesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transfer.model.ListSecurityPoliciesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSecurityPolicies(software.amazon.awssdk.services.transfer.model.ListSecurityPoliciesRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSecurityPoliciesRequest.Builder} avoiding the
     * need to create one manually via {@link ListSecurityPoliciesRequest#builder()}
     * </p>
     *
     * @param listSecurityPoliciesRequest
     *        A {@link Consumer} that will call methods on {@link ListSecurityPoliciesRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListSecurityPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListSecurityPolicies" target="_top">AWS
     *      API Documentation</a>
     */
    default ListSecurityPoliciesPublisher listSecurityPoliciesPaginator(
            Consumer<ListSecurityPoliciesRequest.Builder> listSecurityPoliciesRequest) {
        return listSecurityPoliciesPaginator(ListSecurityPoliciesRequest.builder().applyMutation(listSecurityPoliciesRequest)
                .build());
    }

    /**
     * <p>
     * Lists the file transfer protocol-enabled servers that are associated with your AWS account.
     * </p>
     *
     * @param listServersRequest
     * @return A Java Future containing the result of the ListServers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListServers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListServers" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListServersResponse> listServers(ListServersRequest listServersRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the file transfer protocol-enabled servers that are associated with your AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListServersRequest.Builder} avoiding the need to
     * create one manually via {@link ListServersRequest#builder()}
     * </p>
     *
     * @param listServersRequest
     *        A {@link Consumer} that will call methods on {@link ListServersRequest.Builder} to create a request.
     * @return A Java Future containing the result of the ListServers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListServers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListServers" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListServersResponse> listServers(Consumer<ListServersRequest.Builder> listServersRequest) {
        return listServers(ListServersRequest.builder().applyMutation(listServersRequest).build());
    }

    /**
     * <p>
     * Lists the file transfer protocol-enabled servers that are associated with your AWS account.
     * </p>
     *
     * @return A Java Future containing the result of the ListServers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListServers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListServers" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListServersResponse> listServers() {
        return listServers(ListServersRequest.builder().build());
    }

    /**
     * <p>
     * Lists the file transfer protocol-enabled servers that are associated with your AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listServers(software.amazon.awssdk.services.transfer.model.ListServersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListServersPublisher publisher = client.listServersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListServersPublisher publisher = client.listServersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transfer.model.ListServersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transfer.model.ListServersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listServers(software.amazon.awssdk.services.transfer.model.ListServersRequest)} operation.</b>
     * </p>
     *
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListServers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListServers" target="_top">AWS API
     *      Documentation</a>
     */
    default ListServersPublisher listServersPaginator() {
        return listServersPaginator(ListServersRequest.builder().build());
    }

    /**
     * <p>
     * Lists the file transfer protocol-enabled servers that are associated with your AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listServers(software.amazon.awssdk.services.transfer.model.ListServersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListServersPublisher publisher = client.listServersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListServersPublisher publisher = client.listServersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transfer.model.ListServersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transfer.model.ListServersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listServers(software.amazon.awssdk.services.transfer.model.ListServersRequest)} operation.</b>
     * </p>
     *
     * @param listServersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListServers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListServers" target="_top">AWS API
     *      Documentation</a>
     */
    default ListServersPublisher listServersPaginator(ListServersRequest listServersRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the file transfer protocol-enabled servers that are associated with your AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listServers(software.amazon.awssdk.services.transfer.model.ListServersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListServersPublisher publisher = client.listServersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListServersPublisher publisher = client.listServersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transfer.model.ListServersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transfer.model.ListServersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listServers(software.amazon.awssdk.services.transfer.model.ListServersRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListServersRequest.Builder} avoiding the need to
     * create one manually via {@link ListServersRequest#builder()}
     * </p>
     *
     * @param listServersRequest
     *        A {@link Consumer} that will call methods on {@link ListServersRequest.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListServers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListServers" target="_top">AWS API
     *      Documentation</a>
     */
    default ListServersPublisher listServersPaginator(Consumer<ListServersRequest.Builder> listServersRequest) {
        return listServersPaginator(ListServersRequest.builder().applyMutation(listServersRequest).build());
    }

    /**
     * <p>
     * Lists all of the tags associated with the Amazon Resource Number (ARN) you specify. The resource can be a user,
     * server, or role.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all of the tags associated with the Amazon Resource Number (ARN) you specify. The resource can be a user,
     * server, or role.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on {@link ListTagsForResourceRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Lists all of the tags associated with the Amazon Resource Number (ARN) you specify. The resource can be a user,
     * server, or role.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTagsForResource(software.amazon.awssdk.services.transfer.model.ListTagsForResourceRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListTagsForResourcePublisher publisher = client.listTagsForResourcePaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListTagsForResourcePublisher publisher = client.listTagsForResourcePaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transfer.model.ListTagsForResourceResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transfer.model.ListTagsForResourceResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTagsForResource(software.amazon.awssdk.services.transfer.model.ListTagsForResourceRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    default ListTagsForResourcePublisher listTagsForResourcePaginator(ListTagsForResourceRequest listTagsForResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all of the tags associated with the Amazon Resource Number (ARN) you specify. The resource can be a user,
     * server, or role.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTagsForResource(software.amazon.awssdk.services.transfer.model.ListTagsForResourceRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListTagsForResourcePublisher publisher = client.listTagsForResourcePaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListTagsForResourcePublisher publisher = client.listTagsForResourcePaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transfer.model.ListTagsForResourceResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transfer.model.ListTagsForResourceResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTagsForResource(software.amazon.awssdk.services.transfer.model.ListTagsForResourceRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on {@link ListTagsForResourceRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    default ListTagsForResourcePublisher listTagsForResourcePaginator(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) {
        return listTagsForResourcePaginator(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest)
                .build());
    }

    /**
     * <p>
     * Lists the users for a file transfer protocol-enabled server that you specify by passing the <code>ServerId</code>
     * parameter.
     * </p>
     *
     * @param listUsersRequest
     * @return A Java Future containing the result of the ListUsers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListUsersResponse> listUsers(ListUsersRequest listUsersRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the users for a file transfer protocol-enabled server that you specify by passing the <code>ServerId</code>
     * parameter.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListUsersRequest.Builder} avoiding the need to
     * create one manually via {@link ListUsersRequest#builder()}
     * </p>
     *
     * @param listUsersRequest
     *        A {@link Consumer} that will call methods on {@link ListUsersRequest.Builder} to create a request.
     * @return A Java Future containing the result of the ListUsers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListUsersResponse> listUsers(Consumer<ListUsersRequest.Builder> listUsersRequest) {
        return listUsers(ListUsersRequest.builder().applyMutation(listUsersRequest).build());
    }

    /**
     * <p>
     * Lists the users for a file transfer protocol-enabled server that you specify by passing the <code>ServerId</code>
     * parameter.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listUsers(software.amazon.awssdk.services.transfer.model.ListUsersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListUsersPublisher publisher = client.listUsersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListUsersPublisher publisher = client.listUsersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transfer.model.ListUsersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transfer.model.ListUsersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listUsers(software.amazon.awssdk.services.transfer.model.ListUsersRequest)} operation.</b>
     * </p>
     *
     * @param listUsersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    default ListUsersPublisher listUsersPaginator(ListUsersRequest listUsersRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the users for a file transfer protocol-enabled server that you specify by passing the <code>ServerId</code>
     * parameter.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listUsers(software.amazon.awssdk.services.transfer.model.ListUsersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListUsersPublisher publisher = client.listUsersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListUsersPublisher publisher = client.listUsersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transfer.model.ListUsersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transfer.model.ListUsersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listUsers(software.amazon.awssdk.services.transfer.model.ListUsersRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListUsersRequest.Builder} avoiding the need to
     * create one manually via {@link ListUsersRequest#builder()}
     * </p>
     *
     * @param listUsersRequest
     *        A {@link Consumer} that will call methods on {@link ListUsersRequest.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    default ListUsersPublisher listUsersPaginator(Consumer<ListUsersRequest.Builder> listUsersRequest) {
        return listUsersPaginator(ListUsersRequest.builder().applyMutation(listUsersRequest).build());
    }

    /**
     * <p>
     * Changes the state of a file transfer protocol-enabled server from <code>OFFLINE</code> to <code>ONLINE</code>. It
     * has no impact on a server that is already <code>ONLINE</code>. An <code>ONLINE</code> server can accept and
     * process file transfer jobs.
     * </p>
     * <p>
     * The state of <code>STARTING</code> indicates that the server is in an intermediate state, either not fully able
     * to respond, or not fully online. The values of <code>START_FAILED</code> can indicate an error condition.
     * </p>
     * <p>
     * No response is returned from this call.
     * </p>
     *
     * @param startServerRequest
     * @return A Java Future containing the result of the StartServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.StartServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/StartServer" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<StartServerResponse> startServer(StartServerRequest startServerRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Changes the state of a file transfer protocol-enabled server from <code>OFFLINE</code> to <code>ONLINE</code>. It
     * has no impact on a server that is already <code>ONLINE</code>. An <code>ONLINE</code> server can accept and
     * process file transfer jobs.
     * </p>
     * <p>
     * The state of <code>STARTING</code> indicates that the server is in an intermediate state, either not fully able
     * to respond, or not fully online. The values of <code>START_FAILED</code> can indicate an error condition.
     * </p>
     * <p>
     * No response is returned from this call.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartServerRequest.Builder} avoiding the need to
     * create one manually via {@link StartServerRequest#builder()}
     * </p>
     *
     * @param startServerRequest
     *        A {@link Consumer} that will call methods on {@link StartServerRequest.Builder} to create a request.
     * @return A Java Future containing the result of the StartServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.StartServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/StartServer" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<StartServerResponse> startServer(Consumer<StartServerRequest.Builder> startServerRequest) {
        return startServer(StartServerRequest.builder().applyMutation(startServerRequest).build());
    }

    /**
     * <p>
     * Changes the state of a file transfer protocol-enabled server from <code>ONLINE</code> to <code>OFFLINE</code>. An
     * <code>OFFLINE</code> server cannot accept and process file transfer jobs. Information tied to your server, such
     * as server and user properties, are not affected by stopping your server.
     * </p>
     * <note>
     * <p>
     * Stopping the server will not reduce or impact your file transfer protocol endpoint billing; you must delete the
     * server to stop being billed.
     * </p>
     * </note>
     * <p>
     * The state of <code>STOPPING</code> indicates that the server is in an intermediate state, either not fully able
     * to respond, or not fully offline. The values of <code>STOP_FAILED</code> can indicate an error condition.
     * </p>
     * <p>
     * No response is returned from this call.
     * </p>
     *
     * @param stopServerRequest
     * @return A Java Future containing the result of the StopServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.StopServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/StopServer" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<StopServerResponse> stopServer(StopServerRequest stopServerRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Changes the state of a file transfer protocol-enabled server from <code>ONLINE</code> to <code>OFFLINE</code>. An
     * <code>OFFLINE</code> server cannot accept and process file transfer jobs. Information tied to your server, such
     * as server and user properties, are not affected by stopping your server.
     * </p>
     * <note>
     * <p>
     * Stopping the server will not reduce or impact your file transfer protocol endpoint billing; you must delete the
     * server to stop being billed.
     * </p>
     * </note>
     * <p>
     * The state of <code>STOPPING</code> indicates that the server is in an intermediate state, either not fully able
     * to respond, or not fully offline. The values of <code>STOP_FAILED</code> can indicate an error condition.
     * </p>
     * <p>
     * No response is returned from this call.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StopServerRequest.Builder} avoiding the need to
     * create one manually via {@link StopServerRequest#builder()}
     * </p>
     *
     * @param stopServerRequest
     *        A {@link Consumer} that will call methods on {@link StopServerRequest.Builder} to create a request.
     * @return A Java Future containing the result of the StopServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.StopServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/StopServer" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<StopServerResponse> stopServer(Consumer<StopServerRequest.Builder> stopServerRequest) {
        return stopServer(StopServerRequest.builder().applyMutation(stopServerRequest).build());
    }

    /**
     * <p>
     * Attaches a key-value pair to a resource, as identified by its Amazon Resource Name (ARN). Resources are users,
     * servers, roles, and other entities.
     * </p>
     * <p>
     * There is no response returned from this call.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Attaches a key-value pair to a resource, as identified by its Amazon Resource Name (ARN). Resources are users,
     * servers, roles, and other entities.
     * </p>
     * <p>
     * There is no response returned from this call.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on {@link TagResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * If the <code>IdentityProviderType</code> of a file transfer protocol-enabled server is <code>API_Gateway</code>,
     * tests whether your API Gateway is set up successfully. We highly recommend that you call this operation to test
     * your authentication method as soon as you create your server. By doing so, you can troubleshoot issues with the
     * API Gateway integration to ensure that your users can successfully use the service.
     * </p>
     *
     * @param testIdentityProviderRequest
     * @return A Java Future containing the result of the TestIdentityProvider operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.TestIdentityProvider
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/TestIdentityProvider" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<TestIdentityProviderResponse> testIdentityProvider(
            TestIdentityProviderRequest testIdentityProviderRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * If the <code>IdentityProviderType</code> of a file transfer protocol-enabled server is <code>API_Gateway</code>,
     * tests whether your API Gateway is set up successfully. We highly recommend that you call this operation to test
     * your authentication method as soon as you create your server. By doing so, you can troubleshoot issues with the
     * API Gateway integration to ensure that your users can successfully use the service.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TestIdentityProviderRequest.Builder} avoiding the
     * need to create one manually via {@link TestIdentityProviderRequest#builder()}
     * </p>
     *
     * @param testIdentityProviderRequest
     *        A {@link Consumer} that will call methods on {@link TestIdentityProviderRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the TestIdentityProvider operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.TestIdentityProvider
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/TestIdentityProvider" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<TestIdentityProviderResponse> testIdentityProvider(
            Consumer<TestIdentityProviderRequest.Builder> testIdentityProviderRequest) {
        return testIdentityProvider(TestIdentityProviderRequest.builder().applyMutation(testIdentityProviderRequest).build());
    }

    /**
     * <p>
     * Detaches a key-value pair from a resource, as identified by its Amazon Resource Name (ARN). Resources are users,
     * servers, roles, and other entities.
     * </p>
     * <p>
     * No response is returned from this call.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Detaches a key-value pair from a resource, as identified by its Amazon Resource Name (ARN). Resources are users,
     * servers, roles, and other entities.
     * </p>
     * <p>
     * No response is returned from this call.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on {@link UntagResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest) {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates the file transfer protocol-enabled server's properties after that server has been created.
     * </p>
     * <p>
     * The <code>UpdateServer</code> call returns the <code>ServerId</code> of the server you updated.
     * </p>
     *
     * @param updateServerRequest
     * @return A Java Future containing the result of the UpdateServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>ConflictException This exception is thrown when the <code>UpdatServer</code> is called for a file
     *         transfer protocol-enabled server that has VPC as the endpoint type and the server's
     *         <code>VpcEndpointID</code> is not in the available state.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceExistsException The requested resource does not exist.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.UpdateServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/UpdateServer" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdateServerResponse> updateServer(UpdateServerRequest updateServerRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates the file transfer protocol-enabled server's properties after that server has been created.
     * </p>
     * <p>
     * The <code>UpdateServer</code> call returns the <code>ServerId</code> of the server you updated.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateServerRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateServerRequest#builder()}
     * </p>
     *
     * @param updateServerRequest
     *        A {@link Consumer} that will call methods on {@link UpdateServerRequest.Builder} to create a request.
     * @return A Java Future containing the result of the UpdateServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>ConflictException This exception is thrown when the <code>UpdatServer</code> is called for a file
     *         transfer protocol-enabled server that has VPC as the endpoint type and the server's
     *         <code>VpcEndpointID</code> is not in the available state.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceExistsException The requested resource does not exist.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.UpdateServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/UpdateServer" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdateServerResponse> updateServer(Consumer<UpdateServerRequest.Builder> updateServerRequest) {
        return updateServer(UpdateServerRequest.builder().applyMutation(updateServerRequest).build());
    }

    /**
     * <p>
     * Assigns new properties to a user. Parameters you pass modify any or all of the following: the home directory,
     * role, and policy for the <code>UserName</code> and <code>ServerId</code> you specify.
     * </p>
     * <p>
     * The response returns the <code>ServerId</code> and the <code>UserName</code> for the updated user.
     * </p>
     *
     * @param updateUserRequest
     * @return A Java Future containing the result of the UpdateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.UpdateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/UpdateUser" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdateUserResponse> updateUser(UpdateUserRequest updateUserRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Assigns new properties to a user. Parameters you pass modify any or all of the following: the home directory,
     * role, and policy for the <code>UserName</code> and <code>ServerId</code> you specify.
     * </p>
     * <p>
     * The response returns the <code>ServerId</code> and the <code>UserName</code> for the updated user.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateUserRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateUserRequest#builder()}
     * </p>
     *
     * @param updateUserRequest
     *        A {@link Consumer} that will call methods on {@link UpdateUserRequest.Builder} to create a request.
     * @return A Java Future containing the result of the UpdateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the AWS Transfer Family service is not
     *         available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the AWS Transfer
     *         Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the AWS Transfer
     *         Family service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.UpdateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/UpdateUser" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdateUserResponse> updateUser(Consumer<UpdateUserRequest.Builder> updateUserRequest) {
        return updateUser(UpdateUserRequest.builder().applyMutation(updateUserRequest).build());
    }
}
