/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transfer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the properties of the access that was specified.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DescribedAccess implements SdkPojo, Serializable, ToCopyableBuilder<DescribedAccess.Builder, DescribedAccess> {
    private static final SdkField<String> HOME_DIRECTORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HomeDirectory").getter(getter(DescribedAccess::homeDirectory)).setter(setter(Builder::homeDirectory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HomeDirectory").build()).build();

    private static final SdkField<List<HomeDirectoryMapEntry>> HOME_DIRECTORY_MAPPINGS_FIELD = SdkField
            .<List<HomeDirectoryMapEntry>> builder(MarshallingType.LIST)
            .memberName("HomeDirectoryMappings")
            .getter(getter(DescribedAccess::homeDirectoryMappings))
            .setter(setter(Builder::homeDirectoryMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HomeDirectoryMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<HomeDirectoryMapEntry> builder(MarshallingType.SDK_POJO)
                                            .constructor(HomeDirectoryMapEntry::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> HOME_DIRECTORY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HomeDirectoryType").getter(getter(DescribedAccess::homeDirectoryTypeAsString))
            .setter(setter(Builder::homeDirectoryType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HomeDirectoryType").build()).build();

    private static final SdkField<String> POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Policy")
            .getter(getter(DescribedAccess::policy)).setter(setter(Builder::policy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Policy").build()).build();

    private static final SdkField<PosixProfile> POSIX_PROFILE_FIELD = SdkField.<PosixProfile> builder(MarshallingType.SDK_POJO)
            .memberName("PosixProfile").getter(getter(DescribedAccess::posixProfile)).setter(setter(Builder::posixProfile))
            .constructor(PosixProfile::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PosixProfile").build()).build();

    private static final SdkField<String> ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Role")
            .getter(getter(DescribedAccess::role)).setter(setter(Builder::role))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Role").build()).build();

    private static final SdkField<String> EXTERNAL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExternalId").getter(getter(DescribedAccess::externalId)).setter(setter(Builder::externalId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExternalId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(HOME_DIRECTORY_FIELD,
            HOME_DIRECTORY_MAPPINGS_FIELD, HOME_DIRECTORY_TYPE_FIELD, POLICY_FIELD, POSIX_PROFILE_FIELD, ROLE_FIELD,
            EXTERNAL_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String homeDirectory;

    private final List<HomeDirectoryMapEntry> homeDirectoryMappings;

    private final String homeDirectoryType;

    private final String policy;

    private final PosixProfile posixProfile;

    private final String role;

    private final String externalId;

    private DescribedAccess(BuilderImpl builder) {
        this.homeDirectory = builder.homeDirectory;
        this.homeDirectoryMappings = builder.homeDirectoryMappings;
        this.homeDirectoryType = builder.homeDirectoryType;
        this.policy = builder.policy;
        this.posixProfile = builder.posixProfile;
        this.role = builder.role;
        this.externalId = builder.externalId;
    }

    /**
     * <p>
     * The landing directory (folder) for a user when they log in to the server using the client.
     * </p>
     * <p>
     * A <code>HomeDirectory</code> example is <code>/bucket_name/home/mydirectory</code>.
     * </p>
     * 
     * @return The landing directory (folder) for a user when they log in to the server using the client.</p>
     *         <p>
     *         A <code>HomeDirectory</code> example is <code>/bucket_name/home/mydirectory</code>.
     */
    public final String homeDirectory() {
        return homeDirectory;
    }

    /**
     * Returns true if the HomeDirectoryMappings property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasHomeDirectoryMappings() {
        return homeDirectoryMappings != null && !(homeDirectoryMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Logical directory mappings that specify what Amazon S3 or Amazon EFS paths and keys should be visible to your
     * user and how you want to make them visible. You must specify the <code>Entry</code> and <code>Target</code> pair,
     * where <code>Entry</code> shows how the path is made visible and <code>Target</code> is the actual Amazon S3 or
     * Amazon EFS path. If you only specify a target, it is displayed as is. You also must ensure that your Amazon Web
     * Services Identity and Access Management (IAM) role provides access to paths in <code>Target</code>. This value
     * can only be set when <code>HomeDirectoryType</code> is set to <i>LOGICAL</i>.
     * </p>
     * <p>
     * In most cases, you can use this value instead of the scope-down policy to lock down the associated access to the
     * designated home directory ("<code>chroot</code>"). To do this, you can set <code>Entry</code> to '/' and set
     * <code>Target</code> to the <code>HomeDirectory</code> parameter value.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasHomeDirectoryMappings()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Logical directory mappings that specify what Amazon S3 or Amazon EFS paths and keys should be visible to
     *         your user and how you want to make them visible. You must specify the <code>Entry</code> and
     *         <code>Target</code> pair, where <code>Entry</code> shows how the path is made visible and
     *         <code>Target</code> is the actual Amazon S3 or Amazon EFS path. If you only specify a target, it is
     *         displayed as is. You also must ensure that your Amazon Web Services Identity and Access Management (IAM)
     *         role provides access to paths in <code>Target</code>. This value can only be set when
     *         <code>HomeDirectoryType</code> is set to <i>LOGICAL</i>.</p>
     *         <p>
     *         In most cases, you can use this value instead of the scope-down policy to lock down the associated access
     *         to the designated home directory ("<code>chroot</code>"). To do this, you can set <code>Entry</code> to
     *         '/' and set <code>Target</code> to the <code>HomeDirectory</code> parameter value.
     */
    public final List<HomeDirectoryMapEntry> homeDirectoryMappings() {
        return homeDirectoryMappings;
    }

    /**
     * <p>
     * The type of landing directory (folder) you want your users' home directory to be when they log into the server.
     * If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket or EFS paths as is in their
     * file transfer protocol clients. If you set it <code>LOGICAL</code>, you will need to provide mappings in the
     * <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 or EFS paths visible to your users.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #homeDirectoryType}
     * will return {@link HomeDirectoryType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #homeDirectoryTypeAsString}.
     * </p>
     * 
     * @return The type of landing directory (folder) you want your users' home directory to be when they log into the
     *         server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket or EFS paths
     *         as is in their file transfer protocol clients. If you set it <code>LOGICAL</code>, you will need to
     *         provide mappings in the <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 or EFS
     *         paths visible to your users.
     * @see HomeDirectoryType
     */
    public final HomeDirectoryType homeDirectoryType() {
        return HomeDirectoryType.fromValue(homeDirectoryType);
    }

    /**
     * <p>
     * The type of landing directory (folder) you want your users' home directory to be when they log into the server.
     * If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket or EFS paths as is in their
     * file transfer protocol clients. If you set it <code>LOGICAL</code>, you will need to provide mappings in the
     * <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 or EFS paths visible to your users.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #homeDirectoryType}
     * will return {@link HomeDirectoryType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #homeDirectoryTypeAsString}.
     * </p>
     * 
     * @return The type of landing directory (folder) you want your users' home directory to be when they log into the
     *         server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket or EFS paths
     *         as is in their file transfer protocol clients. If you set it <code>LOGICAL</code>, you will need to
     *         provide mappings in the <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 or EFS
     *         paths visible to your users.
     * @see HomeDirectoryType
     */
    public final String homeDirectoryTypeAsString() {
        return homeDirectoryType;
    }

    /**
     * <p>
     * A scope-down policy for your user so that you can use the same IAM role across multiple users. This policy scopes
     * down user access to portions of their Amazon S3 bucket. Variables that you can use inside this policy include
     * <code>${Transfer:UserName}</code>, <code>${Transfer:HomeDirectory}</code>, and
     * <code>${Transfer:HomeBucket}</code>.
     * </p>
     * 
     * @return A scope-down policy for your user so that you can use the same IAM role across multiple users. This
     *         policy scopes down user access to portions of their Amazon S3 bucket. Variables that you can use inside
     *         this policy include <code>${Transfer:UserName}</code>, <code>${Transfer:HomeDirectory}</code>, and
     *         <code>${Transfer:HomeBucket}</code>.
     */
    public final String policy() {
        return policy;
    }

    /**
     * Returns the value of the PosixProfile property for this object.
     * 
     * @return The value of the PosixProfile property for this object.
     */
    public final PosixProfile posixProfile() {
        return posixProfile;
    }

    /**
     * <p>
     * Specifies the Amazon Resource Name (ARN) of the IAM role that controls your users' access to your Amazon S3
     * bucket or EFS file system. The policies attached to this role determine the level of access that you want to
     * provide your users when transferring files into and out of your Amazon S3 bucket or EFS file system. The IAM role
     * should also contain a trust relationship that allows the server to access your resources when servicing your
     * users' transfer requests.
     * </p>
     * 
     * @return Specifies the Amazon Resource Name (ARN) of the IAM role that controls your users' access to your Amazon
     *         S3 bucket or EFS file system. The policies attached to this role determine the level of access that you
     *         want to provide your users when transferring files into and out of your Amazon S3 bucket or EFS file
     *         system. The IAM role should also contain a trust relationship that allows the server to access your
     *         resources when servicing your users' transfer requests.
     */
    public final String role() {
        return role;
    }

    /**
     * <p>
     * A unique identifier that is required to identify specific groups within your directory. The users of the group
     * that you associate have access to your Amazon S3 or Amazon EFS resources over the enabled protocols using Amazon
     * Web Services Transfer Family. If you know the group name, you can view the SID values by running the following
     * command using Windows PowerShell.
     * </p>
     * <p>
     * <code>Get-ADGroup -Filter {samAccountName -like "<i>YourGroupName</i>*"} -Properties * | Select SamAccountName,ObjectSid</code>
     * </p>
     * <p>
     * In that command, replace <i>YourGroupName</i> with the name of your Active Directory group.
     * </p>
     * <p>
     * The regex used to validate this parameter is a string of characters consisting of uppercase and lowercase
     * alphanumeric characters with no spaces. You can also include underscores or any of the following characters:
     * =,.@:/-
     * </p>
     * 
     * @return A unique identifier that is required to identify specific groups within your directory. The users of the
     *         group that you associate have access to your Amazon S3 or Amazon EFS resources over the enabled protocols
     *         using Amazon Web Services Transfer Family. If you know the group name, you can view the SID values by
     *         running the following command using Windows PowerShell.</p>
     *         <p>
     *         <code>Get-ADGroup -Filter {samAccountName -like "<i>YourGroupName</i>*"} -Properties * | Select SamAccountName,ObjectSid</code>
     *         </p>
     *         <p>
     *         In that command, replace <i>YourGroupName</i> with the name of your Active Directory group.
     *         </p>
     *         <p>
     *         The regex used to validate this parameter is a string of characters consisting of uppercase and lowercase
     *         alphanumeric characters with no spaces. You can also include underscores or any of the following
     *         characters: =,.@:/-
     */
    public final String externalId() {
        return externalId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(homeDirectory());
        hashCode = 31 * hashCode + Objects.hashCode(hasHomeDirectoryMappings() ? homeDirectoryMappings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(homeDirectoryTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(policy());
        hashCode = 31 * hashCode + Objects.hashCode(posixProfile());
        hashCode = 31 * hashCode + Objects.hashCode(role());
        hashCode = 31 * hashCode + Objects.hashCode(externalId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DescribedAccess)) {
            return false;
        }
        DescribedAccess other = (DescribedAccess) obj;
        return Objects.equals(homeDirectory(), other.homeDirectory())
                && hasHomeDirectoryMappings() == other.hasHomeDirectoryMappings()
                && Objects.equals(homeDirectoryMappings(), other.homeDirectoryMappings())
                && Objects.equals(homeDirectoryTypeAsString(), other.homeDirectoryTypeAsString())
                && Objects.equals(policy(), other.policy()) && Objects.equals(posixProfile(), other.posixProfile())
                && Objects.equals(role(), other.role()) && Objects.equals(externalId(), other.externalId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DescribedAccess").add("HomeDirectory", homeDirectory())
                .add("HomeDirectoryMappings", hasHomeDirectoryMappings() ? homeDirectoryMappings() : null)
                .add("HomeDirectoryType", homeDirectoryTypeAsString()).add("Policy", policy())
                .add("PosixProfile", posixProfile()).add("Role", role()).add("ExternalId", externalId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "HomeDirectory":
            return Optional.ofNullable(clazz.cast(homeDirectory()));
        case "HomeDirectoryMappings":
            return Optional.ofNullable(clazz.cast(homeDirectoryMappings()));
        case "HomeDirectoryType":
            return Optional.ofNullable(clazz.cast(homeDirectoryTypeAsString()));
        case "Policy":
            return Optional.ofNullable(clazz.cast(policy()));
        case "PosixProfile":
            return Optional.ofNullable(clazz.cast(posixProfile()));
        case "Role":
            return Optional.ofNullable(clazz.cast(role()));
        case "ExternalId":
            return Optional.ofNullable(clazz.cast(externalId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DescribedAccess, T> g) {
        return obj -> g.apply((DescribedAccess) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DescribedAccess> {
        /**
         * <p>
         * The landing directory (folder) for a user when they log in to the server using the client.
         * </p>
         * <p>
         * A <code>HomeDirectory</code> example is <code>/bucket_name/home/mydirectory</code>.
         * </p>
         * 
         * @param homeDirectory
         *        The landing directory (folder) for a user when they log in to the server using the client.</p>
         *        <p>
         *        A <code>HomeDirectory</code> example is <code>/bucket_name/home/mydirectory</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder homeDirectory(String homeDirectory);

        /**
         * <p>
         * Logical directory mappings that specify what Amazon S3 or Amazon EFS paths and keys should be visible to your
         * user and how you want to make them visible. You must specify the <code>Entry</code> and <code>Target</code>
         * pair, where <code>Entry</code> shows how the path is made visible and <code>Target</code> is the actual
         * Amazon S3 or Amazon EFS path. If you only specify a target, it is displayed as is. You also must ensure that
         * your Amazon Web Services Identity and Access Management (IAM) role provides access to paths in
         * <code>Target</code>. This value can only be set when <code>HomeDirectoryType</code> is set to <i>LOGICAL</i>.
         * </p>
         * <p>
         * In most cases, you can use this value instead of the scope-down policy to lock down the associated access to
         * the designated home directory ("<code>chroot</code>"). To do this, you can set <code>Entry</code> to '/' and
         * set <code>Target</code> to the <code>HomeDirectory</code> parameter value.
         * </p>
         * 
         * @param homeDirectoryMappings
         *        Logical directory mappings that specify what Amazon S3 or Amazon EFS paths and keys should be visible
         *        to your user and how you want to make them visible. You must specify the <code>Entry</code> and
         *        <code>Target</code> pair, where <code>Entry</code> shows how the path is made visible and
         *        <code>Target</code> is the actual Amazon S3 or Amazon EFS path. If you only specify a target, it is
         *        displayed as is. You also must ensure that your Amazon Web Services Identity and Access Management
         *        (IAM) role provides access to paths in <code>Target</code>. This value can only be set when
         *        <code>HomeDirectoryType</code> is set to <i>LOGICAL</i>.</p>
         *        <p>
         *        In most cases, you can use this value instead of the scope-down policy to lock down the associated
         *        access to the designated home directory ("<code>chroot</code>"). To do this, you can set
         *        <code>Entry</code> to '/' and set <code>Target</code> to the <code>HomeDirectory</code> parameter
         *        value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder homeDirectoryMappings(Collection<HomeDirectoryMapEntry> homeDirectoryMappings);

        /**
         * <p>
         * Logical directory mappings that specify what Amazon S3 or Amazon EFS paths and keys should be visible to your
         * user and how you want to make them visible. You must specify the <code>Entry</code> and <code>Target</code>
         * pair, where <code>Entry</code> shows how the path is made visible and <code>Target</code> is the actual
         * Amazon S3 or Amazon EFS path. If you only specify a target, it is displayed as is. You also must ensure that
         * your Amazon Web Services Identity and Access Management (IAM) role provides access to paths in
         * <code>Target</code>. This value can only be set when <code>HomeDirectoryType</code> is set to <i>LOGICAL</i>.
         * </p>
         * <p>
         * In most cases, you can use this value instead of the scope-down policy to lock down the associated access to
         * the designated home directory ("<code>chroot</code>"). To do this, you can set <code>Entry</code> to '/' and
         * set <code>Target</code> to the <code>HomeDirectory</code> parameter value.
         * </p>
         * 
         * @param homeDirectoryMappings
         *        Logical directory mappings that specify what Amazon S3 or Amazon EFS paths and keys should be visible
         *        to your user and how you want to make them visible. You must specify the <code>Entry</code> and
         *        <code>Target</code> pair, where <code>Entry</code> shows how the path is made visible and
         *        <code>Target</code> is the actual Amazon S3 or Amazon EFS path. If you only specify a target, it is
         *        displayed as is. You also must ensure that your Amazon Web Services Identity and Access Management
         *        (IAM) role provides access to paths in <code>Target</code>. This value can only be set when
         *        <code>HomeDirectoryType</code> is set to <i>LOGICAL</i>.</p>
         *        <p>
         *        In most cases, you can use this value instead of the scope-down policy to lock down the associated
         *        access to the designated home directory ("<code>chroot</code>"). To do this, you can set
         *        <code>Entry</code> to '/' and set <code>Target</code> to the <code>HomeDirectory</code> parameter
         *        value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder homeDirectoryMappings(HomeDirectoryMapEntry... homeDirectoryMappings);

        /**
         * <p>
         * Logical directory mappings that specify what Amazon S3 or Amazon EFS paths and keys should be visible to your
         * user and how you want to make them visible. You must specify the <code>Entry</code> and <code>Target</code>
         * pair, where <code>Entry</code> shows how the path is made visible and <code>Target</code> is the actual
         * Amazon S3 or Amazon EFS path. If you only specify a target, it is displayed as is. You also must ensure that
         * your Amazon Web Services Identity and Access Management (IAM) role provides access to paths in
         * <code>Target</code>. This value can only be set when <code>HomeDirectoryType</code> is set to <i>LOGICAL</i>.
         * </p>
         * <p>
         * In most cases, you can use this value instead of the scope-down policy to lock down the associated access to
         * the designated home directory ("<code>chroot</code>"). To do this, you can set <code>Entry</code> to '/' and
         * set <code>Target</code> to the <code>HomeDirectory</code> parameter value.
         * </p>
         * This is a convenience that creates an instance of the {@link List<HomeDirectoryMapEntry>.Builder} avoiding
         * the need to create one manually via {@link List<HomeDirectoryMapEntry>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<HomeDirectoryMapEntry>.Builder#build()} is called
         * immediately and its result is passed to {@link #homeDirectoryMappings(List<HomeDirectoryMapEntry>)}.
         * 
         * @param homeDirectoryMappings
         *        a consumer that will call methods on {@link List<HomeDirectoryMapEntry>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #homeDirectoryMappings(List<HomeDirectoryMapEntry>)
         */
        Builder homeDirectoryMappings(Consumer<HomeDirectoryMapEntry.Builder>... homeDirectoryMappings);

        /**
         * <p>
         * The type of landing directory (folder) you want your users' home directory to be when they log into the
         * server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket or EFS paths as
         * is in their file transfer protocol clients. If you set it <code>LOGICAL</code>, you will need to provide
         * mappings in the <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 or EFS paths visible to
         * your users.
         * </p>
         * 
         * @param homeDirectoryType
         *        The type of landing directory (folder) you want your users' home directory to be when they log into
         *        the server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket or EFS
         *        paths as is in their file transfer protocol clients. If you set it <code>LOGICAL</code>, you will need
         *        to provide mappings in the <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 or
         *        EFS paths visible to your users.
         * @see HomeDirectoryType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HomeDirectoryType
         */
        Builder homeDirectoryType(String homeDirectoryType);

        /**
         * <p>
         * The type of landing directory (folder) you want your users' home directory to be when they log into the
         * server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket or EFS paths as
         * is in their file transfer protocol clients. If you set it <code>LOGICAL</code>, you will need to provide
         * mappings in the <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 or EFS paths visible to
         * your users.
         * </p>
         * 
         * @param homeDirectoryType
         *        The type of landing directory (folder) you want your users' home directory to be when they log into
         *        the server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket or EFS
         *        paths as is in their file transfer protocol clients. If you set it <code>LOGICAL</code>, you will need
         *        to provide mappings in the <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 or
         *        EFS paths visible to your users.
         * @see HomeDirectoryType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HomeDirectoryType
         */
        Builder homeDirectoryType(HomeDirectoryType homeDirectoryType);

        /**
         * <p>
         * A scope-down policy for your user so that you can use the same IAM role across multiple users. This policy
         * scopes down user access to portions of their Amazon S3 bucket. Variables that you can use inside this policy
         * include <code>${Transfer:UserName}</code>, <code>${Transfer:HomeDirectory}</code>, and
         * <code>${Transfer:HomeBucket}</code>.
         * </p>
         * 
         * @param policy
         *        A scope-down policy for your user so that you can use the same IAM role across multiple users. This
         *        policy scopes down user access to portions of their Amazon S3 bucket. Variables that you can use
         *        inside this policy include <code>${Transfer:UserName}</code>, <code>${Transfer:HomeDirectory}</code>,
         *        and <code>${Transfer:HomeBucket}</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policy(String policy);

        /**
         * Sets the value of the PosixProfile property for this object.
         *
         * @param posixProfile
         *        The new value for the PosixProfile property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder posixProfile(PosixProfile posixProfile);

        /**
         * Sets the value of the PosixProfile property for this object.
         *
         * This is a convenience that creates an instance of the {@link PosixProfile.Builder} avoiding the need to
         * create one manually via {@link PosixProfile#builder()}.
         *
         * When the {@link Consumer} completes, {@link PosixProfile.Builder#build()} is called immediately and its
         * result is passed to {@link #posixProfile(PosixProfile)}.
         * 
         * @param posixProfile
         *        a consumer that will call methods on {@link PosixProfile.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #posixProfile(PosixProfile)
         */
        default Builder posixProfile(Consumer<PosixProfile.Builder> posixProfile) {
            return posixProfile(PosixProfile.builder().applyMutation(posixProfile).build());
        }

        /**
         * <p>
         * Specifies the Amazon Resource Name (ARN) of the IAM role that controls your users' access to your Amazon S3
         * bucket or EFS file system. The policies attached to this role determine the level of access that you want to
         * provide your users when transferring files into and out of your Amazon S3 bucket or EFS file system. The IAM
         * role should also contain a trust relationship that allows the server to access your resources when servicing
         * your users' transfer requests.
         * </p>
         * 
         * @param role
         *        Specifies the Amazon Resource Name (ARN) of the IAM role that controls your users' access to your
         *        Amazon S3 bucket or EFS file system. The policies attached to this role determine the level of access
         *        that you want to provide your users when transferring files into and out of your Amazon S3 bucket or
         *        EFS file system. The IAM role should also contain a trust relationship that allows the server to
         *        access your resources when servicing your users' transfer requests.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder role(String role);

        /**
         * <p>
         * A unique identifier that is required to identify specific groups within your directory. The users of the
         * group that you associate have access to your Amazon S3 or Amazon EFS resources over the enabled protocols
         * using Amazon Web Services Transfer Family. If you know the group name, you can view the SID values by running
         * the following command using Windows PowerShell.
         * </p>
         * <p>
         * <code>Get-ADGroup -Filter {samAccountName -like "<i>YourGroupName</i>*"} -Properties * | Select SamAccountName,ObjectSid</code>
         * </p>
         * <p>
         * In that command, replace <i>YourGroupName</i> with the name of your Active Directory group.
         * </p>
         * <p>
         * The regex used to validate this parameter is a string of characters consisting of uppercase and lowercase
         * alphanumeric characters with no spaces. You can also include underscores or any of the following characters:
         * =,.@:/-
         * </p>
         * 
         * @param externalId
         *        A unique identifier that is required to identify specific groups within your directory. The users of
         *        the group that you associate have access to your Amazon S3 or Amazon EFS resources over the enabled
         *        protocols using Amazon Web Services Transfer Family. If you know the group name, you can view the SID
         *        values by running the following command using Windows PowerShell.</p>
         *        <p>
         *        <code>Get-ADGroup -Filter {samAccountName -like "<i>YourGroupName</i>*"} -Properties * | Select SamAccountName,ObjectSid</code>
         *        </p>
         *        <p>
         *        In that command, replace <i>YourGroupName</i> with the name of your Active Directory group.
         *        </p>
         *        <p>
         *        The regex used to validate this parameter is a string of characters consisting of uppercase and
         *        lowercase alphanumeric characters with no spaces. You can also include underscores or any of the
         *        following characters: =,.@:/-
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder externalId(String externalId);
    }

    static final class BuilderImpl implements Builder {
        private String homeDirectory;

        private List<HomeDirectoryMapEntry> homeDirectoryMappings = DefaultSdkAutoConstructList.getInstance();

        private String homeDirectoryType;

        private String policy;

        private PosixProfile posixProfile;

        private String role;

        private String externalId;

        private BuilderImpl() {
        }

        private BuilderImpl(DescribedAccess model) {
            homeDirectory(model.homeDirectory);
            homeDirectoryMappings(model.homeDirectoryMappings);
            homeDirectoryType(model.homeDirectoryType);
            policy(model.policy);
            posixProfile(model.posixProfile);
            role(model.role);
            externalId(model.externalId);
        }

        public final String getHomeDirectory() {
            return homeDirectory;
        }

        @Override
        public final Builder homeDirectory(String homeDirectory) {
            this.homeDirectory = homeDirectory;
            return this;
        }

        public final void setHomeDirectory(String homeDirectory) {
            this.homeDirectory = homeDirectory;
        }

        public final List<HomeDirectoryMapEntry.Builder> getHomeDirectoryMappings() {
            List<HomeDirectoryMapEntry.Builder> result = HomeDirectoryMappingsCopier.copyToBuilder(this.homeDirectoryMappings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder homeDirectoryMappings(Collection<HomeDirectoryMapEntry> homeDirectoryMappings) {
            this.homeDirectoryMappings = HomeDirectoryMappingsCopier.copy(homeDirectoryMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder homeDirectoryMappings(HomeDirectoryMapEntry... homeDirectoryMappings) {
            homeDirectoryMappings(Arrays.asList(homeDirectoryMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder homeDirectoryMappings(Consumer<HomeDirectoryMapEntry.Builder>... homeDirectoryMappings) {
            homeDirectoryMappings(Stream.of(homeDirectoryMappings)
                    .map(c -> HomeDirectoryMapEntry.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setHomeDirectoryMappings(Collection<HomeDirectoryMapEntry.BuilderImpl> homeDirectoryMappings) {
            this.homeDirectoryMappings = HomeDirectoryMappingsCopier.copyFromBuilder(homeDirectoryMappings);
        }

        public final String getHomeDirectoryType() {
            return homeDirectoryType;
        }

        @Override
        public final Builder homeDirectoryType(String homeDirectoryType) {
            this.homeDirectoryType = homeDirectoryType;
            return this;
        }

        @Override
        public final Builder homeDirectoryType(HomeDirectoryType homeDirectoryType) {
            this.homeDirectoryType(homeDirectoryType == null ? null : homeDirectoryType.toString());
            return this;
        }

        public final void setHomeDirectoryType(String homeDirectoryType) {
            this.homeDirectoryType = homeDirectoryType;
        }

        public final String getPolicy() {
            return policy;
        }

        @Override
        public final Builder policy(String policy) {
            this.policy = policy;
            return this;
        }

        public final void setPolicy(String policy) {
            this.policy = policy;
        }

        public final PosixProfile.Builder getPosixProfile() {
            return posixProfile != null ? posixProfile.toBuilder() : null;
        }

        @Override
        public final Builder posixProfile(PosixProfile posixProfile) {
            this.posixProfile = posixProfile;
            return this;
        }

        public final void setPosixProfile(PosixProfile.BuilderImpl posixProfile) {
            this.posixProfile = posixProfile != null ? posixProfile.build() : null;
        }

        public final String getRole() {
            return role;
        }

        @Override
        public final Builder role(String role) {
            this.role = role;
            return this;
        }

        public final void setRole(String role) {
            this.role = role;
        }

        public final String getExternalId() {
            return externalId;
        }

        @Override
        public final Builder externalId(String externalId) {
            this.externalId = externalId;
            return this;
        }

        public final void setExternalId(String externalId) {
            this.externalId = externalId;
        }

        @Override
        public DescribedAccess build() {
            return new DescribedAccess(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
