/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transfer.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateUserRequest extends TransferRequest implements
        ToCopyableBuilder<UpdateUserRequest.Builder, UpdateUserRequest> {
    private static final SdkField<String> HOME_DIRECTORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HomeDirectory").getter(getter(UpdateUserRequest::homeDirectory)).setter(setter(Builder::homeDirectory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HomeDirectory").build()).build();

    private static final SdkField<String> HOME_DIRECTORY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HomeDirectoryType").getter(getter(UpdateUserRequest::homeDirectoryTypeAsString))
            .setter(setter(Builder::homeDirectoryType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HomeDirectoryType").build()).build();

    private static final SdkField<List<HomeDirectoryMapEntry>> HOME_DIRECTORY_MAPPINGS_FIELD = SdkField
            .<List<HomeDirectoryMapEntry>> builder(MarshallingType.LIST)
            .memberName("HomeDirectoryMappings")
            .getter(getter(UpdateUserRequest::homeDirectoryMappings))
            .setter(setter(Builder::homeDirectoryMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HomeDirectoryMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<HomeDirectoryMapEntry> builder(MarshallingType.SDK_POJO)
                                            .constructor(HomeDirectoryMapEntry::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Policy")
            .getter(getter(UpdateUserRequest::policy)).setter(setter(Builder::policy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Policy").build()).build();

    private static final SdkField<PosixProfile> POSIX_PROFILE_FIELD = SdkField.<PosixProfile> builder(MarshallingType.SDK_POJO)
            .memberName("PosixProfile").getter(getter(UpdateUserRequest::posixProfile)).setter(setter(Builder::posixProfile))
            .constructor(PosixProfile::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PosixProfile").build()).build();

    private static final SdkField<String> ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Role")
            .getter(getter(UpdateUserRequest::role)).setter(setter(Builder::role))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Role").build()).build();

    private static final SdkField<String> SERVER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServerId").getter(getter(UpdateUserRequest::serverId)).setter(setter(Builder::serverId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerId").build()).build();

    private static final SdkField<String> USER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UserName").getter(getter(UpdateUserRequest::userName)).setter(setter(Builder::userName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserName").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(HOME_DIRECTORY_FIELD,
            HOME_DIRECTORY_TYPE_FIELD, HOME_DIRECTORY_MAPPINGS_FIELD, POLICY_FIELD, POSIX_PROFILE_FIELD, ROLE_FIELD,
            SERVER_ID_FIELD, USER_NAME_FIELD));

    private final String homeDirectory;

    private final String homeDirectoryType;

    private final List<HomeDirectoryMapEntry> homeDirectoryMappings;

    private final String policy;

    private final PosixProfile posixProfile;

    private final String role;

    private final String serverId;

    private final String userName;

    private UpdateUserRequest(BuilderImpl builder) {
        super(builder);
        this.homeDirectory = builder.homeDirectory;
        this.homeDirectoryType = builder.homeDirectoryType;
        this.homeDirectoryMappings = builder.homeDirectoryMappings;
        this.policy = builder.policy;
        this.posixProfile = builder.posixProfile;
        this.role = builder.role;
        this.serverId = builder.serverId;
        this.userName = builder.userName;
    }

    /**
     * <p>
     * The landing directory (folder) for a user when they log in to the server using the client.
     * </p>
     * <p>
     * A <code>HomeDirectory</code> example is <code>/bucket_name/home/mydirectory</code>.
     * </p>
     * 
     * @return The landing directory (folder) for a user when they log in to the server using the client.</p>
     *         <p>
     *         A <code>HomeDirectory</code> example is <code>/bucket_name/home/mydirectory</code>.
     */
    public final String homeDirectory() {
        return homeDirectory;
    }

    /**
     * <p>
     * The type of landing directory (folder) you want your users' home directory to be when they log into the server.
     * If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket or EFS paths as is in their
     * file transfer protocol clients. If you set it <code>LOGICAL</code>, you need to provide mappings in the
     * <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 or EFS paths visible to your users.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #homeDirectoryType}
     * will return {@link HomeDirectoryType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #homeDirectoryTypeAsString}.
     * </p>
     * 
     * @return The type of landing directory (folder) you want your users' home directory to be when they log into the
     *         server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket or EFS paths
     *         as is in their file transfer protocol clients. If you set it <code>LOGICAL</code>, you need to provide
     *         mappings in the <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 or EFS paths
     *         visible to your users.
     * @see HomeDirectoryType
     */
    public final HomeDirectoryType homeDirectoryType() {
        return HomeDirectoryType.fromValue(homeDirectoryType);
    }

    /**
     * <p>
     * The type of landing directory (folder) you want your users' home directory to be when they log into the server.
     * If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket or EFS paths as is in their
     * file transfer protocol clients. If you set it <code>LOGICAL</code>, you need to provide mappings in the
     * <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 or EFS paths visible to your users.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #homeDirectoryType}
     * will return {@link HomeDirectoryType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #homeDirectoryTypeAsString}.
     * </p>
     * 
     * @return The type of landing directory (folder) you want your users' home directory to be when they log into the
     *         server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket or EFS paths
     *         as is in their file transfer protocol clients. If you set it <code>LOGICAL</code>, you need to provide
     *         mappings in the <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 or EFS paths
     *         visible to your users.
     * @see HomeDirectoryType
     */
    public final String homeDirectoryTypeAsString() {
        return homeDirectoryType;
    }

    /**
     * For responses, this returns true if the service returned a value for the HomeDirectoryMappings property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasHomeDirectoryMappings() {
        return homeDirectoryMappings != null && !(homeDirectoryMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Logical directory mappings that specify what Amazon S3 or Amazon EFS paths and keys should be visible to your
     * user and how you want to make them visible. You must specify the <code>Entry</code> and <code>Target</code> pair,
     * where <code>Entry</code> shows how the path is made visible and <code>Target</code> is the actual Amazon S3 or
     * Amazon EFS path. If you only specify a target, it is displayed as is. You also must ensure that your Amazon Web
     * Services Identity and Access Management (IAM) role provides access to paths in <code>Target</code>. This value
     * can only be set when <code>HomeDirectoryType</code> is set to <i>LOGICAL</i>.
     * </p>
     * <p>
     * The following is an <code>Entry</code> and <code>Target</code> pair example.
     * </p>
     * <p>
     * <code>[ { "Entry": "/directory1", "Target": "/bucket_name/home/mydirectory" } ]</code>
     * </p>
     * <p>
     * In most cases, you can use this value instead of the session policy to lock down your user to the designated home
     * directory ("<code>chroot</code>"). To do this, you can set <code>Entry</code> to '/' and set <code>Target</code>
     * to the HomeDirectory parameter value.
     * </p>
     * <p>
     * The following is an <code>Entry</code> and <code>Target</code> pair example for <code>chroot</code>.
     * </p>
     * <p>
     * <code>[ { "Entry": "/", "Target": "/bucket_name/home/mydirectory" } ]</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasHomeDirectoryMappings} method.
     * </p>
     * 
     * @return Logical directory mappings that specify what Amazon S3 or Amazon EFS paths and keys should be visible to
     *         your user and how you want to make them visible. You must specify the <code>Entry</code> and
     *         <code>Target</code> pair, where <code>Entry</code> shows how the path is made visible and
     *         <code>Target</code> is the actual Amazon S3 or Amazon EFS path. If you only specify a target, it is
     *         displayed as is. You also must ensure that your Amazon Web Services Identity and Access Management (IAM)
     *         role provides access to paths in <code>Target</code>. This value can only be set when
     *         <code>HomeDirectoryType</code> is set to <i>LOGICAL</i>.</p>
     *         <p>
     *         The following is an <code>Entry</code> and <code>Target</code> pair example.
     *         </p>
     *         <p>
     *         <code>[ { "Entry": "/directory1", "Target": "/bucket_name/home/mydirectory" } ]</code>
     *         </p>
     *         <p>
     *         In most cases, you can use this value instead of the session policy to lock down your user to the
     *         designated home directory ("<code>chroot</code>"). To do this, you can set <code>Entry</code> to '/' and
     *         set <code>Target</code> to the HomeDirectory parameter value.
     *         </p>
     *         <p>
     *         The following is an <code>Entry</code> and <code>Target</code> pair example for <code>chroot</code>.
     *         </p>
     *         <p>
     *         <code>[ { "Entry": "/", "Target": "/bucket_name/home/mydirectory" } ]</code>
     */
    public final List<HomeDirectoryMapEntry> homeDirectoryMappings() {
        return homeDirectoryMappings;
    }

    /**
     * <p>
     * A session policy for your user so that you can use the same IAM role across multiple users. This policy scopes
     * down user access to portions of their Amazon S3 bucket. Variables that you can use inside this policy include
     * <code>${Transfer:UserName}</code>, <code>${Transfer:HomeDirectory}</code>, and
     * <code>${Transfer:HomeBucket}</code>.
     * </p>
     * <note>
     * <p>
     * This only applies when the domain of <code>ServerId</code> is S3. EFS does not use session policies.
     * </p>
     * <p>
     * For session policies, Amazon Web Services Transfer Family stores the policy as a JSON blob, instead of the Amazon
     * Resource Name (ARN) of the policy. You save the policy as a JSON blob and pass it in the <code>Policy</code>
     * argument.
     * </p>
     * <p>
     * For an example of a session policy, see <a
     * href="https://docs.aws.amazon.com/transfer/latest/userguide/session-policy">Creating a session policy</a>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/STS/latest/APIReference/API_AssumeRole.html">AssumeRole</a> in the <i>Amazon
     * Web Services Security Token Service API Reference</i>.
     * </p>
     * </note>
     * 
     * @return A session policy for your user so that you can use the same IAM role across multiple users. This policy
     *         scopes down user access to portions of their Amazon S3 bucket. Variables that you can use inside this
     *         policy include <code>${Transfer:UserName}</code>, <code>${Transfer:HomeDirectory}</code>, and
     *         <code>${Transfer:HomeBucket}</code>.</p> <note>
     *         <p>
     *         This only applies when the domain of <code>ServerId</code> is S3. EFS does not use session policies.
     *         </p>
     *         <p>
     *         For session policies, Amazon Web Services Transfer Family stores the policy as a JSON blob, instead of
     *         the Amazon Resource Name (ARN) of the policy. You save the policy as a JSON blob and pass it in the
     *         <code>Policy</code> argument.
     *         </p>
     *         <p>
     *         For an example of a session policy, see <a
     *         href="https://docs.aws.amazon.com/transfer/latest/userguide/session-policy">Creating a session
     *         policy</a>.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/STS/latest/APIReference/API_AssumeRole.html">AssumeRole</a> in the
     *         <i>Amazon Web Services Security Token Service API Reference</i>.
     *         </p>
     */
    public final String policy() {
        return policy;
    }

    /**
     * <p>
     * Specifies the full POSIX identity, including user ID (<code>Uid</code>), group ID (<code>Gid</code>), and any
     * secondary groups IDs (<code>SecondaryGids</code>), that controls your users' access to your Amazon Elastic File
     * Systems (Amazon EFS). The POSIX permissions that are set on files and directories in your file system determines
     * the level of access your users get when transferring files into and out of your Amazon EFS file systems.
     * </p>
     * 
     * @return Specifies the full POSIX identity, including user ID (<code>Uid</code>), group ID (<code>Gid</code>), and
     *         any secondary groups IDs (<code>SecondaryGids</code>), that controls your users' access to your Amazon
     *         Elastic File Systems (Amazon EFS). The POSIX permissions that are set on files and directories in your
     *         file system determines the level of access your users get when transferring files into and out of your
     *         Amazon EFS file systems.
     */
    public final PosixProfile posixProfile() {
        return posixProfile;
    }

    /**
     * <p>
     * Specifies the Amazon Resource Name (ARN) of the IAM role that controls your users' access to your Amazon S3
     * bucket or EFS file system. The policies attached to this role determine the level of access that you want to
     * provide your users when transferring files into and out of your Amazon S3 bucket or EFS file system. The IAM role
     * should also contain a trust relationship that allows the server to access your resources when servicing your
     * users' transfer requests.
     * </p>
     * 
     * @return Specifies the Amazon Resource Name (ARN) of the IAM role that controls your users' access to your Amazon
     *         S3 bucket or EFS file system. The policies attached to this role determine the level of access that you
     *         want to provide your users when transferring files into and out of your Amazon S3 bucket or EFS file
     *         system. The IAM role should also contain a trust relationship that allows the server to access your
     *         resources when servicing your users' transfer requests.
     */
    public final String role() {
        return role;
    }

    /**
     * <p>
     * A system-assigned unique identifier for a server instance that the user account is assigned to.
     * </p>
     * 
     * @return A system-assigned unique identifier for a server instance that the user account is assigned to.
     */
    public final String serverId() {
        return serverId;
    }

    /**
     * <p>
     * A unique string that identifies a user and is associated with a server as specified by the <code>ServerId</code>.
     * This user name must be a minimum of 3 and a maximum of 100 characters long. The following are valid characters:
     * a-z, A-Z, 0-9, underscore '_', hyphen '-', period '.', and at sign '@'. The user name can't start with a hyphen,
     * period, or at sign.
     * </p>
     * 
     * @return A unique string that identifies a user and is associated with a server as specified by the
     *         <code>ServerId</code>. This user name must be a minimum of 3 and a maximum of 100 characters long. The
     *         following are valid characters: a-z, A-Z, 0-9, underscore '_', hyphen '-', period '.', and at sign '@'.
     *         The user name can't start with a hyphen, period, or at sign.
     */
    public final String userName() {
        return userName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(homeDirectory());
        hashCode = 31 * hashCode + Objects.hashCode(homeDirectoryTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasHomeDirectoryMappings() ? homeDirectoryMappings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(policy());
        hashCode = 31 * hashCode + Objects.hashCode(posixProfile());
        hashCode = 31 * hashCode + Objects.hashCode(role());
        hashCode = 31 * hashCode + Objects.hashCode(serverId());
        hashCode = 31 * hashCode + Objects.hashCode(userName());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateUserRequest)) {
            return false;
        }
        UpdateUserRequest other = (UpdateUserRequest) obj;
        return Objects.equals(homeDirectory(), other.homeDirectory())
                && Objects.equals(homeDirectoryTypeAsString(), other.homeDirectoryTypeAsString())
                && hasHomeDirectoryMappings() == other.hasHomeDirectoryMappings()
                && Objects.equals(homeDirectoryMappings(), other.homeDirectoryMappings())
                && Objects.equals(policy(), other.policy()) && Objects.equals(posixProfile(), other.posixProfile())
                && Objects.equals(role(), other.role()) && Objects.equals(serverId(), other.serverId())
                && Objects.equals(userName(), other.userName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UpdateUserRequest").add("HomeDirectory", homeDirectory())
                .add("HomeDirectoryType", homeDirectoryTypeAsString())
                .add("HomeDirectoryMappings", hasHomeDirectoryMappings() ? homeDirectoryMappings() : null)
                .add("Policy", policy()).add("PosixProfile", posixProfile()).add("Role", role()).add("ServerId", serverId())
                .add("UserName", userName()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "HomeDirectory":
            return Optional.ofNullable(clazz.cast(homeDirectory()));
        case "HomeDirectoryType":
            return Optional.ofNullable(clazz.cast(homeDirectoryTypeAsString()));
        case "HomeDirectoryMappings":
            return Optional.ofNullable(clazz.cast(homeDirectoryMappings()));
        case "Policy":
            return Optional.ofNullable(clazz.cast(policy()));
        case "PosixProfile":
            return Optional.ofNullable(clazz.cast(posixProfile()));
        case "Role":
            return Optional.ofNullable(clazz.cast(role()));
        case "ServerId":
            return Optional.ofNullable(clazz.cast(serverId()));
        case "UserName":
            return Optional.ofNullable(clazz.cast(userName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UpdateUserRequest, T> g) {
        return obj -> g.apply((UpdateUserRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends TransferRequest.Builder, SdkPojo, CopyableBuilder<Builder, UpdateUserRequest> {
        /**
         * <p>
         * The landing directory (folder) for a user when they log in to the server using the client.
         * </p>
         * <p>
         * A <code>HomeDirectory</code> example is <code>/bucket_name/home/mydirectory</code>.
         * </p>
         * 
         * @param homeDirectory
         *        The landing directory (folder) for a user when they log in to the server using the client.</p>
         *        <p>
         *        A <code>HomeDirectory</code> example is <code>/bucket_name/home/mydirectory</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder homeDirectory(String homeDirectory);

        /**
         * <p>
         * The type of landing directory (folder) you want your users' home directory to be when they log into the
         * server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket or EFS paths as
         * is in their file transfer protocol clients. If you set it <code>LOGICAL</code>, you need to provide mappings
         * in the <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 or EFS paths visible to your
         * users.
         * </p>
         * 
         * @param homeDirectoryType
         *        The type of landing directory (folder) you want your users' home directory to be when they log into
         *        the server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket or EFS
         *        paths as is in their file transfer protocol clients. If you set it <code>LOGICAL</code>, you need to
         *        provide mappings in the <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 or EFS
         *        paths visible to your users.
         * @see HomeDirectoryType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HomeDirectoryType
         */
        Builder homeDirectoryType(String homeDirectoryType);

        /**
         * <p>
         * The type of landing directory (folder) you want your users' home directory to be when they log into the
         * server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket or EFS paths as
         * is in their file transfer protocol clients. If you set it <code>LOGICAL</code>, you need to provide mappings
         * in the <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 or EFS paths visible to your
         * users.
         * </p>
         * 
         * @param homeDirectoryType
         *        The type of landing directory (folder) you want your users' home directory to be when they log into
         *        the server. If you set it to <code>PATH</code>, the user will see the absolute Amazon S3 bucket or EFS
         *        paths as is in their file transfer protocol clients. If you set it <code>LOGICAL</code>, you need to
         *        provide mappings in the <code>HomeDirectoryMappings</code> for how you want to make Amazon S3 or EFS
         *        paths visible to your users.
         * @see HomeDirectoryType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HomeDirectoryType
         */
        Builder homeDirectoryType(HomeDirectoryType homeDirectoryType);

        /**
         * <p>
         * Logical directory mappings that specify what Amazon S3 or Amazon EFS paths and keys should be visible to your
         * user and how you want to make them visible. You must specify the <code>Entry</code> and <code>Target</code>
         * pair, where <code>Entry</code> shows how the path is made visible and <code>Target</code> is the actual
         * Amazon S3 or Amazon EFS path. If you only specify a target, it is displayed as is. You also must ensure that
         * your Amazon Web Services Identity and Access Management (IAM) role provides access to paths in
         * <code>Target</code>. This value can only be set when <code>HomeDirectoryType</code> is set to <i>LOGICAL</i>.
         * </p>
         * <p>
         * The following is an <code>Entry</code> and <code>Target</code> pair example.
         * </p>
         * <p>
         * <code>[ { "Entry": "/directory1", "Target": "/bucket_name/home/mydirectory" } ]</code>
         * </p>
         * <p>
         * In most cases, you can use this value instead of the session policy to lock down your user to the designated
         * home directory ("<code>chroot</code>"). To do this, you can set <code>Entry</code> to '/' and set
         * <code>Target</code> to the HomeDirectory parameter value.
         * </p>
         * <p>
         * The following is an <code>Entry</code> and <code>Target</code> pair example for <code>chroot</code>.
         * </p>
         * <p>
         * <code>[ { "Entry": "/", "Target": "/bucket_name/home/mydirectory" } ]</code>
         * </p>
         * 
         * @param homeDirectoryMappings
         *        Logical directory mappings that specify what Amazon S3 or Amazon EFS paths and keys should be visible
         *        to your user and how you want to make them visible. You must specify the <code>Entry</code> and
         *        <code>Target</code> pair, where <code>Entry</code> shows how the path is made visible and
         *        <code>Target</code> is the actual Amazon S3 or Amazon EFS path. If you only specify a target, it is
         *        displayed as is. You also must ensure that your Amazon Web Services Identity and Access Management
         *        (IAM) role provides access to paths in <code>Target</code>. This value can only be set when
         *        <code>HomeDirectoryType</code> is set to <i>LOGICAL</i>.</p>
         *        <p>
         *        The following is an <code>Entry</code> and <code>Target</code> pair example.
         *        </p>
         *        <p>
         *        <code>[ { "Entry": "/directory1", "Target": "/bucket_name/home/mydirectory" } ]</code>
         *        </p>
         *        <p>
         *        In most cases, you can use this value instead of the session policy to lock down your user to the
         *        designated home directory ("<code>chroot</code>"). To do this, you can set <code>Entry</code> to '/'
         *        and set <code>Target</code> to the HomeDirectory parameter value.
         *        </p>
         *        <p>
         *        The following is an <code>Entry</code> and <code>Target</code> pair example for <code>chroot</code>.
         *        </p>
         *        <p>
         *        <code>[ { "Entry": "/", "Target": "/bucket_name/home/mydirectory" } ]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder homeDirectoryMappings(Collection<HomeDirectoryMapEntry> homeDirectoryMappings);

        /**
         * <p>
         * Logical directory mappings that specify what Amazon S3 or Amazon EFS paths and keys should be visible to your
         * user and how you want to make them visible. You must specify the <code>Entry</code> and <code>Target</code>
         * pair, where <code>Entry</code> shows how the path is made visible and <code>Target</code> is the actual
         * Amazon S3 or Amazon EFS path. If you only specify a target, it is displayed as is. You also must ensure that
         * your Amazon Web Services Identity and Access Management (IAM) role provides access to paths in
         * <code>Target</code>. This value can only be set when <code>HomeDirectoryType</code> is set to <i>LOGICAL</i>.
         * </p>
         * <p>
         * The following is an <code>Entry</code> and <code>Target</code> pair example.
         * </p>
         * <p>
         * <code>[ { "Entry": "/directory1", "Target": "/bucket_name/home/mydirectory" } ]</code>
         * </p>
         * <p>
         * In most cases, you can use this value instead of the session policy to lock down your user to the designated
         * home directory ("<code>chroot</code>"). To do this, you can set <code>Entry</code> to '/' and set
         * <code>Target</code> to the HomeDirectory parameter value.
         * </p>
         * <p>
         * The following is an <code>Entry</code> and <code>Target</code> pair example for <code>chroot</code>.
         * </p>
         * <p>
         * <code>[ { "Entry": "/", "Target": "/bucket_name/home/mydirectory" } ]</code>
         * </p>
         * 
         * @param homeDirectoryMappings
         *        Logical directory mappings that specify what Amazon S3 or Amazon EFS paths and keys should be visible
         *        to your user and how you want to make them visible. You must specify the <code>Entry</code> and
         *        <code>Target</code> pair, where <code>Entry</code> shows how the path is made visible and
         *        <code>Target</code> is the actual Amazon S3 or Amazon EFS path. If you only specify a target, it is
         *        displayed as is. You also must ensure that your Amazon Web Services Identity and Access Management
         *        (IAM) role provides access to paths in <code>Target</code>. This value can only be set when
         *        <code>HomeDirectoryType</code> is set to <i>LOGICAL</i>.</p>
         *        <p>
         *        The following is an <code>Entry</code> and <code>Target</code> pair example.
         *        </p>
         *        <p>
         *        <code>[ { "Entry": "/directory1", "Target": "/bucket_name/home/mydirectory" } ]</code>
         *        </p>
         *        <p>
         *        In most cases, you can use this value instead of the session policy to lock down your user to the
         *        designated home directory ("<code>chroot</code>"). To do this, you can set <code>Entry</code> to '/'
         *        and set <code>Target</code> to the HomeDirectory parameter value.
         *        </p>
         *        <p>
         *        The following is an <code>Entry</code> and <code>Target</code> pair example for <code>chroot</code>.
         *        </p>
         *        <p>
         *        <code>[ { "Entry": "/", "Target": "/bucket_name/home/mydirectory" } ]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder homeDirectoryMappings(HomeDirectoryMapEntry... homeDirectoryMappings);

        /**
         * <p>
         * Logical directory mappings that specify what Amazon S3 or Amazon EFS paths and keys should be visible to your
         * user and how you want to make them visible. You must specify the <code>Entry</code> and <code>Target</code>
         * pair, where <code>Entry</code> shows how the path is made visible and <code>Target</code> is the actual
         * Amazon S3 or Amazon EFS path. If you only specify a target, it is displayed as is. You also must ensure that
         * your Amazon Web Services Identity and Access Management (IAM) role provides access to paths in
         * <code>Target</code>. This value can only be set when <code>HomeDirectoryType</code> is set to <i>LOGICAL</i>.
         * </p>
         * <p>
         * The following is an <code>Entry</code> and <code>Target</code> pair example.
         * </p>
         * <p>
         * <code>[ { "Entry": "/directory1", "Target": "/bucket_name/home/mydirectory" } ]</code>
         * </p>
         * <p>
         * In most cases, you can use this value instead of the session policy to lock down your user to the designated
         * home directory ("<code>chroot</code>"). To do this, you can set <code>Entry</code> to '/' and set
         * <code>Target</code> to the HomeDirectory parameter value.
         * </p>
         * <p>
         * The following is an <code>Entry</code> and <code>Target</code> pair example for <code>chroot</code>.
         * </p>
         * <p>
         * <code>[ { "Entry": "/", "Target": "/bucket_name/home/mydirectory" } ]</code>
         * </p>
         * This is a convenience method that creates an instance of the {@link List<HomeDirectoryMapEntry>.Builder}
         * avoiding the need to create one manually via {@link List<HomeDirectoryMapEntry>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<HomeDirectoryMapEntry>.Builder#build()} is called
         * immediately and its result is passed to {@link #homeDirectoryMappings(List<HomeDirectoryMapEntry>)}.
         * 
         * @param homeDirectoryMappings
         *        a consumer that will call methods on {@link List<HomeDirectoryMapEntry>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #homeDirectoryMappings(List<HomeDirectoryMapEntry>)
         */
        Builder homeDirectoryMappings(Consumer<HomeDirectoryMapEntry.Builder>... homeDirectoryMappings);

        /**
         * <p>
         * A session policy for your user so that you can use the same IAM role across multiple users. This policy
         * scopes down user access to portions of their Amazon S3 bucket. Variables that you can use inside this policy
         * include <code>${Transfer:UserName}</code>, <code>${Transfer:HomeDirectory}</code>, and
         * <code>${Transfer:HomeBucket}</code>.
         * </p>
         * <note>
         * <p>
         * This only applies when the domain of <code>ServerId</code> is S3. EFS does not use session policies.
         * </p>
         * <p>
         * For session policies, Amazon Web Services Transfer Family stores the policy as a JSON blob, instead of the
         * Amazon Resource Name (ARN) of the policy. You save the policy as a JSON blob and pass it in the
         * <code>Policy</code> argument.
         * </p>
         * <p>
         * For an example of a session policy, see <a
         * href="https://docs.aws.amazon.com/transfer/latest/userguide/session-policy">Creating a session policy</a>.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/STS/latest/APIReference/API_AssumeRole.html">AssumeRole</a> in the
         * <i>Amazon Web Services Security Token Service API Reference</i>.
         * </p>
         * </note>
         * 
         * @param policy
         *        A session policy for your user so that you can use the same IAM role across multiple users. This
         *        policy scopes down user access to portions of their Amazon S3 bucket. Variables that you can use
         *        inside this policy include <code>${Transfer:UserName}</code>, <code>${Transfer:HomeDirectory}</code>,
         *        and <code>${Transfer:HomeBucket}</code>.</p> <note>
         *        <p>
         *        This only applies when the domain of <code>ServerId</code> is S3. EFS does not use session policies.
         *        </p>
         *        <p>
         *        For session policies, Amazon Web Services Transfer Family stores the policy as a JSON blob, instead of
         *        the Amazon Resource Name (ARN) of the policy. You save the policy as a JSON blob and pass it in the
         *        <code>Policy</code> argument.
         *        </p>
         *        <p>
         *        For an example of a session policy, see <a
         *        href="https://docs.aws.amazon.com/transfer/latest/userguide/session-policy">Creating a session
         *        policy</a>.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/STS/latest/APIReference/API_AssumeRole.html">AssumeRole</a> in the
         *        <i>Amazon Web Services Security Token Service API Reference</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policy(String policy);

        /**
         * <p>
         * Specifies the full POSIX identity, including user ID (<code>Uid</code>), group ID (<code>Gid</code>), and any
         * secondary groups IDs (<code>SecondaryGids</code>), that controls your users' access to your Amazon Elastic
         * File Systems (Amazon EFS). The POSIX permissions that are set on files and directories in your file system
         * determines the level of access your users get when transferring files into and out of your Amazon EFS file
         * systems.
         * </p>
         * 
         * @param posixProfile
         *        Specifies the full POSIX identity, including user ID (<code>Uid</code>), group ID (<code>Gid</code>),
         *        and any secondary groups IDs (<code>SecondaryGids</code>), that controls your users' access to your
         *        Amazon Elastic File Systems (Amazon EFS). The POSIX permissions that are set on files and directories
         *        in your file system determines the level of access your users get when transferring files into and out
         *        of your Amazon EFS file systems.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder posixProfile(PosixProfile posixProfile);

        /**
         * <p>
         * Specifies the full POSIX identity, including user ID (<code>Uid</code>), group ID (<code>Gid</code>), and any
         * secondary groups IDs (<code>SecondaryGids</code>), that controls your users' access to your Amazon Elastic
         * File Systems (Amazon EFS). The POSIX permissions that are set on files and directories in your file system
         * determines the level of access your users get when transferring files into and out of your Amazon EFS file
         * systems.
         * </p>
         * This is a convenience method that creates an instance of the {@link PosixProfile.Builder} avoiding the need
         * to create one manually via {@link PosixProfile#builder()}.
         *
         * When the {@link Consumer} completes, {@link PosixProfile.Builder#build()} is called immediately and its
         * result is passed to {@link #posixProfile(PosixProfile)}.
         * 
         * @param posixProfile
         *        a consumer that will call methods on {@link PosixProfile.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #posixProfile(PosixProfile)
         */
        default Builder posixProfile(Consumer<PosixProfile.Builder> posixProfile) {
            return posixProfile(PosixProfile.builder().applyMutation(posixProfile).build());
        }

        /**
         * <p>
         * Specifies the Amazon Resource Name (ARN) of the IAM role that controls your users' access to your Amazon S3
         * bucket or EFS file system. The policies attached to this role determine the level of access that you want to
         * provide your users when transferring files into and out of your Amazon S3 bucket or EFS file system. The IAM
         * role should also contain a trust relationship that allows the server to access your resources when servicing
         * your users' transfer requests.
         * </p>
         * 
         * @param role
         *        Specifies the Amazon Resource Name (ARN) of the IAM role that controls your users' access to your
         *        Amazon S3 bucket or EFS file system. The policies attached to this role determine the level of access
         *        that you want to provide your users when transferring files into and out of your Amazon S3 bucket or
         *        EFS file system. The IAM role should also contain a trust relationship that allows the server to
         *        access your resources when servicing your users' transfer requests.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder role(String role);

        /**
         * <p>
         * A system-assigned unique identifier for a server instance that the user account is assigned to.
         * </p>
         * 
         * @param serverId
         *        A system-assigned unique identifier for a server instance that the user account is assigned to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverId(String serverId);

        /**
         * <p>
         * A unique string that identifies a user and is associated with a server as specified by the
         * <code>ServerId</code>. This user name must be a minimum of 3 and a maximum of 100 characters long. The
         * following are valid characters: a-z, A-Z, 0-9, underscore '_', hyphen '-', period '.', and at sign '@'. The
         * user name can't start with a hyphen, period, or at sign.
         * </p>
         * 
         * @param userName
         *        A unique string that identifies a user and is associated with a server as specified by the
         *        <code>ServerId</code>. This user name must be a minimum of 3 and a maximum of 100 characters long. The
         *        following are valid characters: a-z, A-Z, 0-9, underscore '_', hyphen '-', period '.', and at sign
         *        '@'. The user name can't start with a hyphen, period, or at sign.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userName(String userName);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends TransferRequest.BuilderImpl implements Builder {
        private String homeDirectory;

        private String homeDirectoryType;

        private List<HomeDirectoryMapEntry> homeDirectoryMappings = DefaultSdkAutoConstructList.getInstance();

        private String policy;

        private PosixProfile posixProfile;

        private String role;

        private String serverId;

        private String userName;

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateUserRequest model) {
            super(model);
            homeDirectory(model.homeDirectory);
            homeDirectoryType(model.homeDirectoryType);
            homeDirectoryMappings(model.homeDirectoryMappings);
            policy(model.policy);
            posixProfile(model.posixProfile);
            role(model.role);
            serverId(model.serverId);
            userName(model.userName);
        }

        public final String getHomeDirectory() {
            return homeDirectory;
        }

        public final void setHomeDirectory(String homeDirectory) {
            this.homeDirectory = homeDirectory;
        }

        @Override
        public final Builder homeDirectory(String homeDirectory) {
            this.homeDirectory = homeDirectory;
            return this;
        }

        public final String getHomeDirectoryType() {
            return homeDirectoryType;
        }

        public final void setHomeDirectoryType(String homeDirectoryType) {
            this.homeDirectoryType = homeDirectoryType;
        }

        @Override
        public final Builder homeDirectoryType(String homeDirectoryType) {
            this.homeDirectoryType = homeDirectoryType;
            return this;
        }

        @Override
        public final Builder homeDirectoryType(HomeDirectoryType homeDirectoryType) {
            this.homeDirectoryType(homeDirectoryType == null ? null : homeDirectoryType.toString());
            return this;
        }

        public final List<HomeDirectoryMapEntry.Builder> getHomeDirectoryMappings() {
            List<HomeDirectoryMapEntry.Builder> result = HomeDirectoryMappingsCopier.copyToBuilder(this.homeDirectoryMappings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setHomeDirectoryMappings(Collection<HomeDirectoryMapEntry.BuilderImpl> homeDirectoryMappings) {
            this.homeDirectoryMappings = HomeDirectoryMappingsCopier.copyFromBuilder(homeDirectoryMappings);
        }

        @Override
        public final Builder homeDirectoryMappings(Collection<HomeDirectoryMapEntry> homeDirectoryMappings) {
            this.homeDirectoryMappings = HomeDirectoryMappingsCopier.copy(homeDirectoryMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder homeDirectoryMappings(HomeDirectoryMapEntry... homeDirectoryMappings) {
            homeDirectoryMappings(Arrays.asList(homeDirectoryMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder homeDirectoryMappings(Consumer<HomeDirectoryMapEntry.Builder>... homeDirectoryMappings) {
            homeDirectoryMappings(Stream.of(homeDirectoryMappings)
                    .map(c -> HomeDirectoryMapEntry.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getPolicy() {
            return policy;
        }

        public final void setPolicy(String policy) {
            this.policy = policy;
        }

        @Override
        public final Builder policy(String policy) {
            this.policy = policy;
            return this;
        }

        public final PosixProfile.Builder getPosixProfile() {
            return posixProfile != null ? posixProfile.toBuilder() : null;
        }

        public final void setPosixProfile(PosixProfile.BuilderImpl posixProfile) {
            this.posixProfile = posixProfile != null ? posixProfile.build() : null;
        }

        @Override
        public final Builder posixProfile(PosixProfile posixProfile) {
            this.posixProfile = posixProfile;
            return this;
        }

        public final String getRole() {
            return role;
        }

        public final void setRole(String role) {
            this.role = role;
        }

        @Override
        public final Builder role(String role) {
            this.role = role;
            return this;
        }

        public final String getServerId() {
            return serverId;
        }

        public final void setServerId(String serverId) {
            this.serverId = serverId;
        }

        @Override
        public final Builder serverId(String serverId) {
            this.serverId = serverId;
            return this;
        }

        public final String getUserName() {
            return userName;
        }

        public final void setUserName(String userName) {
            this.userName = userName;
        }

        @Override
        public final Builder userName(String userName) {
            this.userName = userName;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public UpdateUserRequest build() {
            return new UpdateUserRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
