/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transfer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Returns properties of the execution that is specified.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ListedExecution implements SdkPojo, Serializable, ToCopyableBuilder<ListedExecution.Builder, ListedExecution> {
    private static final SdkField<String> EXECUTION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExecutionId").getter(getter(ListedExecution::executionId)).setter(setter(Builder::executionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionId").build()).build();

    private static final SdkField<FileLocation> INITIAL_FILE_LOCATION_FIELD = SdkField
            .<FileLocation> builder(MarshallingType.SDK_POJO).memberName("InitialFileLocation")
            .getter(getter(ListedExecution::initialFileLocation)).setter(setter(Builder::initialFileLocation))
            .constructor(FileLocation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InitialFileLocation").build())
            .build();

    private static final SdkField<ServiceMetadata> SERVICE_METADATA_FIELD = SdkField
            .<ServiceMetadata> builder(MarshallingType.SDK_POJO).memberName("ServiceMetadata")
            .getter(getter(ListedExecution::serviceMetadata)).setter(setter(Builder::serviceMetadata))
            .constructor(ServiceMetadata::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceMetadata").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(ListedExecution::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EXECUTION_ID_FIELD,
            INITIAL_FILE_LOCATION_FIELD, SERVICE_METADATA_FIELD, STATUS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String executionId;

    private final FileLocation initialFileLocation;

    private final ServiceMetadata serviceMetadata;

    private final String status;

    private ListedExecution(BuilderImpl builder) {
        this.executionId = builder.executionId;
        this.initialFileLocation = builder.initialFileLocation;
        this.serviceMetadata = builder.serviceMetadata;
        this.status = builder.status;
    }

    /**
     * <p>
     * A unique identifier for the execution of a workflow.
     * </p>
     * 
     * @return A unique identifier for the execution of a workflow.
     */
    public final String executionId() {
        return executionId;
    }

    /**
     * <p>
     * A structure that describes the Amazon S3 or EFS file location. This is the file location when the execution
     * begins: if the file is being copied, this is the initial (as opposed to destination) file location.
     * </p>
     * 
     * @return A structure that describes the Amazon S3 or EFS file location. This is the file location when the
     *         execution begins: if the file is being copied, this is the initial (as opposed to destination) file
     *         location.
     */
    public final FileLocation initialFileLocation() {
        return initialFileLocation;
    }

    /**
     * <p>
     * A container object for the session details associated with a workflow.
     * </p>
     * 
     * @return A container object for the session details associated with a workflow.
     */
    public final ServiceMetadata serviceMetadata() {
        return serviceMetadata;
    }

    /**
     * <p>
     * The status is one of the execution. Can be in progress, completed, exception encountered, or handling the
     * exception.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status is one of the execution. Can be in progress, completed, exception encountered, or handling the
     *         exception.
     * @see ExecutionStatus
     */
    public final ExecutionStatus status() {
        return ExecutionStatus.fromValue(status);
    }

    /**
     * <p>
     * The status is one of the execution. Can be in progress, completed, exception encountered, or handling the
     * exception.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status is one of the execution. Can be in progress, completed, exception encountered, or handling the
     *         exception.
     * @see ExecutionStatus
     */
    public final String statusAsString() {
        return status;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(executionId());
        hashCode = 31 * hashCode + Objects.hashCode(initialFileLocation());
        hashCode = 31 * hashCode + Objects.hashCode(serviceMetadata());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ListedExecution)) {
            return false;
        }
        ListedExecution other = (ListedExecution) obj;
        return Objects.equals(executionId(), other.executionId())
                && Objects.equals(initialFileLocation(), other.initialFileLocation())
                && Objects.equals(serviceMetadata(), other.serviceMetadata())
                && Objects.equals(statusAsString(), other.statusAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ListedExecution").add("ExecutionId", executionId())
                .add("InitialFileLocation", initialFileLocation()).add("ServiceMetadata", serviceMetadata())
                .add("Status", statusAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ExecutionId":
            return Optional.ofNullable(clazz.cast(executionId()));
        case "InitialFileLocation":
            return Optional.ofNullable(clazz.cast(initialFileLocation()));
        case "ServiceMetadata":
            return Optional.ofNullable(clazz.cast(serviceMetadata()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ListedExecution, T> g) {
        return obj -> g.apply((ListedExecution) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ListedExecution> {
        /**
         * <p>
         * A unique identifier for the execution of a workflow.
         * </p>
         * 
         * @param executionId
         *        A unique identifier for the execution of a workflow.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionId(String executionId);

        /**
         * <p>
         * A structure that describes the Amazon S3 or EFS file location. This is the file location when the execution
         * begins: if the file is being copied, this is the initial (as opposed to destination) file location.
         * </p>
         * 
         * @param initialFileLocation
         *        A structure that describes the Amazon S3 or EFS file location. This is the file location when the
         *        execution begins: if the file is being copied, this is the initial (as opposed to destination) file
         *        location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder initialFileLocation(FileLocation initialFileLocation);

        /**
         * <p>
         * A structure that describes the Amazon S3 or EFS file location. This is the file location when the execution
         * begins: if the file is being copied, this is the initial (as opposed to destination) file location.
         * </p>
         * This is a convenience method that creates an instance of the {@link FileLocation.Builder} avoiding the need
         * to create one manually via {@link FileLocation#builder()}.
         *
         * When the {@link Consumer} completes, {@link FileLocation.Builder#build()} is called immediately and its
         * result is passed to {@link #initialFileLocation(FileLocation)}.
         * 
         * @param initialFileLocation
         *        a consumer that will call methods on {@link FileLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #initialFileLocation(FileLocation)
         */
        default Builder initialFileLocation(Consumer<FileLocation.Builder> initialFileLocation) {
            return initialFileLocation(FileLocation.builder().applyMutation(initialFileLocation).build());
        }

        /**
         * <p>
         * A container object for the session details associated with a workflow.
         * </p>
         * 
         * @param serviceMetadata
         *        A container object for the session details associated with a workflow.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceMetadata(ServiceMetadata serviceMetadata);

        /**
         * <p>
         * A container object for the session details associated with a workflow.
         * </p>
         * This is a convenience method that creates an instance of the {@link ServiceMetadata.Builder} avoiding the
         * need to create one manually via {@link ServiceMetadata#builder()}.
         *
         * When the {@link Consumer} completes, {@link ServiceMetadata.Builder#build()} is called immediately and its
         * result is passed to {@link #serviceMetadata(ServiceMetadata)}.
         * 
         * @param serviceMetadata
         *        a consumer that will call methods on {@link ServiceMetadata.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #serviceMetadata(ServiceMetadata)
         */
        default Builder serviceMetadata(Consumer<ServiceMetadata.Builder> serviceMetadata) {
            return serviceMetadata(ServiceMetadata.builder().applyMutation(serviceMetadata).build());
        }

        /**
         * <p>
         * The status is one of the execution. Can be in progress, completed, exception encountered, or handling the
         * exception.
         * </p>
         * 
         * @param status
         *        The status is one of the execution. Can be in progress, completed, exception encountered, or handling
         *        the exception.
         * @see ExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutionStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status is one of the execution. Can be in progress, completed, exception encountered, or handling the
         * exception.
         * </p>
         * 
         * @param status
         *        The status is one of the execution. Can be in progress, completed, exception encountered, or handling
         *        the exception.
         * @see ExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutionStatus
         */
        Builder status(ExecutionStatus status);
    }

    static final class BuilderImpl implements Builder {
        private String executionId;

        private FileLocation initialFileLocation;

        private ServiceMetadata serviceMetadata;

        private String status;

        private BuilderImpl() {
        }

        private BuilderImpl(ListedExecution model) {
            executionId(model.executionId);
            initialFileLocation(model.initialFileLocation);
            serviceMetadata(model.serviceMetadata);
            status(model.status);
        }

        public final String getExecutionId() {
            return executionId;
        }

        public final void setExecutionId(String executionId) {
            this.executionId = executionId;
        }

        @Override
        public final Builder executionId(String executionId) {
            this.executionId = executionId;
            return this;
        }

        public final FileLocation.Builder getInitialFileLocation() {
            return initialFileLocation != null ? initialFileLocation.toBuilder() : null;
        }

        public final void setInitialFileLocation(FileLocation.BuilderImpl initialFileLocation) {
            this.initialFileLocation = initialFileLocation != null ? initialFileLocation.build() : null;
        }

        @Override
        public final Builder initialFileLocation(FileLocation initialFileLocation) {
            this.initialFileLocation = initialFileLocation;
            return this;
        }

        public final ServiceMetadata.Builder getServiceMetadata() {
            return serviceMetadata != null ? serviceMetadata.toBuilder() : null;
        }

        public final void setServiceMetadata(ServiceMetadata.BuilderImpl serviceMetadata) {
            this.serviceMetadata = serviceMetadata != null ? serviceMetadata.build() : null;
        }

        @Override
        public final Builder serviceMetadata(ServiceMetadata serviceMetadata) {
            this.serviceMetadata = serviceMetadata;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ExecutionStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        @Override
        public ListedExecution build() {
            return new ListedExecution(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
