/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transfer.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The protocol settings that are configured for your server.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProtocolDetails implements SdkPojo, Serializable, ToCopyableBuilder<ProtocolDetails.Builder, ProtocolDetails> {
    private static final SdkField<String> PASSIVE_IP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PassiveIp").getter(getter(ProtocolDetails::passiveIp)).setter(setter(Builder::passiveIp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PassiveIp").build()).build();

    private static final SdkField<String> TLS_SESSION_RESUMPTION_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TlsSessionResumptionMode").getter(getter(ProtocolDetails::tlsSessionResumptionModeAsString))
            .setter(setter(Builder::tlsSessionResumptionMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TlsSessionResumptionMode").build())
            .build();

    private static final SdkField<String> SET_STAT_OPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SetStatOption").getter(getter(ProtocolDetails::setStatOptionAsString))
            .setter(setter(Builder::setStatOption))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SetStatOption").build()).build();

    private static final SdkField<List<String>> AS2_TRANSPORTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("As2Transports")
            .getter(getter(ProtocolDetails::as2TransportsAsStrings))
            .setter(setter(Builder::as2TransportsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("As2Transports").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PASSIVE_IP_FIELD,
            TLS_SESSION_RESUMPTION_MODE_FIELD, SET_STAT_OPTION_FIELD, AS2_TRANSPORTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String passiveIp;

    private final String tlsSessionResumptionMode;

    private final String setStatOption;

    private final List<String> as2Transports;

    private ProtocolDetails(BuilderImpl builder) {
        this.passiveIp = builder.passiveIp;
        this.tlsSessionResumptionMode = builder.tlsSessionResumptionMode;
        this.setStatOption = builder.setStatOption;
        this.as2Transports = builder.as2Transports;
    }

    /**
     * <p>
     * Indicates passive mode, for FTP and FTPS protocols. Enter a single IPv4 address, such as the public IP address of
     * a firewall, router, or load balancer. For example:
     * </p>
     * <p>
     * <code> aws transfer update-server --protocol-details PassiveIp=<i>0.0.0.0</i> </code>
     * </p>
     * <p>
     * Replace <code> <i>0.0.0.0</i> </code> in the example above with the actual IP address you want to use.
     * </p>
     * <note>
     * <p>
     * If you change the <code>PassiveIp</code> value, you must stop and then restart your Transfer Family server for
     * the change to take effect. For details on using passive mode (PASV) in a NAT environment, see <a href=
     * "http://aws.amazon.com/blogs/storage/configuring-your-ftps-server-behind-a-firewall-or-nat-with-aws-transfer-family/"
     * >Configuring your FTPS server behind a firewall or NAT with Transfer Family</a>.
     * </p>
     * </note>
     * 
     * @return Indicates passive mode, for FTP and FTPS protocols. Enter a single IPv4 address, such as the public IP
     *         address of a firewall, router, or load balancer. For example: </p>
     *         <p>
     *         <code> aws transfer update-server --protocol-details PassiveIp=<i>0.0.0.0</i> </code>
     *         </p>
     *         <p>
     *         Replace <code> <i>0.0.0.0</i> </code> in the example above with the actual IP address you want to use.
     *         </p>
     *         <note>
     *         <p>
     *         If you change the <code>PassiveIp</code> value, you must stop and then restart your Transfer Family
     *         server for the change to take effect. For details on using passive mode (PASV) in a NAT environment, see
     *         <a href=
     *         "http://aws.amazon.com/blogs/storage/configuring-your-ftps-server-behind-a-firewall-or-nat-with-aws-transfer-family/"
     *         >Configuring your FTPS server behind a firewall or NAT with Transfer Family</a>.
     *         </p>
     */
    public final String passiveIp() {
        return passiveIp;
    }

    /**
     * <p>
     * A property used with Transfer Family servers that use the FTPS protocol. TLS Session Resumption provides a
     * mechanism to resume or share a negotiated secret key between the control and data connection for an FTPS session.
     * <code>TlsSessionResumptionMode</code> determines whether or not the server resumes recent, negotiated sessions
     * through a unique session ID. This property is available during <code>CreateServer</code> and
     * <code>UpdateServer</code> calls. If a <code>TlsSessionResumptionMode</code> value is not specified during
     * <code>CreateServer</code>, it is set to <code>ENFORCED</code> by default.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DISABLED</code>: the server does not process TLS session resumption client requests and creates a new TLS
     * session for each request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ENABLED</code>: the server processes and accepts clients that are performing TLS session resumption. The
     * server doesn't reject client data connections that do not perform the TLS session resumption client processing.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ENFORCED</code>: the server processes and accepts clients that are performing TLS session resumption. The
     * server rejects client data connections that do not perform the TLS session resumption client processing. Before
     * you set the value to <code>ENFORCED</code>, test your clients.
     * </p>
     * <note>
     * <p>
     * Not all FTPS clients perform TLS session resumption. So, if you choose to enforce TLS session resumption, you
     * prevent any connections from FTPS clients that don't perform the protocol negotiation. To determine whether or
     * not you can use the <code>ENFORCED</code> value, you need to test your clients.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #tlsSessionResumptionMode} will return {@link TlsSessionResumptionMode#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #tlsSessionResumptionModeAsString}.
     * </p>
     * 
     * @return A property used with Transfer Family servers that use the FTPS protocol. TLS Session Resumption provides
     *         a mechanism to resume or share a negotiated secret key between the control and data connection for an
     *         FTPS session. <code>TlsSessionResumptionMode</code> determines whether or not the server resumes recent,
     *         negotiated sessions through a unique session ID. This property is available during
     *         <code>CreateServer</code> and <code>UpdateServer</code> calls. If a <code>TlsSessionResumptionMode</code>
     *         value is not specified during <code>CreateServer</code>, it is set to <code>ENFORCED</code> by
     *         default.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DISABLED</code>: the server does not process TLS session resumption client requests and creates a
     *         new TLS session for each request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ENABLED</code>: the server processes and accepts clients that are performing TLS session
     *         resumption. The server doesn't reject client data connections that do not perform the TLS session
     *         resumption client processing.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ENFORCED</code>: the server processes and accepts clients that are performing TLS session
     *         resumption. The server rejects client data connections that do not perform the TLS session resumption
     *         client processing. Before you set the value to <code>ENFORCED</code>, test your clients.
     *         </p>
     *         <note>
     *         <p>
     *         Not all FTPS clients perform TLS session resumption. So, if you choose to enforce TLS session resumption,
     *         you prevent any connections from FTPS clients that don't perform the protocol negotiation. To determine
     *         whether or not you can use the <code>ENFORCED</code> value, you need to test your clients.
     *         </p>
     *         </note></li>
     * @see TlsSessionResumptionMode
     */
    public final TlsSessionResumptionMode tlsSessionResumptionMode() {
        return TlsSessionResumptionMode.fromValue(tlsSessionResumptionMode);
    }

    /**
     * <p>
     * A property used with Transfer Family servers that use the FTPS protocol. TLS Session Resumption provides a
     * mechanism to resume or share a negotiated secret key between the control and data connection for an FTPS session.
     * <code>TlsSessionResumptionMode</code> determines whether or not the server resumes recent, negotiated sessions
     * through a unique session ID. This property is available during <code>CreateServer</code> and
     * <code>UpdateServer</code> calls. If a <code>TlsSessionResumptionMode</code> value is not specified during
     * <code>CreateServer</code>, it is set to <code>ENFORCED</code> by default.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DISABLED</code>: the server does not process TLS session resumption client requests and creates a new TLS
     * session for each request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ENABLED</code>: the server processes and accepts clients that are performing TLS session resumption. The
     * server doesn't reject client data connections that do not perform the TLS session resumption client processing.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ENFORCED</code>: the server processes and accepts clients that are performing TLS session resumption. The
     * server rejects client data connections that do not perform the TLS session resumption client processing. Before
     * you set the value to <code>ENFORCED</code>, test your clients.
     * </p>
     * <note>
     * <p>
     * Not all FTPS clients perform TLS session resumption. So, if you choose to enforce TLS session resumption, you
     * prevent any connections from FTPS clients that don't perform the protocol negotiation. To determine whether or
     * not you can use the <code>ENFORCED</code> value, you need to test your clients.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #tlsSessionResumptionMode} will return {@link TlsSessionResumptionMode#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #tlsSessionResumptionModeAsString}.
     * </p>
     * 
     * @return A property used with Transfer Family servers that use the FTPS protocol. TLS Session Resumption provides
     *         a mechanism to resume or share a negotiated secret key between the control and data connection for an
     *         FTPS session. <code>TlsSessionResumptionMode</code> determines whether or not the server resumes recent,
     *         negotiated sessions through a unique session ID. This property is available during
     *         <code>CreateServer</code> and <code>UpdateServer</code> calls. If a <code>TlsSessionResumptionMode</code>
     *         value is not specified during <code>CreateServer</code>, it is set to <code>ENFORCED</code> by
     *         default.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DISABLED</code>: the server does not process TLS session resumption client requests and creates a
     *         new TLS session for each request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ENABLED</code>: the server processes and accepts clients that are performing TLS session
     *         resumption. The server doesn't reject client data connections that do not perform the TLS session
     *         resumption client processing.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ENFORCED</code>: the server processes and accepts clients that are performing TLS session
     *         resumption. The server rejects client data connections that do not perform the TLS session resumption
     *         client processing. Before you set the value to <code>ENFORCED</code>, test your clients.
     *         </p>
     *         <note>
     *         <p>
     *         Not all FTPS clients perform TLS session resumption. So, if you choose to enforce TLS session resumption,
     *         you prevent any connections from FTPS clients that don't perform the protocol negotiation. To determine
     *         whether or not you can use the <code>ENFORCED</code> value, you need to test your clients.
     *         </p>
     *         </note></li>
     * @see TlsSessionResumptionMode
     */
    public final String tlsSessionResumptionModeAsString() {
        return tlsSessionResumptionMode;
    }

    /**
     * <p>
     * Use the <code>SetStatOption</code> to ignore the error that is generated when the client attempts to use
     * <code>SETSTAT</code> on a file you are uploading to an S3 bucket.
     * </p>
     * <p>
     * Some SFTP file transfer clients can attempt to change the attributes of remote files, including timestamp and
     * permissions, using commands, such as <code>SETSTAT</code> when uploading the file. However, these commands are
     * not compatible with object storage systems, such as Amazon S3. Due to this incompatibility, file uploads from
     * these clients can result in errors even when the file is otherwise successfully uploaded.
     * </p>
     * <p>
     * Set the value to <code>ENABLE_NO_OP</code> to have the Transfer Family server ignore the <code>SETSTAT</code>
     * command, and upload files without needing to make any changes to your SFTP client. While the
     * <code>SetStatOption</code> <code>ENABLE_NO_OP</code> setting ignores the error, it does generate a log entry in
     * Amazon CloudWatch Logs, so you can determine when the client is making a <code>SETSTAT</code> call.
     * </p>
     * <note>
     * <p>
     * If you want to preserve the original timestamp for your file, and modify other file attributes using
     * <code>SETSTAT</code>, you can use Amazon EFS as backend storage with Transfer Family.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #setStatOption}
     * will return {@link SetStatOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #setStatOptionAsString}.
     * </p>
     * 
     * @return Use the <code>SetStatOption</code> to ignore the error that is generated when the client attempts to use
     *         <code>SETSTAT</code> on a file you are uploading to an S3 bucket.</p>
     *         <p>
     *         Some SFTP file transfer clients can attempt to change the attributes of remote files, including timestamp
     *         and permissions, using commands, such as <code>SETSTAT</code> when uploading the file. However, these
     *         commands are not compatible with object storage systems, such as Amazon S3. Due to this incompatibility,
     *         file uploads from these clients can result in errors even when the file is otherwise successfully
     *         uploaded.
     *         </p>
     *         <p>
     *         Set the value to <code>ENABLE_NO_OP</code> to have the Transfer Family server ignore the
     *         <code>SETSTAT</code> command, and upload files without needing to make any changes to your SFTP client.
     *         While the <code>SetStatOption</code> <code>ENABLE_NO_OP</code> setting ignores the error, it does
     *         generate a log entry in Amazon CloudWatch Logs, so you can determine when the client is making a
     *         <code>SETSTAT</code> call.
     *         </p>
     *         <note>
     *         <p>
     *         If you want to preserve the original timestamp for your file, and modify other file attributes using
     *         <code>SETSTAT</code>, you can use Amazon EFS as backend storage with Transfer Family.
     *         </p>
     * @see SetStatOption
     */
    public final SetStatOption setStatOption() {
        return SetStatOption.fromValue(setStatOption);
    }

    /**
     * <p>
     * Use the <code>SetStatOption</code> to ignore the error that is generated when the client attempts to use
     * <code>SETSTAT</code> on a file you are uploading to an S3 bucket.
     * </p>
     * <p>
     * Some SFTP file transfer clients can attempt to change the attributes of remote files, including timestamp and
     * permissions, using commands, such as <code>SETSTAT</code> when uploading the file. However, these commands are
     * not compatible with object storage systems, such as Amazon S3. Due to this incompatibility, file uploads from
     * these clients can result in errors even when the file is otherwise successfully uploaded.
     * </p>
     * <p>
     * Set the value to <code>ENABLE_NO_OP</code> to have the Transfer Family server ignore the <code>SETSTAT</code>
     * command, and upload files without needing to make any changes to your SFTP client. While the
     * <code>SetStatOption</code> <code>ENABLE_NO_OP</code> setting ignores the error, it does generate a log entry in
     * Amazon CloudWatch Logs, so you can determine when the client is making a <code>SETSTAT</code> call.
     * </p>
     * <note>
     * <p>
     * If you want to preserve the original timestamp for your file, and modify other file attributes using
     * <code>SETSTAT</code>, you can use Amazon EFS as backend storage with Transfer Family.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #setStatOption}
     * will return {@link SetStatOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #setStatOptionAsString}.
     * </p>
     * 
     * @return Use the <code>SetStatOption</code> to ignore the error that is generated when the client attempts to use
     *         <code>SETSTAT</code> on a file you are uploading to an S3 bucket.</p>
     *         <p>
     *         Some SFTP file transfer clients can attempt to change the attributes of remote files, including timestamp
     *         and permissions, using commands, such as <code>SETSTAT</code> when uploading the file. However, these
     *         commands are not compatible with object storage systems, such as Amazon S3. Due to this incompatibility,
     *         file uploads from these clients can result in errors even when the file is otherwise successfully
     *         uploaded.
     *         </p>
     *         <p>
     *         Set the value to <code>ENABLE_NO_OP</code> to have the Transfer Family server ignore the
     *         <code>SETSTAT</code> command, and upload files without needing to make any changes to your SFTP client.
     *         While the <code>SetStatOption</code> <code>ENABLE_NO_OP</code> setting ignores the error, it does
     *         generate a log entry in Amazon CloudWatch Logs, so you can determine when the client is making a
     *         <code>SETSTAT</code> call.
     *         </p>
     *         <note>
     *         <p>
     *         If you want to preserve the original timestamp for your file, and modify other file attributes using
     *         <code>SETSTAT</code>, you can use Amazon EFS as backend storage with Transfer Family.
     *         </p>
     * @see SetStatOption
     */
    public final String setStatOptionAsString() {
        return setStatOption;
    }

    /**
     * <p>
     * Indicates the transport method for the AS2 messages. Currently, only HTTP is supported.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAs2Transports} method.
     * </p>
     * 
     * @return Indicates the transport method for the AS2 messages. Currently, only HTTP is supported.
     */
    public final List<As2Transport> as2Transports() {
        return As2TransportsCopier.copyStringToEnum(as2Transports);
    }

    /**
     * For responses, this returns true if the service returned a value for the As2Transports property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAs2Transports() {
        return as2Transports != null && !(as2Transports instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Indicates the transport method for the AS2 messages. Currently, only HTTP is supported.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAs2Transports} method.
     * </p>
     * 
     * @return Indicates the transport method for the AS2 messages. Currently, only HTTP is supported.
     */
    public final List<String> as2TransportsAsStrings() {
        return as2Transports;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(passiveIp());
        hashCode = 31 * hashCode + Objects.hashCode(tlsSessionResumptionModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(setStatOptionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasAs2Transports() ? as2TransportsAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProtocolDetails)) {
            return false;
        }
        ProtocolDetails other = (ProtocolDetails) obj;
        return Objects.equals(passiveIp(), other.passiveIp())
                && Objects.equals(tlsSessionResumptionModeAsString(), other.tlsSessionResumptionModeAsString())
                && Objects.equals(setStatOptionAsString(), other.setStatOptionAsString())
                && hasAs2Transports() == other.hasAs2Transports()
                && Objects.equals(as2TransportsAsStrings(), other.as2TransportsAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ProtocolDetails").add("PassiveIp", passiveIp())
                .add("TlsSessionResumptionMode", tlsSessionResumptionModeAsString())
                .add("SetStatOption", setStatOptionAsString())
                .add("As2Transports", hasAs2Transports() ? as2TransportsAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PassiveIp":
            return Optional.ofNullable(clazz.cast(passiveIp()));
        case "TlsSessionResumptionMode":
            return Optional.ofNullable(clazz.cast(tlsSessionResumptionModeAsString()));
        case "SetStatOption":
            return Optional.ofNullable(clazz.cast(setStatOptionAsString()));
        case "As2Transports":
            return Optional.ofNullable(clazz.cast(as2TransportsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ProtocolDetails, T> g) {
        return obj -> g.apply((ProtocolDetails) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProtocolDetails> {
        /**
         * <p>
         * Indicates passive mode, for FTP and FTPS protocols. Enter a single IPv4 address, such as the public IP
         * address of a firewall, router, or load balancer. For example:
         * </p>
         * <p>
         * <code> aws transfer update-server --protocol-details PassiveIp=<i>0.0.0.0</i> </code>
         * </p>
         * <p>
         * Replace <code> <i>0.0.0.0</i> </code> in the example above with the actual IP address you want to use.
         * </p>
         * <note>
         * <p>
         * If you change the <code>PassiveIp</code> value, you must stop and then restart your Transfer Family server
         * for the change to take effect. For details on using passive mode (PASV) in a NAT environment, see <a href=
         * "http://aws.amazon.com/blogs/storage/configuring-your-ftps-server-behind-a-firewall-or-nat-with-aws-transfer-family/"
         * >Configuring your FTPS server behind a firewall or NAT with Transfer Family</a>.
         * </p>
         * </note>
         * 
         * @param passiveIp
         *        Indicates passive mode, for FTP and FTPS protocols. Enter a single IPv4 address, such as the public IP
         *        address of a firewall, router, or load balancer. For example: </p>
         *        <p>
         *        <code> aws transfer update-server --protocol-details PassiveIp=<i>0.0.0.0</i> </code>
         *        </p>
         *        <p>
         *        Replace <code> <i>0.0.0.0</i> </code> in the example above with the actual IP address you want to use.
         *        </p>
         *        <note>
         *        <p>
         *        If you change the <code>PassiveIp</code> value, you must stop and then restart your Transfer Family
         *        server for the change to take effect. For details on using passive mode (PASV) in a NAT environment,
         *        see <a href=
         *        "http://aws.amazon.com/blogs/storage/configuring-your-ftps-server-behind-a-firewall-or-nat-with-aws-transfer-family/"
         *        >Configuring your FTPS server behind a firewall or NAT with Transfer Family</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder passiveIp(String passiveIp);

        /**
         * <p>
         * A property used with Transfer Family servers that use the FTPS protocol. TLS Session Resumption provides a
         * mechanism to resume or share a negotiated secret key between the control and data connection for an FTPS
         * session. <code>TlsSessionResumptionMode</code> determines whether or not the server resumes recent,
         * negotiated sessions through a unique session ID. This property is available during <code>CreateServer</code>
         * and <code>UpdateServer</code> calls. If a <code>TlsSessionResumptionMode</code> value is not specified during
         * <code>CreateServer</code>, it is set to <code>ENFORCED</code> by default.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DISABLED</code>: the server does not process TLS session resumption client requests and creates a new
         * TLS session for each request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ENABLED</code>: the server processes and accepts clients that are performing TLS session resumption.
         * The server doesn't reject client data connections that do not perform the TLS session resumption client
         * processing.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ENFORCED</code>: the server processes and accepts clients that are performing TLS session resumption.
         * The server rejects client data connections that do not perform the TLS session resumption client processing.
         * Before you set the value to <code>ENFORCED</code>, test your clients.
         * </p>
         * <note>
         * <p>
         * Not all FTPS clients perform TLS session resumption. So, if you choose to enforce TLS session resumption, you
         * prevent any connections from FTPS clients that don't perform the protocol negotiation. To determine whether
         * or not you can use the <code>ENFORCED</code> value, you need to test your clients.
         * </p>
         * </note></li>
         * </ul>
         * 
         * @param tlsSessionResumptionMode
         *        A property used with Transfer Family servers that use the FTPS protocol. TLS Session Resumption
         *        provides a mechanism to resume or share a negotiated secret key between the control and data
         *        connection for an FTPS session. <code>TlsSessionResumptionMode</code> determines whether or not the
         *        server resumes recent, negotiated sessions through a unique session ID. This property is available
         *        during <code>CreateServer</code> and <code>UpdateServer</code> calls. If a
         *        <code>TlsSessionResumptionMode</code> value is not specified during <code>CreateServer</code>, it is
         *        set to <code>ENFORCED</code> by default.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DISABLED</code>: the server does not process TLS session resumption client requests and creates
         *        a new TLS session for each request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ENABLED</code>: the server processes and accepts clients that are performing TLS session
         *        resumption. The server doesn't reject client data connections that do not perform the TLS session
         *        resumption client processing.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ENFORCED</code>: the server processes and accepts clients that are performing TLS session
         *        resumption. The server rejects client data connections that do not perform the TLS session resumption
         *        client processing. Before you set the value to <code>ENFORCED</code>, test your clients.
         *        </p>
         *        <note>
         *        <p>
         *        Not all FTPS clients perform TLS session resumption. So, if you choose to enforce TLS session
         *        resumption, you prevent any connections from FTPS clients that don't perform the protocol negotiation.
         *        To determine whether or not you can use the <code>ENFORCED</code> value, you need to test your
         *        clients.
         *        </p>
         *        </note></li>
         * @see TlsSessionResumptionMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TlsSessionResumptionMode
         */
        Builder tlsSessionResumptionMode(String tlsSessionResumptionMode);

        /**
         * <p>
         * A property used with Transfer Family servers that use the FTPS protocol. TLS Session Resumption provides a
         * mechanism to resume or share a negotiated secret key between the control and data connection for an FTPS
         * session. <code>TlsSessionResumptionMode</code> determines whether or not the server resumes recent,
         * negotiated sessions through a unique session ID. This property is available during <code>CreateServer</code>
         * and <code>UpdateServer</code> calls. If a <code>TlsSessionResumptionMode</code> value is not specified during
         * <code>CreateServer</code>, it is set to <code>ENFORCED</code> by default.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DISABLED</code>: the server does not process TLS session resumption client requests and creates a new
         * TLS session for each request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ENABLED</code>: the server processes and accepts clients that are performing TLS session resumption.
         * The server doesn't reject client data connections that do not perform the TLS session resumption client
         * processing.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ENFORCED</code>: the server processes and accepts clients that are performing TLS session resumption.
         * The server rejects client data connections that do not perform the TLS session resumption client processing.
         * Before you set the value to <code>ENFORCED</code>, test your clients.
         * </p>
         * <note>
         * <p>
         * Not all FTPS clients perform TLS session resumption. So, if you choose to enforce TLS session resumption, you
         * prevent any connections from FTPS clients that don't perform the protocol negotiation. To determine whether
         * or not you can use the <code>ENFORCED</code> value, you need to test your clients.
         * </p>
         * </note></li>
         * </ul>
         * 
         * @param tlsSessionResumptionMode
         *        A property used with Transfer Family servers that use the FTPS protocol. TLS Session Resumption
         *        provides a mechanism to resume or share a negotiated secret key between the control and data
         *        connection for an FTPS session. <code>TlsSessionResumptionMode</code> determines whether or not the
         *        server resumes recent, negotiated sessions through a unique session ID. This property is available
         *        during <code>CreateServer</code> and <code>UpdateServer</code> calls. If a
         *        <code>TlsSessionResumptionMode</code> value is not specified during <code>CreateServer</code>, it is
         *        set to <code>ENFORCED</code> by default.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DISABLED</code>: the server does not process TLS session resumption client requests and creates
         *        a new TLS session for each request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ENABLED</code>: the server processes and accepts clients that are performing TLS session
         *        resumption. The server doesn't reject client data connections that do not perform the TLS session
         *        resumption client processing.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ENFORCED</code>: the server processes and accepts clients that are performing TLS session
         *        resumption. The server rejects client data connections that do not perform the TLS session resumption
         *        client processing. Before you set the value to <code>ENFORCED</code>, test your clients.
         *        </p>
         *        <note>
         *        <p>
         *        Not all FTPS clients perform TLS session resumption. So, if you choose to enforce TLS session
         *        resumption, you prevent any connections from FTPS clients that don't perform the protocol negotiation.
         *        To determine whether or not you can use the <code>ENFORCED</code> value, you need to test your
         *        clients.
         *        </p>
         *        </note></li>
         * @see TlsSessionResumptionMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TlsSessionResumptionMode
         */
        Builder tlsSessionResumptionMode(TlsSessionResumptionMode tlsSessionResumptionMode);

        /**
         * <p>
         * Use the <code>SetStatOption</code> to ignore the error that is generated when the client attempts to use
         * <code>SETSTAT</code> on a file you are uploading to an S3 bucket.
         * </p>
         * <p>
         * Some SFTP file transfer clients can attempt to change the attributes of remote files, including timestamp and
         * permissions, using commands, such as <code>SETSTAT</code> when uploading the file. However, these commands
         * are not compatible with object storage systems, such as Amazon S3. Due to this incompatibility, file uploads
         * from these clients can result in errors even when the file is otherwise successfully uploaded.
         * </p>
         * <p>
         * Set the value to <code>ENABLE_NO_OP</code> to have the Transfer Family server ignore the <code>SETSTAT</code>
         * command, and upload files without needing to make any changes to your SFTP client. While the
         * <code>SetStatOption</code> <code>ENABLE_NO_OP</code> setting ignores the error, it does generate a log entry
         * in Amazon CloudWatch Logs, so you can determine when the client is making a <code>SETSTAT</code> call.
         * </p>
         * <note>
         * <p>
         * If you want to preserve the original timestamp for your file, and modify other file attributes using
         * <code>SETSTAT</code>, you can use Amazon EFS as backend storage with Transfer Family.
         * </p>
         * </note>
         * 
         * @param setStatOption
         *        Use the <code>SetStatOption</code> to ignore the error that is generated when the client attempts to
         *        use <code>SETSTAT</code> on a file you are uploading to an S3 bucket.</p>
         *        <p>
         *        Some SFTP file transfer clients can attempt to change the attributes of remote files, including
         *        timestamp and permissions, using commands, such as <code>SETSTAT</code> when uploading the file.
         *        However, these commands are not compatible with object storage systems, such as Amazon S3. Due to this
         *        incompatibility, file uploads from these clients can result in errors even when the file is otherwise
         *        successfully uploaded.
         *        </p>
         *        <p>
         *        Set the value to <code>ENABLE_NO_OP</code> to have the Transfer Family server ignore the
         *        <code>SETSTAT</code> command, and upload files without needing to make any changes to your SFTP
         *        client. While the <code>SetStatOption</code> <code>ENABLE_NO_OP</code> setting ignores the error, it
         *        does generate a log entry in Amazon CloudWatch Logs, so you can determine when the client is making a
         *        <code>SETSTAT</code> call.
         *        </p>
         *        <note>
         *        <p>
         *        If you want to preserve the original timestamp for your file, and modify other file attributes using
         *        <code>SETSTAT</code>, you can use Amazon EFS as backend storage with Transfer Family.
         *        </p>
         * @see SetStatOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SetStatOption
         */
        Builder setStatOption(String setStatOption);

        /**
         * <p>
         * Use the <code>SetStatOption</code> to ignore the error that is generated when the client attempts to use
         * <code>SETSTAT</code> on a file you are uploading to an S3 bucket.
         * </p>
         * <p>
         * Some SFTP file transfer clients can attempt to change the attributes of remote files, including timestamp and
         * permissions, using commands, such as <code>SETSTAT</code> when uploading the file. However, these commands
         * are not compatible with object storage systems, such as Amazon S3. Due to this incompatibility, file uploads
         * from these clients can result in errors even when the file is otherwise successfully uploaded.
         * </p>
         * <p>
         * Set the value to <code>ENABLE_NO_OP</code> to have the Transfer Family server ignore the <code>SETSTAT</code>
         * command, and upload files without needing to make any changes to your SFTP client. While the
         * <code>SetStatOption</code> <code>ENABLE_NO_OP</code> setting ignores the error, it does generate a log entry
         * in Amazon CloudWatch Logs, so you can determine when the client is making a <code>SETSTAT</code> call.
         * </p>
         * <note>
         * <p>
         * If you want to preserve the original timestamp for your file, and modify other file attributes using
         * <code>SETSTAT</code>, you can use Amazon EFS as backend storage with Transfer Family.
         * </p>
         * </note>
         * 
         * @param setStatOption
         *        Use the <code>SetStatOption</code> to ignore the error that is generated when the client attempts to
         *        use <code>SETSTAT</code> on a file you are uploading to an S3 bucket.</p>
         *        <p>
         *        Some SFTP file transfer clients can attempt to change the attributes of remote files, including
         *        timestamp and permissions, using commands, such as <code>SETSTAT</code> when uploading the file.
         *        However, these commands are not compatible with object storage systems, such as Amazon S3. Due to this
         *        incompatibility, file uploads from these clients can result in errors even when the file is otherwise
         *        successfully uploaded.
         *        </p>
         *        <p>
         *        Set the value to <code>ENABLE_NO_OP</code> to have the Transfer Family server ignore the
         *        <code>SETSTAT</code> command, and upload files without needing to make any changes to your SFTP
         *        client. While the <code>SetStatOption</code> <code>ENABLE_NO_OP</code> setting ignores the error, it
         *        does generate a log entry in Amazon CloudWatch Logs, so you can determine when the client is making a
         *        <code>SETSTAT</code> call.
         *        </p>
         *        <note>
         *        <p>
         *        If you want to preserve the original timestamp for your file, and modify other file attributes using
         *        <code>SETSTAT</code>, you can use Amazon EFS as backend storage with Transfer Family.
         *        </p>
         * @see SetStatOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SetStatOption
         */
        Builder setStatOption(SetStatOption setStatOption);

        /**
         * <p>
         * Indicates the transport method for the AS2 messages. Currently, only HTTP is supported.
         * </p>
         * 
         * @param as2Transports
         *        Indicates the transport method for the AS2 messages. Currently, only HTTP is supported.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder as2TransportsWithStrings(Collection<String> as2Transports);

        /**
         * <p>
         * Indicates the transport method for the AS2 messages. Currently, only HTTP is supported.
         * </p>
         * 
         * @param as2Transports
         *        Indicates the transport method for the AS2 messages. Currently, only HTTP is supported.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder as2TransportsWithStrings(String... as2Transports);

        /**
         * <p>
         * Indicates the transport method for the AS2 messages. Currently, only HTTP is supported.
         * </p>
         * 
         * @param as2Transports
         *        Indicates the transport method for the AS2 messages. Currently, only HTTP is supported.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder as2Transports(Collection<As2Transport> as2Transports);

        /**
         * <p>
         * Indicates the transport method for the AS2 messages. Currently, only HTTP is supported.
         * </p>
         * 
         * @param as2Transports
         *        Indicates the transport method for the AS2 messages. Currently, only HTTP is supported.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder as2Transports(As2Transport... as2Transports);
    }

    static final class BuilderImpl implements Builder {
        private String passiveIp;

        private String tlsSessionResumptionMode;

        private String setStatOption;

        private List<String> as2Transports = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ProtocolDetails model) {
            passiveIp(model.passiveIp);
            tlsSessionResumptionMode(model.tlsSessionResumptionMode);
            setStatOption(model.setStatOption);
            as2TransportsWithStrings(model.as2Transports);
        }

        public final String getPassiveIp() {
            return passiveIp;
        }

        public final void setPassiveIp(String passiveIp) {
            this.passiveIp = passiveIp;
        }

        @Override
        public final Builder passiveIp(String passiveIp) {
            this.passiveIp = passiveIp;
            return this;
        }

        public final String getTlsSessionResumptionMode() {
            return tlsSessionResumptionMode;
        }

        public final void setTlsSessionResumptionMode(String tlsSessionResumptionMode) {
            this.tlsSessionResumptionMode = tlsSessionResumptionMode;
        }

        @Override
        public final Builder tlsSessionResumptionMode(String tlsSessionResumptionMode) {
            this.tlsSessionResumptionMode = tlsSessionResumptionMode;
            return this;
        }

        @Override
        public final Builder tlsSessionResumptionMode(TlsSessionResumptionMode tlsSessionResumptionMode) {
            this.tlsSessionResumptionMode(tlsSessionResumptionMode == null ? null : tlsSessionResumptionMode.toString());
            return this;
        }

        public final String getSetStatOption() {
            return setStatOption;
        }

        public final void setSetStatOption(String setStatOption) {
            this.setStatOption = setStatOption;
        }

        @Override
        @Transient
        public final Builder setStatOption(String setStatOption) {
            this.setStatOption = setStatOption;
            return this;
        }

        @Override
        @Transient
        public final Builder setStatOption(SetStatOption setStatOption) {
            this.setStatOption(setStatOption == null ? null : setStatOption.toString());
            return this;
        }

        public final Collection<String> getAs2Transports() {
            if (as2Transports instanceof SdkAutoConstructList) {
                return null;
            }
            return as2Transports;
        }

        public final void setAs2Transports(Collection<String> as2Transports) {
            this.as2Transports = As2TransportsCopier.copy(as2Transports);
        }

        @Override
        public final Builder as2TransportsWithStrings(Collection<String> as2Transports) {
            this.as2Transports = As2TransportsCopier.copy(as2Transports);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder as2TransportsWithStrings(String... as2Transports) {
            as2TransportsWithStrings(Arrays.asList(as2Transports));
            return this;
        }

        @Override
        public final Builder as2Transports(Collection<As2Transport> as2Transports) {
            this.as2Transports = As2TransportsCopier.copyEnumToString(as2Transports);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder as2Transports(As2Transport... as2Transports) {
            as2Transports(Arrays.asList(as2Transports));
            return this;
        }

        @Override
        public ProtocolDetails build() {
            return new ProtocolDetails(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
