/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transfer.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the properties of a file transfer protocol-enabled server that was specified.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DescribedServer implements SdkPojo, Serializable, ToCopyableBuilder<DescribedServer.Builder, DescribedServer> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(DescribedServer::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> CERTIFICATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Certificate").getter(getter(DescribedServer::certificate)).setter(setter(Builder::certificate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Certificate").build()).build();

    private static final SdkField<ProtocolDetails> PROTOCOL_DETAILS_FIELD = SdkField
            .<ProtocolDetails> builder(MarshallingType.SDK_POJO).memberName("ProtocolDetails")
            .getter(getter(DescribedServer::protocolDetails)).setter(setter(Builder::protocolDetails))
            .constructor(ProtocolDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProtocolDetails").build()).build();

    private static final SdkField<String> DOMAIN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Domain")
            .getter(getter(DescribedServer::domainAsString)).setter(setter(Builder::domain))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Domain").build()).build();

    private static final SdkField<EndpointDetails> ENDPOINT_DETAILS_FIELD = SdkField
            .<EndpointDetails> builder(MarshallingType.SDK_POJO).memberName("EndpointDetails")
            .getter(getter(DescribedServer::endpointDetails)).setter(setter(Builder::endpointDetails))
            .constructor(EndpointDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointDetails").build()).build();

    private static final SdkField<String> ENDPOINT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EndpointType").getter(getter(DescribedServer::endpointTypeAsString))
            .setter(setter(Builder::endpointType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointType").build()).build();

    private static final SdkField<String> HOST_KEY_FINGERPRINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HostKeyFingerprint").getter(getter(DescribedServer::hostKeyFingerprint))
            .setter(setter(Builder::hostKeyFingerprint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HostKeyFingerprint").build())
            .build();

    private static final SdkField<IdentityProviderDetails> IDENTITY_PROVIDER_DETAILS_FIELD = SdkField
            .<IdentityProviderDetails> builder(MarshallingType.SDK_POJO).memberName("IdentityProviderDetails")
            .getter(getter(DescribedServer::identityProviderDetails)).setter(setter(Builder::identityProviderDetails))
            .constructor(IdentityProviderDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdentityProviderDetails").build())
            .build();

    private static final SdkField<String> IDENTITY_PROVIDER_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IdentityProviderType").getter(getter(DescribedServer::identityProviderTypeAsString))
            .setter(setter(Builder::identityProviderType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdentityProviderType").build())
            .build();

    private static final SdkField<String> LOGGING_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LoggingRole").getter(getter(DescribedServer::loggingRole)).setter(setter(Builder::loggingRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LoggingRole").build()).build();

    private static final SdkField<List<String>> PROTOCOLS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Protocols")
            .getter(getter(DescribedServer::protocolsAsStrings))
            .setter(setter(Builder::protocolsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Protocols").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SECURITY_POLICY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecurityPolicyName").getter(getter(DescribedServer::securityPolicyName))
            .setter(setter(Builder::securityPolicyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityPolicyName").build())
            .build();

    private static final SdkField<String> SERVER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServerId").getter(getter(DescribedServer::serverId)).setter(setter(Builder::serverId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerId").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(DescribedServer::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(DescribedServer::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> USER_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("UserCount").getter(getter(DescribedServer::userCount)).setter(setter(Builder::userCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserCount").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, CERTIFICATE_FIELD,
            PROTOCOL_DETAILS_FIELD, DOMAIN_FIELD, ENDPOINT_DETAILS_FIELD, ENDPOINT_TYPE_FIELD, HOST_KEY_FINGERPRINT_FIELD,
            IDENTITY_PROVIDER_DETAILS_FIELD, IDENTITY_PROVIDER_TYPE_FIELD, LOGGING_ROLE_FIELD, PROTOCOLS_FIELD,
            SECURITY_POLICY_NAME_FIELD, SERVER_ID_FIELD, STATE_FIELD, TAGS_FIELD, USER_COUNT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final String certificate;

    private final ProtocolDetails protocolDetails;

    private final String domain;

    private final EndpointDetails endpointDetails;

    private final String endpointType;

    private final String hostKeyFingerprint;

    private final IdentityProviderDetails identityProviderDetails;

    private final String identityProviderType;

    private final String loggingRole;

    private final List<String> protocols;

    private final String securityPolicyName;

    private final String serverId;

    private final String state;

    private final List<Tag> tags;

    private final Integer userCount;

    private DescribedServer(BuilderImpl builder) {
        this.arn = builder.arn;
        this.certificate = builder.certificate;
        this.protocolDetails = builder.protocolDetails;
        this.domain = builder.domain;
        this.endpointDetails = builder.endpointDetails;
        this.endpointType = builder.endpointType;
        this.hostKeyFingerprint = builder.hostKeyFingerprint;
        this.identityProviderDetails = builder.identityProviderDetails;
        this.identityProviderType = builder.identityProviderType;
        this.loggingRole = builder.loggingRole;
        this.protocols = builder.protocols;
        this.securityPolicyName = builder.securityPolicyName;
        this.serverId = builder.serverId;
        this.state = builder.state;
        this.tags = builder.tags;
        this.userCount = builder.userCount;
    }

    /**
     * <p>
     * Specifies the unique Amazon Resource Name (ARN) of the server.
     * </p>
     * 
     * @return Specifies the unique Amazon Resource Name (ARN) of the server.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * Specifies the ARN of the Amazon Web ServicesCertificate Manager (ACM) certificate. Required when
     * <code>Protocols</code> is set to <code>FTPS</code>.
     * </p>
     * 
     * @return Specifies the ARN of the Amazon Web ServicesCertificate Manager (ACM) certificate. Required when
     *         <code>Protocols</code> is set to <code>FTPS</code>.
     */
    public final String certificate() {
        return certificate;
    }

    /**
     * <p>
     * The protocol settings that are configured for your server.
     * </p>
     * <p>
     * Use the <code>PassiveIp</code> parameter to indicate passive mode. Enter a single dotted-quad IPv4 address, such
     * as the external IP address of a firewall, router, or load balancer.
     * </p>
     * 
     * @return The protocol settings that are configured for your server. </p>
     *         <p>
     *         Use the <code>PassiveIp</code> parameter to indicate passive mode. Enter a single dotted-quad IPv4
     *         address, such as the external IP address of a firewall, router, or load balancer.
     */
    public final ProtocolDetails protocolDetails() {
        return protocolDetails;
    }

    /**
     * <p>
     * Specifies the domain of the storage system that is used for file transfers.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #domain} will
     * return {@link Domain#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #domainAsString}.
     * </p>
     * 
     * @return Specifies the domain of the storage system that is used for file transfers.
     * @see Domain
     */
    public final Domain domain() {
        return Domain.fromValue(domain);
    }

    /**
     * <p>
     * Specifies the domain of the storage system that is used for file transfers.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #domain} will
     * return {@link Domain#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #domainAsString}.
     * </p>
     * 
     * @return Specifies the domain of the storage system that is used for file transfers.
     * @see Domain
     */
    public final String domainAsString() {
        return domain;
    }

    /**
     * <p>
     * The virtual private cloud (VPC) endpoint settings that are configured for your server. When you host your
     * endpoint within your VPC, you can make it accessible only to resources within your VPC, or you can attach Elastic
     * IP addresses and make it accessible to clients over the internet. Your VPC's default security groups are
     * automatically assigned to your endpoint.
     * </p>
     * 
     * @return The virtual private cloud (VPC) endpoint settings that are configured for your server. When you host your
     *         endpoint within your VPC, you can make it accessible only to resources within your VPC, or you can attach
     *         Elastic IP addresses and make it accessible to clients over the internet. Your VPC's default security
     *         groups are automatically assigned to your endpoint.
     */
    public final EndpointDetails endpointDetails() {
        return endpointDetails;
    }

    /**
     * <p>
     * Defines the type of endpoint that your server is connected to. If your server is connected to a VPC endpoint,
     * your server isn't accessible over the public internet.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #endpointType} will
     * return {@link EndpointType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #endpointTypeAsString}.
     * </p>
     * 
     * @return Defines the type of endpoint that your server is connected to. If your server is connected to a VPC
     *         endpoint, your server isn't accessible over the public internet.
     * @see EndpointType
     */
    public final EndpointType endpointType() {
        return EndpointType.fromValue(endpointType);
    }

    /**
     * <p>
     * Defines the type of endpoint that your server is connected to. If your server is connected to a VPC endpoint,
     * your server isn't accessible over the public internet.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #endpointType} will
     * return {@link EndpointType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #endpointTypeAsString}.
     * </p>
     * 
     * @return Defines the type of endpoint that your server is connected to. If your server is connected to a VPC
     *         endpoint, your server isn't accessible over the public internet.
     * @see EndpointType
     */
    public final String endpointTypeAsString() {
        return endpointType;
    }

    /**
     * <p>
     * Specifies the Base64-encoded SHA256 fingerprint of the server's host key. This value is equivalent to the output
     * of the <code>ssh-keygen -l -f my-new-server-key</code> command.
     * </p>
     * 
     * @return Specifies the Base64-encoded SHA256 fingerprint of the server's host key. This value is equivalent to the
     *         output of the <code>ssh-keygen -l -f my-new-server-key</code> command.
     */
    public final String hostKeyFingerprint() {
        return hostKeyFingerprint;
    }

    /**
     * <p>
     * Specifies information to call a customer-supplied authentication API. This field is not populated when the
     * <code>IdentityProviderType</code> of a server is <code>AWS_DIRECTORY_SERVICE</code> or
     * <code>SERVICE_MANAGED</code>.
     * </p>
     * 
     * @return Specifies information to call a customer-supplied authentication API. This field is not populated when
     *         the <code>IdentityProviderType</code> of a server is <code>AWS_DIRECTORY_SERVICE</code> or
     *         <code>SERVICE_MANAGED</code>.
     */
    public final IdentityProviderDetails identityProviderDetails() {
        return identityProviderDetails;
    }

    /**
     * <p>
     * Specifies the mode of authentication for a server. The default value is <code>SERVICE_MANAGED</code>, which
     * allows you to store and access user credentials within the Amazon Web Services Transfer Family service.
     * </p>
     * <p>
     * Use <code>AWS_DIRECTORY_SERVICE</code> to provide access to Active Directory groups in Amazon Web Services
     * Managed Active Directory or Microsoft Active Directory in your on-premises environment or in Amazon Web Services
     * using AD Connectors. This option also requires you to provide a Directory ID using the
     * <code>IdentityProviderDetails</code> parameter.
     * </p>
     * <p>
     * Use the <code>API_GATEWAY</code> value to integrate with an identity provider of your choosing. The
     * <code>API_GATEWAY</code> setting requires you to provide an API Gateway endpoint URL to call for authentication
     * using the <code>IdentityProviderDetails</code> parameter.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #identityProviderType} will return {@link IdentityProviderType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #identityProviderTypeAsString}.
     * </p>
     * 
     * @return Specifies the mode of authentication for a server. The default value is <code>SERVICE_MANAGED</code>,
     *         which allows you to store and access user credentials within the Amazon Web Services Transfer Family
     *         service.</p>
     *         <p>
     *         Use <code>AWS_DIRECTORY_SERVICE</code> to provide access to Active Directory groups in Amazon Web
     *         Services Managed Active Directory or Microsoft Active Directory in your on-premises environment or in
     *         Amazon Web Services using AD Connectors. This option also requires you to provide a Directory ID using
     *         the <code>IdentityProviderDetails</code> parameter.
     *         </p>
     *         <p>
     *         Use the <code>API_GATEWAY</code> value to integrate with an identity provider of your choosing. The
     *         <code>API_GATEWAY</code> setting requires you to provide an API Gateway endpoint URL to call for
     *         authentication using the <code>IdentityProviderDetails</code> parameter.
     * @see IdentityProviderType
     */
    public final IdentityProviderType identityProviderType() {
        return IdentityProviderType.fromValue(identityProviderType);
    }

    /**
     * <p>
     * Specifies the mode of authentication for a server. The default value is <code>SERVICE_MANAGED</code>, which
     * allows you to store and access user credentials within the Amazon Web Services Transfer Family service.
     * </p>
     * <p>
     * Use <code>AWS_DIRECTORY_SERVICE</code> to provide access to Active Directory groups in Amazon Web Services
     * Managed Active Directory or Microsoft Active Directory in your on-premises environment or in Amazon Web Services
     * using AD Connectors. This option also requires you to provide a Directory ID using the
     * <code>IdentityProviderDetails</code> parameter.
     * </p>
     * <p>
     * Use the <code>API_GATEWAY</code> value to integrate with an identity provider of your choosing. The
     * <code>API_GATEWAY</code> setting requires you to provide an API Gateway endpoint URL to call for authentication
     * using the <code>IdentityProviderDetails</code> parameter.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #identityProviderType} will return {@link IdentityProviderType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #identityProviderTypeAsString}.
     * </p>
     * 
     * @return Specifies the mode of authentication for a server. The default value is <code>SERVICE_MANAGED</code>,
     *         which allows you to store and access user credentials within the Amazon Web Services Transfer Family
     *         service.</p>
     *         <p>
     *         Use <code>AWS_DIRECTORY_SERVICE</code> to provide access to Active Directory groups in Amazon Web
     *         Services Managed Active Directory or Microsoft Active Directory in your on-premises environment or in
     *         Amazon Web Services using AD Connectors. This option also requires you to provide a Directory ID using
     *         the <code>IdentityProviderDetails</code> parameter.
     *         </p>
     *         <p>
     *         Use the <code>API_GATEWAY</code> value to integrate with an identity provider of your choosing. The
     *         <code>API_GATEWAY</code> setting requires you to provide an API Gateway endpoint URL to call for
     *         authentication using the <code>IdentityProviderDetails</code> parameter.
     * @see IdentityProviderType
     */
    public final String identityProviderTypeAsString() {
        return identityProviderType;
    }

    /**
     * <p>
     * Specifies the Amazon Resource Name (ARN) of the Amazon Web Services Identity and Access Management (IAM) role
     * that allows a server to turn on Amazon CloudWatch logging for Amazon S3 or Amazon EFS events. When set, user
     * activity can be viewed in your CloudWatch logs.
     * </p>
     * 
     * @return Specifies the Amazon Resource Name (ARN) of the Amazon Web Services Identity and Access Management (IAM)
     *         role that allows a server to turn on Amazon CloudWatch logging for Amazon S3 or Amazon EFS events. When
     *         set, user activity can be viewed in your CloudWatch logs.
     */
    public final String loggingRole() {
        return loggingRole;
    }

    /**
     * <p>
     * Specifies the file transfer protocol or protocols over which your file transfer protocol client can connect to
     * your server's endpoint. The available protocols are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProtocols} method.
     * </p>
     * 
     * @return Specifies the file transfer protocol or protocols over which your file transfer protocol client can
     *         connect to your server's endpoint. The available protocols are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
     *         </p>
     *         </li>
     */
    public final List<Protocol> protocols() {
        return ProtocolsCopier.copyStringToEnum(protocols);
    }

    /**
     * For responses, this returns true if the service returned a value for the Protocols property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasProtocols() {
        return protocols != null && !(protocols instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the file transfer protocol or protocols over which your file transfer protocol client can connect to
     * your server's endpoint. The available protocols are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProtocols} method.
     * </p>
     * 
     * @return Specifies the file transfer protocol or protocols over which your file transfer protocol client can
     *         connect to your server's endpoint. The available protocols are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
     *         </p>
     *         </li>
     */
    public final List<String> protocolsAsStrings() {
        return protocols;
    }

    /**
     * <p>
     * Specifies the name of the security policy that is attached to the server.
     * </p>
     * 
     * @return Specifies the name of the security policy that is attached to the server.
     */
    public final String securityPolicyName() {
        return securityPolicyName;
    }

    /**
     * <p>
     * Specifies the unique system-assigned identifier for a server that you instantiate.
     * </p>
     * 
     * @return Specifies the unique system-assigned identifier for a server that you instantiate.
     */
    public final String serverId() {
        return serverId;
    }

    /**
     * <p>
     * Specifies the condition of a server for the server that was described. A value of <code>ONLINE</code> indicates
     * that the server can accept jobs and transfer files. A <code>State</code> value of <code>OFFLINE</code> means that
     * the server cannot perform file transfer operations.
     * </p>
     * <p>
     * The states of <code>STARTING</code> and <code>STOPPING</code> indicate that the server is in an intermediate
     * state, either not fully able to respond, or not fully offline. The values of <code>START_FAILED</code> or
     * <code>STOP_FAILED</code> can indicate an error condition.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link State#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return Specifies the condition of a server for the server that was described. A value of <code>ONLINE</code>
     *         indicates that the server can accept jobs and transfer files. A <code>State</code> value of
     *         <code>OFFLINE</code> means that the server cannot perform file transfer operations.</p>
     *         <p>
     *         The states of <code>STARTING</code> and <code>STOPPING</code> indicate that the server is in an
     *         intermediate state, either not fully able to respond, or not fully offline. The values of
     *         <code>START_FAILED</code> or <code>STOP_FAILED</code> can indicate an error condition.
     * @see State
     */
    public final State state() {
        return State.fromValue(state);
    }

    /**
     * <p>
     * Specifies the condition of a server for the server that was described. A value of <code>ONLINE</code> indicates
     * that the server can accept jobs and transfer files. A <code>State</code> value of <code>OFFLINE</code> means that
     * the server cannot perform file transfer operations.
     * </p>
     * <p>
     * The states of <code>STARTING</code> and <code>STOPPING</code> indicate that the server is in an intermediate
     * state, either not fully able to respond, or not fully offline. The values of <code>START_FAILED</code> or
     * <code>STOP_FAILED</code> can indicate an error condition.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link State#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return Specifies the condition of a server for the server that was described. A value of <code>ONLINE</code>
     *         indicates that the server can accept jobs and transfer files. A <code>State</code> value of
     *         <code>OFFLINE</code> means that the server cannot perform file transfer operations.</p>
     *         <p>
     *         The states of <code>STARTING</code> and <code>STOPPING</code> indicate that the server is in an
     *         intermediate state, either not fully able to respond, or not fully offline. The values of
     *         <code>START_FAILED</code> or <code>STOP_FAILED</code> can indicate an error condition.
     * @see State
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the key-value pairs that you can use to search for and group servers that were assigned to the server
     * that was described.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Specifies the key-value pairs that you can use to search for and group servers that were assigned to the
     *         server that was described.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Specifies the number of users that are assigned to a server you specified with the <code>ServerId</code>.
     * </p>
     * 
     * @return Specifies the number of users that are assigned to a server you specified with the <code>ServerId</code>.
     */
    public final Integer userCount() {
        return userCount;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(certificate());
        hashCode = 31 * hashCode + Objects.hashCode(protocolDetails());
        hashCode = 31 * hashCode + Objects.hashCode(domainAsString());
        hashCode = 31 * hashCode + Objects.hashCode(endpointDetails());
        hashCode = 31 * hashCode + Objects.hashCode(endpointTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hostKeyFingerprint());
        hashCode = 31 * hashCode + Objects.hashCode(identityProviderDetails());
        hashCode = 31 * hashCode + Objects.hashCode(identityProviderTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(loggingRole());
        hashCode = 31 * hashCode + Objects.hashCode(hasProtocols() ? protocolsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(securityPolicyName());
        hashCode = 31 * hashCode + Objects.hashCode(serverId());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(userCount());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DescribedServer)) {
            return false;
        }
        DescribedServer other = (DescribedServer) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(certificate(), other.certificate())
                && Objects.equals(protocolDetails(), other.protocolDetails())
                && Objects.equals(domainAsString(), other.domainAsString())
                && Objects.equals(endpointDetails(), other.endpointDetails())
                && Objects.equals(endpointTypeAsString(), other.endpointTypeAsString())
                && Objects.equals(hostKeyFingerprint(), other.hostKeyFingerprint())
                && Objects.equals(identityProviderDetails(), other.identityProviderDetails())
                && Objects.equals(identityProviderTypeAsString(), other.identityProviderTypeAsString())
                && Objects.equals(loggingRole(), other.loggingRole()) && hasProtocols() == other.hasProtocols()
                && Objects.equals(protocolsAsStrings(), other.protocolsAsStrings())
                && Objects.equals(securityPolicyName(), other.securityPolicyName())
                && Objects.equals(serverId(), other.serverId()) && Objects.equals(stateAsString(), other.stateAsString())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(userCount(), other.userCount());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DescribedServer").add("Arn", arn()).add("Certificate", certificate())
                .add("ProtocolDetails", protocolDetails()).add("Domain", domainAsString())
                .add("EndpointDetails", endpointDetails()).add("EndpointType", endpointTypeAsString())
                .add("HostKeyFingerprint", hostKeyFingerprint()).add("IdentityProviderDetails", identityProviderDetails())
                .add("IdentityProviderType", identityProviderTypeAsString()).add("LoggingRole", loggingRole())
                .add("Protocols", hasProtocols() ? protocolsAsStrings() : null).add("SecurityPolicyName", securityPolicyName())
                .add("ServerId", serverId()).add("State", stateAsString()).add("Tags", hasTags() ? tags() : null)
                .add("UserCount", userCount()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Certificate":
            return Optional.ofNullable(clazz.cast(certificate()));
        case "ProtocolDetails":
            return Optional.ofNullable(clazz.cast(protocolDetails()));
        case "Domain":
            return Optional.ofNullable(clazz.cast(domainAsString()));
        case "EndpointDetails":
            return Optional.ofNullable(clazz.cast(endpointDetails()));
        case "EndpointType":
            return Optional.ofNullable(clazz.cast(endpointTypeAsString()));
        case "HostKeyFingerprint":
            return Optional.ofNullable(clazz.cast(hostKeyFingerprint()));
        case "IdentityProviderDetails":
            return Optional.ofNullable(clazz.cast(identityProviderDetails()));
        case "IdentityProviderType":
            return Optional.ofNullable(clazz.cast(identityProviderTypeAsString()));
        case "LoggingRole":
            return Optional.ofNullable(clazz.cast(loggingRole()));
        case "Protocols":
            return Optional.ofNullable(clazz.cast(protocolsAsStrings()));
        case "SecurityPolicyName":
            return Optional.ofNullable(clazz.cast(securityPolicyName()));
        case "ServerId":
            return Optional.ofNullable(clazz.cast(serverId()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "UserCount":
            return Optional.ofNullable(clazz.cast(userCount()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DescribedServer, T> g) {
        return obj -> g.apply((DescribedServer) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DescribedServer> {
        /**
         * <p>
         * Specifies the unique Amazon Resource Name (ARN) of the server.
         * </p>
         * 
         * @param arn
         *        Specifies the unique Amazon Resource Name (ARN) of the server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * Specifies the ARN of the Amazon Web ServicesCertificate Manager (ACM) certificate. Required when
         * <code>Protocols</code> is set to <code>FTPS</code>.
         * </p>
         * 
         * @param certificate
         *        Specifies the ARN of the Amazon Web ServicesCertificate Manager (ACM) certificate. Required when
         *        <code>Protocols</code> is set to <code>FTPS</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificate(String certificate);

        /**
         * <p>
         * The protocol settings that are configured for your server.
         * </p>
         * <p>
         * Use the <code>PassiveIp</code> parameter to indicate passive mode. Enter a single dotted-quad IPv4 address,
         * such as the external IP address of a firewall, router, or load balancer.
         * </p>
         * 
         * @param protocolDetails
         *        The protocol settings that are configured for your server. </p>
         *        <p>
         *        Use the <code>PassiveIp</code> parameter to indicate passive mode. Enter a single dotted-quad IPv4
         *        address, such as the external IP address of a firewall, router, or load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocolDetails(ProtocolDetails protocolDetails);

        /**
         * <p>
         * The protocol settings that are configured for your server.
         * </p>
         * <p>
         * Use the <code>PassiveIp</code> parameter to indicate passive mode. Enter a single dotted-quad IPv4 address,
         * such as the external IP address of a firewall, router, or load balancer.
         * </p>
         * This is a convenience that creates an instance of the {@link ProtocolDetails.Builder} avoiding the need to
         * create one manually via {@link ProtocolDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link ProtocolDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #protocolDetails(ProtocolDetails)}.
         * 
         * @param protocolDetails
         *        a consumer that will call methods on {@link ProtocolDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #protocolDetails(ProtocolDetails)
         */
        default Builder protocolDetails(Consumer<ProtocolDetails.Builder> protocolDetails) {
            return protocolDetails(ProtocolDetails.builder().applyMutation(protocolDetails).build());
        }

        /**
         * <p>
         * Specifies the domain of the storage system that is used for file transfers.
         * </p>
         * 
         * @param domain
         *        Specifies the domain of the storage system that is used for file transfers.
         * @see Domain
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Domain
         */
        Builder domain(String domain);

        /**
         * <p>
         * Specifies the domain of the storage system that is used for file transfers.
         * </p>
         * 
         * @param domain
         *        Specifies the domain of the storage system that is used for file transfers.
         * @see Domain
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Domain
         */
        Builder domain(Domain domain);

        /**
         * <p>
         * The virtual private cloud (VPC) endpoint settings that are configured for your server. When you host your
         * endpoint within your VPC, you can make it accessible only to resources within your VPC, or you can attach
         * Elastic IP addresses and make it accessible to clients over the internet. Your VPC's default security groups
         * are automatically assigned to your endpoint.
         * </p>
         * 
         * @param endpointDetails
         *        The virtual private cloud (VPC) endpoint settings that are configured for your server. When you host
         *        your endpoint within your VPC, you can make it accessible only to resources within your VPC, or you
         *        can attach Elastic IP addresses and make it accessible to clients over the internet. Your VPC's
         *        default security groups are automatically assigned to your endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointDetails(EndpointDetails endpointDetails);

        /**
         * <p>
         * The virtual private cloud (VPC) endpoint settings that are configured for your server. When you host your
         * endpoint within your VPC, you can make it accessible only to resources within your VPC, or you can attach
         * Elastic IP addresses and make it accessible to clients over the internet. Your VPC's default security groups
         * are automatically assigned to your endpoint.
         * </p>
         * This is a convenience that creates an instance of the {@link EndpointDetails.Builder} avoiding the need to
         * create one manually via {@link EndpointDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link EndpointDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #endpointDetails(EndpointDetails)}.
         * 
         * @param endpointDetails
         *        a consumer that will call methods on {@link EndpointDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #endpointDetails(EndpointDetails)
         */
        default Builder endpointDetails(Consumer<EndpointDetails.Builder> endpointDetails) {
            return endpointDetails(EndpointDetails.builder().applyMutation(endpointDetails).build());
        }

        /**
         * <p>
         * Defines the type of endpoint that your server is connected to. If your server is connected to a VPC endpoint,
         * your server isn't accessible over the public internet.
         * </p>
         * 
         * @param endpointType
         *        Defines the type of endpoint that your server is connected to. If your server is connected to a VPC
         *        endpoint, your server isn't accessible over the public internet.
         * @see EndpointType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EndpointType
         */
        Builder endpointType(String endpointType);

        /**
         * <p>
         * Defines the type of endpoint that your server is connected to. If your server is connected to a VPC endpoint,
         * your server isn't accessible over the public internet.
         * </p>
         * 
         * @param endpointType
         *        Defines the type of endpoint that your server is connected to. If your server is connected to a VPC
         *        endpoint, your server isn't accessible over the public internet.
         * @see EndpointType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EndpointType
         */
        Builder endpointType(EndpointType endpointType);

        /**
         * <p>
         * Specifies the Base64-encoded SHA256 fingerprint of the server's host key. This value is equivalent to the
         * output of the <code>ssh-keygen -l -f my-new-server-key</code> command.
         * </p>
         * 
         * @param hostKeyFingerprint
         *        Specifies the Base64-encoded SHA256 fingerprint of the server's host key. This value is equivalent to
         *        the output of the <code>ssh-keygen -l -f my-new-server-key</code> command.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hostKeyFingerprint(String hostKeyFingerprint);

        /**
         * <p>
         * Specifies information to call a customer-supplied authentication API. This field is not populated when the
         * <code>IdentityProviderType</code> of a server is <code>AWS_DIRECTORY_SERVICE</code> or
         * <code>SERVICE_MANAGED</code>.
         * </p>
         * 
         * @param identityProviderDetails
         *        Specifies information to call a customer-supplied authentication API. This field is not populated when
         *        the <code>IdentityProviderType</code> of a server is <code>AWS_DIRECTORY_SERVICE</code> or
         *        <code>SERVICE_MANAGED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identityProviderDetails(IdentityProviderDetails identityProviderDetails);

        /**
         * <p>
         * Specifies information to call a customer-supplied authentication API. This field is not populated when the
         * <code>IdentityProviderType</code> of a server is <code>AWS_DIRECTORY_SERVICE</code> or
         * <code>SERVICE_MANAGED</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link IdentityProviderDetails.Builder} avoiding the
         * need to create one manually via {@link IdentityProviderDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link IdentityProviderDetails.Builder#build()} is called immediately
         * and its result is passed to {@link #identityProviderDetails(IdentityProviderDetails)}.
         * 
         * @param identityProviderDetails
         *        a consumer that will call methods on {@link IdentityProviderDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #identityProviderDetails(IdentityProviderDetails)
         */
        default Builder identityProviderDetails(Consumer<IdentityProviderDetails.Builder> identityProviderDetails) {
            return identityProviderDetails(IdentityProviderDetails.builder().applyMutation(identityProviderDetails).build());
        }

        /**
         * <p>
         * Specifies the mode of authentication for a server. The default value is <code>SERVICE_MANAGED</code>, which
         * allows you to store and access user credentials within the Amazon Web Services Transfer Family service.
         * </p>
         * <p>
         * Use <code>AWS_DIRECTORY_SERVICE</code> to provide access to Active Directory groups in Amazon Web Services
         * Managed Active Directory or Microsoft Active Directory in your on-premises environment or in Amazon Web
         * Services using AD Connectors. This option also requires you to provide a Directory ID using the
         * <code>IdentityProviderDetails</code> parameter.
         * </p>
         * <p>
         * Use the <code>API_GATEWAY</code> value to integrate with an identity provider of your choosing. The
         * <code>API_GATEWAY</code> setting requires you to provide an API Gateway endpoint URL to call for
         * authentication using the <code>IdentityProviderDetails</code> parameter.
         * </p>
         * 
         * @param identityProviderType
         *        Specifies the mode of authentication for a server. The default value is <code>SERVICE_MANAGED</code>,
         *        which allows you to store and access user credentials within the Amazon Web Services Transfer Family
         *        service.</p>
         *        <p>
         *        Use <code>AWS_DIRECTORY_SERVICE</code> to provide access to Active Directory groups in Amazon Web
         *        Services Managed Active Directory or Microsoft Active Directory in your on-premises environment or in
         *        Amazon Web Services using AD Connectors. This option also requires you to provide a Directory ID using
         *        the <code>IdentityProviderDetails</code> parameter.
         *        </p>
         *        <p>
         *        Use the <code>API_GATEWAY</code> value to integrate with an identity provider of your choosing. The
         *        <code>API_GATEWAY</code> setting requires you to provide an API Gateway endpoint URL to call for
         *        authentication using the <code>IdentityProviderDetails</code> parameter.
         * @see IdentityProviderType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IdentityProviderType
         */
        Builder identityProviderType(String identityProviderType);

        /**
         * <p>
         * Specifies the mode of authentication for a server. The default value is <code>SERVICE_MANAGED</code>, which
         * allows you to store and access user credentials within the Amazon Web Services Transfer Family service.
         * </p>
         * <p>
         * Use <code>AWS_DIRECTORY_SERVICE</code> to provide access to Active Directory groups in Amazon Web Services
         * Managed Active Directory or Microsoft Active Directory in your on-premises environment or in Amazon Web
         * Services using AD Connectors. This option also requires you to provide a Directory ID using the
         * <code>IdentityProviderDetails</code> parameter.
         * </p>
         * <p>
         * Use the <code>API_GATEWAY</code> value to integrate with an identity provider of your choosing. The
         * <code>API_GATEWAY</code> setting requires you to provide an API Gateway endpoint URL to call for
         * authentication using the <code>IdentityProviderDetails</code> parameter.
         * </p>
         * 
         * @param identityProviderType
         *        Specifies the mode of authentication for a server. The default value is <code>SERVICE_MANAGED</code>,
         *        which allows you to store and access user credentials within the Amazon Web Services Transfer Family
         *        service.</p>
         *        <p>
         *        Use <code>AWS_DIRECTORY_SERVICE</code> to provide access to Active Directory groups in Amazon Web
         *        Services Managed Active Directory or Microsoft Active Directory in your on-premises environment or in
         *        Amazon Web Services using AD Connectors. This option also requires you to provide a Directory ID using
         *        the <code>IdentityProviderDetails</code> parameter.
         *        </p>
         *        <p>
         *        Use the <code>API_GATEWAY</code> value to integrate with an identity provider of your choosing. The
         *        <code>API_GATEWAY</code> setting requires you to provide an API Gateway endpoint URL to call for
         *        authentication using the <code>IdentityProviderDetails</code> parameter.
         * @see IdentityProviderType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IdentityProviderType
         */
        Builder identityProviderType(IdentityProviderType identityProviderType);

        /**
         * <p>
         * Specifies the Amazon Resource Name (ARN) of the Amazon Web Services Identity and Access Management (IAM) role
         * that allows a server to turn on Amazon CloudWatch logging for Amazon S3 or Amazon EFS events. When set, user
         * activity can be viewed in your CloudWatch logs.
         * </p>
         * 
         * @param loggingRole
         *        Specifies the Amazon Resource Name (ARN) of the Amazon Web Services Identity and Access Management
         *        (IAM) role that allows a server to turn on Amazon CloudWatch logging for Amazon S3 or Amazon EFS
         *        events. When set, user activity can be viewed in your CloudWatch logs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loggingRole(String loggingRole);

        /**
         * <p>
         * Specifies the file transfer protocol or protocols over which your file transfer protocol client can connect
         * to your server's endpoint. The available protocols are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
         * </p>
         * </li>
         * </ul>
         * 
         * @param protocols
         *        Specifies the file transfer protocol or protocols over which your file transfer protocol client can
         *        connect to your server's endpoint. The available protocols are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocolsWithStrings(Collection<String> protocols);

        /**
         * <p>
         * Specifies the file transfer protocol or protocols over which your file transfer protocol client can connect
         * to your server's endpoint. The available protocols are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
         * </p>
         * </li>
         * </ul>
         * 
         * @param protocols
         *        Specifies the file transfer protocol or protocols over which your file transfer protocol client can
         *        connect to your server's endpoint. The available protocols are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocolsWithStrings(String... protocols);

        /**
         * <p>
         * Specifies the file transfer protocol or protocols over which your file transfer protocol client can connect
         * to your server's endpoint. The available protocols are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
         * </p>
         * </li>
         * </ul>
         * 
         * @param protocols
         *        Specifies the file transfer protocol or protocols over which your file transfer protocol client can
         *        connect to your server's endpoint. The available protocols are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocols(Collection<Protocol> protocols);

        /**
         * <p>
         * Specifies the file transfer protocol or protocols over which your file transfer protocol client can connect
         * to your server's endpoint. The available protocols are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
         * </p>
         * </li>
         * </ul>
         * 
         * @param protocols
         *        Specifies the file transfer protocol or protocols over which your file transfer protocol client can
         *        connect to your server's endpoint. The available protocols are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SFTP</code> (Secure Shell (SSH) File Transfer Protocol): File transfer over SSH
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FTPS</code> (File Transfer Protocol Secure): File transfer with TLS encryption
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FTP</code> (File Transfer Protocol): Unencrypted file transfer
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocols(Protocol... protocols);

        /**
         * <p>
         * Specifies the name of the security policy that is attached to the server.
         * </p>
         * 
         * @param securityPolicyName
         *        Specifies the name of the security policy that is attached to the server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityPolicyName(String securityPolicyName);

        /**
         * <p>
         * Specifies the unique system-assigned identifier for a server that you instantiate.
         * </p>
         * 
         * @param serverId
         *        Specifies the unique system-assigned identifier for a server that you instantiate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverId(String serverId);

        /**
         * <p>
         * Specifies the condition of a server for the server that was described. A value of <code>ONLINE</code>
         * indicates that the server can accept jobs and transfer files. A <code>State</code> value of
         * <code>OFFLINE</code> means that the server cannot perform file transfer operations.
         * </p>
         * <p>
         * The states of <code>STARTING</code> and <code>STOPPING</code> indicate that the server is in an intermediate
         * state, either not fully able to respond, or not fully offline. The values of <code>START_FAILED</code> or
         * <code>STOP_FAILED</code> can indicate an error condition.
         * </p>
         * 
         * @param state
         *        Specifies the condition of a server for the server that was described. A value of <code>ONLINE</code>
         *        indicates that the server can accept jobs and transfer files. A <code>State</code> value of
         *        <code>OFFLINE</code> means that the server cannot perform file transfer operations.</p>
         *        <p>
         *        The states of <code>STARTING</code> and <code>STOPPING</code> indicate that the server is in an
         *        intermediate state, either not fully able to respond, or not fully offline. The values of
         *        <code>START_FAILED</code> or <code>STOP_FAILED</code> can indicate an error condition.
         * @see State
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see State
         */
        Builder state(String state);

        /**
         * <p>
         * Specifies the condition of a server for the server that was described. A value of <code>ONLINE</code>
         * indicates that the server can accept jobs and transfer files. A <code>State</code> value of
         * <code>OFFLINE</code> means that the server cannot perform file transfer operations.
         * </p>
         * <p>
         * The states of <code>STARTING</code> and <code>STOPPING</code> indicate that the server is in an intermediate
         * state, either not fully able to respond, or not fully offline. The values of <code>START_FAILED</code> or
         * <code>STOP_FAILED</code> can indicate an error condition.
         * </p>
         * 
         * @param state
         *        Specifies the condition of a server for the server that was described. A value of <code>ONLINE</code>
         *        indicates that the server can accept jobs and transfer files. A <code>State</code> value of
         *        <code>OFFLINE</code> means that the server cannot perform file transfer operations.</p>
         *        <p>
         *        The states of <code>STARTING</code> and <code>STOPPING</code> indicate that the server is in an
         *        intermediate state, either not fully able to respond, or not fully offline. The values of
         *        <code>START_FAILED</code> or <code>STOP_FAILED</code> can indicate an error condition.
         * @see State
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see State
         */
        Builder state(State state);

        /**
         * <p>
         * Specifies the key-value pairs that you can use to search for and group servers that were assigned to the
         * server that was described.
         * </p>
         * 
         * @param tags
         *        Specifies the key-value pairs that you can use to search for and group servers that were assigned to
         *        the server that was described.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Specifies the key-value pairs that you can use to search for and group servers that were assigned to the
         * server that was described.
         * </p>
         * 
         * @param tags
         *        Specifies the key-value pairs that you can use to search for and group servers that were assigned to
         *        the server that was described.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Specifies the key-value pairs that you can use to search for and group servers that were assigned to the
         * server that was described.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Specifies the number of users that are assigned to a server you specified with the <code>ServerId</code>.
         * </p>
         * 
         * @param userCount
         *        Specifies the number of users that are assigned to a server you specified with the
         *        <code>ServerId</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userCount(Integer userCount);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private String certificate;

        private ProtocolDetails protocolDetails;

        private String domain;

        private EndpointDetails endpointDetails;

        private String endpointType;

        private String hostKeyFingerprint;

        private IdentityProviderDetails identityProviderDetails;

        private String identityProviderType;

        private String loggingRole;

        private List<String> protocols = DefaultSdkAutoConstructList.getInstance();

        private String securityPolicyName;

        private String serverId;

        private String state;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private Integer userCount;

        private BuilderImpl() {
        }

        private BuilderImpl(DescribedServer model) {
            arn(model.arn);
            certificate(model.certificate);
            protocolDetails(model.protocolDetails);
            domain(model.domain);
            endpointDetails(model.endpointDetails);
            endpointType(model.endpointType);
            hostKeyFingerprint(model.hostKeyFingerprint);
            identityProviderDetails(model.identityProviderDetails);
            identityProviderType(model.identityProviderType);
            loggingRole(model.loggingRole);
            protocolsWithStrings(model.protocols);
            securityPolicyName(model.securityPolicyName);
            serverId(model.serverId);
            state(model.state);
            tags(model.tags);
            userCount(model.userCount);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        @Transient
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getCertificate() {
            return certificate;
        }

        public final void setCertificate(String certificate) {
            this.certificate = certificate;
        }

        @Override
        @Transient
        public final Builder certificate(String certificate) {
            this.certificate = certificate;
            return this;
        }

        public final ProtocolDetails.Builder getProtocolDetails() {
            return protocolDetails != null ? protocolDetails.toBuilder() : null;
        }

        public final void setProtocolDetails(ProtocolDetails.BuilderImpl protocolDetails) {
            this.protocolDetails = protocolDetails != null ? protocolDetails.build() : null;
        }

        @Override
        @Transient
        public final Builder protocolDetails(ProtocolDetails protocolDetails) {
            this.protocolDetails = protocolDetails;
            return this;
        }

        public final String getDomain() {
            return domain;
        }

        public final void setDomain(String domain) {
            this.domain = domain;
        }

        @Override
        @Transient
        public final Builder domain(String domain) {
            this.domain = domain;
            return this;
        }

        @Override
        @Transient
        public final Builder domain(Domain domain) {
            this.domain(domain == null ? null : domain.toString());
            return this;
        }

        public final EndpointDetails.Builder getEndpointDetails() {
            return endpointDetails != null ? endpointDetails.toBuilder() : null;
        }

        public final void setEndpointDetails(EndpointDetails.BuilderImpl endpointDetails) {
            this.endpointDetails = endpointDetails != null ? endpointDetails.build() : null;
        }

        @Override
        @Transient
        public final Builder endpointDetails(EndpointDetails endpointDetails) {
            this.endpointDetails = endpointDetails;
            return this;
        }

        public final String getEndpointType() {
            return endpointType;
        }

        public final void setEndpointType(String endpointType) {
            this.endpointType = endpointType;
        }

        @Override
        @Transient
        public final Builder endpointType(String endpointType) {
            this.endpointType = endpointType;
            return this;
        }

        @Override
        @Transient
        public final Builder endpointType(EndpointType endpointType) {
            this.endpointType(endpointType == null ? null : endpointType.toString());
            return this;
        }

        public final String getHostKeyFingerprint() {
            return hostKeyFingerprint;
        }

        public final void setHostKeyFingerprint(String hostKeyFingerprint) {
            this.hostKeyFingerprint = hostKeyFingerprint;
        }

        @Override
        @Transient
        public final Builder hostKeyFingerprint(String hostKeyFingerprint) {
            this.hostKeyFingerprint = hostKeyFingerprint;
            return this;
        }

        public final IdentityProviderDetails.Builder getIdentityProviderDetails() {
            return identityProviderDetails != null ? identityProviderDetails.toBuilder() : null;
        }

        public final void setIdentityProviderDetails(IdentityProviderDetails.BuilderImpl identityProviderDetails) {
            this.identityProviderDetails = identityProviderDetails != null ? identityProviderDetails.build() : null;
        }

        @Override
        @Transient
        public final Builder identityProviderDetails(IdentityProviderDetails identityProviderDetails) {
            this.identityProviderDetails = identityProviderDetails;
            return this;
        }

        public final String getIdentityProviderType() {
            return identityProviderType;
        }

        public final void setIdentityProviderType(String identityProviderType) {
            this.identityProviderType = identityProviderType;
        }

        @Override
        @Transient
        public final Builder identityProviderType(String identityProviderType) {
            this.identityProviderType = identityProviderType;
            return this;
        }

        @Override
        @Transient
        public final Builder identityProviderType(IdentityProviderType identityProviderType) {
            this.identityProviderType(identityProviderType == null ? null : identityProviderType.toString());
            return this;
        }

        public final String getLoggingRole() {
            return loggingRole;
        }

        public final void setLoggingRole(String loggingRole) {
            this.loggingRole = loggingRole;
        }

        @Override
        @Transient
        public final Builder loggingRole(String loggingRole) {
            this.loggingRole = loggingRole;
            return this;
        }

        public final Collection<String> getProtocols() {
            if (protocols instanceof SdkAutoConstructList) {
                return null;
            }
            return protocols;
        }

        public final void setProtocols(Collection<String> protocols) {
            this.protocols = ProtocolsCopier.copy(protocols);
        }

        @Override
        @Transient
        public final Builder protocolsWithStrings(Collection<String> protocols) {
            this.protocols = ProtocolsCopier.copy(protocols);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder protocolsWithStrings(String... protocols) {
            protocolsWithStrings(Arrays.asList(protocols));
            return this;
        }

        @Override
        @Transient
        public final Builder protocols(Collection<Protocol> protocols) {
            this.protocols = ProtocolsCopier.copyEnumToString(protocols);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder protocols(Protocol... protocols) {
            protocols(Arrays.asList(protocols));
            return this;
        }

        public final String getSecurityPolicyName() {
            return securityPolicyName;
        }

        public final void setSecurityPolicyName(String securityPolicyName) {
            this.securityPolicyName = securityPolicyName;
        }

        @Override
        @Transient
        public final Builder securityPolicyName(String securityPolicyName) {
            this.securityPolicyName = securityPolicyName;
            return this;
        }

        public final String getServerId() {
            return serverId;
        }

        public final void setServerId(String serverId) {
            this.serverId = serverId;
        }

        @Override
        @Transient
        public final Builder serverId(String serverId) {
            this.serverId = serverId;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        @Transient
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        @Transient
        public final Builder state(State state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        @Transient
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Integer getUserCount() {
            return userCount;
        }

        public final void setUserCount(Integer userCount) {
            this.userCount = userCount;
        }

        @Override
        @Transient
        public final Builder userCount(Integer userCount) {
            this.userCount = userCount;
            return this;
        }

        @Override
        public DescribedServer build() {
            return new DescribedServer(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
