/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transfer;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.transfer.model.AccessDeniedException;
import software.amazon.awssdk.services.transfer.model.ConflictException;
import software.amazon.awssdk.services.transfer.model.CreateAccessRequest;
import software.amazon.awssdk.services.transfer.model.CreateAccessResponse;
import software.amazon.awssdk.services.transfer.model.CreateServerRequest;
import software.amazon.awssdk.services.transfer.model.CreateServerResponse;
import software.amazon.awssdk.services.transfer.model.CreateUserRequest;
import software.amazon.awssdk.services.transfer.model.CreateUserResponse;
import software.amazon.awssdk.services.transfer.model.CreateWorkflowRequest;
import software.amazon.awssdk.services.transfer.model.CreateWorkflowResponse;
import software.amazon.awssdk.services.transfer.model.DeleteAccessRequest;
import software.amazon.awssdk.services.transfer.model.DeleteAccessResponse;
import software.amazon.awssdk.services.transfer.model.DeleteServerRequest;
import software.amazon.awssdk.services.transfer.model.DeleteServerResponse;
import software.amazon.awssdk.services.transfer.model.DeleteSshPublicKeyRequest;
import software.amazon.awssdk.services.transfer.model.DeleteSshPublicKeyResponse;
import software.amazon.awssdk.services.transfer.model.DeleteUserRequest;
import software.amazon.awssdk.services.transfer.model.DeleteUserResponse;
import software.amazon.awssdk.services.transfer.model.DeleteWorkflowRequest;
import software.amazon.awssdk.services.transfer.model.DeleteWorkflowResponse;
import software.amazon.awssdk.services.transfer.model.DescribeAccessRequest;
import software.amazon.awssdk.services.transfer.model.DescribeAccessResponse;
import software.amazon.awssdk.services.transfer.model.DescribeExecutionRequest;
import software.amazon.awssdk.services.transfer.model.DescribeExecutionResponse;
import software.amazon.awssdk.services.transfer.model.DescribeSecurityPolicyRequest;
import software.amazon.awssdk.services.transfer.model.DescribeSecurityPolicyResponse;
import software.amazon.awssdk.services.transfer.model.DescribeServerRequest;
import software.amazon.awssdk.services.transfer.model.DescribeServerResponse;
import software.amazon.awssdk.services.transfer.model.DescribeUserRequest;
import software.amazon.awssdk.services.transfer.model.DescribeUserResponse;
import software.amazon.awssdk.services.transfer.model.DescribeWorkflowRequest;
import software.amazon.awssdk.services.transfer.model.DescribeWorkflowResponse;
import software.amazon.awssdk.services.transfer.model.ImportSshPublicKeyRequest;
import software.amazon.awssdk.services.transfer.model.ImportSshPublicKeyResponse;
import software.amazon.awssdk.services.transfer.model.InternalServiceErrorException;
import software.amazon.awssdk.services.transfer.model.InvalidNextTokenException;
import software.amazon.awssdk.services.transfer.model.InvalidRequestException;
import software.amazon.awssdk.services.transfer.model.ListAccessesRequest;
import software.amazon.awssdk.services.transfer.model.ListAccessesResponse;
import software.amazon.awssdk.services.transfer.model.ListExecutionsRequest;
import software.amazon.awssdk.services.transfer.model.ListExecutionsResponse;
import software.amazon.awssdk.services.transfer.model.ListSecurityPoliciesRequest;
import software.amazon.awssdk.services.transfer.model.ListSecurityPoliciesResponse;
import software.amazon.awssdk.services.transfer.model.ListServersRequest;
import software.amazon.awssdk.services.transfer.model.ListServersResponse;
import software.amazon.awssdk.services.transfer.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.transfer.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.transfer.model.ListUsersRequest;
import software.amazon.awssdk.services.transfer.model.ListUsersResponse;
import software.amazon.awssdk.services.transfer.model.ListWorkflowsRequest;
import software.amazon.awssdk.services.transfer.model.ListWorkflowsResponse;
import software.amazon.awssdk.services.transfer.model.ResourceExistsException;
import software.amazon.awssdk.services.transfer.model.ResourceNotFoundException;
import software.amazon.awssdk.services.transfer.model.SendWorkflowStepStateRequest;
import software.amazon.awssdk.services.transfer.model.SendWorkflowStepStateResponse;
import software.amazon.awssdk.services.transfer.model.ServiceUnavailableException;
import software.amazon.awssdk.services.transfer.model.StartServerRequest;
import software.amazon.awssdk.services.transfer.model.StartServerResponse;
import software.amazon.awssdk.services.transfer.model.StopServerRequest;
import software.amazon.awssdk.services.transfer.model.StopServerResponse;
import software.amazon.awssdk.services.transfer.model.TagResourceRequest;
import software.amazon.awssdk.services.transfer.model.TagResourceResponse;
import software.amazon.awssdk.services.transfer.model.TestIdentityProviderRequest;
import software.amazon.awssdk.services.transfer.model.TestIdentityProviderResponse;
import software.amazon.awssdk.services.transfer.model.ThrottlingException;
import software.amazon.awssdk.services.transfer.model.TransferException;
import software.amazon.awssdk.services.transfer.model.TransferRequest;
import software.amazon.awssdk.services.transfer.model.UntagResourceRequest;
import software.amazon.awssdk.services.transfer.model.UntagResourceResponse;
import software.amazon.awssdk.services.transfer.model.UpdateAccessRequest;
import software.amazon.awssdk.services.transfer.model.UpdateAccessResponse;
import software.amazon.awssdk.services.transfer.model.UpdateServerRequest;
import software.amazon.awssdk.services.transfer.model.UpdateServerResponse;
import software.amazon.awssdk.services.transfer.model.UpdateUserRequest;
import software.amazon.awssdk.services.transfer.model.UpdateUserResponse;
import software.amazon.awssdk.services.transfer.paginators.ListAccessesPublisher;
import software.amazon.awssdk.services.transfer.paginators.ListExecutionsPublisher;
import software.amazon.awssdk.services.transfer.paginators.ListSecurityPoliciesPublisher;
import software.amazon.awssdk.services.transfer.paginators.ListServersPublisher;
import software.amazon.awssdk.services.transfer.paginators.ListTagsForResourcePublisher;
import software.amazon.awssdk.services.transfer.paginators.ListUsersPublisher;
import software.amazon.awssdk.services.transfer.paginators.ListWorkflowsPublisher;
import software.amazon.awssdk.services.transfer.transform.CreateAccessRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.CreateServerRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.CreateUserRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.CreateWorkflowRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.DeleteAccessRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.DeleteServerRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.DeleteSshPublicKeyRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.DeleteUserRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.DeleteWorkflowRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.DescribeAccessRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.DescribeExecutionRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.DescribeSecurityPolicyRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.DescribeServerRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.DescribeUserRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.DescribeWorkflowRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.ImportSshPublicKeyRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.ListAccessesRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.ListExecutionsRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.ListSecurityPoliciesRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.ListServersRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.ListUsersRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.ListWorkflowsRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.SendWorkflowStepStateRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.StartServerRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.StopServerRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.TestIdentityProviderRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.UpdateAccessRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.UpdateServerRequestMarshaller;
import software.amazon.awssdk.services.transfer.transform.UpdateUserRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link TransferAsyncClient}.
 *
 * @see TransferAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTransferAsyncClient implements TransferAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultTransferAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultTransferAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Used by administrators to choose which groups in the directory should have access to upload and download files
     * over the enabled protocols using Amazon Web Services Transfer Family. For example, a Microsoft Active Directory
     * might contain 50,000 users, but only a small fraction might need the ability to transfer files to the server. An
     * administrator can use <code>CreateAccess</code> to limit the access to the correct set of users who need this
     * ability.
     * </p>
     *
     * @param createAccessRequest
     * @return A Java Future containing the result of the CreateAccess operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceExistsException The requested resource does not exist.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.CreateAccess
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/CreateAccess" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAccessResponse> createAccess(CreateAccessRequest createAccessRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAccessRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAccess");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAccessResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateAccessResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAccessResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAccessRequest, CreateAccessResponse>()
                            .withOperationName("CreateAccess").withMarshaller(new CreateAccessRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createAccessRequest));
            CompletableFuture<CreateAccessResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Instantiates an auto-scaling virtual server based on the selected file transfer protocol in Amazon Web Services.
     * When you make updates to your file transfer protocol-enabled server or when you work with users, use the
     * service-generated <code>ServerId</code> property that is assigned to the newly created server.
     * </p>
     *
     * @param createServerRequest
     * @return A Java Future containing the result of the CreateServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceExistsException The requested resource does not exist.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.CreateServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/CreateServer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateServerResponse> createServer(CreateServerRequest createServerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createServerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateServer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateServerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateServerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateServerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateServerRequest, CreateServerResponse>()
                            .withOperationName("CreateServer").withMarshaller(new CreateServerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createServerRequest));
            CompletableFuture<CreateServerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a user and associates them with an existing file transfer protocol-enabled server. You can only create
     * and associate users with servers that have the <code>IdentityProviderType</code> set to
     * <code>SERVICE_MANAGED</code>. Using parameters for <code>CreateUser</code>, you can specify the user name, set
     * the home directory, store the user's public key, and assign the user's Amazon Web Services Identity and Access
     * Management (IAM) role. You can also optionally add a session policy, and assign metadata with tags that can be
     * used to group and search for users.
     * </p>
     *
     * @param createUserRequest
     * @return A Java Future containing the result of the CreateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceExistsException The requested resource does not exist.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.CreateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/CreateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateUserResponse> createUser(CreateUserRequest createUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateUserRequest, CreateUserResponse>().withOperationName("CreateUser")
                            .withMarshaller(new CreateUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createUserRequest));
            CompletableFuture<CreateUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows you to create a workflow with specified steps and step details the workflow invokes after file transfer
     * completes. After creating a workflow, you can associate the workflow created with any transfer servers by
     * specifying the <code>workflow-details</code> field in <code>CreateServer</code> and <code>UpdateServer</code>
     * operations.
     * </p>
     *
     * @param createWorkflowRequest
     * @return A Java Future containing the result of the CreateWorkflow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceExistsException The requested resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.CreateWorkflow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/CreateWorkflow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWorkflowResponse> createWorkflow(CreateWorkflowRequest createWorkflowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWorkflowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorkflow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWorkflowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateWorkflowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateWorkflowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWorkflowRequest, CreateWorkflowResponse>()
                            .withOperationName("CreateWorkflow")
                            .withMarshaller(new CreateWorkflowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createWorkflowRequest));
            CompletableFuture<CreateWorkflowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows you to delete the access specified in the <code>ServerID</code> and <code>ExternalID</code> parameters.
     * </p>
     *
     * @param deleteAccessRequest
     * @return A Java Future containing the result of the DeleteAccess operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DeleteAccess
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DeleteAccess" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAccessResponse> deleteAccess(DeleteAccessRequest deleteAccessRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAccessRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAccess");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAccessResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteAccessResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAccessResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAccessRequest, DeleteAccessResponse>()
                            .withOperationName("DeleteAccess").withMarshaller(new DeleteAccessRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteAccessRequest));
            CompletableFuture<DeleteAccessResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the file transfer protocol-enabled server that you specify.
     * </p>
     * <p>
     * No response returns from this operation.
     * </p>
     *
     * @param deleteServerRequest
     * @return A Java Future containing the result of the DeleteServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DeleteServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DeleteServer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteServerResponse> deleteServer(DeleteServerRequest deleteServerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteServerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteServer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteServerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteServerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteServerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteServerRequest, DeleteServerResponse>()
                            .withOperationName("DeleteServer").withMarshaller(new DeleteServerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteServerRequest));
            CompletableFuture<DeleteServerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a user's Secure Shell (SSH) public key.
     * </p>
     *
     * @param deleteSshPublicKeyRequest
     * @return A Java Future containing the result of the DeleteSshPublicKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DeleteSshPublicKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DeleteSshPublicKey" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSshPublicKeyResponse> deleteSshPublicKey(DeleteSshPublicKeyRequest deleteSshPublicKeyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSshPublicKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSshPublicKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSshPublicKeyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSshPublicKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSshPublicKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSshPublicKeyRequest, DeleteSshPublicKeyResponse>()
                            .withOperationName("DeleteSshPublicKey")
                            .withMarshaller(new DeleteSshPublicKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteSshPublicKeyRequest));
            CompletableFuture<DeleteSshPublicKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the user belonging to a file transfer protocol-enabled server you specify.
     * </p>
     * <p>
     * No response returns from this operation.
     * </p>
     * <note>
     * <p>
     * When you delete a user from a server, the user's information is lost.
     * </p>
     * </note>
     *
     * @param deleteUserRequest
     * @return A Java Future containing the result of the DeleteUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DeleteUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DeleteUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteUserResponse> deleteUser(DeleteUserRequest deleteUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteUserRequest, DeleteUserResponse>().withOperationName("DeleteUser")
                            .withMarshaller(new DeleteUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteUserRequest));
            CompletableFuture<DeleteUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified workflow.
     * </p>
     *
     * @param deleteWorkflowRequest
     * @return A Java Future containing the result of the DeleteWorkflow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DeleteWorkflow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DeleteWorkflow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWorkflowResponse> deleteWorkflow(DeleteWorkflowRequest deleteWorkflowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWorkflowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorkflow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWorkflowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteWorkflowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteWorkflowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWorkflowRequest, DeleteWorkflowResponse>()
                            .withOperationName("DeleteWorkflow")
                            .withMarshaller(new DeleteWorkflowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteWorkflowRequest));
            CompletableFuture<DeleteWorkflowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the access that is assigned to the specific file transfer protocol-enabled server, as identified by its
     * <code>ServerId</code> property and its <code>ExternalID</code>.
     * </p>
     * <p>
     * The response from this call returns the properties of the access that is associated with the
     * <code>ServerId</code> value that was specified.
     * </p>
     *
     * @param describeAccessRequest
     * @return A Java Future containing the result of the DescribeAccess operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DescribeAccess
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DescribeAccess" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAccessResponse> describeAccess(DescribeAccessRequest describeAccessRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAccessRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAccess");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAccessResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAccessResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAccessResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAccessRequest, DescribeAccessResponse>()
                            .withOperationName("DescribeAccess")
                            .withMarshaller(new DescribeAccessRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeAccessRequest));
            CompletableFuture<DescribeAccessResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * You can use <code>DescribeExecution</code> to check the details of the execution of the specified workflow.
     * </p>
     *
     * @param describeExecutionRequest
     * @return A Java Future containing the result of the DescribeExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DescribeExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DescribeExecution" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeExecutionResponse> describeExecution(DescribeExecutionRequest describeExecutionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeExecutionRequest, DescribeExecutionResponse>()
                            .withOperationName("DescribeExecution")
                            .withMarshaller(new DescribeExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeExecutionRequest));
            CompletableFuture<DescribeExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the security policy that is attached to your file transfer protocol-enabled server. The response
     * contains a description of the security policy's properties. For more information about security policies, see <a
     * href="https://docs.aws.amazon.com/transfer/latest/userguide/security-policies.html">Working with security
     * policies</a>.
     * </p>
     *
     * @param describeSecurityPolicyRequest
     * @return A Java Future containing the result of the DescribeSecurityPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DescribeSecurityPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DescribeSecurityPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSecurityPolicyResponse> describeSecurityPolicy(
            DescribeSecurityPolicyRequest describeSecurityPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSecurityPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSecurityPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSecurityPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSecurityPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSecurityPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSecurityPolicyRequest, DescribeSecurityPolicyResponse>()
                            .withOperationName("DescribeSecurityPolicy")
                            .withMarshaller(new DescribeSecurityPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeSecurityPolicyRequest));
            CompletableFuture<DescribeSecurityPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a file transfer protocol-enabled server that you specify by passing the <code>ServerId</code>
     * parameter.
     * </p>
     * <p>
     * The response contains a description of a server's properties. When you set <code>EndpointType</code> to VPC, the
     * response will contain the <code>EndpointDetails</code>.
     * </p>
     *
     * @param describeServerRequest
     * @return A Java Future containing the result of the DescribeServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DescribeServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DescribeServer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeServerResponse> describeServer(DescribeServerRequest describeServerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeServerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeServer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeServerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeServerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeServerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeServerRequest, DescribeServerResponse>()
                            .withOperationName("DescribeServer")
                            .withMarshaller(new DescribeServerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeServerRequest));
            CompletableFuture<DescribeServerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the user assigned to the specific file transfer protocol-enabled server, as identified by its
     * <code>ServerId</code> property.
     * </p>
     * <p>
     * The response from this call returns the properties of the user associated with the <code>ServerId</code> value
     * that was specified.
     * </p>
     *
     * @param describeUserRequest
     * @return A Java Future containing the result of the DescribeUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DescribeUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DescribeUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeUserResponse> describeUser(DescribeUserRequest describeUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeUserRequest, DescribeUserResponse>()
                            .withOperationName("DescribeUser").withMarshaller(new DescribeUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeUserRequest));
            CompletableFuture<DescribeUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the specified workflow.
     * </p>
     *
     * @param describeWorkflowRequest
     * @return A Java Future containing the result of the DescribeWorkflow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.DescribeWorkflow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/DescribeWorkflow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeWorkflowResponse> describeWorkflow(DescribeWorkflowRequest describeWorkflowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeWorkflowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeWorkflow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeWorkflowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeWorkflowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeWorkflowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeWorkflowRequest, DescribeWorkflowResponse>()
                            .withOperationName("DescribeWorkflow")
                            .withMarshaller(new DescribeWorkflowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeWorkflowRequest));
            CompletableFuture<DescribeWorkflowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a Secure Shell (SSH) public key to a user account identified by a <code>UserName</code> value assigned to
     * the specific file transfer protocol-enabled server, identified by <code>ServerId</code>.
     * </p>
     * <p>
     * The response returns the <code>UserName</code> value, the <code>ServerId</code> value, and the name of the
     * <code>SshPublicKeyId</code>.
     * </p>
     *
     * @param importSshPublicKeyRequest
     * @return A Java Future containing the result of the ImportSshPublicKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceExistsException The requested resource does not exist.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ImportSshPublicKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ImportSshPublicKey" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ImportSshPublicKeyResponse> importSshPublicKey(ImportSshPublicKeyRequest importSshPublicKeyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, importSshPublicKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportSshPublicKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ImportSshPublicKeyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ImportSshPublicKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ImportSshPublicKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ImportSshPublicKeyRequest, ImportSshPublicKeyResponse>()
                            .withOperationName("ImportSshPublicKey")
                            .withMarshaller(new ImportSshPublicKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(importSshPublicKeyRequest));
            CompletableFuture<ImportSshPublicKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the details for all the accesses you have on your server.
     * </p>
     *
     * @param listAccessesRequest
     * @return A Java Future containing the result of the ListAccesses operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListAccesses
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListAccesses" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAccessesResponse> listAccesses(ListAccessesRequest listAccessesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAccessesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAccesses");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAccessesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAccessesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAccessesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAccessesRequest, ListAccessesResponse>()
                            .withOperationName("ListAccesses").withMarshaller(new ListAccessesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAccessesRequest));
            CompletableFuture<ListAccessesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the details for all the accesses you have on your server.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listAccesses(software.amazon.awssdk.services.transfer.model.ListAccessesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListAccessesPublisher publisher = client.listAccessesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListAccessesPublisher publisher = client.listAccessesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transfer.model.ListAccessesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transfer.model.ListAccessesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAccesses(software.amazon.awssdk.services.transfer.model.ListAccessesRequest)} operation.</b>
     * </p>
     *
     * @param listAccessesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListAccesses
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListAccesses" target="_top">AWS API
     *      Documentation</a>
     */
    public ListAccessesPublisher listAccessesPaginator(ListAccessesRequest listAccessesRequest) {
        return new ListAccessesPublisher(this, applyPaginatorUserAgent(listAccessesRequest));
    }

    /**
     * <p>
     * Lists all executions for the specified workflow.
     * </p>
     *
     * @param listExecutionsRequest
     * @return A Java Future containing the result of the ListExecutions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListExecutions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListExecutionsResponse> listExecutions(ListExecutionsRequest listExecutionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExecutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExecutions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListExecutionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListExecutionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListExecutionsRequest, ListExecutionsResponse>()
                            .withOperationName("ListExecutions")
                            .withMarshaller(new ListExecutionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listExecutionsRequest));
            CompletableFuture<ListExecutionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all executions for the specified workflow.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listExecutions(software.amazon.awssdk.services.transfer.model.ListExecutionsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListExecutionsPublisher publisher = client.listExecutionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListExecutionsPublisher publisher = client.listExecutionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transfer.model.ListExecutionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transfer.model.ListExecutionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listExecutions(software.amazon.awssdk.services.transfer.model.ListExecutionsRequest)} operation.</b>
     * </p>
     *
     * @param listExecutionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListExecutions" target="_top">AWS API
     *      Documentation</a>
     */
    public ListExecutionsPublisher listExecutionsPaginator(ListExecutionsRequest listExecutionsRequest) {
        return new ListExecutionsPublisher(this, applyPaginatorUserAgent(listExecutionsRequest));
    }

    /**
     * <p>
     * Lists the security policies that are attached to your file transfer protocol-enabled servers.
     * </p>
     *
     * @param listSecurityPoliciesRequest
     * @return A Java Future containing the result of the ListSecurityPolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListSecurityPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListSecurityPolicies" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSecurityPoliciesResponse> listSecurityPolicies(
            ListSecurityPoliciesRequest listSecurityPoliciesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSecurityPoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSecurityPolicies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSecurityPoliciesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSecurityPoliciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSecurityPoliciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSecurityPoliciesRequest, ListSecurityPoliciesResponse>()
                            .withOperationName("ListSecurityPolicies")
                            .withMarshaller(new ListSecurityPoliciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listSecurityPoliciesRequest));
            CompletableFuture<ListSecurityPoliciesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the security policies that are attached to your file transfer protocol-enabled servers.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSecurityPolicies(software.amazon.awssdk.services.transfer.model.ListSecurityPoliciesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListSecurityPoliciesPublisher publisher = client.listSecurityPoliciesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListSecurityPoliciesPublisher publisher = client.listSecurityPoliciesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transfer.model.ListSecurityPoliciesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transfer.model.ListSecurityPoliciesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSecurityPolicies(software.amazon.awssdk.services.transfer.model.ListSecurityPoliciesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSecurityPoliciesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListSecurityPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListSecurityPolicies" target="_top">AWS
     *      API Documentation</a>
     */
    public ListSecurityPoliciesPublisher listSecurityPoliciesPaginator(ListSecurityPoliciesRequest listSecurityPoliciesRequest) {
        return new ListSecurityPoliciesPublisher(this, applyPaginatorUserAgent(listSecurityPoliciesRequest));
    }

    /**
     * <p>
     * Lists the file transfer protocol-enabled servers that are associated with your Amazon Web Services account.
     * </p>
     *
     * @param listServersRequest
     * @return A Java Future containing the result of the ListServers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListServers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListServers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListServersResponse> listServers(ListServersRequest listServersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListServersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListServersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListServersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListServersRequest, ListServersResponse>()
                            .withOperationName("ListServers").withMarshaller(new ListServersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listServersRequest));
            CompletableFuture<ListServersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the file transfer protocol-enabled servers that are associated with your Amazon Web Services account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listServers(software.amazon.awssdk.services.transfer.model.ListServersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListServersPublisher publisher = client.listServersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListServersPublisher publisher = client.listServersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transfer.model.ListServersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transfer.model.ListServersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listServers(software.amazon.awssdk.services.transfer.model.ListServersRequest)} operation.</b>
     * </p>
     *
     * @param listServersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListServers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListServers" target="_top">AWS API
     *      Documentation</a>
     */
    public ListServersPublisher listServersPaginator(ListServersRequest listServersRequest) {
        return new ListServersPublisher(this, applyPaginatorUserAgent(listServersRequest));
    }

    /**
     * <p>
     * Lists all of the tags associated with the Amazon Resource Name (ARN) that you specify. The resource can be a
     * user, server, or role.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all of the tags associated with the Amazon Resource Name (ARN) that you specify. The resource can be a
     * user, server, or role.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTagsForResource(software.amazon.awssdk.services.transfer.model.ListTagsForResourceRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListTagsForResourcePublisher publisher = client.listTagsForResourcePaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListTagsForResourcePublisher publisher = client.listTagsForResourcePaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transfer.model.ListTagsForResourceResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transfer.model.ListTagsForResourceResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTagsForResource(software.amazon.awssdk.services.transfer.model.ListTagsForResourceRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    public ListTagsForResourcePublisher listTagsForResourcePaginator(ListTagsForResourceRequest listTagsForResourceRequest) {
        return new ListTagsForResourcePublisher(this, applyPaginatorUserAgent(listTagsForResourceRequest));
    }

    /**
     * <p>
     * Lists the users for a file transfer protocol-enabled server that you specify by passing the <code>ServerId</code>
     * parameter.
     * </p>
     *
     * @param listUsersRequest
     * @return A Java Future containing the result of the ListUsers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListUsersResponse> listUsers(ListUsersRequest listUsersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListUsers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListUsersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListUsersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListUsersRequest, ListUsersResponse>().withOperationName("ListUsers")
                            .withMarshaller(new ListUsersRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listUsersRequest));
            CompletableFuture<ListUsersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the users for a file transfer protocol-enabled server that you specify by passing the <code>ServerId</code>
     * parameter.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listUsers(software.amazon.awssdk.services.transfer.model.ListUsersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListUsersPublisher publisher = client.listUsersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListUsersPublisher publisher = client.listUsersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transfer.model.ListUsersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transfer.model.ListUsersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listUsers(software.amazon.awssdk.services.transfer.model.ListUsersRequest)} operation.</b>
     * </p>
     *
     * @param listUsersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    public ListUsersPublisher listUsersPaginator(ListUsersRequest listUsersRequest) {
        return new ListUsersPublisher(this, applyPaginatorUserAgent(listUsersRequest));
    }

    /**
     * <p>
     * Lists all of your workflows.
     * </p>
     *
     * @param listWorkflowsRequest
     * @return A Java Future containing the result of the ListWorkflows operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListWorkflows
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListWorkflows" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListWorkflowsResponse> listWorkflows(ListWorkflowsRequest listWorkflowsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkflowsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkflows");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWorkflowsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListWorkflowsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWorkflowsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWorkflowsRequest, ListWorkflowsResponse>()
                            .withOperationName("ListWorkflows")
                            .withMarshaller(new ListWorkflowsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listWorkflowsRequest));
            CompletableFuture<ListWorkflowsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all of your workflows.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listWorkflows(software.amazon.awssdk.services.transfer.model.ListWorkflowsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListWorkflowsPublisher publisher = client.listWorkflowsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.transfer.paginators.ListWorkflowsPublisher publisher = client.listWorkflowsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.transfer.model.ListWorkflowsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.transfer.model.ListWorkflowsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listWorkflows(software.amazon.awssdk.services.transfer.model.ListWorkflowsRequest)} operation.</b>
     * </p>
     *
     * @param listWorkflowsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> parameter that was passed is invalid.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.ListWorkflows
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/ListWorkflows" target="_top">AWS API
     *      Documentation</a>
     */
    public ListWorkflowsPublisher listWorkflowsPaginator(ListWorkflowsRequest listWorkflowsRequest) {
        return new ListWorkflowsPublisher(this, applyPaginatorUserAgent(listWorkflowsRequest));
    }

    /**
     * <p>
     * Sends a callback for asynchronous custom steps.
     * </p>
     * <p>
     * The <code>ExecutionId</code>, <code>WorkflowId</code>, and <code>Token</code> are passed to the target resource
     * during execution of a custom step of a workflow. You must include those with their callback as well as providing
     * a status.
     * </p>
     *
     * @param sendWorkflowStepStateRequest
     * @return A Java Future containing the result of the SendWorkflowStepState operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.SendWorkflowStepState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/SendWorkflowStepState"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SendWorkflowStepStateResponse> sendWorkflowStepState(
            SendWorkflowStepStateRequest sendWorkflowStepStateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendWorkflowStepStateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendWorkflowStepState");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendWorkflowStepStateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SendWorkflowStepStateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendWorkflowStepStateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendWorkflowStepStateRequest, SendWorkflowStepStateResponse>()
                            .withOperationName("SendWorkflowStepState")
                            .withMarshaller(new SendWorkflowStepStateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(sendWorkflowStepStateRequest));
            CompletableFuture<SendWorkflowStepStateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Changes the state of a file transfer protocol-enabled server from <code>OFFLINE</code> to <code>ONLINE</code>. It
     * has no impact on a server that is already <code>ONLINE</code>. An <code>ONLINE</code> server can accept and
     * process file transfer jobs.
     * </p>
     * <p>
     * The state of <code>STARTING</code> indicates that the server is in an intermediate state, either not fully able
     * to respond, or not fully online. The values of <code>START_FAILED</code> can indicate an error condition.
     * </p>
     * <p>
     * No response is returned from this call.
     * </p>
     *
     * @param startServerRequest
     * @return A Java Future containing the result of the StartServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.StartServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/StartServer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartServerResponse> startServer(StartServerRequest startServerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startServerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartServer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartServerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartServerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartServerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartServerRequest, StartServerResponse>()
                            .withOperationName("StartServer").withMarshaller(new StartServerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startServerRequest));
            CompletableFuture<StartServerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Changes the state of a file transfer protocol-enabled server from <code>ONLINE</code> to <code>OFFLINE</code>. An
     * <code>OFFLINE</code> server cannot accept and process file transfer jobs. Information tied to your server, such
     * as server and user properties, are not affected by stopping your server.
     * </p>
     * <note>
     * <p>
     * Stopping the server will not reduce or impact your file transfer protocol endpoint billing; you must delete the
     * server to stop being billed.
     * </p>
     * </note>
     * <p>
     * The state of <code>STOPPING</code> indicates that the server is in an intermediate state, either not fully able
     * to respond, or not fully offline. The values of <code>STOP_FAILED</code> can indicate an error condition.
     * </p>
     * <p>
     * No response is returned from this call.
     * </p>
     *
     * @param stopServerRequest
     * @return A Java Future containing the result of the StopServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.StopServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/StopServer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopServerResponse> stopServer(StopServerRequest stopServerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopServerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopServer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopServerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopServerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopServerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopServerRequest, StopServerResponse>().withOperationName("StopServer")
                            .withMarshaller(new StopServerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopServerRequest));
            CompletableFuture<StopServerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Attaches a key-value pair to a resource, as identified by its Amazon Resource Name (ARN). Resources are users,
     * servers, roles, and other entities.
     * </p>
     * <p>
     * There is no response returned from this call.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * If the <code>IdentityProviderType</code> of a file transfer protocol-enabled server is
     * <code>AWS_DIRECTORY_SERVICE</code> or <code>API_Gateway</code>, tests whether your identity provider is set up
     * successfully. We highly recommend that you call this operation to test your authentication method as soon as you
     * create your server. By doing so, you can troubleshoot issues with the identity provider integration to ensure
     * that your users can successfully use the service.
     * </p>
     * <p>
     * The <code>ServerId</code> and <code>UserName</code> parameters are required. The <code>ServerProtocol</code>,
     * <code>SourceIp</code>, and <code>UserPassword</code> are all optional.
     * </p>
     * <note>
     * <p>
     * You cannot use <code>TestIdentityProvider</code> if the <code>IdentityProviderType</code> of your server is
     * <code>SERVICE_MANAGED</code>.
     * </p>
     * </note>
     * <ul>
     * <li>
     * <p>
     * If you provide any incorrect values for any parameters, the <code>Response</code> field is empty.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you provide a server ID for a server that uses service-managed users, you get an error:
     * </p>
     * <p>
     * <code> An error occurred (InvalidRequestException) when calling the TestIdentityProvider operation: s-<i>server-ID</i> not configured for external auth </code>
     * </p>
     * </li>
     * <li>
     * <p>
     * If you enter a Server ID for the <code>--server-id</code> parameter that does not identify an actual Transfer
     * server, you receive the following error:
     * </p>
     * <p>
     * <code>An error occurred (ResourceNotFoundException) when calling the TestIdentityProvider operation: Unknown server</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param testIdentityProviderRequest
     * @return A Java Future containing the result of the TestIdentityProvider operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.TestIdentityProvider
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/TestIdentityProvider" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TestIdentityProviderResponse> testIdentityProvider(
            TestIdentityProviderRequest testIdentityProviderRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, testIdentityProviderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TestIdentityProvider");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TestIdentityProviderResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, TestIdentityProviderResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TestIdentityProviderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TestIdentityProviderRequest, TestIdentityProviderResponse>()
                            .withOperationName("TestIdentityProvider")
                            .withMarshaller(new TestIdentityProviderRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(testIdentityProviderRequest));
            CompletableFuture<TestIdentityProviderResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Detaches a key-value pair from a resource, as identified by its Amazon Resource Name (ARN). Resources are users,
     * servers, roles, and other entities.
     * </p>
     * <p>
     * No response is returned from this call.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows you to update parameters for the access specified in the <code>ServerID</code> and <code>ExternalID</code>
     * parameters.
     * </p>
     *
     * @param updateAccessRequest
     * @return A Java Future containing the result of the UpdateAccess operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceExistsException The requested resource does not exist.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.UpdateAccess
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/UpdateAccess" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAccessResponse> updateAccess(UpdateAccessRequest updateAccessRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAccessRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAccess");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAccessResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateAccessResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAccessResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAccessRequest, UpdateAccessResponse>()
                            .withOperationName("UpdateAccess").withMarshaller(new UpdateAccessRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateAccessRequest));
            CompletableFuture<UpdateAccessResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the file transfer protocol-enabled server's properties after that server has been created.
     * </p>
     * <p>
     * The <code>UpdateServer</code> call returns the <code>ServerId</code> of the server you updated.
     * </p>
     *
     * @param updateServerRequest
     * @return A Java Future containing the result of the UpdateServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>ConflictException This exception is thrown when the <code>UpdateServer</code> is called for a file
     *         transfer protocol-enabled server that has VPC as the endpoint type and the server's
     *         <code>VpcEndpointID</code> is not in the available state.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceExistsException The requested resource does not exist.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.UpdateServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/UpdateServer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateServerResponse> updateServer(UpdateServerRequest updateServerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateServer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateServerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateServerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateServerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateServerRequest, UpdateServerResponse>()
                            .withOperationName("UpdateServer").withMarshaller(new UpdateServerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateServerRequest));
            CompletableFuture<UpdateServerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns new properties to a user. Parameters you pass modify any or all of the following: the home directory,
     * role, and policy for the <code>UserName</code> and <code>ServerId</code> you specify.
     * </p>
     * <p>
     * The response returns the <code>ServerId</code> and the <code>UserName</code> for the updated user.
     * </p>
     *
     * @param updateUserRequest
     * @return A Java Future containing the result of the UpdateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The request has failed because the Amazon Web ServicesTransfer Family
     *         service is not available.</li>
     *         <li>InternalServiceErrorException This exception is thrown when an error occurs in the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource is not found by the Amazon Web
     *         ServicesTransfer Family service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TransferException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TransferAsyncClient.UpdateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/transfer-2018-11-05/UpdateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateUserResponse> updateUser(UpdateUserRequest updateUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Transfer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateUserRequest, UpdateUserResponse>().withOperationName("UpdateUser")
                            .withMarshaller(new UpdateUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateUserRequest));
            CompletableFuture<UpdateUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(TransferException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceError")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceExistsException")
                                .exceptionBuilderSupplier(ResourceExistsException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends TransferRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
