/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transfer.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateAgreementRequest extends TransferRequest implements
        ToCopyableBuilder<CreateAgreementRequest.Builder, CreateAgreementRequest> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(CreateAgreementRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> SERVER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServerId").getter(getter(CreateAgreementRequest::serverId)).setter(setter(Builder::serverId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerId").build()).build();

    private static final SdkField<String> LOCAL_PROFILE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LocalProfileId").getter(getter(CreateAgreementRequest::localProfileId))
            .setter(setter(Builder::localProfileId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LocalProfileId").build()).build();

    private static final SdkField<String> PARTNER_PROFILE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PartnerProfileId").getter(getter(CreateAgreementRequest::partnerProfileId))
            .setter(setter(Builder::partnerProfileId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PartnerProfileId").build()).build();

    private static final SdkField<String> BASE_DIRECTORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BaseDirectory").getter(getter(CreateAgreementRequest::baseDirectory))
            .setter(setter(Builder::baseDirectory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BaseDirectory").build()).build();

    private static final SdkField<String> ACCESS_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccessRole").getter(getter(CreateAgreementRequest::accessRole)).setter(setter(Builder::accessRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessRole").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(CreateAgreementRequest::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateAgreementRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> PRESERVE_FILENAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PreserveFilename").getter(getter(CreateAgreementRequest::preserveFilenameAsString))
            .setter(setter(Builder::preserveFilename))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreserveFilename").build()).build();

    private static final SdkField<String> ENFORCE_MESSAGE_SIGNING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EnforceMessageSigning").getter(getter(CreateAgreementRequest::enforceMessageSigningAsString))
            .setter(setter(Builder::enforceMessageSigning))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnforceMessageSigning").build())
            .build();

    private static final SdkField<CustomDirectoriesType> CUSTOM_DIRECTORIES_FIELD = SdkField
            .<CustomDirectoriesType> builder(MarshallingType.SDK_POJO).memberName("CustomDirectories")
            .getter(getter(CreateAgreementRequest::customDirectories)).setter(setter(Builder::customDirectories))
            .constructor(CustomDirectoriesType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomDirectories").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD,
            SERVER_ID_FIELD, LOCAL_PROFILE_ID_FIELD, PARTNER_PROFILE_ID_FIELD, BASE_DIRECTORY_FIELD, ACCESS_ROLE_FIELD,
            STATUS_FIELD, TAGS_FIELD, PRESERVE_FILENAME_FIELD, ENFORCE_MESSAGE_SIGNING_FIELD, CUSTOM_DIRECTORIES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String description;

    private final String serverId;

    private final String localProfileId;

    private final String partnerProfileId;

    private final String baseDirectory;

    private final String accessRole;

    private final String status;

    private final List<Tag> tags;

    private final String preserveFilename;

    private final String enforceMessageSigning;

    private final CustomDirectoriesType customDirectories;

    private CreateAgreementRequest(BuilderImpl builder) {
        super(builder);
        this.description = builder.description;
        this.serverId = builder.serverId;
        this.localProfileId = builder.localProfileId;
        this.partnerProfileId = builder.partnerProfileId;
        this.baseDirectory = builder.baseDirectory;
        this.accessRole = builder.accessRole;
        this.status = builder.status;
        this.tags = builder.tags;
        this.preserveFilename = builder.preserveFilename;
        this.enforceMessageSigning = builder.enforceMessageSigning;
        this.customDirectories = builder.customDirectories;
    }

    /**
     * <p>
     * A name or short description to identify the agreement.
     * </p>
     * 
     * @return A name or short description to identify the agreement.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * A system-assigned unique identifier for a server instance. This is the specific server that the agreement uses.
     * </p>
     * 
     * @return A system-assigned unique identifier for a server instance. This is the specific server that the agreement
     *         uses.
     */
    public final String serverId() {
        return serverId;
    }

    /**
     * <p>
     * A unique identifier for the AS2 local profile.
     * </p>
     * 
     * @return A unique identifier for the AS2 local profile.
     */
    public final String localProfileId() {
        return localProfileId;
    }

    /**
     * <p>
     * A unique identifier for the partner profile used in the agreement.
     * </p>
     * 
     * @return A unique identifier for the partner profile used in the agreement.
     */
    public final String partnerProfileId() {
        return partnerProfileId;
    }

    /**
     * <p>
     * The landing directory (folder) for files transferred by using the AS2 protocol.
     * </p>
     * <p>
     * A <code>BaseDirectory</code> example is <code>/<i>amzn-s3-demo-bucket</i>/home/mydirectory</code>.
     * </p>
     * 
     * @return The landing directory (folder) for files transferred by using the AS2 protocol.</p>
     *         <p>
     *         A <code>BaseDirectory</code> example is <code>/<i>amzn-s3-demo-bucket</i>/home/mydirectory</code>.
     */
    public final String baseDirectory() {
        return baseDirectory;
    }

    /**
     * <p>
     * Connectors are used to send files using either the AS2 or SFTP protocol. For the access role, provide the Amazon
     * Resource Name (ARN) of the Identity and Access Management role to use.
     * </p>
     * <p>
     * <b>For AS2 connectors</b>
     * </p>
     * <p>
     * With AS2, you can send files by calling <code>StartFileTransfer</code> and specifying the file paths in the
     * request parameter, <code>SendFilePaths</code>. We use the file’s parent directory (for example, for
     * <code>--send-file-paths /bucket/dir/file.txt</code>, parent directory is <code>/bucket/dir/</code>) to
     * temporarily store a processed AS2 message file, store the MDN when we receive them from the partner, and write a
     * final JSON file containing relevant metadata of the transmission. So, the <code>AccessRole</code> needs to
     * provide read and write access to the parent directory of the file location used in the
     * <code>StartFileTransfer</code> request. Additionally, you need to provide read and write access to the parent
     * directory of the files that you intend to send with <code>StartFileTransfer</code>.
     * </p>
     * <p>
     * If you are using Basic authentication for your AS2 connector, the access role requires the
     * <code>secretsmanager:GetSecretValue</code> permission for the secret. If the secret is encrypted using a
     * customer-managed key instead of the Amazon Web Services managed key in Secrets Manager, then the role also needs
     * the <code>kms:Decrypt</code> permission for that key.
     * </p>
     * <p>
     * <b>For SFTP connectors</b>
     * </p>
     * <p>
     * Make sure that the access role provides read and write access to the parent directory of the file location that's
     * used in the <code>StartFileTransfer</code> request. Additionally, make sure that the role provides
     * <code>secretsmanager:GetSecretValue</code> permission to Secrets Manager.
     * </p>
     * 
     * @return Connectors are used to send files using either the AS2 or SFTP protocol. For the access role, provide the
     *         Amazon Resource Name (ARN) of the Identity and Access Management role to use.</p>
     *         <p>
     *         <b>For AS2 connectors</b>
     *         </p>
     *         <p>
     *         With AS2, you can send files by calling <code>StartFileTransfer</code> and specifying the file paths in
     *         the request parameter, <code>SendFilePaths</code>. We use the file’s parent directory (for example, for
     *         <code>--send-file-paths /bucket/dir/file.txt</code>, parent directory is <code>/bucket/dir/</code>) to
     *         temporarily store a processed AS2 message file, store the MDN when we receive them from the partner, and
     *         write a final JSON file containing relevant metadata of the transmission. So, the <code>AccessRole</code>
     *         needs to provide read and write access to the parent directory of the file location used in the
     *         <code>StartFileTransfer</code> request. Additionally, you need to provide read and write access to the
     *         parent directory of the files that you intend to send with <code>StartFileTransfer</code>.
     *         </p>
     *         <p>
     *         If you are using Basic authentication for your AS2 connector, the access role requires the
     *         <code>secretsmanager:GetSecretValue</code> permission for the secret. If the secret is encrypted using a
     *         customer-managed key instead of the Amazon Web Services managed key in Secrets Manager, then the role
     *         also needs the <code>kms:Decrypt</code> permission for that key.
     *         </p>
     *         <p>
     *         <b>For SFTP connectors</b>
     *         </p>
     *         <p>
     *         Make sure that the access role provides read and write access to the parent directory of the file
     *         location that's used in the <code>StartFileTransfer</code> request. Additionally, make sure that the role
     *         provides <code>secretsmanager:GetSecretValue</code> permission to Secrets Manager.
     */
    public final String accessRole() {
        return accessRole;
    }

    /**
     * <p>
     * The status of the agreement. The agreement can be either <code>ACTIVE</code> or <code>INACTIVE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AgreementStatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the agreement. The agreement can be either <code>ACTIVE</code> or <code>INACTIVE</code>.
     * @see AgreementStatusType
     */
    public final AgreementStatusType status() {
        return AgreementStatusType.fromValue(status);
    }

    /**
     * <p>
     * The status of the agreement. The agreement can be either <code>ACTIVE</code> or <code>INACTIVE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AgreementStatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the agreement. The agreement can be either <code>ACTIVE</code> or <code>INACTIVE</code>.
     * @see AgreementStatusType
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Key-value pairs that can be used to group and search for agreements.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Key-value pairs that can be used to group and search for agreements.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Determines whether or not Transfer Family appends a unique string of characters to the end of the AS2 message
     * payload filename when saving it.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ENABLED</code>: the filename provided by your trading parter is preserved when the file is saved.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DISABLED</code> (default value): when Transfer Family saves the file, the filename is adjusted, as
     * described in <a
     * href="https://docs.aws.amazon.com/transfer/latest/userguide/send-as2-messages.html#file-names-as2">File names and
     * locations</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #preserveFilename}
     * will return {@link PreserveFilenameType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #preserveFilenameAsString}.
     * </p>
     * 
     * @return Determines whether or not Transfer Family appends a unique string of characters to the end of the AS2
     *         message payload filename when saving it. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ENABLED</code>: the filename provided by your trading parter is preserved when the file is saved.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DISABLED</code> (default value): when Transfer Family saves the file, the filename is adjusted, as
     *         described in <a
     *         href="https://docs.aws.amazon.com/transfer/latest/userguide/send-as2-messages.html#file-names-as2">File
     *         names and locations</a>.
     *         </p>
     *         </li>
     * @see PreserveFilenameType
     */
    public final PreserveFilenameType preserveFilename() {
        return PreserveFilenameType.fromValue(preserveFilename);
    }

    /**
     * <p>
     * Determines whether or not Transfer Family appends a unique string of characters to the end of the AS2 message
     * payload filename when saving it.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ENABLED</code>: the filename provided by your trading parter is preserved when the file is saved.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DISABLED</code> (default value): when Transfer Family saves the file, the filename is adjusted, as
     * described in <a
     * href="https://docs.aws.amazon.com/transfer/latest/userguide/send-as2-messages.html#file-names-as2">File names and
     * locations</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #preserveFilename}
     * will return {@link PreserveFilenameType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #preserveFilenameAsString}.
     * </p>
     * 
     * @return Determines whether or not Transfer Family appends a unique string of characters to the end of the AS2
     *         message payload filename when saving it. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ENABLED</code>: the filename provided by your trading parter is preserved when the file is saved.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DISABLED</code> (default value): when Transfer Family saves the file, the filename is adjusted, as
     *         described in <a
     *         href="https://docs.aws.amazon.com/transfer/latest/userguide/send-as2-messages.html#file-names-as2">File
     *         names and locations</a>.
     *         </p>
     *         </li>
     * @see PreserveFilenameType
     */
    public final String preserveFilenameAsString() {
        return preserveFilename;
    }

    /**
     * <p>
     * Determines whether or not unsigned messages from your trading partners will be accepted.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ENABLED</code>: Transfer Family rejects unsigned messages from your trading partner.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DISABLED</code> (default value): Transfer Family accepts unsigned messages from your trading partner.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #enforceMessageSigning} will return {@link EnforceMessageSigningType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #enforceMessageSigningAsString}.
     * </p>
     * 
     * @return Determines whether or not unsigned messages from your trading partners will be accepted. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ENABLED</code>: Transfer Family rejects unsigned messages from your trading partner.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DISABLED</code> (default value): Transfer Family accepts unsigned messages from your trading
     *         partner.
     *         </p>
     *         </li>
     * @see EnforceMessageSigningType
     */
    public final EnforceMessageSigningType enforceMessageSigning() {
        return EnforceMessageSigningType.fromValue(enforceMessageSigning);
    }

    /**
     * <p>
     * Determines whether or not unsigned messages from your trading partners will be accepted.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ENABLED</code>: Transfer Family rejects unsigned messages from your trading partner.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DISABLED</code> (default value): Transfer Family accepts unsigned messages from your trading partner.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #enforceMessageSigning} will return {@link EnforceMessageSigningType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #enforceMessageSigningAsString}.
     * </p>
     * 
     * @return Determines whether or not unsigned messages from your trading partners will be accepted. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ENABLED</code>: Transfer Family rejects unsigned messages from your trading partner.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DISABLED</code> (default value): Transfer Family accepts unsigned messages from your trading
     *         partner.
     *         </p>
     *         </li>
     * @see EnforceMessageSigningType
     */
    public final String enforceMessageSigningAsString() {
        return enforceMessageSigning;
    }

    /**
     * <p>
     * A <code>CustomDirectoriesType</code> structure. This structure specifies custom directories for storing various
     * AS2 message files. You can specify directories for the following types of files.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Failed files
     * </p>
     * </li>
     * <li>
     * <p>
     * MDN files
     * </p>
     * </li>
     * <li>
     * <p>
     * Payload files
     * </p>
     * </li>
     * <li>
     * <p>
     * Status files
     * </p>
     * </li>
     * <li>
     * <p>
     * Temporary files
     * </p>
     * </li>
     * </ul>
     * 
     * @return A <code>CustomDirectoriesType</code> structure. This structure specifies custom directories for storing
     *         various AS2 message files. You can specify directories for the following types of files.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Failed files
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         MDN files
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Payload files
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Status files
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Temporary files
     *         </p>
     *         </li>
     */
    public final CustomDirectoriesType customDirectories() {
        return customDirectories;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(serverId());
        hashCode = 31 * hashCode + Objects.hashCode(localProfileId());
        hashCode = 31 * hashCode + Objects.hashCode(partnerProfileId());
        hashCode = 31 * hashCode + Objects.hashCode(baseDirectory());
        hashCode = 31 * hashCode + Objects.hashCode(accessRole());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(preserveFilenameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(enforceMessageSigningAsString());
        hashCode = 31 * hashCode + Objects.hashCode(customDirectories());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateAgreementRequest)) {
            return false;
        }
        CreateAgreementRequest other = (CreateAgreementRequest) obj;
        return Objects.equals(description(), other.description()) && Objects.equals(serverId(), other.serverId())
                && Objects.equals(localProfileId(), other.localProfileId())
                && Objects.equals(partnerProfileId(), other.partnerProfileId())
                && Objects.equals(baseDirectory(), other.baseDirectory()) && Objects.equals(accessRole(), other.accessRole())
                && Objects.equals(statusAsString(), other.statusAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags())
                && Objects.equals(preserveFilenameAsString(), other.preserveFilenameAsString())
                && Objects.equals(enforceMessageSigningAsString(), other.enforceMessageSigningAsString())
                && Objects.equals(customDirectories(), other.customDirectories());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateAgreementRequest").add("Description", description()).add("ServerId", serverId())
                .add("LocalProfileId", localProfileId()).add("PartnerProfileId", partnerProfileId())
                .add("BaseDirectory", baseDirectory()).add("AccessRole", accessRole()).add("Status", statusAsString())
                .add("Tags", hasTags() ? tags() : null).add("PreserveFilename", preserveFilenameAsString())
                .add("EnforceMessageSigning", enforceMessageSigningAsString()).add("CustomDirectories", customDirectories())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "ServerId":
            return Optional.ofNullable(clazz.cast(serverId()));
        case "LocalProfileId":
            return Optional.ofNullable(clazz.cast(localProfileId()));
        case "PartnerProfileId":
            return Optional.ofNullable(clazz.cast(partnerProfileId()));
        case "BaseDirectory":
            return Optional.ofNullable(clazz.cast(baseDirectory()));
        case "AccessRole":
            return Optional.ofNullable(clazz.cast(accessRole()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "PreserveFilename":
            return Optional.ofNullable(clazz.cast(preserveFilenameAsString()));
        case "EnforceMessageSigning":
            return Optional.ofNullable(clazz.cast(enforceMessageSigningAsString()));
        case "CustomDirectories":
            return Optional.ofNullable(clazz.cast(customDirectories()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Description", DESCRIPTION_FIELD);
        map.put("ServerId", SERVER_ID_FIELD);
        map.put("LocalProfileId", LOCAL_PROFILE_ID_FIELD);
        map.put("PartnerProfileId", PARTNER_PROFILE_ID_FIELD);
        map.put("BaseDirectory", BASE_DIRECTORY_FIELD);
        map.put("AccessRole", ACCESS_ROLE_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("Tags", TAGS_FIELD);
        map.put("PreserveFilename", PRESERVE_FILENAME_FIELD);
        map.put("EnforceMessageSigning", ENFORCE_MESSAGE_SIGNING_FIELD);
        map.put("CustomDirectories", CUSTOM_DIRECTORIES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateAgreementRequest, T> g) {
        return obj -> g.apply((CreateAgreementRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends TransferRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateAgreementRequest> {
        /**
         * <p>
         * A name or short description to identify the agreement.
         * </p>
         * 
         * @param description
         *        A name or short description to identify the agreement.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A system-assigned unique identifier for a server instance. This is the specific server that the agreement
         * uses.
         * </p>
         * 
         * @param serverId
         *        A system-assigned unique identifier for a server instance. This is the specific server that the
         *        agreement uses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverId(String serverId);

        /**
         * <p>
         * A unique identifier for the AS2 local profile.
         * </p>
         * 
         * @param localProfileId
         *        A unique identifier for the AS2 local profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder localProfileId(String localProfileId);

        /**
         * <p>
         * A unique identifier for the partner profile used in the agreement.
         * </p>
         * 
         * @param partnerProfileId
         *        A unique identifier for the partner profile used in the agreement.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partnerProfileId(String partnerProfileId);

        /**
         * <p>
         * The landing directory (folder) for files transferred by using the AS2 protocol.
         * </p>
         * <p>
         * A <code>BaseDirectory</code> example is <code>/<i>amzn-s3-demo-bucket</i>/home/mydirectory</code>.
         * </p>
         * 
         * @param baseDirectory
         *        The landing directory (folder) for files transferred by using the AS2 protocol.</p>
         *        <p>
         *        A <code>BaseDirectory</code> example is <code>/<i>amzn-s3-demo-bucket</i>/home/mydirectory</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder baseDirectory(String baseDirectory);

        /**
         * <p>
         * Connectors are used to send files using either the AS2 or SFTP protocol. For the access role, provide the
         * Amazon Resource Name (ARN) of the Identity and Access Management role to use.
         * </p>
         * <p>
         * <b>For AS2 connectors</b>
         * </p>
         * <p>
         * With AS2, you can send files by calling <code>StartFileTransfer</code> and specifying the file paths in the
         * request parameter, <code>SendFilePaths</code>. We use the file’s parent directory (for example, for
         * <code>--send-file-paths /bucket/dir/file.txt</code>, parent directory is <code>/bucket/dir/</code>) to
         * temporarily store a processed AS2 message file, store the MDN when we receive them from the partner, and
         * write a final JSON file containing relevant metadata of the transmission. So, the <code>AccessRole</code>
         * needs to provide read and write access to the parent directory of the file location used in the
         * <code>StartFileTransfer</code> request. Additionally, you need to provide read and write access to the parent
         * directory of the files that you intend to send with <code>StartFileTransfer</code>.
         * </p>
         * <p>
         * If you are using Basic authentication for your AS2 connector, the access role requires the
         * <code>secretsmanager:GetSecretValue</code> permission for the secret. If the secret is encrypted using a
         * customer-managed key instead of the Amazon Web Services managed key in Secrets Manager, then the role also
         * needs the <code>kms:Decrypt</code> permission for that key.
         * </p>
         * <p>
         * <b>For SFTP connectors</b>
         * </p>
         * <p>
         * Make sure that the access role provides read and write access to the parent directory of the file location
         * that's used in the <code>StartFileTransfer</code> request. Additionally, make sure that the role provides
         * <code>secretsmanager:GetSecretValue</code> permission to Secrets Manager.
         * </p>
         * 
         * @param accessRole
         *        Connectors are used to send files using either the AS2 or SFTP protocol. For the access role, provide
         *        the Amazon Resource Name (ARN) of the Identity and Access Management role to use.</p>
         *        <p>
         *        <b>For AS2 connectors</b>
         *        </p>
         *        <p>
         *        With AS2, you can send files by calling <code>StartFileTransfer</code> and specifying the file paths
         *        in the request parameter, <code>SendFilePaths</code>. We use the file’s parent directory (for example,
         *        for <code>--send-file-paths /bucket/dir/file.txt</code>, parent directory is <code>/bucket/dir/</code>
         *        ) to temporarily store a processed AS2 message file, store the MDN when we receive them from the
         *        partner, and write a final JSON file containing relevant metadata of the transmission. So, the
         *        <code>AccessRole</code> needs to provide read and write access to the parent directory of the file
         *        location used in the <code>StartFileTransfer</code> request. Additionally, you need to provide read
         *        and write access to the parent directory of the files that you intend to send with
         *        <code>StartFileTransfer</code>.
         *        </p>
         *        <p>
         *        If you are using Basic authentication for your AS2 connector, the access role requires the
         *        <code>secretsmanager:GetSecretValue</code> permission for the secret. If the secret is encrypted using
         *        a customer-managed key instead of the Amazon Web Services managed key in Secrets Manager, then the
         *        role also needs the <code>kms:Decrypt</code> permission for that key.
         *        </p>
         *        <p>
         *        <b>For SFTP connectors</b>
         *        </p>
         *        <p>
         *        Make sure that the access role provides read and write access to the parent directory of the file
         *        location that's used in the <code>StartFileTransfer</code> request. Additionally, make sure that the
         *        role provides <code>secretsmanager:GetSecretValue</code> permission to Secrets Manager.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessRole(String accessRole);

        /**
         * <p>
         * The status of the agreement. The agreement can be either <code>ACTIVE</code> or <code>INACTIVE</code>.
         * </p>
         * 
         * @param status
         *        The status of the agreement. The agreement can be either <code>ACTIVE</code> or <code>INACTIVE</code>.
         * @see AgreementStatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AgreementStatusType
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the agreement. The agreement can be either <code>ACTIVE</code> or <code>INACTIVE</code>.
         * </p>
         * 
         * @param status
         *        The status of the agreement. The agreement can be either <code>ACTIVE</code> or <code>INACTIVE</code>.
         * @see AgreementStatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AgreementStatusType
         */
        Builder status(AgreementStatusType status);

        /**
         * <p>
         * Key-value pairs that can be used to group and search for agreements.
         * </p>
         * 
         * @param tags
         *        Key-value pairs that can be used to group and search for agreements.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Key-value pairs that can be used to group and search for agreements.
         * </p>
         * 
         * @param tags
         *        Key-value pairs that can be used to group and search for agreements.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Key-value pairs that can be used to group and search for agreements.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.transfer.model.Tag.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.transfer.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.transfer.model.Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.transfer.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Determines whether or not Transfer Family appends a unique string of characters to the end of the AS2 message
         * payload filename when saving it.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ENABLED</code>: the filename provided by your trading parter is preserved when the file is saved.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DISABLED</code> (default value): when Transfer Family saves the file, the filename is adjusted, as
         * described in <a
         * href="https://docs.aws.amazon.com/transfer/latest/userguide/send-as2-messages.html#file-names-as2">File names
         * and locations</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param preserveFilename
         *        Determines whether or not Transfer Family appends a unique string of characters to the end of the AS2
         *        message payload filename when saving it. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ENABLED</code>: the filename provided by your trading parter is preserved when the file is
         *        saved.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DISABLED</code> (default value): when Transfer Family saves the file, the filename is adjusted,
         *        as described in <a
         *        href="https://docs.aws.amazon.com/transfer/latest/userguide/send-as2-messages.html#file-names-as2"
         *        >File names and locations</a>.
         *        </p>
         *        </li>
         * @see PreserveFilenameType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PreserveFilenameType
         */
        Builder preserveFilename(String preserveFilename);

        /**
         * <p>
         * Determines whether or not Transfer Family appends a unique string of characters to the end of the AS2 message
         * payload filename when saving it.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ENABLED</code>: the filename provided by your trading parter is preserved when the file is saved.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DISABLED</code> (default value): when Transfer Family saves the file, the filename is adjusted, as
         * described in <a
         * href="https://docs.aws.amazon.com/transfer/latest/userguide/send-as2-messages.html#file-names-as2">File names
         * and locations</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param preserveFilename
         *        Determines whether or not Transfer Family appends a unique string of characters to the end of the AS2
         *        message payload filename when saving it. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ENABLED</code>: the filename provided by your trading parter is preserved when the file is
         *        saved.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DISABLED</code> (default value): when Transfer Family saves the file, the filename is adjusted,
         *        as described in <a
         *        href="https://docs.aws.amazon.com/transfer/latest/userguide/send-as2-messages.html#file-names-as2"
         *        >File names and locations</a>.
         *        </p>
         *        </li>
         * @see PreserveFilenameType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PreserveFilenameType
         */
        Builder preserveFilename(PreserveFilenameType preserveFilename);

        /**
         * <p>
         * Determines whether or not unsigned messages from your trading partners will be accepted.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ENABLED</code>: Transfer Family rejects unsigned messages from your trading partner.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DISABLED</code> (default value): Transfer Family accepts unsigned messages from your trading partner.
         * </p>
         * </li>
         * </ul>
         * 
         * @param enforceMessageSigning
         *        Determines whether or not unsigned messages from your trading partners will be accepted. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ENABLED</code>: Transfer Family rejects unsigned messages from your trading partner.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DISABLED</code> (default value): Transfer Family accepts unsigned messages from your trading
         *        partner.
         *        </p>
         *        </li>
         * @see EnforceMessageSigningType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EnforceMessageSigningType
         */
        Builder enforceMessageSigning(String enforceMessageSigning);

        /**
         * <p>
         * Determines whether or not unsigned messages from your trading partners will be accepted.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ENABLED</code>: Transfer Family rejects unsigned messages from your trading partner.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DISABLED</code> (default value): Transfer Family accepts unsigned messages from your trading partner.
         * </p>
         * </li>
         * </ul>
         * 
         * @param enforceMessageSigning
         *        Determines whether or not unsigned messages from your trading partners will be accepted. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ENABLED</code>: Transfer Family rejects unsigned messages from your trading partner.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DISABLED</code> (default value): Transfer Family accepts unsigned messages from your trading
         *        partner.
         *        </p>
         *        </li>
         * @see EnforceMessageSigningType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EnforceMessageSigningType
         */
        Builder enforceMessageSigning(EnforceMessageSigningType enforceMessageSigning);

        /**
         * <p>
         * A <code>CustomDirectoriesType</code> structure. This structure specifies custom directories for storing
         * various AS2 message files. You can specify directories for the following types of files.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Failed files
         * </p>
         * </li>
         * <li>
         * <p>
         * MDN files
         * </p>
         * </li>
         * <li>
         * <p>
         * Payload files
         * </p>
         * </li>
         * <li>
         * <p>
         * Status files
         * </p>
         * </li>
         * <li>
         * <p>
         * Temporary files
         * </p>
         * </li>
         * </ul>
         * 
         * @param customDirectories
         *        A <code>CustomDirectoriesType</code> structure. This structure specifies custom directories for
         *        storing various AS2 message files. You can specify directories for the following types of files.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Failed files
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        MDN files
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Payload files
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Status files
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Temporary files
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customDirectories(CustomDirectoriesType customDirectories);

        /**
         * <p>
         * A <code>CustomDirectoriesType</code> structure. This structure specifies custom directories for storing
         * various AS2 message files. You can specify directories for the following types of files.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Failed files
         * </p>
         * </li>
         * <li>
         * <p>
         * MDN files
         * </p>
         * </li>
         * <li>
         * <p>
         * Payload files
         * </p>
         * </li>
         * <li>
         * <p>
         * Status files
         * </p>
         * </li>
         * <li>
         * <p>
         * Temporary files
         * </p>
         * </li>
         * </ul>
         * This is a convenience method that creates an instance of the {@link CustomDirectoriesType.Builder} avoiding
         * the need to create one manually via {@link CustomDirectoriesType#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CustomDirectoriesType.Builder#build()} is called immediately and
         * its result is passed to {@link #customDirectories(CustomDirectoriesType)}.
         * 
         * @param customDirectories
         *        a consumer that will call methods on {@link CustomDirectoriesType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #customDirectories(CustomDirectoriesType)
         */
        default Builder customDirectories(Consumer<CustomDirectoriesType.Builder> customDirectories) {
            return customDirectories(CustomDirectoriesType.builder().applyMutation(customDirectories).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends TransferRequest.BuilderImpl implements Builder {
        private String description;

        private String serverId;

        private String localProfileId;

        private String partnerProfileId;

        private String baseDirectory;

        private String accessRole;

        private String status;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String preserveFilename;

        private String enforceMessageSigning;

        private CustomDirectoriesType customDirectories;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateAgreementRequest model) {
            super(model);
            description(model.description);
            serverId(model.serverId);
            localProfileId(model.localProfileId);
            partnerProfileId(model.partnerProfileId);
            baseDirectory(model.baseDirectory);
            accessRole(model.accessRole);
            status(model.status);
            tags(model.tags);
            preserveFilename(model.preserveFilename);
            enforceMessageSigning(model.enforceMessageSigning);
            customDirectories(model.customDirectories);
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getServerId() {
            return serverId;
        }

        public final void setServerId(String serverId) {
            this.serverId = serverId;
        }

        @Override
        public final Builder serverId(String serverId) {
            this.serverId = serverId;
            return this;
        }

        public final String getLocalProfileId() {
            return localProfileId;
        }

        public final void setLocalProfileId(String localProfileId) {
            this.localProfileId = localProfileId;
        }

        @Override
        public final Builder localProfileId(String localProfileId) {
            this.localProfileId = localProfileId;
            return this;
        }

        public final String getPartnerProfileId() {
            return partnerProfileId;
        }

        public final void setPartnerProfileId(String partnerProfileId) {
            this.partnerProfileId = partnerProfileId;
        }

        @Override
        public final Builder partnerProfileId(String partnerProfileId) {
            this.partnerProfileId = partnerProfileId;
            return this;
        }

        public final String getBaseDirectory() {
            return baseDirectory;
        }

        public final void setBaseDirectory(String baseDirectory) {
            this.baseDirectory = baseDirectory;
        }

        @Override
        public final Builder baseDirectory(String baseDirectory) {
            this.baseDirectory = baseDirectory;
            return this;
        }

        public final String getAccessRole() {
            return accessRole;
        }

        public final void setAccessRole(String accessRole) {
            this.accessRole = accessRole;
        }

        @Override
        public final Builder accessRole(String accessRole) {
            this.accessRole = accessRole;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(AgreementStatusType status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getPreserveFilename() {
            return preserveFilename;
        }

        public final void setPreserveFilename(String preserveFilename) {
            this.preserveFilename = preserveFilename;
        }

        @Override
        public final Builder preserveFilename(String preserveFilename) {
            this.preserveFilename = preserveFilename;
            return this;
        }

        @Override
        public final Builder preserveFilename(PreserveFilenameType preserveFilename) {
            this.preserveFilename(preserveFilename == null ? null : preserveFilename.toString());
            return this;
        }

        public final String getEnforceMessageSigning() {
            return enforceMessageSigning;
        }

        public final void setEnforceMessageSigning(String enforceMessageSigning) {
            this.enforceMessageSigning = enforceMessageSigning;
        }

        @Override
        public final Builder enforceMessageSigning(String enforceMessageSigning) {
            this.enforceMessageSigning = enforceMessageSigning;
            return this;
        }

        @Override
        public final Builder enforceMessageSigning(EnforceMessageSigningType enforceMessageSigning) {
            this.enforceMessageSigning(enforceMessageSigning == null ? null : enforceMessageSigning.toString());
            return this;
        }

        public final CustomDirectoriesType.Builder getCustomDirectories() {
            return customDirectories != null ? customDirectories.toBuilder() : null;
        }

        public final void setCustomDirectories(CustomDirectoriesType.BuilderImpl customDirectories) {
            this.customDirectories = customDirectories != null ? customDirectories.build() : null;
        }

        @Override
        public final Builder customDirectories(CustomDirectoriesType customDirectories) {
            this.customDirectories = customDirectories;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateAgreementRequest build() {
            return new CreateAgreementRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
