/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transfer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the details for an SFTP connector object. The connector object is used for transferring files to and from a
 * partner's SFTP server.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SftpConnectorConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<SftpConnectorConfig.Builder, SftpConnectorConfig> {
    private static final SdkField<String> USER_SECRET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UserSecretId").getter(getter(SftpConnectorConfig::userSecretId)).setter(setter(Builder::userSecretId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserSecretId").build()).build();

    private static final SdkField<List<String>> TRUSTED_HOST_KEYS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("TrustedHostKeys")
            .getter(getter(SftpConnectorConfig::trustedHostKeys))
            .setter(setter(Builder::trustedHostKeys))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrustedHostKeys").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> MAX_CONCURRENT_CONNECTIONS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxConcurrentConnections").getter(getter(SftpConnectorConfig::maxConcurrentConnections))
            .setter(setter(Builder::maxConcurrentConnections))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxConcurrentConnections").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(USER_SECRET_ID_FIELD,
            TRUSTED_HOST_KEYS_FIELD, MAX_CONCURRENT_CONNECTIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String userSecretId;

    private final List<String> trustedHostKeys;

    private final Integer maxConcurrentConnections;

    private SftpConnectorConfig(BuilderImpl builder) {
        this.userSecretId = builder.userSecretId;
        this.trustedHostKeys = builder.trustedHostKeys;
        this.maxConcurrentConnections = builder.maxConcurrentConnections;
    }

    /**
     * <p>
     * The identifier for the secret (in Amazon Web Services Secrets Manager) that contains the SFTP user's private key,
     * password, or both. The identifier must be the Amazon Resource Name (ARN) of the secret.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * Required when creating an SFTP connector
     * </p>
     * </li>
     * <li>
     * <p>
     * Optional when updating an existing SFTP connector
     * </p>
     * </li>
     * </ul>
     * </note>
     * 
     * @return The identifier for the secret (in Amazon Web Services Secrets Manager) that contains the SFTP user's
     *         private key, password, or both. The identifier must be the Amazon Resource Name (ARN) of the secret.</p>
     *         <note>
     *         <ul>
     *         <li>
     *         <p>
     *         Required when creating an SFTP connector
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Optional when updating an existing SFTP connector
     *         </p>
     *         </li>
     *         </ul>
     */
    public final String userSecretId() {
        return userSecretId;
    }

    /**
     * For responses, this returns true if the service returned a value for the TrustedHostKeys property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTrustedHostKeys() {
        return trustedHostKeys != null && !(trustedHostKeys instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The public portion of the host key, or keys, that are used to identify the external server to which you are
     * connecting. You can use the <code>ssh-keyscan</code> command against the SFTP server to retrieve the necessary
     * key.
     * </p>
     * <note>
     * <p>
     * <code>TrustedHostKeys</code> is optional for <code>CreateConnector</code>. If not provided, you can use
     * <code>TestConnection</code> to retrieve the server host key during the initial connection attempt, and
     * subsequently update the connector with the observed host key.
     * </p>
     * </note>
     * <p>
     * The three standard SSH public key format elements are <code>&lt;key type&gt;</code>,
     * <code>&lt;body base64&gt;</code>, and an optional <code>&lt;comment&gt;</code>, with spaces between each element.
     * Specify only the <code>&lt;key type&gt;</code> and <code>&lt;body base64&gt;</code>: do not enter the
     * <code>&lt;comment&gt;</code> portion of the key.
     * </p>
     * <p>
     * For the trusted host key, Transfer Family accepts RSA and ECDSA keys.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For RSA keys, the <code>&lt;key type&gt;</code> string is <code>ssh-rsa</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For ECDSA keys, the <code>&lt;key type&gt;</code> string is either <code>ecdsa-sha2-nistp256</code>,
     * <code>ecdsa-sha2-nistp384</code>, or <code>ecdsa-sha2-nistp521</code>, depending on the size of the key you
     * generated.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Run this command to retrieve the SFTP server host key, where your SFTP server name is <code>ftp.host.com</code>.
     * </p>
     * <p>
     * <code>ssh-keyscan ftp.host.com</code>
     * </p>
     * <p>
     * This prints the public host key to standard output.
     * </p>
     * <p>
     * <code>ftp.host.com ssh-rsa AAAAB3Nza...&lt;long-string-for-public-key</code>
     * </p>
     * <p>
     * Copy and paste this string into the <code>TrustedHostKeys</code> field for the <code>create-connector</code>
     * command or into the <b>Trusted host keys</b> field in the console.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTrustedHostKeys} method.
     * </p>
     * 
     * @return The public portion of the host key, or keys, that are used to identify the external server to which you
     *         are connecting. You can use the <code>ssh-keyscan</code> command against the SFTP server to retrieve the
     *         necessary key.</p> <note>
     *         <p>
     *         <code>TrustedHostKeys</code> is optional for <code>CreateConnector</code>. If not provided, you can use
     *         <code>TestConnection</code> to retrieve the server host key during the initial connection attempt, and
     *         subsequently update the connector with the observed host key.
     *         </p>
     *         </note>
     *         <p>
     *         The three standard SSH public key format elements are <code>&lt;key type&gt;</code>,
     *         <code>&lt;body base64&gt;</code>, and an optional <code>&lt;comment&gt;</code>, with spaces between each
     *         element. Specify only the <code>&lt;key type&gt;</code> and <code>&lt;body base64&gt;</code>: do not
     *         enter the <code>&lt;comment&gt;</code> portion of the key.
     *         </p>
     *         <p>
     *         For the trusted host key, Transfer Family accepts RSA and ECDSA keys.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For RSA keys, the <code>&lt;key type&gt;</code> string is <code>ssh-rsa</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For ECDSA keys, the <code>&lt;key type&gt;</code> string is either <code>ecdsa-sha2-nistp256</code>,
     *         <code>ecdsa-sha2-nistp384</code>, or <code>ecdsa-sha2-nistp521</code>, depending on the size of the key
     *         you generated.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Run this command to retrieve the SFTP server host key, where your SFTP server name is
     *         <code>ftp.host.com</code>.
     *         </p>
     *         <p>
     *         <code>ssh-keyscan ftp.host.com</code>
     *         </p>
     *         <p>
     *         This prints the public host key to standard output.
     *         </p>
     *         <p>
     *         <code>ftp.host.com ssh-rsa AAAAB3Nza...&lt;long-string-for-public-key</code>
     *         </p>
     *         <p>
     *         Copy and paste this string into the <code>TrustedHostKeys</code> field for the
     *         <code>create-connector</code> command or into the <b>Trusted host keys</b> field in the console.
     */
    public final List<String> trustedHostKeys() {
        return trustedHostKeys;
    }

    /**
     * <p>
     * Specify the number of concurrent connections that your connector creates to the remote server. The default value
     * is <code>5</code> (this is also the maximum value allowed).
     * </p>
     * <p>
     * This parameter specifies the number of active connections that your connector can establish with the remote
     * server at the same time. Increasing this value can enhance connector performance when transferring large file
     * batches by enabling parallel operations.
     * </p>
     * 
     * @return Specify the number of concurrent connections that your connector creates to the remote server. The
     *         default value is <code>5</code> (this is also the maximum value allowed).</p>
     *         <p>
     *         This parameter specifies the number of active connections that your connector can establish with the
     *         remote server at the same time. Increasing this value can enhance connector performance when transferring
     *         large file batches by enabling parallel operations.
     */
    public final Integer maxConcurrentConnections() {
        return maxConcurrentConnections;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(userSecretId());
        hashCode = 31 * hashCode + Objects.hashCode(hasTrustedHostKeys() ? trustedHostKeys() : null);
        hashCode = 31 * hashCode + Objects.hashCode(maxConcurrentConnections());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SftpConnectorConfig)) {
            return false;
        }
        SftpConnectorConfig other = (SftpConnectorConfig) obj;
        return Objects.equals(userSecretId(), other.userSecretId()) && hasTrustedHostKeys() == other.hasTrustedHostKeys()
                && Objects.equals(trustedHostKeys(), other.trustedHostKeys())
                && Objects.equals(maxConcurrentConnections(), other.maxConcurrentConnections());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SftpConnectorConfig").add("UserSecretId", userSecretId())
                .add("TrustedHostKeys", hasTrustedHostKeys() ? trustedHostKeys() : null)
                .add("MaxConcurrentConnections", maxConcurrentConnections()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "UserSecretId":
            return Optional.ofNullable(clazz.cast(userSecretId()));
        case "TrustedHostKeys":
            return Optional.ofNullable(clazz.cast(trustedHostKeys()));
        case "MaxConcurrentConnections":
            return Optional.ofNullable(clazz.cast(maxConcurrentConnections()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("UserSecretId", USER_SECRET_ID_FIELD);
        map.put("TrustedHostKeys", TRUSTED_HOST_KEYS_FIELD);
        map.put("MaxConcurrentConnections", MAX_CONCURRENT_CONNECTIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SftpConnectorConfig, T> g) {
        return obj -> g.apply((SftpConnectorConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SftpConnectorConfig> {
        /**
         * <p>
         * The identifier for the secret (in Amazon Web Services Secrets Manager) that contains the SFTP user's private
         * key, password, or both. The identifier must be the Amazon Resource Name (ARN) of the secret.
         * </p>
         * <note>
         * <ul>
         * <li>
         * <p>
         * Required when creating an SFTP connector
         * </p>
         * </li>
         * <li>
         * <p>
         * Optional when updating an existing SFTP connector
         * </p>
         * </li>
         * </ul>
         * </note>
         * 
         * @param userSecretId
         *        The identifier for the secret (in Amazon Web Services Secrets Manager) that contains the SFTP user's
         *        private key, password, or both. The identifier must be the Amazon Resource Name (ARN) of the
         *        secret.</p> <note>
         *        <ul>
         *        <li>
         *        <p>
         *        Required when creating an SFTP connector
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Optional when updating an existing SFTP connector
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userSecretId(String userSecretId);

        /**
         * <p>
         * The public portion of the host key, or keys, that are used to identify the external server to which you are
         * connecting. You can use the <code>ssh-keyscan</code> command against the SFTP server to retrieve the
         * necessary key.
         * </p>
         * <note>
         * <p>
         * <code>TrustedHostKeys</code> is optional for <code>CreateConnector</code>. If not provided, you can use
         * <code>TestConnection</code> to retrieve the server host key during the initial connection attempt, and
         * subsequently update the connector with the observed host key.
         * </p>
         * </note>
         * <p>
         * The three standard SSH public key format elements are <code>&lt;key type&gt;</code>,
         * <code>&lt;body base64&gt;</code>, and an optional <code>&lt;comment&gt;</code>, with spaces between each
         * element. Specify only the <code>&lt;key type&gt;</code> and <code>&lt;body base64&gt;</code>: do not enter
         * the <code>&lt;comment&gt;</code> portion of the key.
         * </p>
         * <p>
         * For the trusted host key, Transfer Family accepts RSA and ECDSA keys.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For RSA keys, the <code>&lt;key type&gt;</code> string is <code>ssh-rsa</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For ECDSA keys, the <code>&lt;key type&gt;</code> string is either <code>ecdsa-sha2-nistp256</code>,
         * <code>ecdsa-sha2-nistp384</code>, or <code>ecdsa-sha2-nistp521</code>, depending on the size of the key you
         * generated.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Run this command to retrieve the SFTP server host key, where your SFTP server name is
         * <code>ftp.host.com</code>.
         * </p>
         * <p>
         * <code>ssh-keyscan ftp.host.com</code>
         * </p>
         * <p>
         * This prints the public host key to standard output.
         * </p>
         * <p>
         * <code>ftp.host.com ssh-rsa AAAAB3Nza...&lt;long-string-for-public-key</code>
         * </p>
         * <p>
         * Copy and paste this string into the <code>TrustedHostKeys</code> field for the <code>create-connector</code>
         * command or into the <b>Trusted host keys</b> field in the console.
         * </p>
         * 
         * @param trustedHostKeys
         *        The public portion of the host key, or keys, that are used to identify the external server to which
         *        you are connecting. You can use the <code>ssh-keyscan</code> command against the SFTP server to
         *        retrieve the necessary key.</p> <note>
         *        <p>
         *        <code>TrustedHostKeys</code> is optional for <code>CreateConnector</code>. If not provided, you can
         *        use <code>TestConnection</code> to retrieve the server host key during the initial connection attempt,
         *        and subsequently update the connector with the observed host key.
         *        </p>
         *        </note>
         *        <p>
         *        The three standard SSH public key format elements are <code>&lt;key type&gt;</code>,
         *        <code>&lt;body base64&gt;</code>, and an optional <code>&lt;comment&gt;</code>, with spaces between
         *        each element. Specify only the <code>&lt;key type&gt;</code> and <code>&lt;body base64&gt;</code>: do
         *        not enter the <code>&lt;comment&gt;</code> portion of the key.
         *        </p>
         *        <p>
         *        For the trusted host key, Transfer Family accepts RSA and ECDSA keys.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For RSA keys, the <code>&lt;key type&gt;</code> string is <code>ssh-rsa</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For ECDSA keys, the <code>&lt;key type&gt;</code> string is either <code>ecdsa-sha2-nistp256</code>,
         *        <code>ecdsa-sha2-nistp384</code>, or <code>ecdsa-sha2-nistp521</code>, depending on the size of the
         *        key you generated.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Run this command to retrieve the SFTP server host key, where your SFTP server name is
         *        <code>ftp.host.com</code>.
         *        </p>
         *        <p>
         *        <code>ssh-keyscan ftp.host.com</code>
         *        </p>
         *        <p>
         *        This prints the public host key to standard output.
         *        </p>
         *        <p>
         *        <code>ftp.host.com ssh-rsa AAAAB3Nza...&lt;long-string-for-public-key</code>
         *        </p>
         *        <p>
         *        Copy and paste this string into the <code>TrustedHostKeys</code> field for the
         *        <code>create-connector</code> command or into the <b>Trusted host keys</b> field in the console.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trustedHostKeys(Collection<String> trustedHostKeys);

        /**
         * <p>
         * The public portion of the host key, or keys, that are used to identify the external server to which you are
         * connecting. You can use the <code>ssh-keyscan</code> command against the SFTP server to retrieve the
         * necessary key.
         * </p>
         * <note>
         * <p>
         * <code>TrustedHostKeys</code> is optional for <code>CreateConnector</code>. If not provided, you can use
         * <code>TestConnection</code> to retrieve the server host key during the initial connection attempt, and
         * subsequently update the connector with the observed host key.
         * </p>
         * </note>
         * <p>
         * The three standard SSH public key format elements are <code>&lt;key type&gt;</code>,
         * <code>&lt;body base64&gt;</code>, and an optional <code>&lt;comment&gt;</code>, with spaces between each
         * element. Specify only the <code>&lt;key type&gt;</code> and <code>&lt;body base64&gt;</code>: do not enter
         * the <code>&lt;comment&gt;</code> portion of the key.
         * </p>
         * <p>
         * For the trusted host key, Transfer Family accepts RSA and ECDSA keys.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For RSA keys, the <code>&lt;key type&gt;</code> string is <code>ssh-rsa</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For ECDSA keys, the <code>&lt;key type&gt;</code> string is either <code>ecdsa-sha2-nistp256</code>,
         * <code>ecdsa-sha2-nistp384</code>, or <code>ecdsa-sha2-nistp521</code>, depending on the size of the key you
         * generated.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Run this command to retrieve the SFTP server host key, where your SFTP server name is
         * <code>ftp.host.com</code>.
         * </p>
         * <p>
         * <code>ssh-keyscan ftp.host.com</code>
         * </p>
         * <p>
         * This prints the public host key to standard output.
         * </p>
         * <p>
         * <code>ftp.host.com ssh-rsa AAAAB3Nza...&lt;long-string-for-public-key</code>
         * </p>
         * <p>
         * Copy and paste this string into the <code>TrustedHostKeys</code> field for the <code>create-connector</code>
         * command or into the <b>Trusted host keys</b> field in the console.
         * </p>
         * 
         * @param trustedHostKeys
         *        The public portion of the host key, or keys, that are used to identify the external server to which
         *        you are connecting. You can use the <code>ssh-keyscan</code> command against the SFTP server to
         *        retrieve the necessary key.</p> <note>
         *        <p>
         *        <code>TrustedHostKeys</code> is optional for <code>CreateConnector</code>. If not provided, you can
         *        use <code>TestConnection</code> to retrieve the server host key during the initial connection attempt,
         *        and subsequently update the connector with the observed host key.
         *        </p>
         *        </note>
         *        <p>
         *        The three standard SSH public key format elements are <code>&lt;key type&gt;</code>,
         *        <code>&lt;body base64&gt;</code>, and an optional <code>&lt;comment&gt;</code>, with spaces between
         *        each element. Specify only the <code>&lt;key type&gt;</code> and <code>&lt;body base64&gt;</code>: do
         *        not enter the <code>&lt;comment&gt;</code> portion of the key.
         *        </p>
         *        <p>
         *        For the trusted host key, Transfer Family accepts RSA and ECDSA keys.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For RSA keys, the <code>&lt;key type&gt;</code> string is <code>ssh-rsa</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For ECDSA keys, the <code>&lt;key type&gt;</code> string is either <code>ecdsa-sha2-nistp256</code>,
         *        <code>ecdsa-sha2-nistp384</code>, or <code>ecdsa-sha2-nistp521</code>, depending on the size of the
         *        key you generated.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Run this command to retrieve the SFTP server host key, where your SFTP server name is
         *        <code>ftp.host.com</code>.
         *        </p>
         *        <p>
         *        <code>ssh-keyscan ftp.host.com</code>
         *        </p>
         *        <p>
         *        This prints the public host key to standard output.
         *        </p>
         *        <p>
         *        <code>ftp.host.com ssh-rsa AAAAB3Nza...&lt;long-string-for-public-key</code>
         *        </p>
         *        <p>
         *        Copy and paste this string into the <code>TrustedHostKeys</code> field for the
         *        <code>create-connector</code> command or into the <b>Trusted host keys</b> field in the console.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trustedHostKeys(String... trustedHostKeys);

        /**
         * <p>
         * Specify the number of concurrent connections that your connector creates to the remote server. The default
         * value is <code>5</code> (this is also the maximum value allowed).
         * </p>
         * <p>
         * This parameter specifies the number of active connections that your connector can establish with the remote
         * server at the same time. Increasing this value can enhance connector performance when transferring large file
         * batches by enabling parallel operations.
         * </p>
         * 
         * @param maxConcurrentConnections
         *        Specify the number of concurrent connections that your connector creates to the remote server. The
         *        default value is <code>5</code> (this is also the maximum value allowed).</p>
         *        <p>
         *        This parameter specifies the number of active connections that your connector can establish with the
         *        remote server at the same time. Increasing this value can enhance connector performance when
         *        transferring large file batches by enabling parallel operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxConcurrentConnections(Integer maxConcurrentConnections);
    }

    static final class BuilderImpl implements Builder {
        private String userSecretId;

        private List<String> trustedHostKeys = DefaultSdkAutoConstructList.getInstance();

        private Integer maxConcurrentConnections;

        private BuilderImpl() {
        }

        private BuilderImpl(SftpConnectorConfig model) {
            userSecretId(model.userSecretId);
            trustedHostKeys(model.trustedHostKeys);
            maxConcurrentConnections(model.maxConcurrentConnections);
        }

        public final String getUserSecretId() {
            return userSecretId;
        }

        public final void setUserSecretId(String userSecretId) {
            this.userSecretId = userSecretId;
        }

        @Override
        public final Builder userSecretId(String userSecretId) {
            this.userSecretId = userSecretId;
            return this;
        }

        public final Collection<String> getTrustedHostKeys() {
            if (trustedHostKeys instanceof SdkAutoConstructList) {
                return null;
            }
            return trustedHostKeys;
        }

        public final void setTrustedHostKeys(Collection<String> trustedHostKeys) {
            this.trustedHostKeys = SftpConnectorTrustedHostKeyListCopier.copy(trustedHostKeys);
        }

        @Override
        public final Builder trustedHostKeys(Collection<String> trustedHostKeys) {
            this.trustedHostKeys = SftpConnectorTrustedHostKeyListCopier.copy(trustedHostKeys);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder trustedHostKeys(String... trustedHostKeys) {
            trustedHostKeys(Arrays.asList(trustedHostKeys));
            return this;
        }

        public final Integer getMaxConcurrentConnections() {
            return maxConcurrentConnections;
        }

        public final void setMaxConcurrentConnections(Integer maxConcurrentConnections) {
            this.maxConcurrentConnections = maxConcurrentConnections;
        }

        @Override
        public final Builder maxConcurrentConnections(Integer maxConcurrentConnections) {
            this.maxConcurrentConnections = maxConcurrentConnections;
            return this;
        }

        @Override
        public SftpConnectorConfig build() {
            return new SftpConnectorConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
