/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.translate;

import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.client.ClientExecutionParams;
import software.amazon.awssdk.core.client.ClientHandler;
import software.amazon.awssdk.core.client.SdkClientHandler;
import software.amazon.awssdk.core.config.ClientConfiguration;
import software.amazon.awssdk.core.config.SyncClientConfiguration;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.exception.SdkServiceException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.protocol.json.JsonClientMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorResponseMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorShapeMetadata;
import software.amazon.awssdk.core.protocol.json.JsonOperationMetadata;
import software.amazon.awssdk.core.protocol.json.SdkJsonProtocolFactory;
import software.amazon.awssdk.services.translate.model.InternalServerException;
import software.amazon.awssdk.services.translate.model.InvalidRequestException;
import software.amazon.awssdk.services.translate.model.ServiceUnavailableException;
import software.amazon.awssdk.services.translate.model.TextSizeLimitExceededException;
import software.amazon.awssdk.services.translate.model.TooManyRequestsException;
import software.amazon.awssdk.services.translate.model.TranslateException;
import software.amazon.awssdk.services.translate.model.TranslateTextRequest;
import software.amazon.awssdk.services.translate.model.TranslateTextResponse;
import software.amazon.awssdk.services.translate.model.UnsupportedLanguagePairException;
import software.amazon.awssdk.services.translate.transform.TranslateTextRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.TranslateTextResponseUnmarshaller;

/**
 * Internal implementation of {@link TranslateClient}.
 *
 * @see TranslateClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTranslateClient implements TranslateClient {
    private final ClientHandler clientHandler;

    private final SdkJsonProtocolFactory protocolFactory;

    private final ClientConfiguration clientConfiguration;

    protected DefaultTranslateClient(SyncClientConfiguration clientConfiguration) {
        this.clientHandler = new SdkClientHandler(clientConfiguration, null);
        this.protocolFactory = init();
        this.clientConfiguration = clientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Translates input text from the source language to the target language. You can translate between English (en) and
     * one of the following languages, or between one of the following languages and English.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Arabic (ar)
     * </p>
     * </li>
     * <li>
     * <p>
     * Chinese (Simplified) (zh)
     * </p>
     * </li>
     * <li>
     * <p>
     * French (fr)
     * </p>
     * </li>
     * <li>
     * <p>
     * German (de)
     * </p>
     * </li>
     * <li>
     * <p>
     * Portuguese (pt)
     * </p>
     * </li>
     * <li>
     * <p>
     * Spanish (es)
     * </p>
     * </li>
     * </ul>
     *
     * @param translateTextRequest
     * @return Result of the TranslateText operation returned by the service.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws TextSizeLimitExceededException
     *         The size of the input text exceeds the length constraint for the <code>Text</code> field. Try again with
     *         a shorter text.
     * @throws TooManyRequestsException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws UnsupportedLanguagePairException
     *         Amazon Translate cannot translate input text in the source language into this target language. For more
     *         information, see <a>how-to-error-msg</a>.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ServiceUnavailableException
     *         Amazon Translate is unavailable. Retry your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.TranslateText
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/TranslateText" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TranslateTextResponse translateText(TranslateTextRequest translateTextRequest) throws InvalidRequestException,
            TextSizeLimitExceededException, TooManyRequestsException, UnsupportedLanguagePairException, InternalServerException,
            ServiceUnavailableException, SdkServiceException, SdkClientException, TranslateException {

        HttpResponseHandler<TranslateTextResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new TranslateTextResponseUnmarshaller());

        HttpResponseHandler<SdkServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<TranslateTextRequest, TranslateTextResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(translateTextRequest).withMarshaller(new TranslateTextRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<SdkServiceException> createErrorResponseHandler() {
        return protocolFactory.createErrorResponseHandler(new JsonErrorResponseMetadata());
    }

    private software.amazon.awssdk.core.protocol.json.SdkJsonProtocolFactory init() {
        return new SdkJsonProtocolFactory(new JsonClientMetadata()
                .withProtocolVersion("1.1")
                .withSupportsCbor(false)
                .withSupportsIon(false)
                .withBaseServiceExceptionClass(software.amazon.awssdk.services.translate.model.TranslateException.class)
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("TooManyRequestsException").withModeledClass(
                                TooManyRequestsException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InternalServerException").withModeledClass(
                                InternalServerException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InvalidRequestException").withModeledClass(
                                InvalidRequestException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("TextSizeLimitExceededException").withModeledClass(
                                TextSizeLimitExceededException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("UnsupportedLanguagePairException").withModeledClass(
                                UnsupportedLanguagePairException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ServiceUnavailableException").withModeledClass(
                                ServiceUnavailableException.class)));
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
