/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.translate;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.translate.model.DeleteTerminologyRequest;
import software.amazon.awssdk.services.translate.model.DeleteTerminologyResponse;
import software.amazon.awssdk.services.translate.model.DescribeTextTranslationJobRequest;
import software.amazon.awssdk.services.translate.model.DescribeTextTranslationJobResponse;
import software.amazon.awssdk.services.translate.model.DetectedLanguageLowConfidenceException;
import software.amazon.awssdk.services.translate.model.GetTerminologyRequest;
import software.amazon.awssdk.services.translate.model.GetTerminologyResponse;
import software.amazon.awssdk.services.translate.model.ImportTerminologyRequest;
import software.amazon.awssdk.services.translate.model.ImportTerminologyResponse;
import software.amazon.awssdk.services.translate.model.InternalServerException;
import software.amazon.awssdk.services.translate.model.InvalidFilterException;
import software.amazon.awssdk.services.translate.model.InvalidParameterValueException;
import software.amazon.awssdk.services.translate.model.InvalidRequestException;
import software.amazon.awssdk.services.translate.model.LimitExceededException;
import software.amazon.awssdk.services.translate.model.ListTerminologiesRequest;
import software.amazon.awssdk.services.translate.model.ListTerminologiesResponse;
import software.amazon.awssdk.services.translate.model.ListTextTranslationJobsRequest;
import software.amazon.awssdk.services.translate.model.ListTextTranslationJobsResponse;
import software.amazon.awssdk.services.translate.model.ResourceNotFoundException;
import software.amazon.awssdk.services.translate.model.ServiceUnavailableException;
import software.amazon.awssdk.services.translate.model.StartTextTranslationJobRequest;
import software.amazon.awssdk.services.translate.model.StartTextTranslationJobResponse;
import software.amazon.awssdk.services.translate.model.StopTextTranslationJobRequest;
import software.amazon.awssdk.services.translate.model.StopTextTranslationJobResponse;
import software.amazon.awssdk.services.translate.model.TextSizeLimitExceededException;
import software.amazon.awssdk.services.translate.model.TooManyRequestsException;
import software.amazon.awssdk.services.translate.model.TranslateException;
import software.amazon.awssdk.services.translate.model.TranslateRequest;
import software.amazon.awssdk.services.translate.model.TranslateTextRequest;
import software.amazon.awssdk.services.translate.model.TranslateTextResponse;
import software.amazon.awssdk.services.translate.model.UnsupportedLanguagePairException;
import software.amazon.awssdk.services.translate.paginators.ListTerminologiesIterable;
import software.amazon.awssdk.services.translate.paginators.ListTextTranslationJobsIterable;
import software.amazon.awssdk.services.translate.transform.DeleteTerminologyRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.DescribeTextTranslationJobRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.GetTerminologyRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.ImportTerminologyRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.ListTerminologiesRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.ListTextTranslationJobsRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.StartTextTranslationJobRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.StopTextTranslationJobRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.TranslateTextRequestMarshaller;

/**
 * Internal implementation of {@link TranslateClient}.
 *
 * @see TranslateClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTranslateClient implements TranslateClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultTranslateClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * A synchronous action that deletes a custom terminology.
     * </p>
     *
     * @param deleteTerminologyRequest
     * @return Result of the DeleteTerminology operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource you are looking for has not been found. Review the resource you're looking for and see if a
     *         different resource will accomplish your needs before retrying the revised request.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.DeleteTerminology
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/DeleteTerminology" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteTerminologyResponse deleteTerminology(DeleteTerminologyRequest deleteTerminologyRequest)
            throws ResourceNotFoundException, TooManyRequestsException, InternalServerException, AwsServiceException,
            SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTerminologyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteTerminologyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTerminology");

            return clientHandler.execute(new ClientExecutionParams<DeleteTerminologyRequest, DeleteTerminologyResponse>()
                    .withOperationName("DeleteTerminology").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteTerminologyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteTerminologyRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTerminologyRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the properties associated with an asycnhronous batch translation job including name, ID, status, source and
     * target languages, input/output S3 buckets, and so on.
     * </p>
     *
     * @param describeTextTranslationJobRequest
     * @return Result of the DescribeTextTranslationJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource you are looking for has not been found. Review the resource you're looking for and see if a
     *         different resource will accomplish your needs before retrying the revised request.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.DescribeTextTranslationJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/DescribeTextTranslationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeTextTranslationJobResponse describeTextTranslationJob(
            DescribeTextTranslationJobRequest describeTextTranslationJobRequest) throws ResourceNotFoundException,
            TooManyRequestsException, InternalServerException, AwsServiceException, SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTextTranslationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeTextTranslationJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTextTranslationJob");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeTextTranslationJobRequest, DescribeTextTranslationJobResponse>()
                            .withOperationName("DescribeTextTranslationJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeTextTranslationJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeTextTranslationJobRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    describeTextTranslationJobRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a custom terminology.
     * </p>
     *
     * @param getTerminologyRequest
     * @return Result of the GetTerminology operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource you are looking for has not been found. Review the resource you're looking for and see if a
     *         different resource will accomplish your needs before retrying the revised request.
     * @throws InvalidParameterValueException
     *         The value of the parameter is invalid. Review the value of the parameter you are using to correct it, and
     *         then retry your operation.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.GetTerminology
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/GetTerminology" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTerminologyResponse getTerminology(GetTerminologyRequest getTerminologyRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, TooManyRequestsException, InternalServerException, AwsServiceException,
            SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTerminologyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTerminologyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTerminology");

            return clientHandler.execute(new ClientExecutionParams<GetTerminologyRequest, GetTerminologyResponse>()
                    .withOperationName("GetTerminology").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getTerminologyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTerminologyRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTerminologyRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates a custom terminology, depending on whether or not one already exists for the given terminology
     * name. Importing a terminology with the same name as an existing one will merge the terminologies based on the
     * chosen merge strategy. Currently, the only supported merge strategy is OVERWRITE, and so the imported terminology
     * will overwrite an existing terminology of the same name.
     * </p>
     * <p>
     * If you import a terminology that overwrites an existing one, the new terminology take up to 10 minutes to fully
     * propagate and be available for use in a translation due to cache policies with the DataPlane service that
     * performs the translations.
     * </p>
     *
     * @param importTerminologyRequest
     * @return Result of the ImportTerminology operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of the parameter is invalid. Review the value of the parameter you are using to correct it, and
     *         then retry your operation.
     * @throws LimitExceededException
     *         The specified limit has been exceeded. Review your request and retry it with a quantity below the stated
     *         limit.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.ImportTerminology
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ImportTerminology" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ImportTerminologyResponse importTerminology(ImportTerminologyRequest importTerminologyRequest)
            throws InvalidParameterValueException, LimitExceededException, TooManyRequestsException, InternalServerException,
            AwsServiceException, SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ImportTerminologyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ImportTerminologyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportTerminology");

            return clientHandler.execute(new ClientExecutionParams<ImportTerminologyRequest, ImportTerminologyResponse>()
                    .withOperationName("ImportTerminology").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(importTerminologyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ImportTerminologyRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, importTerminologyRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of custom terminologies associated with your account.
     * </p>
     *
     * @param listTerminologiesRequest
     * @return Result of the ListTerminologies operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of the parameter is invalid. Review the value of the parameter you are using to correct it, and
     *         then retry your operation.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.ListTerminologies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListTerminologies" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTerminologiesResponse listTerminologies(ListTerminologiesRequest listTerminologiesRequest)
            throws InvalidParameterValueException, TooManyRequestsException, InternalServerException, AwsServiceException,
            SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTerminologiesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTerminologiesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTerminologies");

            return clientHandler.execute(new ClientExecutionParams<ListTerminologiesRequest, ListTerminologiesResponse>()
                    .withOperationName("ListTerminologies").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTerminologiesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTerminologiesRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTerminologiesRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of custom terminologies associated with your account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTerminologies(software.amazon.awssdk.services.translate.model.ListTerminologiesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.translate.paginators.ListTerminologiesIterable responses = client.listTerminologiesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.translate.paginators.ListTerminologiesIterable responses = client
     *             .listTerminologiesPaginator(request);
     *     for (software.amazon.awssdk.services.translate.model.ListTerminologiesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.translate.paginators.ListTerminologiesIterable responses = client.listTerminologiesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTerminologies(software.amazon.awssdk.services.translate.model.ListTerminologiesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTerminologiesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterValueException
     *         The value of the parameter is invalid. Review the value of the parameter you are using to correct it, and
     *         then retry your operation.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.ListTerminologies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListTerminologies" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTerminologiesIterable listTerminologiesPaginator(ListTerminologiesRequest listTerminologiesRequest)
            throws InvalidParameterValueException, TooManyRequestsException, InternalServerException, AwsServiceException,
            SdkClientException, TranslateException {
        return new ListTerminologiesIterable(this, applyPaginatorUserAgent(listTerminologiesRequest));
    }

    /**
     * <p>
     * Gets a list of the batch translation jobs that you have submitted.
     * </p>
     *
     * @param listTextTranslationJobsRequest
     * @return Result of the ListTextTranslationJobs operation returned by the service.
     * @throws InvalidRequestException
     *         The request that you made is invalid. Check your request to determine why it's invalid and then retry the
     *         request.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws InvalidFilterException
     *         The filter specified for the operation is invalid. Specify a different filter.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.ListTextTranslationJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListTextTranslationJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTextTranslationJobsResponse listTextTranslationJobs(ListTextTranslationJobsRequest listTextTranslationJobsRequest)
            throws InvalidRequestException, TooManyRequestsException, InvalidFilterException, InternalServerException,
            AwsServiceException, SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTextTranslationJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTextTranslationJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTextTranslationJobs");

            return clientHandler
                    .execute(new ClientExecutionParams<ListTextTranslationJobsRequest, ListTextTranslationJobsResponse>()
                            .withOperationName("ListTextTranslationJobs").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listTextTranslationJobsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListTextTranslationJobsRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTextTranslationJobsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a list of the batch translation jobs that you have submitted.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTextTranslationJobs(software.amazon.awssdk.services.translate.model.ListTextTranslationJobsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.translate.paginators.ListTextTranslationJobsIterable responses = client.listTextTranslationJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.translate.paginators.ListTextTranslationJobsIterable responses = client
     *             .listTextTranslationJobsPaginator(request);
     *     for (software.amazon.awssdk.services.translate.model.ListTextTranslationJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.translate.paginators.ListTextTranslationJobsIterable responses = client.listTextTranslationJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTextTranslationJobs(software.amazon.awssdk.services.translate.model.ListTextTranslationJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTextTranslationJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request that you made is invalid. Check your request to determine why it's invalid and then retry the
     *         request.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws InvalidFilterException
     *         The filter specified for the operation is invalid. Specify a different filter.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.ListTextTranslationJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListTextTranslationJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTextTranslationJobsIterable listTextTranslationJobsPaginator(
            ListTextTranslationJobsRequest listTextTranslationJobsRequest) throws InvalidRequestException,
            TooManyRequestsException, InvalidFilterException, InternalServerException, AwsServiceException, SdkClientException,
            TranslateException {
        return new ListTextTranslationJobsIterable(this, applyPaginatorUserAgent(listTextTranslationJobsRequest));
    }

    /**
     * <p>
     * Starts an asynchronous batch translation job. Batch translation jobs can be used to translate large volumes of
     * text across multiple documents at once. For more information, see <a>async</a>.
     * </p>
     * <p>
     * Batch translation jobs can be described with the <a>DescribeTextTranslationJob</a> operation, listed with the
     * <a>ListTextTranslationJobs</a> operation, and stopped with the <a>StopTextTranslationJob</a> operation.
     * </p>
     * <note>
     * <p>
     * Amazon Translate does not support batch translation of multiple source languages at once.
     * </p>
     * </note>
     *
     * @param startTextTranslationJobRequest
     * @return Result of the StartTextTranslationJob operation returned by the service.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws UnsupportedLanguagePairException
     *         Amazon Translate does not support translation from the language of the source text into the requested
     *         target language. For more information, see <a>how-to-error-msg</a>.
     * @throws InvalidRequestException
     *         The request that you made is invalid. Check your request to determine why it's invalid and then retry the
     *         request.
     * @throws ResourceNotFoundException
     *         The resource you are looking for has not been found. Review the resource you're looking for and see if a
     *         different resource will accomplish your needs before retrying the revised request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.StartTextTranslationJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/StartTextTranslationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartTextTranslationJobResponse startTextTranslationJob(StartTextTranslationJobRequest startTextTranslationJobRequest)
            throws TooManyRequestsException, UnsupportedLanguagePairException, InvalidRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartTextTranslationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartTextTranslationJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartTextTranslationJob");

            return clientHandler
                    .execute(new ClientExecutionParams<StartTextTranslationJobRequest, StartTextTranslationJobResponse>()
                            .withOperationName("StartTextTranslationJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(startTextTranslationJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartTextTranslationJobRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startTextTranslationJobRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops an asynchronous batch translation job that is in progress.
     * </p>
     * <p>
     * If the job's state is <code>IN_PROGRESS</code>, the job will be marked for termination and put into the
     * <code>STOP_REQUESTED</code> state. If the job completes before it can be stopped, it is put into the
     * <code>COMPLETED</code> state. Otherwise, the job is put into the <code>STOPPED</code> state.
     * </p>
     * <p>
     * Asynchronous batch translation jobs are started with the <a>StartTextTranslationJob</a> operation. You can use
     * the <a>DescribeTextTranslationJob</a> or <a>ListTextTranslationJobs</a> operations to get a batch translation
     * job's <code>JobId</code>.
     * </p>
     *
     * @param stopTextTranslationJobRequest
     * @return Result of the StopTextTranslationJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource you are looking for has not been found. Review the resource you're looking for and see if a
     *         different resource will accomplish your needs before retrying the revised request.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.StopTextTranslationJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/StopTextTranslationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StopTextTranslationJobResponse stopTextTranslationJob(StopTextTranslationJobRequest stopTextTranslationJobRequest)
            throws ResourceNotFoundException, TooManyRequestsException, InternalServerException, AwsServiceException,
            SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopTextTranslationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StopTextTranslationJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopTextTranslationJob");

            return clientHandler
                    .execute(new ClientExecutionParams<StopTextTranslationJobRequest, StopTextTranslationJobResponse>()
                            .withOperationName("StopTextTranslationJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(stopTextTranslationJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StopTextTranslationJobRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopTextTranslationJobRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Translates input text from the source language to the target language. For a list of available languages and
     * language codes, see <a>what-is-languages</a>.
     * </p>
     *
     * @param translateTextRequest
     * @return Result of the TranslateText operation returned by the service.
     * @throws InvalidRequestException
     *         The request that you made is invalid. Check your request to determine why it's invalid and then retry the
     *         request.
     * @throws TextSizeLimitExceededException
     *         The size of the text you submitted exceeds the size limit. Reduce the size of the text or use a smaller
     *         document and then retry your request.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws UnsupportedLanguagePairException
     *         Amazon Translate does not support translation from the language of the source text into the requested
     *         target language. For more information, see <a>how-to-error-msg</a>.
     * @throws DetectedLanguageLowConfidenceException
     *         The confidence that Amazon Comprehend accurately detected the source language is low. If a low confidence
     *         level is acceptable for your application, you can use the language in the exception to call Amazon
     *         Translate again. For more information, see the <a
     *         href="https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectDominantLanguage.html"
     *         >DetectDominantLanguage</a> operation in the <i>Amazon Comprehend Developer Guide</i>.
     * @throws ResourceNotFoundException
     *         The resource you are looking for has not been found. Review the resource you're looking for and see if a
     *         different resource will accomplish your needs before retrying the revised request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ServiceUnavailableException
     *         The Amazon Translate service is temporarily unavailable. Please wait a bit and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.TranslateText
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/TranslateText" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TranslateTextResponse translateText(TranslateTextRequest translateTextRequest) throws InvalidRequestException,
            TextSizeLimitExceededException, TooManyRequestsException, UnsupportedLanguagePairException,
            DetectedLanguageLowConfidenceException, ResourceNotFoundException, InternalServerException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TranslateTextResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TranslateTextResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TranslateText");

            return clientHandler.execute(new ClientExecutionParams<TranslateTextRequest, TranslateTextResponse>()
                    .withOperationName("TranslateText").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(translateTextRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TranslateTextRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, translateTextRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(TranslateException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidFilterException")
                                .exceptionBuilderSupplier(InvalidFilterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DetectedLanguageLowConfidenceException")
                                .exceptionBuilderSupplier(DetectedLanguageLowConfidenceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                                .exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedLanguagePairException")
                                .exceptionBuilderSupplier(UnsupportedLanguagePairException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends TranslateRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
