/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.translate.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The properties of a parallel data resource.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ParallelDataProperties implements SdkPojo, Serializable,
        ToCopyableBuilder<ParallelDataProperties.Builder, ParallelDataProperties> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ParallelDataProperties::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(ParallelDataProperties::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(ParallelDataProperties::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(ParallelDataProperties::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> SOURCE_LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceLanguageCode").getter(getter(ParallelDataProperties::sourceLanguageCode))
            .setter(setter(Builder::sourceLanguageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceLanguageCode").build())
            .build();

    private static final SdkField<List<String>> TARGET_LANGUAGE_CODES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("TargetLanguageCodes")
            .getter(getter(ParallelDataProperties::targetLanguageCodes))
            .setter(setter(Builder::targetLanguageCodes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetLanguageCodes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ParallelDataConfig> PARALLEL_DATA_CONFIG_FIELD = SdkField
            .<ParallelDataConfig> builder(MarshallingType.SDK_POJO).memberName("ParallelDataConfig")
            .getter(getter(ParallelDataProperties::parallelDataConfig)).setter(setter(Builder::parallelDataConfig))
            .constructor(ParallelDataConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParallelDataConfig").build())
            .build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Message")
            .getter(getter(ParallelDataProperties::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Message").build()).build();

    private static final SdkField<Long> IMPORTED_DATA_SIZE_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("ImportedDataSize").getter(getter(ParallelDataProperties::importedDataSize))
            .setter(setter(Builder::importedDataSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImportedDataSize").build()).build();

    private static final SdkField<Long> IMPORTED_RECORD_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("ImportedRecordCount").getter(getter(ParallelDataProperties::importedRecordCount))
            .setter(setter(Builder::importedRecordCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImportedRecordCount").build())
            .build();

    private static final SdkField<Long> FAILED_RECORD_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("FailedRecordCount").getter(getter(ParallelDataProperties::failedRecordCount))
            .setter(setter(Builder::failedRecordCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailedRecordCount").build()).build();

    private static final SdkField<Long> SKIPPED_RECORD_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("SkippedRecordCount").getter(getter(ParallelDataProperties::skippedRecordCount))
            .setter(setter(Builder::skippedRecordCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SkippedRecordCount").build())
            .build();

    private static final SdkField<EncryptionKey> ENCRYPTION_KEY_FIELD = SdkField
            .<EncryptionKey> builder(MarshallingType.SDK_POJO).memberName("EncryptionKey")
            .getter(getter(ParallelDataProperties::encryptionKey)).setter(setter(Builder::encryptionKey))
            .constructor(EncryptionKey::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionKey").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAt").getter(getter(ParallelDataProperties::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedAt").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdatedAt").getter(getter(ParallelDataProperties::lastUpdatedAt))
            .setter(setter(Builder::lastUpdatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedAt").build()).build();

    private static final SdkField<String> LATEST_UPDATE_ATTEMPT_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LatestUpdateAttemptStatus").getter(getter(ParallelDataProperties::latestUpdateAttemptStatusAsString))
            .setter(setter(Builder::latestUpdateAttemptStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LatestUpdateAttemptStatus").build())
            .build();

    private static final SdkField<Instant> LATEST_UPDATE_ATTEMPT_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LatestUpdateAttemptAt").getter(getter(ParallelDataProperties::latestUpdateAttemptAt))
            .setter(setter(Builder::latestUpdateAttemptAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LatestUpdateAttemptAt").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ARN_FIELD,
            DESCRIPTION_FIELD, STATUS_FIELD, SOURCE_LANGUAGE_CODE_FIELD, TARGET_LANGUAGE_CODES_FIELD, PARALLEL_DATA_CONFIG_FIELD,
            MESSAGE_FIELD, IMPORTED_DATA_SIZE_FIELD, IMPORTED_RECORD_COUNT_FIELD, FAILED_RECORD_COUNT_FIELD,
            SKIPPED_RECORD_COUNT_FIELD, ENCRYPTION_KEY_FIELD, CREATED_AT_FIELD, LAST_UPDATED_AT_FIELD,
            LATEST_UPDATE_ATTEMPT_STATUS_FIELD, LATEST_UPDATE_ATTEMPT_AT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String arn;

    private final String description;

    private final String status;

    private final String sourceLanguageCode;

    private final List<String> targetLanguageCodes;

    private final ParallelDataConfig parallelDataConfig;

    private final String message;

    private final Long importedDataSize;

    private final Long importedRecordCount;

    private final Long failedRecordCount;

    private final Long skippedRecordCount;

    private final EncryptionKey encryptionKey;

    private final Instant createdAt;

    private final Instant lastUpdatedAt;

    private final String latestUpdateAttemptStatus;

    private final Instant latestUpdateAttemptAt;

    private ParallelDataProperties(BuilderImpl builder) {
        this.name = builder.name;
        this.arn = builder.arn;
        this.description = builder.description;
        this.status = builder.status;
        this.sourceLanguageCode = builder.sourceLanguageCode;
        this.targetLanguageCodes = builder.targetLanguageCodes;
        this.parallelDataConfig = builder.parallelDataConfig;
        this.message = builder.message;
        this.importedDataSize = builder.importedDataSize;
        this.importedRecordCount = builder.importedRecordCount;
        this.failedRecordCount = builder.failedRecordCount;
        this.skippedRecordCount = builder.skippedRecordCount;
        this.encryptionKey = builder.encryptionKey;
        this.createdAt = builder.createdAt;
        this.lastUpdatedAt = builder.lastUpdatedAt;
        this.latestUpdateAttemptStatus = builder.latestUpdateAttemptStatus;
        this.latestUpdateAttemptAt = builder.latestUpdateAttemptAt;
    }

    /**
     * <p>
     * The custom name assigned to the parallel data resource.
     * </p>
     * 
     * @return The custom name assigned to the parallel data resource.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the parallel data resource.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the parallel data resource.
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * The description assigned to the parallel data resource.
     * </p>
     * 
     * @return The description assigned to the parallel data resource.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The status of the parallel data resource. When the parallel data is ready for you to use, the status is
     * <code>ACTIVE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ParallelDataStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the parallel data resource. When the parallel data is ready for you to use, the status is
     *         <code>ACTIVE</code>.
     * @see ParallelDataStatus
     */
    public ParallelDataStatus status() {
        return ParallelDataStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the parallel data resource. When the parallel data is ready for you to use, the status is
     * <code>ACTIVE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ParallelDataStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the parallel data resource. When the parallel data is ready for you to use, the status is
     *         <code>ACTIVE</code>.
     * @see ParallelDataStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The source language of the translations in the parallel data file.
     * </p>
     * 
     * @return The source language of the translations in the parallel data file.
     */
    public String sourceLanguageCode() {
        return sourceLanguageCode;
    }

    /**
     * Returns true if the TargetLanguageCodes property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasTargetLanguageCodes() {
        return targetLanguageCodes != null && !(targetLanguageCodes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The language codes for the target languages available in the parallel data file. All possible target languages
     * are returned as an array.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTargetLanguageCodes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The language codes for the target languages available in the parallel data file. All possible target
     *         languages are returned as an array.
     */
    public List<String> targetLanguageCodes() {
        return targetLanguageCodes;
    }

    /**
     * <p>
     * Specifies the format and S3 location of the parallel data input file.
     * </p>
     * 
     * @return Specifies the format and S3 location of the parallel data input file.
     */
    public ParallelDataConfig parallelDataConfig() {
        return parallelDataConfig;
    }

    /**
     * <p>
     * Additional information from Amazon Translate about the parallel data resource.
     * </p>
     * 
     * @return Additional information from Amazon Translate about the parallel data resource.
     */
    public String message() {
        return message;
    }

    /**
     * <p>
     * The number of UTF-8 characters that Amazon Translate imported from the parallel data input file. This number
     * includes only the characters in your translation examples. It does not include characters that are used to format
     * your file. For example, if you provided a Translation Memory Exchange (.tmx) file, this number does not include
     * the tags.
     * </p>
     * 
     * @return The number of UTF-8 characters that Amazon Translate imported from the parallel data input file. This
     *         number includes only the characters in your translation examples. It does not include characters that are
     *         used to format your file. For example, if you provided a Translation Memory Exchange (.tmx) file, this
     *         number does not include the tags.
     */
    public Long importedDataSize() {
        return importedDataSize;
    }

    /**
     * <p>
     * The number of records successfully imported from the parallel data input file.
     * </p>
     * 
     * @return The number of records successfully imported from the parallel data input file.
     */
    public Long importedRecordCount() {
        return importedRecordCount;
    }

    /**
     * <p>
     * The number of records unsuccessfully imported from the parallel data input file.
     * </p>
     * 
     * @return The number of records unsuccessfully imported from the parallel data input file.
     */
    public Long failedRecordCount() {
        return failedRecordCount;
    }

    /**
     * <p>
     * The number of items in the input file that Amazon Translate skipped when you created or updated the parallel data
     * resource. For example, Amazon Translate skips empty records, empty target texts, and empty lines.
     * </p>
     * 
     * @return The number of items in the input file that Amazon Translate skipped when you created or updated the
     *         parallel data resource. For example, Amazon Translate skips empty records, empty target texts, and empty
     *         lines.
     */
    public Long skippedRecordCount() {
        return skippedRecordCount;
    }

    /**
     * Returns the value of the EncryptionKey property for this object.
     * 
     * @return The value of the EncryptionKey property for this object.
     */
    public EncryptionKey encryptionKey() {
        return encryptionKey;
    }

    /**
     * <p>
     * The time at which the parallel data resource was created.
     * </p>
     * 
     * @return The time at which the parallel data resource was created.
     */
    public Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The time at which the parallel data resource was last updated.
     * </p>
     * 
     * @return The time at which the parallel data resource was last updated.
     */
    public Instant lastUpdatedAt() {
        return lastUpdatedAt;
    }

    /**
     * <p>
     * The status of the most recent update attempt for the parallel data resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #latestUpdateAttemptStatus} will return {@link ParallelDataStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #latestUpdateAttemptStatusAsString}.
     * </p>
     * 
     * @return The status of the most recent update attempt for the parallel data resource.
     * @see ParallelDataStatus
     */
    public ParallelDataStatus latestUpdateAttemptStatus() {
        return ParallelDataStatus.fromValue(latestUpdateAttemptStatus);
    }

    /**
     * <p>
     * The status of the most recent update attempt for the parallel data resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #latestUpdateAttemptStatus} will return {@link ParallelDataStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #latestUpdateAttemptStatusAsString}.
     * </p>
     * 
     * @return The status of the most recent update attempt for the parallel data resource.
     * @see ParallelDataStatus
     */
    public String latestUpdateAttemptStatusAsString() {
        return latestUpdateAttemptStatus;
    }

    /**
     * <p>
     * The time that the most recent update was attempted.
     * </p>
     * 
     * @return The time that the most recent update was attempted.
     */
    public Instant latestUpdateAttemptAt() {
        return latestUpdateAttemptAt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(sourceLanguageCode());
        hashCode = 31 * hashCode + Objects.hashCode(hasTargetLanguageCodes() ? targetLanguageCodes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(parallelDataConfig());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(importedDataSize());
        hashCode = 31 * hashCode + Objects.hashCode(importedRecordCount());
        hashCode = 31 * hashCode + Objects.hashCode(failedRecordCount());
        hashCode = 31 * hashCode + Objects.hashCode(skippedRecordCount());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionKey());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(latestUpdateAttemptStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(latestUpdateAttemptAt());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ParallelDataProperties)) {
            return false;
        }
        ParallelDataProperties other = (ParallelDataProperties) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn())
                && Objects.equals(description(), other.description()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(sourceLanguageCode(), other.sourceLanguageCode())
                && hasTargetLanguageCodes() == other.hasTargetLanguageCodes()
                && Objects.equals(targetLanguageCodes(), other.targetLanguageCodes())
                && Objects.equals(parallelDataConfig(), other.parallelDataConfig()) && Objects.equals(message(), other.message())
                && Objects.equals(importedDataSize(), other.importedDataSize())
                && Objects.equals(importedRecordCount(), other.importedRecordCount())
                && Objects.equals(failedRecordCount(), other.failedRecordCount())
                && Objects.equals(skippedRecordCount(), other.skippedRecordCount())
                && Objects.equals(encryptionKey(), other.encryptionKey()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(lastUpdatedAt(), other.lastUpdatedAt())
                && Objects.equals(latestUpdateAttemptStatusAsString(), other.latestUpdateAttemptStatusAsString())
                && Objects.equals(latestUpdateAttemptAt(), other.latestUpdateAttemptAt());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ParallelDataProperties").add("Name", name()).add("Arn", arn()).add("Description", description())
                .add("Status", statusAsString()).add("SourceLanguageCode", sourceLanguageCode())
                .add("TargetLanguageCodes", hasTargetLanguageCodes() ? targetLanguageCodes() : null)
                .add("ParallelDataConfig", parallelDataConfig()).add("Message", message())
                .add("ImportedDataSize", importedDataSize()).add("ImportedRecordCount", importedRecordCount())
                .add("FailedRecordCount", failedRecordCount()).add("SkippedRecordCount", skippedRecordCount())
                .add("EncryptionKey", encryptionKey()).add("CreatedAt", createdAt()).add("LastUpdatedAt", lastUpdatedAt())
                .add("LatestUpdateAttemptStatus", latestUpdateAttemptStatusAsString())
                .add("LatestUpdateAttemptAt", latestUpdateAttemptAt()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "SourceLanguageCode":
            return Optional.ofNullable(clazz.cast(sourceLanguageCode()));
        case "TargetLanguageCodes":
            return Optional.ofNullable(clazz.cast(targetLanguageCodes()));
        case "ParallelDataConfig":
            return Optional.ofNullable(clazz.cast(parallelDataConfig()));
        case "Message":
            return Optional.ofNullable(clazz.cast(message()));
        case "ImportedDataSize":
            return Optional.ofNullable(clazz.cast(importedDataSize()));
        case "ImportedRecordCount":
            return Optional.ofNullable(clazz.cast(importedRecordCount()));
        case "FailedRecordCount":
            return Optional.ofNullable(clazz.cast(failedRecordCount()));
        case "SkippedRecordCount":
            return Optional.ofNullable(clazz.cast(skippedRecordCount()));
        case "EncryptionKey":
            return Optional.ofNullable(clazz.cast(encryptionKey()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "LastUpdatedAt":
            return Optional.ofNullable(clazz.cast(lastUpdatedAt()));
        case "LatestUpdateAttemptStatus":
            return Optional.ofNullable(clazz.cast(latestUpdateAttemptStatusAsString()));
        case "LatestUpdateAttemptAt":
            return Optional.ofNullable(clazz.cast(latestUpdateAttemptAt()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ParallelDataProperties, T> g) {
        return obj -> g.apply((ParallelDataProperties) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ParallelDataProperties> {
        /**
         * <p>
         * The custom name assigned to the parallel data resource.
         * </p>
         * 
         * @param name
         *        The custom name assigned to the parallel data resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the parallel data resource.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the parallel data resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The description assigned to the parallel data resource.
         * </p>
         * 
         * @param description
         *        The description assigned to the parallel data resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The status of the parallel data resource. When the parallel data is ready for you to use, the status is
         * <code>ACTIVE</code>.
         * </p>
         * 
         * @param status
         *        The status of the parallel data resource. When the parallel data is ready for you to use, the status
         *        is <code>ACTIVE</code>.
         * @see ParallelDataStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParallelDataStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the parallel data resource. When the parallel data is ready for you to use, the status is
         * <code>ACTIVE</code>.
         * </p>
         * 
         * @param status
         *        The status of the parallel data resource. When the parallel data is ready for you to use, the status
         *        is <code>ACTIVE</code>.
         * @see ParallelDataStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParallelDataStatus
         */
        Builder status(ParallelDataStatus status);

        /**
         * <p>
         * The source language of the translations in the parallel data file.
         * </p>
         * 
         * @param sourceLanguageCode
         *        The source language of the translations in the parallel data file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceLanguageCode(String sourceLanguageCode);

        /**
         * <p>
         * The language codes for the target languages available in the parallel data file. All possible target
         * languages are returned as an array.
         * </p>
         * 
         * @param targetLanguageCodes
         *        The language codes for the target languages available in the parallel data file. All possible target
         *        languages are returned as an array.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetLanguageCodes(Collection<String> targetLanguageCodes);

        /**
         * <p>
         * The language codes for the target languages available in the parallel data file. All possible target
         * languages are returned as an array.
         * </p>
         * 
         * @param targetLanguageCodes
         *        The language codes for the target languages available in the parallel data file. All possible target
         *        languages are returned as an array.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetLanguageCodes(String... targetLanguageCodes);

        /**
         * <p>
         * Specifies the format and S3 location of the parallel data input file.
         * </p>
         * 
         * @param parallelDataConfig
         *        Specifies the format and S3 location of the parallel data input file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parallelDataConfig(ParallelDataConfig parallelDataConfig);

        /**
         * <p>
         * Specifies the format and S3 location of the parallel data input file.
         * </p>
         * This is a convenience that creates an instance of the {@link ParallelDataConfig.Builder} avoiding the need to
         * create one manually via {@link ParallelDataConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link ParallelDataConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #parallelDataConfig(ParallelDataConfig)}.
         * 
         * @param parallelDataConfig
         *        a consumer that will call methods on {@link ParallelDataConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parallelDataConfig(ParallelDataConfig)
         */
        default Builder parallelDataConfig(Consumer<ParallelDataConfig.Builder> parallelDataConfig) {
            return parallelDataConfig(ParallelDataConfig.builder().applyMutation(parallelDataConfig).build());
        }

        /**
         * <p>
         * Additional information from Amazon Translate about the parallel data resource.
         * </p>
         * 
         * @param message
         *        Additional information from Amazon Translate about the parallel data resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);

        /**
         * <p>
         * The number of UTF-8 characters that Amazon Translate imported from the parallel data input file. This number
         * includes only the characters in your translation examples. It does not include characters that are used to
         * format your file. For example, if you provided a Translation Memory Exchange (.tmx) file, this number does
         * not include the tags.
         * </p>
         * 
         * @param importedDataSize
         *        The number of UTF-8 characters that Amazon Translate imported from the parallel data input file. This
         *        number includes only the characters in your translation examples. It does not include characters that
         *        are used to format your file. For example, if you provided a Translation Memory Exchange (.tmx) file,
         *        this number does not include the tags.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder importedDataSize(Long importedDataSize);

        /**
         * <p>
         * The number of records successfully imported from the parallel data input file.
         * </p>
         * 
         * @param importedRecordCount
         *        The number of records successfully imported from the parallel data input file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder importedRecordCount(Long importedRecordCount);

        /**
         * <p>
         * The number of records unsuccessfully imported from the parallel data input file.
         * </p>
         * 
         * @param failedRecordCount
         *        The number of records unsuccessfully imported from the parallel data input file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failedRecordCount(Long failedRecordCount);

        /**
         * <p>
         * The number of items in the input file that Amazon Translate skipped when you created or updated the parallel
         * data resource. For example, Amazon Translate skips empty records, empty target texts, and empty lines.
         * </p>
         * 
         * @param skippedRecordCount
         *        The number of items in the input file that Amazon Translate skipped when you created or updated the
         *        parallel data resource. For example, Amazon Translate skips empty records, empty target texts, and
         *        empty lines.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder skippedRecordCount(Long skippedRecordCount);

        /**
         * Sets the value of the EncryptionKey property for this object.
         *
         * @param encryptionKey
         *        The new value for the EncryptionKey property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionKey(EncryptionKey encryptionKey);

        /**
         * Sets the value of the EncryptionKey property for this object.
         *
         * This is a convenience that creates an instance of the {@link EncryptionKey.Builder} avoiding the need to
         * create one manually via {@link EncryptionKey#builder()}.
         *
         * When the {@link Consumer} completes, {@link EncryptionKey.Builder#build()} is called immediately and its
         * result is passed to {@link #encryptionKey(EncryptionKey)}.
         * 
         * @param encryptionKey
         *        a consumer that will call methods on {@link EncryptionKey.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionKey(EncryptionKey)
         */
        default Builder encryptionKey(Consumer<EncryptionKey.Builder> encryptionKey) {
            return encryptionKey(EncryptionKey.builder().applyMutation(encryptionKey).build());
        }

        /**
         * <p>
         * The time at which the parallel data resource was created.
         * </p>
         * 
         * @param createdAt
         *        The time at which the parallel data resource was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The time at which the parallel data resource was last updated.
         * </p>
         * 
         * @param lastUpdatedAt
         *        The time at which the parallel data resource was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedAt(Instant lastUpdatedAt);

        /**
         * <p>
         * The status of the most recent update attempt for the parallel data resource.
         * </p>
         * 
         * @param latestUpdateAttemptStatus
         *        The status of the most recent update attempt for the parallel data resource.
         * @see ParallelDataStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParallelDataStatus
         */
        Builder latestUpdateAttemptStatus(String latestUpdateAttemptStatus);

        /**
         * <p>
         * The status of the most recent update attempt for the parallel data resource.
         * </p>
         * 
         * @param latestUpdateAttemptStatus
         *        The status of the most recent update attempt for the parallel data resource.
         * @see ParallelDataStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParallelDataStatus
         */
        Builder latestUpdateAttemptStatus(ParallelDataStatus latestUpdateAttemptStatus);

        /**
         * <p>
         * The time that the most recent update was attempted.
         * </p>
         * 
         * @param latestUpdateAttemptAt
         *        The time that the most recent update was attempted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder latestUpdateAttemptAt(Instant latestUpdateAttemptAt);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String arn;

        private String description;

        private String status;

        private String sourceLanguageCode;

        private List<String> targetLanguageCodes = DefaultSdkAutoConstructList.getInstance();

        private ParallelDataConfig parallelDataConfig;

        private String message;

        private Long importedDataSize;

        private Long importedRecordCount;

        private Long failedRecordCount;

        private Long skippedRecordCount;

        private EncryptionKey encryptionKey;

        private Instant createdAt;

        private Instant lastUpdatedAt;

        private String latestUpdateAttemptStatus;

        private Instant latestUpdateAttemptAt;

        private BuilderImpl() {
        }

        private BuilderImpl(ParallelDataProperties model) {
            name(model.name);
            arn(model.arn);
            description(model.description);
            status(model.status);
            sourceLanguageCode(model.sourceLanguageCode);
            targetLanguageCodes(model.targetLanguageCodes);
            parallelDataConfig(model.parallelDataConfig);
            message(model.message);
            importedDataSize(model.importedDataSize);
            importedRecordCount(model.importedRecordCount);
            failedRecordCount(model.failedRecordCount);
            skippedRecordCount(model.skippedRecordCount);
            encryptionKey(model.encryptionKey);
            createdAt(model.createdAt);
            lastUpdatedAt(model.lastUpdatedAt);
            latestUpdateAttemptStatus(model.latestUpdateAttemptStatus);
            latestUpdateAttemptAt(model.latestUpdateAttemptAt);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ParallelDataStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getSourceLanguageCode() {
            return sourceLanguageCode;
        }

        @Override
        public final Builder sourceLanguageCode(String sourceLanguageCode) {
            this.sourceLanguageCode = sourceLanguageCode;
            return this;
        }

        public final void setSourceLanguageCode(String sourceLanguageCode) {
            this.sourceLanguageCode = sourceLanguageCode;
        }

        public final Collection<String> getTargetLanguageCodes() {
            if (targetLanguageCodes instanceof SdkAutoConstructList) {
                return null;
            }
            return targetLanguageCodes;
        }

        @Override
        public final Builder targetLanguageCodes(Collection<String> targetLanguageCodes) {
            this.targetLanguageCodes = LanguageCodeStringListCopier.copy(targetLanguageCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetLanguageCodes(String... targetLanguageCodes) {
            targetLanguageCodes(Arrays.asList(targetLanguageCodes));
            return this;
        }

        public final void setTargetLanguageCodes(Collection<String> targetLanguageCodes) {
            this.targetLanguageCodes = LanguageCodeStringListCopier.copy(targetLanguageCodes);
        }

        public final ParallelDataConfig.Builder getParallelDataConfig() {
            return parallelDataConfig != null ? parallelDataConfig.toBuilder() : null;
        }

        @Override
        public final Builder parallelDataConfig(ParallelDataConfig parallelDataConfig) {
            this.parallelDataConfig = parallelDataConfig;
            return this;
        }

        public final void setParallelDataConfig(ParallelDataConfig.BuilderImpl parallelDataConfig) {
            this.parallelDataConfig = parallelDataConfig != null ? parallelDataConfig.build() : null;
        }

        public final String getMessage() {
            return message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        public final Long getImportedDataSize() {
            return importedDataSize;
        }

        @Override
        public final Builder importedDataSize(Long importedDataSize) {
            this.importedDataSize = importedDataSize;
            return this;
        }

        public final void setImportedDataSize(Long importedDataSize) {
            this.importedDataSize = importedDataSize;
        }

        public final Long getImportedRecordCount() {
            return importedRecordCount;
        }

        @Override
        public final Builder importedRecordCount(Long importedRecordCount) {
            this.importedRecordCount = importedRecordCount;
            return this;
        }

        public final void setImportedRecordCount(Long importedRecordCount) {
            this.importedRecordCount = importedRecordCount;
        }

        public final Long getFailedRecordCount() {
            return failedRecordCount;
        }

        @Override
        public final Builder failedRecordCount(Long failedRecordCount) {
            this.failedRecordCount = failedRecordCount;
            return this;
        }

        public final void setFailedRecordCount(Long failedRecordCount) {
            this.failedRecordCount = failedRecordCount;
        }

        public final Long getSkippedRecordCount() {
            return skippedRecordCount;
        }

        @Override
        public final Builder skippedRecordCount(Long skippedRecordCount) {
            this.skippedRecordCount = skippedRecordCount;
            return this;
        }

        public final void setSkippedRecordCount(Long skippedRecordCount) {
            this.skippedRecordCount = skippedRecordCount;
        }

        public final EncryptionKey.Builder getEncryptionKey() {
            return encryptionKey != null ? encryptionKey.toBuilder() : null;
        }

        @Override
        public final Builder encryptionKey(EncryptionKey encryptionKey) {
            this.encryptionKey = encryptionKey;
            return this;
        }

        public final void setEncryptionKey(EncryptionKey.BuilderImpl encryptionKey) {
            this.encryptionKey = encryptionKey != null ? encryptionKey.build() : null;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        public final Instant getLastUpdatedAt() {
            return lastUpdatedAt;
        }

        @Override
        public final Builder lastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
            return this;
        }

        public final void setLastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
        }

        public final String getLatestUpdateAttemptStatus() {
            return latestUpdateAttemptStatus;
        }

        @Override
        public final Builder latestUpdateAttemptStatus(String latestUpdateAttemptStatus) {
            this.latestUpdateAttemptStatus = latestUpdateAttemptStatus;
            return this;
        }

        @Override
        public final Builder latestUpdateAttemptStatus(ParallelDataStatus latestUpdateAttemptStatus) {
            this.latestUpdateAttemptStatus(latestUpdateAttemptStatus == null ? null : latestUpdateAttemptStatus.toString());
            return this;
        }

        public final void setLatestUpdateAttemptStatus(String latestUpdateAttemptStatus) {
            this.latestUpdateAttemptStatus = latestUpdateAttemptStatus;
        }

        public final Instant getLatestUpdateAttemptAt() {
            return latestUpdateAttemptAt;
        }

        @Override
        public final Builder latestUpdateAttemptAt(Instant latestUpdateAttemptAt) {
            this.latestUpdateAttemptAt = latestUpdateAttemptAt;
            return this;
        }

        public final void setLatestUpdateAttemptAt(Instant latestUpdateAttemptAt) {
            this.latestUpdateAttemptAt = latestUpdateAttemptAt;
        }

        @Override
        public ParallelDataProperties build() {
            return new ParallelDataProperties(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
