/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.translate.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about a translation job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TextTranslationJobProperties implements SdkPojo, Serializable,
        ToCopyableBuilder<TextTranslationJobProperties.Builder, TextTranslationJobProperties> {
    private static final SdkField<String> JOB_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("JobId")
            .getter(getter(TextTranslationJobProperties::jobId)).setter(setter(Builder::jobId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobId").build()).build();

    private static final SdkField<String> JOB_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("JobName").getter(getter(TextTranslationJobProperties::jobName)).setter(setter(Builder::jobName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobName").build()).build();

    private static final SdkField<String> JOB_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("JobStatus").getter(getter(TextTranslationJobProperties::jobStatusAsString))
            .setter(setter(Builder::jobStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobStatus").build()).build();

    private static final SdkField<JobDetails> JOB_DETAILS_FIELD = SdkField.<JobDetails> builder(MarshallingType.SDK_POJO)
            .memberName("JobDetails").getter(getter(TextTranslationJobProperties::jobDetails))
            .setter(setter(Builder::jobDetails)).constructor(JobDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobDetails").build()).build();

    private static final SdkField<String> SOURCE_LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceLanguageCode").getter(getter(TextTranslationJobProperties::sourceLanguageCode))
            .setter(setter(Builder::sourceLanguageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceLanguageCode").build())
            .build();

    private static final SdkField<List<String>> TARGET_LANGUAGE_CODES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("TargetLanguageCodes")
            .getter(getter(TextTranslationJobProperties::targetLanguageCodes))
            .setter(setter(Builder::targetLanguageCodes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetLanguageCodes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> TERMINOLOGY_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("TerminologyNames")
            .getter(getter(TextTranslationJobProperties::terminologyNames))
            .setter(setter(Builder::terminologyNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TerminologyNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> PARALLEL_DATA_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ParallelDataNames")
            .getter(getter(TextTranslationJobProperties::parallelDataNames))
            .setter(setter(Builder::parallelDataNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParallelDataNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Message")
            .getter(getter(TextTranslationJobProperties::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Message").build()).build();

    private static final SdkField<Instant> SUBMITTED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("SubmittedTime").getter(getter(TextTranslationJobProperties::submittedTime))
            .setter(setter(Builder::submittedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubmittedTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndTime").getter(getter(TextTranslationJobProperties::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTime").build()).build();

    private static final SdkField<InputDataConfig> INPUT_DATA_CONFIG_FIELD = SdkField
            .<InputDataConfig> builder(MarshallingType.SDK_POJO).memberName("InputDataConfig")
            .getter(getter(TextTranslationJobProperties::inputDataConfig)).setter(setter(Builder::inputDataConfig))
            .constructor(InputDataConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InputDataConfig").build()).build();

    private static final SdkField<OutputDataConfig> OUTPUT_DATA_CONFIG_FIELD = SdkField
            .<OutputDataConfig> builder(MarshallingType.SDK_POJO).memberName("OutputDataConfig")
            .getter(getter(TextTranslationJobProperties::outputDataConfig)).setter(setter(Builder::outputDataConfig))
            .constructor(OutputDataConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputDataConfig").build()).build();

    private static final SdkField<String> DATA_ACCESS_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataAccessRoleArn").getter(getter(TextTranslationJobProperties::dataAccessRoleArn))
            .setter(setter(Builder::dataAccessRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataAccessRoleArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(JOB_ID_FIELD, JOB_NAME_FIELD,
            JOB_STATUS_FIELD, JOB_DETAILS_FIELD, SOURCE_LANGUAGE_CODE_FIELD, TARGET_LANGUAGE_CODES_FIELD,
            TERMINOLOGY_NAMES_FIELD, PARALLEL_DATA_NAMES_FIELD, MESSAGE_FIELD, SUBMITTED_TIME_FIELD, END_TIME_FIELD,
            INPUT_DATA_CONFIG_FIELD, OUTPUT_DATA_CONFIG_FIELD, DATA_ACCESS_ROLE_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String jobId;

    private final String jobName;

    private final String jobStatus;

    private final JobDetails jobDetails;

    private final String sourceLanguageCode;

    private final List<String> targetLanguageCodes;

    private final List<String> terminologyNames;

    private final List<String> parallelDataNames;

    private final String message;

    private final Instant submittedTime;

    private final Instant endTime;

    private final InputDataConfig inputDataConfig;

    private final OutputDataConfig outputDataConfig;

    private final String dataAccessRoleArn;

    private TextTranslationJobProperties(BuilderImpl builder) {
        this.jobId = builder.jobId;
        this.jobName = builder.jobName;
        this.jobStatus = builder.jobStatus;
        this.jobDetails = builder.jobDetails;
        this.sourceLanguageCode = builder.sourceLanguageCode;
        this.targetLanguageCodes = builder.targetLanguageCodes;
        this.terminologyNames = builder.terminologyNames;
        this.parallelDataNames = builder.parallelDataNames;
        this.message = builder.message;
        this.submittedTime = builder.submittedTime;
        this.endTime = builder.endTime;
        this.inputDataConfig = builder.inputDataConfig;
        this.outputDataConfig = builder.outputDataConfig;
        this.dataAccessRoleArn = builder.dataAccessRoleArn;
    }

    /**
     * <p>
     * The ID of the translation job.
     * </p>
     * 
     * @return The ID of the translation job.
     */
    public final String jobId() {
        return jobId;
    }

    /**
     * <p>
     * The user-defined name of the translation job.
     * </p>
     * 
     * @return The user-defined name of the translation job.
     */
    public final String jobName() {
        return jobName;
    }

    /**
     * <p>
     * The status of the translation job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #jobStatus} will
     * return {@link JobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #jobStatusAsString}.
     * </p>
     * 
     * @return The status of the translation job.
     * @see JobStatus
     */
    public final JobStatus jobStatus() {
        return JobStatus.fromValue(jobStatus);
    }

    /**
     * <p>
     * The status of the translation job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #jobStatus} will
     * return {@link JobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #jobStatusAsString}.
     * </p>
     * 
     * @return The status of the translation job.
     * @see JobStatus
     */
    public final String jobStatusAsString() {
        return jobStatus;
    }

    /**
     * <p>
     * The number of documents successfully and unsuccessfully processed during the translation job.
     * </p>
     * 
     * @return The number of documents successfully and unsuccessfully processed during the translation job.
     */
    public final JobDetails jobDetails() {
        return jobDetails;
    }

    /**
     * <p>
     * The language code of the language of the source text. The language must be a language supported by Amazon
     * Translate.
     * </p>
     * 
     * @return The language code of the language of the source text. The language must be a language supported by Amazon
     *         Translate.
     */
    public final String sourceLanguageCode() {
        return sourceLanguageCode;
    }

    /**
     * Returns true if the TargetLanguageCodes property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasTargetLanguageCodes() {
        return targetLanguageCodes != null && !(targetLanguageCodes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The language code of the language of the target text. The language must be a language supported by Amazon
     * Translate.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTargetLanguageCodes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The language code of the language of the target text. The language must be a language supported by Amazon
     *         Translate.
     */
    public final List<String> targetLanguageCodes() {
        return targetLanguageCodes;
    }

    /**
     * Returns true if the TerminologyNames property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasTerminologyNames() {
        return terminologyNames != null && !(terminologyNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list containing the names of the terminologies applied to a translation job. Only one terminology can be
     * applied per <a>StartTextTranslationJob</a> request at this time.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTerminologyNames()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list containing the names of the terminologies applied to a translation job. Only one terminology can
     *         be applied per <a>StartTextTranslationJob</a> request at this time.
     */
    public final List<String> terminologyNames() {
        return terminologyNames;
    }

    /**
     * Returns true if the ParallelDataNames property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasParallelDataNames() {
        return parallelDataNames != null && !(parallelDataNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list containing the names of the parallel data resources applied to the translation job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasParallelDataNames()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list containing the names of the parallel data resources applied to the translation job.
     */
    public final List<String> parallelDataNames() {
        return parallelDataNames;
    }

    /**
     * <p>
     * An explanation of any errors that may have occured during the translation job.
     * </p>
     * 
     * @return An explanation of any errors that may have occured during the translation job.
     */
    public final String message() {
        return message;
    }

    /**
     * <p>
     * The time at which the translation job was submitted.
     * </p>
     * 
     * @return The time at which the translation job was submitted.
     */
    public final Instant submittedTime() {
        return submittedTime;
    }

    /**
     * <p>
     * The time at which the translation job ended.
     * </p>
     * 
     * @return The time at which the translation job ended.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The input configuration properties that were specified when the job was requested.
     * </p>
     * 
     * @return The input configuration properties that were specified when the job was requested.
     */
    public final InputDataConfig inputDataConfig() {
        return inputDataConfig;
    }

    /**
     * <p>
     * The output configuration properties that were specified when the job was requested.
     * </p>
     * 
     * @return The output configuration properties that were specified when the job was requested.
     */
    public final OutputDataConfig outputDataConfig() {
        return outputDataConfig;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an AWS Identity Access and Management (IAM) role that granted Amazon Translate
     * read access to the job's input data.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of an AWS Identity Access and Management (IAM) role that granted Amazon
     *         Translate read access to the job's input data.
     */
    public final String dataAccessRoleArn() {
        return dataAccessRoleArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(jobId());
        hashCode = 31 * hashCode + Objects.hashCode(jobName());
        hashCode = 31 * hashCode + Objects.hashCode(jobStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(jobDetails());
        hashCode = 31 * hashCode + Objects.hashCode(sourceLanguageCode());
        hashCode = 31 * hashCode + Objects.hashCode(hasTargetLanguageCodes() ? targetLanguageCodes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTerminologyNames() ? terminologyNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasParallelDataNames() ? parallelDataNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(submittedTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(inputDataConfig());
        hashCode = 31 * hashCode + Objects.hashCode(outputDataConfig());
        hashCode = 31 * hashCode + Objects.hashCode(dataAccessRoleArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TextTranslationJobProperties)) {
            return false;
        }
        TextTranslationJobProperties other = (TextTranslationJobProperties) obj;
        return Objects.equals(jobId(), other.jobId()) && Objects.equals(jobName(), other.jobName())
                && Objects.equals(jobStatusAsString(), other.jobStatusAsString())
                && Objects.equals(jobDetails(), other.jobDetails())
                && Objects.equals(sourceLanguageCode(), other.sourceLanguageCode())
                && hasTargetLanguageCodes() == other.hasTargetLanguageCodes()
                && Objects.equals(targetLanguageCodes(), other.targetLanguageCodes())
                && hasTerminologyNames() == other.hasTerminologyNames()
                && Objects.equals(terminologyNames(), other.terminologyNames())
                && hasParallelDataNames() == other.hasParallelDataNames()
                && Objects.equals(parallelDataNames(), other.parallelDataNames()) && Objects.equals(message(), other.message())
                && Objects.equals(submittedTime(), other.submittedTime()) && Objects.equals(endTime(), other.endTime())
                && Objects.equals(inputDataConfig(), other.inputDataConfig())
                && Objects.equals(outputDataConfig(), other.outputDataConfig())
                && Objects.equals(dataAccessRoleArn(), other.dataAccessRoleArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TextTranslationJobProperties").add("JobId", jobId()).add("JobName", jobName())
                .add("JobStatus", jobStatusAsString()).add("JobDetails", jobDetails())
                .add("SourceLanguageCode", sourceLanguageCode())
                .add("TargetLanguageCodes", hasTargetLanguageCodes() ? targetLanguageCodes() : null)
                .add("TerminologyNames", hasTerminologyNames() ? terminologyNames() : null)
                .add("ParallelDataNames", hasParallelDataNames() ? parallelDataNames() : null).add("Message", message())
                .add("SubmittedTime", submittedTime()).add("EndTime", endTime()).add("InputDataConfig", inputDataConfig())
                .add("OutputDataConfig", outputDataConfig()).add("DataAccessRoleArn", dataAccessRoleArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "JobId":
            return Optional.ofNullable(clazz.cast(jobId()));
        case "JobName":
            return Optional.ofNullable(clazz.cast(jobName()));
        case "JobStatus":
            return Optional.ofNullable(clazz.cast(jobStatusAsString()));
        case "JobDetails":
            return Optional.ofNullable(clazz.cast(jobDetails()));
        case "SourceLanguageCode":
            return Optional.ofNullable(clazz.cast(sourceLanguageCode()));
        case "TargetLanguageCodes":
            return Optional.ofNullable(clazz.cast(targetLanguageCodes()));
        case "TerminologyNames":
            return Optional.ofNullable(clazz.cast(terminologyNames()));
        case "ParallelDataNames":
            return Optional.ofNullable(clazz.cast(parallelDataNames()));
        case "Message":
            return Optional.ofNullable(clazz.cast(message()));
        case "SubmittedTime":
            return Optional.ofNullable(clazz.cast(submittedTime()));
        case "EndTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "InputDataConfig":
            return Optional.ofNullable(clazz.cast(inputDataConfig()));
        case "OutputDataConfig":
            return Optional.ofNullable(clazz.cast(outputDataConfig()));
        case "DataAccessRoleArn":
            return Optional.ofNullable(clazz.cast(dataAccessRoleArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TextTranslationJobProperties, T> g) {
        return obj -> g.apply((TextTranslationJobProperties) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TextTranslationJobProperties> {
        /**
         * <p>
         * The ID of the translation job.
         * </p>
         * 
         * @param jobId
         *        The ID of the translation job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobId(String jobId);

        /**
         * <p>
         * The user-defined name of the translation job.
         * </p>
         * 
         * @param jobName
         *        The user-defined name of the translation job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobName(String jobName);

        /**
         * <p>
         * The status of the translation job.
         * </p>
         * 
         * @param jobStatus
         *        The status of the translation job.
         * @see JobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobStatus
         */
        Builder jobStatus(String jobStatus);

        /**
         * <p>
         * The status of the translation job.
         * </p>
         * 
         * @param jobStatus
         *        The status of the translation job.
         * @see JobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobStatus
         */
        Builder jobStatus(JobStatus jobStatus);

        /**
         * <p>
         * The number of documents successfully and unsuccessfully processed during the translation job.
         * </p>
         * 
         * @param jobDetails
         *        The number of documents successfully and unsuccessfully processed during the translation job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobDetails(JobDetails jobDetails);

        /**
         * <p>
         * The number of documents successfully and unsuccessfully processed during the translation job.
         * </p>
         * This is a convenience that creates an instance of the {@link JobDetails.Builder} avoiding the need to create
         * one manually via {@link JobDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link JobDetails.Builder#build()} is called immediately and its result
         * is passed to {@link #jobDetails(JobDetails)}.
         * 
         * @param jobDetails
         *        a consumer that will call methods on {@link JobDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #jobDetails(JobDetails)
         */
        default Builder jobDetails(Consumer<JobDetails.Builder> jobDetails) {
            return jobDetails(JobDetails.builder().applyMutation(jobDetails).build());
        }

        /**
         * <p>
         * The language code of the language of the source text. The language must be a language supported by Amazon
         * Translate.
         * </p>
         * 
         * @param sourceLanguageCode
         *        The language code of the language of the source text. The language must be a language supported by
         *        Amazon Translate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceLanguageCode(String sourceLanguageCode);

        /**
         * <p>
         * The language code of the language of the target text. The language must be a language supported by Amazon
         * Translate.
         * </p>
         * 
         * @param targetLanguageCodes
         *        The language code of the language of the target text. The language must be a language supported by
         *        Amazon Translate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetLanguageCodes(Collection<String> targetLanguageCodes);

        /**
         * <p>
         * The language code of the language of the target text. The language must be a language supported by Amazon
         * Translate.
         * </p>
         * 
         * @param targetLanguageCodes
         *        The language code of the language of the target text. The language must be a language supported by
         *        Amazon Translate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetLanguageCodes(String... targetLanguageCodes);

        /**
         * <p>
         * A list containing the names of the terminologies applied to a translation job. Only one terminology can be
         * applied per <a>StartTextTranslationJob</a> request at this time.
         * </p>
         * 
         * @param terminologyNames
         *        A list containing the names of the terminologies applied to a translation job. Only one terminology
         *        can be applied per <a>StartTextTranslationJob</a> request at this time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder terminologyNames(Collection<String> terminologyNames);

        /**
         * <p>
         * A list containing the names of the terminologies applied to a translation job. Only one terminology can be
         * applied per <a>StartTextTranslationJob</a> request at this time.
         * </p>
         * 
         * @param terminologyNames
         *        A list containing the names of the terminologies applied to a translation job. Only one terminology
         *        can be applied per <a>StartTextTranslationJob</a> request at this time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder terminologyNames(String... terminologyNames);

        /**
         * <p>
         * A list containing the names of the parallel data resources applied to the translation job.
         * </p>
         * 
         * @param parallelDataNames
         *        A list containing the names of the parallel data resources applied to the translation job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parallelDataNames(Collection<String> parallelDataNames);

        /**
         * <p>
         * A list containing the names of the parallel data resources applied to the translation job.
         * </p>
         * 
         * @param parallelDataNames
         *        A list containing the names of the parallel data resources applied to the translation job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parallelDataNames(String... parallelDataNames);

        /**
         * <p>
         * An explanation of any errors that may have occured during the translation job.
         * </p>
         * 
         * @param message
         *        An explanation of any errors that may have occured during the translation job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);

        /**
         * <p>
         * The time at which the translation job was submitted.
         * </p>
         * 
         * @param submittedTime
         *        The time at which the translation job was submitted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder submittedTime(Instant submittedTime);

        /**
         * <p>
         * The time at which the translation job ended.
         * </p>
         * 
         * @param endTime
         *        The time at which the translation job ended.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The input configuration properties that were specified when the job was requested.
         * </p>
         * 
         * @param inputDataConfig
         *        The input configuration properties that were specified when the job was requested.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputDataConfig(InputDataConfig inputDataConfig);

        /**
         * <p>
         * The input configuration properties that were specified when the job was requested.
         * </p>
         * This is a convenience that creates an instance of the {@link InputDataConfig.Builder} avoiding the need to
         * create one manually via {@link InputDataConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link InputDataConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #inputDataConfig(InputDataConfig)}.
         * 
         * @param inputDataConfig
         *        a consumer that will call methods on {@link InputDataConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #inputDataConfig(InputDataConfig)
         */
        default Builder inputDataConfig(Consumer<InputDataConfig.Builder> inputDataConfig) {
            return inputDataConfig(InputDataConfig.builder().applyMutation(inputDataConfig).build());
        }

        /**
         * <p>
         * The output configuration properties that were specified when the job was requested.
         * </p>
         * 
         * @param outputDataConfig
         *        The output configuration properties that were specified when the job was requested.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputDataConfig(OutputDataConfig outputDataConfig);

        /**
         * <p>
         * The output configuration properties that were specified when the job was requested.
         * </p>
         * This is a convenience that creates an instance of the {@link OutputDataConfig.Builder} avoiding the need to
         * create one manually via {@link OutputDataConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link OutputDataConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #outputDataConfig(OutputDataConfig)}.
         * 
         * @param outputDataConfig
         *        a consumer that will call methods on {@link OutputDataConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputDataConfig(OutputDataConfig)
         */
        default Builder outputDataConfig(Consumer<OutputDataConfig.Builder> outputDataConfig) {
            return outputDataConfig(OutputDataConfig.builder().applyMutation(outputDataConfig).build());
        }

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an AWS Identity Access and Management (IAM) role that granted Amazon
         * Translate read access to the job's input data.
         * </p>
         * 
         * @param dataAccessRoleArn
         *        The Amazon Resource Name (ARN) of an AWS Identity Access and Management (IAM) role that granted Amazon
         *        Translate read access to the job's input data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataAccessRoleArn(String dataAccessRoleArn);
    }

    static final class BuilderImpl implements Builder {
        private String jobId;

        private String jobName;

        private String jobStatus;

        private JobDetails jobDetails;

        private String sourceLanguageCode;

        private List<String> targetLanguageCodes = DefaultSdkAutoConstructList.getInstance();

        private List<String> terminologyNames = DefaultSdkAutoConstructList.getInstance();

        private List<String> parallelDataNames = DefaultSdkAutoConstructList.getInstance();

        private String message;

        private Instant submittedTime;

        private Instant endTime;

        private InputDataConfig inputDataConfig;

        private OutputDataConfig outputDataConfig;

        private String dataAccessRoleArn;

        private BuilderImpl() {
        }

        private BuilderImpl(TextTranslationJobProperties model) {
            jobId(model.jobId);
            jobName(model.jobName);
            jobStatus(model.jobStatus);
            jobDetails(model.jobDetails);
            sourceLanguageCode(model.sourceLanguageCode);
            targetLanguageCodes(model.targetLanguageCodes);
            terminologyNames(model.terminologyNames);
            parallelDataNames(model.parallelDataNames);
            message(model.message);
            submittedTime(model.submittedTime);
            endTime(model.endTime);
            inputDataConfig(model.inputDataConfig);
            outputDataConfig(model.outputDataConfig);
            dataAccessRoleArn(model.dataAccessRoleArn);
        }

        public final String getJobId() {
            return jobId;
        }

        @Override
        public final Builder jobId(String jobId) {
            this.jobId = jobId;
            return this;
        }

        public final void setJobId(String jobId) {
            this.jobId = jobId;
        }

        public final String getJobName() {
            return jobName;
        }

        @Override
        public final Builder jobName(String jobName) {
            this.jobName = jobName;
            return this;
        }

        public final void setJobName(String jobName) {
            this.jobName = jobName;
        }

        public final String getJobStatus() {
            return jobStatus;
        }

        @Override
        public final Builder jobStatus(String jobStatus) {
            this.jobStatus = jobStatus;
            return this;
        }

        @Override
        public final Builder jobStatus(JobStatus jobStatus) {
            this.jobStatus(jobStatus == null ? null : jobStatus.toString());
            return this;
        }

        public final void setJobStatus(String jobStatus) {
            this.jobStatus = jobStatus;
        }

        public final JobDetails.Builder getJobDetails() {
            return jobDetails != null ? jobDetails.toBuilder() : null;
        }

        @Override
        public final Builder jobDetails(JobDetails jobDetails) {
            this.jobDetails = jobDetails;
            return this;
        }

        public final void setJobDetails(JobDetails.BuilderImpl jobDetails) {
            this.jobDetails = jobDetails != null ? jobDetails.build() : null;
        }

        public final String getSourceLanguageCode() {
            return sourceLanguageCode;
        }

        @Override
        public final Builder sourceLanguageCode(String sourceLanguageCode) {
            this.sourceLanguageCode = sourceLanguageCode;
            return this;
        }

        public final void setSourceLanguageCode(String sourceLanguageCode) {
            this.sourceLanguageCode = sourceLanguageCode;
        }

        public final Collection<String> getTargetLanguageCodes() {
            if (targetLanguageCodes instanceof SdkAutoConstructList) {
                return null;
            }
            return targetLanguageCodes;
        }

        @Override
        public final Builder targetLanguageCodes(Collection<String> targetLanguageCodes) {
            this.targetLanguageCodes = TargetLanguageCodeStringListCopier.copy(targetLanguageCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetLanguageCodes(String... targetLanguageCodes) {
            targetLanguageCodes(Arrays.asList(targetLanguageCodes));
            return this;
        }

        public final void setTargetLanguageCodes(Collection<String> targetLanguageCodes) {
            this.targetLanguageCodes = TargetLanguageCodeStringListCopier.copy(targetLanguageCodes);
        }

        public final Collection<String> getTerminologyNames() {
            if (terminologyNames instanceof SdkAutoConstructList) {
                return null;
            }
            return terminologyNames;
        }

        @Override
        public final Builder terminologyNames(Collection<String> terminologyNames) {
            this.terminologyNames = ResourceNameListCopier.copy(terminologyNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder terminologyNames(String... terminologyNames) {
            terminologyNames(Arrays.asList(terminologyNames));
            return this;
        }

        public final void setTerminologyNames(Collection<String> terminologyNames) {
            this.terminologyNames = ResourceNameListCopier.copy(terminologyNames);
        }

        public final Collection<String> getParallelDataNames() {
            if (parallelDataNames instanceof SdkAutoConstructList) {
                return null;
            }
            return parallelDataNames;
        }

        @Override
        public final Builder parallelDataNames(Collection<String> parallelDataNames) {
            this.parallelDataNames = ResourceNameListCopier.copy(parallelDataNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parallelDataNames(String... parallelDataNames) {
            parallelDataNames(Arrays.asList(parallelDataNames));
            return this;
        }

        public final void setParallelDataNames(Collection<String> parallelDataNames) {
            this.parallelDataNames = ResourceNameListCopier.copy(parallelDataNames);
        }

        public final String getMessage() {
            return message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        public final Instant getSubmittedTime() {
            return submittedTime;
        }

        @Override
        public final Builder submittedTime(Instant submittedTime) {
            this.submittedTime = submittedTime;
            return this;
        }

        public final void setSubmittedTime(Instant submittedTime) {
            this.submittedTime = submittedTime;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        public final InputDataConfig.Builder getInputDataConfig() {
            return inputDataConfig != null ? inputDataConfig.toBuilder() : null;
        }

        @Override
        public final Builder inputDataConfig(InputDataConfig inputDataConfig) {
            this.inputDataConfig = inputDataConfig;
            return this;
        }

        public final void setInputDataConfig(InputDataConfig.BuilderImpl inputDataConfig) {
            this.inputDataConfig = inputDataConfig != null ? inputDataConfig.build() : null;
        }

        public final OutputDataConfig.Builder getOutputDataConfig() {
            return outputDataConfig != null ? outputDataConfig.toBuilder() : null;
        }

        @Override
        public final Builder outputDataConfig(OutputDataConfig outputDataConfig) {
            this.outputDataConfig = outputDataConfig;
            return this;
        }

        public final void setOutputDataConfig(OutputDataConfig.BuilderImpl outputDataConfig) {
            this.outputDataConfig = outputDataConfig != null ? outputDataConfig.build() : null;
        }

        public final String getDataAccessRoleArn() {
            return dataAccessRoleArn;
        }

        @Override
        public final Builder dataAccessRoleArn(String dataAccessRoleArn) {
            this.dataAccessRoleArn = dataAccessRoleArn;
            return this;
        }

        public final void setDataAccessRoleArn(String dataAccessRoleArn) {
            this.dataAccessRoleArn = dataAccessRoleArn;
        }

        @Override
        public TextTranslationJobProperties build() {
            return new TextTranslationJobProperties(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
