/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.translate;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.translate.model.ConcurrentModificationException;
import software.amazon.awssdk.services.translate.model.ConflictException;
import software.amazon.awssdk.services.translate.model.CreateParallelDataRequest;
import software.amazon.awssdk.services.translate.model.CreateParallelDataResponse;
import software.amazon.awssdk.services.translate.model.DeleteParallelDataRequest;
import software.amazon.awssdk.services.translate.model.DeleteParallelDataResponse;
import software.amazon.awssdk.services.translate.model.DeleteTerminologyRequest;
import software.amazon.awssdk.services.translate.model.DeleteTerminologyResponse;
import software.amazon.awssdk.services.translate.model.DescribeTextTranslationJobRequest;
import software.amazon.awssdk.services.translate.model.DescribeTextTranslationJobResponse;
import software.amazon.awssdk.services.translate.model.DetectedLanguageLowConfidenceException;
import software.amazon.awssdk.services.translate.model.GetParallelDataRequest;
import software.amazon.awssdk.services.translate.model.GetParallelDataResponse;
import software.amazon.awssdk.services.translate.model.GetTerminologyRequest;
import software.amazon.awssdk.services.translate.model.GetTerminologyResponse;
import software.amazon.awssdk.services.translate.model.ImportTerminologyRequest;
import software.amazon.awssdk.services.translate.model.ImportTerminologyResponse;
import software.amazon.awssdk.services.translate.model.InternalServerException;
import software.amazon.awssdk.services.translate.model.InvalidFilterException;
import software.amazon.awssdk.services.translate.model.InvalidParameterValueException;
import software.amazon.awssdk.services.translate.model.InvalidRequestException;
import software.amazon.awssdk.services.translate.model.LimitExceededException;
import software.amazon.awssdk.services.translate.model.ListLanguagesRequest;
import software.amazon.awssdk.services.translate.model.ListLanguagesResponse;
import software.amazon.awssdk.services.translate.model.ListParallelDataRequest;
import software.amazon.awssdk.services.translate.model.ListParallelDataResponse;
import software.amazon.awssdk.services.translate.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.translate.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.translate.model.ListTerminologiesRequest;
import software.amazon.awssdk.services.translate.model.ListTerminologiesResponse;
import software.amazon.awssdk.services.translate.model.ListTextTranslationJobsRequest;
import software.amazon.awssdk.services.translate.model.ListTextTranslationJobsResponse;
import software.amazon.awssdk.services.translate.model.ResourceNotFoundException;
import software.amazon.awssdk.services.translate.model.ServiceUnavailableException;
import software.amazon.awssdk.services.translate.model.StartTextTranslationJobRequest;
import software.amazon.awssdk.services.translate.model.StartTextTranslationJobResponse;
import software.amazon.awssdk.services.translate.model.StopTextTranslationJobRequest;
import software.amazon.awssdk.services.translate.model.StopTextTranslationJobResponse;
import software.amazon.awssdk.services.translate.model.TagResourceRequest;
import software.amazon.awssdk.services.translate.model.TagResourceResponse;
import software.amazon.awssdk.services.translate.model.TextSizeLimitExceededException;
import software.amazon.awssdk.services.translate.model.TooManyRequestsException;
import software.amazon.awssdk.services.translate.model.TooManyTagsException;
import software.amazon.awssdk.services.translate.model.TranslateException;
import software.amazon.awssdk.services.translate.model.TranslateRequest;
import software.amazon.awssdk.services.translate.model.TranslateTextRequest;
import software.amazon.awssdk.services.translate.model.TranslateTextResponse;
import software.amazon.awssdk.services.translate.model.UnsupportedDisplayLanguageCodeException;
import software.amazon.awssdk.services.translate.model.UnsupportedLanguagePairException;
import software.amazon.awssdk.services.translate.model.UntagResourceRequest;
import software.amazon.awssdk.services.translate.model.UntagResourceResponse;
import software.amazon.awssdk.services.translate.model.UpdateParallelDataRequest;
import software.amazon.awssdk.services.translate.model.UpdateParallelDataResponse;
import software.amazon.awssdk.services.translate.paginators.ListLanguagesPublisher;
import software.amazon.awssdk.services.translate.paginators.ListParallelDataPublisher;
import software.amazon.awssdk.services.translate.paginators.ListTerminologiesPublisher;
import software.amazon.awssdk.services.translate.paginators.ListTextTranslationJobsPublisher;
import software.amazon.awssdk.services.translate.transform.CreateParallelDataRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.DeleteParallelDataRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.DeleteTerminologyRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.DescribeTextTranslationJobRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.GetParallelDataRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.GetTerminologyRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.ImportTerminologyRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.ListLanguagesRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.ListParallelDataRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.ListTerminologiesRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.ListTextTranslationJobsRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.StartTextTranslationJobRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.StopTextTranslationJobRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.TranslateTextRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.UpdateParallelDataRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link TranslateAsyncClient}.
 *
 * @see TranslateAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTranslateAsyncClient implements TranslateAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultTranslateAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultTranslateAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a parallel data resource in Amazon Translate by importing an input file from Amazon S3. Parallel data
     * files contain examples that show how you want segments of text to be translated. By adding parallel data, you can
     * influence the style, tone, and word choice in your translation output.
     * </p>
     *
     * @param createParallelDataRequest
     * @return A Java Future containing the result of the CreateParallelData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>InvalidRequestException The request that you made is not valid. Check your request to determine why
     *         it's not valid and then retry the request.</li>
     *         <li>LimitExceededException The specified limit has been exceeded. Review your request and retry it with a
     *         quantity below the stated limit.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>TooManyTagsException</li>
     *         <li>ConflictException There was a conflict processing the request. Try your request again.</li>
     *         <li>ConcurrentModificationException Another modification is being made. That modification must complete
     *         before you can make your change.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.CreateParallelData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/CreateParallelData" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateParallelDataResponse> createParallelData(CreateParallelDataRequest createParallelDataRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createParallelDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateParallelData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateParallelDataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateParallelDataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateParallelDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateParallelDataRequest, CreateParallelDataResponse>()
                            .withOperationName("CreateParallelData")
                            .withMarshaller(new CreateParallelDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createParallelDataRequest));
            CompletableFuture<CreateParallelDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a parallel data resource in Amazon Translate.
     * </p>
     *
     * @param deleteParallelDataRequest
     * @return A Java Future containing the result of the DeleteParallelData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException Another modification is being made. That modification must complete
     *         before you can make your change.</li>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.DeleteParallelData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/DeleteParallelData" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteParallelDataResponse> deleteParallelData(DeleteParallelDataRequest deleteParallelDataRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteParallelDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteParallelData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteParallelDataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteParallelDataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteParallelDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteParallelDataRequest, DeleteParallelDataResponse>()
                            .withOperationName("DeleteParallelData")
                            .withMarshaller(new DeleteParallelDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteParallelDataRequest));
            CompletableFuture<DeleteParallelDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * A synchronous action that deletes a custom terminology.
     * </p>
     *
     * @param deleteTerminologyRequest
     * @return A Java Future containing the result of the DeleteTerminology operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.DeleteTerminology
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/DeleteTerminology" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTerminologyResponse> deleteTerminology(DeleteTerminologyRequest deleteTerminologyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTerminologyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTerminology");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTerminologyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteTerminologyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTerminologyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTerminologyRequest, DeleteTerminologyResponse>()
                            .withOperationName("DeleteTerminology")
                            .withMarshaller(new DeleteTerminologyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteTerminologyRequest));
            CompletableFuture<DeleteTerminologyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the properties associated with an asynchronous batch translation job including name, ID, status, source and
     * target languages, input/output S3 buckets, and so on.
     * </p>
     *
     * @param describeTextTranslationJobRequest
     * @return A Java Future containing the result of the DescribeTextTranslationJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.DescribeTextTranslationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/DescribeTextTranslationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTextTranslationJobResponse> describeTextTranslationJob(
            DescribeTextTranslationJobRequest describeTextTranslationJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTextTranslationJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTextTranslationJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTextTranslationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeTextTranslationJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeTextTranslationJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTextTranslationJobRequest, DescribeTextTranslationJobResponse>()
                            .withOperationName("DescribeTextTranslationJob")
                            .withMarshaller(new DescribeTextTranslationJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeTextTranslationJobRequest));
            CompletableFuture<DescribeTextTranslationJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about a parallel data resource.
     * </p>
     *
     * @param getParallelDataRequest
     * @return A Java Future containing the result of the GetParallelData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.GetParallelData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/GetParallelData" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetParallelDataResponse> getParallelData(GetParallelDataRequest getParallelDataRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getParallelDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetParallelData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetParallelDataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetParallelDataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetParallelDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetParallelDataRequest, GetParallelDataResponse>()
                            .withOperationName("GetParallelData")
                            .withMarshaller(new GetParallelDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getParallelDataRequest));
            CompletableFuture<GetParallelDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a custom terminology.
     * </p>
     *
     * @param getTerminologyRequest
     * @return A Java Future containing the result of the GetTerminology operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.GetTerminology
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/GetTerminology" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTerminologyResponse> getTerminology(GetTerminologyRequest getTerminologyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTerminologyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTerminology");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTerminologyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTerminologyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTerminologyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTerminologyRequest, GetTerminologyResponse>()
                            .withOperationName("GetTerminology")
                            .withMarshaller(new GetTerminologyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getTerminologyRequest));
            CompletableFuture<GetTerminologyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates a custom terminology, depending on whether one already exists for the given terminology name.
     * Importing a terminology with the same name as an existing one will merge the terminologies based on the chosen
     * merge strategy. The only supported merge strategy is OVERWRITE, where the imported terminology overwrites the
     * existing terminology of the same name.
     * </p>
     * <p>
     * If you import a terminology that overwrites an existing one, the new terminology takes up to 10 minutes to fully
     * propagate. After that, translations have access to the new terminology.
     * </p>
     *
     * @param importTerminologyRequest
     * @return A Java Future containing the result of the ImportTerminology operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>LimitExceededException The specified limit has been exceeded. Review your request and retry it with a
     *         quantity below the stated limit.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>TooManyTagsException</li>
     *         <li>ConcurrentModificationException Another modification is being made. That modification must complete
     *         before you can make your change.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.ImportTerminology
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ImportTerminology" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ImportTerminologyResponse> importTerminology(ImportTerminologyRequest importTerminologyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, importTerminologyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportTerminology");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ImportTerminologyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ImportTerminologyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ImportTerminologyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ImportTerminologyRequest, ImportTerminologyResponse>()
                            .withOperationName("ImportTerminology")
                            .withMarshaller(new ImportTerminologyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(importTerminologyRequest));
            CompletableFuture<ImportTerminologyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a list of languages (RFC-5646 codes and names) that Amazon Translate supports.
     * </p>
     *
     * @param listLanguagesRequest
     * @return A Java Future containing the result of the ListLanguages operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>UnsupportedDisplayLanguageCodeException Requested display language code is not supported.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.ListLanguages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListLanguages" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListLanguagesResponse> listLanguages(ListLanguagesRequest listLanguagesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLanguagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLanguages");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLanguagesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListLanguagesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLanguagesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLanguagesRequest, ListLanguagesResponse>()
                            .withOperationName("ListLanguages")
                            .withMarshaller(new ListLanguagesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLanguagesRequest));
            CompletableFuture<ListLanguagesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a list of languages (RFC-5646 codes and names) that Amazon Translate supports.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listLanguages(software.amazon.awssdk.services.translate.model.ListLanguagesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.translate.paginators.ListLanguagesPublisher publisher = client.listLanguagesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.translate.paginators.ListLanguagesPublisher publisher = client.listLanguagesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.translate.model.ListLanguagesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.translate.model.ListLanguagesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLanguages(software.amazon.awssdk.services.translate.model.ListLanguagesRequest)} operation.</b>
     * </p>
     *
     * @param listLanguagesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>UnsupportedDisplayLanguageCodeException Requested display language code is not supported.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.ListLanguages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListLanguages" target="_top">AWS API
     *      Documentation</a>
     */
    public ListLanguagesPublisher listLanguagesPaginator(ListLanguagesRequest listLanguagesRequest) {
        return new ListLanguagesPublisher(this, applyPaginatorUserAgent(listLanguagesRequest));
    }

    /**
     * <p>
     * Provides a list of your parallel data resources in Amazon Translate.
     * </p>
     *
     * @param listParallelDataRequest
     * @return A Java Future containing the result of the ListParallelData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.ListParallelData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListParallelData" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListParallelDataResponse> listParallelData(ListParallelDataRequest listParallelDataRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listParallelDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListParallelData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListParallelDataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListParallelDataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListParallelDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListParallelDataRequest, ListParallelDataResponse>()
                            .withOperationName("ListParallelData")
                            .withMarshaller(new ListParallelDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listParallelDataRequest));
            CompletableFuture<ListParallelDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a list of your parallel data resources in Amazon Translate.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listParallelData(software.amazon.awssdk.services.translate.model.ListParallelDataRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.translate.paginators.ListParallelDataPublisher publisher = client.listParallelDataPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.translate.paginators.ListParallelDataPublisher publisher = client.listParallelDataPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.translate.model.ListParallelDataResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.translate.model.ListParallelDataResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listParallelData(software.amazon.awssdk.services.translate.model.ListParallelDataRequest)} operation.</b>
     * </p>
     *
     * @param listParallelDataRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.ListParallelData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListParallelData" target="_top">AWS
     *      API Documentation</a>
     */
    public ListParallelDataPublisher listParallelDataPaginator(ListParallelDataRequest listParallelDataRequest) {
        return new ListParallelDataPublisher(this, applyPaginatorUserAgent(listParallelDataRequest));
    }

    /**
     * Invokes the ListTagsForResource operation asynchronously.
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a list of custom terminologies associated with your account.
     * </p>
     *
     * @param listTerminologiesRequest
     * @return A Java Future containing the result of the ListTerminologies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.ListTerminologies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListTerminologies" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTerminologiesResponse> listTerminologies(ListTerminologiesRequest listTerminologiesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTerminologiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTerminologies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTerminologiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTerminologiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTerminologiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTerminologiesRequest, ListTerminologiesResponse>()
                            .withOperationName("ListTerminologies")
                            .withMarshaller(new ListTerminologiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTerminologiesRequest));
            CompletableFuture<ListTerminologiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a list of custom terminologies associated with your account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTerminologies(software.amazon.awssdk.services.translate.model.ListTerminologiesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.translate.paginators.ListTerminologiesPublisher publisher = client.listTerminologiesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.translate.paginators.ListTerminologiesPublisher publisher = client.listTerminologiesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.translate.model.ListTerminologiesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.translate.model.ListTerminologiesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTerminologies(software.amazon.awssdk.services.translate.model.ListTerminologiesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTerminologiesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.ListTerminologies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListTerminologies" target="_top">AWS
     *      API Documentation</a>
     */
    public ListTerminologiesPublisher listTerminologiesPaginator(ListTerminologiesRequest listTerminologiesRequest) {
        return new ListTerminologiesPublisher(this, applyPaginatorUserAgent(listTerminologiesRequest));
    }

    /**
     * <p>
     * Gets a list of the batch translation jobs that you have submitted.
     * </p>
     *
     * @param listTextTranslationJobsRequest
     * @return A Java Future containing the result of the ListTextTranslationJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request that you made is not valid. Check your request to determine why
     *         it's not valid and then retry the request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>InvalidFilterException The filter specified for the operation is not valid. Specify a different
     *         filter.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.ListTextTranslationJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListTextTranslationJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTextTranslationJobsResponse> listTextTranslationJobs(
            ListTextTranslationJobsRequest listTextTranslationJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTextTranslationJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTextTranslationJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTextTranslationJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTextTranslationJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTextTranslationJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTextTranslationJobsRequest, ListTextTranslationJobsResponse>()
                            .withOperationName("ListTextTranslationJobs")
                            .withMarshaller(new ListTextTranslationJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTextTranslationJobsRequest));
            CompletableFuture<ListTextTranslationJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list of the batch translation jobs that you have submitted.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTextTranslationJobs(software.amazon.awssdk.services.translate.model.ListTextTranslationJobsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.translate.paginators.ListTextTranslationJobsPublisher publisher = client.listTextTranslationJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.translate.paginators.ListTextTranslationJobsPublisher publisher = client.listTextTranslationJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.translate.model.ListTextTranslationJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.translate.model.ListTextTranslationJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTextTranslationJobs(software.amazon.awssdk.services.translate.model.ListTextTranslationJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTextTranslationJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request that you made is not valid. Check your request to determine why
     *         it's not valid and then retry the request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>InvalidFilterException The filter specified for the operation is not valid. Specify a different
     *         filter.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.ListTextTranslationJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListTextTranslationJobs"
     *      target="_top">AWS API Documentation</a>
     */
    public ListTextTranslationJobsPublisher listTextTranslationJobsPaginator(
            ListTextTranslationJobsRequest listTextTranslationJobsRequest) {
        return new ListTextTranslationJobsPublisher(this, applyPaginatorUserAgent(listTextTranslationJobsRequest));
    }

    /**
     * <p>
     * Starts an asynchronous batch translation job. Batch translation jobs can be used to translate large volumes of
     * text across multiple documents at once. For more information, see <a>async</a>.
     * </p>
     * <p>
     * Batch translation jobs can be described with the <a>DescribeTextTranslationJob</a> operation, listed with the
     * <a>ListTextTranslationJobs</a> operation, and stopped with the <a>StopTextTranslationJob</a> operation.
     * </p>
     * <note>
     * <p>
     * Amazon Translate does not support batch translation of multiple source languages at once.
     * </p>
     * </note>
     *
     * @param startTextTranslationJobRequest
     * @return A Java Future containing the result of the StartTextTranslationJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>UnsupportedLanguagePairException Amazon Translate does not support translation from the language of
     *         the source text into the requested target language. For more information, see <a>how-to-error-msg</a>.</li>
     *         <li>InvalidRequestException The request that you made is not valid. Check your request to determine why
     *         it's not valid and then retry the request.</li>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.StartTextTranslationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/StartTextTranslationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartTextTranslationJobResponse> startTextTranslationJob(
            StartTextTranslationJobRequest startTextTranslationJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startTextTranslationJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartTextTranslationJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartTextTranslationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartTextTranslationJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartTextTranslationJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartTextTranslationJobRequest, StartTextTranslationJobResponse>()
                            .withOperationName("StartTextTranslationJob")
                            .withMarshaller(new StartTextTranslationJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startTextTranslationJobRequest));
            CompletableFuture<StartTextTranslationJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops an asynchronous batch translation job that is in progress.
     * </p>
     * <p>
     * If the job's state is <code>IN_PROGRESS</code>, the job will be marked for termination and put into the
     * <code>STOP_REQUESTED</code> state. If the job completes before it can be stopped, it is put into the
     * <code>COMPLETED</code> state. Otherwise, the job is put into the <code>STOPPED</code> state.
     * </p>
     * <p>
     * Asynchronous batch translation jobs are started with the <a>StartTextTranslationJob</a> operation. You can use
     * the <a>DescribeTextTranslationJob</a> or <a>ListTextTranslationJobs</a> operations to get a batch translation
     * job's <code>JobId</code>.
     * </p>
     *
     * @param stopTextTranslationJobRequest
     * @return A Java Future containing the result of the StopTextTranslationJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.StopTextTranslationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/StopTextTranslationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopTextTranslationJobResponse> stopTextTranslationJob(
            StopTextTranslationJobRequest stopTextTranslationJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopTextTranslationJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopTextTranslationJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopTextTranslationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopTextTranslationJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopTextTranslationJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopTextTranslationJobRequest, StopTextTranslationJobResponse>()
                            .withOperationName("StopTextTranslationJob")
                            .withMarshaller(new StopTextTranslationJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopTextTranslationJobRequest));
            CompletableFuture<StopTextTranslationJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the TagResource operation asynchronously.
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>ConcurrentModificationException Another modification is being made. That modification must complete
     *         before you can make your change.</li>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>TooManyTagsException</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Translates input text from the source language to the target language. For a list of available languages and
     * language codes, see <a>what-is-languages</a>.
     * </p>
     *
     * @param translateTextRequest
     * @return A Java Future containing the result of the TranslateText operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request that you made is not valid. Check your request to determine why
     *         it's not valid and then retry the request.</li>
     *         <li>TextSizeLimitExceededException The size of the text you submitted exceeds the size limit. Reduce the
     *         size of the text or use a smaller document and then retry your request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>UnsupportedLanguagePairException Amazon Translate does not support translation from the language of
     *         the source text into the requested target language. For more information, see <a>how-to-error-msg</a>.</li>
     *         <li>DetectedLanguageLowConfidenceException The confidence that Amazon Comprehend accurately detected the
     *         source language is low. If a low confidence level is acceptable for your application, you can use the
     *         language in the exception to call Amazon Translate again. For more information, see the <a
     *         href="https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectDominantLanguage.html"
     *         >DetectDominantLanguage</a> operation in the <i>Amazon Comprehend Developer Guide</i>.</li>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ServiceUnavailableException The Amazon Translate service is temporarily unavailable. Wait a bit and
     *         then retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.TranslateText
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/TranslateText" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TranslateTextResponse> translateText(TranslateTextRequest translateTextRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, translateTextRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TranslateText");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TranslateTextResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TranslateTextResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TranslateTextResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TranslateTextRequest, TranslateTextResponse>()
                            .withOperationName("TranslateText")
                            .withMarshaller(new TranslateTextRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(translateTextRequest));
            CompletableFuture<TranslateTextResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the UntagResource operation asynchronously.
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>ConcurrentModificationException Another modification is being made. That modification must complete
     *         before you can make your change.</li>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a previously created parallel data resource by importing a new input file from Amazon S3.
     * </p>
     *
     * @param updateParallelDataRequest
     * @return A Java Future containing the result of the UpdateParallelData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException Another modification is being made. That modification must complete
     *         before you can make your change.</li>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>InvalidRequestException The request that you made is not valid. Check your request to determine why
     *         it's not valid and then retry the request.</li>
     *         <li>LimitExceededException The specified limit has been exceeded. Review your request and retry it with a
     *         quantity below the stated limit.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>ConflictException There was a conflict processing the request. Try your request again.</li>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.UpdateParallelData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/UpdateParallelData" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateParallelDataResponse> updateParallelData(UpdateParallelDataRequest updateParallelDataRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateParallelDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateParallelData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateParallelDataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateParallelDataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateParallelDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateParallelDataRequest, UpdateParallelDataResponse>()
                            .withOperationName("UpdateParallelData")
                            .withMarshaller(new UpdateParallelDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateParallelDataRequest));
            CompletableFuture<UpdateParallelDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(TranslateException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DetectedLanguageLowConfidenceException")
                                .exceptionBuilderSupplier(DetectedLanguageLowConfidenceException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedLanguagePairException")
                                .exceptionBuilderSupplier(UnsupportedLanguagePairException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidFilterException")
                                .exceptionBuilderSupplier(InvalidFilterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                                .exceptionBuilderSupplier(TextSizeLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedDisplayLanguageCodeException")
                                .exceptionBuilderSupplier(UnsupportedDisplayLanguageCodeException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends TranslateRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
