/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.translate.model;

import java.beans.Transient;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class TranslateTextRequest extends TranslateRequest implements
        ToCopyableBuilder<TranslateTextRequest.Builder, TranslateTextRequest> {
    private static final SdkField<String> TEXT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Text")
            .getter(getter(TranslateTextRequest::text)).setter(setter(Builder::text))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Text").build()).build();

    private static final SdkField<List<String>> TERMINOLOGY_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("TerminologyNames")
            .getter(getter(TranslateTextRequest::terminologyNames))
            .setter(setter(Builder::terminologyNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TerminologyNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SOURCE_LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceLanguageCode").getter(getter(TranslateTextRequest::sourceLanguageCode))
            .setter(setter(Builder::sourceLanguageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceLanguageCode").build())
            .build();

    private static final SdkField<String> TARGET_LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TargetLanguageCode").getter(getter(TranslateTextRequest::targetLanguageCode))
            .setter(setter(Builder::targetLanguageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetLanguageCode").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TEXT_FIELD,
            TERMINOLOGY_NAMES_FIELD, SOURCE_LANGUAGE_CODE_FIELD, TARGET_LANGUAGE_CODE_FIELD));

    private final String text;

    private final List<String> terminologyNames;

    private final String sourceLanguageCode;

    private final String targetLanguageCode;

    private TranslateTextRequest(BuilderImpl builder) {
        super(builder);
        this.text = builder.text;
        this.terminologyNames = builder.terminologyNames;
        this.sourceLanguageCode = builder.sourceLanguageCode;
        this.targetLanguageCode = builder.targetLanguageCode;
    }

    /**
     * <p>
     * The text to translate. The text string can be a maximum of 5,000 bytes long. Depending on your character set,
     * this may be fewer than 5,000 characters.
     * </p>
     * 
     * @return The text to translate. The text string can be a maximum of 5,000 bytes long. Depending on your character
     *         set, this may be fewer than 5,000 characters.
     */
    public final String text() {
        return text;
    }

    /**
     * For responses, this returns true if the service returned a value for the TerminologyNames property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTerminologyNames() {
        return terminologyNames != null && !(terminologyNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The name of the terminology list file to be used in the TranslateText request. You can use 1 terminology list at
     * most in a <code>TranslateText</code> request. Terminology lists can contain a maximum of 256 terms.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTerminologyNames} method.
     * </p>
     * 
     * @return The name of the terminology list file to be used in the TranslateText request. You can use 1 terminology
     *         list at most in a <code>TranslateText</code> request. Terminology lists can contain a maximum of 256
     *         terms.
     */
    public final List<String> terminologyNames() {
        return terminologyNames;
    }

    /**
     * <p>
     * The language code for the language of the source text. The language must be a language supported by Amazon
     * Translate. For a list of language codes, see <a>what-is-languages</a>.
     * </p>
     * <p>
     * To have Amazon Translate determine the source language of your text, you can specify <code>auto</code> in the
     * <code>SourceLanguageCode</code> field. If you specify <code>auto</code>, Amazon Translate will call <a
     * href="https://docs.aws.amazon.com/comprehend/latest/dg/comprehend-general.html">Amazon Comprehend</a> to
     * determine the source language.
     * </p>
     * 
     * @return The language code for the language of the source text. The language must be a language supported by
     *         Amazon Translate. For a list of language codes, see <a>what-is-languages</a>.</p>
     *         <p>
     *         To have Amazon Translate determine the source language of your text, you can specify <code>auto</code> in
     *         the <code>SourceLanguageCode</code> field. If you specify <code>auto</code>, Amazon Translate will call
     *         <a href="https://docs.aws.amazon.com/comprehend/latest/dg/comprehend-general.html">Amazon Comprehend</a>
     *         to determine the source language.
     */
    public final String sourceLanguageCode() {
        return sourceLanguageCode;
    }

    /**
     * <p>
     * The language code requested for the language of the target text. The language must be a language supported by
     * Amazon Translate.
     * </p>
     * 
     * @return The language code requested for the language of the target text. The language must be a language
     *         supported by Amazon Translate.
     */
    public final String targetLanguageCode() {
        return targetLanguageCode;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(text());
        hashCode = 31 * hashCode + Objects.hashCode(hasTerminologyNames() ? terminologyNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(sourceLanguageCode());
        hashCode = 31 * hashCode + Objects.hashCode(targetLanguageCode());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TranslateTextRequest)) {
            return false;
        }
        TranslateTextRequest other = (TranslateTextRequest) obj;
        return Objects.equals(text(), other.text()) && hasTerminologyNames() == other.hasTerminologyNames()
                && Objects.equals(terminologyNames(), other.terminologyNames())
                && Objects.equals(sourceLanguageCode(), other.sourceLanguageCode())
                && Objects.equals(targetLanguageCode(), other.targetLanguageCode());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TranslateTextRequest").add("Text", text())
                .add("TerminologyNames", hasTerminologyNames() ? terminologyNames() : null)
                .add("SourceLanguageCode", sourceLanguageCode()).add("TargetLanguageCode", targetLanguageCode()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Text":
            return Optional.ofNullable(clazz.cast(text()));
        case "TerminologyNames":
            return Optional.ofNullable(clazz.cast(terminologyNames()));
        case "SourceLanguageCode":
            return Optional.ofNullable(clazz.cast(sourceLanguageCode()));
        case "TargetLanguageCode":
            return Optional.ofNullable(clazz.cast(targetLanguageCode()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TranslateTextRequest, T> g) {
        return obj -> g.apply((TranslateTextRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends TranslateRequest.Builder, SdkPojo, CopyableBuilder<Builder, TranslateTextRequest> {
        /**
         * <p>
         * The text to translate. The text string can be a maximum of 5,000 bytes long. Depending on your character set,
         * this may be fewer than 5,000 characters.
         * </p>
         * 
         * @param text
         *        The text to translate. The text string can be a maximum of 5,000 bytes long. Depending on your
         *        character set, this may be fewer than 5,000 characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder text(String text);

        /**
         * <p>
         * The name of the terminology list file to be used in the TranslateText request. You can use 1 terminology list
         * at most in a <code>TranslateText</code> request. Terminology lists can contain a maximum of 256 terms.
         * </p>
         * 
         * @param terminologyNames
         *        The name of the terminology list file to be used in the TranslateText request. You can use 1
         *        terminology list at most in a <code>TranslateText</code> request. Terminology lists can contain a
         *        maximum of 256 terms.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder terminologyNames(Collection<String> terminologyNames);

        /**
         * <p>
         * The name of the terminology list file to be used in the TranslateText request. You can use 1 terminology list
         * at most in a <code>TranslateText</code> request. Terminology lists can contain a maximum of 256 terms.
         * </p>
         * 
         * @param terminologyNames
         *        The name of the terminology list file to be used in the TranslateText request. You can use 1
         *        terminology list at most in a <code>TranslateText</code> request. Terminology lists can contain a
         *        maximum of 256 terms.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder terminologyNames(String... terminologyNames);

        /**
         * <p>
         * The language code for the language of the source text. The language must be a language supported by Amazon
         * Translate. For a list of language codes, see <a>what-is-languages</a>.
         * </p>
         * <p>
         * To have Amazon Translate determine the source language of your text, you can specify <code>auto</code> in the
         * <code>SourceLanguageCode</code> field. If you specify <code>auto</code>, Amazon Translate will call <a
         * href="https://docs.aws.amazon.com/comprehend/latest/dg/comprehend-general.html">Amazon Comprehend</a> to
         * determine the source language.
         * </p>
         * 
         * @param sourceLanguageCode
         *        The language code for the language of the source text. The language must be a language supported by
         *        Amazon Translate. For a list of language codes, see <a>what-is-languages</a>.</p>
         *        <p>
         *        To have Amazon Translate determine the source language of your text, you can specify <code>auto</code>
         *        in the <code>SourceLanguageCode</code> field. If you specify <code>auto</code>, Amazon Translate will
         *        call <a href="https://docs.aws.amazon.com/comprehend/latest/dg/comprehend-general.html">Amazon
         *        Comprehend</a> to determine the source language.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceLanguageCode(String sourceLanguageCode);

        /**
         * <p>
         * The language code requested for the language of the target text. The language must be a language supported by
         * Amazon Translate.
         * </p>
         * 
         * @param targetLanguageCode
         *        The language code requested for the language of the target text. The language must be a language
         *        supported by Amazon Translate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetLanguageCode(String targetLanguageCode);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends TranslateRequest.BuilderImpl implements Builder {
        private String text;

        private List<String> terminologyNames = DefaultSdkAutoConstructList.getInstance();

        private String sourceLanguageCode;

        private String targetLanguageCode;

        private BuilderImpl() {
        }

        private BuilderImpl(TranslateTextRequest model) {
            super(model);
            text(model.text);
            terminologyNames(model.terminologyNames);
            sourceLanguageCode(model.sourceLanguageCode);
            targetLanguageCode(model.targetLanguageCode);
        }

        public final String getText() {
            return text;
        }

        public final void setText(String text) {
            this.text = text;
        }

        @Override
        @Transient
        public final Builder text(String text) {
            this.text = text;
            return this;
        }

        public final Collection<String> getTerminologyNames() {
            if (terminologyNames instanceof SdkAutoConstructList) {
                return null;
            }
            return terminologyNames;
        }

        public final void setTerminologyNames(Collection<String> terminologyNames) {
            this.terminologyNames = ResourceNameListCopier.copy(terminologyNames);
        }

        @Override
        @Transient
        public final Builder terminologyNames(Collection<String> terminologyNames) {
            this.terminologyNames = ResourceNameListCopier.copy(terminologyNames);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder terminologyNames(String... terminologyNames) {
            terminologyNames(Arrays.asList(terminologyNames));
            return this;
        }

        public final String getSourceLanguageCode() {
            return sourceLanguageCode;
        }

        public final void setSourceLanguageCode(String sourceLanguageCode) {
            this.sourceLanguageCode = sourceLanguageCode;
        }

        @Override
        @Transient
        public final Builder sourceLanguageCode(String sourceLanguageCode) {
            this.sourceLanguageCode = sourceLanguageCode;
            return this;
        }

        public final String getTargetLanguageCode() {
            return targetLanguageCode;
        }

        public final void setTargetLanguageCode(String targetLanguageCode) {
            this.targetLanguageCode = targetLanguageCode;
        }

        @Override
        @Transient
        public final Builder targetLanguageCode(String targetLanguageCode) {
            this.targetLanguageCode = targetLanguageCode;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public TranslateTextRequest build() {
            return new TranslateTextRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
