/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.translate;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.translate.internal.TranslateServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.translate.model.ConcurrentModificationException;
import software.amazon.awssdk.services.translate.model.ConflictException;
import software.amazon.awssdk.services.translate.model.CreateParallelDataRequest;
import software.amazon.awssdk.services.translate.model.CreateParallelDataResponse;
import software.amazon.awssdk.services.translate.model.DeleteParallelDataRequest;
import software.amazon.awssdk.services.translate.model.DeleteParallelDataResponse;
import software.amazon.awssdk.services.translate.model.DeleteTerminologyRequest;
import software.amazon.awssdk.services.translate.model.DeleteTerminologyResponse;
import software.amazon.awssdk.services.translate.model.DescribeTextTranslationJobRequest;
import software.amazon.awssdk.services.translate.model.DescribeTextTranslationJobResponse;
import software.amazon.awssdk.services.translate.model.DetectedLanguageLowConfidenceException;
import software.amazon.awssdk.services.translate.model.GetParallelDataRequest;
import software.amazon.awssdk.services.translate.model.GetParallelDataResponse;
import software.amazon.awssdk.services.translate.model.GetTerminologyRequest;
import software.amazon.awssdk.services.translate.model.GetTerminologyResponse;
import software.amazon.awssdk.services.translate.model.ImportTerminologyRequest;
import software.amazon.awssdk.services.translate.model.ImportTerminologyResponse;
import software.amazon.awssdk.services.translate.model.InternalServerException;
import software.amazon.awssdk.services.translate.model.InvalidFilterException;
import software.amazon.awssdk.services.translate.model.InvalidParameterValueException;
import software.amazon.awssdk.services.translate.model.InvalidRequestException;
import software.amazon.awssdk.services.translate.model.LimitExceededException;
import software.amazon.awssdk.services.translate.model.ListLanguagesRequest;
import software.amazon.awssdk.services.translate.model.ListLanguagesResponse;
import software.amazon.awssdk.services.translate.model.ListParallelDataRequest;
import software.amazon.awssdk.services.translate.model.ListParallelDataResponse;
import software.amazon.awssdk.services.translate.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.translate.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.translate.model.ListTerminologiesRequest;
import software.amazon.awssdk.services.translate.model.ListTerminologiesResponse;
import software.amazon.awssdk.services.translate.model.ListTextTranslationJobsRequest;
import software.amazon.awssdk.services.translate.model.ListTextTranslationJobsResponse;
import software.amazon.awssdk.services.translate.model.ResourceNotFoundException;
import software.amazon.awssdk.services.translate.model.ServiceUnavailableException;
import software.amazon.awssdk.services.translate.model.StartTextTranslationJobRequest;
import software.amazon.awssdk.services.translate.model.StartTextTranslationJobResponse;
import software.amazon.awssdk.services.translate.model.StopTextTranslationJobRequest;
import software.amazon.awssdk.services.translate.model.StopTextTranslationJobResponse;
import software.amazon.awssdk.services.translate.model.TagResourceRequest;
import software.amazon.awssdk.services.translate.model.TagResourceResponse;
import software.amazon.awssdk.services.translate.model.TextSizeLimitExceededException;
import software.amazon.awssdk.services.translate.model.TooManyRequestsException;
import software.amazon.awssdk.services.translate.model.TooManyTagsException;
import software.amazon.awssdk.services.translate.model.TranslateDocumentRequest;
import software.amazon.awssdk.services.translate.model.TranslateDocumentResponse;
import software.amazon.awssdk.services.translate.model.TranslateException;
import software.amazon.awssdk.services.translate.model.TranslateTextRequest;
import software.amazon.awssdk.services.translate.model.TranslateTextResponse;
import software.amazon.awssdk.services.translate.model.UnsupportedDisplayLanguageCodeException;
import software.amazon.awssdk.services.translate.model.UnsupportedLanguagePairException;
import software.amazon.awssdk.services.translate.model.UntagResourceRequest;
import software.amazon.awssdk.services.translate.model.UntagResourceResponse;
import software.amazon.awssdk.services.translate.model.UpdateParallelDataRequest;
import software.amazon.awssdk.services.translate.model.UpdateParallelDataResponse;
import software.amazon.awssdk.services.translate.transform.CreateParallelDataRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.DeleteParallelDataRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.DeleteTerminologyRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.DescribeTextTranslationJobRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.GetParallelDataRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.GetTerminologyRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.ImportTerminologyRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.ListLanguagesRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.ListParallelDataRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.ListTerminologiesRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.ListTextTranslationJobsRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.StartTextTranslationJobRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.StopTextTranslationJobRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.TranslateDocumentRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.TranslateTextRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.UpdateParallelDataRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link TranslateClient}.
 *
 * @see TranslateClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTranslateClient implements TranslateClient {
    private static final Logger log = Logger.loggerFor(DefaultTranslateClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultTranslateClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a parallel data resource in Amazon Translate by importing an input file from Amazon S3. Parallel data
     * files contain examples that show how you want segments of text to be translated. By adding parallel data, you can
     * influence the style, tone, and word choice in your translation output.
     * </p>
     *
     * @param createParallelDataRequest
     * @return Result of the CreateParallelData operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of the parameter is not valid. Review the value of the parameter you are using to correct it,
     *         and then retry your operation.
     * @throws InvalidRequestException
     *         The request that you made is not valid. Check your request to determine why it's not valid and then retry
     *         the request.
     * @throws LimitExceededException
     *         The specified limit has been exceeded. Review your request and retry it with a quantity below the stated
     *         limit.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws TooManyTagsException
     *         You have added too many tags to this resource. The maximum is 50 tags.
     * @throws ConflictException
     *         There was a conflict processing the request. Try your request again.
     * @throws ConcurrentModificationException
     *         Another modification is being made. That modification must complete before you can make your change.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.CreateParallelData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/CreateParallelData" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateParallelDataResponse createParallelData(CreateParallelDataRequest createParallelDataRequest)
            throws InvalidParameterValueException, InvalidRequestException, LimitExceededException, TooManyRequestsException,
            TooManyTagsException, ConflictException, ConcurrentModificationException, InternalServerException,
            AwsServiceException, SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateParallelDataResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateParallelDataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createParallelDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createParallelDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateParallelData");

            return clientHandler.execute(new ClientExecutionParams<CreateParallelDataRequest, CreateParallelDataResponse>()
                    .withOperationName("CreateParallelData").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createParallelDataRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateParallelDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a parallel data resource in Amazon Translate.
     * </p>
     *
     * @param deleteParallelDataRequest
     * @return Result of the DeleteParallelData operation returned by the service.
     * @throws ConcurrentModificationException
     *         Another modification is being made. That modification must complete before you can make your change.
     * @throws ResourceNotFoundException
     *         The resource you are looking for has not been found. Review the resource you're looking for and see if a
     *         different resource will accomplish your needs before retrying the revised request.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.DeleteParallelData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/DeleteParallelData" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteParallelDataResponse deleteParallelData(DeleteParallelDataRequest deleteParallelDataRequest)
            throws ConcurrentModificationException, ResourceNotFoundException, TooManyRequestsException, InternalServerException,
            AwsServiceException, SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteParallelDataResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteParallelDataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteParallelDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteParallelDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteParallelData");

            return clientHandler.execute(new ClientExecutionParams<DeleteParallelDataRequest, DeleteParallelDataResponse>()
                    .withOperationName("DeleteParallelData").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteParallelDataRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteParallelDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * A synchronous action that deletes a custom terminology.
     * </p>
     *
     * @param deleteTerminologyRequest
     * @return Result of the DeleteTerminology operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource you are looking for has not been found. Review the resource you're looking for and see if a
     *         different resource will accomplish your needs before retrying the revised request.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws InvalidParameterValueException
     *         The value of the parameter is not valid. Review the value of the parameter you are using to correct it,
     *         and then retry your operation.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.DeleteTerminology
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/DeleteTerminology" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteTerminologyResponse deleteTerminology(DeleteTerminologyRequest deleteTerminologyRequest)
            throws ResourceNotFoundException, TooManyRequestsException, InvalidParameterValueException, InternalServerException,
            AwsServiceException, SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTerminologyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteTerminologyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTerminologyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTerminologyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTerminology");

            return clientHandler.execute(new ClientExecutionParams<DeleteTerminologyRequest, DeleteTerminologyResponse>()
                    .withOperationName("DeleteTerminology").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteTerminologyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteTerminologyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the properties associated with an asynchronous batch translation job including name, ID, status, source and
     * target languages, input/output S3 buckets, and so on.
     * </p>
     *
     * @param describeTextTranslationJobRequest
     * @return Result of the DescribeTextTranslationJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource you are looking for has not been found. Review the resource you're looking for and see if a
     *         different resource will accomplish your needs before retrying the revised request.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.DescribeTextTranslationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/DescribeTextTranslationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeTextTranslationJobResponse describeTextTranslationJob(
            DescribeTextTranslationJobRequest describeTextTranslationJobRequest) throws ResourceNotFoundException,
            TooManyRequestsException, InternalServerException, AwsServiceException, SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTextTranslationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeTextTranslationJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeTextTranslationJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTextTranslationJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTextTranslationJob");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeTextTranslationJobRequest, DescribeTextTranslationJobResponse>()
                            .withOperationName("DescribeTextTranslationJob").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeTextTranslationJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeTextTranslationJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides information about a parallel data resource.
     * </p>
     *
     * @param getParallelDataRequest
     * @return Result of the GetParallelData operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource you are looking for has not been found. Review the resource you're looking for and see if a
     *         different resource will accomplish your needs before retrying the revised request.
     * @throws InvalidParameterValueException
     *         The value of the parameter is not valid. Review the value of the parameter you are using to correct it,
     *         and then retry your operation.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.GetParallelData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/GetParallelData" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetParallelDataResponse getParallelData(GetParallelDataRequest getParallelDataRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, TooManyRequestsException, InternalServerException,
            AwsServiceException, SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetParallelDataResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetParallelDataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getParallelDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getParallelDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetParallelData");

            return clientHandler.execute(new ClientExecutionParams<GetParallelDataRequest, GetParallelDataResponse>()
                    .withOperationName("GetParallelData").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getParallelDataRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetParallelDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a custom terminology.
     * </p>
     *
     * @param getTerminologyRequest
     * @return Result of the GetTerminology operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource you are looking for has not been found. Review the resource you're looking for and see if a
     *         different resource will accomplish your needs before retrying the revised request.
     * @throws InvalidParameterValueException
     *         The value of the parameter is not valid. Review the value of the parameter you are using to correct it,
     *         and then retry your operation.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.GetTerminology
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/GetTerminology" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTerminologyResponse getTerminology(GetTerminologyRequest getTerminologyRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, TooManyRequestsException, InternalServerException, AwsServiceException,
            SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTerminologyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTerminologyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTerminologyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTerminologyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTerminology");

            return clientHandler.execute(new ClientExecutionParams<GetTerminologyRequest, GetTerminologyResponse>()
                    .withOperationName("GetTerminology").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getTerminologyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTerminologyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates a custom terminology, depending on whether one already exists for the given terminology name.
     * Importing a terminology with the same name as an existing one will merge the terminologies based on the chosen
     * merge strategy. The only supported merge strategy is OVERWRITE, where the imported terminology overwrites the
     * existing terminology of the same name.
     * </p>
     * <p>
     * If you import a terminology that overwrites an existing one, the new terminology takes up to 10 minutes to fully
     * propagate. After that, translations have access to the new terminology.
     * </p>
     *
     * @param importTerminologyRequest
     * @return Result of the ImportTerminology operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of the parameter is not valid. Review the value of the parameter you are using to correct it,
     *         and then retry your operation.
     * @throws LimitExceededException
     *         The specified limit has been exceeded. Review your request and retry it with a quantity below the stated
     *         limit.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws TooManyTagsException
     *         You have added too many tags to this resource. The maximum is 50 tags.
     * @throws ConcurrentModificationException
     *         Another modification is being made. That modification must complete before you can make your change.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.ImportTerminology
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ImportTerminology" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ImportTerminologyResponse importTerminology(ImportTerminologyRequest importTerminologyRequest)
            throws InvalidParameterValueException, LimitExceededException, TooManyRequestsException, TooManyTagsException,
            ConcurrentModificationException, InternalServerException, AwsServiceException, SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ImportTerminologyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ImportTerminologyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(importTerminologyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, importTerminologyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportTerminology");

            return clientHandler.execute(new ClientExecutionParams<ImportTerminologyRequest, ImportTerminologyResponse>()
                    .withOperationName("ImportTerminology").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(importTerminologyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ImportTerminologyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of languages (RFC-5646 codes and names) that Amazon Translate supports.
     * </p>
     *
     * @param listLanguagesRequest
     * @return Result of the ListLanguages operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of the parameter is not valid. Review the value of the parameter you are using to correct it,
     *         and then retry your operation.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws UnsupportedDisplayLanguageCodeException
     *         Requested display language code is not supported.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.ListLanguages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListLanguages" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListLanguagesResponse listLanguages(ListLanguagesRequest listLanguagesRequest) throws InvalidParameterValueException,
            TooManyRequestsException, UnsupportedDisplayLanguageCodeException, InternalServerException, AwsServiceException,
            SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListLanguagesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListLanguagesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listLanguagesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLanguagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLanguages");

            return clientHandler.execute(new ClientExecutionParams<ListLanguagesRequest, ListLanguagesResponse>()
                    .withOperationName("ListLanguages").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listLanguagesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListLanguagesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of your parallel data resources in Amazon Translate.
     * </p>
     *
     * @param listParallelDataRequest
     * @return Result of the ListParallelData operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of the parameter is not valid. Review the value of the parameter you are using to correct it,
     *         and then retry your operation.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.ListParallelData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListParallelData" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListParallelDataResponse listParallelData(ListParallelDataRequest listParallelDataRequest)
            throws InvalidParameterValueException, TooManyRequestsException, InternalServerException, AwsServiceException,
            SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListParallelDataResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListParallelDataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listParallelDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listParallelDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListParallelData");

            return clientHandler.execute(new ClientExecutionParams<ListParallelDataRequest, ListParallelDataResponse>()
                    .withOperationName("ListParallelData").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listParallelDataRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListParallelDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all tags associated with a given Amazon Translate resource. For more information, see <a
     * href="https://docs.aws.amazon.com/translate/latest/dg/tagging.html"> Tagging your resources</a>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of the parameter is not valid. Review the value of the parameter you are using to correct it,
     *         and then retry your operation.
     * @throws ResourceNotFoundException
     *         The resource you are looking for has not been found. Review the resource you're looking for and see if a
     *         different resource will accomplish your needs before retrying the revised request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InvalidParameterValueException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of custom terminologies associated with your account.
     * </p>
     *
     * @param listTerminologiesRequest
     * @return Result of the ListTerminologies operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of the parameter is not valid. Review the value of the parameter you are using to correct it,
     *         and then retry your operation.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.ListTerminologies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListTerminologies" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTerminologiesResponse listTerminologies(ListTerminologiesRequest listTerminologiesRequest)
            throws InvalidParameterValueException, TooManyRequestsException, InternalServerException, AwsServiceException,
            SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTerminologiesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTerminologiesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTerminologiesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTerminologiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTerminologies");

            return clientHandler.execute(new ClientExecutionParams<ListTerminologiesRequest, ListTerminologiesResponse>()
                    .withOperationName("ListTerminologies").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTerminologiesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTerminologiesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a list of the batch translation jobs that you have submitted.
     * </p>
     *
     * @param listTextTranslationJobsRequest
     * @return Result of the ListTextTranslationJobs operation returned by the service.
     * @throws InvalidRequestException
     *         The request that you made is not valid. Check your request to determine why it's not valid and then retry
     *         the request.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws InvalidFilterException
     *         The filter specified for the operation is not valid. Specify a different filter.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.ListTextTranslationJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListTextTranslationJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTextTranslationJobsResponse listTextTranslationJobs(ListTextTranslationJobsRequest listTextTranslationJobsRequest)
            throws InvalidRequestException, TooManyRequestsException, InvalidFilterException, InternalServerException,
            AwsServiceException, SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTextTranslationJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTextTranslationJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTextTranslationJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTextTranslationJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTextTranslationJobs");

            return clientHandler
                    .execute(new ClientExecutionParams<ListTextTranslationJobsRequest, ListTextTranslationJobsResponse>()
                            .withOperationName("ListTextTranslationJobs").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listTextTranslationJobsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListTextTranslationJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts an asynchronous batch translation job. Use batch translation jobs to translate large volumes of text
     * across multiple documents at once. For batch translation, you can input documents with different source languages
     * (specify <code>auto</code> as the source language). You can specify one or more target languages. Batch
     * translation translates each input document into each of the target languages. For more information, see <a
     * href="https://docs.aws.amazon.com/translate/latest/dg/async.html">Asynchronous batch processing</a>.
     * </p>
     * <p>
     * Batch translation jobs can be described with the <a>DescribeTextTranslationJob</a> operation, listed with the
     * <a>ListTextTranslationJobs</a> operation, and stopped with the <a>StopTextTranslationJob</a> operation.
     * </p>
     *
     * @param startTextTranslationJobRequest
     * @return Result of the StartTextTranslationJob operation returned by the service.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws UnsupportedLanguagePairException
     *         Amazon Translate does not support translation from the language of the source text into the requested
     *         target language. For more information, see <a
     *         href="https://docs.aws.amazon.com/translate/latest/dg/what-is-languages.html">Supported languages</a>.
     * @throws InvalidRequestException
     *         The request that you made is not valid. Check your request to determine why it's not valid and then retry
     *         the request.
     * @throws ResourceNotFoundException
     *         The resource you are looking for has not been found. Review the resource you're looking for and see if a
     *         different resource will accomplish your needs before retrying the revised request.
     * @throws InvalidParameterValueException
     *         The value of the parameter is not valid. Review the value of the parameter you are using to correct it,
     *         and then retry your operation.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.StartTextTranslationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/StartTextTranslationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartTextTranslationJobResponse startTextTranslationJob(StartTextTranslationJobRequest startTextTranslationJobRequest)
            throws TooManyRequestsException, UnsupportedLanguagePairException, InvalidRequestException,
            ResourceNotFoundException, InvalidParameterValueException, InternalServerException, AwsServiceException,
            SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartTextTranslationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartTextTranslationJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startTextTranslationJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startTextTranslationJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartTextTranslationJob");

            return clientHandler
                    .execute(new ClientExecutionParams<StartTextTranslationJobRequest, StartTextTranslationJobResponse>()
                            .withOperationName("StartTextTranslationJob").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(startTextTranslationJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartTextTranslationJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops an asynchronous batch translation job that is in progress.
     * </p>
     * <p>
     * If the job's state is <code>IN_PROGRESS</code>, the job will be marked for termination and put into the
     * <code>STOP_REQUESTED</code> state. If the job completes before it can be stopped, it is put into the
     * <code>COMPLETED</code> state. Otherwise, the job is put into the <code>STOPPED</code> state.
     * </p>
     * <p>
     * Asynchronous batch translation jobs are started with the <a>StartTextTranslationJob</a> operation. You can use
     * the <a>DescribeTextTranslationJob</a> or <a>ListTextTranslationJobs</a> operations to get a batch translation
     * job's <code>JobId</code>.
     * </p>
     *
     * @param stopTextTranslationJobRequest
     * @return Result of the StopTextTranslationJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource you are looking for has not been found. Review the resource you're looking for and see if a
     *         different resource will accomplish your needs before retrying the revised request.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.StopTextTranslationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/StopTextTranslationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StopTextTranslationJobResponse stopTextTranslationJob(StopTextTranslationJobRequest stopTextTranslationJobRequest)
            throws ResourceNotFoundException, TooManyRequestsException, InternalServerException, AwsServiceException,
            SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopTextTranslationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StopTextTranslationJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopTextTranslationJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopTextTranslationJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopTextTranslationJob");

            return clientHandler
                    .execute(new ClientExecutionParams<StopTextTranslationJobRequest, StopTextTranslationJobResponse>()
                            .withOperationName("StopTextTranslationJob").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(stopTextTranslationJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StopTextTranslationJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associates a specific tag with a resource. A tag is a key-value pair that adds as a metadata to a resource. For
     * more information, see <a href="https://docs.aws.amazon.com/translate/latest/dg/tagging.html"> Tagging your
     * resources</a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of the parameter is not valid. Review the value of the parameter you are using to correct it,
     *         and then retry your operation.
     * @throws ConcurrentModificationException
     *         Another modification is being made. That modification must complete before you can make your change.
     * @throws ResourceNotFoundException
     *         The resource you are looking for has not been found. Review the resource you're looking for and see if a
     *         different resource will accomplish your needs before retrying the revised request.
     * @throws TooManyTagsException
     *         You have added too many tags to this resource. The maximum is 50 tags.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InvalidParameterValueException,
            ConcurrentModificationException, ResourceNotFoundException, TooManyTagsException, InternalServerException,
            AwsServiceException, SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Translates the input document from the source language to the target language. This synchronous operation
     * supports text, HTML, or Word documents as the input document. <code>TranslateDocument</code> supports
     * translations from English to any supported language, and from any supported language to English. Therefore,
     * specify either the source language code or the target language code as “en” (English).
     * </p>
     * <p>
     * If you set the <code>Formality</code> parameter, the request will fail if the target language does not support
     * formality. For a list of target languages that support formality, see <a
     * href="https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-formality.html">Setting
     * formality</a>.
     * </p>
     *
     * @param translateDocumentRequest
     * @return Result of the TranslateDocument operation returned by the service.
     * @throws InvalidRequestException
     *         The request that you made is not valid. Check your request to determine why it's not valid and then retry
     *         the request.
     * @throws LimitExceededException
     *         The specified limit has been exceeded. Review your request and retry it with a quantity below the stated
     *         limit.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws ResourceNotFoundException
     *         The resource you are looking for has not been found. Review the resource you're looking for and see if a
     *         different resource will accomplish your needs before retrying the revised request.
     * @throws UnsupportedLanguagePairException
     *         Amazon Translate does not support translation from the language of the source text into the requested
     *         target language. For more information, see <a
     *         href="https://docs.aws.amazon.com/translate/latest/dg/what-is-languages.html">Supported languages</a>.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ServiceUnavailableException
     *         The Amazon Translate service is temporarily unavailable. Wait a bit and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.TranslateDocument
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/TranslateDocument" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TranslateDocumentResponse translateDocument(TranslateDocumentRequest translateDocumentRequest)
            throws InvalidRequestException, LimitExceededException, TooManyRequestsException, ResourceNotFoundException,
            UnsupportedLanguagePairException, InternalServerException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TranslateDocumentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TranslateDocumentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(translateDocumentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, translateDocumentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TranslateDocument");

            return clientHandler.execute(new ClientExecutionParams<TranslateDocumentRequest, TranslateDocumentResponse>()
                    .withOperationName("TranslateDocument").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(translateDocumentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TranslateDocumentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Translates input text from the source language to the target language. For a list of available languages and
     * language codes, see <a href="https://docs.aws.amazon.com/translate/latest/dg/what-is-languages.html">Supported
     * languages</a>.
     * </p>
     *
     * @param translateTextRequest
     * @return Result of the TranslateText operation returned by the service.
     * @throws InvalidRequestException
     *         The request that you made is not valid. Check your request to determine why it's not valid and then retry
     *         the request.
     * @throws TextSizeLimitExceededException
     *         The size of the text you submitted exceeds the size limit. Reduce the size of the text or use a smaller
     *         document and then retry your request.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws UnsupportedLanguagePairException
     *         Amazon Translate does not support translation from the language of the source text into the requested
     *         target language. For more information, see <a
     *         href="https://docs.aws.amazon.com/translate/latest/dg/what-is-languages.html">Supported languages</a>.
     * @throws DetectedLanguageLowConfidenceException
     *         The confidence that Amazon Comprehend accurately detected the source language is low. If a low confidence
     *         level is acceptable for your application, you can use the language in the exception to call Amazon
     *         Translate again. For more information, see the <a
     *         href="https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectDominantLanguage.html"
     *         >DetectDominantLanguage</a> operation in the <i>Amazon Comprehend Developer Guide</i>.
     * @throws ResourceNotFoundException
     *         The resource you are looking for has not been found. Review the resource you're looking for and see if a
     *         different resource will accomplish your needs before retrying the revised request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ServiceUnavailableException
     *         The Amazon Translate service is temporarily unavailable. Wait a bit and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.TranslateText
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/TranslateText" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TranslateTextResponse translateText(TranslateTextRequest translateTextRequest) throws InvalidRequestException,
            TextSizeLimitExceededException, TooManyRequestsException, UnsupportedLanguagePairException,
            DetectedLanguageLowConfidenceException, ResourceNotFoundException, InternalServerException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TranslateTextResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TranslateTextResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(translateTextRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, translateTextRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TranslateText");

            return clientHandler.execute(new ClientExecutionParams<TranslateTextRequest, TranslateTextResponse>()
                    .withOperationName("TranslateText").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(translateTextRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TranslateTextRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a specific tag associated with an Amazon Translate resource. For more information, see <a
     * href="https://docs.aws.amazon.com/translate/latest/dg/tagging.html"> Tagging your resources</a>.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of the parameter is not valid. Review the value of the parameter you are using to correct it,
     *         and then retry your operation.
     * @throws ConcurrentModificationException
     *         Another modification is being made. That modification must complete before you can make your change.
     * @throws ResourceNotFoundException
     *         The resource you are looking for has not been found. Review the resource you're looking for and see if a
     *         different resource will accomplish your needs before retrying the revised request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InvalidParameterValueException,
            ConcurrentModificationException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a previously created parallel data resource by importing a new input file from Amazon S3.
     * </p>
     *
     * @param updateParallelDataRequest
     * @return Result of the UpdateParallelData operation returned by the service.
     * @throws ConcurrentModificationException
     *         Another modification is being made. That modification must complete before you can make your change.
     * @throws InvalidParameterValueException
     *         The value of the parameter is not valid. Review the value of the parameter you are using to correct it,
     *         and then retry your operation.
     * @throws InvalidRequestException
     *         The request that you made is not valid. Check your request to determine why it's not valid and then retry
     *         the request.
     * @throws LimitExceededException
     *         The specified limit has been exceeded. Review your request and retry it with a quantity below the stated
     *         limit.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again.
     * @throws ConflictException
     *         There was a conflict processing the request. Try your request again.
     * @throws ResourceNotFoundException
     *         The resource you are looking for has not been found. Review the resource you're looking for and see if a
     *         different resource will accomplish your needs before retrying the revised request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TranslateException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TranslateClient.UpdateParallelData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/UpdateParallelData" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateParallelDataResponse updateParallelData(UpdateParallelDataRequest updateParallelDataRequest)
            throws ConcurrentModificationException, InvalidParameterValueException, InvalidRequestException,
            LimitExceededException, TooManyRequestsException, ConflictException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, TranslateException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateParallelDataResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateParallelDataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateParallelDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateParallelDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateParallelData");

            return clientHandler.execute(new ClientExecutionParams<UpdateParallelDataRequest, UpdateParallelDataResponse>()
                    .withOperationName("UpdateParallelData").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateParallelDataRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateParallelDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        TranslateServiceClientConfigurationBuilder serviceConfigBuilder = new TranslateServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(TranslateException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DetectedLanguageLowConfidenceException")
                                .exceptionBuilderSupplier(DetectedLanguageLowConfidenceException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedLanguagePairException")
                                .exceptionBuilderSupplier(UnsupportedLanguagePairException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidFilterException")
                                .exceptionBuilderSupplier(InvalidFilterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                                .exceptionBuilderSupplier(TextSizeLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedDisplayLanguageCodeException")
                                .exceptionBuilderSupplier(UnsupportedDisplayLanguageCodeException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build());
    }

    @Override
    public final TranslateServiceClientConfiguration serviceClientConfiguration() {
        return new TranslateServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
