/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.translate.model;

import java.beans.Transient;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class TranslateDocumentRequest extends TranslateRequest implements
        ToCopyableBuilder<TranslateDocumentRequest.Builder, TranslateDocumentRequest> {
    private static final SdkField<Document> DOCUMENT_FIELD = SdkField.<Document> builder(MarshallingType.SDK_POJO)
            .memberName("Document").getter(getter(TranslateDocumentRequest::document)).setter(setter(Builder::document))
            .constructor(Document::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Document").build()).build();

    private static final SdkField<List<String>> TERMINOLOGY_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("TerminologyNames")
            .getter(getter(TranslateDocumentRequest::terminologyNames))
            .setter(setter(Builder::terminologyNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TerminologyNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SOURCE_LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceLanguageCode").getter(getter(TranslateDocumentRequest::sourceLanguageCode))
            .setter(setter(Builder::sourceLanguageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceLanguageCode").build())
            .build();

    private static final SdkField<String> TARGET_LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TargetLanguageCode").getter(getter(TranslateDocumentRequest::targetLanguageCode))
            .setter(setter(Builder::targetLanguageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetLanguageCode").build())
            .build();

    private static final SdkField<TranslationSettings> SETTINGS_FIELD = SdkField
            .<TranslationSettings> builder(MarshallingType.SDK_POJO).memberName("Settings")
            .getter(getter(TranslateDocumentRequest::settings)).setter(setter(Builder::settings))
            .constructor(TranslationSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Settings").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DOCUMENT_FIELD,
            TERMINOLOGY_NAMES_FIELD, SOURCE_LANGUAGE_CODE_FIELD, TARGET_LANGUAGE_CODE_FIELD, SETTINGS_FIELD));

    private final Document document;

    private final List<String> terminologyNames;

    private final String sourceLanguageCode;

    private final String targetLanguageCode;

    private final TranslationSettings settings;

    private TranslateDocumentRequest(BuilderImpl builder) {
        super(builder);
        this.document = builder.document;
        this.terminologyNames = builder.terminologyNames;
        this.sourceLanguageCode = builder.sourceLanguageCode;
        this.targetLanguageCode = builder.targetLanguageCode;
        this.settings = builder.settings;
    }

    /**
     * <p>
     * The content and content type for the document to be translated. The document size must not exceed 100 KB.
     * </p>
     * 
     * @return The content and content type for the document to be translated. The document size must not exceed 100 KB.
     */
    public final Document document() {
        return document;
    }

    /**
     * For responses, this returns true if the service returned a value for the TerminologyNames property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTerminologyNames() {
        return terminologyNames != null && !(terminologyNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The name of a terminology list file to add to the translation job. This file provides source terms and the
     * desired translation for each term. A terminology list can contain a maximum of 256 terms. You can use one custom
     * terminology resource in your translation request.
     * </p>
     * <p>
     * Use the <a>ListTerminologies</a> operation to get the available terminology lists.
     * </p>
     * <p>
     * For more information about custom terminology lists, see <a
     * href="https://docs.aws.amazon.com/translate/latest/dg/how-custom-terminology.html">Custom terminology</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTerminologyNames} method.
     * </p>
     * 
     * @return The name of a terminology list file to add to the translation job. This file provides source terms and
     *         the desired translation for each term. A terminology list can contain a maximum of 256 terms. You can use
     *         one custom terminology resource in your translation request.</p>
     *         <p>
     *         Use the <a>ListTerminologies</a> operation to get the available terminology lists.
     *         </p>
     *         <p>
     *         For more information about custom terminology lists, see <a
     *         href="https://docs.aws.amazon.com/translate/latest/dg/how-custom-terminology.html">Custom
     *         terminology</a>.
     */
    public final List<String> terminologyNames() {
        return terminologyNames;
    }

    /**
     * <p>
     * The language code for the language of the source text. For a list of supported language codes, see <a
     * href="https://docs.aws.amazon.com/translate/latest/dg/what-is-languages.html">Supported languages</a>.
     * </p>
     * <p>
     * To have Amazon Translate determine the source language of your text, you can specify <code>auto</code> in the
     * <code>SourceLanguageCode</code> field. If you specify <code>auto</code>, Amazon Translate will call <a
     * href="https://docs.aws.amazon.com/comprehend/latest/dg/comprehend-general.html">Amazon Comprehend</a> to
     * determine the source language.
     * </p>
     * <note>
     * <p>
     * If you specify <code>auto</code>, you must send the <code>TranslateDocument</code> request in a region that
     * supports Amazon Comprehend. Otherwise, the request returns an error indicating that autodetect is not supported.
     * </p>
     * </note>
     * 
     * @return The language code for the language of the source text. For a list of supported language codes, see <a
     *         href="https://docs.aws.amazon.com/translate/latest/dg/what-is-languages.html">Supported
     *         languages</a>.</p>
     *         <p>
     *         To have Amazon Translate determine the source language of your text, you can specify <code>auto</code> in
     *         the <code>SourceLanguageCode</code> field. If you specify <code>auto</code>, Amazon Translate will call
     *         <a href="https://docs.aws.amazon.com/comprehend/latest/dg/comprehend-general.html">Amazon Comprehend</a>
     *         to determine the source language.
     *         </p>
     *         <note>
     *         <p>
     *         If you specify <code>auto</code>, you must send the <code>TranslateDocument</code> request in a region
     *         that supports Amazon Comprehend. Otherwise, the request returns an error indicating that autodetect is
     *         not supported.
     *         </p>
     */
    public final String sourceLanguageCode() {
        return sourceLanguageCode;
    }

    /**
     * <p>
     * The language code requested for the translated document. For a list of supported language codes, see <a
     * href="https://docs.aws.amazon.com/translate/latest/dg/what-is-languages.html">Supported languages</a>.
     * </p>
     * 
     * @return The language code requested for the translated document. For a list of supported language codes, see <a
     *         href="https://docs.aws.amazon.com/translate/latest/dg/what-is-languages.html">Supported languages</a>.
     */
    public final String targetLanguageCode() {
        return targetLanguageCode;
    }

    /**
     * <p>
     * Settings to configure your translation output. You can configure the following options:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Brevity: not supported.
     * </p>
     * </li>
     * <li>
     * <p>
     * Formality: sets the formality level of the output text.
     * </p>
     * </li>
     * <li>
     * <p>
     * Profanity: masks profane words and phrases in your translation output.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Settings to configure your translation output. You can configure the following options:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Brevity: not supported.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Formality: sets the formality level of the output text.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Profanity: masks profane words and phrases in your translation output.
     *         </p>
     *         </li>
     */
    public final TranslationSettings settings() {
        return settings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(document());
        hashCode = 31 * hashCode + Objects.hashCode(hasTerminologyNames() ? terminologyNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(sourceLanguageCode());
        hashCode = 31 * hashCode + Objects.hashCode(targetLanguageCode());
        hashCode = 31 * hashCode + Objects.hashCode(settings());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TranslateDocumentRequest)) {
            return false;
        }
        TranslateDocumentRequest other = (TranslateDocumentRequest) obj;
        return Objects.equals(document(), other.document()) && hasTerminologyNames() == other.hasTerminologyNames()
                && Objects.equals(terminologyNames(), other.terminologyNames())
                && Objects.equals(sourceLanguageCode(), other.sourceLanguageCode())
                && Objects.equals(targetLanguageCode(), other.targetLanguageCode())
                && Objects.equals(settings(), other.settings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TranslateDocumentRequest").add("Document", document())
                .add("TerminologyNames", hasTerminologyNames() ? terminologyNames() : null)
                .add("SourceLanguageCode", sourceLanguageCode()).add("TargetLanguageCode", targetLanguageCode())
                .add("Settings", settings()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Document":
            return Optional.ofNullable(clazz.cast(document()));
        case "TerminologyNames":
            return Optional.ofNullable(clazz.cast(terminologyNames()));
        case "SourceLanguageCode":
            return Optional.ofNullable(clazz.cast(sourceLanguageCode()));
        case "TargetLanguageCode":
            return Optional.ofNullable(clazz.cast(targetLanguageCode()));
        case "Settings":
            return Optional.ofNullable(clazz.cast(settings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TranslateDocumentRequest, T> g) {
        return obj -> g.apply((TranslateDocumentRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends TranslateRequest.Builder, SdkPojo, CopyableBuilder<Builder, TranslateDocumentRequest> {
        /**
         * <p>
         * The content and content type for the document to be translated. The document size must not exceed 100 KB.
         * </p>
         * 
         * @param document
         *        The content and content type for the document to be translated. The document size must not exceed 100
         *        KB.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder document(Document document);

        /**
         * <p>
         * The content and content type for the document to be translated. The document size must not exceed 100 KB.
         * </p>
         * This is a convenience method that creates an instance of the {@link Document.Builder} avoiding the need to
         * create one manually via {@link Document#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Document.Builder#build()} is called immediately and its result is
         * passed to {@link #document(Document)}.
         * 
         * @param document
         *        a consumer that will call methods on {@link Document.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #document(Document)
         */
        default Builder document(Consumer<Document.Builder> document) {
            return document(Document.builder().applyMutation(document).build());
        }

        /**
         * <p>
         * The name of a terminology list file to add to the translation job. This file provides source terms and the
         * desired translation for each term. A terminology list can contain a maximum of 256 terms. You can use one
         * custom terminology resource in your translation request.
         * </p>
         * <p>
         * Use the <a>ListTerminologies</a> operation to get the available terminology lists.
         * </p>
         * <p>
         * For more information about custom terminology lists, see <a
         * href="https://docs.aws.amazon.com/translate/latest/dg/how-custom-terminology.html">Custom terminology</a>.
         * </p>
         * 
         * @param terminologyNames
         *        The name of a terminology list file to add to the translation job. This file provides source terms and
         *        the desired translation for each term. A terminology list can contain a maximum of 256 terms. You can
         *        use one custom terminology resource in your translation request.</p>
         *        <p>
         *        Use the <a>ListTerminologies</a> operation to get the available terminology lists.
         *        </p>
         *        <p>
         *        For more information about custom terminology lists, see <a
         *        href="https://docs.aws.amazon.com/translate/latest/dg/how-custom-terminology.html">Custom
         *        terminology</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder terminologyNames(Collection<String> terminologyNames);

        /**
         * <p>
         * The name of a terminology list file to add to the translation job. This file provides source terms and the
         * desired translation for each term. A terminology list can contain a maximum of 256 terms. You can use one
         * custom terminology resource in your translation request.
         * </p>
         * <p>
         * Use the <a>ListTerminologies</a> operation to get the available terminology lists.
         * </p>
         * <p>
         * For more information about custom terminology lists, see <a
         * href="https://docs.aws.amazon.com/translate/latest/dg/how-custom-terminology.html">Custom terminology</a>.
         * </p>
         * 
         * @param terminologyNames
         *        The name of a terminology list file to add to the translation job. This file provides source terms and
         *        the desired translation for each term. A terminology list can contain a maximum of 256 terms. You can
         *        use one custom terminology resource in your translation request.</p>
         *        <p>
         *        Use the <a>ListTerminologies</a> operation to get the available terminology lists.
         *        </p>
         *        <p>
         *        For more information about custom terminology lists, see <a
         *        href="https://docs.aws.amazon.com/translate/latest/dg/how-custom-terminology.html">Custom
         *        terminology</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder terminologyNames(String... terminologyNames);

        /**
         * <p>
         * The language code for the language of the source text. For a list of supported language codes, see <a
         * href="https://docs.aws.amazon.com/translate/latest/dg/what-is-languages.html">Supported languages</a>.
         * </p>
         * <p>
         * To have Amazon Translate determine the source language of your text, you can specify <code>auto</code> in the
         * <code>SourceLanguageCode</code> field. If you specify <code>auto</code>, Amazon Translate will call <a
         * href="https://docs.aws.amazon.com/comprehend/latest/dg/comprehend-general.html">Amazon Comprehend</a> to
         * determine the source language.
         * </p>
         * <note>
         * <p>
         * If you specify <code>auto</code>, you must send the <code>TranslateDocument</code> request in a region that
         * supports Amazon Comprehend. Otherwise, the request returns an error indicating that autodetect is not
         * supported.
         * </p>
         * </note>
         * 
         * @param sourceLanguageCode
         *        The language code for the language of the source text. For a list of supported language codes, see <a
         *        href="https://docs.aws.amazon.com/translate/latest/dg/what-is-languages.html">Supported
         *        languages</a>.</p>
         *        <p>
         *        To have Amazon Translate determine the source language of your text, you can specify <code>auto</code>
         *        in the <code>SourceLanguageCode</code> field. If you specify <code>auto</code>, Amazon Translate will
         *        call <a href="https://docs.aws.amazon.com/comprehend/latest/dg/comprehend-general.html">Amazon
         *        Comprehend</a> to determine the source language.
         *        </p>
         *        <note>
         *        <p>
         *        If you specify <code>auto</code>, you must send the <code>TranslateDocument</code> request in a region
         *        that supports Amazon Comprehend. Otherwise, the request returns an error indicating that autodetect is
         *        not supported.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceLanguageCode(String sourceLanguageCode);

        /**
         * <p>
         * The language code requested for the translated document. For a list of supported language codes, see <a
         * href="https://docs.aws.amazon.com/translate/latest/dg/what-is-languages.html">Supported languages</a>.
         * </p>
         * 
         * @param targetLanguageCode
         *        The language code requested for the translated document. For a list of supported language codes, see
         *        <a href="https://docs.aws.amazon.com/translate/latest/dg/what-is-languages.html">Supported
         *        languages</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetLanguageCode(String targetLanguageCode);

        /**
         * <p>
         * Settings to configure your translation output. You can configure the following options:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Brevity: not supported.
         * </p>
         * </li>
         * <li>
         * <p>
         * Formality: sets the formality level of the output text.
         * </p>
         * </li>
         * <li>
         * <p>
         * Profanity: masks profane words and phrases in your translation output.
         * </p>
         * </li>
         * </ul>
         * 
         * @param settings
         *        Settings to configure your translation output. You can configure the following options:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Brevity: not supported.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Formality: sets the formality level of the output text.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Profanity: masks profane words and phrases in your translation output.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder settings(TranslationSettings settings);

        /**
         * <p>
         * Settings to configure your translation output. You can configure the following options:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Brevity: not supported.
         * </p>
         * </li>
         * <li>
         * <p>
         * Formality: sets the formality level of the output text.
         * </p>
         * </li>
         * <li>
         * <p>
         * Profanity: masks profane words and phrases in your translation output.
         * </p>
         * </li>
         * </ul>
         * This is a convenience method that creates an instance of the {@link TranslationSettings.Builder} avoiding the
         * need to create one manually via {@link TranslationSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TranslationSettings.Builder#build()} is called immediately and
         * its result is passed to {@link #settings(TranslationSettings)}.
         * 
         * @param settings
         *        a consumer that will call methods on {@link TranslationSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #settings(TranslationSettings)
         */
        default Builder settings(Consumer<TranslationSettings.Builder> settings) {
            return settings(TranslationSettings.builder().applyMutation(settings).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends TranslateRequest.BuilderImpl implements Builder {
        private Document document;

        private List<String> terminologyNames = DefaultSdkAutoConstructList.getInstance();

        private String sourceLanguageCode;

        private String targetLanguageCode;

        private TranslationSettings settings;

        private BuilderImpl() {
        }

        private BuilderImpl(TranslateDocumentRequest model) {
            super(model);
            document(model.document);
            terminologyNames(model.terminologyNames);
            sourceLanguageCode(model.sourceLanguageCode);
            targetLanguageCode(model.targetLanguageCode);
            settings(model.settings);
        }

        public final Document.Builder getDocument() {
            return document != null ? document.toBuilder() : null;
        }

        public final void setDocument(Document.BuilderImpl document) {
            this.document = document != null ? document.build() : null;
        }

        @Override
        public final Builder document(Document document) {
            this.document = document;
            return this;
        }

        public final Collection<String> getTerminologyNames() {
            if (terminologyNames instanceof SdkAutoConstructList) {
                return null;
            }
            return terminologyNames;
        }

        public final void setTerminologyNames(Collection<String> terminologyNames) {
            this.terminologyNames = ResourceNameListCopier.copy(terminologyNames);
        }

        @Override
        public final Builder terminologyNames(Collection<String> terminologyNames) {
            this.terminologyNames = ResourceNameListCopier.copy(terminologyNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder terminologyNames(String... terminologyNames) {
            terminologyNames(Arrays.asList(terminologyNames));
            return this;
        }

        public final String getSourceLanguageCode() {
            return sourceLanguageCode;
        }

        public final void setSourceLanguageCode(String sourceLanguageCode) {
            this.sourceLanguageCode = sourceLanguageCode;
        }

        @Override
        public final Builder sourceLanguageCode(String sourceLanguageCode) {
            this.sourceLanguageCode = sourceLanguageCode;
            return this;
        }

        public final String getTargetLanguageCode() {
            return targetLanguageCode;
        }

        public final void setTargetLanguageCode(String targetLanguageCode) {
            this.targetLanguageCode = targetLanguageCode;
        }

        @Override
        public final Builder targetLanguageCode(String targetLanguageCode) {
            this.targetLanguageCode = targetLanguageCode;
            return this;
        }

        public final TranslationSettings.Builder getSettings() {
            return settings != null ? settings.toBuilder() : null;
        }

        public final void setSettings(TranslationSettings.BuilderImpl settings) {
            this.settings = settings != null ? settings.build() : null;
        }

        @Override
        @Transient
        public final Builder settings(TranslationSettings settings) {
            this.settings = settings;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public TranslateDocumentRequest build() {
            return new TranslateDocumentRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
