/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.translate.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The properties of the custom terminology.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TerminologyProperties implements SdkPojo, Serializable,
        ToCopyableBuilder<TerminologyProperties.Builder, TerminologyProperties> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(TerminologyProperties::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(TerminologyProperties::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(TerminologyProperties::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> SOURCE_LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceLanguageCode").getter(getter(TerminologyProperties::sourceLanguageCode))
            .setter(setter(Builder::sourceLanguageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceLanguageCode").build())
            .build();

    private static final SdkField<List<String>> TARGET_LANGUAGE_CODES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("TargetLanguageCodes")
            .getter(getter(TerminologyProperties::targetLanguageCodes))
            .setter(setter(Builder::targetLanguageCodes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetLanguageCodes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<EncryptionKey> ENCRYPTION_KEY_FIELD = SdkField
            .<EncryptionKey> builder(MarshallingType.SDK_POJO).memberName("EncryptionKey")
            .getter(getter(TerminologyProperties::encryptionKey)).setter(setter(Builder::encryptionKey))
            .constructor(EncryptionKey::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionKey").build()).build();

    private static final SdkField<Integer> SIZE_BYTES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("SizeBytes").getter(getter(TerminologyProperties::sizeBytes)).setter(setter(Builder::sizeBytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SizeBytes").build()).build();

    private static final SdkField<Integer> TERM_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("TermCount").getter(getter(TerminologyProperties::termCount)).setter(setter(Builder::termCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TermCount").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAt").getter(getter(TerminologyProperties::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedAt").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdatedAt").getter(getter(TerminologyProperties::lastUpdatedAt))
            .setter(setter(Builder::lastUpdatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedAt").build()).build();

    private static final SdkField<String> DIRECTIONALITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Directionality").getter(getter(TerminologyProperties::directionalityAsString))
            .setter(setter(Builder::directionality))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Directionality").build()).build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Message")
            .getter(getter(TerminologyProperties::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Message").build()).build();

    private static final SdkField<Integer> SKIPPED_TERM_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("SkippedTermCount").getter(getter(TerminologyProperties::skippedTermCount))
            .setter(setter(Builder::skippedTermCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SkippedTermCount").build()).build();

    private static final SdkField<String> FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Format")
            .getter(getter(TerminologyProperties::formatAsString)).setter(setter(Builder::format))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Format").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, DESCRIPTION_FIELD,
            ARN_FIELD, SOURCE_LANGUAGE_CODE_FIELD, TARGET_LANGUAGE_CODES_FIELD, ENCRYPTION_KEY_FIELD, SIZE_BYTES_FIELD,
            TERM_COUNT_FIELD, CREATED_AT_FIELD, LAST_UPDATED_AT_FIELD, DIRECTIONALITY_FIELD, MESSAGE_FIELD,
            SKIPPED_TERM_COUNT_FIELD, FORMAT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("Name", NAME_FIELD);
                    put("Description", DESCRIPTION_FIELD);
                    put("Arn", ARN_FIELD);
                    put("SourceLanguageCode", SOURCE_LANGUAGE_CODE_FIELD);
                    put("TargetLanguageCodes", TARGET_LANGUAGE_CODES_FIELD);
                    put("EncryptionKey", ENCRYPTION_KEY_FIELD);
                    put("SizeBytes", SIZE_BYTES_FIELD);
                    put("TermCount", TERM_COUNT_FIELD);
                    put("CreatedAt", CREATED_AT_FIELD);
                    put("LastUpdatedAt", LAST_UPDATED_AT_FIELD);
                    put("Directionality", DIRECTIONALITY_FIELD);
                    put("Message", MESSAGE_FIELD);
                    put("SkippedTermCount", SKIPPED_TERM_COUNT_FIELD);
                    put("Format", FORMAT_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String description;

    private final String arn;

    private final String sourceLanguageCode;

    private final List<String> targetLanguageCodes;

    private final EncryptionKey encryptionKey;

    private final Integer sizeBytes;

    private final Integer termCount;

    private final Instant createdAt;

    private final Instant lastUpdatedAt;

    private final String directionality;

    private final String message;

    private final Integer skippedTermCount;

    private final String format;

    private TerminologyProperties(BuilderImpl builder) {
        this.name = builder.name;
        this.description = builder.description;
        this.arn = builder.arn;
        this.sourceLanguageCode = builder.sourceLanguageCode;
        this.targetLanguageCodes = builder.targetLanguageCodes;
        this.encryptionKey = builder.encryptionKey;
        this.sizeBytes = builder.sizeBytes;
        this.termCount = builder.termCount;
        this.createdAt = builder.createdAt;
        this.lastUpdatedAt = builder.lastUpdatedAt;
        this.directionality = builder.directionality;
        this.message = builder.message;
        this.skippedTermCount = builder.skippedTermCount;
        this.format = builder.format;
    }

    /**
     * <p>
     * The name of the custom terminology.
     * </p>
     * 
     * @return The name of the custom terminology.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The description of the custom terminology properties.
     * </p>
     * 
     * @return The description of the custom terminology properties.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the custom terminology.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the custom terminology.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The language code for the source text of the translation request for which the custom terminology is being used.
     * </p>
     * 
     * @return The language code for the source text of the translation request for which the custom terminology is
     *         being used.
     */
    public final String sourceLanguageCode() {
        return sourceLanguageCode;
    }

    /**
     * For responses, this returns true if the service returned a value for the TargetLanguageCodes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTargetLanguageCodes() {
        return targetLanguageCodes != null && !(targetLanguageCodes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The language codes for the target languages available with the custom terminology resource. All possible target
     * languages are returned in array.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTargetLanguageCodes} method.
     * </p>
     * 
     * @return The language codes for the target languages available with the custom terminology resource. All possible
     *         target languages are returned in array.
     */
    public final List<String> targetLanguageCodes() {
        return targetLanguageCodes;
    }

    /**
     * <p>
     * The encryption key for the custom terminology.
     * </p>
     * 
     * @return The encryption key for the custom terminology.
     */
    public final EncryptionKey encryptionKey() {
        return encryptionKey;
    }

    /**
     * <p>
     * The size of the file used when importing a custom terminology.
     * </p>
     * 
     * @return The size of the file used when importing a custom terminology.
     */
    public final Integer sizeBytes() {
        return sizeBytes;
    }

    /**
     * <p>
     * The number of terms included in the custom terminology.
     * </p>
     * 
     * @return The number of terms included in the custom terminology.
     */
    public final Integer termCount() {
        return termCount;
    }

    /**
     * <p>
     * The time at which the custom terminology was created, based on the timestamp.
     * </p>
     * 
     * @return The time at which the custom terminology was created, based on the timestamp.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The time at which the custom terminology was last update, based on the timestamp.
     * </p>
     * 
     * @return The time at which the custom terminology was last update, based on the timestamp.
     */
    public final Instant lastUpdatedAt() {
        return lastUpdatedAt;
    }

    /**
     * <p>
     * The directionality of your terminology resource indicates whether it has one source language (uni-directional) or
     * multiple (multi-directional).
     * </p>
     * <dl>
     * <dt>UNI</dt>
     * <dd>
     * <p>
     * The terminology resource has one source language (the first column in a CSV file), and all of its other languages
     * are target languages.
     * </p>
     * </dd>
     * <dt>MULTI</dt>
     * <dd>
     * <p>
     * Any language in the terminology resource can be the source language.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #directionality}
     * will return {@link Directionality#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #directionalityAsString}.
     * </p>
     * 
     * @return The directionality of your terminology resource indicates whether it has one source language
     *         (uni-directional) or multiple (multi-directional). </p>
     *         <dl>
     *         <dt>UNI</dt>
     *         <dd>
     *         <p>
     *         The terminology resource has one source language (the first column in a CSV file), and all of its other
     *         languages are target languages.
     *         </p>
     *         </dd>
     *         <dt>MULTI</dt>
     *         <dd>
     *         <p>
     *         Any language in the terminology resource can be the source language.
     *         </p>
     *         </dd>
     * @see Directionality
     */
    public final Directionality directionality() {
        return Directionality.fromValue(directionality);
    }

    /**
     * <p>
     * The directionality of your terminology resource indicates whether it has one source language (uni-directional) or
     * multiple (multi-directional).
     * </p>
     * <dl>
     * <dt>UNI</dt>
     * <dd>
     * <p>
     * The terminology resource has one source language (the first column in a CSV file), and all of its other languages
     * are target languages.
     * </p>
     * </dd>
     * <dt>MULTI</dt>
     * <dd>
     * <p>
     * Any language in the terminology resource can be the source language.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #directionality}
     * will return {@link Directionality#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #directionalityAsString}.
     * </p>
     * 
     * @return The directionality of your terminology resource indicates whether it has one source language
     *         (uni-directional) or multiple (multi-directional). </p>
     *         <dl>
     *         <dt>UNI</dt>
     *         <dd>
     *         <p>
     *         The terminology resource has one source language (the first column in a CSV file), and all of its other
     *         languages are target languages.
     *         </p>
     *         </dd>
     *         <dt>MULTI</dt>
     *         <dd>
     *         <p>
     *         Any language in the terminology resource can be the source language.
     *         </p>
     *         </dd>
     * @see Directionality
     */
    public final String directionalityAsString() {
        return directionality;
    }

    /**
     * <p>
     * Additional information from Amazon Translate about the terminology resource.
     * </p>
     * 
     * @return Additional information from Amazon Translate about the terminology resource.
     */
    public final String message() {
        return message;
    }

    /**
     * <p>
     * The number of terms in the input file that Amazon Translate skipped when you created or updated the terminology
     * resource.
     * </p>
     * 
     * @return The number of terms in the input file that Amazon Translate skipped when you created or updated the
     *         terminology resource.
     */
    public final Integer skippedTermCount() {
        return skippedTermCount;
    }

    /**
     * <p>
     * The format of the custom terminology input file.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link TerminologyDataFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #formatAsString}.
     * </p>
     * 
     * @return The format of the custom terminology input file.
     * @see TerminologyDataFormat
     */
    public final TerminologyDataFormat format() {
        return TerminologyDataFormat.fromValue(format);
    }

    /**
     * <p>
     * The format of the custom terminology input file.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link TerminologyDataFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #formatAsString}.
     * </p>
     * 
     * @return The format of the custom terminology input file.
     * @see TerminologyDataFormat
     */
    public final String formatAsString() {
        return format;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(sourceLanguageCode());
        hashCode = 31 * hashCode + Objects.hashCode(hasTargetLanguageCodes() ? targetLanguageCodes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(encryptionKey());
        hashCode = 31 * hashCode + Objects.hashCode(sizeBytes());
        hashCode = 31 * hashCode + Objects.hashCode(termCount());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(directionalityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(skippedTermCount());
        hashCode = 31 * hashCode + Objects.hashCode(formatAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TerminologyProperties)) {
            return false;
        }
        TerminologyProperties other = (TerminologyProperties) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && Objects.equals(arn(), other.arn()) && Objects.equals(sourceLanguageCode(), other.sourceLanguageCode())
                && hasTargetLanguageCodes() == other.hasTargetLanguageCodes()
                && Objects.equals(targetLanguageCodes(), other.targetLanguageCodes())
                && Objects.equals(encryptionKey(), other.encryptionKey()) && Objects.equals(sizeBytes(), other.sizeBytes())
                && Objects.equals(termCount(), other.termCount()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(lastUpdatedAt(), other.lastUpdatedAt())
                && Objects.equals(directionalityAsString(), other.directionalityAsString())
                && Objects.equals(message(), other.message()) && Objects.equals(skippedTermCount(), other.skippedTermCount())
                && Objects.equals(formatAsString(), other.formatAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TerminologyProperties").add("Name", name()).add("Description", description()).add("Arn", arn())
                .add("SourceLanguageCode", sourceLanguageCode())
                .add("TargetLanguageCodes", hasTargetLanguageCodes() ? targetLanguageCodes() : null)
                .add("EncryptionKey", encryptionKey()).add("SizeBytes", sizeBytes()).add("TermCount", termCount())
                .add("CreatedAt", createdAt()).add("LastUpdatedAt", lastUpdatedAt())
                .add("Directionality", directionalityAsString()).add("Message", message())
                .add("SkippedTermCount", skippedTermCount()).add("Format", formatAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "SourceLanguageCode":
            return Optional.ofNullable(clazz.cast(sourceLanguageCode()));
        case "TargetLanguageCodes":
            return Optional.ofNullable(clazz.cast(targetLanguageCodes()));
        case "EncryptionKey":
            return Optional.ofNullable(clazz.cast(encryptionKey()));
        case "SizeBytes":
            return Optional.ofNullable(clazz.cast(sizeBytes()));
        case "TermCount":
            return Optional.ofNullable(clazz.cast(termCount()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "LastUpdatedAt":
            return Optional.ofNullable(clazz.cast(lastUpdatedAt()));
        case "Directionality":
            return Optional.ofNullable(clazz.cast(directionalityAsString()));
        case "Message":
            return Optional.ofNullable(clazz.cast(message()));
        case "SkippedTermCount":
            return Optional.ofNullable(clazz.cast(skippedTermCount()));
        case "Format":
            return Optional.ofNullable(clazz.cast(formatAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<TerminologyProperties, T> g) {
        return obj -> g.apply((TerminologyProperties) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TerminologyProperties> {
        /**
         * <p>
         * The name of the custom terminology.
         * </p>
         * 
         * @param name
         *        The name of the custom terminology.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The description of the custom terminology properties.
         * </p>
         * 
         * @param description
         *        The description of the custom terminology properties.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the custom terminology.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the custom terminology.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The language code for the source text of the translation request for which the custom terminology is being
         * used.
         * </p>
         * 
         * @param sourceLanguageCode
         *        The language code for the source text of the translation request for which the custom terminology is
         *        being used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceLanguageCode(String sourceLanguageCode);

        /**
         * <p>
         * The language codes for the target languages available with the custom terminology resource. All possible
         * target languages are returned in array.
         * </p>
         * 
         * @param targetLanguageCodes
         *        The language codes for the target languages available with the custom terminology resource. All
         *        possible target languages are returned in array.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetLanguageCodes(Collection<String> targetLanguageCodes);

        /**
         * <p>
         * The language codes for the target languages available with the custom terminology resource. All possible
         * target languages are returned in array.
         * </p>
         * 
         * @param targetLanguageCodes
         *        The language codes for the target languages available with the custom terminology resource. All
         *        possible target languages are returned in array.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetLanguageCodes(String... targetLanguageCodes);

        /**
         * <p>
         * The encryption key for the custom terminology.
         * </p>
         * 
         * @param encryptionKey
         *        The encryption key for the custom terminology.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionKey(EncryptionKey encryptionKey);

        /**
         * <p>
         * The encryption key for the custom terminology.
         * </p>
         * This is a convenience method that creates an instance of the {@link EncryptionKey.Builder} avoiding the need
         * to create one manually via {@link EncryptionKey#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EncryptionKey.Builder#build()} is called immediately and its
         * result is passed to {@link #encryptionKey(EncryptionKey)}.
         * 
         * @param encryptionKey
         *        a consumer that will call methods on {@link EncryptionKey.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionKey(EncryptionKey)
         */
        default Builder encryptionKey(Consumer<EncryptionKey.Builder> encryptionKey) {
            return encryptionKey(EncryptionKey.builder().applyMutation(encryptionKey).build());
        }

        /**
         * <p>
         * The size of the file used when importing a custom terminology.
         * </p>
         * 
         * @param sizeBytes
         *        The size of the file used when importing a custom terminology.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sizeBytes(Integer sizeBytes);

        /**
         * <p>
         * The number of terms included in the custom terminology.
         * </p>
         * 
         * @param termCount
         *        The number of terms included in the custom terminology.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder termCount(Integer termCount);

        /**
         * <p>
         * The time at which the custom terminology was created, based on the timestamp.
         * </p>
         * 
         * @param createdAt
         *        The time at which the custom terminology was created, based on the timestamp.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The time at which the custom terminology was last update, based on the timestamp.
         * </p>
         * 
         * @param lastUpdatedAt
         *        The time at which the custom terminology was last update, based on the timestamp.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedAt(Instant lastUpdatedAt);

        /**
         * <p>
         * The directionality of your terminology resource indicates whether it has one source language
         * (uni-directional) or multiple (multi-directional).
         * </p>
         * <dl>
         * <dt>UNI</dt>
         * <dd>
         * <p>
         * The terminology resource has one source language (the first column in a CSV file), and all of its other
         * languages are target languages.
         * </p>
         * </dd>
         * <dt>MULTI</dt>
         * <dd>
         * <p>
         * Any language in the terminology resource can be the source language.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param directionality
         *        The directionality of your terminology resource indicates whether it has one source language
         *        (uni-directional) or multiple (multi-directional). </p>
         *        <dl>
         *        <dt>UNI</dt>
         *        <dd>
         *        <p>
         *        The terminology resource has one source language (the first column in a CSV file), and all of its
         *        other languages are target languages.
         *        </p>
         *        </dd>
         *        <dt>MULTI</dt>
         *        <dd>
         *        <p>
         *        Any language in the terminology resource can be the source language.
         *        </p>
         *        </dd>
         * @see Directionality
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Directionality
         */
        Builder directionality(String directionality);

        /**
         * <p>
         * The directionality of your terminology resource indicates whether it has one source language
         * (uni-directional) or multiple (multi-directional).
         * </p>
         * <dl>
         * <dt>UNI</dt>
         * <dd>
         * <p>
         * The terminology resource has one source language (the first column in a CSV file), and all of its other
         * languages are target languages.
         * </p>
         * </dd>
         * <dt>MULTI</dt>
         * <dd>
         * <p>
         * Any language in the terminology resource can be the source language.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param directionality
         *        The directionality of your terminology resource indicates whether it has one source language
         *        (uni-directional) or multiple (multi-directional). </p>
         *        <dl>
         *        <dt>UNI</dt>
         *        <dd>
         *        <p>
         *        The terminology resource has one source language (the first column in a CSV file), and all of its
         *        other languages are target languages.
         *        </p>
         *        </dd>
         *        <dt>MULTI</dt>
         *        <dd>
         *        <p>
         *        Any language in the terminology resource can be the source language.
         *        </p>
         *        </dd>
         * @see Directionality
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Directionality
         */
        Builder directionality(Directionality directionality);

        /**
         * <p>
         * Additional information from Amazon Translate about the terminology resource.
         * </p>
         * 
         * @param message
         *        Additional information from Amazon Translate about the terminology resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);

        /**
         * <p>
         * The number of terms in the input file that Amazon Translate skipped when you created or updated the
         * terminology resource.
         * </p>
         * 
         * @param skippedTermCount
         *        The number of terms in the input file that Amazon Translate skipped when you created or updated the
         *        terminology resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder skippedTermCount(Integer skippedTermCount);

        /**
         * <p>
         * The format of the custom terminology input file.
         * </p>
         * 
         * @param format
         *        The format of the custom terminology input file.
         * @see TerminologyDataFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TerminologyDataFormat
         */
        Builder format(String format);

        /**
         * <p>
         * The format of the custom terminology input file.
         * </p>
         * 
         * @param format
         *        The format of the custom terminology input file.
         * @see TerminologyDataFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TerminologyDataFormat
         */
        Builder format(TerminologyDataFormat format);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String description;

        private String arn;

        private String sourceLanguageCode;

        private List<String> targetLanguageCodes = DefaultSdkAutoConstructList.getInstance();

        private EncryptionKey encryptionKey;

        private Integer sizeBytes;

        private Integer termCount;

        private Instant createdAt;

        private Instant lastUpdatedAt;

        private String directionality;

        private String message;

        private Integer skippedTermCount;

        private String format;

        private BuilderImpl() {
        }

        private BuilderImpl(TerminologyProperties model) {
            name(model.name);
            description(model.description);
            arn(model.arn);
            sourceLanguageCode(model.sourceLanguageCode);
            targetLanguageCodes(model.targetLanguageCodes);
            encryptionKey(model.encryptionKey);
            sizeBytes(model.sizeBytes);
            termCount(model.termCount);
            createdAt(model.createdAt);
            lastUpdatedAt(model.lastUpdatedAt);
            directionality(model.directionality);
            message(model.message);
            skippedTermCount(model.skippedTermCount);
            format(model.format);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getSourceLanguageCode() {
            return sourceLanguageCode;
        }

        public final void setSourceLanguageCode(String sourceLanguageCode) {
            this.sourceLanguageCode = sourceLanguageCode;
        }

        @Override
        public final Builder sourceLanguageCode(String sourceLanguageCode) {
            this.sourceLanguageCode = sourceLanguageCode;
            return this;
        }

        public final Collection<String> getTargetLanguageCodes() {
            if (targetLanguageCodes instanceof SdkAutoConstructList) {
                return null;
            }
            return targetLanguageCodes;
        }

        public final void setTargetLanguageCodes(Collection<String> targetLanguageCodes) {
            this.targetLanguageCodes = LanguageCodeStringListCopier.copy(targetLanguageCodes);
        }

        @Override
        public final Builder targetLanguageCodes(Collection<String> targetLanguageCodes) {
            this.targetLanguageCodes = LanguageCodeStringListCopier.copy(targetLanguageCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetLanguageCodes(String... targetLanguageCodes) {
            targetLanguageCodes(Arrays.asList(targetLanguageCodes));
            return this;
        }

        public final EncryptionKey.Builder getEncryptionKey() {
            return encryptionKey != null ? encryptionKey.toBuilder() : null;
        }

        public final void setEncryptionKey(EncryptionKey.BuilderImpl encryptionKey) {
            this.encryptionKey = encryptionKey != null ? encryptionKey.build() : null;
        }

        @Override
        public final Builder encryptionKey(EncryptionKey encryptionKey) {
            this.encryptionKey = encryptionKey;
            return this;
        }

        public final Integer getSizeBytes() {
            return sizeBytes;
        }

        public final void setSizeBytes(Integer sizeBytes) {
            this.sizeBytes = sizeBytes;
        }

        @Override
        public final Builder sizeBytes(Integer sizeBytes) {
            this.sizeBytes = sizeBytes;
            return this;
        }

        public final Integer getTermCount() {
            return termCount;
        }

        public final void setTermCount(Integer termCount) {
            this.termCount = termCount;
        }

        @Override
        public final Builder termCount(Integer termCount) {
            this.termCount = termCount;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getLastUpdatedAt() {
            return lastUpdatedAt;
        }

        public final void setLastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
        }

        @Override
        public final Builder lastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
            return this;
        }

        public final String getDirectionality() {
            return directionality;
        }

        public final void setDirectionality(String directionality) {
            this.directionality = directionality;
        }

        @Override
        public final Builder directionality(String directionality) {
            this.directionality = directionality;
            return this;
        }

        @Override
        public final Builder directionality(Directionality directionality) {
            this.directionality(directionality == null ? null : directionality.toString());
            return this;
        }

        public final String getMessage() {
            return message;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final Integer getSkippedTermCount() {
            return skippedTermCount;
        }

        public final void setSkippedTermCount(Integer skippedTermCount) {
            this.skippedTermCount = skippedTermCount;
        }

        @Override
        public final Builder skippedTermCount(Integer skippedTermCount) {
            this.skippedTermCount = skippedTermCount;
            return this;
        }

        public final String getFormat() {
            return format;
        }

        public final void setFormat(String format) {
            this.format = format;
        }

        @Override
        public final Builder format(String format) {
            this.format = format;
            return this;
        }

        @Override
        public final Builder format(TerminologyDataFormat format) {
            this.format(format == null ? null : format.toString());
            return this;
        }

        @Override
        public TerminologyProperties build() {
            return new TerminologyProperties(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
