/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.translate.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Settings to configure your translation output. You can configure the following options:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Brevity: reduces the length of the translation output for most translations. Available for <code>TranslateText</code>
 * only.
 * </p>
 * </li>
 * <li>
 * <p>
 * Formality: sets the formality level of the translation output.
 * </p>
 * </li>
 * <li>
 * <p>
 * Profanity: masks profane words and phrases in the translation output.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TranslationSettings implements SdkPojo, Serializable,
        ToCopyableBuilder<TranslationSettings.Builder, TranslationSettings> {
    private static final SdkField<String> FORMALITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Formality").getter(getter(TranslationSettings::formalityAsString)).setter(setter(Builder::formality))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Formality").build()).build();

    private static final SdkField<String> PROFANITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Profanity").getter(getter(TranslationSettings::profanityAsString)).setter(setter(Builder::profanity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Profanity").build()).build();

    private static final SdkField<String> BREVITY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Brevity")
            .getter(getter(TranslationSettings::brevityAsString)).setter(setter(Builder::brevity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Brevity").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FORMALITY_FIELD,
            PROFANITY_FIELD, BREVITY_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("Formality", FORMALITY_FIELD);
                    put("Profanity", PROFANITY_FIELD);
                    put("Brevity", BREVITY_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String formality;

    private final String profanity;

    private final String brevity;

    private TranslationSettings(BuilderImpl builder) {
        this.formality = builder.formality;
        this.profanity = builder.profanity;
        this.brevity = builder.brevity;
    }

    /**
     * <p>
     * You can specify the desired level of formality for translations to supported target languages. The formality
     * setting controls the level of formal language usage (also known as <a
     * href="https://en.wikipedia.org/wiki/Register_(sociolinguistics)">register</a>) in the translation output. You can
     * set the value to informal or formal. If you don't specify a value for formality, or if the target language
     * doesn't support formality, the translation will ignore the formality setting.
     * </p>
     * <p>
     * If you specify multiple target languages for the job, translate ignores the formality setting for any unsupported
     * target language.
     * </p>
     * <p>
     * For a list of target languages that support formality, see <a href=
     * "https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-formality.html#customizing-translations-formality-languages"
     * >Supported languages</a> in the Amazon Translate Developer Guide.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #formality} will
     * return {@link Formality#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formalityAsString}.
     * </p>
     * 
     * @return You can specify the desired level of formality for translations to supported target languages. The
     *         formality setting controls the level of formal language usage (also known as <a
     *         href="https://en.wikipedia.org/wiki/Register_(sociolinguistics)">register</a>) in the translation output.
     *         You can set the value to informal or formal. If you don't specify a value for formality, or if the target
     *         language doesn't support formality, the translation will ignore the formality setting.</p>
     *         <p>
     *         If you specify multiple target languages for the job, translate ignores the formality setting for any
     *         unsupported target language.
     *         </p>
     *         <p>
     *         For a list of target languages that support formality, see <a href=
     *         "https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-formality.html#customizing-translations-formality-languages"
     *         >Supported languages</a> in the Amazon Translate Developer Guide.
     * @see Formality
     */
    public final Formality formality() {
        return Formality.fromValue(formality);
    }

    /**
     * <p>
     * You can specify the desired level of formality for translations to supported target languages. The formality
     * setting controls the level of formal language usage (also known as <a
     * href="https://en.wikipedia.org/wiki/Register_(sociolinguistics)">register</a>) in the translation output. You can
     * set the value to informal or formal. If you don't specify a value for formality, or if the target language
     * doesn't support formality, the translation will ignore the formality setting.
     * </p>
     * <p>
     * If you specify multiple target languages for the job, translate ignores the formality setting for any unsupported
     * target language.
     * </p>
     * <p>
     * For a list of target languages that support formality, see <a href=
     * "https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-formality.html#customizing-translations-formality-languages"
     * >Supported languages</a> in the Amazon Translate Developer Guide.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #formality} will
     * return {@link Formality#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formalityAsString}.
     * </p>
     * 
     * @return You can specify the desired level of formality for translations to supported target languages. The
     *         formality setting controls the level of formal language usage (also known as <a
     *         href="https://en.wikipedia.org/wiki/Register_(sociolinguistics)">register</a>) in the translation output.
     *         You can set the value to informal or formal. If you don't specify a value for formality, or if the target
     *         language doesn't support formality, the translation will ignore the formality setting.</p>
     *         <p>
     *         If you specify multiple target languages for the job, translate ignores the formality setting for any
     *         unsupported target language.
     *         </p>
     *         <p>
     *         For a list of target languages that support formality, see <a href=
     *         "https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-formality.html#customizing-translations-formality-languages"
     *         >Supported languages</a> in the Amazon Translate Developer Guide.
     * @see Formality
     */
    public final String formalityAsString() {
        return formality;
    }

    /**
     * <p>
     * You can enable the profanity setting if you want to mask profane words and phrases in your translation output.
     * </p>
     * <p>
     * To mask profane words and phrases, Amazon Translate replaces them with the grawlix string “?$#@$“. This
     * 5-character sequence is used for each profane word or phrase, regardless of the length or number of words.
     * </p>
     * <p>
     * Amazon Translate doesn't detect profanity in all of its supported languages. For languages that don't support
     * profanity detection, see <a href=
     * "https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-profanity.html#customizing-translations-profanity-languages"
     * >Unsupported languages</a> in the Amazon Translate Developer Guide.
     * </p>
     * <p>
     * If you specify multiple target languages for the job, all the target languages must support profanity masking. If
     * any of the target languages don't support profanity masking, the translation job won't mask profanity for any
     * target language.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #profanity} will
     * return {@link Profanity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #profanityAsString}.
     * </p>
     * 
     * @return You can enable the profanity setting if you want to mask profane words and phrases in your translation
     *         output.</p>
     *         <p>
     *         To mask profane words and phrases, Amazon Translate replaces them with the grawlix string “?$#@$“. This
     *         5-character sequence is used for each profane word or phrase, regardless of the length or number of
     *         words.
     *         </p>
     *         <p>
     *         Amazon Translate doesn't detect profanity in all of its supported languages. For languages that don't
     *         support profanity detection, see <a href=
     *         "https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-profanity.html#customizing-translations-profanity-languages"
     *         >Unsupported languages</a> in the Amazon Translate Developer Guide.
     *         </p>
     *         <p>
     *         If you specify multiple target languages for the job, all the target languages must support profanity
     *         masking. If any of the target languages don't support profanity masking, the translation job won't mask
     *         profanity for any target language.
     * @see Profanity
     */
    public final Profanity profanity() {
        return Profanity.fromValue(profanity);
    }

    /**
     * <p>
     * You can enable the profanity setting if you want to mask profane words and phrases in your translation output.
     * </p>
     * <p>
     * To mask profane words and phrases, Amazon Translate replaces them with the grawlix string “?$#@$“. This
     * 5-character sequence is used for each profane word or phrase, regardless of the length or number of words.
     * </p>
     * <p>
     * Amazon Translate doesn't detect profanity in all of its supported languages. For languages that don't support
     * profanity detection, see <a href=
     * "https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-profanity.html#customizing-translations-profanity-languages"
     * >Unsupported languages</a> in the Amazon Translate Developer Guide.
     * </p>
     * <p>
     * If you specify multiple target languages for the job, all the target languages must support profanity masking. If
     * any of the target languages don't support profanity masking, the translation job won't mask profanity for any
     * target language.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #profanity} will
     * return {@link Profanity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #profanityAsString}.
     * </p>
     * 
     * @return You can enable the profanity setting if you want to mask profane words and phrases in your translation
     *         output.</p>
     *         <p>
     *         To mask profane words and phrases, Amazon Translate replaces them with the grawlix string “?$#@$“. This
     *         5-character sequence is used for each profane word or phrase, regardless of the length or number of
     *         words.
     *         </p>
     *         <p>
     *         Amazon Translate doesn't detect profanity in all of its supported languages. For languages that don't
     *         support profanity detection, see <a href=
     *         "https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-profanity.html#customizing-translations-profanity-languages"
     *         >Unsupported languages</a> in the Amazon Translate Developer Guide.
     *         </p>
     *         <p>
     *         If you specify multiple target languages for the job, all the target languages must support profanity
     *         masking. If any of the target languages don't support profanity masking, the translation job won't mask
     *         profanity for any target language.
     * @see Profanity
     */
    public final String profanityAsString() {
        return profanity;
    }

    /**
     * <p>
     * When you turn on brevity, Amazon Translate reduces the length of the translation output for most translations
     * (when compared with the same translation with brevity turned off). By default, brevity is turned off.
     * </p>
     * <p>
     * If you turn on brevity for a translation request with an unsupported language pair, the translation proceeds with
     * the brevity setting turned off.
     * </p>
     * <p>
     * For the language pairs that brevity supports, see <a
     * href="https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-brevity">Using brevity</a> in the
     * Amazon Translate Developer Guide.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #brevity} will
     * return {@link Brevity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #brevityAsString}.
     * </p>
     * 
     * @return When you turn on brevity, Amazon Translate reduces the length of the translation output for most
     *         translations (when compared with the same translation with brevity turned off). By default, brevity is
     *         turned off.</p>
     *         <p>
     *         If you turn on brevity for a translation request with an unsupported language pair, the translation
     *         proceeds with the brevity setting turned off.
     *         </p>
     *         <p>
     *         For the language pairs that brevity supports, see <a
     *         href="https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-brevity">Using brevity</a>
     *         in the Amazon Translate Developer Guide.
     * @see Brevity
     */
    public final Brevity brevity() {
        return Brevity.fromValue(brevity);
    }

    /**
     * <p>
     * When you turn on brevity, Amazon Translate reduces the length of the translation output for most translations
     * (when compared with the same translation with brevity turned off). By default, brevity is turned off.
     * </p>
     * <p>
     * If you turn on brevity for a translation request with an unsupported language pair, the translation proceeds with
     * the brevity setting turned off.
     * </p>
     * <p>
     * For the language pairs that brevity supports, see <a
     * href="https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-brevity">Using brevity</a> in the
     * Amazon Translate Developer Guide.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #brevity} will
     * return {@link Brevity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #brevityAsString}.
     * </p>
     * 
     * @return When you turn on brevity, Amazon Translate reduces the length of the translation output for most
     *         translations (when compared with the same translation with brevity turned off). By default, brevity is
     *         turned off.</p>
     *         <p>
     *         If you turn on brevity for a translation request with an unsupported language pair, the translation
     *         proceeds with the brevity setting turned off.
     *         </p>
     *         <p>
     *         For the language pairs that brevity supports, see <a
     *         href="https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-brevity">Using brevity</a>
     *         in the Amazon Translate Developer Guide.
     * @see Brevity
     */
    public final String brevityAsString() {
        return brevity;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(formalityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(profanityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(brevityAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TranslationSettings)) {
            return false;
        }
        TranslationSettings other = (TranslationSettings) obj;
        return Objects.equals(formalityAsString(), other.formalityAsString())
                && Objects.equals(profanityAsString(), other.profanityAsString())
                && Objects.equals(brevityAsString(), other.brevityAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TranslationSettings").add("Formality", formalityAsString())
                .add("Profanity", profanityAsString()).add("Brevity", brevityAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Formality":
            return Optional.ofNullable(clazz.cast(formalityAsString()));
        case "Profanity":
            return Optional.ofNullable(clazz.cast(profanityAsString()));
        case "Brevity":
            return Optional.ofNullable(clazz.cast(brevityAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<TranslationSettings, T> g) {
        return obj -> g.apply((TranslationSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TranslationSettings> {
        /**
         * <p>
         * You can specify the desired level of formality for translations to supported target languages. The formality
         * setting controls the level of formal language usage (also known as <a
         * href="https://en.wikipedia.org/wiki/Register_(sociolinguistics)">register</a>) in the translation output. You
         * can set the value to informal or formal. If you don't specify a value for formality, or if the target
         * language doesn't support formality, the translation will ignore the formality setting.
         * </p>
         * <p>
         * If you specify multiple target languages for the job, translate ignores the formality setting for any
         * unsupported target language.
         * </p>
         * <p>
         * For a list of target languages that support formality, see <a href=
         * "https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-formality.html#customizing-translations-formality-languages"
         * >Supported languages</a> in the Amazon Translate Developer Guide.
         * </p>
         * 
         * @param formality
         *        You can specify the desired level of formality for translations to supported target languages. The
         *        formality setting controls the level of formal language usage (also known as <a
         *        href="https://en.wikipedia.org/wiki/Register_(sociolinguistics)">register</a>) in the translation
         *        output. You can set the value to informal or formal. If you don't specify a value for formality, or if
         *        the target language doesn't support formality, the translation will ignore the formality setting.</p>
         *        <p>
         *        If you specify multiple target languages for the job, translate ignores the formality setting for any
         *        unsupported target language.
         *        </p>
         *        <p>
         *        For a list of target languages that support formality, see <a href=
         *        "https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-formality.html#customizing-translations-formality-languages"
         *        >Supported languages</a> in the Amazon Translate Developer Guide.
         * @see Formality
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Formality
         */
        Builder formality(String formality);

        /**
         * <p>
         * You can specify the desired level of formality for translations to supported target languages. The formality
         * setting controls the level of formal language usage (also known as <a
         * href="https://en.wikipedia.org/wiki/Register_(sociolinguistics)">register</a>) in the translation output. You
         * can set the value to informal or formal. If you don't specify a value for formality, or if the target
         * language doesn't support formality, the translation will ignore the formality setting.
         * </p>
         * <p>
         * If you specify multiple target languages for the job, translate ignores the formality setting for any
         * unsupported target language.
         * </p>
         * <p>
         * For a list of target languages that support formality, see <a href=
         * "https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-formality.html#customizing-translations-formality-languages"
         * >Supported languages</a> in the Amazon Translate Developer Guide.
         * </p>
         * 
         * @param formality
         *        You can specify the desired level of formality for translations to supported target languages. The
         *        formality setting controls the level of formal language usage (also known as <a
         *        href="https://en.wikipedia.org/wiki/Register_(sociolinguistics)">register</a>) in the translation
         *        output. You can set the value to informal or formal. If you don't specify a value for formality, or if
         *        the target language doesn't support formality, the translation will ignore the formality setting.</p>
         *        <p>
         *        If you specify multiple target languages for the job, translate ignores the formality setting for any
         *        unsupported target language.
         *        </p>
         *        <p>
         *        For a list of target languages that support formality, see <a href=
         *        "https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-formality.html#customizing-translations-formality-languages"
         *        >Supported languages</a> in the Amazon Translate Developer Guide.
         * @see Formality
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Formality
         */
        Builder formality(Formality formality);

        /**
         * <p>
         * You can enable the profanity setting if you want to mask profane words and phrases in your translation
         * output.
         * </p>
         * <p>
         * To mask profane words and phrases, Amazon Translate replaces them with the grawlix string “?$#@$“. This
         * 5-character sequence is used for each profane word or phrase, regardless of the length or number of words.
         * </p>
         * <p>
         * Amazon Translate doesn't detect profanity in all of its supported languages. For languages that don't support
         * profanity detection, see <a href=
         * "https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-profanity.html#customizing-translations-profanity-languages"
         * >Unsupported languages</a> in the Amazon Translate Developer Guide.
         * </p>
         * <p>
         * If you specify multiple target languages for the job, all the target languages must support profanity
         * masking. If any of the target languages don't support profanity masking, the translation job won't mask
         * profanity for any target language.
         * </p>
         * 
         * @param profanity
         *        You can enable the profanity setting if you want to mask profane words and phrases in your translation
         *        output.</p>
         *        <p>
         *        To mask profane words and phrases, Amazon Translate replaces them with the grawlix string “?$#@$“.
         *        This 5-character sequence is used for each profane word or phrase, regardless of the length or number
         *        of words.
         *        </p>
         *        <p>
         *        Amazon Translate doesn't detect profanity in all of its supported languages. For languages that don't
         *        support profanity detection, see <a href=
         *        "https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-profanity.html#customizing-translations-profanity-languages"
         *        >Unsupported languages</a> in the Amazon Translate Developer Guide.
         *        </p>
         *        <p>
         *        If you specify multiple target languages for the job, all the target languages must support profanity
         *        masking. If any of the target languages don't support profanity masking, the translation job won't
         *        mask profanity for any target language.
         * @see Profanity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Profanity
         */
        Builder profanity(String profanity);

        /**
         * <p>
         * You can enable the profanity setting if you want to mask profane words and phrases in your translation
         * output.
         * </p>
         * <p>
         * To mask profane words and phrases, Amazon Translate replaces them with the grawlix string “?$#@$“. This
         * 5-character sequence is used for each profane word or phrase, regardless of the length or number of words.
         * </p>
         * <p>
         * Amazon Translate doesn't detect profanity in all of its supported languages. For languages that don't support
         * profanity detection, see <a href=
         * "https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-profanity.html#customizing-translations-profanity-languages"
         * >Unsupported languages</a> in the Amazon Translate Developer Guide.
         * </p>
         * <p>
         * If you specify multiple target languages for the job, all the target languages must support profanity
         * masking. If any of the target languages don't support profanity masking, the translation job won't mask
         * profanity for any target language.
         * </p>
         * 
         * @param profanity
         *        You can enable the profanity setting if you want to mask profane words and phrases in your translation
         *        output.</p>
         *        <p>
         *        To mask profane words and phrases, Amazon Translate replaces them with the grawlix string “?$#@$“.
         *        This 5-character sequence is used for each profane word or phrase, regardless of the length or number
         *        of words.
         *        </p>
         *        <p>
         *        Amazon Translate doesn't detect profanity in all of its supported languages. For languages that don't
         *        support profanity detection, see <a href=
         *        "https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-profanity.html#customizing-translations-profanity-languages"
         *        >Unsupported languages</a> in the Amazon Translate Developer Guide.
         *        </p>
         *        <p>
         *        If you specify multiple target languages for the job, all the target languages must support profanity
         *        masking. If any of the target languages don't support profanity masking, the translation job won't
         *        mask profanity for any target language.
         * @see Profanity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Profanity
         */
        Builder profanity(Profanity profanity);

        /**
         * <p>
         * When you turn on brevity, Amazon Translate reduces the length of the translation output for most translations
         * (when compared with the same translation with brevity turned off). By default, brevity is turned off.
         * </p>
         * <p>
         * If you turn on brevity for a translation request with an unsupported language pair, the translation proceeds
         * with the brevity setting turned off.
         * </p>
         * <p>
         * For the language pairs that brevity supports, see <a
         * href="https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-brevity">Using brevity</a> in
         * the Amazon Translate Developer Guide.
         * </p>
         * 
         * @param brevity
         *        When you turn on brevity, Amazon Translate reduces the length of the translation output for most
         *        translations (when compared with the same translation with brevity turned off). By default, brevity is
         *        turned off.</p>
         *        <p>
         *        If you turn on brevity for a translation request with an unsupported language pair, the translation
         *        proceeds with the brevity setting turned off.
         *        </p>
         *        <p>
         *        For the language pairs that brevity supports, see <a
         *        href="https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-brevity">Using
         *        brevity</a> in the Amazon Translate Developer Guide.
         * @see Brevity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Brevity
         */
        Builder brevity(String brevity);

        /**
         * <p>
         * When you turn on brevity, Amazon Translate reduces the length of the translation output for most translations
         * (when compared with the same translation with brevity turned off). By default, brevity is turned off.
         * </p>
         * <p>
         * If you turn on brevity for a translation request with an unsupported language pair, the translation proceeds
         * with the brevity setting turned off.
         * </p>
         * <p>
         * For the language pairs that brevity supports, see <a
         * href="https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-brevity">Using brevity</a> in
         * the Amazon Translate Developer Guide.
         * </p>
         * 
         * @param brevity
         *        When you turn on brevity, Amazon Translate reduces the length of the translation output for most
         *        translations (when compared with the same translation with brevity turned off). By default, brevity is
         *        turned off.</p>
         *        <p>
         *        If you turn on brevity for a translation request with an unsupported language pair, the translation
         *        proceeds with the brevity setting turned off.
         *        </p>
         *        <p>
         *        For the language pairs that brevity supports, see <a
         *        href="https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-brevity">Using
         *        brevity</a> in the Amazon Translate Developer Guide.
         * @see Brevity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Brevity
         */
        Builder brevity(Brevity brevity);
    }

    static final class BuilderImpl implements Builder {
        private String formality;

        private String profanity;

        private String brevity;

        private BuilderImpl() {
        }

        private BuilderImpl(TranslationSettings model) {
            formality(model.formality);
            profanity(model.profanity);
            brevity(model.brevity);
        }

        public final String getFormality() {
            return formality;
        }

        public final void setFormality(String formality) {
            this.formality = formality;
        }

        @Override
        public final Builder formality(String formality) {
            this.formality = formality;
            return this;
        }

        @Override
        public final Builder formality(Formality formality) {
            this.formality(formality == null ? null : formality.toString());
            return this;
        }

        public final String getProfanity() {
            return profanity;
        }

        public final void setProfanity(String profanity) {
            this.profanity = profanity;
        }

        @Override
        public final Builder profanity(String profanity) {
            this.profanity = profanity;
            return this;
        }

        @Override
        public final Builder profanity(Profanity profanity) {
            this.profanity(profanity == null ? null : profanity.toString());
            return this;
        }

        public final String getBrevity() {
            return brevity;
        }

        public final void setBrevity(String brevity) {
            this.brevity = brevity;
        }

        @Override
        public final Builder brevity(String brevity) {
            this.brevity = brevity;
            return this;
        }

        @Override
        public final Builder brevity(Brevity brevity) {
            this.brevity(brevity == null ? null : brevity.toString());
            return this;
        }

        @Override
        public TranslationSettings build() {
            return new TranslationSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
