/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.translate;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.translate.internal.TranslateServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.translate.model.ConcurrentModificationException;
import software.amazon.awssdk.services.translate.model.ConflictException;
import software.amazon.awssdk.services.translate.model.CreateParallelDataRequest;
import software.amazon.awssdk.services.translate.model.CreateParallelDataResponse;
import software.amazon.awssdk.services.translate.model.DeleteParallelDataRequest;
import software.amazon.awssdk.services.translate.model.DeleteParallelDataResponse;
import software.amazon.awssdk.services.translate.model.DeleteTerminologyRequest;
import software.amazon.awssdk.services.translate.model.DeleteTerminologyResponse;
import software.amazon.awssdk.services.translate.model.DescribeTextTranslationJobRequest;
import software.amazon.awssdk.services.translate.model.DescribeTextTranslationJobResponse;
import software.amazon.awssdk.services.translate.model.DetectedLanguageLowConfidenceException;
import software.amazon.awssdk.services.translate.model.GetParallelDataRequest;
import software.amazon.awssdk.services.translate.model.GetParallelDataResponse;
import software.amazon.awssdk.services.translate.model.GetTerminologyRequest;
import software.amazon.awssdk.services.translate.model.GetTerminologyResponse;
import software.amazon.awssdk.services.translate.model.ImportTerminologyRequest;
import software.amazon.awssdk.services.translate.model.ImportTerminologyResponse;
import software.amazon.awssdk.services.translate.model.InternalServerException;
import software.amazon.awssdk.services.translate.model.InvalidFilterException;
import software.amazon.awssdk.services.translate.model.InvalidParameterValueException;
import software.amazon.awssdk.services.translate.model.InvalidRequestException;
import software.amazon.awssdk.services.translate.model.LimitExceededException;
import software.amazon.awssdk.services.translate.model.ListLanguagesRequest;
import software.amazon.awssdk.services.translate.model.ListLanguagesResponse;
import software.amazon.awssdk.services.translate.model.ListParallelDataRequest;
import software.amazon.awssdk.services.translate.model.ListParallelDataResponse;
import software.amazon.awssdk.services.translate.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.translate.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.translate.model.ListTerminologiesRequest;
import software.amazon.awssdk.services.translate.model.ListTerminologiesResponse;
import software.amazon.awssdk.services.translate.model.ListTextTranslationJobsRequest;
import software.amazon.awssdk.services.translate.model.ListTextTranslationJobsResponse;
import software.amazon.awssdk.services.translate.model.ResourceNotFoundException;
import software.amazon.awssdk.services.translate.model.ServiceUnavailableException;
import software.amazon.awssdk.services.translate.model.StartTextTranslationJobRequest;
import software.amazon.awssdk.services.translate.model.StartTextTranslationJobResponse;
import software.amazon.awssdk.services.translate.model.StopTextTranslationJobRequest;
import software.amazon.awssdk.services.translate.model.StopTextTranslationJobResponse;
import software.amazon.awssdk.services.translate.model.TagResourceRequest;
import software.amazon.awssdk.services.translate.model.TagResourceResponse;
import software.amazon.awssdk.services.translate.model.TextSizeLimitExceededException;
import software.amazon.awssdk.services.translate.model.TooManyRequestsException;
import software.amazon.awssdk.services.translate.model.TooManyTagsException;
import software.amazon.awssdk.services.translate.model.TranslateDocumentRequest;
import software.amazon.awssdk.services.translate.model.TranslateDocumentResponse;
import software.amazon.awssdk.services.translate.model.TranslateException;
import software.amazon.awssdk.services.translate.model.TranslateTextRequest;
import software.amazon.awssdk.services.translate.model.TranslateTextResponse;
import software.amazon.awssdk.services.translate.model.UnsupportedDisplayLanguageCodeException;
import software.amazon.awssdk.services.translate.model.UnsupportedLanguagePairException;
import software.amazon.awssdk.services.translate.model.UntagResourceRequest;
import software.amazon.awssdk.services.translate.model.UntagResourceResponse;
import software.amazon.awssdk.services.translate.model.UpdateParallelDataRequest;
import software.amazon.awssdk.services.translate.model.UpdateParallelDataResponse;
import software.amazon.awssdk.services.translate.transform.CreateParallelDataRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.DeleteParallelDataRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.DeleteTerminologyRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.DescribeTextTranslationJobRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.GetParallelDataRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.GetTerminologyRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.ImportTerminologyRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.ListLanguagesRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.ListParallelDataRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.ListTerminologiesRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.ListTextTranslationJobsRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.StartTextTranslationJobRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.StopTextTranslationJobRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.TranslateDocumentRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.TranslateTextRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.translate.transform.UpdateParallelDataRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link TranslateAsyncClient}.
 *
 * @see TranslateAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTranslateAsyncClient implements TranslateAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultTranslateAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultTranslateAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a parallel data resource in Amazon Translate by importing an input file from Amazon S3. Parallel data
     * files contain examples that show how you want segments of text to be translated. By adding parallel data, you can
     * influence the style, tone, and word choice in your translation output.
     * </p>
     *
     * @param createParallelDataRequest
     * @return A Java Future containing the result of the CreateParallelData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>InvalidRequestException The request that you made is not valid. Check your request to determine why
     *         it's not valid and then retry the request.</li>
     *         <li>LimitExceededException The specified limit has been exceeded. Review your request and retry it with a
     *         quantity below the stated limit.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>TooManyTagsException You have added too many tags to this resource. The maximum is 50 tags.</li>
     *         <li>ConflictException There was a conflict processing the request. Try your request again.</li>
     *         <li>ConcurrentModificationException Another modification is being made. That modification must complete
     *         before you can make your change.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.CreateParallelData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/CreateParallelData" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateParallelDataResponse> createParallelData(CreateParallelDataRequest createParallelDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createParallelDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createParallelDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateParallelData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateParallelDataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateParallelDataResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DetectedLanguageLowConfidenceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DetectedLanguageLowConfidenceException")
                            .httpStatusCode(400).exceptionBuilderSupplier(DetectedLanguageLowConfidenceException::builder)
                            .build());
                case "InvalidParameterValueException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidParameterValueException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedLanguagePairException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedLanguagePairException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedLanguagePairException::builder).build());
                case "InvalidFilterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidFilterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidFilterException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "TextSizeLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "UnsupportedDisplayLanguageCodeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDisplayLanguageCodeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedDisplayLanguageCodeException::builder)
                            .build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateParallelDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateParallelDataRequest, CreateParallelDataResponse>()
                            .withOperationName("CreateParallelData").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateParallelDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createParallelDataRequest));
            CompletableFuture<CreateParallelDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a parallel data resource in Amazon Translate.
     * </p>
     *
     * @param deleteParallelDataRequest
     * @return A Java Future containing the result of the DeleteParallelData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConcurrentModificationException Another modification is being made. That modification must complete
     *         before you can make your change.</li>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.DeleteParallelData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/DeleteParallelData" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteParallelDataResponse> deleteParallelData(DeleteParallelDataRequest deleteParallelDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteParallelDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteParallelDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteParallelData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteParallelDataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteParallelDataResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DetectedLanguageLowConfidenceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DetectedLanguageLowConfidenceException")
                            .httpStatusCode(400).exceptionBuilderSupplier(DetectedLanguageLowConfidenceException::builder)
                            .build());
                case "InvalidParameterValueException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidParameterValueException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedLanguagePairException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedLanguagePairException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedLanguagePairException::builder).build());
                case "InvalidFilterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidFilterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidFilterException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "TextSizeLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "UnsupportedDisplayLanguageCodeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDisplayLanguageCodeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedDisplayLanguageCodeException::builder)
                            .build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteParallelDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteParallelDataRequest, DeleteParallelDataResponse>()
                            .withOperationName("DeleteParallelData").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteParallelDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteParallelDataRequest));
            CompletableFuture<DeleteParallelDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * A synchronous action that deletes a custom terminology.
     * </p>
     *
     * @param deleteTerminologyRequest
     * @return A Java Future containing the result of the DeleteTerminology operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.DeleteTerminology
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/DeleteTerminology" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTerminologyResponse> deleteTerminology(DeleteTerminologyRequest deleteTerminologyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTerminologyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTerminologyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTerminology");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTerminologyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteTerminologyResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DetectedLanguageLowConfidenceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DetectedLanguageLowConfidenceException")
                            .httpStatusCode(400).exceptionBuilderSupplier(DetectedLanguageLowConfidenceException::builder)
                            .build());
                case "InvalidParameterValueException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidParameterValueException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedLanguagePairException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedLanguagePairException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedLanguagePairException::builder).build());
                case "InvalidFilterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidFilterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidFilterException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "TextSizeLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "UnsupportedDisplayLanguageCodeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDisplayLanguageCodeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedDisplayLanguageCodeException::builder)
                            .build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteTerminologyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTerminologyRequest, DeleteTerminologyResponse>()
                            .withOperationName("DeleteTerminology").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteTerminologyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteTerminologyRequest));
            CompletableFuture<DeleteTerminologyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the properties associated with an asynchronous batch translation job including name, ID, status, source and
     * target languages, input/output S3 buckets, and so on.
     * </p>
     *
     * @param describeTextTranslationJobRequest
     * @return A Java Future containing the result of the DescribeTextTranslationJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.DescribeTextTranslationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/DescribeTextTranslationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTextTranslationJobResponse> describeTextTranslationJob(
            DescribeTextTranslationJobRequest describeTextTranslationJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeTextTranslationJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTextTranslationJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTextTranslationJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTextTranslationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeTextTranslationJobResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DetectedLanguageLowConfidenceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DetectedLanguageLowConfidenceException")
                            .httpStatusCode(400).exceptionBuilderSupplier(DetectedLanguageLowConfidenceException::builder)
                            .build());
                case "InvalidParameterValueException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidParameterValueException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedLanguagePairException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedLanguagePairException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedLanguagePairException::builder).build());
                case "InvalidFilterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidFilterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidFilterException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "TextSizeLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "UnsupportedDisplayLanguageCodeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDisplayLanguageCodeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedDisplayLanguageCodeException::builder)
                            .build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DescribeTextTranslationJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTextTranslationJobRequest, DescribeTextTranslationJobResponse>()
                            .withOperationName("DescribeTextTranslationJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeTextTranslationJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeTextTranslationJobRequest));
            CompletableFuture<DescribeTextTranslationJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about a parallel data resource.
     * </p>
     *
     * @param getParallelDataRequest
     * @return A Java Future containing the result of the GetParallelData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.GetParallelData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/GetParallelData" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetParallelDataResponse> getParallelData(GetParallelDataRequest getParallelDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getParallelDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getParallelDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetParallelData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetParallelDataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetParallelDataResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DetectedLanguageLowConfidenceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DetectedLanguageLowConfidenceException")
                            .httpStatusCode(400).exceptionBuilderSupplier(DetectedLanguageLowConfidenceException::builder)
                            .build());
                case "InvalidParameterValueException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidParameterValueException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedLanguagePairException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedLanguagePairException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedLanguagePairException::builder).build());
                case "InvalidFilterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidFilterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidFilterException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "TextSizeLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "UnsupportedDisplayLanguageCodeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDisplayLanguageCodeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedDisplayLanguageCodeException::builder)
                            .build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetParallelDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetParallelDataRequest, GetParallelDataResponse>()
                            .withOperationName("GetParallelData").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetParallelDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getParallelDataRequest));
            CompletableFuture<GetParallelDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a custom terminology.
     * </p>
     *
     * @param getTerminologyRequest
     * @return A Java Future containing the result of the GetTerminology operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.GetTerminology
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/GetTerminology" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTerminologyResponse> getTerminology(GetTerminologyRequest getTerminologyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTerminologyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTerminologyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTerminology");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTerminologyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTerminologyResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DetectedLanguageLowConfidenceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DetectedLanguageLowConfidenceException")
                            .httpStatusCode(400).exceptionBuilderSupplier(DetectedLanguageLowConfidenceException::builder)
                            .build());
                case "InvalidParameterValueException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidParameterValueException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedLanguagePairException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedLanguagePairException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedLanguagePairException::builder).build());
                case "InvalidFilterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidFilterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidFilterException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "TextSizeLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "UnsupportedDisplayLanguageCodeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDisplayLanguageCodeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedDisplayLanguageCodeException::builder)
                            .build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetTerminologyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTerminologyRequest, GetTerminologyResponse>()
                            .withOperationName("GetTerminology").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTerminologyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTerminologyRequest));
            CompletableFuture<GetTerminologyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates a custom terminology, depending on whether one already exists for the given terminology name.
     * Importing a terminology with the same name as an existing one will merge the terminologies based on the chosen
     * merge strategy. The only supported merge strategy is OVERWRITE, where the imported terminology overwrites the
     * existing terminology of the same name.
     * </p>
     * <p>
     * If you import a terminology that overwrites an existing one, the new terminology takes up to 10 minutes to fully
     * propagate. After that, translations have access to the new terminology.
     * </p>
     *
     * @param importTerminologyRequest
     * @return A Java Future containing the result of the ImportTerminology operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>LimitExceededException The specified limit has been exceeded. Review your request and retry it with a
     *         quantity below the stated limit.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>TooManyTagsException You have added too many tags to this resource. The maximum is 50 tags.</li>
     *         <li>ConcurrentModificationException Another modification is being made. That modification must complete
     *         before you can make your change.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.ImportTerminology
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ImportTerminology" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ImportTerminologyResponse> importTerminology(ImportTerminologyRequest importTerminologyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(importTerminologyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, importTerminologyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportTerminology");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ImportTerminologyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ImportTerminologyResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DetectedLanguageLowConfidenceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DetectedLanguageLowConfidenceException")
                            .httpStatusCode(400).exceptionBuilderSupplier(DetectedLanguageLowConfidenceException::builder)
                            .build());
                case "InvalidParameterValueException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidParameterValueException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedLanguagePairException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedLanguagePairException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedLanguagePairException::builder).build());
                case "InvalidFilterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidFilterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidFilterException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "TextSizeLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "UnsupportedDisplayLanguageCodeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDisplayLanguageCodeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedDisplayLanguageCodeException::builder)
                            .build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ImportTerminologyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ImportTerminologyRequest, ImportTerminologyResponse>()
                            .withOperationName("ImportTerminology").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ImportTerminologyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(importTerminologyRequest));
            CompletableFuture<ImportTerminologyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a list of languages (RFC-5646 codes and names) that Amazon Translate supports.
     * </p>
     *
     * @param listLanguagesRequest
     * @return A Java Future containing the result of the ListLanguages operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>UnsupportedDisplayLanguageCodeException Requested display language code is not supported.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.ListLanguages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListLanguages" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListLanguagesResponse> listLanguages(ListLanguagesRequest listLanguagesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listLanguagesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLanguagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLanguages");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLanguagesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListLanguagesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DetectedLanguageLowConfidenceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DetectedLanguageLowConfidenceException")
                            .httpStatusCode(400).exceptionBuilderSupplier(DetectedLanguageLowConfidenceException::builder)
                            .build());
                case "InvalidParameterValueException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidParameterValueException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedLanguagePairException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedLanguagePairException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedLanguagePairException::builder).build());
                case "InvalidFilterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidFilterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidFilterException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "TextSizeLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "UnsupportedDisplayLanguageCodeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDisplayLanguageCodeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedDisplayLanguageCodeException::builder)
                            .build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListLanguagesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLanguagesRequest, ListLanguagesResponse>()
                            .withOperationName("ListLanguages").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListLanguagesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listLanguagesRequest));
            CompletableFuture<ListLanguagesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a list of your parallel data resources in Amazon Translate.
     * </p>
     *
     * @param listParallelDataRequest
     * @return A Java Future containing the result of the ListParallelData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.ListParallelData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListParallelData" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListParallelDataResponse> listParallelData(ListParallelDataRequest listParallelDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listParallelDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listParallelDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListParallelData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListParallelDataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListParallelDataResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DetectedLanguageLowConfidenceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DetectedLanguageLowConfidenceException")
                            .httpStatusCode(400).exceptionBuilderSupplier(DetectedLanguageLowConfidenceException::builder)
                            .build());
                case "InvalidParameterValueException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidParameterValueException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedLanguagePairException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedLanguagePairException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedLanguagePairException::builder).build());
                case "InvalidFilterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidFilterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidFilterException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "TextSizeLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "UnsupportedDisplayLanguageCodeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDisplayLanguageCodeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedDisplayLanguageCodeException::builder)
                            .build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListParallelDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListParallelDataRequest, ListParallelDataResponse>()
                            .withOperationName("ListParallelData").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListParallelDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listParallelDataRequest));
            CompletableFuture<ListParallelDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all tags associated with a given Amazon Translate resource. For more information, see <a
     * href="https://docs.aws.amazon.com/translate/latest/dg/tagging.html"> Tagging your resources</a>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DetectedLanguageLowConfidenceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DetectedLanguageLowConfidenceException")
                            .httpStatusCode(400).exceptionBuilderSupplier(DetectedLanguageLowConfidenceException::builder)
                            .build());
                case "InvalidParameterValueException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidParameterValueException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedLanguagePairException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedLanguagePairException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedLanguagePairException::builder).build());
                case "InvalidFilterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidFilterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidFilterException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "TextSizeLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "UnsupportedDisplayLanguageCodeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDisplayLanguageCodeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedDisplayLanguageCodeException::builder)
                            .build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a list of custom terminologies associated with your account.
     * </p>
     *
     * @param listTerminologiesRequest
     * @return A Java Future containing the result of the ListTerminologies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.ListTerminologies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListTerminologies" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTerminologiesResponse> listTerminologies(ListTerminologiesRequest listTerminologiesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTerminologiesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTerminologiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTerminologies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTerminologiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTerminologiesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DetectedLanguageLowConfidenceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DetectedLanguageLowConfidenceException")
                            .httpStatusCode(400).exceptionBuilderSupplier(DetectedLanguageLowConfidenceException::builder)
                            .build());
                case "InvalidParameterValueException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidParameterValueException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedLanguagePairException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedLanguagePairException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedLanguagePairException::builder).build());
                case "InvalidFilterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidFilterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidFilterException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "TextSizeLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "UnsupportedDisplayLanguageCodeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDisplayLanguageCodeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedDisplayLanguageCodeException::builder)
                            .build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTerminologiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTerminologiesRequest, ListTerminologiesResponse>()
                            .withOperationName("ListTerminologies").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTerminologiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTerminologiesRequest));
            CompletableFuture<ListTerminologiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list of the batch translation jobs that you have submitted.
     * </p>
     *
     * @param listTextTranslationJobsRequest
     * @return A Java Future containing the result of the ListTextTranslationJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request that you made is not valid. Check your request to determine why
     *         it's not valid and then retry the request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>InvalidFilterException The filter specified for the operation is not valid. Specify a different
     *         filter.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.ListTextTranslationJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/ListTextTranslationJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTextTranslationJobsResponse> listTextTranslationJobs(
            ListTextTranslationJobsRequest listTextTranslationJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTextTranslationJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTextTranslationJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTextTranslationJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTextTranslationJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTextTranslationJobsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DetectedLanguageLowConfidenceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DetectedLanguageLowConfidenceException")
                            .httpStatusCode(400).exceptionBuilderSupplier(DetectedLanguageLowConfidenceException::builder)
                            .build());
                case "InvalidParameterValueException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidParameterValueException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedLanguagePairException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedLanguagePairException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedLanguagePairException::builder).build());
                case "InvalidFilterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidFilterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidFilterException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "TextSizeLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "UnsupportedDisplayLanguageCodeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDisplayLanguageCodeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedDisplayLanguageCodeException::builder)
                            .build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTextTranslationJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTextTranslationJobsRequest, ListTextTranslationJobsResponse>()
                            .withOperationName("ListTextTranslationJobs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTextTranslationJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTextTranslationJobsRequest));
            CompletableFuture<ListTextTranslationJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts an asynchronous batch translation job. Use batch translation jobs to translate large volumes of text
     * across multiple documents at once. For batch translation, you can input documents with different source languages
     * (specify <code>auto</code> as the source language). You can specify one or more target languages. Batch
     * translation translates each input document into each of the target languages. For more information, see <a
     * href="https://docs.aws.amazon.com/translate/latest/dg/async.html">Asynchronous batch processing</a>.
     * </p>
     * <p>
     * Batch translation jobs can be described with the <a>DescribeTextTranslationJob</a> operation, listed with the
     * <a>ListTextTranslationJobs</a> operation, and stopped with the <a>StopTextTranslationJob</a> operation.
     * </p>
     *
     * @param startTextTranslationJobRequest
     * @return A Java Future containing the result of the StartTextTranslationJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>UnsupportedLanguagePairException Amazon Translate does not support translation from the language of
     *         the source text into the requested target language. For more information, see <a
     *         href="https://docs.aws.amazon.com/translate/latest/dg/what-is-languages.html">Supported languages</a>.</li>
     *         <li>InvalidRequestException The request that you made is not valid. Check your request to determine why
     *         it's not valid and then retry the request.</li>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.StartTextTranslationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/StartTextTranslationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartTextTranslationJobResponse> startTextTranslationJob(
            StartTextTranslationJobRequest startTextTranslationJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startTextTranslationJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startTextTranslationJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartTextTranslationJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartTextTranslationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartTextTranslationJobResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DetectedLanguageLowConfidenceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DetectedLanguageLowConfidenceException")
                            .httpStatusCode(400).exceptionBuilderSupplier(DetectedLanguageLowConfidenceException::builder)
                            .build());
                case "InvalidParameterValueException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidParameterValueException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedLanguagePairException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedLanguagePairException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedLanguagePairException::builder).build());
                case "InvalidFilterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidFilterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidFilterException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "TextSizeLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "UnsupportedDisplayLanguageCodeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDisplayLanguageCodeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedDisplayLanguageCodeException::builder)
                            .build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StartTextTranslationJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartTextTranslationJobRequest, StartTextTranslationJobResponse>()
                            .withOperationName("StartTextTranslationJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartTextTranslationJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startTextTranslationJobRequest));
            CompletableFuture<StartTextTranslationJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops an asynchronous batch translation job that is in progress.
     * </p>
     * <p>
     * If the job's state is <code>IN_PROGRESS</code>, the job will be marked for termination and put into the
     * <code>STOP_REQUESTED</code> state. If the job completes before it can be stopped, it is put into the
     * <code>COMPLETED</code> state. Otherwise, the job is put into the <code>STOPPED</code> state.
     * </p>
     * <p>
     * Asynchronous batch translation jobs are started with the <a>StartTextTranslationJob</a> operation. You can use
     * the <a>DescribeTextTranslationJob</a> or <a>ListTextTranslationJobs</a> operations to get a batch translation
     * job's <code>JobId</code>.
     * </p>
     *
     * @param stopTextTranslationJobRequest
     * @return A Java Future containing the result of the StopTextTranslationJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.StopTextTranslationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/StopTextTranslationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopTextTranslationJobResponse> stopTextTranslationJob(
            StopTextTranslationJobRequest stopTextTranslationJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopTextTranslationJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopTextTranslationJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopTextTranslationJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopTextTranslationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopTextTranslationJobResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DetectedLanguageLowConfidenceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DetectedLanguageLowConfidenceException")
                            .httpStatusCode(400).exceptionBuilderSupplier(DetectedLanguageLowConfidenceException::builder)
                            .build());
                case "InvalidParameterValueException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidParameterValueException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedLanguagePairException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedLanguagePairException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedLanguagePairException::builder).build());
                case "InvalidFilterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidFilterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidFilterException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "TextSizeLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "UnsupportedDisplayLanguageCodeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDisplayLanguageCodeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedDisplayLanguageCodeException::builder)
                            .build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StopTextTranslationJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopTextTranslationJobRequest, StopTextTranslationJobResponse>()
                            .withOperationName("StopTextTranslationJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopTextTranslationJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopTextTranslationJobRequest));
            CompletableFuture<StopTextTranslationJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates a specific tag with a resource. A tag is a key-value pair that adds as a metadata to a resource. For
     * more information, see <a href="https://docs.aws.amazon.com/translate/latest/dg/tagging.html"> Tagging your
     * resources</a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>ConcurrentModificationException Another modification is being made. That modification must complete
     *         before you can make your change.</li>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>TooManyTagsException You have added too many tags to this resource. The maximum is 50 tags.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DetectedLanguageLowConfidenceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DetectedLanguageLowConfidenceException")
                            .httpStatusCode(400).exceptionBuilderSupplier(DetectedLanguageLowConfidenceException::builder)
                            .build());
                case "InvalidParameterValueException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidParameterValueException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedLanguagePairException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedLanguagePairException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedLanguagePairException::builder).build());
                case "InvalidFilterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidFilterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidFilterException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "TextSizeLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "UnsupportedDisplayLanguageCodeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDisplayLanguageCodeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedDisplayLanguageCodeException::builder)
                            .build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Translates the input document from the source language to the target language. This synchronous operation
     * supports text, HTML, or Word documents as the input document. <code>TranslateDocument</code> supports
     * translations from English to any supported language, and from any supported language to English. Therefore,
     * specify either the source language code or the target language code as “en” (English).
     * </p>
     * <p>
     * If you set the <code>Formality</code> parameter, the request will fail if the target language does not support
     * formality. For a list of target languages that support formality, see <a
     * href="https://docs.aws.amazon.com/translate/latest/dg/customizing-translations-formality.html">Setting
     * formality</a>.
     * </p>
     *
     * @param translateDocumentRequest
     * @return A Java Future containing the result of the TranslateDocument operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request that you made is not valid. Check your request to determine why
     *         it's not valid and then retry the request.</li>
     *         <li>LimitExceededException The specified limit has been exceeded. Review your request and retry it with a
     *         quantity below the stated limit.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>UnsupportedLanguagePairException Amazon Translate does not support translation from the language of
     *         the source text into the requested target language. For more information, see <a
     *         href="https://docs.aws.amazon.com/translate/latest/dg/what-is-languages.html">Supported languages</a>.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ServiceUnavailableException The Amazon Translate service is temporarily unavailable. Wait a bit and
     *         then retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.TranslateDocument
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/TranslateDocument" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TranslateDocumentResponse> translateDocument(TranslateDocumentRequest translateDocumentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(translateDocumentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, translateDocumentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TranslateDocument");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TranslateDocumentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, TranslateDocumentResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DetectedLanguageLowConfidenceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DetectedLanguageLowConfidenceException")
                            .httpStatusCode(400).exceptionBuilderSupplier(DetectedLanguageLowConfidenceException::builder)
                            .build());
                case "InvalidParameterValueException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidParameterValueException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedLanguagePairException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedLanguagePairException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedLanguagePairException::builder).build());
                case "InvalidFilterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidFilterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidFilterException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "TextSizeLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "UnsupportedDisplayLanguageCodeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDisplayLanguageCodeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedDisplayLanguageCodeException::builder)
                            .build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<TranslateDocumentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TranslateDocumentRequest, TranslateDocumentResponse>()
                            .withOperationName("TranslateDocument").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TranslateDocumentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(translateDocumentRequest));
            CompletableFuture<TranslateDocumentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Translates input text from the source language to the target language. For a list of available languages and
     * language codes, see <a href="https://docs.aws.amazon.com/translate/latest/dg/what-is-languages.html">Supported
     * languages</a>.
     * </p>
     *
     * @param translateTextRequest
     * @return A Java Future containing the result of the TranslateText operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request that you made is not valid. Check your request to determine why
     *         it's not valid and then retry the request.</li>
     *         <li>TextSizeLimitExceededException The size of the text you submitted exceeds the size limit. Reduce the
     *         size of the text or use a smaller document and then retry your request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>UnsupportedLanguagePairException Amazon Translate does not support translation from the language of
     *         the source text into the requested target language. For more information, see <a
     *         href="https://docs.aws.amazon.com/translate/latest/dg/what-is-languages.html">Supported languages</a>.</li>
     *         <li>DetectedLanguageLowConfidenceException The confidence that Amazon Comprehend accurately detected the
     *         source language is low. If a low confidence level is acceptable for your application, you can use the
     *         language in the exception to call Amazon Translate again. For more information, see the <a
     *         href="https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectDominantLanguage.html"
     *         >DetectDominantLanguage</a> operation in the <i>Amazon Comprehend Developer Guide</i>.</li>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ServiceUnavailableException The Amazon Translate service is temporarily unavailable. Wait a bit and
     *         then retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.TranslateText
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/TranslateText" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TranslateTextResponse> translateText(TranslateTextRequest translateTextRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(translateTextRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, translateTextRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TranslateText");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TranslateTextResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TranslateTextResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DetectedLanguageLowConfidenceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DetectedLanguageLowConfidenceException")
                            .httpStatusCode(400).exceptionBuilderSupplier(DetectedLanguageLowConfidenceException::builder)
                            .build());
                case "InvalidParameterValueException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidParameterValueException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedLanguagePairException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedLanguagePairException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedLanguagePairException::builder).build());
                case "InvalidFilterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidFilterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidFilterException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "TextSizeLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "UnsupportedDisplayLanguageCodeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDisplayLanguageCodeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedDisplayLanguageCodeException::builder)
                            .build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<TranslateTextResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TranslateTextRequest, TranslateTextResponse>()
                            .withOperationName("TranslateText").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TranslateTextRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(translateTextRequest));
            CompletableFuture<TranslateTextResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a specific tag associated with an Amazon Translate resource. For more information, see <a
     * href="https://docs.aws.amazon.com/translate/latest/dg/tagging.html"> Tagging your resources</a>.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>ConcurrentModificationException Another modification is being made. That modification must complete
     *         before you can make your change.</li>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DetectedLanguageLowConfidenceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DetectedLanguageLowConfidenceException")
                            .httpStatusCode(400).exceptionBuilderSupplier(DetectedLanguageLowConfidenceException::builder)
                            .build());
                case "InvalidParameterValueException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidParameterValueException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedLanguagePairException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedLanguagePairException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedLanguagePairException::builder).build());
                case "InvalidFilterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidFilterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidFilterException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "TextSizeLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "UnsupportedDisplayLanguageCodeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDisplayLanguageCodeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedDisplayLanguageCodeException::builder)
                            .build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a previously created parallel data resource by importing a new input file from Amazon S3.
     * </p>
     *
     * @param updateParallelDataRequest
     * @return A Java Future containing the result of the UpdateParallelData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConcurrentModificationException Another modification is being made. That modification must complete
     *         before you can make your change.</li>
     *         <li>InvalidParameterValueException The value of the parameter is not valid. Review the value of the
     *         parameter you are using to correct it, and then retry your operation.</li>
     *         <li>InvalidRequestException The request that you made is not valid. Check your request to determine why
     *         it's not valid and then retry the request.</li>
     *         <li>LimitExceededException The specified limit has been exceeded. Review your request and retry it with a
     *         quantity below the stated limit.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again.</li>
     *         <li>ConflictException There was a conflict processing the request. Try your request again.</li>
     *         <li>ResourceNotFoundException The resource you are looking for has not been found. Review the resource
     *         you're looking for and see if a different resource will accomplish your needs before retrying the revised
     *         request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TranslateException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample TranslateAsyncClient.UpdateParallelData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/translate-2017-07-01/UpdateParallelData" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateParallelDataResponse> updateParallelData(UpdateParallelDataRequest updateParallelDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateParallelDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateParallelDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Translate");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateParallelData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateParallelDataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateParallelDataResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DetectedLanguageLowConfidenceException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DetectedLanguageLowConfidenceException")
                            .httpStatusCode(400).exceptionBuilderSupplier(DetectedLanguageLowConfidenceException::builder)
                            .build());
                case "InvalidParameterValueException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                            .httpStatusCode(400).exceptionBuilderSupplier(InvalidParameterValueException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedLanguagePairException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedLanguagePairException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedLanguagePairException::builder).build());
                case "InvalidFilterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidFilterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidFilterException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "TextSizeLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "UnsupportedDisplayLanguageCodeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedDisplayLanguageCodeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedDisplayLanguageCodeException::builder)
                            .build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateParallelDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateParallelDataRequest, UpdateParallelDataResponse>()
                            .withOperationName("UpdateParallelData").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateParallelDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateParallelDataRequest));
            CompletableFuture<UpdateParallelDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final TranslateServiceClientConfiguration serviceClientConfiguration() {
        return new TranslateServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(TranslateException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON).protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        TranslateServiceClientConfigurationBuilder serviceConfigBuilder = new TranslateServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
