/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.trustedadvisor;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.trustedadvisor.internal.TrustedAdvisorServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.trustedadvisor.model.AccessDeniedException;
import software.amazon.awssdk.services.trustedadvisor.model.BatchUpdateRecommendationResourceExclusionRequest;
import software.amazon.awssdk.services.trustedadvisor.model.BatchUpdateRecommendationResourceExclusionResponse;
import software.amazon.awssdk.services.trustedadvisor.model.ConflictException;
import software.amazon.awssdk.services.trustedadvisor.model.GetOrganizationRecommendationRequest;
import software.amazon.awssdk.services.trustedadvisor.model.GetOrganizationRecommendationResponse;
import software.amazon.awssdk.services.trustedadvisor.model.GetRecommendationRequest;
import software.amazon.awssdk.services.trustedadvisor.model.GetRecommendationResponse;
import software.amazon.awssdk.services.trustedadvisor.model.InternalServerException;
import software.amazon.awssdk.services.trustedadvisor.model.ListChecksRequest;
import software.amazon.awssdk.services.trustedadvisor.model.ListChecksResponse;
import software.amazon.awssdk.services.trustedadvisor.model.ListOrganizationRecommendationAccountsRequest;
import software.amazon.awssdk.services.trustedadvisor.model.ListOrganizationRecommendationAccountsResponse;
import software.amazon.awssdk.services.trustedadvisor.model.ListOrganizationRecommendationResourcesRequest;
import software.amazon.awssdk.services.trustedadvisor.model.ListOrganizationRecommendationResourcesResponse;
import software.amazon.awssdk.services.trustedadvisor.model.ListOrganizationRecommendationsRequest;
import software.amazon.awssdk.services.trustedadvisor.model.ListOrganizationRecommendationsResponse;
import software.amazon.awssdk.services.trustedadvisor.model.ListRecommendationResourcesRequest;
import software.amazon.awssdk.services.trustedadvisor.model.ListRecommendationResourcesResponse;
import software.amazon.awssdk.services.trustedadvisor.model.ListRecommendationsRequest;
import software.amazon.awssdk.services.trustedadvisor.model.ListRecommendationsResponse;
import software.amazon.awssdk.services.trustedadvisor.model.ResourceNotFoundException;
import software.amazon.awssdk.services.trustedadvisor.model.ThrottlingException;
import software.amazon.awssdk.services.trustedadvisor.model.TrustedAdvisorException;
import software.amazon.awssdk.services.trustedadvisor.model.UpdateOrganizationRecommendationLifecycleRequest;
import software.amazon.awssdk.services.trustedadvisor.model.UpdateOrganizationRecommendationLifecycleResponse;
import software.amazon.awssdk.services.trustedadvisor.model.UpdateRecommendationLifecycleRequest;
import software.amazon.awssdk.services.trustedadvisor.model.UpdateRecommendationLifecycleResponse;
import software.amazon.awssdk.services.trustedadvisor.model.ValidationException;
import software.amazon.awssdk.services.trustedadvisor.transform.BatchUpdateRecommendationResourceExclusionRequestMarshaller;
import software.amazon.awssdk.services.trustedadvisor.transform.GetOrganizationRecommendationRequestMarshaller;
import software.amazon.awssdk.services.trustedadvisor.transform.GetRecommendationRequestMarshaller;
import software.amazon.awssdk.services.trustedadvisor.transform.ListChecksRequestMarshaller;
import software.amazon.awssdk.services.trustedadvisor.transform.ListOrganizationRecommendationAccountsRequestMarshaller;
import software.amazon.awssdk.services.trustedadvisor.transform.ListOrganizationRecommendationResourcesRequestMarshaller;
import software.amazon.awssdk.services.trustedadvisor.transform.ListOrganizationRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.trustedadvisor.transform.ListRecommendationResourcesRequestMarshaller;
import software.amazon.awssdk.services.trustedadvisor.transform.ListRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.trustedadvisor.transform.UpdateOrganizationRecommendationLifecycleRequestMarshaller;
import software.amazon.awssdk.services.trustedadvisor.transform.UpdateRecommendationLifecycleRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link TrustedAdvisorAsyncClient}.
 *
 * @see TrustedAdvisorAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTrustedAdvisorAsyncClient implements TrustedAdvisorAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultTrustedAdvisorAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultTrustedAdvisorAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Update one or more exclusion status for a list of recommendation resources
     * </p>
     *
     * @param batchUpdateRecommendationResourceExclusionRequest
     * @return A Java Future containing the result of the BatchUpdateRecommendationResourceExclusion operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException Exception that access has been denied due to insufficient access</li>
     *         <li>ConflictException Exception that the request was denied due to conflictions in state</li>
     *         <li>InternalServerException Exception to notify that an unexpected internal error occurred during
     *         processing of the request</li>
     *         <li>ValidationException Exception that the request failed to satisfy service constraints</li>
     *         <li>ThrottlingException Exception to notify that requests are being throttled</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TrustedAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TrustedAdvisorAsyncClient.BatchUpdateRecommendationResourceExclusion
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/trustedadvisor-2022-09-15/BatchUpdateRecommendationResourceExclusion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchUpdateRecommendationResourceExclusionResponse> batchUpdateRecommendationResourceExclusion(
            BatchUpdateRecommendationResourceExclusionRequest batchUpdateRecommendationResourceExclusionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                batchUpdateRecommendationResourceExclusionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                batchUpdateRecommendationResourceExclusionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TrustedAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchUpdateRecommendationResourceExclusion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchUpdateRecommendationResourceExclusionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, BatchUpdateRecommendationResourceExclusionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchUpdateRecommendationResourceExclusionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchUpdateRecommendationResourceExclusionRequest, BatchUpdateRecommendationResourceExclusionResponse>()
                            .withOperationName("BatchUpdateRecommendationResourceExclusion")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchUpdateRecommendationResourceExclusionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchUpdateRecommendationResourceExclusionRequest));
            CompletableFuture<BatchUpdateRecommendationResourceExclusionResponse> whenCompleted = executeFuture.whenComplete((r,
                    e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get a specific recommendation within an AWS Organizations organization. This API supports only prioritized
     * recommendations.
     * </p>
     *
     * @param getOrganizationRecommendationRequest
     * @return A Java Future containing the result of the GetOrganizationRecommendation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException Exception that access has been denied due to insufficient access</li>
     *         <li>InternalServerException Exception to notify that an unexpected internal error occurred during
     *         processing of the request</li>
     *         <li>ValidationException Exception that the request failed to satisfy service constraints</li>
     *         <li>ResourceNotFoundException Exception that the requested resource has not been found</li>
     *         <li>ThrottlingException Exception to notify that requests are being throttled</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TrustedAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TrustedAdvisorAsyncClient.GetOrganizationRecommendation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/trustedadvisor-2022-09-15/GetOrganizationRecommendation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetOrganizationRecommendationResponse> getOrganizationRecommendation(
            GetOrganizationRecommendationRequest getOrganizationRecommendationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getOrganizationRecommendationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getOrganizationRecommendationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TrustedAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOrganizationRecommendation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetOrganizationRecommendationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetOrganizationRecommendationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetOrganizationRecommendationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetOrganizationRecommendationRequest, GetOrganizationRecommendationResponse>()
                            .withOperationName("GetOrganizationRecommendation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetOrganizationRecommendationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getOrganizationRecommendationRequest));
            CompletableFuture<GetOrganizationRecommendationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get a specific Recommendation
     * </p>
     *
     * @param getRecommendationRequest
     * @return A Java Future containing the result of the GetRecommendation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException Exception that access has been denied due to insufficient access</li>
     *         <li>InternalServerException Exception to notify that an unexpected internal error occurred during
     *         processing of the request</li>
     *         <li>ValidationException Exception that the request failed to satisfy service constraints</li>
     *         <li>ResourceNotFoundException Exception that the requested resource has not been found</li>
     *         <li>ThrottlingException Exception to notify that requests are being throttled</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TrustedAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TrustedAdvisorAsyncClient.GetRecommendation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/trustedadvisor-2022-09-15/GetRecommendation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRecommendationResponse> getRecommendation(GetRecommendationRequest getRecommendationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRecommendationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRecommendationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TrustedAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecommendation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRecommendationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRecommendationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRecommendationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRecommendationRequest, GetRecommendationResponse>()
                            .withOperationName("GetRecommendation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRecommendationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRecommendationRequest));
            CompletableFuture<GetRecommendationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List a filterable set of Checks
     * </p>
     *
     * @param listChecksRequest
     * @return A Java Future containing the result of the ListChecks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException Exception that access has been denied due to insufficient access</li>
     *         <li>InternalServerException Exception to notify that an unexpected internal error occurred during
     *         processing of the request</li>
     *         <li>ValidationException Exception that the request failed to satisfy service constraints</li>
     *         <li>ThrottlingException Exception to notify that requests are being throttled</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TrustedAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TrustedAdvisorAsyncClient.ListChecks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/trustedadvisor-2022-09-15/ListChecks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListChecksResponse> listChecks(ListChecksRequest listChecksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listChecksRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChecksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TrustedAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChecks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListChecksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListChecksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListChecksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListChecksRequest, ListChecksResponse>().withOperationName("ListChecks")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListChecksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listChecksRequest));
            CompletableFuture<ListChecksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the accounts that own the resources for an organization aggregate recommendation. This API only supports
     * prioritized recommendations.
     * </p>
     *
     * @param listOrganizationRecommendationAccountsRequest
     * @return A Java Future containing the result of the ListOrganizationRecommendationAccounts operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException Exception that access has been denied due to insufficient access</li>
     *         <li>InternalServerException Exception to notify that an unexpected internal error occurred during
     *         processing of the request</li>
     *         <li>ValidationException Exception that the request failed to satisfy service constraints</li>
     *         <li>ResourceNotFoundException Exception that the requested resource has not been found</li>
     *         <li>ThrottlingException Exception to notify that requests are being throttled</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TrustedAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TrustedAdvisorAsyncClient.ListOrganizationRecommendationAccounts
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/trustedadvisor-2022-09-15/ListOrganizationRecommendationAccounts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListOrganizationRecommendationAccountsResponse> listOrganizationRecommendationAccounts(
            ListOrganizationRecommendationAccountsRequest listOrganizationRecommendationAccountsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listOrganizationRecommendationAccountsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listOrganizationRecommendationAccountsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TrustedAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOrganizationRecommendationAccounts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListOrganizationRecommendationAccountsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListOrganizationRecommendationAccountsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListOrganizationRecommendationAccountsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListOrganizationRecommendationAccountsRequest, ListOrganizationRecommendationAccountsResponse>()
                            .withOperationName("ListOrganizationRecommendationAccounts").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListOrganizationRecommendationAccountsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listOrganizationRecommendationAccountsRequest));
            CompletableFuture<ListOrganizationRecommendationAccountsResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List Resources of a Recommendation within an Organization. This API only supports prioritized recommendations.
     * </p>
     *
     * @param listOrganizationRecommendationResourcesRequest
     * @return A Java Future containing the result of the ListOrganizationRecommendationResources operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException Exception that access has been denied due to insufficient access</li>
     *         <li>InternalServerException Exception to notify that an unexpected internal error occurred during
     *         processing of the request</li>
     *         <li>ValidationException Exception that the request failed to satisfy service constraints</li>
     *         <li>ResourceNotFoundException Exception that the requested resource has not been found</li>
     *         <li>ThrottlingException Exception to notify that requests are being throttled</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TrustedAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TrustedAdvisorAsyncClient.ListOrganizationRecommendationResources
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/trustedadvisor-2022-09-15/ListOrganizationRecommendationResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListOrganizationRecommendationResourcesResponse> listOrganizationRecommendationResources(
            ListOrganizationRecommendationResourcesRequest listOrganizationRecommendationResourcesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listOrganizationRecommendationResourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listOrganizationRecommendationResourcesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TrustedAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOrganizationRecommendationResources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListOrganizationRecommendationResourcesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListOrganizationRecommendationResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListOrganizationRecommendationResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListOrganizationRecommendationResourcesRequest, ListOrganizationRecommendationResourcesResponse>()
                            .withOperationName("ListOrganizationRecommendationResources").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListOrganizationRecommendationResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listOrganizationRecommendationResourcesRequest));
            CompletableFuture<ListOrganizationRecommendationResourcesResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List a filterable set of Recommendations within an Organization. This API only supports prioritized
     * recommendations.
     * </p>
     *
     * @param listOrganizationRecommendationsRequest
     * @return A Java Future containing the result of the ListOrganizationRecommendations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException Exception that access has been denied due to insufficient access</li>
     *         <li>InternalServerException Exception to notify that an unexpected internal error occurred during
     *         processing of the request</li>
     *         <li>ValidationException Exception that the request failed to satisfy service constraints</li>
     *         <li>ThrottlingException Exception to notify that requests are being throttled</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TrustedAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TrustedAdvisorAsyncClient.ListOrganizationRecommendations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/trustedadvisor-2022-09-15/ListOrganizationRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListOrganizationRecommendationsResponse> listOrganizationRecommendations(
            ListOrganizationRecommendationsRequest listOrganizationRecommendationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listOrganizationRecommendationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listOrganizationRecommendationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TrustedAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOrganizationRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListOrganizationRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListOrganizationRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListOrganizationRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListOrganizationRecommendationsRequest, ListOrganizationRecommendationsResponse>()
                            .withOperationName("ListOrganizationRecommendations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListOrganizationRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listOrganizationRecommendationsRequest));
            CompletableFuture<ListOrganizationRecommendationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List Resources of a Recommendation
     * </p>
     *
     * @param listRecommendationResourcesRequest
     * @return A Java Future containing the result of the ListRecommendationResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException Exception that access has been denied due to insufficient access</li>
     *         <li>InternalServerException Exception to notify that an unexpected internal error occurred during
     *         processing of the request</li>
     *         <li>ValidationException Exception that the request failed to satisfy service constraints</li>
     *         <li>ResourceNotFoundException Exception that the requested resource has not been found</li>
     *         <li>ThrottlingException Exception to notify that requests are being throttled</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TrustedAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TrustedAdvisorAsyncClient.ListRecommendationResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/trustedadvisor-2022-09-15/ListRecommendationResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRecommendationResourcesResponse> listRecommendationResources(
            ListRecommendationResourcesRequest listRecommendationResourcesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRecommendationResourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRecommendationResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TrustedAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRecommendationResources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRecommendationResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRecommendationResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRecommendationResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRecommendationResourcesRequest, ListRecommendationResourcesResponse>()
                            .withOperationName("ListRecommendationResources").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRecommendationResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRecommendationResourcesRequest));
            CompletableFuture<ListRecommendationResourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List a filterable set of Recommendations
     * </p>
     *
     * @param listRecommendationsRequest
     * @return A Java Future containing the result of the ListRecommendations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException Exception that access has been denied due to insufficient access</li>
     *         <li>InternalServerException Exception to notify that an unexpected internal error occurred during
     *         processing of the request</li>
     *         <li>ValidationException Exception that the request failed to satisfy service constraints</li>
     *         <li>ThrottlingException Exception to notify that requests are being throttled</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TrustedAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TrustedAdvisorAsyncClient.ListRecommendations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/trustedadvisor-2022-09-15/ListRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRecommendationsResponse> listRecommendations(
            ListRecommendationsRequest listRecommendationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRecommendationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRecommendationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TrustedAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRecommendationsRequest, ListRecommendationsResponse>()
                            .withOperationName("ListRecommendations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRecommendationsRequest));
            CompletableFuture<ListRecommendationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update the lifecycle of a Recommendation within an Organization. This API only supports prioritized
     * recommendations.
     * </p>
     *
     * @param updateOrganizationRecommendationLifecycleRequest
     * @return A Java Future containing the result of the UpdateOrganizationRecommendationLifecycle operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException Exception that access has been denied due to insufficient access</li>
     *         <li>ConflictException Exception that the request was denied due to conflictions in state</li>
     *         <li>InternalServerException Exception to notify that an unexpected internal error occurred during
     *         processing of the request</li>
     *         <li>ValidationException Exception that the request failed to satisfy service constraints</li>
     *         <li>ResourceNotFoundException Exception that the requested resource has not been found</li>
     *         <li>ThrottlingException Exception to notify that requests are being throttled</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TrustedAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TrustedAdvisorAsyncClient.UpdateOrganizationRecommendationLifecycle
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/trustedadvisor-2022-09-15/UpdateOrganizationRecommendationLifecycle"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateOrganizationRecommendationLifecycleResponse> updateOrganizationRecommendationLifecycle(
            UpdateOrganizationRecommendationLifecycleRequest updateOrganizationRecommendationLifecycleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                updateOrganizationRecommendationLifecycleRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateOrganizationRecommendationLifecycleRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TrustedAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateOrganizationRecommendationLifecycle");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateOrganizationRecommendationLifecycleResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateOrganizationRecommendationLifecycleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateOrganizationRecommendationLifecycleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateOrganizationRecommendationLifecycleRequest, UpdateOrganizationRecommendationLifecycleResponse>()
                            .withOperationName("UpdateOrganizationRecommendationLifecycle")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateOrganizationRecommendationLifecycleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateOrganizationRecommendationLifecycleRequest));
            CompletableFuture<UpdateOrganizationRecommendationLifecycleResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update the lifecyle of a Recommendation. This API only supports prioritized recommendations.
     * </p>
     *
     * @param updateRecommendationLifecycleRequest
     * @return A Java Future containing the result of the UpdateRecommendationLifecycle operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException Exception that access has been denied due to insufficient access</li>
     *         <li>ConflictException Exception that the request was denied due to conflictions in state</li>
     *         <li>InternalServerException Exception to notify that an unexpected internal error occurred during
     *         processing of the request</li>
     *         <li>ValidationException Exception that the request failed to satisfy service constraints</li>
     *         <li>ResourceNotFoundException Exception that the requested resource has not been found</li>
     *         <li>ThrottlingException Exception to notify that requests are being throttled</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TrustedAdvisorException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample TrustedAdvisorAsyncClient.UpdateRecommendationLifecycle
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/trustedadvisor-2022-09-15/UpdateRecommendationLifecycle"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRecommendationLifecycleResponse> updateRecommendationLifecycle(
            UpdateRecommendationLifecycleRequest updateRecommendationLifecycleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRecommendationLifecycleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateRecommendationLifecycleRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "TrustedAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRecommendationLifecycle");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRecommendationLifecycleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateRecommendationLifecycleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRecommendationLifecycleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRecommendationLifecycleRequest, UpdateRecommendationLifecycleResponse>()
                            .withOperationName("UpdateRecommendationLifecycle").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateRecommendationLifecycleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateRecommendationLifecycleRequest));
            CompletableFuture<UpdateRecommendationLifecycleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final TrustedAdvisorServiceClientConfiguration serviceClientConfiguration() {
        return new TrustedAdvisorServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(TrustedAdvisorException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        TrustedAdvisorServiceClientConfigurationBuilder serviceConfigBuilder = new TrustedAdvisorServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
