/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.verifiedpermissions.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration for an identity source that represents a connection to an Amazon Cognito user pool used as an
 * identity provider for Verified Permissions.
 * </p>
 * <p>
 * This data type is used as a field that is part of an <a
 * href="https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_Configuration.html">Configuration</a>
 * structure that is used as a parameter to the <a
 * href="https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_Configuration.html">Configuration</a>.
 * </p>
 * <p>
 * Example:
 * <code>"CognitoUserPoolConfiguration":{"UserPoolArn":"cognito-idp:us-east-1:123456789012:userpool/us-east-1_1a2b3c4d5","ClientIds": ["a1b2c3d4e5f6g7h8i9j0kalbmc"]}</code>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CognitoUserPoolConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<CognitoUserPoolConfiguration.Builder, CognitoUserPoolConfiguration> {
    private static final SdkField<String> USER_POOL_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("userPoolArn").getter(getter(CognitoUserPoolConfiguration::userPoolArn))
            .setter(setter(Builder::userPoolArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("userPoolArn").build()).build();

    private static final SdkField<List<String>> CLIENT_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("clientIds")
            .getter(getter(CognitoUserPoolConfiguration::clientIds))
            .setter(setter(Builder::clientIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clientIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(USER_POOL_ARN_FIELD,
            CLIENT_IDS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String userPoolArn;

    private final List<String> clientIds;

    private CognitoUserPoolConfiguration(BuilderImpl builder) {
        this.userPoolArn = builder.userPoolArn;
        this.clientIds = builder.clientIds;
    }

    /**
     * <p>
     * The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     * (ARN)</a> of the Amazon Cognito user pool that contains the identities to be authorized.
     * </p>
     * <p>
     * Example: <code>"UserPoolArn": "cognito-idp:us-east-1:123456789012:userpool/us-east-1_1a2b3c4d5"</code>
     * </p>
     * 
     * @return The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource
     *         Name (ARN)</a> of the Amazon Cognito user pool that contains the identities to be authorized.</p>
     *         <p>
     *         Example: <code>"UserPoolArn": "cognito-idp:us-east-1:123456789012:userpool/us-east-1_1a2b3c4d5"</code>
     */
    public final String userPoolArn() {
        return userPoolArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the ClientIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasClientIds() {
        return clientIds != null && !(clientIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The unique application client IDs that are associated with the specified Amazon Cognito user pool.
     * </p>
     * <p>
     * Example: <code>"ClientIds": ["&amp;ExampleCogClientId;"]</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasClientIds} method.
     * </p>
     * 
     * @return The unique application client IDs that are associated with the specified Amazon Cognito user pool.</p>
     *         <p>
     *         Example: <code>"ClientIds": ["&amp;ExampleCogClientId;"]</code>
     */
    public final List<String> clientIds() {
        return clientIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(userPoolArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasClientIds() ? clientIds() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CognitoUserPoolConfiguration)) {
            return false;
        }
        CognitoUserPoolConfiguration other = (CognitoUserPoolConfiguration) obj;
        return Objects.equals(userPoolArn(), other.userPoolArn()) && hasClientIds() == other.hasClientIds()
                && Objects.equals(clientIds(), other.clientIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CognitoUserPoolConfiguration").add("UserPoolArn", userPoolArn())
                .add("ClientIds", hasClientIds() ? clientIds() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "userPoolArn":
            return Optional.ofNullable(clazz.cast(userPoolArn()));
        case "clientIds":
            return Optional.ofNullable(clazz.cast(clientIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CognitoUserPoolConfiguration, T> g) {
        return obj -> g.apply((CognitoUserPoolConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CognitoUserPoolConfiguration> {
        /**
         * <p>
         * The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
         * (ARN)</a> of the Amazon Cognito user pool that contains the identities to be authorized.
         * </p>
         * <p>
         * Example: <code>"UserPoolArn": "cognito-idp:us-east-1:123456789012:userpool/us-east-1_1a2b3c4d5"</code>
         * </p>
         * 
         * @param userPoolArn
         *        The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
         *        Resource Name (ARN)</a> of the Amazon Cognito user pool that contains the identities to be
         *        authorized.</p>
         *        <p>
         *        Example: <code>"UserPoolArn": "cognito-idp:us-east-1:123456789012:userpool/us-east-1_1a2b3c4d5"</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userPoolArn(String userPoolArn);

        /**
         * <p>
         * The unique application client IDs that are associated with the specified Amazon Cognito user pool.
         * </p>
         * <p>
         * Example: <code>"ClientIds": ["&amp;ExampleCogClientId;"]</code>
         * </p>
         * 
         * @param clientIds
         *        The unique application client IDs that are associated with the specified Amazon Cognito user pool.</p>
         *        <p>
         *        Example: <code>"ClientIds": ["&amp;ExampleCogClientId;"]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientIds(Collection<String> clientIds);

        /**
         * <p>
         * The unique application client IDs that are associated with the specified Amazon Cognito user pool.
         * </p>
         * <p>
         * Example: <code>"ClientIds": ["&amp;ExampleCogClientId;"]</code>
         * </p>
         * 
         * @param clientIds
         *        The unique application client IDs that are associated with the specified Amazon Cognito user pool.</p>
         *        <p>
         *        Example: <code>"ClientIds": ["&amp;ExampleCogClientId;"]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientIds(String... clientIds);
    }

    static final class BuilderImpl implements Builder {
        private String userPoolArn;

        private List<String> clientIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CognitoUserPoolConfiguration model) {
            userPoolArn(model.userPoolArn);
            clientIds(model.clientIds);
        }

        public final String getUserPoolArn() {
            return userPoolArn;
        }

        public final void setUserPoolArn(String userPoolArn) {
            this.userPoolArn = userPoolArn;
        }

        @Override
        public final Builder userPoolArn(String userPoolArn) {
            this.userPoolArn = userPoolArn;
            return this;
        }

        public final Collection<String> getClientIds() {
            if (clientIds instanceof SdkAutoConstructList) {
                return null;
            }
            return clientIds;
        }

        public final void setClientIds(Collection<String> clientIds) {
            this.clientIds = ClientIdsCopier.copy(clientIds);
        }

        @Override
        public final Builder clientIds(Collection<String> clientIds) {
            this.clientIds = ClientIdsCopier.copy(clientIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder clientIds(String... clientIds) {
            clientIds(Arrays.asList(clientIds));
            return this;
        }

        @Override
        public CognitoUserPoolConfiguration build() {
            return new CognitoUserPoolConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
