/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.verifiedpermissions;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.verifiedpermissions.internal.VerifiedPermissionsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.verifiedpermissions.model.AccessDeniedException;
import software.amazon.awssdk.services.verifiedpermissions.model.ConflictException;
import software.amazon.awssdk.services.verifiedpermissions.model.CreateIdentitySourceRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.CreateIdentitySourceResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.CreatePolicyRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.CreatePolicyResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.CreatePolicyStoreRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.CreatePolicyStoreResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.CreatePolicyTemplateRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.CreatePolicyTemplateResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.DeleteIdentitySourceRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.DeleteIdentitySourceResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.DeletePolicyRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.DeletePolicyResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.DeletePolicyStoreRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.DeletePolicyStoreResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.DeletePolicyTemplateRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.DeletePolicyTemplateResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.GetIdentitySourceRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.GetIdentitySourceResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.GetPolicyRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.GetPolicyResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.GetPolicyStoreRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.GetPolicyStoreResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.GetPolicyTemplateRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.GetPolicyTemplateResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.GetSchemaRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.GetSchemaResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.InternalServerException;
import software.amazon.awssdk.services.verifiedpermissions.model.IsAuthorizedRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.IsAuthorizedResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.IsAuthorizedWithTokenRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.IsAuthorizedWithTokenResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.ListIdentitySourcesRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.ListIdentitySourcesResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.ListPoliciesRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.ListPoliciesResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.ListPolicyStoresRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.ListPolicyStoresResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.ListPolicyTemplatesRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.ListPolicyTemplatesResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.PutSchemaRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.PutSchemaResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.ResourceNotFoundException;
import software.amazon.awssdk.services.verifiedpermissions.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.verifiedpermissions.model.ThrottlingException;
import software.amazon.awssdk.services.verifiedpermissions.model.UpdateIdentitySourceRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.UpdateIdentitySourceResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.UpdatePolicyRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.UpdatePolicyResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.UpdatePolicyStoreRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.UpdatePolicyStoreResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.UpdatePolicyTemplateRequest;
import software.amazon.awssdk.services.verifiedpermissions.model.UpdatePolicyTemplateResponse;
import software.amazon.awssdk.services.verifiedpermissions.model.ValidationException;
import software.amazon.awssdk.services.verifiedpermissions.model.VerifiedPermissionsException;
import software.amazon.awssdk.services.verifiedpermissions.transform.CreateIdentitySourceRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.CreatePolicyRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.CreatePolicyStoreRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.CreatePolicyTemplateRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.DeleteIdentitySourceRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.DeletePolicyRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.DeletePolicyStoreRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.DeletePolicyTemplateRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.GetIdentitySourceRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.GetPolicyRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.GetPolicyStoreRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.GetPolicyTemplateRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.GetSchemaRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.IsAuthorizedRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.IsAuthorizedWithTokenRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.ListIdentitySourcesRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.ListPoliciesRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.ListPolicyStoresRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.ListPolicyTemplatesRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.PutSchemaRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.UpdateIdentitySourceRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.UpdatePolicyRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.UpdatePolicyStoreRequestMarshaller;
import software.amazon.awssdk.services.verifiedpermissions.transform.UpdatePolicyTemplateRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link VerifiedPermissionsAsyncClient}.
 *
 * @see VerifiedPermissionsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultVerifiedPermissionsAsyncClient implements VerifiedPermissionsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultVerifiedPermissionsAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final VerifiedPermissionsServiceClientConfiguration serviceClientConfiguration;

    protected DefaultVerifiedPermissionsAsyncClient(VerifiedPermissionsServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a reference to an Amazon Cognito user pool as an external identity provider (IdP).
     * </p>
     * <p>
     * After you create an identity source, you can use the identities provided by the IdP as proxies for the principal
     * in authorization queries that use the <a
     * href="https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_IsAuthorizedWithToken.html"
     * >IsAuthorizedWithToken</a> operation. These identities take the form of tokens that contain claims about the
     * user, such as IDs, attributes and group memberships. Amazon Cognito provides both identity tokens and access
     * tokens, and Verified Permissions can use either or both. Any combination of identity and access tokens results in
     * the same Cedar principal. Verified Permissions automatically translates the information about the identities into
     * the standard Cedar attributes that can be evaluated by your policies. Because the Amazon Cognito identity and
     * access tokens can contain different information, the tokens you choose to use determine which principal
     * attributes are available to access when evaluating Cedar policies.
     * </p>
     * <important>
     * <p>
     * If you delete a Amazon Cognito user pool or user, tokens from that deleted pool or that deleted user continue to
     * be usable until they expire.
     * </p>
     * </important> <note>
     * <p>
     * To reference a user from this identity source in your Cedar policies, use the following syntax.
     * </p>
     * <p>
     * <i>IdentityType::"&lt;CognitoUserPoolIdentifier&gt;|&lt;CognitoClientId&gt;</i>
     * </p>
     * <p>
     * Where <code>IdentityType</code> is the string that you provide to the <code>PrincipalEntityType</code> parameter
     * for this operation. The <code>CognitoUserPoolId</code> and <code>CognitoClientId</code> are defined by the Amazon
     * Cognito user pool.
     * </p>
     * </note> <note>
     * <p>
     * Verified Permissions is <i> <a href="https://wikipedia.org/wiki/Eventual_consistency">eventually consistent</a>
     * </i>. It can take a few seconds for a new or changed element to be propagate through the service and be visible
     * in the results of other Verified Permissions operations.
     * </p>
     * </note>
     *
     * @param createIdentitySourceRequest
     * @return A Java Future containing the result of the CreateIdentitySource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>ServiceQuotaExceededException The request failed because it would cause a service quota to be
     *         exceeded.</li>
     *         <li>ConflictException The request failed because another request to modify a resource occurred at the
     *         same.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request failed because it references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.CreateIdentitySource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/CreateIdentitySource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateIdentitySourceResponse> createIdentitySource(
            CreateIdentitySourceRequest createIdentitySourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createIdentitySourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createIdentitySourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateIdentitySource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateIdentitySourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateIdentitySourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateIdentitySourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateIdentitySourceRequest, CreateIdentitySourceResponse>()
                            .withOperationName("CreateIdentitySource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateIdentitySourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createIdentitySourceRequest));
            CompletableFuture<CreateIdentitySourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a Cedar policy and saves it in the specified policy store. You can create either a static policy or a
     * policy linked to a policy template.
     * </p>
     * <ul>
     * <li>
     * <p>
     * To create a static policy, provide the Cedar policy text in the <code>StaticPolicy</code> section of the
     * <code>PolicyDefinition</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To create a policy that is dynamically linked to a policy template, specify the policy template ID and the
     * principal and resource to associate with this policy in the <code>templateLinked</code> section of the
     * <code>PolicyDefinition</code>. If the policy template is ever updated, any policies linked to the policy template
     * automatically use the updated template.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * Creating a policy causes it to be validated against the schema in the policy store. If the policy doesn't pass
     * validation, the operation fails and the policy isn't stored.
     * </p>
     * </note> <note>
     * <p>
     * Verified Permissions is <i> <a href="https://wikipedia.org/wiki/Eventual_consistency">eventually consistent</a>
     * </i>. It can take a few seconds for a new or changed element to be propagate through the service and be visible
     * in the results of other Verified Permissions operations.
     * </p>
     * </note>
     *
     * @param createPolicyRequest
     * @return A Java Future containing the result of the CreatePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>ServiceQuotaExceededException The request failed because it would cause a service quota to be
     *         exceeded.</li>
     *         <li>ConflictException The request failed because another request to modify a resource occurred at the
     *         same.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request failed because it references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.CreatePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/CreatePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePolicyResponse> createPolicy(CreatePolicyRequest createPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPolicyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreatePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePolicyRequest, CreatePolicyResponse>()
                            .withOperationName("CreatePolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPolicyRequest));
            CompletableFuture<CreatePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a policy store. A policy store is a container for policy resources.
     * </p>
     * <note>
     * <p>
     * Although <a href="https://docs.cedarpolicy.com/schema/schema.html#namespace">Cedar supports multiple
     * namespaces</a>, Verified Permissions currently supports only one namespace per policy store.
     * </p>
     * </note> <note>
     * <p>
     * Verified Permissions is <i> <a href="https://wikipedia.org/wiki/Eventual_consistency">eventually consistent</a>
     * </i>. It can take a few seconds for a new or changed element to be propagate through the service and be visible
     * in the results of other Verified Permissions operations.
     * </p>
     * </note>
     *
     * @param createPolicyStoreRequest
     * @return A Java Future containing the result of the CreatePolicyStore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>ServiceQuotaExceededException The request failed because it would cause a service quota to be
     *         exceeded.</li>
     *         <li>ConflictException The request failed because another request to modify a resource occurred at the
     *         same.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.CreatePolicyStore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/CreatePolicyStore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePolicyStoreResponse> createPolicyStore(CreatePolicyStoreRequest createPolicyStoreRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPolicyStoreRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPolicyStoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePolicyStore");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePolicyStoreResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePolicyStoreResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePolicyStoreResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePolicyStoreRequest, CreatePolicyStoreResponse>()
                            .withOperationName("CreatePolicyStore").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePolicyStoreRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPolicyStoreRequest));
            CompletableFuture<CreatePolicyStoreResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a policy template. A template can use placeholders for the principal and resource. A template must be
     * instantiated into a policy by associating it with specific principals and resources to use for the placeholders.
     * That instantiated policy can then be considered in authorization decisions. The instantiated policy works
     * identically to any other policy, except that it is dynamically linked to the template. If the template changes,
     * then any policies that are linked to that template are immediately updated as well.
     * </p>
     * <note>
     * <p>
     * Verified Permissions is <i> <a href="https://wikipedia.org/wiki/Eventual_consistency">eventually consistent</a>
     * </i>. It can take a few seconds for a new or changed element to be propagate through the service and be visible
     * in the results of other Verified Permissions operations.
     * </p>
     * </note>
     *
     * @param createPolicyTemplateRequest
     * @return A Java Future containing the result of the CreatePolicyTemplate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>ServiceQuotaExceededException The request failed because it would cause a service quota to be
     *         exceeded.</li>
     *         <li>ConflictException The request failed because another request to modify a resource occurred at the
     *         same.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request failed because it references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.CreatePolicyTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/CreatePolicyTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePolicyTemplateResponse> createPolicyTemplate(
            CreatePolicyTemplateRequest createPolicyTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPolicyTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPolicyTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePolicyTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePolicyTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePolicyTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePolicyTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePolicyTemplateRequest, CreatePolicyTemplateResponse>()
                            .withOperationName("CreatePolicyTemplate").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePolicyTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPolicyTemplateRequest));
            CompletableFuture<CreatePolicyTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an identity source that references an identity provider (IdP) such as Amazon Cognito. After you delete
     * the identity source, you can no longer use tokens for identities from that identity source to represent
     * principals in authorization queries made using <a
     * href="https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_IsAuthorizedWithToken.html"
     * >IsAuthorizedWithToken</a>. operations.
     * </p>
     *
     * @param deleteIdentitySourceRequest
     * @return A Java Future containing the result of the DeleteIdentitySource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>ConflictException The request failed because another request to modify a resource occurred at the
     *         same.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request failed because it references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.DeleteIdentitySource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/DeleteIdentitySource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteIdentitySourceResponse> deleteIdentitySource(
            DeleteIdentitySourceRequest deleteIdentitySourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteIdentitySourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteIdentitySourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteIdentitySource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteIdentitySourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteIdentitySourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteIdentitySourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteIdentitySourceRequest, DeleteIdentitySourceResponse>()
                            .withOperationName("DeleteIdentitySource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteIdentitySourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteIdentitySourceRequest));
            CompletableFuture<DeleteIdentitySourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified policy from the policy store.
     * </p>
     * <p>
     * This operation is idempotent; if you specify a policy that doesn't exist, the request response returns a
     * successful <code>HTTP 200</code> status code.
     * </p>
     *
     * @param deletePolicyRequest
     * @return A Java Future containing the result of the DeletePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>ConflictException The request failed because another request to modify a resource occurred at the
     *         same.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request failed because it references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.DeletePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/DeletePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePolicyResponse> deletePolicy(DeletePolicyRequest deletePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePolicyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeletePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePolicyRequest, DeletePolicyResponse>()
                            .withOperationName("DeletePolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeletePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePolicyRequest));
            CompletableFuture<DeletePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified policy store.
     * </p>
     * <p>
     * This operation is idempotent. If you specify a policy store that does not exist, the request response will still
     * return a successful HTTP 200 status code.
     * </p>
     *
     * @param deletePolicyStoreRequest
     * @return A Java Future containing the result of the DeletePolicyStore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.DeletePolicyStore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/DeletePolicyStore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePolicyStoreResponse> deletePolicyStore(DeletePolicyStoreRequest deletePolicyStoreRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePolicyStoreRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePolicyStoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePolicyStore");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePolicyStoreResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePolicyStoreResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePolicyStoreResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePolicyStoreRequest, DeletePolicyStoreResponse>()
                            .withOperationName("DeletePolicyStore").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeletePolicyStoreRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePolicyStoreRequest));
            CompletableFuture<DeletePolicyStoreResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified policy template from the policy store.
     * </p>
     * <important>
     * <p>
     * This operation also deletes any policies that were created from the specified policy template. Those policies are
     * immediately removed from all future API responses, and are asynchronously deleted from the policy store.
     * </p>
     * </important>
     *
     * @param deletePolicyTemplateRequest
     * @return A Java Future containing the result of the DeletePolicyTemplate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>ConflictException The request failed because another request to modify a resource occurred at the
     *         same.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request failed because it references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.DeletePolicyTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/DeletePolicyTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePolicyTemplateResponse> deletePolicyTemplate(
            DeletePolicyTemplateRequest deletePolicyTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePolicyTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePolicyTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePolicyTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePolicyTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePolicyTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePolicyTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePolicyTemplateRequest, DeletePolicyTemplateResponse>()
                            .withOperationName("DeletePolicyTemplate").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeletePolicyTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePolicyTemplateRequest));
            CompletableFuture<DeletePolicyTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the details about the specified identity source.
     * </p>
     *
     * @param getIdentitySourceRequest
     * @return A Java Future containing the result of the GetIdentitySource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request failed because it references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.GetIdentitySource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/GetIdentitySource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetIdentitySourceResponse> getIdentitySource(GetIdentitySourceRequest getIdentitySourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getIdentitySourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getIdentitySourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetIdentitySource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetIdentitySourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetIdentitySourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetIdentitySourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetIdentitySourceRequest, GetIdentitySourceResponse>()
                            .withOperationName("GetIdentitySource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetIdentitySourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getIdentitySourceRequest));
            CompletableFuture<GetIdentitySourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about the specified policy.
     * </p>
     *
     * @param getPolicyRequest
     * @return A Java Future containing the result of the GetPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request failed because it references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.GetPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/GetPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPolicyResponse> getPolicy(GetPolicyRequest getPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPolicyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPolicyRequest, GetPolicyResponse>().withOperationName("GetPolicy")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPolicyRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getPolicyRequest));
            CompletableFuture<GetPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details about a policy store.
     * </p>
     *
     * @param getPolicyStoreRequest
     * @return A Java Future containing the result of the GetPolicyStore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request failed because it references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.GetPolicyStore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/GetPolicyStore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPolicyStoreResponse> getPolicyStore(GetPolicyStoreRequest getPolicyStoreRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPolicyStoreRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPolicyStoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPolicyStore");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPolicyStoreResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPolicyStoreResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPolicyStoreResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPolicyStoreRequest, GetPolicyStoreResponse>()
                            .withOperationName("GetPolicyStore").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPolicyStoreRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPolicyStoreRequest));
            CompletableFuture<GetPolicyStoreResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve the details for the specified policy template in the specified policy store.
     * </p>
     *
     * @param getPolicyTemplateRequest
     * @return A Java Future containing the result of the GetPolicyTemplate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request failed because it references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.GetPolicyTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/GetPolicyTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPolicyTemplateResponse> getPolicyTemplate(GetPolicyTemplateRequest getPolicyTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPolicyTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPolicyTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPolicyTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPolicyTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPolicyTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPolicyTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPolicyTemplateRequest, GetPolicyTemplateResponse>()
                            .withOperationName("GetPolicyTemplate").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPolicyTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPolicyTemplateRequest));
            CompletableFuture<GetPolicyTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve the details for the specified schema in the specified policy store.
     * </p>
     *
     * @param getSchemaRequest
     * @return A Java Future containing the result of the GetSchema operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request failed because it references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.GetSchema
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/GetSchema" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSchemaResponse> getSchema(GetSchemaRequest getSchemaRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSchemaRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSchemaRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSchema");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSchemaResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetSchemaResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSchemaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSchemaRequest, GetSchemaResponse>().withOperationName("GetSchema")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSchemaRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getSchemaRequest));
            CompletableFuture<GetSchemaResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Makes an authorization decision about a service request described in the parameters. The information in the
     * parameters can also define additional context that Verified Permissions can include in the evaluation. The
     * request is evaluated against all matching policies in the specified policy store. The result of the decision is
     * either <code>Allow</code> or <code>Deny</code>, along with a list of the policies that resulted in the decision.
     * </p>
     *
     * @param isAuthorizedRequest
     * @return A Java Future containing the result of the IsAuthorized operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request failed because it references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.IsAuthorized
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/IsAuthorized"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<IsAuthorizedResponse> isAuthorized(IsAuthorizedRequest isAuthorizedRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(isAuthorizedRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, isAuthorizedRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "IsAuthorized");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<IsAuthorizedResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    IsAuthorizedResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<IsAuthorizedResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<IsAuthorizedRequest, IsAuthorizedResponse>()
                            .withOperationName("IsAuthorized").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new IsAuthorizedRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(isAuthorizedRequest));
            CompletableFuture<IsAuthorizedResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Makes an authorization decision about a service request described in the parameters. The principal in this
     * request comes from an external identity source in the form of an identity token formatted as a <a
     * href="https://wikipedia.org/wiki/JSON_Web_Token">JSON web token (JWT)</a>. The information in the parameters can
     * also define additional context that Verified Permissions can include in the evaluation. The request is evaluated
     * against all matching policies in the specified policy store. The result of the decision is either
     * <code>Allow</code> or <code>Deny</code>, along with a list of the policies that resulted in the decision.
     * </p>
     * <important>
     * <p>
     * If you specify the <code>identityToken</code> parameter, then this operation derives the principal from that
     * token. You must not also include that principal in the <code>entities</code> parameter or the operation fails and
     * reports a conflict between the two entity sources.
     * </p>
     * <p>
     * If you provide only an <code>accessToken</code>, then you can include the entity as part of the
     * <code>entities</code> parameter to provide additional attributes.
     * </p>
     * </important>
     * <p>
     * At this time, Verified Permissions accepts tokens from only Amazon Cognito.
     * </p>
     * <p>
     * Verified Permissions validates each token that is specified in a request by checking its expiration date and its
     * signature.
     * </p>
     * <important>
     * <p>
     * If you delete a Amazon Cognito user pool or user, tokens from that deleted pool or that deleted user continue to
     * be usable until they expire.
     * </p>
     * </important>
     *
     * @param isAuthorizedWithTokenRequest
     * @return A Java Future containing the result of the IsAuthorizedWithToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request failed because it references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.IsAuthorizedWithToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/IsAuthorizedWithToken"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<IsAuthorizedWithTokenResponse> isAuthorizedWithToken(
            IsAuthorizedWithTokenRequest isAuthorizedWithTokenRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(isAuthorizedWithTokenRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, isAuthorizedWithTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "IsAuthorizedWithToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<IsAuthorizedWithTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, IsAuthorizedWithTokenResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<IsAuthorizedWithTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<IsAuthorizedWithTokenRequest, IsAuthorizedWithTokenResponse>()
                            .withOperationName("IsAuthorizedWithToken").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new IsAuthorizedWithTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(isAuthorizedWithTokenRequest));
            CompletableFuture<IsAuthorizedWithTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a paginated list of all of the identity sources defined in the specified policy store.
     * </p>
     *
     * @param listIdentitySourcesRequest
     * @return A Java Future containing the result of the ListIdentitySources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request failed because it references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.ListIdentitySources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/ListIdentitySources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListIdentitySourcesResponse> listIdentitySources(
            ListIdentitySourcesRequest listIdentitySourcesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listIdentitySourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIdentitySourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIdentitySources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListIdentitySourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListIdentitySourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListIdentitySourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListIdentitySourcesRequest, ListIdentitySourcesResponse>()
                            .withOperationName("ListIdentitySources").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListIdentitySourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listIdentitySourcesRequest));
            CompletableFuture<ListIdentitySourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a paginated list of all policies stored in the specified policy store.
     * </p>
     *
     * @param listPoliciesRequest
     * @return A Java Future containing the result of the ListPolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request failed because it references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.ListPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/ListPolicies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPoliciesResponse> listPolicies(ListPoliciesRequest listPoliciesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPoliciesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPolicies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPoliciesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListPoliciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPoliciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPoliciesRequest, ListPoliciesResponse>()
                            .withOperationName("ListPolicies").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPoliciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPoliciesRequest));
            CompletableFuture<ListPoliciesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a paginated list of all policy stores in the calling Amazon Web Services account.
     * </p>
     *
     * @param listPolicyStoresRequest
     * @return A Java Future containing the result of the ListPolicyStores operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.ListPolicyStores
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/ListPolicyStores"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPolicyStoresResponse> listPolicyStores(ListPolicyStoresRequest listPolicyStoresRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPolicyStoresRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPolicyStoresRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPolicyStores");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPolicyStoresResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPolicyStoresResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPolicyStoresResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPolicyStoresRequest, ListPolicyStoresResponse>()
                            .withOperationName("ListPolicyStores").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPolicyStoresRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPolicyStoresRequest));
            CompletableFuture<ListPolicyStoresResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a paginated list of all policy templates in the specified policy store.
     * </p>
     *
     * @param listPolicyTemplatesRequest
     * @return A Java Future containing the result of the ListPolicyTemplates operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request failed because it references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.ListPolicyTemplates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/ListPolicyTemplates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPolicyTemplatesResponse> listPolicyTemplates(
            ListPolicyTemplatesRequest listPolicyTemplatesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPolicyTemplatesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPolicyTemplatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPolicyTemplates");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPolicyTemplatesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPolicyTemplatesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPolicyTemplatesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPolicyTemplatesRequest, ListPolicyTemplatesResponse>()
                            .withOperationName("ListPolicyTemplates").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPolicyTemplatesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPolicyTemplatesRequest));
            CompletableFuture<ListPolicyTemplatesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates the policy schema in the specified policy store. The schema is used to validate any Cedar
     * policies and policy templates submitted to the policy store. Any changes to the schema validate only policies and
     * templates submitted after the schema change. Existing policies and templates are not re-evaluated against the
     * changed schema. If you later update a policy, then it is evaluated against the new schema at that time.
     * </p>
     * <note>
     * <p>
     * Verified Permissions is <i> <a href="https://wikipedia.org/wiki/Eventual_consistency">eventually consistent</a>
     * </i>. It can take a few seconds for a new or changed element to be propagate through the service and be visible
     * in the results of other Verified Permissions operations.
     * </p>
     * </note>
     *
     * @param putSchemaRequest
     * @return A Java Future containing the result of the PutSchema operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>ServiceQuotaExceededException The request failed because it would cause a service quota to be
     *         exceeded.</li>
     *         <li>ConflictException The request failed because another request to modify a resource occurred at the
     *         same.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request failed because it references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.PutSchema
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/PutSchema" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutSchemaResponse> putSchema(PutSchemaRequest putSchemaRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putSchemaRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putSchemaRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutSchema");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutSchemaResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutSchemaResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutSchemaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutSchemaRequest, PutSchemaResponse>().withOperationName("PutSchema")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutSchemaRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(putSchemaRequest));
            CompletableFuture<PutSchemaResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified identity source to use a new identity provider (IdP) source, or to change the mapping of
     * identities from the IdP to a different principal entity type.
     * </p>
     * <note>
     * <p>
     * Verified Permissions is <i> <a href="https://wikipedia.org/wiki/Eventual_consistency">eventually consistent</a>
     * </i>. It can take a few seconds for a new or changed element to be propagate through the service and be visible
     * in the results of other Verified Permissions operations.
     * </p>
     * </note>
     *
     * @param updateIdentitySourceRequest
     * @return A Java Future containing the result of the UpdateIdentitySource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>ConflictException The request failed because another request to modify a resource occurred at the
     *         same.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request failed because it references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.UpdateIdentitySource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/UpdateIdentitySource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateIdentitySourceResponse> updateIdentitySource(
            UpdateIdentitySourceRequest updateIdentitySourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateIdentitySourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateIdentitySourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateIdentitySource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateIdentitySourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateIdentitySourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateIdentitySourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateIdentitySourceRequest, UpdateIdentitySourceResponse>()
                            .withOperationName("UpdateIdentitySource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateIdentitySourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateIdentitySourceRequest));
            CompletableFuture<UpdateIdentitySourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies a Cedar static policy in the specified policy store. You can change only certain elements of the <a
     * href=
     * "https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_UpdatePolicyInput.html#amazonverifiedpermissions-UpdatePolicy-request-UpdatePolicyDefinition"
     * >UpdatePolicyDefinition</a> parameter. You can directly update only static policies. To change a template-linked
     * policy, you must update the template instead, using <a
     * href="https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_UpdatePolicyTemplate.html"
     * >UpdatePolicyTemplate</a>.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * If policy validation is enabled in the policy store, then updating a static policy causes Verified Permissions to
     * validate the policy against the schema in the policy store. If the updated static policy doesn't pass validation,
     * the operation fails and the update isn't stored.
     * </p>
     * </li>
     * <li>
     * <p>
     * When you edit a static policy, You can change only certain elements of a static policy:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The action referenced by the policy.
     * </p>
     * </li>
     * <li>
     * <p>
     * A condition clause, such as when and unless.
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can't change these elements of a static policy:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Changing a policy from a static policy to a template-linked policy.
     * </p>
     * </li>
     * <li>
     * <p>
     * Changing the effect of a static policy from permit or forbid.
     * </p>
     * </li>
     * <li>
     * <p>
     * The principal referenced by a static policy.
     * </p>
     * </li>
     * <li>
     * <p>
     * The resource referenced by a static policy.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * To update a template-linked policy, you must update the template instead.
     * </p>
     * </li>
     * </ul>
     * </note> <note>
     * <p>
     * Verified Permissions is <i> <a href="https://wikipedia.org/wiki/Eventual_consistency">eventually consistent</a>
     * </i>. It can take a few seconds for a new or changed element to be propagate through the service and be visible
     * in the results of other Verified Permissions operations.
     * </p>
     * </note>
     *
     * @param updatePolicyRequest
     * @return A Java Future containing the result of the UpdatePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>ServiceQuotaExceededException The request failed because it would cause a service quota to be
     *         exceeded.</li>
     *         <li>ConflictException The request failed because another request to modify a resource occurred at the
     *         same.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request failed because it references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.UpdatePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/UpdatePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePolicyResponse> updatePolicy(UpdatePolicyRequest updatePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePolicyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdatePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePolicyRequest, UpdatePolicyResponse>()
                            .withOperationName("UpdatePolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdatePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePolicyRequest));
            CompletableFuture<UpdatePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies the validation setting for a policy store.
     * </p>
     * <note>
     * <p>
     * Verified Permissions is <i> <a href="https://wikipedia.org/wiki/Eventual_consistency">eventually consistent</a>
     * </i>. It can take a few seconds for a new or changed element to be propagate through the service and be visible
     * in the results of other Verified Permissions operations.
     * </p>
     * </note>
     *
     * @param updatePolicyStoreRequest
     * @return A Java Future containing the result of the UpdatePolicyStore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>ConflictException The request failed because another request to modify a resource occurred at the
     *         same.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request failed because it references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.UpdatePolicyStore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/UpdatePolicyStore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePolicyStoreResponse> updatePolicyStore(UpdatePolicyStoreRequest updatePolicyStoreRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePolicyStoreRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePolicyStoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePolicyStore");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePolicyStoreResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePolicyStoreResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePolicyStoreResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePolicyStoreRequest, UpdatePolicyStoreResponse>()
                            .withOperationName("UpdatePolicyStore").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdatePolicyStoreRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePolicyStoreRequest));
            CompletableFuture<UpdatePolicyStoreResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified policy template. You can update only the description and the some elements of the <a href=
     * "https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_UpdatePolicyTemplate.html#amazonverifiedpermissions-UpdatePolicyTemplate-request-policyBody"
     * >policyBody</a>.
     * </p>
     * <important>
     * <p>
     * Changes you make to the policy template content are immediately (within the constraints of eventual consistency)
     * reflected in authorization decisions that involve all template-linked policies instantiated from this template.
     * </p>
     * </important> <note>
     * <p>
     * Verified Permissions is <i> <a href="https://wikipedia.org/wiki/Eventual_consistency">eventually consistent</a>
     * </i>. It can take a few seconds for a new or changed element to be propagate through the service and be visible
     * in the results of other Verified Permissions operations.
     * </p>
     * </note>
     *
     * @param updatePolicyTemplateRequest
     * @return A Java Future containing the result of the UpdatePolicyTemplate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because one or more input parameters don't satisfy their
     *         constraint requirements. The output is provided as a list of fields and a reason for each field that
     *         isn't valid.</p>
     *         <p>
     *         The possible reasons include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>UnrecognizedEntityType</b>
     *         </p>
     *         <p>
     *         The policy includes an entity type that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnrecognizedActionId</b>
     *         </p>
     *         <p>
     *         The policy includes an action id that isn't found in the schema.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InvalidActionApplication</b>
     *         </p>
     *         <p>
     *         The policy includes an action that, according to the schema, doesn't support the specified principal and
     *         resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnexpectedType</b>
     *         </p>
     *         <p>
     *         The policy included an operand that isn't a valid type for the specified operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IncompatibleTypes</b>
     *         </p>
     *         <p>
     *         The types of elements included in a <code>set</code>, or the types of expressions used in an
     *         <code>if...then...else</code> clause aren't compatible in this context.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>MissingAttribute</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that isn't specified in the schema. Test for
     *         the existence of the attribute first before attempting to access its value. For more information, see the
     *         <a href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UnsafeOptionalAttributeAccess</b>
     *         </p>
     *         <p>
     *         The policy attempts to access a record or entity attribute that is optional and isn't guaranteed to be
     *         present. Test for the existence of the attribute first before attempting to access its value. For more
     *         information, see the <a
     *         href="https://docs.cedarpolicy.com/policies/syntax-operators.html#has-presence-of-attribute-test">has
     *         (presence of attribute test) operator</a> in the <i>Cedar Policy Language Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ImpossiblePolicy</b>
     *         </p>
     *         <p>
     *         Cedar has determined that a policy condition always evaluates to false. If the policy is always false, it
     *         can never apply to any query, and so it can never affect an authorization decision.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WrongNumberArguments</b>
     *         </p>
     *         <p>
     *         The policy references an extension type with the wrong number of arguments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FunctionArgumentValidationError</b>
     *         </p>
     *         <p>
     *         Cedar couldn't parse the argument passed to an extension type. For example, a string that is to be parsed
     *         as an IPv4 address can contain only digits and the period character.
     *         </p>
     *         </li></li>
     *         <li>ConflictException The request failed because another request to modify a resource occurred at the
     *         same.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request failed because it references a resource that doesn't exist.</li>
     *         <li>ThrottlingException The request failed because it exceeded a throttling quota.</li>
     *         <li>InternalServerException The request failed because of an internal error. Try your request again later
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VerifiedPermissionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample VerifiedPermissionsAsyncClient.UpdatePolicyTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/verifiedpermissions-2021-12-01/UpdatePolicyTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePolicyTemplateResponse> updatePolicyTemplate(
            UpdatePolicyTemplateRequest updatePolicyTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePolicyTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePolicyTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "VerifiedPermissions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePolicyTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePolicyTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePolicyTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePolicyTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePolicyTemplateRequest, UpdatePolicyTemplateResponse>()
                            .withOperationName("UpdatePolicyTemplate").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdatePolicyTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePolicyTemplateRequest));
            CompletableFuture<UpdatePolicyTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final VerifiedPermissionsServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(VerifiedPermissionsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        VerifiedPermissionsServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = VerifiedPermissionsServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
