/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.verifiedpermissions.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains configuration details of a Amazon Cognito user pool for use with an identity source.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateCognitoUserPoolConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<UpdateCognitoUserPoolConfiguration.Builder, UpdateCognitoUserPoolConfiguration> {
    private static final SdkField<String> USER_POOL_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("userPoolArn").getter(getter(UpdateCognitoUserPoolConfiguration::userPoolArn))
            .setter(setter(Builder::userPoolArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("userPoolArn").build()).build();

    private static final SdkField<List<String>> CLIENT_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("clientIds")
            .getter(getter(UpdateCognitoUserPoolConfiguration::clientIds))
            .setter(setter(Builder::clientIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clientIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<UpdateCognitoGroupConfiguration> GROUP_CONFIGURATION_FIELD = SdkField
            .<UpdateCognitoGroupConfiguration> builder(MarshallingType.SDK_POJO).memberName("groupConfiguration")
            .getter(getter(UpdateCognitoUserPoolConfiguration::groupConfiguration)).setter(setter(Builder::groupConfiguration))
            .constructor(UpdateCognitoGroupConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("groupConfiguration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(USER_POOL_ARN_FIELD,
            CLIENT_IDS_FIELD, GROUP_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String userPoolArn;

    private final List<String> clientIds;

    private final UpdateCognitoGroupConfiguration groupConfiguration;

    private UpdateCognitoUserPoolConfiguration(BuilderImpl builder) {
        this.userPoolArn = builder.userPoolArn;
        this.clientIds = builder.clientIds;
        this.groupConfiguration = builder.groupConfiguration;
    }

    /**
     * <p>
     * The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     * (ARN)</a> of the Amazon Cognito user pool associated with this identity source.
     * </p>
     * 
     * @return The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource
     *         Name (ARN)</a> of the Amazon Cognito user pool associated with this identity source.
     */
    public final String userPoolArn() {
        return userPoolArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the ClientIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasClientIds() {
        return clientIds != null && !(clientIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The client ID of an app client that is configured for the specified Amazon Cognito user pool.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasClientIds} method.
     * </p>
     * 
     * @return The client ID of an app client that is configured for the specified Amazon Cognito user pool.
     */
    public final List<String> clientIds() {
        return clientIds;
    }

    /**
     * <p>
     * The configuration of the user groups from an Amazon Cognito user pool identity source.
     * </p>
     * 
     * @return The configuration of the user groups from an Amazon Cognito user pool identity source.
     */
    public final UpdateCognitoGroupConfiguration groupConfiguration() {
        return groupConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(userPoolArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasClientIds() ? clientIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(groupConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateCognitoUserPoolConfiguration)) {
            return false;
        }
        UpdateCognitoUserPoolConfiguration other = (UpdateCognitoUserPoolConfiguration) obj;
        return Objects.equals(userPoolArn(), other.userPoolArn()) && hasClientIds() == other.hasClientIds()
                && Objects.equals(clientIds(), other.clientIds())
                && Objects.equals(groupConfiguration(), other.groupConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UpdateCognitoUserPoolConfiguration").add("UserPoolArn", userPoolArn())
                .add("ClientIds", clientIds() == null ? null : "*** Sensitive Data Redacted ***")
                .add("GroupConfiguration", groupConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "userPoolArn":
            return Optional.ofNullable(clazz.cast(userPoolArn()));
        case "clientIds":
            return Optional.ofNullable(clazz.cast(clientIds()));
        case "groupConfiguration":
            return Optional.ofNullable(clazz.cast(groupConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UpdateCognitoUserPoolConfiguration, T> g) {
        return obj -> g.apply((UpdateCognitoUserPoolConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, UpdateCognitoUserPoolConfiguration> {
        /**
         * <p>
         * The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
         * (ARN)</a> of the Amazon Cognito user pool associated with this identity source.
         * </p>
         * 
         * @param userPoolArn
         *        The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
         *        Resource Name (ARN)</a> of the Amazon Cognito user pool associated with this identity source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userPoolArn(String userPoolArn);

        /**
         * <p>
         * The client ID of an app client that is configured for the specified Amazon Cognito user pool.
         * </p>
         * 
         * @param clientIds
         *        The client ID of an app client that is configured for the specified Amazon Cognito user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientIds(Collection<String> clientIds);

        /**
         * <p>
         * The client ID of an app client that is configured for the specified Amazon Cognito user pool.
         * </p>
         * 
         * @param clientIds
         *        The client ID of an app client that is configured for the specified Amazon Cognito user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientIds(String... clientIds);

        /**
         * <p>
         * The configuration of the user groups from an Amazon Cognito user pool identity source.
         * </p>
         * 
         * @param groupConfiguration
         *        The configuration of the user groups from an Amazon Cognito user pool identity source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupConfiguration(UpdateCognitoGroupConfiguration groupConfiguration);

        /**
         * <p>
         * The configuration of the user groups from an Amazon Cognito user pool identity source.
         * </p>
         * This is a convenience method that creates an instance of the {@link UpdateCognitoGroupConfiguration.Builder}
         * avoiding the need to create one manually via {@link UpdateCognitoGroupConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link UpdateCognitoGroupConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #groupConfiguration(UpdateCognitoGroupConfiguration)}.
         * 
         * @param groupConfiguration
         *        a consumer that will call methods on {@link UpdateCognitoGroupConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #groupConfiguration(UpdateCognitoGroupConfiguration)
         */
        default Builder groupConfiguration(Consumer<UpdateCognitoGroupConfiguration.Builder> groupConfiguration) {
            return groupConfiguration(UpdateCognitoGroupConfiguration.builder().applyMutation(groupConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String userPoolArn;

        private List<String> clientIds = DefaultSdkAutoConstructList.getInstance();

        private UpdateCognitoGroupConfiguration groupConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateCognitoUserPoolConfiguration model) {
            userPoolArn(model.userPoolArn);
            clientIds(model.clientIds);
            groupConfiguration(model.groupConfiguration);
        }

        public final String getUserPoolArn() {
            return userPoolArn;
        }

        public final void setUserPoolArn(String userPoolArn) {
            this.userPoolArn = userPoolArn;
        }

        @Override
        public final Builder userPoolArn(String userPoolArn) {
            this.userPoolArn = userPoolArn;
            return this;
        }

        public final Collection<String> getClientIds() {
            if (clientIds instanceof SdkAutoConstructList) {
                return null;
            }
            return clientIds;
        }

        public final void setClientIds(Collection<String> clientIds) {
            this.clientIds = ClientIdsCopier.copy(clientIds);
        }

        @Override
        public final Builder clientIds(Collection<String> clientIds) {
            this.clientIds = ClientIdsCopier.copy(clientIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder clientIds(String... clientIds) {
            clientIds(Arrays.asList(clientIds));
            return this;
        }

        public final UpdateCognitoGroupConfiguration.Builder getGroupConfiguration() {
            return groupConfiguration != null ? groupConfiguration.toBuilder() : null;
        }

        public final void setGroupConfiguration(UpdateCognitoGroupConfiguration.BuilderImpl groupConfiguration) {
            this.groupConfiguration = groupConfiguration != null ? groupConfiguration.build() : null;
        }

        @Override
        public final Builder groupConfiguration(UpdateCognitoGroupConfiguration groupConfiguration) {
            this.groupConfiguration = groupConfiguration;
            return this;
        }

        @Override
        public UpdateCognitoUserPoolConfiguration build() {
            return new UpdateCognitoUserPoolConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
