/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.verifiedpermissions.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about a policy.
 * </p>
 * <p>
 * This data type is used as a response parameter for the <a
 * href="https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_ListPolicies.html">ListPolicies</a>
 * operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PolicyItem implements SdkPojo, Serializable, ToCopyableBuilder<PolicyItem.Builder, PolicyItem> {
    private static final SdkField<String> POLICY_STORE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("policyStoreId").getter(getter(PolicyItem::policyStoreId)).setter(setter(Builder::policyStoreId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("policyStoreId").build()).build();

    private static final SdkField<String> POLICY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("policyId").getter(getter(PolicyItem::policyId)).setter(setter(Builder::policyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("policyId").build()).build();

    private static final SdkField<String> POLICY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("policyType").getter(getter(PolicyItem::policyTypeAsString)).setter(setter(Builder::policyType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("policyType").build()).build();

    private static final SdkField<EntityIdentifier> PRINCIPAL_FIELD = SdkField
            .<EntityIdentifier> builder(MarshallingType.SDK_POJO).memberName("principal").getter(getter(PolicyItem::principal))
            .setter(setter(Builder::principal)).constructor(EntityIdentifier::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("principal").build()).build();

    private static final SdkField<EntityIdentifier> RESOURCE_FIELD = SdkField
            .<EntityIdentifier> builder(MarshallingType.SDK_POJO).memberName("resource").getter(getter(PolicyItem::resource))
            .setter(setter(Builder::resource)).constructor(EntityIdentifier::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resource").build()).build();

    private static final SdkField<List<ActionIdentifier>> ACTIONS_FIELD = SdkField
            .<List<ActionIdentifier>> builder(MarshallingType.LIST)
            .memberName("actions")
            .getter(getter(PolicyItem::actions))
            .setter(setter(Builder::actions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("actions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ActionIdentifier> builder(MarshallingType.SDK_POJO)
                                            .constructor(ActionIdentifier::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<PolicyDefinitionItem> DEFINITION_FIELD = SdkField
            .<PolicyDefinitionItem> builder(MarshallingType.SDK_POJO).memberName("definition")
            .getter(getter(PolicyItem::definition)).setter(setter(Builder::definition))
            .constructor(PolicyDefinitionItem::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("definition").build()).build();

    private static final SdkField<Instant> CREATED_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdDate")
            .getter(getter(PolicyItem::createdDate))
            .setter(setter(Builder::createdDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdDate").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> LAST_UPDATED_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastUpdatedDate")
            .getter(getter(PolicyItem::lastUpdatedDate))
            .setter(setter(Builder::lastUpdatedDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdatedDate").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> EFFECT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("effect")
            .getter(getter(PolicyItem::effectAsString)).setter(setter(Builder::effect))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("effect").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(POLICY_STORE_ID_FIELD,
            POLICY_ID_FIELD, POLICY_TYPE_FIELD, PRINCIPAL_FIELD, RESOURCE_FIELD, ACTIONS_FIELD, DEFINITION_FIELD,
            CREATED_DATE_FIELD, LAST_UPDATED_DATE_FIELD, EFFECT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String policyStoreId;

    private final String policyId;

    private final String policyType;

    private final EntityIdentifier principal;

    private final EntityIdentifier resource;

    private final List<ActionIdentifier> actions;

    private final PolicyDefinitionItem definition;

    private final Instant createdDate;

    private final Instant lastUpdatedDate;

    private final String effect;

    private PolicyItem(BuilderImpl builder) {
        this.policyStoreId = builder.policyStoreId;
        this.policyId = builder.policyId;
        this.policyType = builder.policyType;
        this.principal = builder.principal;
        this.resource = builder.resource;
        this.actions = builder.actions;
        this.definition = builder.definition;
        this.createdDate = builder.createdDate;
        this.lastUpdatedDate = builder.lastUpdatedDate;
        this.effect = builder.effect;
    }

    /**
     * <p>
     * The identifier of the PolicyStore where the policy you want information about is stored.
     * </p>
     * 
     * @return The identifier of the PolicyStore where the policy you want information about is stored.
     */
    public final String policyStoreId() {
        return policyStoreId;
    }

    /**
     * <p>
     * The identifier of the policy you want information about.
     * </p>
     * 
     * @return The identifier of the policy you want information about.
     */
    public final String policyId() {
        return policyId;
    }

    /**
     * <p>
     * The type of the policy. This is one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>static</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>templateLinked</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #policyType} will
     * return {@link PolicyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #policyTypeAsString}.
     * </p>
     * 
     * @return The type of the policy. This is one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>static</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>templateLinked</code>
     *         </p>
     *         </li>
     * @see PolicyType
     */
    public final PolicyType policyType() {
        return PolicyType.fromValue(policyType);
    }

    /**
     * <p>
     * The type of the policy. This is one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>static</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>templateLinked</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #policyType} will
     * return {@link PolicyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #policyTypeAsString}.
     * </p>
     * 
     * @return The type of the policy. This is one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>static</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>templateLinked</code>
     *         </p>
     *         </li>
     * @see PolicyType
     */
    public final String policyTypeAsString() {
        return policyType;
    }

    /**
     * <p>
     * The principal associated with the policy.
     * </p>
     * 
     * @return The principal associated with the policy.
     */
    public final EntityIdentifier principal() {
        return principal;
    }

    /**
     * <p>
     * The resource associated with the policy.
     * </p>
     * 
     * @return The resource associated with the policy.
     */
    public final EntityIdentifier resource() {
        return resource;
    }

    /**
     * For responses, this returns true if the service returned a value for the Actions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasActions() {
        return actions != null && !(actions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The action that a policy permits or forbids. For example,
     * <code>{"actions": [{"actionId": "ViewPhoto", "actionType": "PhotoFlash::Action"}, {"entityID": "SharePhoto", "entityType": "PhotoFlash::Action"}]}</code>
     * .
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasActions} method.
     * </p>
     * 
     * @return The action that a policy permits or forbids. For example,
     *         <code>{"actions": [{"actionId": "ViewPhoto", "actionType": "PhotoFlash::Action"}, {"entityID": "SharePhoto", "entityType": "PhotoFlash::Action"}]}</code>
     *         .
     */
    public final List<ActionIdentifier> actions() {
        return actions;
    }

    /**
     * <p>
     * The policy definition of an item in the list of policies returned.
     * </p>
     * 
     * @return The policy definition of an item in the list of policies returned.
     */
    public final PolicyDefinitionItem definition() {
        return definition;
    }

    /**
     * <p>
     * The date and time the policy was created.
     * </p>
     * 
     * @return The date and time the policy was created.
     */
    public final Instant createdDate() {
        return createdDate;
    }

    /**
     * <p>
     * The date and time the policy was most recently updated.
     * </p>
     * 
     * @return The date and time the policy was most recently updated.
     */
    public final Instant lastUpdatedDate() {
        return lastUpdatedDate;
    }

    /**
     * <p>
     * The effect of the decision that a policy returns to an authorization request. For example,
     * <code>"effect": "Permit"</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #effect} will
     * return {@link PolicyEffect#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #effectAsString}.
     * </p>
     * 
     * @return The effect of the decision that a policy returns to an authorization request. For example,
     *         <code>"effect": "Permit"</code>.
     * @see PolicyEffect
     */
    public final PolicyEffect effect() {
        return PolicyEffect.fromValue(effect);
    }

    /**
     * <p>
     * The effect of the decision that a policy returns to an authorization request. For example,
     * <code>"effect": "Permit"</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #effect} will
     * return {@link PolicyEffect#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #effectAsString}.
     * </p>
     * 
     * @return The effect of the decision that a policy returns to an authorization request. For example,
     *         <code>"effect": "Permit"</code>.
     * @see PolicyEffect
     */
    public final String effectAsString() {
        return effect;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(policyStoreId());
        hashCode = 31 * hashCode + Objects.hashCode(policyId());
        hashCode = 31 * hashCode + Objects.hashCode(policyTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(principal());
        hashCode = 31 * hashCode + Objects.hashCode(resource());
        hashCode = 31 * hashCode + Objects.hashCode(hasActions() ? actions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(definition());
        hashCode = 31 * hashCode + Objects.hashCode(createdDate());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedDate());
        hashCode = 31 * hashCode + Objects.hashCode(effectAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PolicyItem)) {
            return false;
        }
        PolicyItem other = (PolicyItem) obj;
        return Objects.equals(policyStoreId(), other.policyStoreId()) && Objects.equals(policyId(), other.policyId())
                && Objects.equals(policyTypeAsString(), other.policyTypeAsString())
                && Objects.equals(principal(), other.principal()) && Objects.equals(resource(), other.resource())
                && hasActions() == other.hasActions() && Objects.equals(actions(), other.actions())
                && Objects.equals(definition(), other.definition()) && Objects.equals(createdDate(), other.createdDate())
                && Objects.equals(lastUpdatedDate(), other.lastUpdatedDate())
                && Objects.equals(effectAsString(), other.effectAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PolicyItem").add("PolicyStoreId", policyStoreId()).add("PolicyId", policyId())
                .add("PolicyType", policyTypeAsString()).add("Principal", principal()).add("Resource", resource())
                .add("Actions", hasActions() ? actions() : null).add("Definition", definition())
                .add("CreatedDate", createdDate()).add("LastUpdatedDate", lastUpdatedDate()).add("Effect", effectAsString())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "policyStoreId":
            return Optional.ofNullable(clazz.cast(policyStoreId()));
        case "policyId":
            return Optional.ofNullable(clazz.cast(policyId()));
        case "policyType":
            return Optional.ofNullable(clazz.cast(policyTypeAsString()));
        case "principal":
            return Optional.ofNullable(clazz.cast(principal()));
        case "resource":
            return Optional.ofNullable(clazz.cast(resource()));
        case "actions":
            return Optional.ofNullable(clazz.cast(actions()));
        case "definition":
            return Optional.ofNullable(clazz.cast(definition()));
        case "createdDate":
            return Optional.ofNullable(clazz.cast(createdDate()));
        case "lastUpdatedDate":
            return Optional.ofNullable(clazz.cast(lastUpdatedDate()));
        case "effect":
            return Optional.ofNullable(clazz.cast(effectAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PolicyItem, T> g) {
        return obj -> g.apply((PolicyItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PolicyItem> {
        /**
         * <p>
         * The identifier of the PolicyStore where the policy you want information about is stored.
         * </p>
         * 
         * @param policyStoreId
         *        The identifier of the PolicyStore where the policy you want information about is stored.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyStoreId(String policyStoreId);

        /**
         * <p>
         * The identifier of the policy you want information about.
         * </p>
         * 
         * @param policyId
         *        The identifier of the policy you want information about.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyId(String policyId);

        /**
         * <p>
         * The type of the policy. This is one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>static</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>templateLinked</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param policyType
         *        The type of the policy. This is one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>static</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>templateLinked</code>
         *        </p>
         *        </li>
         * @see PolicyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyType
         */
        Builder policyType(String policyType);

        /**
         * <p>
         * The type of the policy. This is one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>static</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>templateLinked</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param policyType
         *        The type of the policy. This is one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>static</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>templateLinked</code>
         *        </p>
         *        </li>
         * @see PolicyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyType
         */
        Builder policyType(PolicyType policyType);

        /**
         * <p>
         * The principal associated with the policy.
         * </p>
         * 
         * @param principal
         *        The principal associated with the policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder principal(EntityIdentifier principal);

        /**
         * <p>
         * The principal associated with the policy.
         * </p>
         * This is a convenience method that creates an instance of the {@link EntityIdentifier.Builder} avoiding the
         * need to create one manually via {@link EntityIdentifier#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EntityIdentifier.Builder#build()} is called immediately and its
         * result is passed to {@link #principal(EntityIdentifier)}.
         * 
         * @param principal
         *        a consumer that will call methods on {@link EntityIdentifier.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #principal(EntityIdentifier)
         */
        default Builder principal(Consumer<EntityIdentifier.Builder> principal) {
            return principal(EntityIdentifier.builder().applyMutation(principal).build());
        }

        /**
         * <p>
         * The resource associated with the policy.
         * </p>
         * 
         * @param resource
         *        The resource associated with the policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resource(EntityIdentifier resource);

        /**
         * <p>
         * The resource associated with the policy.
         * </p>
         * This is a convenience method that creates an instance of the {@link EntityIdentifier.Builder} avoiding the
         * need to create one manually via {@link EntityIdentifier#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EntityIdentifier.Builder#build()} is called immediately and its
         * result is passed to {@link #resource(EntityIdentifier)}.
         * 
         * @param resource
         *        a consumer that will call methods on {@link EntityIdentifier.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resource(EntityIdentifier)
         */
        default Builder resource(Consumer<EntityIdentifier.Builder> resource) {
            return resource(EntityIdentifier.builder().applyMutation(resource).build());
        }

        /**
         * <p>
         * The action that a policy permits or forbids. For example,
         * <code>{"actions": [{"actionId": "ViewPhoto", "actionType": "PhotoFlash::Action"}, {"entityID": "SharePhoto", "entityType": "PhotoFlash::Action"}]}</code>
         * .
         * </p>
         * 
         * @param actions
         *        The action that a policy permits or forbids. For example,
         *        <code>{"actions": [{"actionId": "ViewPhoto", "actionType": "PhotoFlash::Action"}, {"entityID": "SharePhoto", "entityType": "PhotoFlash::Action"}]}</code>
         *        .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(Collection<ActionIdentifier> actions);

        /**
         * <p>
         * The action that a policy permits or forbids. For example,
         * <code>{"actions": [{"actionId": "ViewPhoto", "actionType": "PhotoFlash::Action"}, {"entityID": "SharePhoto", "entityType": "PhotoFlash::Action"}]}</code>
         * .
         * </p>
         * 
         * @param actions
         *        The action that a policy permits or forbids. For example,
         *        <code>{"actions": [{"actionId": "ViewPhoto", "actionType": "PhotoFlash::Action"}, {"entityID": "SharePhoto", "entityType": "PhotoFlash::Action"}]}</code>
         *        .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(ActionIdentifier... actions);

        /**
         * <p>
         * The action that a policy permits or forbids. For example,
         * <code>{"actions": [{"actionId": "ViewPhoto", "actionType": "PhotoFlash::Action"}, {"entityID": "SharePhoto", "entityType": "PhotoFlash::Action"}]}</code>
         * .
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.verifiedpermissions.model.ActionIdentifier.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.verifiedpermissions.model.ActionIdentifier#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.verifiedpermissions.model.ActionIdentifier.Builder#build()} is called
         * immediately and its result is passed to {@link #actions(List<ActionIdentifier>)}.
         * 
         * @param actions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.verifiedpermissions.model.ActionIdentifier.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #actions(java.util.Collection<ActionIdentifier>)
         */
        Builder actions(Consumer<ActionIdentifier.Builder>... actions);

        /**
         * <p>
         * The policy definition of an item in the list of policies returned.
         * </p>
         * 
         * @param definition
         *        The policy definition of an item in the list of policies returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder definition(PolicyDefinitionItem definition);

        /**
         * <p>
         * The policy definition of an item in the list of policies returned.
         * </p>
         * This is a convenience method that creates an instance of the {@link PolicyDefinitionItem.Builder} avoiding
         * the need to create one manually via {@link PolicyDefinitionItem#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PolicyDefinitionItem.Builder#build()} is called immediately and
         * its result is passed to {@link #definition(PolicyDefinitionItem)}.
         * 
         * @param definition
         *        a consumer that will call methods on {@link PolicyDefinitionItem.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #definition(PolicyDefinitionItem)
         */
        default Builder definition(Consumer<PolicyDefinitionItem.Builder> definition) {
            return definition(PolicyDefinitionItem.builder().applyMutation(definition).build());
        }

        /**
         * <p>
         * The date and time the policy was created.
         * </p>
         * 
         * @param createdDate
         *        The date and time the policy was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdDate(Instant createdDate);

        /**
         * <p>
         * The date and time the policy was most recently updated.
         * </p>
         * 
         * @param lastUpdatedDate
         *        The date and time the policy was most recently updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedDate(Instant lastUpdatedDate);

        /**
         * <p>
         * The effect of the decision that a policy returns to an authorization request. For example,
         * <code>"effect": "Permit"</code>.
         * </p>
         * 
         * @param effect
         *        The effect of the decision that a policy returns to an authorization request. For example,
         *        <code>"effect": "Permit"</code>.
         * @see PolicyEffect
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyEffect
         */
        Builder effect(String effect);

        /**
         * <p>
         * The effect of the decision that a policy returns to an authorization request. For example,
         * <code>"effect": "Permit"</code>.
         * </p>
         * 
         * @param effect
         *        The effect of the decision that a policy returns to an authorization request. For example,
         *        <code>"effect": "Permit"</code>.
         * @see PolicyEffect
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyEffect
         */
        Builder effect(PolicyEffect effect);
    }

    static final class BuilderImpl implements Builder {
        private String policyStoreId;

        private String policyId;

        private String policyType;

        private EntityIdentifier principal;

        private EntityIdentifier resource;

        private List<ActionIdentifier> actions = DefaultSdkAutoConstructList.getInstance();

        private PolicyDefinitionItem definition;

        private Instant createdDate;

        private Instant lastUpdatedDate;

        private String effect;

        private BuilderImpl() {
        }

        private BuilderImpl(PolicyItem model) {
            policyStoreId(model.policyStoreId);
            policyId(model.policyId);
            policyType(model.policyType);
            principal(model.principal);
            resource(model.resource);
            actions(model.actions);
            definition(model.definition);
            createdDate(model.createdDate);
            lastUpdatedDate(model.lastUpdatedDate);
            effect(model.effect);
        }

        public final String getPolicyStoreId() {
            return policyStoreId;
        }

        public final void setPolicyStoreId(String policyStoreId) {
            this.policyStoreId = policyStoreId;
        }

        @Override
        public final Builder policyStoreId(String policyStoreId) {
            this.policyStoreId = policyStoreId;
            return this;
        }

        public final String getPolicyId() {
            return policyId;
        }

        public final void setPolicyId(String policyId) {
            this.policyId = policyId;
        }

        @Override
        public final Builder policyId(String policyId) {
            this.policyId = policyId;
            return this;
        }

        public final String getPolicyType() {
            return policyType;
        }

        public final void setPolicyType(String policyType) {
            this.policyType = policyType;
        }

        @Override
        public final Builder policyType(String policyType) {
            this.policyType = policyType;
            return this;
        }

        @Override
        public final Builder policyType(PolicyType policyType) {
            this.policyType(policyType == null ? null : policyType.toString());
            return this;
        }

        public final EntityIdentifier.Builder getPrincipal() {
            return principal != null ? principal.toBuilder() : null;
        }

        public final void setPrincipal(EntityIdentifier.BuilderImpl principal) {
            this.principal = principal != null ? principal.build() : null;
        }

        @Override
        public final Builder principal(EntityIdentifier principal) {
            this.principal = principal;
            return this;
        }

        public final EntityIdentifier.Builder getResource() {
            return resource != null ? resource.toBuilder() : null;
        }

        public final void setResource(EntityIdentifier.BuilderImpl resource) {
            this.resource = resource != null ? resource.build() : null;
        }

        @Override
        public final Builder resource(EntityIdentifier resource) {
            this.resource = resource;
            return this;
        }

        public final List<ActionIdentifier.Builder> getActions() {
            List<ActionIdentifier.Builder> result = ActionIdentifierListCopier.copyToBuilder(this.actions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setActions(Collection<ActionIdentifier.BuilderImpl> actions) {
            this.actions = ActionIdentifierListCopier.copyFromBuilder(actions);
        }

        @Override
        public final Builder actions(Collection<ActionIdentifier> actions) {
            this.actions = ActionIdentifierListCopier.copy(actions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actions(ActionIdentifier... actions) {
            actions(Arrays.asList(actions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actions(Consumer<ActionIdentifier.Builder>... actions) {
            actions(Stream.of(actions).map(c -> ActionIdentifier.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final PolicyDefinitionItem.Builder getDefinition() {
            return definition != null ? definition.toBuilder() : null;
        }

        public final void setDefinition(PolicyDefinitionItem.BuilderImpl definition) {
            this.definition = definition != null ? definition.build() : null;
        }

        @Override
        public final Builder definition(PolicyDefinitionItem definition) {
            this.definition = definition;
            return this;
        }

        public final Instant getCreatedDate() {
            return createdDate;
        }

        public final void setCreatedDate(Instant createdDate) {
            this.createdDate = createdDate;
        }

        @Override
        public final Builder createdDate(Instant createdDate) {
            this.createdDate = createdDate;
            return this;
        }

        public final Instant getLastUpdatedDate() {
            return lastUpdatedDate;
        }

        public final void setLastUpdatedDate(Instant lastUpdatedDate) {
            this.lastUpdatedDate = lastUpdatedDate;
        }

        @Override
        public final Builder lastUpdatedDate(Instant lastUpdatedDate) {
            this.lastUpdatedDate = lastUpdatedDate;
            return this;
        }

        public final String getEffect() {
            return effect;
        }

        public final void setEffect(String effect) {
            this.effect = effect;
        }

        @Override
        public final Builder effect(String effect) {
            this.effect = effect;
            return this;
        }

        @Override
        public final Builder effect(PolicyEffect effect) {
            this.effect(effect == null ? null : effect.toString());
            return this;
        }

        @Override
        public PolicyItem build() {
            return new PolicyItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
